/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2015 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::motionInterpolation

Description
    Base class for interpolation of cell displacement fields, generated by
    fvMotionSolvers, to the points. This base class implements the default
    method which applies volPointInterpolation only.

SourceFiles
    motionInterpolation.C

\*---------------------------------------------------------------------------*/

#ifndef motionInterpolation_H
#define motionInterpolation_H

#include "fvMesh.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                         Class motionInterpolation Declaration
\*---------------------------------------------------------------------------*/

class motionInterpolation
{
    // Private data

        //- Reference to the FV mesh
        const fvMesh& mesh_;


    // Private Member Functions

        //- No copy construct
        motionInterpolation(const motionInterpolation&) = delete;

        //- No copy assignment
        void operator=(const motionInterpolation&) = delete;


public:

    //- Runtime type information
    TypeName("motionInterpolation");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            autoPtr,
            motionInterpolation,
            Istream,
            (
                const fvMesh& mesh,
                Istream& entry
            ),
            (mesh, entry)
        );


    // Selectors

        //- Select default
        static autoPtr<motionInterpolation> New(const fvMesh& mesh);

        //- Select from stream
        static autoPtr<motionInterpolation> New
        (
            const fvMesh& mesh,
            Istream& entry
        );


    // Constructors

        //- Construct from an fvMesh
        motionInterpolation(const fvMesh& mesh);

        //- Construct from an fvMesh and an Istream
        motionInterpolation(const fvMesh& mesh, Istream& entry);


    //- Destructor
    virtual ~motionInterpolation();


    // Member Functions

        //- Return const-reference to the mesh
        const fvMesh& mesh() const
        {
            return mesh_;
        }

        //- Interpolate the given scalar cell displacement
        virtual void interpolate
        (
            const volScalarField&,
            pointScalarField&
        ) const;

        //- Interpolate the given vector cell displacement
        virtual void interpolate
        (
            const volVectorField&,
            pointVectorField&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
