#! /usr/bin/env python
# -*- coding: utf-8 -*-

#   OpenTeacher
#   depends on: python-qt4
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

import xml.dom.minidom
import xml.parsers.expat
import datetime
import words
import sys

import errors

class XmlFile(xml.dom.minidom.Document):
	"""XmlFile is a class used by the export methods of the filetypes to create XML when they need to."""
	def __init__(self):
		xml.dom.minidom.Document.__init__(self)

	def save(self, location):
		location = location.encode(sys.getfilesystemencoding())
		fileObject = open(location, "w")
		self.writexml(fileObject, encoding="UTF-8")
		fileObject.close()

	def createTextNode(self, text):
		text = unicode(text).encode("UTF-8")
		return xml.dom.minidom.Document.createTextNode(self, text)

class ABBYLingvoTutorFile:
	"""The representation of an ABBY Lingvo Tutor (.xml) file"""
	def __init__(self, location):
		self.wordList = words.WordList()
		self.location = location

	def load(self):
		try:
			dom = xml.dom.minidom.parse(self.location.encode(sys.getfilesystemencoding()))
		except xml.parsers.expat.ExpatError, e:
			print e
			raise errors.ContentError()
		try:
			root = dom.getElementsByTagName("dictionary")[0]
		except IndexError, e:
			print e
			raise errors.ContentError()

		if root.hasAttribute("title"):
			self.wordList.title = root.getAttribute("title")
		for card in root.getElementsByTagName("card"):
			word = words.Word()
			try:
				word.question = card.getElementsByTagName("word")[0].childNodes[0].nodeValue
			except IndexError, e:
				print e
				raise errors.ContentError()
			try:
				translations = card.getElementsByTagName("translations")[0]
			except IndexError, e:
				print e
				raise errors.ContentError()
			answers = translations.getElementsByTagName("word")
			try:
				word.answer = answers[0].childNodes[0].nodeValue
			except IndexError:
				pass
			try:
				answers = answers[1:]
				answers = [a.childNodes[0].nodeValue for a in answers]
				word.secondAnswer = u", ".join(answers)
			except IndexError:
				pass
			self.wordList.addWord(word)

class OpenTeacherFile:
	"""The OpenTeacherFile class represents an xml file""" 
	def __init__(self, location):
		self.wordList = words.WordList()
		self.location = location

	def load(self):
		try:
			dom = xml.dom.minidom.parse(self.location.encode(sys.getfilesystemencoding()))
		except xml.parsers.expat.ExpatError, e:
			print e
			raise errors.ContentError()

		try:
			self.wordList.title = dom.getElementsByTagName("title")[0].childNodes[0].nodeValue
		except IndexError:
			pass

		try:
			self.wordList.questionLanguage = dom.getElementsByTagName("question_language")[0].childNodes[0].nodeValue
		except IndexError:
			pass

		try:
			self.wordList.answerLanguage = dom.getElementsByTagName("answer_language")[0].childNodes[0].nodeValue
		except IndexError:
			pass

		wordsDom = dom.getElementsByTagName("word")
		for wordDom in wordsDom:
			word = words.Word()

			try:
				results = wordDom.getElementsByTagName("results")[0].childNodes[0].nodeValue.split("/")

				word.question = wordDom.getElementsByTagName("known")[0].childNodes[0].nodeValue
				word.answer = wordDom.getElementsByTagName("foreign")[0].childNodes[0].nodeValue
				word.errorCount = results[0]
				word.testCount = results[1]
			except IndexError, e:
				print e
				raise errors.ContentError

			try:
				word.secondAnswer = wordDom.getElementsByTagName("second")[0].childNodes[0].nodeValue
			except IndexError:
				pass
			self.wordList.addWord(word)

	def save(self):
		#create an xml document
		xml = XmlFile()
		#create a <root> element
		rootElement = xml.createElement("root")
		xml.appendChild(rootElement)

		#create a <title> element
		titleElement = xml.createElement("title")
		titleElement.appendChild(xml.createTextNode(self.wordList.title))
		rootElement.appendChild(titleElement)

		#create a <question_language> element
		questionLanguageElement = xml.createElement("question_language")
		questionLanguageElement.appendChild(xml.createTextNode(self.wordList.questionLanguage))
		rootElement.appendChild(questionLanguageElement)

		#create a <answer_language> element
		answerLanguageElement = xml.createElement("answer_language")
		answerLanguageElement.appendChild(xml.createTextNode(self.wordList.answerLanguage))
		rootElement.appendChild(answerLanguageElement)

		for word in self.wordList:
			#create a <word> element
			wordElement = xml.createElement("word")
			rootElement.appendChild(wordElement)
			#create a <known> element in <word>
			knownElement = xml.createElement("known")
			knownElement.appendChild(xml.createTextNode(word.question))
			wordElement.appendChild(knownElement)
			#create a <foreign> element in <word>
			foreignElement = xml.createElement("foreign")
			foreignElement.appendChild(xml.createTextNode(word.answer))
			wordElement.appendChild(foreignElement)
			#create a <second> element in <word>
			if word.secondAnswerSet():
				secondElement = xml.createElement("second")
				secondElement.appendChild(xml.createTextNode(word.secondAnswer))
				wordElement.appendChild(secondElement)
			#create a <results> element in <word>
			resultsElement = xml.createElement("results")
			resultsElement.appendChild(xml.createTextNode(word.results))
			wordElement.appendChild(resultsElement)
		xml.save(self.location)

class Teach2000File:
	def __init__(self, location):
		self.wordList = words.WordList()
		self.location = unicode(location)

	def load(self):
		try:
			dom = xml.dom.minidom.parse(self.location.encode(sys.getfilesystemencoding()))
		except xml.parsers.expat.ExpatError, e:
			print e
			raise errors.ContentError()

		itemDoms = dom.getElementsByTagName("item")
		for itemDom in itemDoms:
			word = words.Word()

			try:
				word.question = itemDom.getElementsByTagName("question")[0].childNodes[0].nodeValue
				answers = itemDom.getElementsByTagName("answer")
				word.answer = answers[0].childNodes[0].nodeValue
			except IndexError, e:
				print e
				raise errors.ContentError


			try:
				word.secondAnswer = answers[1].childNodes[0].nodeValue
			except IndexError:
				pass

			try:
				word.errorCount = itemDom.getElementsByTagName("errors")[0].childNodes[0].nodeValue
				word.testCount = itemDom.getElementsByTagName("testcount")[0].childNodes[0].nodeValue
			except IndexError, e:
				print e
				raise errors.ContentError


			id = itemDom.getAttribute("id")
			word.id = id

			self.wordList.addWord(word)

	def save(self):
		#create an xml document
		xml = XmlFile()
		#create a comment tag, default for a Teach2000 file
		xml.appendChild(xml.createComment("This is a Teach2000 document (http://teach2000.memtrain.com)"))

		#create a <teach2000> element
		teach2000Element = xml.createElement("teach2000")
		xml.appendChild(teach2000Element)

		#set metadata
		#version
		versionElement = xml.createElement("version")
		versionElement.appendChild(xml.createTextNode("831"))
		teach2000Element.appendChild(versionElement)

		#description
		descriptionElement = xml.createElement("description")
		descriptionElement.appendChild(xml.createTextNode("Normal"))
		teach2000Element.appendChild(descriptionElement)

		#filedata
		message_dataElement = xml.createElement("message_data")

		message_dataElement.setAttribute("mm_files_embedded", "N")
		message_dataElement.setAttribute("encrypted", "N")

		teach2000Element.appendChild(message_dataElement)

		#fonts
		font_questionElement = xml.createElement("font_question")
		font_questionElement.appendChild(xml.createTextNode("Arial"))
		message_dataElement.appendChild(font_questionElement)

		font_answerElement = xml.createElement("font_answer")
		font_answerElement.appendChild(xml.createTextNode("Lucida Sans Unicode"))
		message_dataElement.appendChild(font_answerElement)

		#create <items> element
		itemsElement = xml.createElement("items")
		message_dataElement.appendChild(itemsElement)

		#testresults
		testresultsElement = xml.createElement("testresults")
		message_dataElement.appendChild(testresultsElement)

		#mapquizfile
		mapquizfileElement = xml.createElement("mapquizfile")
		message_dataElement.appendChild(mapquizfileElement)

		#Teach2000 doesn't support metadata - so title, questionLanguage and
		#answerLanguage aren't used.

		for word in self.wordList:
			#create a <item> element
			itemElement = xml.createElement("item")
			itemElement.setAttribute("id", unicode(word.id))
			itemsElement.appendChild(itemElement)

			#create <questions> element
			questionsElement = xml.createElement("questions")
			itemElement.appendChild(questionsElement)

			#create <question> element
			questionElement = xml.createElement("question")
			questionElement.setAttribute("id", "0")
			questionElement.appendChild(xml.createTextNode(word.question))
			questionsElement.appendChild(questionElement)

			#create <answers> element
			answersElement = xml.createElement("answers")
			answersElement.setAttribute("type", "0")
			itemElement.appendChild(answersElement)

			#create <answer> element(s)
			answerElement = xml.createElement("answer")
			answerElement.setAttribute("id", "0")
			answerElement.appendChild(xml.createTextNode(word.answer))
			answersElement.appendChild(answerElement)
			if word.secondAnswerSet():
				secondAnswerElement = xml.createElement("answer")
				secondAnswerElement.setAttribute("id", "1")
				secondAnswerElement.appendChild(xml.createTextNode(word.secondAnswer))
				answersElement.appendChild(secondAnswerElement)

			#create <errors> element
			errorsElement = xml.createElement("errors")
			errorsElement.appendChild(xml.createTextNode(word.errorCount))
			itemElement.appendChild(errorsElement)

			#create <testcount> element
			testcountElement = xml.createElement("testcount")
			testcountElement.appendChild(xml.createTextNode(word.testCount))
			itemElement.appendChild(testcountElement)
		xml.save(self.location)

class WrtsFile:
	def __init__(self, location):
		self.wordList = words.WordList()
		self.location = unicode(location)

	def load(self):
		try:
			dom = xml.dom.minidom.parse(self.location.encode(sys.getfilesystemencoding()))
		except xml.parsers.expat.ExpatError, e:
			print e
			raise errors.ContentError()

		try:
			languageDom = dom.getElementsByTagName("taal")[0]
		except IndexError, e:
			print e
			raise errors.ContentError

		try:
			self.wordList.title = dom.getElementsByTagName("titel")[0].childNodes[0].nodeValue
		except IndexError:
			self.wordList.title = ""
		try:
			self.wordList.questionLanguage = languageDom.getElementsByTagName("a")[0].childNodes[0].nodeValue
		except IndexError:
			self.wordList.questionLanguage = ""
		try:
			self.wordList.answerLanguage = languageDom.getElementsByTagName("b")[0].childNodes[0].nodeValue
		except IndexError:
			self.wordList.answerLanguage = ""

		wordDoms = dom.getElementsByTagName("woord")

		for wordDom in wordDoms:
			word = words.Word()

			try:
				word.question = wordDom.getElementsByTagName("a")[0].childNodes[0].nodeValue
				answers = wordDom.getElementsByTagName("b")[0].childNodes[0].nodeValue
			except IndexError, e:
				print e
				raise errors.ContentError

			#Split answers if more; Wrts doesn't know something like a 'second answer'.
			if "," in answers:
				answers = answers.split(",", 1)
				word.answer = answers[0]
				word.secondAnswer = answers[1]
			elif ";" in answers:
				answers = answers.split(";", 1)
				word.answer = answers[0]
				word.secondAnswer = answer[1]
			else:
				word.answer = answers

			self.wordList.addWord(word)

	def save(self):
		#create xml document
		xml = XmlFile()
		#create wrts-comment
		commentElement = xml.createComment("Wrts, http://www.wrts.nl/")
		xml.appendChild(commentElement)
		
		#create a <wrts> element
		wrtsElement = xml.createElement("wrts")
		xml.appendChild(wrtsElement)
		
		#create a <lijst> element (list in english) in <wrts>
		listElement = xml.createElement("lijst")
		listElement.setAttribute("id", "00000000")
		wrtsElement.appendChild(listElement)

		#create a <titel> element (title in english)
		titleElement = xml.createElement("titel")
		titleElement.appendChild(xml.createTextNode(self.wordList.title))
		listElement.appendChild(titleElement)

		#get time in right format.
		time = datetime.datetime.now().strftime("%a, %d %b %Y %H:%M:%S %z")
		
		#create <datum> element (date in english)
		dateElement = xml.createElement("datum")

		#create some elements about time
		downloadedElement = xml.createElement("downloaded")
		createdElement = xml.createElement("created")
		updatedElement = xml.createElement("updated")

		#fill them with current time
		downloadedElement.appendChild(xml.createTextNode(time))
		createdElement.appendChild(xml.createTextNode(time))
		updatedElement.appendChild(xml.createTextNode(time))

		#append them to the <datum> element
		dateElement.appendChild(downloadedElement)
		dateElement.appendChild(createdElement)
		dateElement.appendChild(updatedElement)

		#add the <datum> element to the <lijst> element
		listElement.appendChild(dateElement)

		#add the <author> element to the <lijst> element
		authorElement = xml.createElement("author")
		authorElement.appendChild(xml.createTextNode("Created by: OpenTeacher"))
		authorElement.setAttribute("id", "000000")
		listElement.appendChild(authorElement)

		#add the <taal> element to the <lijst> element (language in english)
		languageElement = xml.createElement("taal")

		questionLanguageElement = xml.createElement("a")
		questionLanguageElement.appendChild(xml.createTextNode(self.wordList.questionLanguage))
		languageElement.appendChild(questionLanguageElement)

		answerLanguageElement = xml.createElement("b")
		answerLanguageElement.appendChild(xml.createTextNode(self.wordList.answerLanguage))
		languageElement.appendChild(answerLanguageElement)

		listElement.appendChild(languageElement)

		for word in self.wordList:
			#create a <woord> element (word in english)
			wordElement = xml.createElement("woord")
			listElement.appendChild(wordElement)
			#create a <a> (question) element in <woord>
			questionElement = xml.createElement("a")
			questionElement.appendChild(xml.createTextNode(word.question))
			wordElement.appendChild(questionElement)
			answer = word.answer
			if word.secondAnswerSet():
				answer += ", " + word.secondAnswer
			#create a <b> (answer) element in <woord>
			answerElement = xml.createElement("b")
			answerElement.appendChild(xml.createTextNode(answer))
			wordElement.appendChild(answerElement)
		xml.save(self.location)

class TextFile:
	"""TextFile exports a wordlist to plain text. This class can save it to
	   a file, or return a unicode string.

	   Methods:
		   getText
		   save
	   Attributes:
		   wordList"""

	def __init__(self, location):
		self.location = location

	def getText(self):
		text = u""

		if self.wordList.titleSet():
			text += self.wordList.title + "\n\n"
		if self.wordList.questionLanguageSet() and self.wordList.answerLanguageSet():
				text += self.wordList.questionLanguage + " - " + self.wordList.answerLanguage + "\n\n"
		maxlen = 8

		for word in self.wordList.words:
			curlen = len(word.question)
			if curlen +1 > maxlen:
				maxlen = curlen +1

		for word in self.wordList.words:
			text += word.question + (maxlen - len(word.question)) * " " + word.answer
			if word.secondAnswerSet():
				text += ", %s\n" % word.secondAnswer
			else:
				text += "\n"
		return text

	def save(self):
		text = self.getText()
		text = text.encode("UTF-8")
		fileObject = open(self.location, "w")
		fileObject.write(text)
		fileObject.close()
