///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>

#include <core/viewport/Window3D.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/SceneRenderer.h>

namespace Scripting {

using namespace boost::python;

void ExportViewport()
{
	class_<CameraViewDescription>("CameraViewDescription", no_init)
		.def_readonly("AspectRatio", &CameraViewDescription::aspectRatio)
		.def_readonly("IsPerspective", &CameraViewDescription::isPerspective)
		.def_readonly("ZNear", &CameraViewDescription::znear)
		.def_readonly("ZFar", &CameraViewDescription::zfar)
		.def_readonly("FieldOfView", &CameraViewDescription::fieldOfView)
		.def_readonly("ViewMatrix", &CameraViewDescription::viewMatrix)
		.def_readonly("InverseViewMatrix", &CameraViewDescription::inverseViewMatrix)
		.def_readonly("ProjectionMatrix", &CameraViewDescription::projectionMatrix)
		.def_readonly("InverseProjectionMatrix", &CameraViewDescription::inverseProjectionMatrix)
	;

	class_<ViewportRecord, bases<RefTarget>, intrusive_ptr<ViewportRecord>, noncopyable>("ViewportRecord", no_init)
	;

	class_<Viewport, noncopyable>("Viewport", no_init)
		.def("UpdateViewport", &Viewport::updateViewport)
		.add_property("Caption", make_function(&Viewport::caption, return_value_policy<copy_const_reference>()))
		.add_property("FieldOfView", &Viewport::fieldOfView, &Viewport::setFieldOfView)
		.add_property("IsPerspectiveProjection", &Viewport::isPerspectiveProjection)
		.add_property("CurrentView", make_function(&Viewport::currentView, return_value_policy<copy_const_reference>()))
		.add_property("Settings", make_function(&Viewport::settings, return_internal_reference<>()))
		.def("GetViewDescription", &Viewport::getViewDescription)
	;

	class_<ViewportManager, noncopyable>("ViewportManager", no_init)
		.add_static_property("Instance", make_function(&ViewportManager::getSingletonInstance, return_value_policy<reference_existing_object>()))
		.add_property("ActiveViewport", make_function(&ViewportManager::activeViewport, return_internal_reference<>()), &ViewportManager::setActiveViewport)
		.add_property("MaximizedViewport", make_function(&ViewportManager::maximizedViewport, return_internal_reference<>()), &ViewportManager::setMaximizedViewport)
		.def("UpdateViewports", &ViewportManager::updateViewports)
		.add_property("Viewports", make_function(&ViewportManager::viewports, return_internal_reference<>()))
	;

	enum_<SceneRenderer::SceneExtentsMode>("SceneExtentsMode")
			.value("ALL", SceneRenderer::ALL)
			.value("SELECTED_OBJECTS", SceneRenderer::SELECTED_OBJECTS)
			.value("RENDERABLE_OBJECTS", SceneRenderer::RENDERABLE_OBJECTS)
	;

	class_<SceneRenderer, noncopyable>("SceneRenderer", no_init)
		.add_static_property("ActiveRenderer", make_function(&SceneRenderer::activeRenderer, return_value_policy<reference_existing_object>()))
		.def("SceneExtents", make_function((Box3 (SceneRenderer::*)(const intrusive_ptr<ViewportRecord>&, TimeTicks, SceneRenderer::SceneExtentsMode))&SceneRenderer::sceneExtents, return_value_policy<return_by_value>()))
	;
}

};
