//
// C++ Implementation: vocabularymodel
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "vocabularymodel.h"

#include <ept/debtags/vocabulary.h>
#include <ept/debtags/tag.h>
#include <wibble/operators.h>

#include "aggregator.h"
#include "eptinstance.h"

#include <cassert>

using namespace wibble::operators;


// NTagModel
#include "vocabularymodelrole.h"

typedef ept::debtags::Tag Tag;
typedef ept::debtags::Facet Facet;



TagWrapper::TagWrapper() {}
TagWrapper::TagWrapper(Tag tag_) : tag(tag_) {}

namespace NTagModel
{

VocabularyModel::VocabularyModel()
{
	const set<Facet> facets = EptInstance::aggregator()->vocabulary().facets();
	
	int i = 0;
	for (set<Facet>::const_iterator it = facets.begin(); it != facets.end(); ++it, ++i)
	{
		_facets.push_back(FacetData(*it, i));
		FacetData* pFd = &(_facets[i]);
		set<Tag> tags = it->tags();
		vector<TagData> tagData;
		int j = 0;
		for (set<Tag>::const_iterator jt = tags.begin(); jt != tags.end(); ++jt, ++j)
		{
			tagData.push_back( TagData(*jt, i) );	
			_tagToTagData[*jt] = make_pair(i, j);
		}
		_tags.push_back(tagData);
	}
	_companionTagsValid = false;
	qDebug("%d facets found", rowCount(QModelIndex()));
}


VocabularyModel::~VocabularyModel()
{
}


int VocabularyModel::rowCount(const QModelIndex & parent) const
{
	// if the number of facets was requested
	if (!parent.isValid())
	{
		return _facets.size();
	}
	else 
	{
		// only the first column has any children
		if (parent.column() != 0)
			return 0;
		const ItemData* pData = (ItemData*) parent.internalPointer();
		assert(pData != 0);
		// if the parent is a facet, the number of children is the number of tags in the facet
		if (pData->isFacet())
		{
			const FacetData* pParentData = (FacetData*) pData;
			const Facet facet = pParentData->facet;
			return facet.tags().size();
		}
		else	// parent is a tag which do not have children (since tag grouping is not supported)
			return 0;
	}
}

QVariant VocabularyModel::data(const QModelIndex& index, int role) const
{
// 	qDebug("Requesting data for row: %d, col: %d", index.row(), index.column());
	// sometime an item -1, -1 is requested
	if (index.row() < 0 || index.column() < 0)
		return QVariant();
	switch (role)
	{
		case Qt::ToolTipRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			assert(pData != 0);
			return pData->description();
		}
		case Qt::DisplayRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			assert(pData != 0);
			if (index.column() == 1)
				return pData->fullname();
			return pData->name();
		}
		case SelectedRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			if (!pData->isFacet())
			{
				const TagData* pTagData = pData->toTagData();
				return pTagData->selected;
			}
			else
				return false;
		}
		case HiddenRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			if (pData->isFacet())
				return pData->toFacetData()->hidden;
			else
				return false;
		}
		case TypeRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			if (!pData->isFacet())
			{
				return TagTypeItem;
			}
			else
				return FacetTypeItem;
		}
		case TagRole:
		{
			const ItemData* pData = (ItemData*) index.internalPointer();
			assert(pData->toTagData());
			Tag tag = pData->toTagData()->tag;
			// dummy to pull in default constructor symbol
// 			TagWrapper tw();
 			return QVariant::fromValue(TagWrapper(tag));
// 			return QVariant();
		}
	}
	return QVariant();
}

QVariant VocabularyModel::headerData( int section, Qt::Orientation orientation, 
	int role) const
{
	if (role == Qt::DisplayRole)
	{
		if (section == 0)
			return "Name";
		if (section == 1)
			return "ID";
	}
	return QVariant();
}

QModelIndex VocabularyModel::parent(const QModelIndex & index) const
{
	if (!index.isValid())
		return QModelIndex();
	const ItemData* pData = (ItemData*) index.internalPointer();
	assert(pData != 0);
	if (pData->isFacet())
	{
// 		qDebug("[parent()] Parent is facet: " + pData->fullname());
		return QModelIndex();	// no parent for facets
	}
	else
	{
		const TagData* pTagData = static_cast<const TagData*>(pData);
		FacetData facetData = _facets.at(pTagData->facetIndex);
// 		qDebug("[parent()] Getting parent for tag " + pTagData->fullname() + " with facet " +
// 			facetData.fullname() + " at row %d", facetData.row);
		return createIndex(facetData.row, 0, 
				const_cast<void*>(dynamic_cast<const void*>(&(_facets[pTagData->facetIndex]))) );
	}
}

QModelIndex VocabularyModel::index (int row, int column, const QModelIndex & parent) const
{
// 	qDebug("[VocabularyModel::index()] requesting index row %i, col %i", row, column);
	
	// requesting index for a facet
	if (row < 0 || column < 0)
		return QModelIndex();	
	if (!parent.isValid())
	{
		if (row >= _facets.size() || column >= 2)
		{
			qWarning("[VocabularyModel::index()] Warning: row or column to large, row: %d, column, %d", row, column);
			return QModelIndex();	
		}
		const FacetData* pFd = &_facets[row];
 		return createIndex(row, column, const_cast<void*>(dynamic_cast<const void*>(pFd)));
	}
	const ItemData* pData = (ItemData*) parent.internalPointer();
	if (!pData->isFacet())
		return QModelIndex();
	assert(dynamic_cast<const FacetData*>(pData) != 0);
	FacetData* pFacetData = (FacetData*) parent.internalPointer();
	Facet facet = pFacetData->facet;
	if (row >= facet.tags().size() || column >= 2)
	{
		qDebug("[VocabularyModel::index()] Warning: row or column to large, row: %d, column, %d", row, column);
		qDebug("[VocabularyModel::index()] Facet: " + pFacetData->fullname());
		return QModelIndex();
	}
	const TagData* pTd = &(_tags[pFacetData->row][row]);
 	return createIndex( row, column, (void*)  pTd);
}


bool VocabularyModel::setData(const QModelIndex& index, const QVariant& value, int role)
{
	qDebug("[VocabularyModel::setData()]: called");
	switch(role)
	{
		case SelectedRole:
		{
			ItemData* pData = (ItemData*) index.internalPointer();
			TagData* pTagData = pData->toTagData();
			// setting selected for facets is not allowed/ignored
			if (!pTagData)
			{
				goto hiddenLabel;
				return false;
			}
			pTagData->selected = value.toBool();
			qDebug("[VocabularyModel::setData()] size before insert/remove: %lu", _selectedTags.size());
			// if the tag is selected
			if (value.toBool())
				_selectedTags.insert(pTagData->tag);
			else	// the tag is unselected
				_selectedTags.erase(pTagData->tag);
			qDebug("[VocabularyModel::setData()] size after insert/remove: %lu", _selectedTags.size());
			_companionTagsValid = false;
// 			reset();
// 			emitAllDataChanged();
			emit(dataChanged(index, index));
			return true;
		}
		case HiddenRole:
		{
			hiddenLabel:
			ItemData* pData = (ItemData*) index.internalPointer();
			if (pData->isFacet())
			{
				pData->toFacetData()->hidden = value.toBool();
				emit(dataChanged(index, index));
				return true;
			}
			else
			{
				qWarning("[VocabularyModel::setData()] trying to set hidden for a tag which is not supported");
			}
		}
		default:
			return QAbstractItemModel::setData(index, value, role);
	}
	return false;
}

void VocabularyModel::emitAllDataChanged()
{
	QModelIndex root = QModelIndex();
	for(int i = 0; i < rowCount(root); ++i)
	{
		QModelIndex parent = index(i, 0, root);
		QModelIndex tl = index(0, 0, parent);
		QModelIndex br = index(rowCount(parent)-1, columnCount(parent)-1, parent);
		qDebug("Top Left Item: " + data(tl, Qt::DisplayRole).toString());
		qDebug("Bottom Right Item: " + data(br, Qt::DisplayRole).toString());
		emit(dataChanged(tl, br));
	}
	QModelIndex parent = index(0, 0, root);
	QModelIndex br = index(rowCount(root)-1, columnCount(parent)-1, root);
	emit(dataChanged(parent, br));
}


QModelIndex VocabularyModel::indexForTag(const Tag& tag, int column) const
{
	map<Tag, pair<int, int> >::const_iterator it = _tagToTagData.find(tag);
	if (it == _tagToTagData.end())
		return QModelIndex();
	pair<int, int> indexes = it->second;
	int facetIndex = indexes.first;
	int tagIndex = indexes.second;
	const TagData* pTd = &(_tags[facetIndex][tagIndex]);
	return createIndex( tagIndex, column, (void*)  pTd);
}


Facet VocabularyModel::getFacet(int index) const
{
	return getElement(EptInstance::aggregator()->vocabulary().facets(), index);
}

const set<Tag>& VocabularyModel::selectedTags() const
{
	return _selectedTags;
/*	_selectedTagsCache.clear();
	// iterate over all facets and collect the selected tags inside those
	QModelIndex parent = QModelIndex();
	for(int i = 0; i < rowCount(parent); ++i)
	{
		QModelIndex index = index(i, 0, parent);
		_selectedTagsCache |= collectSelectedChildItems(index);
	}
	return result;*/
}
	
std::set<Tag> VocabularyModel::collectSelectedChildItems(const QModelIndex& parent) const
{
	std::set<Tag> result;
	for(int i = 0; i < rowCount(parent); ++i)
	{
		QModelIndex index = this->index(i, 0, parent);
		// if the item is selected
		if (data(index, NTagModel::SelectedRole).toBool())
		{
			NTagModel::TagData* pTagData = (NTagModel::TagData*) index.internalPointer();
			result.insert(pTagData->tag);
		}
		result |= collectSelectedChildItems(index);
	}
	return result;
}



}	// NTagModel

#undef emit
//#include <ept/cache/tag.tcc>
//#include <ept/cache/debtags/vocabulary.tcc>
