/*
 * Copyright (c) 2000, 2001, 2002 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define _GNU_SOURCE

#include <grp.h>
#include <pwd.h>
#include <time.h>
#include <dirent.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#define PAM_SM_AUTH
#include <security/pam_modules.h>

#include "public.h"
#include "common.h"

static int debug_flag = 0;

extern int sigfholder(const char *path, int sig);

int
pam_sm_authenticate (pam_handle_t *pamh __attribute__ ((unused)),
		     int flags __attribute__ ((unused)),
		     int argc __attribute__ ((unused)),
		     const char **argv __attribute__ ((unused)))
{
  return PAM_IGNORE;
}

int
pam_sm_setcred (pam_handle_t *pamh __attribute__ ((unused)),
		int flags __attribute__ ((unused)),
		int argc __attribute__ ((unused)),
		const char **argv __attribute__ ((unused)))
{
  return PAM_IGNORE;
}

int
pam_sm_open_session (pam_handle_t *pamh, int flags, int argc,
                     const char **argv)
{
  FILE *fp, *flogout;
  int pw_buflen = 256;
  char *pw_buffer = alloca (pw_buflen);
  struct passwd pw_resultbuf;
  struct passwd *pw = NULL;
  const char *name = NULL;
  const char *configfile = "/etc/logindevperm";
  char *save_perms;
  char *tty = NULL;
  char buf[BUFSIZ];
  int retval;
  unsigned long int i;
  int written = 0;
  /* Flags for options */
  int abort_on_error = 0;

  /* Parse parameters for module */
  for ( ; argc-- > 0; argv++)
    {
      if (strcasecmp (*argv, "abort") == 0)
	abort_on_error = 1;
      else if (strcasecmp (*argv, "debug") == 0)
	debug_flag = 1;
      else
	__log_err (LOG_ERR, "session: Unknown option: %s", *argv);
    }

  /* get the user name */
  if ((retval = pam_get_user (pamh, &name, NULL)) != PAM_SUCCESS)
    return retval;

  if (name == NULL || *name == '\0' || (!isalnum(*name) && *name != '_'))
    {
      if (name)
        __log_err (LOG_ERR, "bad username [%s]", name);
      return PAM_USER_UNKNOWN;
    }

  while (getpwnam_r (name, &pw_resultbuf, pw_buffer, pw_buflen, &pw) != 0
	 && errno == ERANGE)
    {
      errno = 0;
      pw_buflen += 256;
      pw_buffer = alloca (pw_buflen);
    }

  if (pw == NULL)
    return PAM_USER_UNKNOWN;

  retval = pam_get_item (pamh, PAM_TTY, (const void **)&tty);
  if (retval != PAM_SUCCESS || tty == NULL)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       "cannot determine user's tty");
      return PAM_SERVICE_ERR;
    }

  if (debug_flag)
    __log_err (LOG_DEBUG, "User is on tty [%s]", tty);

  save_perms = alloca (strlen (SAVE_PERMS_PATH) + strlen (tty) + 2);
  /* XXX */
  strcpy (save_perms, SAVE_PERMS_PATH);
  strcat (save_perms, tty);
  for (i = strlen (SAVE_PERMS_PATH); i < strlen (save_perms); ++i)
    if (save_perms[i] == '/')
      save_perms[i] = '_';

  fp = fopen(configfile, "r");
  if (fp == NULL)
    {
      if (abort_on_error)
	{
	  __write_message (pamh, flags, PAM_ERROR_MSG,
			   "Error opening %s: %s", configfile,
			   strerror (errno));
	  return PAM_PERM_DENIED;
	}
      else
	{
	  __write_message (pamh, flags, PAM_TEXT_INFO,
			   "Error opening %s: %s", configfile,
			   strerror (errno));
	  return PAM_SUCCESS;
	}
    }

  flogout = fopen(save_perms, "w");
  if (flogout == NULL)
    {
      if (abort_on_error)
	{
	  __write_message (pamh, flags, PAM_ERROR_MSG,
			   "Error opening %s: %s", save_perms,
			   strerror (errno));
	  return PAM_PERM_DENIED;
	}
      else
	{
	  __write_message (pamh, flags, PAM_TEXT_INFO,
			   "Error opening %s: %s", save_perms,
			   strerror (errno));
	  return PAM_SUCCESS;
	}
    }
  else
    chmod (save_perms, 0600);

  while (fgets (buf, sizeof(buf), fp))
    {
#define WSPACE          " \t\n"
      char *device, *cp;
      int perm;

      if ((cp = strchr (buf, '#')) != NULL)
	*cp = 0;                            /* strip comment */
      if ((cp = device = strtok (buf, WSPACE)) == 0)
	continue;                           /* empty or comment */
      if (!(strncmp (device, "/dev/", 5) == 0
	    || (device[0] == ':' && isdigit(device[1]))
	    || (device[0] == '*' && device[1] == '\0'))
	  || (cp = strtok((char *) 0, WSPACE)) == 0
	  || *cp != '0'
	  || sscanf(cp, "%o", &perm) == 0
	  || perm == 0
	  || (perm & 0777) != perm
	  || (cp = strtok((char *) 0, WSPACE)) == 0)
	{
	  if (abort_on_error)
	    {
	      __write_message (pamh, flags, PAM_ERROR_MSG, "bad entry: %s",
			       cp ? cp : "(null)");
	      return PAM_PERM_DENIED;
	    }
	  else
	    {
	      __write_message (pamh, flags, PAM_TEXT_INFO, "bad entry: %s",
			       cp ? cp : "(null)");
	      continue;
	    }
	}

      if ((strcmp (device, "*") == 0) || (strcmp (device, tty) == 0))
	{
	  if (debug_flag)
	    __log_err (LOG_DEBUG, "tty=%s, perm=%o, devices=%s",
		       device, perm, cp);

	  for (cp = strtok (cp, ":"); cp; cp = strtok ((char *) 0, ":"))
	    {
	      login_protect (cp, perm, pw->pw_uid, pw->pw_gid, flogout);
	      ++written;
	    }
	}
    }
  fclose(fp);
  fclose(flogout);
  if (written == 0)
    unlink (save_perms);

  return PAM_SUCCESS;
}

int
pam_sm_close_session (pam_handle_t * pamh, int flags,
                      int argc, const char **argv)
{
  char *tty = NULL;
  int retval;

  /* Parse parameters for module */
  for ( ; argc-- > 0; argv++)
    {
      /* Ignore abort option */
      if (strcasecmp (*argv, "debug") == 0)
	debug_flag = 1;
      else if (strcasecmp (*argv, "abort") != 0)
	__log_err (LOG_ERR, "session: Unknown option: %s", *argv);
    }

  retval = pam_get_item (pamh, PAM_TTY, (const void **)&tty);
  if (retval != PAM_SUCCESS || tty == NULL)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       "cannot determine user's tty");
      return PAM_SERVICE_ERR;
    }

  if (restore_permissions (tty) != 0)
    return PAM_SERVICE_ERR;

  return PAM_SUCCESS;
}

/* static module data */
#ifdef PAM_STATIC
struct pam_module _pam_devperm_modstruct =
{
  "pam_devperm",
  pam_sm_authenticate,
  pam_sm_setcred,
  NULL /* pam_sm_acct_mgmt */,
  pam_sm_open_session,
  pam_sm_close_session,
  NULL /* pam_sm_chauthtok */
};
#endif
