/*
 * dialog_filechooser.cc
 * Copyright (C) Martin Singer 2009-2011 <m_power3@users.sourceforge.net>
 * 
 * pdfchain is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * pdfchain is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "dialog_filechooser.h"


/*** Static Class Variables ***************************************************/

// File Chooser Dialog
//std::string cFCDialog::sCurrentFolder = Pdfchain::Path::DEFAULT_FOLDER_DOCUMENTS;					//TODO: DELETE external variables doesn't work with virtual variables
std::string cFCDialog::sCurrentFolder = Glib::get_user_special_dir( G_USER_DIRECTORY_DOCUMENTS );


/*** File Chooser Dialog ******************************************************/

// Constructor
cFCDialog::cFCDialog( Gtk::Window& parent , const Glib::ustring& title , Gtk::FileChooserAction action , bool select_multiple , Pdfchain::Id::FileType file_type )
:
	Gtk::FileChooserDialog( parent , title , action ),
	eFileType( file_type ),
	pButton_Ok( NULL ),
	pButton_Cancel( NULL )
{
	switch ( file_type ) {
		
		case Pdfchain::Id::FILE_TYPE_ANY:
			break;
		case Pdfchain::Id::FILE_TYPE_PDF:
			add_filter( Glib::RefPtr<cFFilter_Pdf>( new cFFilter_Pdf() ) );
			add_filter( Glib::RefPtr<cFFilter_Any>( new cFFilter_Any() ) );
			break;
		case Pdfchain::Id::FILE_TYPE_FDF:
			add_filter( Glib::RefPtr<cFFilter_Fdf>( new cFFilter_Fdf() ) );
			add_filter( Glib::RefPtr<cFFilter_Text>( new cFFilter_Text() ) );
			add_filter( Glib::RefPtr<cFFilter_Any>( new cFFilter_Any() ) );
			break;
		case Pdfchain::Id::FILE_TYPE_DUMP:
			add_filter( Glib::RefPtr<cFFilter_Dump>( new cFFilter_Dump() ) );
			add_filter( Glib::RefPtr<cFFilter_Text>( new cFFilter_Text() ) );
			add_filter( Glib::RefPtr<cFFilter_Any>( new cFFilter_Any() ) );
			break;
		case Pdfchain::Id::FILE_TYPE_DIR:
			add_filter( Glib::RefPtr<cFFilter_Any>( new cFFilter_Any() ) );
			add_filter( Glib::RefPtr<cFFilter_Pdf>( new cFFilter_Pdf() ) );
			break;
	}

	pButton_Cancel	= add_button( Gtk::Stock::CANCEL	, Gtk::RESPONSE_CANCEL );
	
	switch ( action ) {
		case Gtk::FILE_CHOOSER_ACTION_OPEN:
			
			switch ( select_multiple ) {
				case true:
					pButton_Ok = add_button( Gtk::Stock::ADD , Gtk::RESPONSE_OK );
					break;
				case false:
					pButton_Ok = add_button( Gtk::Stock::OPEN , Gtk::RESPONSE_OK );
					break;
			}
			
			break;
		case Gtk::FILE_CHOOSER_ACTION_SAVE:
			pButton_Ok = add_button( Gtk::Stock::SAVE , Gtk::RESPONSE_OK );
			break;
		
		case Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER:
		case Gtk::FILE_CHOOSER_ACTION_CREATE_FOLDER:
			pButton_Ok = add_button( Gtk::Stock::DIRECTORY , Gtk::RESPONSE_OK );	// Stockicon DIRECTORY doesn't work!?!
			//pButton_Ok = add_button( Gtk::Stock::OPEN , Gtk::RESPONSE_OK );			// Replacement icon for Gtk::Stock::DIRECTORY works!?!
			break;
		
		default:
			pButton_Ok = add_button( Gtk::Stock::OPEN , Gtk::RESPONSE_OK );
			break;
	}

	set_select_multiple( select_multiple );
	set_current_folder( sCurrentFolder );
}


// Destructor
cFCDialog::~cFCDialog()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cFCDialog::~cFCDialog()";	//TEST
#endif
	
	if ( pButton_Ok )		{ delete pButton_Ok;		pButton_Ok		= NULL; }
	if ( pButton_Cancel )	{ delete pButton_Cancel;	pButton_Cancel	= NULL; }
}


// Methode (public) : run
int
cFCDialog::run()
{
	int response_id = Gtk::RESPONSE_NONE;
	set_current_folder( sCurrentFolder );
	
	if ( Gtk::RESPONSE_OK == ( response_id = Gtk::Dialog::run() ) )
		sCurrentFolder = get_current_folder();
	
	return response_id;
}



/*** File Cooser Dialog : Cat - Add Files *************************************/

// Constructor
cFCDialog_Cat_AddFiles::cFCDialog_Cat_AddFiles( Gtk::Window& parent , const Glib::ustring& title , Gtk::FileChooserAction action , bool select_multiple , Pdfchain::Id::FileType file_type )
:
	cFCDialog( parent , title , action , select_multiple , file_type )	// parent_window , title , action , select_multiple , file_type
{	
	set_extra_widget( mExpander );
}


// Destructor
cFCDialog_Cat_AddFiles::~cFCDialog_Cat_AddFiles()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cFCDialog_Cat_AddFiles::~cFCDialog_Cat_AddFiles()";	//TEST
#endif
}


// Method (public) : get String of Entry Pages
Glib::ustring
cFCDialog_Cat_AddFiles::getStr_Pages()
{
	return mExpander.mEntry_Pages.get_text();
}


// Method (public) : get String of ComboBox EvenOdd
Glib::ustring
cFCDialog_Cat_AddFiles::getStr_EvenOdd()
{
	Gtk::TreeModel::iterator	iter		= mExpander.mCBox_EvenOdd.get_active();
	cTMCRecord_Selection*		pt_record	= mExpander.mCBox_EvenOdd.rLStore->getTMCRecord();
	return (*iter)[pt_record->mTMColumn_Label];
}


// Method (public) : get String of ComboBox Rotation
Glib::ustring
cFCDialog_Cat_AddFiles::getStr_Rotation()
{
	Gtk::TreeModel::iterator	iter		= mExpander.mCBox_Rotation.get_active();
	cTMCRecord_Selection*		pt_record	= mExpander.mCBox_Rotation.rLStore->getTMCRecord();
	return (*iter)[pt_record->mTMColumn_Label];
}


// Method (public) : get String of Entry Password
Glib::ustring
cFCDialog_Cat_AddFiles::getStr_Password() {
	return mExpander.mEntry_Password.get_text();
}


// Method (public) : get Command of Combobox EvenOdd
Glib::ustring
cFCDialog_Cat_AddFiles::getCmd_EvenOdd()
{
	Gtk::TreeModel::iterator	iter		= mExpander.mCBox_EvenOdd.get_active();
	cTMCRecord_Selection*		pt_record	= mExpander.mCBox_EvenOdd.rLStore->getTMCRecord();
	return iter->get_value( pt_record->mTMColumn_Command );
}


// Method (public) : get Command of Combobox Rotation
Glib::ustring
cFCDialog_Cat_AddFiles::getCmd_Rotation()
{
	Gtk::TreeModel::iterator	iter		= mExpander.mCBox_Rotation.get_active();
	cTMCRecord_Selection*		pt_record	= mExpander.mCBox_Rotation.rLStore->getTMCRecord();
	return iter->get_value( pt_record->mTMColumn_Command );
}



/*** File Chooser Dialog : Save ***********************************************/

// Constructor
cFCDialog_SaveAs::cFCDialog_SaveAs( Gtk::Window& ref_window , const Glib::ustring& title , Pdfchain::Id::FileType file_type )
:
	cFCDialog( ref_window , title , Gtk::FILE_CHOOSER_ACTION_SAVE , false , file_type ),	// parent_window , title , action , select_multiple , file_type
	mCButton_AddExtension( _("Add _extension") , true ),	// label , mnemonic
	mMDialog_OverwriteConfirm( ref_window , _("Overwrite file?") , false ,  Gtk::MESSAGE_QUESTION , Gtk::BUTTONS_YES_NO  , false )
{
	mCButton_AddExtension.set_active( true );
	set_extra_widget( mCButton_AddExtension );
	//set_do_overwrite_confirmation( true );	// Doesn't work with Auto Add Extension
}


// Destructor
cFCDialog_SaveAs::~cFCDialog_SaveAs()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cFCDialog_SaveAs::~cFCDialog_SaveAs()";	//TEST
#endif
}


// Methode (public) : get filename
std::string
cFCDialog_SaveAs::get_filename()
{
	std::string str_filename = Gtk::FileChooser::get_filename();
	
	switch ( mCButton_AddExtension.get_active() ) {
		case true:
			str_filename = addExtension( str_filename );
			break;
	}

	if ( Gio::File::create_for_path( str_filename )->query_exists() ) {
		mMDialog_OverwriteConfirm.set_secondary_text( Glib::filename_to_utf8( str_filename ) );
		switch ( mMDialog_OverwriteConfirm.run() ) {
			case Gtk::RESPONSE_YES:
				mMDialog_OverwriteConfirm.hide();
				break;
			case Gtk::RESPONSE_NO:
				mMDialog_OverwriteConfirm.hide();
				str_filename = "";
				break;
		}
	}
	
	return str_filename;
}


// Methode (prtected) : add extension
std::string
cFCDialog_SaveAs::addExtension( std::string str_filename )
{
	std::string				str_extension	= "";
	std::string::size_type	size_extension	= 0;
	std::string::size_type	size_filename	= str_filename.length();

	switch ( eFileType ) {
		case Pdfchain::Id::FILE_TYPE_PDF:
			str_extension = Pdfchain::Extension::EXT_PDF;
			break;
		case Pdfchain::Id::FILE_TYPE_FDF:
			str_extension = Pdfchain::Extension::EXT_FDF;
			break;
		case Pdfchain::Id::FILE_TYPE_DUMP:
			str_extension = Pdfchain::Extension::EXT_DUMP;
			break;
	}

	if ( size_extension = str_extension.length() )
		if ( str_filename.compare( size_filename - size_extension , size_filename , str_extension ) )
			str_filename.append( str_extension );
	
	return str_filename;
}


/*** File Chooser Button ******************************************************/

// Constructor
cFCButton::cFCButton( const Glib::ustring& str_title , Gtk::FileChooserAction action , Pdfchain::Id::FileType file_type )
:
	Gtk::FileChooserButton( str_title , action )
{
	switch ( file_type ) {
		case Pdfchain::Id::FILE_TYPE_PDF:
			add_filter( Glib::RefPtr<cFFilter_Pdf>( new cFFilter_Pdf() ) );
			break;
		case Pdfchain::Id::FILE_TYPE_FDF:
			add_filter( Glib::RefPtr<cFFilter_Fdf>( new cFFilter_Fdf() ) );
			add_filter( Glib::RefPtr<cFFilter_Text>( new cFFilter_Text() ) );
			break;
		case Pdfchain::Id::FILE_TYPE_DUMP:
			add_filter( Glib::RefPtr<cFFilter_Dump>( new cFFilter_Dump() ) );
			add_filter( Glib::RefPtr<cFFilter_Text>( new cFFilter_Text() ) );
			break;
	}
			
	add_filter( Glib::RefPtr<cFFilter_Any>( new cFFilter_Any() ) );

	//set_select_multiple( false );
	//set_current_folder( Pdfchain::Path::DEFAULT_FOLDER_DOCUMENTS );			//OLD
	set_current_folder( Glib::get_user_special_dir( G_USER_DIRECTORY_DOCUMENTS ) );
}


// Destructor
cFCButton::~cFCButton()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cFCButton::~cFCButton()";	//TEST
#endif
}