/* parse.c - parse configuration file */
/*
 *  Penguineyes
 *  Copyright (C) 1998/1999 Neil Howie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <glob.h>

#include "penguineyes.h"

const char *default_config_file = NULL;

/* comments stripped for brevity, see glib.h for details */
static	GScannerConfig	scanner_config =
{
    (" \t\n"), (G_CSET_a_2_z "_/" G_CSET_A_2_Z),
    ( G_CSET_a_2_z "_0123456789./" G_CSET_A_2_Z ), ( "#\n" ),
    FALSE, FALSE, TRUE, FALSE, TRUE, TRUE, FALSE, TRUE, FALSE, FALSE,
    FALSE, FALSE, FALSE, FALSE, TRUE, TRUE, FALSE, TRUE, TRUE, TRUE, FALSE
};
enum {
    TOKEN_NEW = G_TOKEN_LAST,
    TOKEN_MASK,		TOKEN_WIDTH,	TOKEN_HEIGHT,	TOKEN_COLOUR,
    TOKEN_EYE,		TOKEN_EYEPIX,	TOKEN_GEO,	TOKEN_RATE,
    TOKEN_THEME,	TOKEN_END,	TOKEN_SUBDIR,	TOKEN_SAY,
    TOKEN_LAUNCH,	TOKEN_DLAUNCH,	TOKEN_FHORIZ,	TOKEN_FVERT,
    TOKEN_FLIP,		TOKEN_DEFAULT,	TOKEN_CLOCK,	TOKEN_SMOOTH,
    TOKEN_SHAND,	TOKEN_MHAND,	TOKEN_HHAND,	TOKEN_ADEFAULT,
    TOKEN_GUILTY,	TOKEN_LAST
};

static struct {
    gchar *name;
    gint token;
} symbols[] = {
    {"image",	TOKEN_NEW	}, {"mask",	TOKEN_MASK	},
    {"width",	TOKEN_WIDTH	}, {"height",	TOKEN_HEIGHT	},
    {"eye_colour",TOKEN_COLOUR	}, {"eye",	TOKEN_EYE	},
    {"pixmap",	TOKEN_EYEPIX	}, {"theme",	TOKEN_THEME	},
    {"geometry",TOKEN_GEO	}, {"rate",	TOKEN_RATE	},
    {"end",	TOKEN_END	}, {"say",	TOKEN_SAY	},
    {"launch",	TOKEN_LAUNCH	}, {"default_launch", TOKEN_DLAUNCH },
    {"flip_v",	TOKEN_FVERT	}, {"flip_h",	TOKEN_FHORIZ	},
    {"flip",	TOKEN_FLIP	}, {"subdir",	TOKEN_SUBDIR	},
    {"clock",	TOKEN_CLOCK	}, {"smooth",	TOKEN_SMOOTH	},
    {"second_hand", TOKEN_SHAND	}, {"minute_hand", TOKEN_MHAND	},
    {"hour_hand", TOKEN_HHAND	}, {"guilty",	TOKEN_GUILTY	},
    {"applet_default", TOKEN_ADEFAULT }, {"default",	TOKEN_DEFAULT	}
};

static const guint nsymbols = sizeof(symbols) / sizeof(symbols[0]);

enum { PARSE_OK, PARSE_ERROR, PARSE_DONE };

GSList *theme_list;
struct theme *default_theme;

static gchar *dname = NULL; // name of default theme

static gchar *image_file(gchar *orig, gchar *path)
{
    gchar *tstr;

    if (FILE_EXISTS(orig))
	return g_strdup(orig);

    if (*orig != '/') {
	tstr = g_strdup_printf("%s/.penguineyes/%s", getenv("HOME"), orig);
	if (FILE_EXISTS(tstr)) return tstr;
	g_free(tstr);

	tstr = g_strdup_printf("%s/.penguineyes/%s/%s", getenv("HOME"),
		path, orig);
	if (FILE_EXISTS(tstr)) return tstr;
	g_free(tstr);

	tstr = g_strdup_printf(DATADIR"/%s", orig);
	if (FILE_EXISTS(tstr)) return tstr;
	g_free(tstr);

	tstr = g_strdup_printf(DATADIR"/%s/%s", path, orig);
	if (FILE_EXISTS(tstr)) return tstr;
	g_free(tstr);
    }

    tstr = g_strdup_printf("unable to find file '%s'", orig);
    g_warning(tstr);
    g_free(tstr);
    return NULL;
}

static gboolean get_int(GScanner *scn, int *var)
{
    int toke;

    toke = g_scanner_peek_next_token(scn);
    if (toke != G_TOKEN_INT)
	return FALSE;

    g_scanner_get_next_token(scn);

    *var = g_scanner_cur_value(scn).v_int;
    return TRUE;
}

static gboolean scan_eye(GScanner *scn, struct theme *pos)
{
    struct eyeBall *ball = (struct eyeBall*)g_malloc(sizeof(struct eyeBall));

    ball->ins = NULL;

    if (!get_int(scn, &(ball->norm.x)) || !get_int(scn, &(ball->norm.y))
	    || !get_int(scn, &(ball->socket.w))
	    || !get_int(scn, &(ball->socket.h))) {
	g_free(ball);
	return FALSE;
    }

    ball->ball.w = ball->ball.h = -1;

    if (get_int(scn, &(ball->ball.w)) && (!get_int(scn, &(ball->ball.h))))
	    ball->ball.h = ball->ball.w;

    if (g_scanner_peek_next_token(scn) == TOKEN_EYEPIX) {
	g_scanner_get_next_token(scn);
	if (g_scanner_get_next_token(scn) != G_TOKEN_STRING) {
	    g_free(ball);
	    return FALSE;
	}

	ball->pfile = image_file(g_scanner_cur_value(scn).v_string, pos->dir);
    } else {
	ball->pfile = NULL;

	if (ball->ball.w == -1) {
	    fprintf(stderr, "You must specify either a size of pixmap"\
		    " for each eye!\n");
	    g_free(ball);
	    return FALSE;
	}
    }
    pos->balls = g_slist_append(pos->balls, ball);

    return TRUE;
}

static struct clock_settings *load_clock(GScanner *scn)
{
    struct clock_settings *c;
    c = (struct clock_settings*)g_malloc(sizeof(struct clock_settings));

    if (!get_int(scn, &(c->pos.x))) return NULL;
    if (!get_int(scn, &(c->pos.y))) return NULL;
    if (!get_int(scn, &(c->dim.w))) return NULL;
    if (!get_int(scn, &(c->dim.h))) return NULL;

    if (g_scanner_peek_next_token(scn) == TOKEN_SMOOTH) {
	c->smooth = TRUE;
	g_scanner_get_next_token(scn);
    } else {
	c->smooth = FALSE;
    }

    c->hands[0].colour.red = 0;
    c->hands[0].colour.blue = 0;
    c->hands[0].colour.green = 0;

    c->hands[1].colour.red = 0;
    c->hands[1].colour.green = 0;
    c->hands[1].colour.blue = 0xff;

    c->hands[2].colour.red = 0xff;
    c->hands[2].colour.green = 0;
    c->hands[2].colour.blue = 0;

    c->hands[0].width[0] = 0.05;
    c->hands[0].width[1] = 0.0;
    c->hands[0].length = 0.5;

    c->hands[1].width[0] = 0.02;
    c->hands[1].width[1] = 0.0;
    c->hands[1].length = 0.75;

    c->hands[2].width[0] = 0.0;
    c->hands[2].width[1] = 0.0;
    c->hands[2].length = 1.00;

    return c;
}

static gboolean load_hand(GScanner *scn, struct clock_settings *clock, int idx)
{
    double mav;
    int k, w[2], l;

    struct hand_settings *hand = &(clock->hands[idx]);

    if (!get_int(scn, &(w[0]))) return FALSE;
    if (!get_int(scn, &(w[1]))) return FALSE;
    if (!get_int(scn, &(l))) return FALSE;

    mav = (clock->dim.w + clock->dim.h)/2.0;

    hand->width[0] = (double)(w[0])/mav;
    hand->width[1] = (double)(w[1])/mav;
    hand->length = (double)(l*2)/(double)(clock->dim.w);

    if (!get_int(scn, &k)) return FALSE;
    hand->colour.red = k;
    if (!get_int(scn, &k)) return FALSE;
    hand->colour.green = k;
    if (!get_int(scn, &k)) return FALSE;
    hand->colour.blue = k;

    return TRUE;
}

static gint parse(GScanner *scn, char **subdir)
{
    struct theme *pos= NULL;
    gint token, k;

    token = g_scanner_peek_next_token(scn);

    if (token == G_TOKEN_EOF) {
	return PARSE_DONE;
    }

    if (token == G_TOKEN_ERROR)
	return PARSE_ERROR;

    if (token == TOKEN_DEFAULT) {
	g_scanner_get_next_token(scn);
	if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
	    return PARSE_ERROR;
#ifdef HAVE_GNOME
	if (!is_gnome_applet)
#endif
	{
	dname = g_strdup(g_scanner_cur_value(scn).v_string);
	}
	return PARSE_OK;
    }

    if (token == TOKEN_ADEFAULT) {
	g_scanner_get_next_token(scn);
	if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
	    return PARSE_ERROR;
#ifdef HAVE_GNOME
	if (is_gnome_applet)
#endif
	{
	dname = g_strdup(g_scanner_cur_value(scn).v_string);
	}
	return PARSE_OK;
    }

    if (token == TOKEN_DLAUNCH) {
	g_scanner_get_next_token(scn);
	if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
	    return PARSE_ERROR;
	defaultLaunch = g_strdup(g_scanner_cur_value(scn).v_string);
	return PARSE_OK;
    }

    if (token == TOKEN_SUBDIR) {
	g_scanner_get_next_token(scn);
	if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
	    return PARSE_ERROR;
	*subdir = g_strdup(g_scanner_cur_value(scn).v_string);
	token = g_scanner_peek_next_token(scn);
    }

    if (token != TOKEN_NEW)
	return PARSE_ERROR;

    g_scanner_get_next_token(scn);

    pos = (struct theme*)g_malloc(sizeof(struct theme));

    pos->clock = NULL;
    pos->colour = NULL;
    pos->balls = NULL;
    pos->say = NULL;
    pos->launch = NULL;
    pos->flags = 0;
    pos->width = pos->height = -1;
    pos->mask = NULL;
    pos->dir = *subdir; /* nb: no strdup, so never free subdir info! */

    token = g_scanner_get_next_token(scn);
    if (token != G_TOKEN_STRING)
	return PARSE_ERROR;

    pos->title = g_strdup(g_scanner_cur_value(scn).v_string);

    token = g_scanner_get_next_token(scn);
    if (token != G_TOKEN_STRING)
	return PARSE_ERROR;

    pos->filename = image_file(g_scanner_cur_value(scn).v_string, pos->dir);

    if (pos->filename != NULL) {
	pos->flags |= T_FLAG_EXISTS;
    } else {
	g_warning("file '%s' doesn't exist", g_scanner_cur_value(scn).v_string);
    }

    while ((token = g_scanner_get_next_token(scn)) != TOKEN_END) {
	switch(token) {
	    case G_TOKEN_EOF:
	    	g_warning("premature end of file!");
	    	return PARSE_ERROR;
	    case G_TOKEN_ERROR:
		return PARSE_ERROR;
	    case TOKEN_MASK:
		if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
		    return PARSE_ERROR;
		pos->mask = image_file(g_scanner_cur_value(scn).v_string,
			pos->dir);
		break;
	    case TOKEN_SAY:
		if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
		    return PARSE_ERROR;
		pos->say = g_strdup(g_scanner_cur_value(scn).v_string);
		break;
	    case TOKEN_LAUNCH:
		if (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
		    return PARSE_ERROR;
		pos->launch = g_strdup(g_scanner_cur_value(scn).v_string);
		break;
	    case TOKEN_FHORIZ:
		pos->flags |= T_FLAG_FLIP_HOR;
		break;
	    case TOKEN_FVERT:
		pos->flags |= T_FLAG_FLIP_VER;
		break;
	    case TOKEN_EYE:
		if (!scan_eye(scn, pos)) return PARSE_ERROR;
		break;
	    case TOKEN_COLOUR:
		if (pos->colour != NULL) g_free(pos->colour);
		pos->colour = (GdkColor*)g_malloc(sizeof(GdkColor));

		if (!get_int(scn, &k)) return PARSE_ERROR;
		pos->colour->red = k;
		if (!get_int(scn, &k)) return PARSE_ERROR;
		pos->colour->green = k;
		if (!get_int(scn, &k)) return PARSE_ERROR;
		pos->colour->blue = k;
		break;
	    case TOKEN_WIDTH:
		if (!get_int(scn, &(pos->width))) return PARSE_ERROR;
		break;
	    case TOKEN_HEIGHT:
		if (!get_int(scn, &(pos->height))) return PARSE_ERROR;
		break;
	    case TOKEN_CLOCK:
		if (!(pos->clock = load_clock(scn))) return PARSE_ERROR;
		break;
	    case TOKEN_HHAND:
		if (!load_hand(scn, pos->clock,0)) return PARSE_ERROR;
		break;
	    case TOKEN_MHAND:
		if (!load_hand(scn, pos->clock,1)) return PARSE_ERROR;
		break;
	    case TOKEN_SHAND:
		if (!load_hand(scn, pos->clock,2)) return PARSE_ERROR;
		break;
	    default:
		g_warning("parse - got a strange token\n");
		return PARSE_ERROR;
	}
    }
    pos->ins_flags = pos->flags;
    theme_list = g_slist_append(theme_list, pos);
    return PARSE_OK;
}

static void scan(char *fname)
{
    GScanner *scn;
    int fd, i;
    int retval;
    char *subdir = NULL;

    scn = g_scanner_new(&scanner_config);

    if ((fd = open(fname, O_RDONLY)) < 0) {
	g_warning("couldn't open %s", fname);
	return;
    }

    g_scanner_input_file (scn, fd);

    for (i = 0; i < nsymbols; i++)
	g_scanner_add_symbol(scn, symbols[i].name,
		GINT_TO_POINTER(symbols[i].token));

    while ((retval = parse(scn, &subdir)) != PARSE_DONE) {
	if (retval == PARSE_ERROR) {
	    g_warning("parse error in '%s': line '%d'", fname, scn->line);
	    exit(1);
	}
    }
    g_scanner_destroy(scn);
    close(fd);
}

static void scan_config_dir(const char *basedir)
{
    char *gpat;
    int i;

    glob_t mrglobby; /* sorry */

    gpat = alloca(strlen(basedir)+22);

    sprintf(gpat, "%s/Config/*", basedir);

    glob(gpat, GLOB_TILDE | GLOB_MARK, NULL, &mrglobby);

    if (mrglobby.gl_pathc == 0) {
	globfree(&mrglobby);
	return;
    }

    /* hack to put default stuff at the front */
    sprintf(gpat, "%s/Config/penguineyesrc", basedir);

    if (FILE_EXISTS(gpat))
	scan(gpat);

    for (i = 0; i < mrglobby.gl_pathc; i++) {
	if (strcmp(gpat, mrglobby.gl_pathv[i]) != 0)
	    scan(mrglobby.gl_pathv[i]);
    }
    globfree(&mrglobby);
}

void getrc(void)
{
    GSList *t;
    char userdir[256];

    scan_config_dir(DATADIR);

    sprintf(userdir, "%s/.penguineyes", getenv("HOME"));

    if (closedir(opendir(userdir)) == -1) {
	fprintf(stderr, "creating directory '%s'\n", userdir);
	mkdir(userdir, 0755);
    }
    sprintf(userdir, "%s/.penguineyes/Config", getenv("HOME"));

    if (closedir(opendir(userdir)) == -1) {
	fprintf(stderr, "creating directory '%s'\n", userdir);
	mkdir(userdir, 0755);
    }

    sprintf(userdir, "%s/.penguineyes", getenv("HOME"));

    scan_config_dir(userdir);

    if (theme_list == NULL) {
	g_error("No themes found at all!");
	exit(1);
    }

    if (dname == NULL)
	dname = g_strdup("default");

    default_theme = NULL;
    for (t = theme_list; t != NULL; t = g_slist_next(t)) {
	if (strcmp(((struct theme*)(t->data))->title, dname) == 0) {
	    default_theme = (struct theme*)(t->data);
	    break;
	}
    }

    if (default_theme == NULL) {// arbitrary choice
	default_theme = (struct theme*)(theme_list->data);
	fprintf(stderr, "couldn't find default theme, using \"%s\" instead\n",
		default_theme->title);
    }

    g_free(dname);
    dname = NULL;
}

void write_defaults()
{
    FILE *def;

    if (!GTK_WIDGET_VISIBLE(mainWindow)) {
#ifdef HAVE_GNOME
	if (!is_gnome_applet)
#endif
	{
	    g_warning("penguineyes: window not visible, not writing defaults\n");
	    return;
	}
    }

    if ((def = fopen(default_config_file, "w")) == NULL) {
	g_warning("penguineyes: Couldn't write defaults file '%s'\n",
		default_config_file);
	return;
    }

    fprintf(def, "# This file is auto-generated, alter at your own risk\n\n");
    fprintf(def, "theme \"%s\"\n", default_theme->title);
#ifdef HAVE_GNOME
    if (!is_gnome_applet)
#endif
    {
	int x = 0, y = 0;
	gdk_window_get_position(mainWindow->window, &x, &y);

	fprintf(def, "geometry %d %d %d %d\n", x, y,
		screen_width, screen_height);
    }
    fprintf(def, "rate %d\n", fps);

    fprintf(def, "flip %d %d\n",
	    default_theme->ins_flags&T_FLAG_FLIP_HOR ? 1 : 0,
	    default_theme->ins_flags&T_FLAG_FLIP_VER ? 1 : 0);

    if (guilty)
	fputs("guilty\n", def);

    fclose(def);
}


static gboolean read_default_file(GScanner *scn, int fd, struct defaults *defa)
{
    int i;

    for (i = 0; i < nsymbols; i++)
	g_scanner_add_symbol(scn, symbols[i].name,
		GINT_TO_POINTER(symbols[i].token));

    if ((g_scanner_get_next_token(scn) != TOKEN_THEME)
	    || (g_scanner_get_next_token(scn) != G_TOKEN_STRING)
	    || (!(defa->theme
		    = get_theme_by_name(g_scanner_cur_value(scn).v_string)))
	    || (g_scanner_get_next_token(scn) != TOKEN_GEO)

	    || (!get_int(scn, &(defa->pos.x)))
	    || (!get_int(scn, &(defa->pos.y)))

	    || (!get_int(scn, &(defa->dim.w)))
	    || (!get_int(scn, &(defa->dim.h)))

	    || (g_scanner_get_next_token(scn) != TOKEN_RATE))
	return FALSE;


    if (fps < 0 && !get_int(scn, &i)) return FALSE;
    defa->fps = i;

    if (g_scanner_get_next_token(scn) != TOKEN_FLIP) return FALSE;
    if (!get_int(scn, &i)) return FALSE;
    defa->fhor = ((i == 1) ? TRUE : FALSE);
    if (!get_int(scn, &i)) return FALSE;
    defa->fver = ((i == 1) ? TRUE : FALSE);

    if (g_scanner_get_next_token(scn) == TOKEN_GUILTY)
	guilty = TRUE;

    return TRUE;
}

struct defaults read_defaults()
{
    GScanner *scn = NULL;
    int fd;

    struct defaults defa = {
	NULL, { -1, -1 }, { -1, -1 }, 24, FALSE, FALSE
    };

    if ((fd = open(default_config_file, O_RDONLY)) > 0) {
	scn = g_scanner_new(&scanner_config);

	g_scanner_input_file (scn, fd);

	read_default_file(scn, fd, &defa);

	g_scanner_destroy(scn);
	close(fd);
    }

    return defa;
}
