/*
 *  Penguineyes, a linuxified version of Xeyes
 *  Copyright (C) 1998/1999/2000 Neil Howie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#include <getopt.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk_imlib.h>

#include "penguineyes.h"

/* global variables */
GdkPixmap *sourcePenguin = NULL;
GtkWidget *mainWindow = NULL;
GdkGC *penguineyes_gc = NULL;
GdkGC *mask_gc = NULL;
gchar fps = -1;
gboolean guilty = FALSE;

gchar *defaultLaunch = "gnome-terminal";

#ifdef HAVE_GNOME
gboolean is_gnome_applet = FALSE;
GtkWidget *appletWidget = NULL;
GtkWidget *widgetButton = NULL;
#endif

#ifndef HAVE_GNOME
#include "gnome-geometry.c" /* link in the GNOME geometry commands */
#endif

/* initial geometry settings */
static struct { int x; int y; int w; int h; } init_geo = { -1, -1, -1, -1 };

static GtkWidget *main_menu;

static gboolean use_random_theme = FALSE;

static void draw_ball(struct eyeBall *ball);

static int spawn_number = 0;

static void penguineyes_quit()
{
    write_defaults();
    gtk_main_quit();
    exit(0);
}

#ifdef HAVE_GNOME
static gint penguin_save_session(GnomeClient *client, gint phase,
	GnomeSaveStyle save_style, gint is_shutdown,
	GnomeInteractStyle interact_style, gint is_fast, gpointer client_data)
{
    char *args[12];
    char fps_string[4];
    char *file = gnome_client_get_config_prefix(client);
    int n_args;

    /*
    if (is_gnome_applet)
	return TRUE;
	*/

    gnome_config_push_prefix(file);

    if (fps <= 0 || fps > 60) { /* fps_string is only 4, so check */
	g_warning("fps gave a strange number!\n");
	fps = 24;
    }
	
    sprintf(fps_string, "%d", fps);

    args[0] = g_strdup(gnome_master_client()->restart_command[0]);
    args[1] = "--config";
    args[2] = g_strdup(default_config_file);
    args[3] = "--theme";
    args[4] = default_theme->title;
    args[5] = "--fps";
    args[6] = fps_string;

    if (!is_gnome_applet) {
	args[7] = "--geometry";
	args[8] = gnome_geometry_string(mainWindow->window);

	n_args = 9;
    } else {
	n_args = 7;
    }

    if (default_theme->ins_flags & T_FLAG_FLIP_HOR)
	args[n_args++] = "--fliph";

    if (default_theme->ins_flags & T_FLAG_FLIP_VER)
	args[n_args++] = "--flipv";

    if (guilty)
	args[n_args++] = "--guilty";

    args[n_args] = NULL;

    gnome_client_set_restart_command(client, n_args, args);

    gnome_config_sync();

    gnome_config_pop_prefix();
    g_free(file);
    g_free(args[0]);
    g_free(args[2]);

    write_defaults();

    return TRUE;
}

static gint appletLaunch(GtkWidget *w, GdkEventButton *event)
{
    gchar *launch = default_theme->launch ? : defaultLaunch;

    if (fork()) return FALSE;

    execl("bash", "bash", "-c", launch, NULL);
    execl("/bin/sh", "sh", "-c", launch, NULL);
    g_warning("exec failed");
    _exit(0);
}

#endif

static struct coord mouse;

/* functions for dragging window */
static gint button_press(GtkWidget *w, GdkEventButton *event)
{
    mouse.x = (int)event->x;
    mouse.y = (int)event->y;

    if (event->button == 3) {
	gtk_menu_popup(GTK_MENU(main_menu), NULL, NULL, NULL, NULL,
		event->button, event->time);

    } else if (event->button == 4 || event->button == 5) {
	/* mouse wheel - sweeeeet! */
	int change = (event->button == 4) ? -8 : 8;

	gdk_window_resize(mainWindow->window,
		screen_width + (screen_width/change),
		screen_height + (screen_height/change));
    }


    return FALSE;
}

static gint resize_event(GtkWidget *w, GdkEventConfigure *event)
{
    if (event->width != screen_width || event->height != screen_height)
	penguin_set_size(event->width, event->height);

    return TRUE;
}

static gint motion_notify(GtkWidget *w, GdkEventMotion *event)
{
    struct coord m;
    int wx, wy;

    m.x = (int)event->x;
    m.y = (int)event->y;


    if (event->state & GDK_BUTTON1_MASK) {
	gdk_window_get_position(mainWindow->window, &wx, &wy);
	wx += (m.x - mouse.x);
	wy += (m.y - mouse.y);

      /* move it - hack, never have coords less than 0 */
	wx = MAX(wx, 0);
	wy = MAX(wy, 0);
	gdk_window_move(mainWindow->window, wx, wy);

    } else if (event->state & GDK_BUTTON2_MASK) {
	wx = screen_width;
	wy = screen_height;

	if (event->state & GDK_SHIFT_MASK) { /* resize, but keep ratio */
	    int dx = m.x - mouse.x;
	    int dy = m.y - mouse.y;

	    if (ABS(dx) > ABS(dy))
		dy = (dx*wy)/wx;
	    else
		dx = (dy*wx)/wy;

	    wx += dx;
	    wy += dy;
	} else {
	    wx += m.x - mouse.x;
	    wy += m.y - mouse.y;
	}

	if (wx <= 0 || wy <= 0)
	    return TRUE;
	
        mouse.x = m.x;
        mouse.y = m.y;

	gdk_window_resize(mainWindow->window, wx, wy);
    }
    return TRUE;
}

static void draw_ball(struct eyeBall *ball)
{
    if (ball->ins->pixmap) {
	gdk_gc_set_clip_origin(ball->ins->gc, ball->ins->screen.x,
		ball->ins->screen.y);
	gdk_gc_set_clip_mask(ball->ins->gc, ball->ins->mask);
	gdk_draw_pixmap(mainWindow->window, ball->ins->gc, ball->ins->pixmap,
		0, 0, ball->ins->screen.x, ball->ins->screen.y,
		ball->ins->rball.w, ball->ins->rball.h);

    } else {
	gdk_draw_arc(mainWindow->window, penguineyes_gc, TRUE,
		ball->ins->screen.x, ball->ins->screen.y, ball->ins->rball.w,
		ball->ins->rball.h, 0, 360*64);
    }

    if (mask_gc) {
	gdk_draw_pixmap(mainWindow->window, mask_gc, sourcePenguin,
		ball->ins->screen.x, ball->ins->screen.y,
		ball->ins->screen.x, ball->ins->screen.y,
		ball->ins->rball.w, ball->ins->rball.h);
    }
}

static void draw_eye(struct eyeBall *ball, struct coord *m)
{
    struct coord b, dif, d, t;
    float angle;

    dif.x = ball->ins->pos.x + (ball->ins->rsocket.w)/2 - m->x;
    dif.y = ball->ins->pos.y + (ball->ins->rsocket.h)/2 - m->y;

    d.x = MIN(ABS(dif.x), (ball->ins->rsocket.w - (ball->ins->rball.w)/2));
    d.y = MIN(ABS(dif.y), (ball->ins->rsocket.h - (ball->ins->rball.h)/2));

    angle = atan2((float)dif.y,(float)dif.x);

    if (!guilty) {
	b.x = ball->ins->pos.x + ((ball->ins->rsocket.w)
		- (gint)(d.x*cos(angle)))/2;
	b.y = ball->ins->pos.y + ((ball->ins->rsocket.h)
		- (gint)(d.y*sin(angle)))/2;
    } else {
	b.x = ball->ins->pos.x + ((ball->ins->rsocket.w)
		+ (gint)(d.x*cos(angle)))/2;
	b.y = ball->ins->pos.y + ((ball->ins->rsocket.h)
		+ (gint)(d.y*sin(angle)))/2;
    }



    t.x = b.x - (ball->ins->rball.w/2);
    t.y = b.y - (ball->ins->rball.h/2);

    if (ball->ins->screen.x == t.x && ball->ins->screen.y == t.y)
	return;

  /* cover up the old eyeball by whatever was underneath it */
    gdk_draw_pixmap(mainWindow->window, penguineyes_gc, sourcePenguin,
	    ball->ins->screen.x, ball->ins->screen.y, ball->ins->screen.x,
	    ball->ins->screen.y, ball->ins->rball.w, ball->ins->rball.h);

    ball->ins->screen.x = t.x;
    ball->ins->screen.y = t.y;

    draw_ball(ball);
}

gint draw_eyes(void)
{
    struct coord m;

    gdk_window_get_pointer(mainWindow->window, &(m.x), &(m.y), NULL);

    g_slist_foreach(themeBalls, (GFunc)draw_eye, (gpointer)&m);
    
    return TRUE;
}

static gint eyes_callback(void) {
    /* small hack, speed up resizing, etc by not 
     * doing this when we're trying to do other things
     */
    if (!gdk_events_pending()) {
	draw_eyes();

	if (default_theme->clock) {
	    update_clock(FALSE);
	    draw_clock();
	}
    }

    return TRUE;
}


static struct theme *get_random_theme(void)
{
    struct theme *nu;
    int l = g_slist_length(theme_list);
    int n;

    if (l <= 0) {
	g_warning("No themes to select!\n");
	return NULL;
    }

    if (default_theme) {
	int p = g_slist_index(theme_list, default_theme);

	n = (p + (rand()%(l-2)) + 1)%l;

    } else {
	n = rand()%l;
    }

    if (!(nu = (struct theme*)g_slist_nth_data(theme_list, n)))
	g_warning("random theme set NULL!!\n");

    return nu;
}

static void set_random_theme(void)
{
    set_theme_auto(get_random_theme());
}

static void reset_theme_size(void)
{
    gdk_window_resize(mainWindow->window,
	    default_theme->width, default_theme->height);
}

static void flip_hor(void)
{
    g_return_if_fail(default_theme);

    default_theme->ins_flags ^= T_FLAG_FLIP_HOR;
    set_theme_size(default_theme, screen_width, screen_height);
}

static void flip_ver(void)
{
    g_return_if_fail(default_theme);

    default_theme->ins_flags ^= T_FLAG_FLIP_VER;
    set_theme_size(default_theme, screen_width, screen_height);
}

static void flip_guilty(void)
{
    guilty = !guilty; /* not guilty, m'lud */
}

static void theme_menu_add(struct theme *t, GtkWidget *submenu)
{
    GtkWidget *entry;
    gchar *mstr;

    if (!(t->flags & T_FLAG_EXISTS)) return;

    mstr = g_strdup_printf((t == default_theme) ? "*%s":" %s", t->title);

    entry = gtk_menu_item_new_with_label(mstr);
    g_free(mstr);

    gtk_menu_append(GTK_MENU(submenu), entry);

    gtk_signal_connect_object(GTK_OBJECT(entry), "activate",
	    GTK_SIGNAL_FUNC(set_theme_auto), (gpointer)t);
    gtk_widget_show(entry);
}

static void add_separator(GtkWidget *menu)
{
    GtkWidget *entry = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(menu), entry);
    gtk_widget_show(entry);
}

static GtkAccelGroup *accel_grp = NULL;

static void add_menu_item(GtkWidget *menu, const gchar *name,
	GtkSignalFunc func, guint accel_key, guint accel_mods)
{
    GtkWidget *entry = gtk_menu_item_new_with_label(name);

#ifdef HAVE_GNOME
    if (!is_gnome_applet)
#endif
    gtk_widget_add_accelerator(entry, "activate", accel_grp, accel_key,
	    accel_mods, GTK_ACCEL_VISIBLE);

    gtk_signal_connect_object(GTK_OBJECT(entry), "activate", func, NULL);
    gtk_menu_append(GTK_MENU(menu), entry);
    gtk_widget_show(entry);
}

void create_right_click_menu(void)
{
    GtkWidget *submenu;
    GtkWidget *entry;

    if (main_menu)
	gtk_widget_destroy(main_menu);

#ifdef HAVE_GNOME
    if (!is_gnome_applet)
#endif
    {
    accel_grp = gtk_accel_group_new();
    gtk_window_add_accel_group(GTK_WINDOW(mainWindow), accel_grp);
    }
    main_menu = gtk_menu_new();

    entry = gtk_menu_item_new_with_label(PACKAGE" v"VERSION);
    gtk_menu_append(GTK_MENU(main_menu), entry);
    gtk_widget_show(entry);

    add_separator(main_menu);

    submenu = gtk_menu_new();
    g_slist_foreach(theme_list, (GFunc)theme_menu_add, submenu);

    entry = gtk_menu_item_new_with_label("Themes");
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(entry), submenu);

    gtk_menu_append(GTK_MENU(main_menu), entry);
    gtk_widget_show(entry);

    add_separator(submenu);
    add_menu_item(submenu, " Random", GTK_SIGNAL_FUNC(set_random_theme),
	    GDK_space, 0);
    add_separator(main_menu);
    add_menu_item(main_menu, "Flip horizontal", GTK_SIGNAL_FUNC(flip_hor),
	    'h', 0);
    add_menu_item(main_menu, "Flip vertical", GTK_SIGNAL_FUNC(flip_ver),
	    'v', 0);

    add_menu_item(main_menu, "Flip guilty/paranoid", GTK_SIGNAL_FUNC(flip_guilty),
	    'g', 0);

    add_menu_item(main_menu, "Reset size", GTK_SIGNAL_FUNC(reset_theme_size),
	    'r', 0);
    add_separator(main_menu);

    add_menu_item(main_menu, "Save", GTK_SIGNAL_FUNC(write_defaults), 's', 0);
    add_menu_item(main_menu, "Quit", GTK_SIGNAL_FUNC(penguineyes_quit),
	    'q', 0);
}

void draw_area(int x, int y, int w, int h)
{
    GSList *l;

    gdk_draw_pixmap(mainWindow->window, penguineyes_gc, sourcePenguin,
	    x, y, x, y, w, h);

    for (l = themeBalls; l; l = g_slist_next(l)) {
	struct eyeBall *ball = (struct eyeBall*)(l->data);
	int bh = ball->ins->rball.h;
	int bw = ball->ins->rball.w;

	if (!((x+w < ball->ins->screen.x || x > ball->ins->screen.x+bw)
		&& (y+h < ball->ins->screen.y || y+h > ball->ins->screen.y+bh)))
	    draw_ball(ball);
    }
}


static gint expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
    int x, y, w, h;
    struct clock_settings *c;

    x = ev->area.x;
    y = ev->area.y;
    w = ev->area.width;
    h = ev->area.height;

    if ((c = default_theme->clock) != NULL) {
	if ((x > (c->rpos.x+c->rdim.w)) || (x + w < (c->rpos.x))
		|| (y > (c->rpos.y+c->rdim.h)) || (y + h < (c->rpos.y))) {
	    c = NULL;
	} else {
	    int x2, y2, x3, y3, xl, yl;

	    x2 = x+w;
	    y2 = y+h;
	    x3 = c->rpos.x + c->rdim.w;
	    y3 = c->rpos.y + c->rdim.h;

	    x = MIN(x, c->rpos.x);
	    y = MIN(y, c->rpos.x);
	    xl = MAX(x2, x3);
	    yl = MAX(y2, y3);
	    w = xl - x;
	    h = yl - y;

	    if (w <= 0 || h <= 0) {
		g_warning("expose_event: invalid w or h!");
		w = h = 1;
	    }
	}
    }

    draw_area(x, y, w, h);

    if (c != NULL)
	draw_clock();

    return FALSE;
}

static char *default_theme_name = NULL;

struct theme *get_theme_by_name(gchar *name)
{
    GSList *tmp;
    int ln;

    if (name == NULL)
	return NULL;

    ln = strlen(name);

    for (tmp = theme_list ;tmp; tmp = g_slist_next(tmp)) {
	struct theme *t = (struct theme*)tmp->data;
	if (strncasecmp(t->title, name, ln) == 0)
	    return t;
    }
    return NULL;
}

#ifdef HAVE_GNOME
static void cb_properties()
{
    gtk_menu_popup(GTK_MENU(main_menu), NULL, NULL, NULL, NULL, 0, 0);
}

#endif

static gboolean init_flip_v = FALSE;
static gboolean init_flip_h = FALSE;

static void penguinInit(void)
{
    gboolean foundtheme = FALSE;
    struct defaults got_def;

    getrc();

    got_def = read_defaults();

    if (use_random_theme) {
	default_theme = get_random_theme();
	foundtheme = TRUE;

	if (init_geo.h == -1)
	    init_geo.h = default_theme->height;
	if (init_geo.w == -1)
	    init_geo.w = default_theme->width;
    }

    if (default_theme_name != NULL) {
	struct theme *tmp = get_theme_by_name(default_theme_name);
	if (tmp) {
	    default_theme = tmp;
	    foundtheme = TRUE;
	}
	g_free(default_theme_name);
	default_theme_name = NULL;
    }

    if (!foundtheme && got_def.theme != NULL)
	default_theme = got_def.theme;

    if (!foundtheme && init_geo.w == -1 && init_geo.h == -1) {
	init_geo.w = got_def.dim.w;
	init_geo.h = got_def.dim.h;
    }

    if (init_geo.x == -1 && init_geo.y == -1 
	    && got_def.pos.x != -1 && got_def.pos.y != -1) {
	init_geo.x = got_def.pos.x;
	init_geo.y = got_def.pos.y;
    }

    signal(SIGTERM, penguineyes_quit);
    signal(SIGINT, penguineyes_quit);

    gtk_widget_push_visual(gdk_imlib_get_visual());
    gtk_widget_push_colormap(gdk_imlib_get_colormap());

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	appletWidget = applet_widget_new("penguineyes");
	mainWindow = gtk_drawing_area_new();

	widgetButton = gtk_button_new();

	gtk_container_add(GTK_CONTAINER(widgetButton), mainWindow);

	applet_widget_add(APPLET_WIDGET(appletWidget), widgetButton);

	gtk_widget_show(widgetButton);

	gtk_container_set_border_width(GTK_CONTAINER(widgetButton), 0);
	gtk_container_set_border_width(GTK_CONTAINER(appletWidget), 0);

	gtk_widget_set_events(mainWindow, GDK_EXPOSURE_MASK);
    } else
#endif
    {
	mainWindow = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_widget_set_events(mainWindow,
	    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK
	    | GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);
    }

    gtk_widget_set_app_paintable(mainWindow, TRUE);

    gtk_widget_realize(mainWindow);


    if (init_flip_h || got_def.fhor)
	default_theme->ins_flags |= T_FLAG_FLIP_HOR;
    else
	default_theme->ins_flags &= ~T_FLAG_FLIP_HOR;

    if (init_flip_v || got_def.fver)
	default_theme->ins_flags |= T_FLAG_FLIP_VER;
    else
	default_theme->ins_flags &= ~T_FLAG_FLIP_VER;

    penguineyes_gc = gdk_gc_new(mainWindow->window);
    
    set_theme_size(default_theme, init_geo.w, init_geo.h);

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	applet_widget_register_stock_callback(APPLET_WIDGET(appletWidget),
		"properties", GNOME_STOCK_MENU_PROP, _("Properties..."),
		cb_properties, NULL);
    }
#endif

    gdk_window_set_decorations(mainWindow->window, 0);
    gdk_window_set_functions(mainWindow->window, 0);
    gdk_window_set_back_pixmap(mainWindow->window, NULL, FALSE);

#ifdef HAVE_GNOME
    gnome_win_hints_set_layer(mainWindow, WIN_LAYER_ONTOP);
    gnome_win_hints_set_state(mainWindow, WIN_STATE_STICKY);
#endif

    if (init_geo.x != -1 && init_geo.y != -1) {
#ifdef HAVE_GNOME
	if (!is_gnome_applet)
#endif
	gtk_widget_set_uposition(mainWindow, init_geo.x, init_geo.y);
    }

    gtk_widget_show(mainWindow);

    gdk_window_raise(mainWindow->window);

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	gtk_signal_connect (GTK_OBJECT(widgetButton), "button_press_event",
	    GTK_SIGNAL_FUNC(appletLaunch), NULL);
    } else
#endif
    {
	gtk_signal_connect (GTK_OBJECT(mainWindow), "motion_notify_event",
		GTK_SIGNAL_FUNC(motion_notify), NULL);

	gtk_signal_connect (GTK_OBJECT(mainWindow), "button_press_event",
		GTK_SIGNAL_FUNC(button_press), NULL);
    }

    gtk_signal_connect (GTK_OBJECT(mainWindow), "delete_event",
	    GTK_SIGNAL_FUNC(penguineyes_quit), NULL);

    gtk_signal_connect (GTK_OBJECT(mainWindow), "destroy",
	    GTK_SIGNAL_FUNC(penguineyes_quit), NULL);

    gtk_signal_connect (GTK_OBJECT(mainWindow), "expose_event",
	    GTK_SIGNAL_FUNC(expose_event), NULL);

    gtk_signal_connect (GTK_OBJECT(mainWindow), "configure_event",
	    GTK_SIGNAL_FUNC(resize_event), NULL);

#ifdef HAVE_GNOME
    {
	GnomeClient *client = gnome_master_client();

	gtk_signal_connect(GTK_OBJECT(client), "save_yourself",
		GTK_SIGNAL_FUNC(penguin_save_session), NULL);

	gtk_signal_connect(GTK_OBJECT(client), "die",
		GTK_SIGNAL_FUNC(penguineyes_quit), NULL);

    }
#endif

    if (fps < 0)
	fps = 24;

    gtk_timeout_add(1000/fps, (GtkFunction)eyes_callback, NULL);


#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	gtk_widget_show(appletWidget);
    }
#endif

    gtk_widget_hide(mainWindow);
    gtk_widget_show(mainWindow);
    gdk_window_resize(mainWindow->window, screen_width, screen_height);
}

enum {
    KEY_A  = -1,
    KEY_W  = -2,
    KEY_H  = -3,
    KEY_F  = -4,
    KEY_T  = -5,
    KEY_R  = -6,
    KEY_G  = -7,
    KEY_FH = -8,
    KEY_FV = -9,
    KEY_C  = -10,
    KEY_S  = -11,
    KEY_GU = -12
};

static void parse_arg(char option, const char *arg)
{
    int tmp;

    switch (option) {
	case '?':
	    puts(PACKAGE " version " VERSION "\n");
	    puts("options:\n"
		"\t-c  --config <file>\t- use alternative config file\n"
		"\t-w  --width  <width>\t- set width to specified size\n"
		"\t-h  --height <height>\t- set height to specified size\n"
		"\t-t  --theme  \"<theme>\"\t- use specified theme\n"
		"\t-f  --fps    <rate>\t- set redraw rate\n"
		"\t-r  --random\t\t- set a random theme\n"
		"\t-V  --flipv\t\t- flip the window vertically\n"
		"\t-H  --fliph\t\t- flip the window horizontally\n"
		"\t-g  --guilty\t\t- start with the eyes looking away from the cursor\n"
		"\t-s  --spawn <number>\t- spawn n windows\n"
#ifdef HAVE_GNOME
		"\t-a  --applet\t\t- create GNOME applet\n"
#endif
		"\t-v  --version\t\t- print version\n"
		"\t-?  --help\t\t- this!\n\n");
	    exit(0);
	    break;
	case 'v': /* print version, then exit */
	    puts(PACKAGE " version " VERSION "\n");
	    exit(0);
	    break;
	case 'c': case KEY_C: /* choose config file */
	    default_config_file = g_strdup(arg);
	    break;
	case 'a': case KEY_A: /* run as an applet, if under GNOME */
#ifdef HAVE_GNOME
	    is_gnome_applet = TRUE;
#else
	    fprintf(stderr, "This version of penguineyes has been "\
		    "compiled without GNOME support\n");
#endif
	    break;
	case 'f': case KEY_F: /* set the frames per second */
	    sscanf(arg, "%d", &tmp);
	    if (tmp < 5) {
		fprintf(stderr, "fps too low, setting to 5\n");
		fps = 2;
	    } else if (tmp > 60) {
		fprintf(stderr, "fps too low, setting to 60\n");
		fps = 60;
	    } else
		fps = tmp;
	    break;
	case 'w': case KEY_W: /* set the width */
	    sscanf(arg, "%d", &tmp);
	    if (tmp <= 0) tmp = -1;

	    init_geo.w = tmp;
	    break;
	case 'h': case KEY_H: /* set the height */
	    sscanf(arg, "%d", &tmp);
	    if (tmp <= 0) tmp = -1;

	    init_geo.h = tmp;
	    break;
	case 't': case KEY_T: /* set the starting theme */
	    default_theme_name = g_strdup(arg);
	    break;
	case 'r': case KEY_R: /* set to use a random theme */
	    use_random_theme = TRUE;
	    break;
	case 'V': case KEY_FV: /* flip vertically */
	    init_flip_v = TRUE;
	    break;
	case 'H': case KEY_FH: /* flip horiz. */
	    init_flip_h = TRUE;
	    break;
	case 's': case KEY_S: /* 'spawn' more penguineyes */
	    sscanf(arg, "%d", &tmp);
	    if (tmp < 1) {
		g_warning("invalid spawn number!");
		break;
	    }
	    spawn_number = tmp;
	    break;
	case 'g': case KEY_G: /* geometry string */
	    if (!gnome_parse_geometry(arg, &(init_geo.x), &(init_geo.y),
			&(init_geo.w), &(init_geo.h))) {
		g_warning("penguineyes: 'geometry' string was not "\
			"parsed correctly\n");
		init_geo.x = init_geo.y = init_geo.w = init_geo.h = -1;
	    }
	    break;
	case 'u': case KEY_GU: /* guilty */
	    guilty = TRUE;
	    break;
	default:
	    g_warning("getopt returned a strange value: %c\n", option);
	    break;
    }
}


#ifndef HAVE_GNOME
static struct option long_options[] = {
    { "help",	 0, 0, '?' },
    { "applet",	 0, 0, 'a' },
    { "version", 0, 0, 'v' },
    { "random",	 0, 0, 'r' },
    { "geometry",1, 0, 'g' },
    { "width",	 1, 0, 'w' },
    { "height",	 1, 0, 'h' },
    { "fps",	 1, 0, 'f' },
    { "theme",	 1, 0, 't' },
    { "flipv",   0, 0, 'V' },
    { "fliph",   0, 0, 'H' },
    { "spawn",   1, 0, 's' },
    { "config",  1, 0, 'c' },
    { "guilty",  0, 0, 'u' },
    { 0,	 0, 0, 0   }
};
#else
static void gnome_parse_arg(
	poptContext state, enum poptCallbackReason reason,
	const struct poptOption *opt, const char *arg, void *data)
{
    parse_arg(opt->val, arg);
}

static struct poptOption cb_options[] = {
    { NULL, '\0', POPT_ARG_CALLBACK, gnome_parse_arg, 0},
    { "applet", 'a', POPT_ARG_NONE, NULL, KEY_A,
	N_("Run as a GNOME applet"), NULL},
    { "random", 'r', POPT_ARG_NONE, NULL, KEY_R,
	N_("Set random theme"), NULL},
    { "width", 'w', POPT_ARG_STRING, NULL, KEY_W,
	N_("set width"), N_("WIDTH")},
    { "height", 'h', POPT_ARG_STRING, NULL, KEY_H,
	N_("set height"), N_("HEIGHT")},
    { "fps", 'f', POPT_ARG_STRING, NULL, KEY_F,
	N_("set redraw rate"), N_("FPS")},
    { "theme", 't', POPT_ARG_STRING, NULL, KEY_T,
	N_("set theme"), N_("THEME")},
    { "geometry", 'g', POPT_ARG_STRING, NULL, KEY_G,
	N_("set window geometry"), N_("GEOMETRY")},
    { "fliph", 'H', POPT_ARG_NONE, NULL, KEY_FH,
	N_("flip horizontally"), NULL},
    { "flipv", 'V', POPT_ARG_NONE, NULL, KEY_FV,
	N_("flip vertically"), NULL},
    { "spawn", 's', POPT_ARG_STRING, NULL, KEY_S,
	N_("Spawn NUMBER windows"), N_("NUMBER") },
    { "config", 'c', POPT_ARG_STRING, NULL, KEY_C,
	N_("set default config file"), N_("FILE") },
    { "guilty", 'u', POPT_ARG_NONE, NULL, KEY_GU,
	N_("set guilty option"), NULL },
    { NULL, '\0', 0, NULL, 0}
};
#endif


#ifndef HAVE_GNOME
static void parse_args(int argc, char *argv[])
{
    int c;

    for ( ; ; ) {
	int index = 0;

	c = getopt_long(argc, argv, "?auvrcVHg:f:w:h:t:s:", long_options, &index);

	if (c < 0)
	    break;

	parse_arg(c, optarg);
    }

    if (optind < argc) {
	fprintf(stderr, "Unknown argument '%s'\n", argv[optind]);
	exit(1);
    }
}
#else
static void parse_args(int argc, char *argv[])
{
    int i;

    for (i = 0; i < argc;i++)
	if (strcmp(argv[i], "-a") == 0 || strcmp(argv[i], "--applet") == 0)
	    is_gnome_applet = TRUE;

    if (is_gnome_applet) {
	applet_widget_init(PACKAGE, VERSION, argc, argv,
		cb_options, 0, NULL);
    } else {
	gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv,
		cb_options, 0, NULL);
    }

}
#endif

/* this is _such_ a horrible hack, but I don't feel like rewritting
 * the code to allow >1 window open at once
 */
static void do_spawn(char *argv0)
{
    char blah[256], blah2[32];
    char *argvc[8];
    int i, x, y;

    if (spawn_number < 1) return;

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	g_warning("You can't spawn applets...");
	return;
    }
#endif

    gdk_window_get_root_origin(mainWindow->window, &x, &y);

    argvc[0] = g_strdup(argv0);
    argvc[1] = "--config";
    argvc[5] = "--theme";
    argvc[6] = g_strdup(default_theme->title);
    argvc[7] = NULL;

    for (i = 1; i < spawn_number; i++) {
	sprintf(blah, "%s-%d", default_config_file, i);
	argvc[2] = blah;

	if (!FILE_EXISTS(blah)) {
	    argvc[3] = "--geometry";
	    sprintf(blah2, "%dx%d+%d+%d", screen_width, screen_height,
		    abs(x-(i*(screen_width/2))), abs(y-(i*(screen_height/8))));
	    argvc[4] = blah2;
	} else {
	    argvc[3] = NULL;
	}

	if (!fork()) {
	    execvp(argvc[0], argvc);

	    g_warning("Exec failed!\n");
	    exit(1);
	}
    }
    g_free(argvc[0]);
    g_free(argvc[6]);
}

static void setup_config_file(void)
{
    if (default_config_file) return;

    /* hmmmm ... hack! */
    default_config_file = g_strdup_printf("%s/.penguineyes/%sdefaults",
	    getenv("HOME"),
#ifdef HAVE_GNOME
	    is_gnome_applet ? "applet_" :
#endif
	    "");
}

/* set up imlib so it's nice and quick, I'll leave dithering to what's the
 * default, since non-dithering looks crap, and doesn't slow _that_ much
 */
static void my_init_imlib(void)
{
    GdkImlibInitParams p;

    p.flags = PARAMS_HIQUALITY|PARAMS_FASTRENDER;
    p.hiquality = 0;
    p.fastrender = 1;
    
    gdk_imlib_init_params(&p);
}

int main(int argc, char *argv[])
{
    srand(getpid());
    gtk_init(&argc, &argv);
    my_init_imlib();

    gdk_imlib_init();

    parse_args(argc, argv);

    setup_config_file();

    penguinInit();

    do_spawn(argv[0]);

#ifdef HAVE_GNOME
    if (is_gnome_applet)
	applet_widget_gtk_main();
    else
#endif
	gtk_main();

    return 0;
}
