#!bin/env python
if __name__ == '__main__':
    import sys, petsc4py
    petsc4py.init(sys.argv)
    del sys, petsc4py

from petsc4py import PETSc as PETSc


print """
Given the problem

        -u''(x) = f(x)  on  (0,1)
           u(x) = 0     at  x=0, x=1

If f(x)=1, the exat solution is

            u(x)= 1/2*x*(1-x)

We solve it by the finite difference method
using PETSc libraries under Python.
"""

opts = PETSc.Options()

# Problem Size
# ------------
N = opts.getInt('N', 100)  # grid points, boundaries not included
h = 1.0/(N+1)              # grid spacing


# Matrix
# ---------------
A = PETSc.Mat()
A.createSeqAIJ((N,N),nz=3)
A.setValue(0, 0,  2, PETSc.InsertMode.INSERT)
A.setValue(0, 1, -1, PETSc.InsertMode.INSERT)
for i in xrange(1,N-1):
    A.setValue(i, i,    2, PETSc.InsertMode.INSERT)
    A.setValue(i, i-1, -1, PETSc.InsertMode.INSERT)
    A.setValue(i, i+1, -1, PETSc.InsertMode.INSERT)
A.setValue(N-1, N-2, -1, PETSc.InsertMode.INSERT)
A.setValue(N-1, N-1,  2, PETSc.InsertMode.INSERT)
A.assemble()
A.scale(1/h)


# Vectors
# -------
def f(x):
    return 1

x, b = A.getVecs()

for i in xrange(N):
    b.setValue(i, f(i*h), PETSc.InsertMode.INSERT)
b.scale(h)


# KSP and PC
# ----------
ksp = PETSc.KSP()
ksp.create(PETSc.COMM_SELF)
# ksp.setType('cg')
# kps.getPC().setType('jacobi')
ksp.setOperators(A,A,'same_nz')
ksp.setFromOptions()

# Solve
# -----
ksp.solve(b, x)


# Error respect to exact solution
# -------------------------------
def u(x):
    return 1.0/2*x*(1-x)

e = x.duplicate()
for i in xrange(N):
    e.setValue(i, u(i*h), PETSc.InsertMode.INSERT)
e.AXPY(-1, x)

print "Error: %f" % e.norm()


# View solution and error
# -------------
if opts.getTruth('draw_sol', False):
    viewer_x = PETSc.ViewerDraw(title='solution')
    viewer_x.view(x)

if opts.getTruth('draw_err', False):
    viewer_e = PETSc.ViewerDraw(title='error')
    viewer_e.view(e)

secs = opts.getInt('sleep', 0)
if secs:
    PETSc.Sys.sleep(secs)
