# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

"""All PIL related issues."""

#License: latest GPL
import datetime, os, re, time, types
from cStringIO import StringIO
from urllib2 import urlopen

import Image
try:
    from ExifTags import TAGS as EXIFTAGS
except:
    #older versions of PIL
    EXIFTAGS    = {}
    
try:
    import pyexiv2
    import _pyexiv2 as exif
except:
    pyexiv2     = None
    exif        = False

try:
    from unicoding import ensure_unicode
    from core.ct import TITLE
    from core.translation import _t
    from formField import is_www_file
except:
    _t = str
    is_www_file = os.path.isfile
    TITLE       = ''
    def ensure_unicode(x): return x

IMAGE_DEFAULT_DPI       = 72
DOT         = '.' #should be same as in core.translations
MONTHS      = (_t('january'),_t('february'),_t('march'),_t('april'),_t('may'),
                _t('june'),_t('july'),_t('august'),_t('september'),
                _t('october'),_t('november'),_t('december'),)
WEEKDAYS    = (_t('monday'),_t('tuesday'),_t('wednesday'),_t('thursday'),
                _t('friday'),_t('saturday'),_t('sunday'))
DATETIME_KEYS   = ['year','month','day','hour','minute','second']
re_DATETIME = re.compile(
                '(?P<year>\d{4})[-:](?P<month>\d{2})[-:](?P<day>\d{2}) '
                '(?P<hour>\d{2}):(?P<minute>\d{2}):(?P<second>\d{2})')
                
re_TAG      = re.compile('(Pil|Exif|Iptc|Pexif|Zexif)([.]\w+)+')
re_KEY = re.compile('(#*)((\w|[.])*)')
WWW_CACHE = {}

def image_open(x):
    if is_www_file(x):
        try:
            return WWW_CACHE[x]
        except KeyError:
            f   = urlopen(x)
            im  = WWW_CACHE[x] = Image.open(StringIO(f.read()))
            return im
    return Image.open(x)

def extract_info_dpi(info,image):
    #info
    image_info              = image.info
    if image_info.has_key('dpi'):
        #also in this case it is possible that dpi is 0
        dpi                 = image_info['dpi']
        if len(dpi) > 1:
            dpi             = dpi[0]
        info[_t('dpi')]  = dpi
    else: dpi               = 0
    if dpi == 0:
        if image_info.has_key('aspect'):
            info[_t('dpi')]  = dpi = image_info['aspect'][0]
        else:
            dpi             = 0
        if dpi == 0:
            info[_t('dpi')]  = IMAGE_DEFAULT_DPI
        
def split_data(new):
    """new - dictionary"""
    value   = new.values()[0]
    #tuples or list
    if type(value) in (types.ListType,types.TupleType):
        if len(value) > 1:
            for k, v in new.items():
                for i, x in enumerate(v):
                    new['%s%s%d'%(k,DOT,i)]  = v[i]
        return
    #datetime strings
    done    = False
    for k, v in new.items():
        if type(v) in types.StringTypes:
            dt      = re_DATETIME.match(v)
            if dt:
                for key in DATETIME_KEYS:
                    new['%s%s%s'%(k,DOT,key)]  = dt.group(key)
                    done    = True
    if done: return 
    #date time values
    if type(value) == datetime.datetime:
        for k, v in new.items():
            for key in DATETIME_KEYS:
                new['%s%s%s'%(k,DOT,key)]  = getattr(v,key)
        
def extract_info_pexif(info,image):
    """pexif = Pil EXIF"""
    #check if exif information is available through pil
    if not hasattr(image,'_getexif'):
        return info
    #add exif information
    try:
        exif    = image._getexif()
    except:
        exif    = None
    if not exif: return
    for key, value in exif.items():
        new    = {'Zexif%s0x%04x'%(DOT,key):value}
        if EXIFTAGS.has_key(key):
            new['Pexif%s%s'%(DOT,EXIFTAGS[key])] = value
        #Check if we can split exif data further
        split_data(new)
        info.update(new)

def extract_info_pyexiv2(info,location=''):
    if not (pyexiv2 and os.path.isfile(location)):
        return
    #official patch of pyexiv2
    try:
        if type(location) is unicode:
            location = location.encode('utf-8')
    except UnicodeEncodeError:
        return
    try:
        image   = pyexiv2.Image(location)
        image.readMetadata()
    except:
        return
    #exif & iptc
    for keys in [image.exifKeys(),image.iptcKeys()]:
        for key in keys:
            try:
                new = {key:image[key]}
                split_data(new)
                info.update(new)
            except:
                pass

def extract_info_location(info,location='',folder=None):
    if not location.strip():
        return
    location                        = ensure_unicode(location)
    dirname                         = os.path.dirname(location)
    if folder:
        info[_t('folder')]          = folder
    else:
        info[_t('folder')]          = dirname
    info[_t('root')],info[_t('foldername')] = os.path.split(info[_t('folder')])
    info[_t('subfolder')]           = dirname[len(info[_t('folder')])+1:]
    info[_t('path')]                = location
    filename                        = os.path.basename(location)
    info[_t('filename')], typ       = os.path.splitext(filename)
    info[_t('type')]                = typ[1:]
    #os time
    try:
        file_stat                  = os.stat(location)
        info[_t('filesize')]       = file_stat.st_size
        st_mtime                   = time.localtime(file_stat.st_mtime)[:7]
    except:
        info[_t('filesize')]       = 0
        st_mtime                   = (0,)*7
    info[_t('year')],info[_t('month')],info[_t('day')],info[_t('hour')],\
        info[_t('minute')],info[_t('second')],info[_t('weekday')] =\
        st_mtime
    info[_t('weekdayname')]        = WEEKDAYS[info[_t('weekday')]]
    info[_t('monthname')]          = MONTHS[info[_t('month')]-1]

def extract_info_image(info,image):
    extract_info_dpi(info,image)
    #retrieve properties
    width, height           = size = image.size
    #summarize all photo properties in info
    info.update({
        _t('width')                     : width,
        _t('height')                    : height,
        _t('mode')                      : image.mode,
        'Pil'+DOT+'Size'                : size,
        'Pil'+DOT+'Format'              : image.format,
        'Pil'+DOT+'FormatDescription'   : image.format_description,
    })
    return info

def extract_info_pil(image):
    return prefix_dict(extract_info_image({},image),'new'+DOT)
    
def extract_info(info, image, location='',image_index=0,folder=None,
        metadata=True):
    info['#']   = image_index
    extract_info_location(info,location,folder)
    extract_info_image(info,image)
    if metadata:
        extract_info_pexif(info,image)
        extract_info_pyexiv2(info,location)
    return info

def prefix_dict(d,prefix,exclude=[]):
    exclude.append(prefix)
    for key, value in d.items():
        add = True
        for e in exclude:
            if key[:len(e)] == e:
                add = False
                break
        if add:
            d[prefix+key] = value
    return d

class TestInfo:
    def __init__(self,info):
        self._keys   = info.keys()
        
    def __getitem__(self,x):
        key_match   = re_KEY.match(x)
        if key_match:
            key     = key_match.group(2)
            if not key: key = '#'
        else:
            key     = x
        if self._is_valid_key(key):
            return '2'
        raise KeyError, x
    
    def _is_valid_key(self,x):
        return x in self._keys or re_TAG.match(x)
    
class Info:
    def __init__(self,dictionary={'#':0}):
        self._dictionary    = dictionary
        self.__setitem__    = self._dictionary.__setitem__
        self.keys           = self._dictionary.keys
        self.items          = self._dictionary.items
        self.update         = self._dictionary.update
        self.values         = self._dictionary.values
        
    def __getitem__(self,x):
        match               = re_KEY.match(x)
        if match:
            number          = match.group(1)
            if number:
                key         = match.group(2)
                if not key: key = '#'
                format      = '%%0%dd'%len(number)
                return format%self._dictionary[key]
        return self._dictionary[x]
    
    def copy(self):
        return Info(self._dictionary.copy())
    
def create_test_info():
    global TEST_INFO
    im      = Image.new('L',(1,1))
    info    = extract_info({},im,'/test/unknown.png')
    TEST_INFO   = TestInfo(info)
    return TEST_INFO
       
def report_invalid_image(f,valid,invalid,folder):
    try:
        im  = image_open(f)
        im.verify()
        valid.append((folder,f))
    except:
        invalid.append(f)
            
class Photo:
    def __init__(self,filename,image_index=0,save_metadata = False,folder=None):
        #validate image
        name = self.current_layer_name  = _t('background')
        layer                           = Layer(image_open(filename))
        self.layers                     = {name:layer}
        self.init_info(layer,filename,image_index,save_metadata,folder)
        
    def get_filename(self,folder,filename,typ):
        self.info[_t('new')+DOT+_t('width')], \
            self.info[_t('new')+DOT+_t('height')] = \
            self.image.size
        return os.path.join(folder,
            '%s.%s'%(filename,typ)).replace('<','%(').replace('>', ')s')%\
            self.__dict__
            
    def init_info(self,layer,location,image_index,save_metadata=False,
            folder=None,prefix=_t('new')+DOT):
        if save_metadata and exif:
            self._exif_source   = location
        else:
            self._exif_source   = None
        self.info_original  = Info()
        extract_info(self.info_original,layer.image,location,image_index,folder)
        self.info           = prefix_dict(self.info_original,prefix)
        
    def get_info(self):
        """Use method for read-only"""
        return self.info.copy()
            
    def get_info_original(self):
        """Use method for read-only"""
        return self.info_original.copy()
            
    #---layers
    def get_layer(self,name=None):
        if name is None:
            name   = self.current_layer_name
        return self.layers[name]
    
    def set_layer(self,layer,name=None):
        if name is None:
            name   = self.current_layer_name
        self.layers[name]   = layer
        
    def as_image(self):
        return self.get_layer().image
        
    #---image operations affecting all layers
    def save(self,filename,*args,**keyw):
        """Saves a flattened image"""
        #todo: flatten layers
        layer   = self.get_layer()
        layer.image.save(filename,*args,**keyw)
        #copy metadata if needed
        self.copy_metadata(filename,software=TITLE)
        #update info
        if hasattr(keyw,'dpi'):
            self.set_attr(_t('dpi'),keyw['dpi'][0])
        extract_info(self.info,layer.image,filename,folder=self.info['folder'],
            metadata=False)
        
    def convert(self,mode,*args,**keyw):
        """Converts all layers to a different mode."""
        for layer in self.layers.values():
            layer.image = layer.image.convert(mode,*args,**keyw)
        self.set_attr(_t('mode'),mode)
            
    def resize(self,size,method):
        """Resizes all layers to a different size"""
        size    = (max(1,size[0]),max(1,size[1]))
        for layer in self.layers.values():
            layer.image = layer.image.resize(size,method)
        self.set_size(size)
            
    def set_size(self,size):
        self.set_attr(_t('width'),size[0])
        self.set_attr(_t('height'),size[1])
        self.set_attr('Pil'+DOT+_t('Size'),size)
        
    def set_attr(self,attr,value):
        self.info[_t('new')+DOT+attr] = value
        
    def update_size(self):
        self.set_size(self.layers.values()[0].image.size)
        return self
    #---pil
    def apply_pil(self,function,*arg,**keyw):
        self.get_layer().apply_pil(function,*arg,**keyw)
        return self
    
    def apply_all_pil(self,function,*arg,**keyw):
        for layer in self.layers.values():
            layer.apply_pil(function,*arg,**keyw)
        return self
    
    #---exif
    def copy_metadata(self,filename,software=None):
        if self._exif_source:
            try:
                exif.copy_metadata(self._exif_source,filename,software)
            except:
                pass

class Layer:
    def __init__(self,image,position=(0,0)):
        self.image      = image
        self.position   = position
        
    def apply_pil(self,function,*arg,**keyw):
        self.image  = function(self.image,*arg,**keyw)
        
if __name__ == '__main__':
    create_test_info()