/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Mirco Müller <macslow@bangang.de>
 */

/**
 * SECTION:pgmtext
 * @short_description: A text drawable.
 * @see_also: #PgmDrawable, #PgmImage.
 *
 * <refsect2>
 * <para>
 * #PgmText is a drawable displaying a text with support for multiple lines,
 * markups and several properties. The specified text is then arranged to fit
 * inside the drawable size. Fonts are specified through face name and
 * attributes. The font height is given in canvas coordinates.
 * </para>
 * </refsect2>
 *
 * FIXME:
 * • Describe the markup syntax.
 * • Describe the different capabilities.
 *
 * Last reviewed on 2007-05-18 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "pgmtext.h"

GST_DEBUG_CATEGORY_STATIC (pgm_text_debug);
#define GST_CAT_DEFAULT pgm_text_debug

static PgmDrawableClass *parent_class = NULL;

/* GObject stuff */

G_DEFINE_TYPE (PgmText, pgm_text, PGM_TYPE_DRAWABLE);

static void
pgm_text_dispose (GObject *object)
{
  PgmText * text = PGM_TEXT (object);

  GST_OBJECT_LOCK (text);

  g_free (text->label);
  g_free (text->font_family);

  GST_OBJECT_UNLOCK (text);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_text_class_init (PgmTextClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_text_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_text_debug, "pgm_text", 0, "text object");
}

static void
pgm_text_init (PgmText* text)
{
  /* Labels */
  text->label = g_strdup ("");

  /* Font properties */
  text->font_family = g_strdup ("Sans");
  text->gravity = PGM_TEXT_GRAVITY_AUTO;
  text->stretch = PGM_TEXT_STRETCH_NORMAL;
  text->style = PGM_TEXT_STYLE_NORMAL;
  text->variant = PGM_TEXT_VARIANT_NORMAL;
  text->weight = PGM_TEXT_WEIGHT_NORMAL;
  text->height = 1.0f;

  /* Text adjustment */
  text->ellipsize = PGM_TEXT_ELLIPSIZE_NONE;
  text->alignment = PGM_TEXT_ALIGN_LEFT;
  text->wrap = PGM_TEXT_WRAP_WORD;
  text->line_spacing = 0.0f;
  text->justify = FALSE;

  /* Outline parameters */
  text->outline_r = 0;
  text->outline_g = 0;
  text->outline_b = 0;
  text->outline_a = 0;
  text->outline_width = 0.0f;
  text->use_markup = FALSE;
}

/* Public methods */

/**
 * pgm_text_new:
 * @markup: the markup string.
 *
 * Creates a new #PgmText instance.
 *
 * MT safe.
 *
 * Returns: a new #PgmText instance.
 */
PgmDrawable *
pgm_text_new (const gchar *markup)
{
  PgmText *text;

  text = g_object_new (PGM_TYPE_TEXT, NULL);
  GST_DEBUG_OBJECT (text, "created new text");

  pgm_text_set_markup (text, markup);

  return PGM_DRAWABLE (text);
}

/**
 * pgm_text_set_label:
 * @text: a #PgmText object.
 * @label: the label string.
 *
 * Sets label of @text to @label. @label has to be a NULL-terminated string.
 * The default label is the empty one "".
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_label (PgmText *text,
                    const gchar *label)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  GST_DEBUG_OBJECT (text, "set label to '%s'", label);

  if (G_UNLIKELY (text->label))
    g_free (text->label);

  text->label = g_strdup (label);
  text->use_markup = FALSE;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_LABEL);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_markup:
 * @text: a #PgmText object.
 * @markup: the markup string.
 *
 * Sets markup of @text to @markup. @markup has to be a NULL-terminated string.
 * The default label is the empty one "".
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_markup (PgmText *text,
                     const gchar *markup)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  GST_DEBUG_OBJECT (text, "set markup to '%s'", markup);

  if (G_UNLIKELY (text->label))
    g_free (text->label);

  text->label = g_strdup (markup);
  text->use_markup = TRUE;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_MARKUP);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_label:
 * @text: a #PgmText object.
 * @label: a pointer to a pointer to a gchar where the label string will be
 * stored. g_free() after use.
 *
 * Retrieves the current label of @text in @label. If a markup is currently
 * used, the full markup is retrieved.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_label (PgmText *text,
                    gchar **label)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (label != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *label = g_strdup (text->label);

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_font_family:
 * @text: a #PgmText object.
 * @font_family: the font-family string.
 *
 * Sets the font family of @text to @font_family. The default font family
 * name is "Sans".
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_font_family (PgmText *text,
                          const gchar *font_family)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  if (G_UNLIKELY (text->font_family))
    g_free (text->font_family);

  text->font_family = g_strdup (font_family);

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_FONT_FAMILY);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_font_family:
 * @text: a #PgmText object.
 * @font_family: a pointer to a pointer to a gchar where the font family
 * string will be stored. g_free() after use.
 *
 * Retrieves the current font-family of @text in @label.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_font_family (PgmText *text,
                          gchar **font_family)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (font_family != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *font_family = g_strdup (text->font_family);

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_font_height:
 * @text: a #PgmText object.
 * @font_height: the text height in canvas-coordinates.
 *
 * Sets the height of @text to @font_height in proportion to the height of the
 * drawable. The default height is 1.0f, which fills the drawable in height.
 * @font_height will be the maximum height of a line of text (i.e. the height of
 * something like "Čq", measured from the bottom of the 'q' to the top of the
 * accent on the 'Č').
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_font_height (PgmText *text,
                          gfloat font_height)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->height = font_height;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_HEIGHT);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_font_height:
 * @text: a #PgmText object.
 * @font_height: a pointer to a gfloat where the text height is going to be
 * stored.
 *
 * Retrieves the height of @text in @height.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_font_height (PgmText *text,
                          gfloat *font_height)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (font_height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *font_height = text->height;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_ellipsize:
 * @text: a #PgmText object.
 * @ellipsize: the text ellipsizing.
 *
 * Sets ellipsizing of @text to @ellipsize. Only takes effect when @text gets
 * to large to fit the drawable size. The default value is
 * #PGM_TEXT_ELLIPSIZE_NONE.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_ellipsize (PgmText *text,
                        PgmTextEllipsize ellipsize)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->ellipsize = ellipsize;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_ELLIPSIZE);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_ellipsize:
 * @text: a #PgmText object.
 * @ellipsize: a pointer to a #PgmTextEllipsize where the text ellipsizing is
 * going to be stored.
 *
 * Retrieves the ellipsizing of @text in @ellipsize.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_ellipsize (PgmText *text,
                        PgmTextEllipsize *ellipsize)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (ellipsize != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *ellipsize = text->ellipsize;

  GST_OBJECT_UNLOCK (text);

  return  PGM_ERROR_OK;
}

/**
 * pgm_text_set_justify:
 * @text: a #PgmText object.
 * @justify: the text justification state.
 *
 * Enables or not justification of @text. If @justify is #TRUE, text is
 * expanded to fill the whole drawable width.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_justify (PgmText *text,
                      gboolean justify)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->justify = justify;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_JUSTIFY);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_justify:
 * @text: a #PgmText object.
 * @justify: a pointer to a #gboolean where the justification state is
 * going to be stored.
 *
 * Retrieves the justification state of @text in @justify.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_justify (PgmText *text,
                      gboolean *justify)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (justify != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *justify = text->justify;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_alignment:
 * @text: a #PgmText object.
 * @alignment: the text alignment.
 *
 * Sets alignment of @text to @alignment. The default value is
 * #PGM_TEXT_ALIGN_LEFT.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_alignment (PgmText *text,
                        PgmTextAlignment alignment)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->alignment = alignment;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_ALIGNMENT);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_alignment:
 * @text: a #PgmText object.
 * @alignment: a pointer to a #PgmTextAlignment where the text alignment is
 * going to be stored.
 *
 * Retrieves the alignment of @text in @alignment.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_alignment (PgmText *text,
                        PgmTextAlignment *alignment)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (alignment != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *alignment = text->alignment;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_wrap:
 * @text: a #PgmText object.
 * @wrap: the text wrapping.
 *
 * Sets wrapping of @text to @wrap. The default value is #PGM_TEXT_WRAP_WORD.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_wrap (PgmText *text,
                   PgmTextWrap wrap)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->wrap = wrap;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_WRAP);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_wrap:
 * @text: a #PgmText object.
 * @wrap: a pointer to a #PgmTextWrap where the text wrapping is going to be
 * stored.
 *
 * Retrieves the wrapping of @text in @wrap.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_wrap (PgmText *text,
                   PgmTextWrap *wrap)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (wrap != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *wrap = text->wrap;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_gravity
 * @text: a #PgmText object.
 * @gravity: the text gravity.
 *
 * Sets gravity of @text to @gravity. The default value is
 * #PGM_TEXT_GRAVITY_AUTO.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_gravity (PgmText *text,
                      PgmTextGravity gravity)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->gravity = gravity;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_GRAVITY);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_gravity:
 * @text: a #PgmText object.
 * @gravity: a pointer to a #PgmTextGravity where the text gravity is going
 * to be stored.
 *
 * Retrieves the gravity of @text in @gravity.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_gravity (PgmText *text,
                      PgmTextGravity *gravity)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (gravity != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *gravity = text->gravity;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_stretch:
 * @text: a #PgmText object.
 * @stretch: the text stretching.
 *
 * Sets stretching of @text to @stretch. The default value is
 * #PGM_TEXT_STRETCH_NORMAL.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_stretch (PgmText *text,
                      PgmTextStretch stretch)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->stretch = stretch;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_STRETCH);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_stretch:
 * @text: a #PgmText object.
 * @stretch: a pointer to a #PgmTextStretch where the text stretching is going
 * to be stored.
 *
 * Retrieves the stretching of @text in @stretch.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_stretch (PgmText *text,
                      PgmTextStretch *stretch)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (stretch != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *stretch = text->stretch;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_style:
 * @text: a #PgmText object.
 * @style: the text style.
 *
 * Sets style of @text to @style. The default value is #PGM_TEXT_STYLE_NORMAL.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_style (PgmText *text,
                    PgmTextStyle style)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->style = style;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_STYLE);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_style
 * @text: a #PgmText object.
 * @style: a pointer to a #PgmTextStyle where the text style is going to be
 * stored.
 *
 * Retrieves the style of @text in @style.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_style (PgmText *text,
                    PgmTextStyle *style)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (style != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *style = text->style;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_variant:
 * @text: a #PgmText object.
 * @variant: the text variant.
 *
 * Sets variant of @text to @variant. The default value is
 * #PGM_TEXT_VARIANT_NORMAL.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_variant (PgmText *text,
                      PgmTextVariant variant)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->variant = variant;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_VARIANT);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_variant:
 * @text: a #PgmText object.
 * @variant: a pointer to a #PgmTextVariant where the text variant is going to
 * be stored.
 *
 * Retrieves the variant of @text in @variant.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_variant (PgmText *text,
                      PgmTextVariant *variant)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (variant != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *variant = text->variant;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_weight:
 * @text: a #PgmText object.
 * @weight: the text weight.
 *
 * Sets weight of @text to @weight. The default value is
 * #PGM_TEXT_WEIGHT_NORMAL.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_weight (PgmText *text,
                     PgmTextWeight weight)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->weight = weight;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_WEIGHT);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_weight:
 * @text: a #PgmText object.
 * @weight: a pointer to a #PgmTextWeight where the text weight is going to
 * be stored.
 *
 * Retrieves the weight of @text in @weight.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_weight (PgmText *text,
                     PgmTextWeight *weight)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (weight != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *weight = text->weight;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_line_spacing:
 * @text: a #PgmText object.
 * @line_spacing: the line spacing.
 *
 * Sets the space between two lines of @text to @line_spacing in canvas
 * coordinates. The default value is 0.0f.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_line_spacing (PgmText *text,
                           gfloat line_spacing)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->line_spacing = MAX (0.0f, line_spacing);

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_LINE_SPACING);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_line_spacing:
 * @text: a #PgmText object.
 * @line_spacing: a pointer to a #gfloat where the line spacing is going to
 * be stored.
 *
 * Retrieves the space between two lines of @text in @line_spacing.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_line_spacing (PgmText *text,
                           gfloat *line_spacing)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (line_spacing != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *line_spacing = text->line_spacing;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_outline_color:
 * @text: a #PgmText object.
 * @red: the text outline red color component.
 * @green: the text outline green color component.
 * @blue: the text outline blue color component.
 * @alpha: the text outline alpha color component.
 *
 * Sets the outline color of @text to (@red, @green, @blue, @alpha).
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_outline_color (PgmText *text,
                            guchar red,
                            guchar green,
                            guchar blue,
                            guchar alpha)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->outline_r = red;
  text->outline_g = green;
  text->outline_b = blue;
  text->outline_a = alpha;

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_OUTLINE_COLOR);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_outline_color:
 * @text: a #PgmText object.
 * @red: a pointer to a #guchar where the outline red color component is going
 * to be stored.
 * @green: a pointer to a #guchar where the outline green color component is
 * going to be stored.
 * @blue: a pointer to a #guchar where the outline blue color component is going
 * to be stored.
 * @alpha: a pointer to a #guchar where the outline alpha color component is
 * going to be stored.
 *
 * Retrieves the outline color of @text in (@red, @green, @blue, @alpha).
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_outline_color (PgmText *text,
                            guchar *red,
                            guchar *green,
                            guchar *blue,
                            guchar *alpha)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (red != NULL, PGM_ERROR_X);
  g_return_val_if_fail (green != NULL, PGM_ERROR_X);
  g_return_val_if_fail (blue != NULL, PGM_ERROR_X);
  g_return_val_if_fail (alpha != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *red = text->outline_r;
  *green = text->outline_g;
  *blue = text->outline_b;
  *alpha = text->outline_a;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_set_outline_width:
 * @text: a #PgmText object.
 * @width: the text outline width.
 *
 * Sets the outline width (thickness) of @text to @width in canvas coordinates.
 * The default value is 0.0f.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_set_outline_width (PgmText *text,
                            gfloat width)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  text->outline_width = MAX (0.0f, width);

  GST_OBJECT_UNLOCK (text);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (text), PGM_TEXT_OUTLINE_WIDTH);

  return PGM_ERROR_OK;
}

/**
 * pgm_text_get_outline_width:
 * @text: a #PgmText object.
 * @width: a pointer to a #gfloat where the text outline width is going to be
 * stored.
 *
 * Retrieves the outline width (thickness) of @text in @width.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_text_get_outline_width (PgmText *text,
                            gfloat *width)
{
  g_return_val_if_fail (PGM_IS_TEXT (text), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (text);

  *width = text->outline_width;

  GST_OBJECT_UNLOCK (text);

  return PGM_ERROR_OK;
}
