/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_TEXTURE_H__
#define __PGM_TEXTURE_H__

#include <pgm/pgmimage.h>
#include "pgmcontext.h"
#include "pgmgldefs.h"

G_BEGIN_DECLS

/* Texture locking */
#define PGM_TEXTURE_LOCK(texture)   (g_mutex_lock (texture->mutex))
#define PGM_TEXTURE_UNLOCK(texture) (g_mutex_unlock (texture->mutex))

/* Type casting */
#define PGM_TEXTURE(obj) ((PgmTexture*) (obj))

typedef struct _PgmTexture PgmTexture;

/* Function pointer types */
typedef void (*PgmTextureUploadFunc) (PgmTexture*, void*);
typedef void (*PgmTextureCreateFunc) (PgmTexture*);
typedef void (*PgmTextureBindFunc)   (PgmTexture*);
typedef void (*PgmTextureUnbindFunc) (PgmTexture*);

/* Texture flags */
typedef enum {
  PGM_TEXTURE_IDENTITY_MATRIX = (1 << 0)
} PgmTextureFlags;

/* Data storage types */
typedef enum {
  PGM_TEXTURE_CLEAN,
  PGM_TEXTURE_BUFFER,
  PGM_TEXTURE_GST_BUFFER,
  PGM_TEXTURE_PIXBUF,
  PGM_TEXTURE_SYSTEM_BUFFER
} PgmTextureStorageType;

struct _PgmTexture {
  PgmTextureStorageType storage;

  union {
    gpointer   buffer;
    GstBuffer *gstbuffer;
    GdkPixbuf *pixbuf;
    gpointer   systembuffer;
  } data;

  PgmImagePixelFormat csp; /* Color space */
  gint   width;            /* Width of the buffer */
  gint   height;           /* Height of the buffer */
  gint   width_pot;        /* Power-of-two width of the buffer */
  gint   height_pot;       /* Power-of-two height of the buffer */
  gfloat norm_width;       /* width / width_pot */
  gfloat norm_height;      /* height / height_pot */
  gfloat inv_norm_width;   /* (width / width_pot)^-1 */
  gfloat inv_norm_height;  /* (height / height_pot)^-1 */
  gint   stride;           /* Stride of the buffer (not used) */
  gint   size;             /* Size of the buffer in bytes */

  /* State flags */
  guint8 flags;

  /* Parameters */
  PgmMat4x4 *matrix;
  PgmGlEnum  filter;
  PgmGlEnum  wrap;

  /* Number of identifiant */
  gint count;

  /* Identifiant(s) */
  guint *id;

  /* I420 and YV12 planar program values */
  guint cbcr_height;
  guint y_stride, cbcr_stride;
  guint cb_offset, cr_offset;
};

/* create a new texture */
PgmTexture *pgm_texture_new               (PgmContext *context);

/* Free a texture */
void        pgm_texture_free              (PgmTexture *texture);

/* Set a raw buffer */
void        pgm_texture_set_buffer        (PgmTexture *texture,
                                           guchar *buffer,
                                           PgmImagePixelFormat csp,
                                           guint width,
                                           guint height,
                                           guint size,
                                           guint stride);

/* Set a system buffer */
void        pgm_texture_set_system_buffer (PgmTexture *texture,
                                           gconstpointer system_buffer,
                                           PgmImagePixelFormat csp,
                                           guint width,
                                           guint height);

/* Set a GdkPixbuf */
void        pgm_texture_set_pixbuf        (PgmTexture *texture,
                                           GdkPixbuf *pixbuf);

/* Set a GstBuffer */
void        pgm_texture_set_gst_buffer    (PgmTexture *texture,
                                           GstBuffer *gstbuffer,
                                           PgmImagePixelFormat csp,
                                           guint width,
                                           guint height,
                                           guint stride);

/* Update a GstBuffer */
void        pgm_texture_update_gst_buffer (PgmTexture *texture,
                                           GstBuffer *gstbuffer);

/* Bind the texture */
void        pgm_texture_bind              (PgmTexture *texture);

/* Unbind the texture */
void        pgm_texture_unbind            (PgmTexture *texture);

/* Generate the texture */
void        pgm_texture_generate          (PgmTexture *texture);

/* Clean up the texture */
void        pgm_texture_clean             (PgmTexture *texture);

/* Upload the texture */
void        pgm_texture_upload            (PgmTexture *texture);

/* Update the texture */
void        pgm_texture_update            (PgmTexture *texture);

/* Set the texture matrix */
void        pgm_texture_set_matrix        (PgmTexture *texture,
                                           PgmMat4x4 *matrix);

G_END_DECLS

#endif /* __PGM_TEXTURE_H__ */
