/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * Context class is used to create and handle the OpenGL ES context. It retrieves
 * the capabilities of the OpenGL ES implementation at runtime and stores the
 * results in its version, limits and feature mask data.
 *
 * It creates a thread dedicated to the rendering. Only this thread communicates
 * with the native OpenGL libraries for the entire process. It allows the
 * application side to do blocking operations without stopping the rendering.
 *
 * Context handles two task queues for other threads to push operations. The
 * first queue is used to push tasks that need to be complete as soon as
 * possible in the rendering thread, the tasks pushed in this queue are called
 * "immediate tasks". Typical immediate tasks can be for example requests to
 * change the viewport size or title. The second queue is used to push
 * tasks that only need to be complete on a viewport update request. Such tasks
 * are called "deferred tasks". Deferred tasks can be for example texture
 * uploads or projection matrix updates. The immediate task queue is flushed
 * each time a task is pushed with the pgm_gles_context_push_immediate_task()
 * function.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmglescontext.h"

#ifdef G_OS_UNIX
#include <unistd.h> /* pipe, close */
#endif /* G_OS_UNIX */

#ifdef WIN32
#include <windows.h>
#include <fcntl.h>    /* _O_BINARY */
#include <io.h>       /* _pipe, _close */
#define pipe(handle)  _pipe(handle, 4096, _O_BINARY)
#define close(handle) _close(handle)
#endif /* WIN32 */

#include <GLES/gl.h>  /* gl* */
#include <string.h>   /* strlen */
#include <ctype.h>    /* isdigit */
#include <math.h>     /* tan */

#ifndef HAVE_OPENGLES_PC_EMULATION
#include "pgmgleseglbackend.h"
#else
#include "pgmglesxbackend.h"
#endif /* HAVE_OPENGLES_PC_EMULATION */

GST_DEBUG_CATEGORY_EXTERN (pgm_gles_debug);
#define GST_CAT_DEFAULT pgm_gles_debug

/* Function prototypes */
static void flush_task_queue (PgmGlesContext *glescontext, GList **queue);
static void render           (PgmGlesContext *glescontext);

/* Map OpenGL ES extension names and feature bits */
typedef struct {
  const gchar *name;
  gint         feature_mask;
} ExtensionMap;

/* OpenGL ES extensions checked at runtime */
static ExtensionMap gles_extensions_map[] = {
  { "GL_IMG_texture_format_BGRA8888", PGM_GLES_FEAT_TEXTURE_FORMAT_BGRA },
  { NULL,                             0                                 }
};

/* OpenGL ES functions binding */
static PgmGlesContextProcAddress gles_proc_address = {
  /* OpenGL ES 1.1 core functions */
  (pgm_gles_enable)               glEnable,
  (pgm_gles_disable)              glDisable,
  (pgm_gles_get_error)            glGetError,
  (pgm_gles_get_string)           glGetString,
  (pgm_gles_enable_client_state)  glEnableClientState,
  (pgm_gles_disable_client_state) glDisableClientState,
  (pgm_gles_vertex_pointer)       glVertexPointer,
  (pgm_gles_color_pointer)        glColorPointer,
  (pgm_gles_tex_coord_pointer)    glTexCoordPointer,
  (pgm_gles_draw_arrays)          glDrawArrays,
  (pgm_gles_draw_elements)        glDrawElements,
  (pgm_gles_color_4f)             glColor4f,
  (pgm_gles_blend_func)           glBlendFunc,
  (pgm_gles_clear)                glClear,
  (pgm_gles_clear_color)          glClearColor,
  (pgm_gles_matrix_mode)          glMatrixMode,
  (pgm_gles_push_matrix)          glPushMatrix,
  (pgm_gles_pop_matrix)           glPopMatrix,
  (pgm_gles_load_identity)        glLoadIdentity,
  (pgm_gles_load_matrix_f)        glLoadMatrixf,
  (pgm_gles_viewport)             glViewport,
  (pgm_gles_flush)                glFlush,
  (pgm_gles_finish)               glFinish,
  (pgm_gles_frustum_f)            glFrustumf,
  (pgm_gles_ortho_f)              glOrthof,
  (pgm_gles_scale_f)              glScalef,
  (pgm_gles_translate_f)          glTranslatef,
  (pgm_gles_rotate_f)             glRotatef,
  (pgm_gles_hint)                 glHint,
  (pgm_gles_shade_model)          glShadeModel,
  (pgm_gles_read_pixels)          glReadPixels,
  (pgm_gles_gen_textures)         glGenTextures,
  (pgm_gles_delete_textures)      glDeleteTextures,
  (pgm_gles_bind_texture)         glBindTexture,
  (pgm_gles_tex_image_2d)         glTexImage2D,
  (pgm_gles_tex_sub_image_2d)     glTexSubImage2D,
  (pgm_gles_tex_parameter_i)      glTexParameteri,
  (pgm_gles_active_texture)       glActiveTexture,
  (pgm_gles_get_integer_v)        glGetIntegerv,
  (pgm_gles_get_float_v)          glGetFloatv,

  /* OpenGL ES extension functions */
  /* ... */
};

/* Update OpenGL ES viewport and projection */
static void
update_projection (PgmGlesContext *glescontext)
{
  PgmGlesViewport *glesviewport = glescontext->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  PgmGlesContextProcAddress *gles = glescontext->gles;
  gint projected_width, projected_height;
  gint projected_x, projected_y;
  PgmMat4x4 *projection, *transpose;

  /* Retrieve viewport informations with lock */
  GST_OBJECT_LOCK (viewport);
  projected_width = viewport->projected_width;
  projected_height = viewport->projected_height;
  projected_x = viewport->projected_x;
  projected_y = viewport->projected_y;
  projection = viewport->projection;
  GST_OBJECT_UNLOCK (viewport);

  /* Set up viewport */
  gles->viewport (projected_x, projected_y, projected_width, projected_height);

  /* Update projection matrix */
  gles->matrix_mode (PGM_GLES_PROJECTION);
  gles->load_identity ();
  transpose = pgm_mat4x4_transpose (projection);
  gles->load_matrix_f (transpose->m);
  pgm_mat4x4_free (transpose);
  gles->matrix_mode (PGM_GLES_MODELVIEW);

  /* Update all the drawables inside the viewport */
  pgm_gles_viewport_update_drawable_projection (glesviewport);
}

/* Set up initial OpenGL ES states */
static void
init_opengles_states (PgmGlesContext *glescontext)
{
  PgmGlesContextProcAddress *gles = glescontext->gles;

  /* Enable texture mapping */
  gles->enable (PGM_GLES_TEXTURE_2D);

  /* Enable blending */
  gles->blend_func (PGM_GLES_SRC_ALPHA, PGM_GLES_ONE_MINUS_SRC_ALPHA);
  gles->enable (PGM_GLES_BLEND);

  /* Disable depth testing */
  gles->disable (PGM_GLES_DEPTH_TEST);
}

/* Check whether an extension is supported by the OpenGL ES implementation given
 * the extension name and the list of supported extensions */
static gboolean
has_opengl_extension (const gchar *extensions,
                      const gchar *extension_name)
{
  gchar *end;
  size_t ext_name_len = strlen (extension_name);
  gchar *p = (gchar*) extensions;
  size_t size;

  if (!extensions)
    return FALSE;

  end = p + strlen (p);
  while (p < end)
    {
      size = strcspn (p, " ");
      if ((ext_name_len == size)  && (strncmp (extension_name, p, size) == 0))
        return TRUE;
      p += size + 1;
    }

  return FALSE;
}

/* Bind the OpenGL ES extension proc addresses */
static void
bind_opengles_extensions (PgmGlesContext *glescontext)
{
  /* No extension functions to bind yet! */
  return;
}

/* Compute the frame rate per second */
static void
compute_frame_rate (PgmGlesContext *glescontext)
{
  GTimeVal current_time;
  gfloat elapsed_time;
  static guint fps = 0;

  /* Retrieve current time and compute the time elapsed since the last tick */
  g_get_current_time (&current_time);
  elapsed_time = (current_time.tv_sec - glescontext->fps_tick_time.tv_sec)
    + (current_time.tv_usec - glescontext->fps_tick_time.tv_usec) * 0.000001f;

  /* Is the second elapsed? */
  if (elapsed_time >= 1.0f)
    {
      glescontext->fps = fps;

      /* Set the new tick time */
      glescontext->fps_tick_time.tv_sec = current_time.tv_sec;
      glescontext->fps_tick_time.tv_usec = current_time.tv_usec;

      /* Reset the counter */
      fps = 0;
    }

  fps++;
}

/* Viewport update task handler */
static void
do_projection (PgmGlesContext *glescontext,
               gpointer data)
{
  update_projection (glescontext);
  pgm_gles_context_update (glescontext);
}

/* Alpha blending update task handler */
static void
do_alpha_blending (PgmGlesContext *glescontext,
                   gpointer data)
{
  gboolean alpha_blending;

  GST_OBJECT_LOCK (glescontext->glesviewport);
  alpha_blending = PGM_VIEWPORT (glescontext->glesviewport)->alpha_blending;
  GST_OBJECT_UNLOCK (glescontext->glesviewport);

  if (alpha_blending)
    glescontext->gles->enable (PGM_GLES_BLEND);
  else
    glescontext->gles->disable (PGM_GLES_BLEND);

  pgm_gles_context_update (glescontext);
}

/* Visibility update task handler */
static void
do_visibility (PgmGlesContext *glescontext,
               gpointer data)
{
  gboolean visible;

  GST_OBJECT_LOCK (glescontext->glesviewport);
  visible = PGM_VIEWPORT (glescontext->glesviewport)->visible;
  GST_OBJECT_UNLOCK (glescontext->glesviewport);

  pgm_gles_backend_set_visibility (glescontext->backend, visible);
}

/* Frame buffer read task handler */
static void
do_read_pixels (PgmGlesContext *glescontext,
                gpointer data)
{
  PgmGlesViewportPixelRectangle *rectangle =
    (PgmGlesViewportPixelRectangle*) data;
  PgmGlesContextProcAddress *gles = glescontext->gles;
  PgmViewport *viewport = PGM_VIEWPORT (glescontext->glesviewport);
  PgmCanvas *canvas = viewport->canvas;

  /* Flush the queues ensuring last rendering requests will be completed */
  pgm_gles_viewport_flush_update_queue (glescontext->glesviewport);
  flush_task_queue (glescontext, &glescontext->immediate_task);
  flush_task_queue (glescontext, &glescontext->deferred_task);

  /* If there's a canvas we need to invert the projection matrix upside-down.
   * Note that if there's no canvas bound, the pixels area will be black. */
  if (G_LIKELY (canvas))
    {
      gles->matrix_mode (PGM_GLES_PROJECTION);
      gles->push_matrix ();
      gles->scale_f (1.0f, -1.0f, 1.0f);
      gles->translate_f (0.0f, -canvas->height, 0.0f);
      gles->matrix_mode (PGM_GLES_MODELVIEW);
    }

  /* Draw the scene, read it from the back buffer, and clear the back buffer */
  render (glescontext);
  gles->read_pixels (rectangle->x, rectangle->y, rectangle->width,
                     rectangle->height, PGM_GLES_RGBA, PGM_GLES_UNSIGNED_BYTE,
                     rectangle->pixels);
  gles->clear (PGM_GLES_COLOR_BUFFER_BIT);

  /* Pop our inverted projection matrix if needed */
  if (G_LIKELY (canvas))
    {
      gles->matrix_mode (PGM_GLES_PROJECTION);
      gles->pop_matrix ();
      gles->matrix_mode (PGM_GLES_MODELVIEW);
    }

  /* Push the read pixels back to the viewport */
  pgm_viewport_push_pixels (viewport, rectangle->width, rectangle->height,
                            rectangle->pixels);

  /* Then finally free the rectangle structure */
  g_slice_free (PgmGlesViewportPixelRectangle, rectangle);
  rectangle = NULL;
}

/* Texture generation task handler */
static void
do_gen_texture (PgmGlesContext *glescontext,
                gpointer data)
{
  pgm_gles_texture_generate (PGM_GLES_TEXTURE_CAST (data));
}

/* Texture clean up task handler */
static void
do_clean_texture (PgmGlesContext *glescontext,
                  gpointer data)
{
  pgm_gles_texture_clean (PGM_GLES_TEXTURE_CAST (data));
}

/* Texture upload task handler */
static void
do_upload_texture (PgmGlesContext *glescontext,
                   gpointer data)
{
  pgm_gles_texture_upload (PGM_GLES_TEXTURE_CAST (data));
}

/* Texture update task handler */
static void
do_update_texture (PgmGlesContext *glescontext,
                   gpointer data)
{
  pgm_gles_texture_update (PGM_GLES_TEXTURE_CAST (data));
}

/* Texture free task handler */
static void
do_free_texture (PgmGlesContext *glescontext,
                 gpointer data)
{
  pgm_gles_texture_free (PGM_GLES_TEXTURE_CAST (data));
}

/* Traverse the layers and draw the drawables */
static void
render (PgmGlesContext *glescontext)
{
  PgmGlesViewport *glesviewport = glescontext->glesviewport;

  g_mutex_lock (glesviewport->layer_lock);

  g_list_foreach (glesviewport->far_layer,
                  (GFunc) pgm_gles_drawable_draw, NULL);
  g_list_foreach (glesviewport->middle_layer,
                  (GFunc) pgm_gles_drawable_draw, NULL);
  g_list_foreach (glesviewport->near_layer,
                  (GFunc) pgm_gles_drawable_draw, NULL);

  g_mutex_unlock (glesviewport->layer_lock);
}

/* Push a task in a queue removing a similar one if any */
static void
push_task (PgmGlesContext *glescontext,
           GList **queue,
           PgmGlesContextTask *task)
{
  GList *_queue = *queue;
  PgmGlesContextTask *queued_task;

  PGM_GLES_CONTEXT_LOCK (glescontext);

  /* Search and remove a similar task from the queue */
  while (_queue)
    {
      queued_task = PGM_GLES_CONTEXT_TASK (_queue->data);

      if (queued_task->type == task->type
          && queued_task->data == task->data)
        {
          GList *next = _queue->next;

          if (_queue->prev)
            _queue->prev->next = next;
          else
            *queue = next;
          if (next)
            next->prev = _queue->prev;

          pgm_gles_context_task_free (PGM_GLES_CONTEXT_TASK (_queue->data));
          g_list_free_1 (_queue);

          _queue = NULL;
        }
      else
        _queue = _queue->next;
    }

  /* And prepend the task in the queue */
  *queue = g_list_prepend (*queue, task);

  PGM_GLES_CONTEXT_UNLOCK (glescontext);
}

/* Removes all the tasks with the given data from a queue */
static void
remove_tasks_with_data (PgmGlesContext *glescontext,
                        GList **queue,
                        gconstpointer data)
{
  GList *_queue = *queue;

  PGM_GLES_CONTEXT_LOCK (glescontext);

  /* Sequentially search for tasks with the corresponding data and remove
   * them until the end of the queue */
  while (_queue)
    {
      if (PGM_GLES_CONTEXT_TASK (_queue->data)->data != data)
        _queue = _queue->next;
      else
        {
          GList *next = _queue->next;

          if (_queue->prev)
            _queue->prev->next = next;
          else
            *queue = next;
          if (next)
            next->prev = _queue->prev;

          pgm_gles_context_task_free (PGM_GLES_CONTEXT_TASK (_queue->data));
          g_list_free_1 (_queue);

          _queue = next;
        }
    }

  PGM_GLES_CONTEXT_UNLOCK (glescontext);
}

/* Frees a task queue with all its remaining tasks */
static void
free_task_queue (GList **queue)
{
  GList *walk = *queue;

  while (walk)
    {
      pgm_gles_context_task_free ((PgmGlesContextTask *) walk->data);
      walk = walk->next;
    }

  g_list_free (*queue);
  *queue = NULL;
}

/* Flushes a task queue calling the correct handler for each tasks */
static void
flush_task_queue (PgmGlesContext *glescontext,
                  GList **queue)
{
  PgmGlesContextTask *task;
  GList *reversed_queue, *walk;

  PGM_GLES_CONTEXT_LOCK (glescontext);
  reversed_queue = g_list_reverse (*queue);
  *queue = NULL;
  PGM_GLES_CONTEXT_UNLOCK (glescontext);

  walk = reversed_queue;
  while (walk)
    {
      task = PGM_GLES_CONTEXT_TASK (walk->data);
      glescontext->task_func[task->type] (glescontext, task->data);
      pgm_gles_context_task_free (task);
      walk = walk->next;
    }

  g_list_free (reversed_queue);
  reversed_queue = NULL;
}

/* i/o watch callback for immediate tasks */
static gboolean
immediate_io_cb (GIOChannel *source,
                 GIOCondition condition,
                 gpointer data)
{
  PgmGlesContext *glescontext = PGM_GLES_CONTEXT (data);
  gchar buf;

  g_io_channel_read_chars (source, &buf, 1, NULL, NULL);

  /* Complete the immediate task queue */
  flush_task_queue (glescontext, &glescontext->immediate_task);

  return TRUE;
}

/* Update source */
static gboolean
update_cb (gpointer data)
{
  PgmGlesContext *glescontext = PGM_GLES_CONTEXT (data);

  /* User defined update callback */
  pgm_viewport_emit_update_pass (PGM_VIEWPORT (glescontext->glesviewport));

  /* Flush update queue */
  pgm_gles_viewport_flush_update_queue (glescontext->glesviewport);

  /* Flush rendering queues */
  flush_task_queue (glescontext, &glescontext->immediate_task);
  flush_task_queue (glescontext, &glescontext->deferred_task);

  /* Render the scene */
  render (glescontext);
  pgm_gles_backend_swap_buffers (glescontext->backend);
  glescontext->gles->clear (PGM_GLES_COLOR_BUFFER_BIT);

  /* Handle frame rate */
  compute_frame_rate (glescontext);

  return TRUE;
}

/* Update timeout used to remove the update idle after 1 second of inactivity */
static gboolean
update_removal_timeout_cb (gpointer data)
{
  PgmGlesContext *glescontext = (PgmGlesContext*) data;
  GTimeVal current_time;

  g_get_current_time (&current_time);

  g_mutex_lock (glescontext->update_mutex);

  /* If the elapsed time since the previous update request rises above one
   * second, remove the auto-update source */
  if ((current_time.tv_sec - glescontext->update_timestamp.tv_sec) >= 1)
    {
      GSource *source;

      /* Get the source from the rendering GMainContext and remove it */
      source = g_main_context_find_source_by_id (glescontext->render_context,
                                                 glescontext->update_tag);
      if (source)
        g_source_destroy (source);

      glescontext->update_tag = 0;
      glescontext->auto_updated = FALSE;
      glescontext->fps = 0;

      g_mutex_unlock (glescontext->update_mutex);

      GST_DEBUG ("removing update source");

      /* Remove ourself from the loop */
      return FALSE;
    }

  g_mutex_unlock (glescontext->update_mutex);

  return TRUE;
}

/* Create a pipe and its i/o channels */
static void
create_io_channels (gint *fd,
                    GIOChannel **in,
                    GIOChannel **out)
{
  if (pipe (fd) == -1)
    {
      GST_ERROR ("cannot create the pipe");
      return;
    }

  *in = g_io_channel_unix_new (fd[1]);
  if (!*in)
    {
      GST_ERROR ("cannot create the input channel");
      return;
    }

  *out = g_io_channel_unix_new (fd[0]);
  if (!*out)
    {
      GST_ERROR ("cannot create the output channel");
      return;
    }
}

/* Close a pipe and unref its i/o channels */
static void
delete_io_channels (gint *fd,
                    GIOChannel **in,
                    GIOChannel **out)
{
  if (*out)
    {
      g_io_channel_unref (*out);
      *out = NULL;
      close (fd[0]);
    }

  if (*in)
    {
      g_io_channel_unref (*in);
      *in = NULL;
      close (fd[1]);
    }
}

/* Write a byte in the given i/o channel */
static inline void
write_char (GIOChannel *channel)
{
  if (G_LIKELY (channel))
    {
      g_io_channel_write_chars (channel, "#", 1, NULL, NULL);
      g_io_channel_flush (channel, NULL);
    }
}

/* Add an input watch to a specific context */
static guint
add_input_watch (GIOChannel *channel,
                 GIOFunc func,
                 gpointer user_data,
                 GMainContext *context)
{
  GSource *source;
  guint id;

  source = g_io_create_watch (channel, G_IO_IN);
  g_source_set_callback (source, (GSourceFunc) func, user_data, NULL);
  id = g_source_attach (source, context);
  g_source_unref (source);

  return id;
}

/* Rendering thread entry point, initialize and launch the rendering loop.
 * FIXME: Correct error handling */
static gpointer
render_loop (gpointer data)
{
  PgmGlesContext *glescontext = (PgmGlesContext *) data;
  PgmGlesContextProcAddress *gles = NULL;
  const gchar *env_var = NULL;
  gint i = 0;

#ifndef HAVE_OPENGLES_PC_EMULATION
  glescontext->backend = pgm_gles_egl_backend_new (glescontext);
#else
  glescontext->backend = pgm_gles_x_backend_new (glescontext);
#endif /* HAVE_OPENGLES_PC_EMULATION */

  pgm_gles_backend_create_window (glescontext->backend);

  glescontext->gles = gles = &gles_proc_address;

  /* Retrieve renderer informations */
  glescontext->version_string = (const gchar*)
    gles->get_string (PGM_GLES_VERSION);
  glescontext->vendor = (const gchar*) gles->get_string (PGM_GLES_VENDOR);
  glescontext->renderer = (const gchar*) gles->get_string (PGM_GLES_RENDERER);
  glescontext->extensions = (const gchar*)
    gles->get_string (PGM_GLES_EXTENSIONS);

  /* Add some debug info */
  GST_INFO ("OpenGL ES vendor: %s", glescontext->vendor);
  GST_INFO ("OpenGL ES renderer: %s", glescontext->renderer);
  GST_INFO ("OpenGL ES version: %s", glescontext->version_string);
  GST_DEBUG ("OpenGL ES extensions: %s", glescontext->extensions);

  /* Retrieve and bind supported extensions */
  while (gles_extensions_map[i].name)
    {
      if (has_opengl_extension (glescontext->extensions,
                                gles_extensions_map[i].name))
        glescontext->feature_mask |= gles_extensions_map[i].feature_mask;
      i++;
    }

  bind_opengles_extensions (glescontext);

  /* Maximum texture size */
  gles->get_integer_v (PGM_GLES_MAX_TEXTURE_SIZE,
                       &glescontext->max_texture_size);
  GST_INFO ("OpenGL ES max texture size: %d", glescontext->max_texture_size);

  /* Init OpenGL ES */
  init_opengles_states (glescontext);
  gles->clear (PGM_GLES_COLOR_BUFFER_BIT);

  /* Add the immediate source to the rendering thread GMainContext */
  glescontext->immediate_tag = add_input_watch (glescontext->immediate_out,
                                                (GIOFunc) immediate_io_cb,
                                                glescontext,
                                                glescontext->render_context);

  /* Get the framerate requested by the user in the environment */
  env_var = g_getenv ("PGM_GLES_FPS");
  if (env_var)
    {
      gint fps = atoi (env_var);
      if (fps > 0)
        {
          GST_INFO ("Requested FPS: %d", fps);
          glescontext->requested_fps = 1000 / fps;
        }
    }

  /* Signal application thread that the rendering thread is now initialized */
  g_mutex_lock (glescontext->init_mutex);
  glescontext->initialized = TRUE;
  g_cond_signal (glescontext->init_cond);
  g_mutex_unlock (glescontext->init_mutex);

  /* And enter the main loop */
  g_main_loop_run (glescontext->render_loop);

  /* Flush the queues */
  flush_task_queue (glescontext, &glescontext->immediate_task);
  flush_task_queue (glescontext, &glescontext->deferred_task);

  /* Clean up rendering thread related data */
  pgm_gles_backend_destroy_window (glescontext->backend);
  gst_object_unref (glescontext->backend);
  glescontext->backend = NULL;

  return NULL;
}

/* Initialize context creating the rendering thread */
static gboolean
init_gles_context (PgmGlesContext *glescontext)
{
  GError *error = NULL;

  /* Structure access */
  glescontext->mutex = g_mutex_new ();

  /* Context and loop creation */
  glescontext->render_context = g_main_context_new ();
  glescontext->render_loop = g_main_loop_new (glescontext->render_context,
                                              FALSE);

  /* Initialization lock */
  glescontext->init_mutex = g_mutex_new ();
  glescontext->init_cond = g_cond_new ();
  glescontext->initialized = FALSE;

  /* Immediate task source */
  glescontext->immediate_fd[0] = -1;
  glescontext->immediate_fd[1] = -1;
  glescontext->immediate_in = NULL;
  glescontext->immediate_out = NULL;
  glescontext->immediate_tag = 0;
  create_io_channels (glescontext->immediate_fd, &glescontext->immediate_in,
                      &glescontext->immediate_out);

  /* Task queues */
  glescontext->immediate_task = NULL;
  glescontext->deferred_task = NULL;

  /* Auto-update */
  glescontext->update_mutex = g_mutex_new ();
  glescontext->auto_updated = FALSE;
  g_get_current_time (&glescontext->update_timestamp);
  glescontext->update_tag = 0;
  glescontext->requested_fps = 0;

  /* Frame rate */
  g_get_current_time (&glescontext->fps_tick_time);
  glescontext->fps = 0;

  /* Create rendering thread */
  glescontext->render_thread = g_thread_create (render_loop, glescontext,
                                                TRUE, &error);
  if (error != NULL)
    {
      GST_ERROR ("couldn't create rendering thread: %s", error->message);
      return FALSE;
    }

  /* Various tasks function binding */
  glescontext->task_func[PGM_GLES_CONTEXT_PROJECTION] =
    GST_DEBUG_FUNCPTR (do_projection);
  glescontext->task_func[PGM_GLES_CONTEXT_VISIBILITY] =
    GST_DEBUG_FUNCPTR (do_visibility);
  glescontext->task_func[PGM_GLES_CONTEXT_ALPHA_BLENDING] =
    GST_DEBUG_FUNCPTR (do_alpha_blending);
  glescontext->task_func[PGM_GLES_CONTEXT_READ_PIXELS] =
    GST_DEBUG_FUNCPTR (do_read_pixels);

  /* Texture related tasks function binding */
  glescontext->task_func[PGM_GLES_CONTEXT_GEN_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_gen_texture);
  glescontext->task_func[PGM_GLES_CONTEXT_CLEAN_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_clean_texture);
  glescontext->task_func[PGM_GLES_CONTEXT_UPLOAD_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_upload_texture);
  glescontext->task_func[PGM_GLES_CONTEXT_UPDATE_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_update_texture);
  glescontext->task_func[PGM_GLES_CONTEXT_FREE_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_free_texture);

  /* Wait for the rendering thread initialization completion */
  g_mutex_lock (glescontext->init_mutex);
  if (!glescontext->initialized)
    g_cond_wait (glescontext->init_cond, glescontext->init_mutex);
  g_mutex_unlock (glescontext->init_mutex);

  return TRUE;
}

/* Dispose context */
static void
dispose_gles_context (PgmGlesContext *glescontext)
{
  /* Iterate the last events */
  while (g_main_context_pending (glescontext->render_context))
    g_main_context_iteration (glescontext->render_context, FALSE);

  /* Remove the immediate source */
  if (glescontext->immediate_tag)
    g_source_remove (glescontext->immediate_tag);

  /* Remove the auto-update source */
  if (glescontext->update_tag)
    {
      GSource *source = g_main_context_find_source_by_id
        (glescontext->render_context, glescontext->update_tag);
      if (source)
        g_source_destroy (source);
    }

  /* Join the rendering thread */
  g_main_loop_quit (glescontext->render_loop);
  g_main_loop_unref (glescontext->render_loop);
  g_main_context_unref (glescontext->render_context);
  g_thread_join (glescontext->render_thread);

  /* Dispose immediate source data */
  delete_io_channels (glescontext->immediate_fd, &glescontext->immediate_in,
                      &glescontext->immediate_out);

  /* Free the task queues */
  free_task_queue (&glescontext->deferred_task);
  free_task_queue (&glescontext->immediate_task);

  /* Auto-update lock */
  g_mutex_free (glescontext->update_mutex);

  /* Initialization lock */
  g_mutex_free (glescontext->init_mutex);
  g_cond_free (glescontext->init_cond);
  glescontext->initialized = FALSE;

  /* Structure access */
  g_mutex_free (glescontext->mutex);
}

/* Public functions */

PgmGlesContext *
pgm_gles_context_new (PgmGlesViewport *glesviewport)
{
  PgmGlesContext *glescontext;
  gboolean initialized;

  glescontext = g_slice_new0 (PgmGlesContext);

  glescontext->glesviewport = glesviewport;

  initialized = init_gles_context (glescontext);
  if (!initialized)
    {
      g_slice_free (PgmGlesContext, glescontext);
      glescontext = NULL;
    }

  return glescontext;
}

void
pgm_gles_context_free (PgmGlesContext *glescontext)
{
  g_return_if_fail (glescontext != NULL);

  dispose_gles_context (glescontext);
  glescontext = NULL;

  g_slice_free (PgmGlesContext, glescontext);
}

PgmGlesContextTask *
pgm_gles_context_task_new (PgmGlesContextTaskType type,
                      gpointer data)
{
  PgmGlesContextTask *task;

  task = g_slice_new0 (PgmGlesContextTask);
  task->type = type;
  task->data = data;

  return task;
}

void
pgm_gles_context_task_free (PgmGlesContextTask *task)
{
  g_return_if_fail (task != NULL);

  task->data = NULL;

  g_slice_free (PgmGlesContextTask, task);
}

void
pgm_gles_context_update (PgmGlesContext *glescontext)
{
  g_return_if_fail (glescontext != NULL);

  g_mutex_lock (glescontext->update_mutex);

  /* Get a timestamp of the last update request */
  g_get_current_time (&glescontext->update_timestamp);

  /* Add the auto-update if the rendering loop is stopped */
  if (G_UNLIKELY (!glescontext->auto_updated))
    {
      GSource *source;

      /* Add the update source to the rendering GMainContext. If the user has
       * not specified a framerate through the PGM_GLES_FPS environment
       * variable, an idle source blocked on the vsync is attached. Otherwise,
       * a timeout source with the specified framerate is attached. */
      if (G_LIKELY (glescontext->requested_fps == 0))
        {
          GST_DEBUG ("adding update idle source");
          source = g_idle_source_new ();
          g_source_set_priority ((GSource*) source, G_PRIORITY_DEFAULT);
        }
      else
        {
          GST_DEBUG ("adding update timeout source");
          source = g_timeout_source_new (glescontext->requested_fps);
        }
      g_source_set_callback (source, update_cb, glescontext, NULL);
      glescontext->update_tag = g_source_attach (source,
                                                 glescontext->render_context);
      g_source_unref (source);

      /* Add the update removal timeout to the rendering GMainContext */
      source = g_timeout_source_new (500);
      g_source_set_callback (source, update_removal_timeout_cb, glescontext,
                             NULL);
      g_source_attach (source, glescontext->render_context);
      g_source_unref (source);

      /* Mark the render loop as auto-updated */
      glescontext->auto_updated = TRUE;
    }

  g_mutex_unlock (glescontext->update_mutex);
}

void
pgm_gles_context_push_immediate_task (PgmGlesContext *glescontext,
                                      PgmGlesContextTask *task)
{
  g_return_if_fail (glescontext != NULL);

  push_task (glescontext, &glescontext->immediate_task, task);
  write_char (glescontext->immediate_in);
}

void
pgm_gles_context_push_deferred_task (PgmGlesContext *glescontext,
                                     PgmGlesContextTask *task)
{
  g_return_if_fail (glescontext != NULL);

  push_task (glescontext, &glescontext->deferred_task, task);
}

void
pgm_gles_context_remove_tasks_with_data (PgmGlesContext *glescontext,
                                         gconstpointer data)
{
  g_return_if_fail (glescontext != NULL);

  remove_tasks_with_data (glescontext, &glescontext->immediate_task, data);
  remove_tasks_with_data (glescontext, &glescontext->deferred_task, data);
}
