/*
 * File adoption routine 
 *
 */
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <getopt.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mman.h>

#include "pimppa.h"
#include "adopt.h"
#include "news.h"
#include "md5.h"

extern MYSQL *src_db, *dst_db;

struct filedesc
{
	char *file;
	char *desc;
};

char *binsearch(char *filename, struct filedesc *descs, int entries);

/*
 * Compares two filenames of a filedesc structure, case-insensitive
 *
 */
int fcomp(struct filedesc *a, struct filedesc *b)
{
    return(strcasecmp(a->file,b->file));
}

/*
 * Checks directory "path" and deletes all files from it which
 * are found to be duplicates or otherwise unsuitable. Adds
 * files to the filearea.
 *
 */
int p_adopt(char *path, int default_area, unsigned long options)
{
	char olddir[PATH_MAX];
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	DIR *dirp;
	struct dirent *dirdata;
	int gained_files=0,gained_bytes=0;
	int deleted_files=0,deleted_bytes=0;
	FILE *fp;
	int nazi=0,verbose=0, domove=0, sloppy=0;
	int entries=0,fd=0,descsize=0;
	char *descbuf=NULL;
	struct filedesc *descs=NULL;
	int strict_md5;
	int to_lowercase,minassnamelen;
	char *value;

	if(options & OPT_NAZI)
		nazi=1;
	if(options & OPT_VERBOSE)
		verbose=1;
	if(options & OPT_MOVE)
		domove=1;
	if(options & OPT_SLOPPY)
		sloppy=1;

	value=p_getmisc(src_db, P_KEY_STRICTMD5);
	if(value)
		strict_md5=atoi(value);
	else
		strict_md5=atoi(P_STRICTMD5);

	value=p_getmisc(src_db, P_KEY_TOLOWERCASE);
	if(value)
		to_lowercase=atoi(value);
	else
		to_lowercase=atoi(P_TOLOWERCASE);

	value=p_getmisc(src_db, P_KEY_MINASSNAMELEN);
	if(value)
		minassnamelen=atoi(value);
	else
		minassnamelen=atoi(P_MIN_ASS_NAMELENGTH);
		
	dirp=opendir(path);
	if(!dirp)
	{
		fprintf(stderr, "Unable to open %s\n", path);
		return(0);
	}

	getcwd(olddir, PATH_MAX);
	chdir(path);

	/* Check if we want to use a description file 'pimppasubjects.txt' */
	if(options & OPT_INSERTSUBJECTS)
	{
		int i,j;
		struct stat st;
		int which;
		char tmpfile[PATH_MAX];
		
		sprintf(tmpfile, "%s/pimppasubjects.txt", olddir);
		
		if((fd=open(tmpfile, O_RDONLY|O_RDWR))>0)
		{
			fstat(fd, &st);
			descsize=st.st_size;
			descbuf=mmap(0, descsize, PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0);
			if(descbuf!=MAP_FAILED)
			{
				for(i=0;i<descsize;i++)
				{
					if(descbuf[i]==0)
						entries++;
				}
				entries=entries/2;
				descs=malloc(entries*sizeof(struct filedesc));
				if(!descs)
				{	
					fprintf(stderr, "UARRK critical malloc() error on descs\n");
					return(-1);
				}

				which=0;i=0;j=0;	// Format: FILENAME\0SUBJECT\0
				while(j<entries && i<descsize)
				{
					if(!which)
						descs[j].file=&descbuf[i];
					else
					{
						descs[j].desc=&descbuf[i];
						j++;
					}
					while(i<descsize && descbuf[i]!=0)
						i++;
					i++;	
					which^=1;
				}
		        qsort(descs, entries, sizeof(struct filedesc),
						  (int(*)(const void*, const void*)) fcomp);
			}
			else
			{
				fprintf(stderr, "mmap() of 'pimppasubjects.txt' failed\n");
				close(fd);
			}
		}
		else
		{
			fprintf(stderr, "Unable to open 'pimppasubjects.txt'\n");
		}
	}
		
	// Ok, parse through the directory to be adopted	
	while((dirdata=readdir(dirp)))
	{
		char fullpath[PATH_MAX];
		int new_file;
		char *filename=dirdata->d_name;
		struct stat st;
		char escaped_fn[2*P_LEN_FILE_NAME+1];
		char escaped_desc[2*1024+1];
		char md5sum[16];
		char md5sum_esc[16*2+1];
		int destarea_id;
		int file_integ=0;

		new_file=1;
		destarea_id=default_area;
		
		if(dirdata->d_reclen>=P_LEN_FILE_NAME)	// Oops, too long filename
		{
			if(verbose)
				printf("%s\n  !! Over %d chars filename, deleted\n",
						dirdata->d_name, P_LEN_FILE_NAME);

			unlink(dirdata->d_name);
			deleted_files++;
			deleted_bytes+=st.st_size;
			continue;
		}
		
		sprintf(fullpath, "%s/%s", path, filename);

		stat(fullpath, &st);
		if(!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			|| S_ISDIR(st.st_mode))
			continue;
		
		if(to_lowercase)				// Convert filename
		{
			char newpath[PATH_MAX];

			p_strtolc(filename);
					
			sprintf(newpath, "%s/%s", path, filename);
			rename(fullpath, newpath);
			
			strcpy(fullpath, newpath);
		}
		
		if(verbose && domove)
			printf("%s\n", filename);

//		printf("at 1\n");
/************ Dupecheck the filename *************************/

		if(!sloppy)
			file_integ=p_dupecheck(src_db, filename, destarea_id);
		else
			file_integ=-1;

		if(file_integ>=0 && file_integ!=INTEG_FAILED)	// Exists, not failed
		{
			if(domove)				// We're moving, but dupe, delete
			{
				unlink(fullpath);
				deleted_files++;
				deleted_bytes+=st.st_size;
			
				if(verbose)
					printf("  !! Exists, deleted\n");
			}
			// else don't readopt an existing file

			continue;
		}
		// After this point, the passed file is either missing
		// from database or broken (INTEG_FAILED)
		
//		printf("at 2\n");

/********** Is the filename valid at all ***********/

		if(!sloppy && !p_checkfilename(dst_db, filename))
		{
			if(verbose)
			{
				if(!domove)
					printf("%s\n", filename);
				
				printf("  !! Invalid filename, deleted\n");
			}

			unlink(fullpath);
			deleted_files++;
			deleted_bytes+=st.st_size;
			continue;
		}
		
//		printf("at 3\n");
	
/************ md5sum calc & dupecheck ***************/

		// Calculate md5sum
		memset(md5sum, 0, 16);
		fp=fopen(fullpath, "rb");
		if(fp)
		{
			if(md5_stream(fp, md5sum))
				fprintf(stderr, "md5_stream() fail on %s\n", fullpath);
			
			fclose(fp);
		}
		
//		printf("at 3a\n");

//		if(!sloppy && strict_md5 && p_md5check(md5sum))
		if(strict_md5 && p_md5check(md5sum))
		{
			// Do not delete existing file thats broken, its just the same file
			if(!domove && file_integ>=0)
				continue;
		
//			printf("at 3c\n");
			
			unlink(fullpath);
			deleted_files++;
			deleted_bytes+=st.st_size;
			
			if(verbose)
			{
				int cnt;
				
				if(!domove)
					printf("%s\n", filename);
				
				printf("  !! md5sum ");
				for(cnt=0;cnt<16;cnt++)
					printf("%02x", (unsigned char)md5sum[cnt]);
				printf(" collided, deleted\n");
				
				continue;
			}
		}
			
		mysql_escape_string(md5sum_esc, md5sum, 16);
		
//		printf("at 4\n");

/**** If we're moving, find the destination area *************/

		if(domove)
		{
			int src_context=0;
			
			p_query(src_db, "SELECT area_context "
					"FROM p_areas "
					"WHERE area_id=%d",
				default_area);
			if((sql_res=mysql_store_result(src_db)))
			{
				if((sql_row=mysql_fetch_row(sql_res)))
				{
					src_context=atoi(sql_row[0]);
				}
				mysql_free_result(sql_res);
			}
			
			destarea_id=p_getdest(filename, src_context);
			if(destarea_id==0)						// It didn't know :(
				destarea_id=default_area;

			if(verbose)
				printf("  => %d", destarea_id);

			// Check the typical target areas. If destarea_id is not
			// among them, reset it to default_area
			if(destarea_id>0 && destarea_id!=default_area)
			{
				p_query(src_db, "SELECT " 
						"  (%d REGEXP area_targets) "
						"FROM p_areas "
 	    			                "WHERE area_id=%d",
				        destarea_id, default_area);
				sql_res=mysql_store_result(src_db);
				if(sql_res)	
				{
					if((sql_row=mysql_fetch_row(sql_res))) 
					{
						if(atoi(sql_row[0])<=0)
						{
							if(verbose)
								printf("	!! Resetting target area %d to default %d\n", destarea_id, default_area);
							destarea_id=default_area;
						}
					}
					mysql_free_result(sql_res);
				}
			}
	
			if(verbose && !sloppy && destarea_id==-1)
				printf("  !! Assign patterns ordered autokill\n");

			if(sloppy && destarea_id==-1)
				destarea_id=default_area;

			if(destarea_id==default_area && nazi)	// Unknown file, delete!
			{ 
				if(verbose)
					printf("  !! NaziSays: Delete!\n");
	
				destarea_id=-1;
			}

			if(destarea_id<0)					// Marked for deletion?
			{	
				unlink(fullpath);
				deleted_files++;
				deleted_bytes+=st.st_size;
				continue;
			}
		
//			printf("at 5\n");
		
/******* Locate destination area path and move file **************/

			p_query(src_db, "SELECT area_path, area_flags "
					"FROM p_areas "
					"WHERE area_id=%d",
					destarea_id);
			sql_res=mysql_store_result(src_db);
			if(sql_res)
			{
				if((sql_row=mysql_fetch_row(sql_res)))	// Ok, move!
				{
					char newpath[PATH_MAX];
		
					sprintf(newpath, "%s%s%s", 
							sql_row[0], 
							(p_checkp(sql_row[0]) ? "" : "/"),
							filename);
			
					if(!p_mv(fullpath, newpath))
					{
						fprintf(stderr, "p_mv() error from %s to %s\n",
								fullpath, newpath);
					}
							
					strcpy(fullpath, newpath);
					
					if(verbose)
						printf("  => %s\n", sql_row[0]);
				}
				mysql_free_result(sql_res);
			}
		}
		else
		{
			int dst_context=0;
			// not moving, we're already on some pimppa filearea
			// Lets create assign pattern
			
//			printf("at 6\n");
			
			p_query(src_db, "SELECT area_context "
					"FROM p_areas "
					"WHERE area_id=%d",
				destarea_id);
			sql_res=mysql_store_result(src_db);
			if(sql_res)
			{
				if((sql_row=mysql_fetch_row(sql_res)))	// Ok, move!
				{
					dst_context=atoi(sql_row[0]);
				}
				mysql_free_result(sql_res);
			}

			p_assign(dst_db, filename, minassnamelen, destarea_id, dst_context);
		}
			
//		printf("at 7\n");

/******  All seems well. Lets punch the file in *********/

		gained_bytes+=st.st_size;
		gained_files++;

		// If we use OPT_INSERTSUBJECTS, look for the subject
		escaped_desc[0]=0;
		if(descs)
		{
			char *this;

			this=binsearch(filename, descs, entries);
			if(this)
				mysql_escape_string(escaped_desc, this, strlen(this));
		}

		mysql_escape_string(escaped_fn, filename, strlen(filename));
		
		p_query(src_db, "REPLACE INTO p_files (file_name, file_size, "
				"  file_area,file_date,file_backup, "
				"  file_md5sum, file_integ,file_trans, "
				"  file_desc) "
				"VALUES ('%s',%ld,%d,FROM_UNIXTIME(%ld),0, "
				"  '%s',%d,%d,'%s')",
			escaped_fn, st.st_size, destarea_id, 
			(long) time(NULL), md5sum_esc, 
			INTEG_NEW,TRANS_NEW,escaped_desc);
		
//		printf("at 8\n");

		if(domove && verbose)
			printf("  == OK\n");
		fflush(stdout);
	}

	closedir(dirp);
	chdir(olddir);

	if(descs)	// OPT_INSERTSUBJECTS
	{
		free(descs);
		munmap(descbuf, descsize);
		close(fd);
	}

	if(options & OPT_STATS)
	{
		printf("Ok, %d files and %d bytes gained, %d files and %d bytes were deleted.\n",
				gained_files,gained_bytes,
				deleted_files, deleted_bytes);
	}

	fflush(stdout);
	return(gained_files);
}

/* 
 * The worlds worst binary search routine (C) IGOR TECHNOLOGIES
 *
 * Looks for "filename" from "descs" having "entries" entries.
 * Returns the description if found, otherwise NULL.
 *
 * Case-insensitive.
 *
 */
char *binsearch(char *filename, struct filedesc *descs, int entries)
{
	int i=entries/2;
	int res;
	int min=0,max=entries-1;	

//	printf("ENTR %d\n", entries);
	while(min<=max && i>=0 && i<=max)
	{
//		printf("LOOK %d MI %d MA %d %s\n", i, min, max, descs[i].file);	
		res=strcasecmp(filename,descs[i].file);
		if(res==0)
			return(descs[i].desc);
		else
		{
			if(res>0)
			{
//				printf("+\n");
				min=i+1;
			}
			else		
			{
//				printf("-\n");	
				max=i-1;
			}
		}
		i=min+(max-min)/2;
	}

	return(NULL);
}
