/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/properties_frame.hpp
 * \brief The window showing the properties of an item.
 * \author Julien Jorge
 */
#ifndef __BF_PROPERTIES_FRAME_HPP__
#define __BF_PROPERTIES_FRAME_HPP__

#include "bf/item_field_edit.hpp"
#include "bf/windows_layout.hpp"

/**
 * \brief Declare the item_field_edit::set_field_value() methods.
 * \param type The type for which the method is implemented.
 * \remark The method for std::list<type> v is also declared.
 */
#define DECLARE_PROXY_SET_FIELD_VALUE(type)                             \
  void set_field_value                                                  \
  ( item_instance& item, const std::string& name, const type& v );      \
                                                                        \
  void set_field_value                                                  \
  ( item_instance& item, const std::string& name, const std::list<type>& v )

namespace bf
{
  /**
   * \brief The properties window of our program.
   * \author Julien Jorge
   */
  class properties_frame:
    public wxPanel,
    public item_field_edit::proxy
  {
  public:
    /** \brief The identifiers of the controls. */
    enum control_id
      {
        IDC_ITEM_PROPERTIES,
        IDC_FIXED_STATE,
        IDC_TEXT_IDENTIFIER,
        IDC_CHANGE_ITEM_CLASS
      }; // enum control_id

  public:
    properties_frame( wxWindow* parent );

    void set_window_layout( windows_layout& layout );

    void set_item( item_instance* item );

  private:
    void update_controls();
    void create_controls();

    void delete_field( item_instance& item, const std::string& name );
    void get_item_identifiers( wxArrayString& id, const type_field& f );

    template<typename Type>
    void do_set_field_value
    ( item_instance& item, const std::string& name, const Type& v );

    DECLARE_PROXY_SET_FIELD_VALUE( integer_type );
    DECLARE_PROXY_SET_FIELD_VALUE( u_integer_type );
    DECLARE_PROXY_SET_FIELD_VALUE( real_type );
    DECLARE_PROXY_SET_FIELD_VALUE( bool_type );
    DECLARE_PROXY_SET_FIELD_VALUE( string_type );
    DECLARE_PROXY_SET_FIELD_VALUE( sprite );
    DECLARE_PROXY_SET_FIELD_VALUE( animation_file_type );
    DECLARE_PROXY_SET_FIELD_VALUE( item_reference_type );
    DECLARE_PROXY_SET_FIELD_VALUE( font_file_type );
    DECLARE_PROXY_SET_FIELD_VALUE( sample_file_type );

    void on_close(wxCloseEvent& event);
    void on_item_focused(wxListEvent& event);
    void on_change_fixed(wxCommandEvent& event);
    void on_validate_id(wxCommandEvent& event);
    void on_change_item_class(wxCommandEvent& event);

  private:
    /** \brief The control in which we display the class of the item. */
    wxStaticText* m_item_class;

    /** \brief The list of properties available for this item. */
    item_field_edit* m_prop;

    /** \brief The control for configure the fixed option. */
    wxCheckBox* m_fixed_box;

    /** \brief The control in which we display the identifiant. */
    wxTextCtrl* m_id_text;

    /** \brief The control in which we display the description of the selected
        field. */
    wxStaticText* m_description;

    /** \brief The windows of the program. */
    windows_layout* m_windows_layout;

    DECLARE_EVENT_TABLE()

  }; // class properties_frame
} // namespace bf

#include "bf/impl/properties_frame.tpp"

#endif // __BF_PROPERTIES_FRAME_HPP__
