/*
 *  Copyright 2000 by Josh Wilmes (josh@hitchhiker.org)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

/* This is a bare-bones implementation of a text-mode (curses) GUI for
 * plex86's bochs UI code.  It support monochrome and color text modes.
 * For best results, use an 80x24 xterm with the 'vga' font.
 * I have only tested this with ncurses, but it's simple enough that I would
 * expect it to work with regular curses with very minor changes, if any. */


#include <curses.h>
#include "bochs.h"

static int in_graphics_mode = 0;

/* include support for color ncurses */
#define CURSES_COLOR 1

#define NO_MOD    0x01
#define SHIFT_MOD 0x02
#define CTRL_MOD  0x04

static Bit32u keymap[0x80][2] = {
  /* 0x00 */ {  0x00                ,  NO_MOD     },
  /* 0x01 */ {  BX_KEY_A            ,  CTRL_MOD   },
  /* 0x02 */ {  BX_KEY_B            ,  CTRL_MOD   },
  /* 0x03 */ {  BX_KEY_C            ,  CTRL_MOD   },
  /* 0x04 */ {  BX_KEY_D            ,  CTRL_MOD   },
  /* 0x05 */ {  BX_KEY_E            ,  CTRL_MOD   },
  /* 0x06 */ {  BX_KEY_F            ,  CTRL_MOD   },
  /* 0x07 */ {  BX_KEY_G            ,  CTRL_MOD   },
  /* 0x08 */ {  BX_KEY_BACKSPACE    ,  NO_MOD     },
  /* 0x09 */ {  BX_KEY_TAB          ,  NO_MOD     },
  /* 0x0A */ {  BX_KEY_J            ,  CTRL_MOD   },
  /* 0x0B */ {  BX_KEY_K            ,  CTRL_MOD   },
  /* 0x0C */ {  BX_KEY_L            ,  CTRL_MOD   },
  /* 0x0D */ {  BX_KEY_ENTER        ,  NO_MOD     },
  /* 0x0E */ {  BX_KEY_N            ,  CTRL_MOD   },
  /* 0x0F */ {  BX_KEY_O            ,  CTRL_MOD   },
  /* 0x10 */ {  BX_KEY_P            ,  CTRL_MOD   },
  /* 0x11 */ {  BX_KEY_Q            ,  CTRL_MOD   },
  /* 0x12 */ {  BX_KEY_R            ,  CTRL_MOD   },
  /* 0x13 */ {  BX_KEY_S            ,  CTRL_MOD   },
  /* 0x14 */ {  BX_KEY_T            ,  CTRL_MOD   },
  /* 0x15 */ {  BX_KEY_U            ,  CTRL_MOD   },
  /* 0x16 */ {  BX_KEY_V            ,  CTRL_MOD   },
  /* 0x17 */ {  BX_KEY_W            ,  CTRL_MOD   },
  /* 0x18 */ {  BX_KEY_X            ,  CTRL_MOD   },
  /* 0x19 */ {  BX_KEY_Y            ,  CTRL_MOD   },
  /* 0x1A */ {  BX_KEY_Z            ,  CTRL_MOD   },

  /* 0x1B */ {  BX_KEY_ESC          ,  NO_MOD     },
  /* 0x1C */ {  0x00                ,  NO_MOD     },
  /* 0x1D */ {  0x00                ,  NO_MOD     },
  /* 0x1E */ {  0x00                ,  NO_MOD     },
  /* 0x1F */ {  0x00                ,  NO_MOD     },
  
  /* 0x20 */ {  BX_KEY_SPACE        ,  NO_MOD     },
  /* 0x21 */ {  BX_KEY_1            ,  SHIFT_MOD  }, /* ! */
  /* 0x22 */ {  BX_KEY_SINGLE_QUOTE ,  SHIFT_MOD  }, /* " */
  /* 0x23 */ {  BX_KEY_3            ,  SHIFT_MOD  }, /* # */
  /* 0x24 */ {  BX_KEY_4            ,  SHIFT_MOD  }, /* $ */
  /* 0x25 */ {  BX_KEY_5            ,  SHIFT_MOD  }, /* % */
  /* 0x26 */ {  BX_KEY_7            ,  SHIFT_MOD  }, /* & */
  /* 0x27 */ {  BX_KEY_SINGLE_QUOTE ,  NO_MOD     },
  /* 0x28 */ {  BX_KEY_9            ,  SHIFT_MOD  },
  /* 0x29 */ {  BX_KEY_0            ,  SHIFT_MOD  },
  /* 0x2A */ {  BX_KEY_8            ,  SHIFT_MOD  }, /* * */

  /* 0x2B */ {  BX_KEY_EQUALS       ,  SHIFT_MOD  }, /* + */
  /* 0x2C */ {  BX_KEY_COMMA        ,  NO_MOD     }, 
  /* 0x2D */ {  BX_KEY_MINUS        ,  NO_MOD     }, 
  /* 0x2E */ {  BX_KEY_PERIOD       ,  NO_MOD     }, 
  /* 0x2F */ {  BX_KEY_SLASH        ,  NO_MOD     }, 

  /* 0x30 */ {  BX_KEY_0            ,  NO_MOD     }, 
  /* 0x31 */ {  BX_KEY_1            ,  NO_MOD     }, 
  /* 0x32 */ {  BX_KEY_2            ,  NO_MOD     }, 
  /* 0x33 */ {  BX_KEY_3            ,  NO_MOD     }, 
  /* 0x34 */ {  BX_KEY_4            ,  NO_MOD     }, 
  /* 0x35 */ {  BX_KEY_5            ,  NO_MOD     }, 
  /* 0x36 */ {  BX_KEY_6            ,  NO_MOD     }, 
  /* 0x37 */ {  BX_KEY_7            ,  NO_MOD     }, 
  /* 0x38 */ {  BX_KEY_8            ,  NO_MOD     }, 
  /* 0x39 */ {  BX_KEY_9            ,  NO_MOD     }, 

  /* 0x3A */ {  BX_KEY_SEMICOLON    ,  SHIFT_MOD  }, /* : */
  /* 0x3B */ {  BX_KEY_SEMICOLON    ,  NO_MOD     }, 
  /* 0x3C */ {  BX_KEY_COMMA        ,  SHIFT_MOD  }, /* < */
  /* 0x3D */ {  BX_KEY_EQUALS       ,  NO_MOD     }, 
  /* 0x3E */ {  BX_KEY_PERIOD       ,  SHIFT_MOD  }, /* > */
  /* 0x3F */ {  BX_KEY_SLASH        ,  SHIFT_MOD  }, /* ? */
  /* 0x40 */ {  BX_KEY_2            ,  SHIFT_MOD  }, /* @ */

  /* 0x41 */ {  BX_KEY_A            ,  SHIFT_MOD  }, 
  /* 0x42 */ {  BX_KEY_B            ,  SHIFT_MOD  }, 
  /* 0x43 */ {  BX_KEY_C            ,  SHIFT_MOD  }, 
  /* 0x44 */ {  BX_KEY_D            ,  SHIFT_MOD  }, 
  /* 0x45 */ {  BX_KEY_E            ,  SHIFT_MOD  }, 
  /* 0x46 */ {  BX_KEY_F            ,  SHIFT_MOD  }, 
  /* 0x47 */ {  BX_KEY_G            ,  SHIFT_MOD  }, 
  /* 0x48 */ {  BX_KEY_H            ,  SHIFT_MOD  }, 
  /* 0x49 */ {  BX_KEY_I            ,  SHIFT_MOD  }, 
  /* 0x4A */ {  BX_KEY_J            ,  SHIFT_MOD  }, 
  /* 0x4B */ {  BX_KEY_K            ,  SHIFT_MOD  }, 
  /* 0x4C */ {  BX_KEY_L            ,  SHIFT_MOD  }, 
  /* 0x4D */ {  BX_KEY_M            ,  SHIFT_MOD  }, 
  /* 0x4E */ {  BX_KEY_N            ,  SHIFT_MOD  }, 
  /* 0x4F */ {  BX_KEY_O            ,  SHIFT_MOD  }, 
  /* 0x50 */ {  BX_KEY_P            ,  SHIFT_MOD  }, 
  /* 0x51 */ {  BX_KEY_Q            ,  SHIFT_MOD  }, 
  /* 0x52 */ {  BX_KEY_R            ,  SHIFT_MOD  }, 
  /* 0x53 */ {  BX_KEY_S            ,  SHIFT_MOD  }, 
  /* 0x54 */ {  BX_KEY_T            ,  SHIFT_MOD  }, 
  /* 0x55 */ {  BX_KEY_U            ,  SHIFT_MOD  }, 
  /* 0x56 */ {  BX_KEY_V            ,  SHIFT_MOD  }, 
  /* 0x57 */ {  BX_KEY_W            ,  SHIFT_MOD  }, 
  /* 0x58 */ {  BX_KEY_X            ,  SHIFT_MOD  }, 
  /* 0x59 */ {  BX_KEY_Y            ,  SHIFT_MOD  }, 
  /* 0x5A */ {  BX_KEY_Z            ,  SHIFT_MOD  }, 

  /* 0x5B */ {  BX_KEY_LEFT_BRACKET ,  NO_MOD     },
  /* 0x5C */ {  BX_KEY_BACKSLASH    ,  NO_MOD     },
  /* 0x5D */ {  BX_KEY_RIGHT_BRACKET,  NO_MOD     },	      
  /* 0x5E */ {  BX_KEY_6            ,  SHIFT_MOD  },  /* ^ */
  /* 0x5F */ {  BX_KEY_MINUS        ,  SHIFT_MOD  },  /* _ */
  /* 0x60 */ {  BX_KEY_GRAVE        ,  SHIFT_MOD  },  /* ` */

  /* 0x61 */ {  BX_KEY_A            ,  NO_MOD     }, 
  /* 0x62 */ {  BX_KEY_B            ,  NO_MOD     }, 
  /* 0x63 */ {  BX_KEY_C            ,  NO_MOD     }, 
  /* 0x64 */ {  BX_KEY_D            ,  NO_MOD     }, 
  /* 0x65 */ {  BX_KEY_E            ,  NO_MOD     }, 
  /* 0x66 */ {  BX_KEY_F            ,  NO_MOD     }, 
  /* 0x67 */ {  BX_KEY_G            ,  NO_MOD     }, 
  /* 0x68 */ {  BX_KEY_H            ,  NO_MOD     }, 
  /* 0x69 */ {  BX_KEY_I            ,  NO_MOD     }, 
  /* 0x6A */ {  BX_KEY_J            ,  NO_MOD     }, 
  /* 0x6B */ {  BX_KEY_K            ,  NO_MOD     }, 
  /* 0x6C */ {  BX_KEY_L            ,  NO_MOD     }, 
  /* 0x6D */ {  BX_KEY_M            ,  NO_MOD     }, 
  /* 0x6E */ {  BX_KEY_N            ,  NO_MOD     }, 
  /* 0x6F */ {  BX_KEY_O            ,  NO_MOD     }, 
  /* 0x70 */ {  BX_KEY_P            ,  NO_MOD     }, 
  /* 0x71 */ {  BX_KEY_Q            ,  NO_MOD     }, 
  /* 0x72 */ {  BX_KEY_R            ,  NO_MOD     }, 
  /* 0x73 */ {  BX_KEY_S            ,  NO_MOD     }, 
  /* 0x74 */ {  BX_KEY_T            ,  NO_MOD     }, 
  /* 0x75 */ {  BX_KEY_U            ,  NO_MOD     }, 
  /* 0x76 */ {  BX_KEY_V            ,  NO_MOD     }, 
  /* 0x77 */ {  BX_KEY_W            ,  NO_MOD     }, 
  /* 0x78 */ {  BX_KEY_X            ,  NO_MOD     }, 
  /* 0x79 */ {  BX_KEY_Y            ,  NO_MOD     }, 
  /* 0x7A */ {  BX_KEY_Z            ,  NO_MOD     }, 

  /* 0x7B */ {  BX_KEY_LEFT_BRACKET ,  SHIFT_MOD  },
  /* 0x7C */ {  BX_KEY_BACKSLASH    ,  SHIFT_MOD  },
  /* 0x7D */ {  BX_KEY_RIGHT_BRACKET,  SHIFT_MOD  },	      
  /* 0x7E */ {  BX_KEY_GRAVE        ,  NO_MOD     },  /* ~ */

  /* 0x7F */ {  BX_KEY_BACKSPACE    ,  NO_MOD     },
};

#ifdef CURSES_COLOR
short curses_colors[8] = {
  /* 0 */ COLOR_BLACK,
  /* 1 */ COLOR_BLUE,
  /* 2 */ COLOR_GREEN,
  /* 3 */ COLOR_CYAN,
  /* 4 */ COLOR_RED,
  /* 5 */ COLOR_MAGENTA,
  /* 6 */ COLOR_YELLOW,
  /* 7 */ COLOR_WHITE
};

short curses_color_map[8][8];
#endif /* CURSES_COLOR */

// ::SPECIFIC_INIT()
//
// Called from gui.cc, once upon program startup, to allow for the
// specific GUI code (X11, BeOS, ...) to be initialized.
//
// th: a 'this' pointer to the gui class.  If a function external to the
//     class needs access, store this pointer and use later.
// argc, argv: not used right now, but the intention is to pass native GUI
//     specific options from the command line.  (X11 options, BeOS options,...)
//
// tilewidth, tileheight: for optimization, graphics_tile_update() passes
//     only updated regions of the screen to the gui code to be redrawn.
//     These define the dimensions of a region (tile).
// headerbar_y:  A headerbar (toolbar) is display on the top of the
//     VGA window, showing floppy status, and other information.  It
//     always assumes the width of the current VGA mode width, but

//     it's height is defined by this parameter.

  void
bx_gui_c::specific_init(bx_gui_c *th, int argc, char **argv, unsigned tilewidth, unsigned tileheight,
                     unsigned headerbar_y)
{
  UNUSED(th);
  UNUSED(argc);
  UNUSED(argv);
  UNUSED(tilewidth);
  UNUSED(tileheight);
  UNUSED(headerbar_y);

  initscr();
  scrollok(stdscr,FALSE);
  cbreak();  /* change to raw() to get rid of ctrl-c */
  noecho();
  nonl();
  keypad(stdscr, TRUE);
  nodelay(stdscr, TRUE);

  fprintf(stderr, "curses.cc: For best results, use the 'vga' font\n");

#ifdef CURSES_COLOR
  start_color();
  if (has_colors() == TRUE)
    fprintf(stderr, "curses.cc: Color is supported by this terminal\n");
  else 
    fprintf(stderr, "curses.cc: Color is not supported by this terminal\n");

  /* allocate color pairs */
  int pairno = 1;
  for (int i = 0; i < 8; i++) {
    for (int j = 0; j < 8; j++) {
      if (i == 0 && j == 0) continue;
      if (init_pair(pairno, curses_colors[i], curses_colors[j]) == ERR) {
        fprintf(stderr, "error allocating color pair (%d/%d)\n", i, j);
        return;
      }
      curses_color_map[i][j] = pairno;
      pairno++;
    }
  }
#endif /* CURSES_COLOR */
}


// ::HANDLE_EVENTS()
//
// Called periodically (vga_update_interval in .bochsrc) so the
// the gui code can poll for keyboard, mouse, and other
// relevant events.

  void
bx_gui_c::handle_events(void)
{
  int key;
  Bit32u key_event = 0;
  Bit32u mod = 0;

  key = getch();

  if (key == ERR) 
    return;

  if (key == 0x0C) { /* control L - refresh screen */
    redrawwin(stdscr);    

    return;  /* for now, we'll trap the control-l.  Perhaps later we will
		want to pass it through to the guest.. if so, just remove
		this return */
  }


  switch (key) {
  case KEY_END:          key_event = BX_KEY_KP_END; break;   
  case KEY_DOWN:         key_event = BX_KEY_KP_DOWN; break;
  case KEY_NPAGE:        key_event = BX_KEY_KP_PAGE_DOWN; break;
  case KEY_LEFT:         key_event = BX_KEY_KP_LEFT; break;
  case KEY_B2:           key_event = BX_KEY_KP_5; break;
  case KEY_RIGHT:        key_event = BX_KEY_KP_RIGHT; break;
  case KEY_HOME:         key_event = BX_KEY_KP_HOME; break;
  case KEY_UP:           key_event = BX_KEY_KP_UP; break;
  case KEY_PPAGE:        key_event = BX_KEY_KP_PAGE_UP; break;
  case KEY_IC:           key_event = BX_KEY_KP_INSERT; break;   
  case KEY_DC:           key_event = BX_KEY_KP_DELETE; break;
  case KEY_BACKSPACE:    key_event = BX_KEY_BACKSPACE; break;
  case KEY_F(1):         key_event = BX_KEY_F1; break;
  case KEY_F(2):         key_event = BX_KEY_F2; break;
  case KEY_F(3):         key_event = BX_KEY_F3; break;
  case KEY_F(4):         key_event = BX_KEY_F4; break;
  case KEY_F(5):         key_event = BX_KEY_F5; break;
  case KEY_F(6):         key_event = BX_KEY_F6; break;
  case KEY_F(7):         key_event = BX_KEY_F7; break;
  case KEY_F(8):         key_event = BX_KEY_F8; break;
  case KEY_F(9):         key_event = BX_KEY_F9; break;
  case KEY_F(10):        key_event = BX_KEY_F10; break;
  case KEY_F(11):        key_event = BX_KEY_F11; break;
  case KEY_F(12):        key_event = BX_KEY_F12; break;
  }

  if (! key_event && key < 0x80) {
    key_event = keymap[key][0];
    mod = keymap[key][1];
  }

  if (! key_event) {    
    fprintf(stderr, "# bx_gui_c::handle_events(): keysym %x unhandled!\n", (unsigned) key);
    return;
  }

  if (mod & CTRL_MOD) 
    pluginGenScancode(BX_KEY_CTRL_L | BX_KEY_PRESSED);
  
  if (mod & SHIFT_MOD) 
    pluginGenScancode(BX_KEY_SHIFT_L | BX_KEY_PRESSED);

  pluginGenScancode(key_event | BX_KEY_PRESSED);
  pluginGenScancode(key_event | BX_KEY_RELEASED);

  if (mod & CTRL_MOD) 
    pluginGenScancode(BX_KEY_CTRL_L | BX_KEY_RELEASED);
  
  if (mod & SHIFT_MOD) 
    pluginGenScancode(BX_KEY_SHIFT_L | BX_KEY_RELEASED);
}


// ::FLUSH()
//
// Called periodically, requesting that the gui code flush all pending
// screen update requests.

  void
bx_gui_c::flush(void)
{
  refresh();
}


// ::CLEAR_SCREEN()
//
// Called to request that the VGA region is cleared.  Don't
// clear the area that defines the headerbar.

  void
bx_gui_c::clear_screen(void)  
{
  clear();
}



// ::TEXT_UPDATE()
//
// Called in a VGA text mode, to update the screen with
// new content.
//
// old_text: array of character/attributes making up the contents
//           of the screen from the last call.  See below
// new_text: array of character/attributes making up the current
//           contents, which should now be displayed.  See below
//
// format of old_text & new_text: each is 4000 bytes long.
//     This represents 80 characters wide by 25 high, with
//     each character being 2 bytes.  The first by is the
//     character value, the second is the attribute byte.
//     I currently don't handle the attribute byte.
//
// cursor_x: new x location of cursor
// cursor_y: new y location of cursor

  void
bx_gui_c::text_update(Bit8u *old_text, Bit8u *new_text,
                      unsigned long cursor_x, unsigned long cursor_y,
		      unsigned rows)
{
  UNUSED(old_text);
  UNUSED(new_text);
  UNUSED(cursor_x);
  UNUSED(cursor_y);

  if (in_graphics_mode) {
    printf("Text mode has been re-entered - display is now available\n");
    in_graphics_mode = 0;
    return;
  }

  move(0,0);

  for (unsigned int i=0; i<rows; i++) {    
    move(i, 0);
    for (unsigned int j=0; j<80; j++) {
      chtype c = new_text[(i*80+j)*2];
      char style = new_text[(i*80+j)*2 + 1];      
      char fgbold  = style & 0x08;
      char bgbold  = (style & 0x80) >> 4;
#ifdef CURSES_COLOR
      char fgcolor = style & 0x07;
      char bgcolor = (style & 0x70) >> 4;      
#endif

      /* remap any characters as necessary */
      switch (c) {
      case 0x00:  c=' ';
      }

      if (fgbold) c = c | A_BOLD;
      if (bgbold) c = c | A_REVERSE;
#ifdef CURSES_COLOR
      c = c | COLOR_PAIR(curses_color_map[fgcolor][bgcolor]);
#endif  // CURSES_COLOR

      addch(c);
    }
  }
  
  move(cursor_y, cursor_x);
}


// ::PALETTE_CHANGE()
//
// Allocate a color in the native GUI, for this color, and put
// it in the colormap location 'index'.
// returns: 0=no screen update needed (color map change has direct effect)
//          1=screen updated needed (redraw using current colormap)

  Boolean
bx_gui_c::palette_change(unsigned index, unsigned red, unsigned green, unsigned blue)
{
  UNUSED(index);
  UNUSED(red);
  UNUSED(green);
  UNUSED(blue);

  return(0);
}


// ::GRAPHICS_TILE_UPDATE()
//
// Called to request that a tile of graphics be drawn to the
// screen, since info in this region has changed.
//
// tile: array of 8bit values representing a block of pixels with
//       dimension equal to the 'tilewidth' & 'tileheight' parameters to
//       ::specific_init().  Each value specifies an index into the
//       array of colors you allocated for ::palette_change()
// x0: x origin of tile
// y0: y origin of tile
//
// note: origin of tile and of window based on (0,0) being in the upper
//       left of the window.

  void
bx_gui_c::graphics_tile_update(Bit8u *tile, unsigned x0, unsigned y0)
{
  UNUSED(tile);
  UNUSED(x0);
  UNUSED(y0);
  
  if (! in_graphics_mode) {
    printf("Graphics mode has been entered - no display is available\n");
    in_graphics_mode = 1;
  }
}



// ::DIMENSION_UPDATE()
//
// Called when the VGA mode changes it's X,Y dimensions.
// Resize the window to this size, but you need to add on
// the height of the headerbar to the Y value.
//
// x: new VGA x size
// y: new VGA y size (add headerbar_y parameter from ::specific_init().

  void
bx_gui_c::dimension_update(unsigned x, unsigned y)
{
  UNUSED(x);
  UNUSED(y);
}


// ::CREATE_BITMAP()
//
// Create a monochrome bitmap of size 'xdim' by 'ydim', which will
// be drawn in the headerbar.  Return an integer ID to the bitmap,
// with which the bitmap can be referenced later.
//
// bmap: packed 8 pixels-per-byte bitmap.  The pixel order is:
//       bit0 is the left most pixel, bit7 is the right most pixel.
// xdim: x dimension of bitmap
// ydim: y dimension of bitmap

  unsigned
bx_gui_c::create_bitmap(const unsigned char *bmap, unsigned xdim, unsigned ydim)
{
  UNUSED(bmap);
  UNUSED(xdim);
  UNUSED(ydim);

  return(0);
}


// ::HEADERBAR_BITMAP()
//
// Called to install a bitmap in the bochs headerbar (toolbar).
//
// bmap_id: will correspond to an ID returned from
//     ::create_bitmap().  'alignment' is either BX_GRAVITY_LEFT
//     or BX_GRAVITY_RIGHT, meaning install the bitmap in the next
//     available leftmost or rightmost space.
// alignment: is either BX_GRAVITY_LEFT or BX_GRAVITY_RIGHT,
//     meaning install the bitmap in the next
//     available leftmost or rightmost space.
// f: a 'C' function pointer to callback when the mouse is clicked in
//     the boundaries of this bitmap.

  unsigned
bx_gui_c::headerbar_bitmap(unsigned bmap_id, unsigned alignment, void (*f)(void))
{
  UNUSED(bmap_id);
  UNUSED(alignment);
  UNUSED(f);

  return(0);
}


// ::SHOW_HEADERBAR()
//
// Show (redraw) the current headerbar, which is composed of
// currently installed bitmaps.

  void
bx_gui_c::show_headerbar(void)
{
}


// ::REPLACE_BITMAP()
//
// Replace the bitmap installed in the headerbar ID slot 'hbar_id',
// with the one specified by 'bmap_id'.  'bmap_id' will have
// been generated by ::create_bitmap().  The old and new bitmap
// must be of the same size.  This allows the bitmap the user
// sees to change, when some action occurs.  For example when
// the user presses on the floppy icon, it then displays
// the ejected status.
//
// hbar_id: headerbar slot ID
// bmap_id: bitmap ID

  void
bx_gui_c::replace_bitmap(unsigned hbar_id, unsigned bmap_id)
{
  UNUSED(hbar_id);
  UNUSED(bmap_id);

}


// ::EXIT()
//
// Called before bochs terminates, to allow for a graceful
// exit from the native GUI mechanism.

  void
bx_gui_c::exit(void)
{
  mvcur(0, COLS-1, LINES-1, 0);
  endwin();
}
