/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "polyxcalc-main.h"
#include "polyxcalc-ui-calculator.h"
#include "polyxcalc-ui-chempad.h"
#include "polyxcalc-ui-recorder.h"


/****************************************************************************
		 This file is the entry point of the

			   polyxcalc module

		   in the polyxmass software suite.
****************************************************************************/


PxmCalcCtxt *
polyxcalc_main_prepare_new_context (PxmPolchemdefSpec *ps)
{
  /* The user has asked that a new calculator be launched. We just
     comply with the command. Note that 'ps' can be NULL if the user
     wants to launch the editor without any specific polymer chemistry
     definition (thus using only atoms from the default 'basic' atom
     definition.

     If ps is non-NULL, then that means that we can get it type
     member, which should allow us to see if it is already available
     in the array of all available polymer chemistry definition in the
     polyxmass global array.
  */
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  PxmAtomSpec *as = NULL;

  PxmCalcCtxt *calcctxt = NULL;


  /*
    First allocate the new calcctxt instance: whatever the 'ps' param,
    we'll need one !
  */
  calcctxt = polyxcalc_calcctxt_new ();

  if (ps != NULL)
    {
      /* We get a pointer to a polymer chemistry definition
	 specification instance (PxmPolchemdefSpec) that contains a
	 'type' member saying of what polchemdef we need here. So all
	 we have to do is check if such asked polchemdef exists
	 already in memory in the form of a polchemdef context
	 (PxmPolchemdefCtxt).
      */
      polchemdefctxt = polyxmass_polchemdefctxt_get_ptr_by_type 
	(ps->type, polyxmass_polchemdefctxtGPA);
      
      if (polchemdefctxt == NULL)
	{
	  /* Apparently not a single polymer definition of type
	     ps->type is available in memory... We need to construct a
	     new polchemdefctxt, load the polymer definition context
	     and set the members of polchemdefctxt properly.
	  */
	  polchemdefctxt = polyxmass_polchemdefctxt_new ();
	  
	  if (FALSE == 
	      polyxmass_polchemdefctxt_init_with_type (polchemdefctxt,
						       ps->type,
						       TRUE))
	    {
	      g_critical (_("%s@%d: failed initializing polchemdefctxt"
		       " for type '%s'\n"),
		     __FILE__, __LINE__, ps->type);

	      polyxmass_polchemdefctxt_free (polchemdefctxt);
	      polyxcalc_calcctxt_free (calcctxt);
	      	      
	      return NULL;
	    }
	  /* End of 
	     if (polchemdefctxt == NULL)
	  */

	  /* At this point, the new polchemdef seems to be faithfully
	     rendered in memory and the context setup correctly. We
	     can add it to the main global array of such instances.
	  */
	  g_ptr_array_add (polyxmass_polchemdefctxtGPA, polchemdefctxt);
	}
      /* End of 
	 if (polchemdefctxt == NULL)
      */
      
      /* At this point 'polchemdefctxt' points to a faithful polymer
	 chemistry definition context that we'll be able to set to the
	 calcctxt instance.
      */
      calcctxt->polchemdefctxt = polchemdefctxt;
      
      /* Immediately ref the polchemdefctxt so that if we have to free
	 the calcctxt, it will be freed if it was allocated here (that
	 is no other context is using it.
      */
      polyxmass_polchemdefctxt_ref (polchemdefctxt);
      
      /* We will use the atoms array from the polymer chemistry
	 definition, and for that all we do is to let
	 calcctxt->atomGPA point to the
	 calcctxt->polchemdefctxt->polchemdef->atomGPA array. We won't
	 free that calcctxt->atomGPA array in polyxcalc_calcctxt_free
	 (). See that function for details.
       */
      calcctxt->atomGPA = calcctxt->polchemdefctxt->polchemdef->atomGPA;
    }
  else
    {
      /* The caller did not give a valid 'ps' as a parameter, so 
	 we do without any polymer chemistry definition....

	 and we only load the atom definition that's default: 'basic'.
      */
      as = polyxmass_globals_get_matching_polchemdef_atomdef 
	(/* gchar *polchemdef */NULL,
	 /* gint *polchemdef_idx */NULL,
	 "basic",
	 /* gint *atomdef_idx */NULL);
      
      /* At this point we MUST have a proper 'as' instance, otherwise that
	 is an error, because we could not go on in any imaginable way.
      */
      if (as == NULL)
	{
	  g_critical (_("%s@%d: Failed finding a matching atom "
			"definition, please ensure that the GNU polyxmass "
			"software suite is correctly installed. "
			"In particular, is the GNU polyxmass-common "
			"package installed?\n"),
		      __FILE__, __LINE__);
	  
	  polyxcalc_calcctxt_free (calcctxt);
	  
	  return NULL;
	}
      
      /* At this point we can load the atom definition file that should
	 sit in the as->file member. Parse the file, which will have a
	 side effect of filling calcctxt->atomGPA with PxmAtom instances.
      */
      if (-1 == pxmchem_atom_render_xml_file (as->file, calcctxt->atomGPA))
	{
	  g_critical (_("%s@%d: failed parsing the atom "
			"definition file '%s'.\n"),
		      __FILE__, __LINE__, as->file);
	  
	  polyxcalc_calcctxt_free (calcctxt);
	}
    }
  
  /* We finally have terminated all the chemistry-related stuff in the
     polyxcalc context (PxmCalcCtxt). We can now go on with the
     graphical interface stuff...
  */
  /* The calculator window, which is the main interface window with
     the user.
  */
  if (NULL == polyxcalc_ui_calculator_wnd_setup_window (calcctxt))
    {
      g_critical (_("%s@%d: failed setting up a new polymer "
		    "mass calculator.\n"),
		  __FILE__, __LINE__);
      
      polyxcalc_calcctxt_free (calcctxt);
      
      return NULL;
    }
  
  /* The chempad window, which is the window where the user can
     display the programmable chemical pad.
  */
  if (NULL == polyxcalc_ui_chempad_wnd_setup_window (calcctxt))
    {
      g_critical (_("%s@%d: failed setting up the chemical pad.\n"),
	     __FILE__, __LINE__);
      
      polyxcalc_calcctxt_free (calcctxt);
      
      return NULL;
    }
  
  /* The recorder window, which is the window where all the actions
     are recorder for the user to collate her work into an electronic
     logbook.
  */
  if (NULL == polyxcalc_ui_recorder_wnd_setup_window (calcctxt))
    {
      g_critical (_("%s@%d: failed setting up the logbook recorder.\n"),
	     __FILE__, __LINE__);
      
      polyxcalc_calcctxt_free (calcctxt);
      
      return NULL;
    }


  /* At this point, we can add the new polyxcalc context to the array
     of such instances.
  */
  g_ptr_array_add (polyxcalc_calcctxtGPA, calcctxt);
  
  return calcctxt;
}

  




