/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-pka-ph-pi.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-acidobasic.h"



GtkWidget *
polyxedit_pka_ph_pi_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;

  gdouble *positive_charges = NULL;
  gdouble *negative_charges = NULL;
  


  g_assert (editctxt != NULL);



  gui_file = 
    g_strdup_printf ("%s/polyxedit-acidobasic.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "ph_pka_net_charge_pi_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "ph_pka_net_charge_pi_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed creating the acido-basic config window\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);

  /* Immediately allocate the doubles that we will feed continuously 
     when performing calculations...
  */
  positive_charges = g_malloc0 (sizeof (gdouble));
  g_object_set_data_full (G_OBJECT (window),
			  "positive_charges", positive_charges, g_free);
  
  negative_charges = g_malloc0 (sizeof (gdouble));
  g_object_set_data_full (G_OBJECT (window),
			  "negative_charges", negative_charges, g_free);
  
  
  /* We may need to displayed timed-out messages in here:
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);
  

  widget = glade_xml_get_widget (xml, "polymer_sequence_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polymer_sequence_name_entry", widget);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  widget = glade_xml_get_widget (xml, "polymer_sequence_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polymer_sequence_id_number_entry", widget);

  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  
  widget = glade_xml_get_widget (xml, "update_sequence_region_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "update_sequence_region_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_pka_ph_pi_wnd_sequence_region_button), 
		    window);

  widget = glade_xml_get_widget (xml, "sel_start_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "sel_start_pos_entry", widget);
  
  widget = glade_xml_get_widget (xml, "sel_end_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "sel_end_pos_entry", widget);
  
  widget = glade_xml_get_widget (xml, "whole_sequence_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "whole_sequence_checkbutton", widget);

  /* At this point we should be able to set the values to the text
     entries above, if some portion of the protein is already selected...
  */
  polyxedit_pka_ph_pi_wnd_sequence_region_button (NULL,
						  window);
  

  widget = glade_xml_get_widget (xml, "hydrogen_potential_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "hydrogen_potential_spinbutton", widget);

  widget = glade_xml_get_widget (xml, "isoelectric_point_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "isoelectric_point_entry", widget);

  widget = glade_xml_get_widget (xml, "treated_pka_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "treated_pka_number_entry", widget);

  widget = glade_xml_get_widget (xml, "positive_charges_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "positive_charges_entry", widget);

  widget = glade_xml_get_widget (xml, "negative_charges_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "negative_charges_entry", widget);

  widget = glade_xml_get_widget (xml, "net_charge_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "net_charge_entry", widget);

  widget = glade_xml_get_widget (xml, "compute_net_charge_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "compute_net_charge_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_pka_ph_pi_wnd_compute_net_charge_button), 
		    window);

  widget = glade_xml_get_widget (xml, "compute_isoelectric_point_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "compute_isoelectric_point_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_pka_ph_pi_wnd_compute_pi_button), 
		    window);


  gtk_widget_show_all (GTK_WIDGET (window));


  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_pka_ph_pi_wnd_delete_event), 
		    editctxt);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_pka_ph_pi_wnd_destroy_event), 
		    editctxt);
  


  /* We have finished setting up the window, and so also using the xml
     data, unref them.
   */
  g_object_unref (G_OBJECT (xml));
  
  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one monomer modification window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("pka_ph_pi_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_pka_ph_pi_wnd_really_close);
  
  g_free (help);
  
  return window;
}
  

gboolean
polyxedit_pka_ph_pi_wnd_get_sequence_interval (GtkWidget *window,
						gint *start_idx,
						gint *end_idx)
{
  GtkWidget *widget = NULL;

  gchar *help = NULL;
  gint len = 0;

  gint start = 0;
  gint end = 0;
  
  PxmEditCtxt *editctxt = NULL;

  


  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  g_assert (start_idx != NULL && end_idx != NULL);
  

  g_assert (editctxt->polymer != NULL);
  g_assert (editctxt->polymer->monomerGPA != NULL);
  len = editctxt->polymer->monomerGPA->len;

  if (len == 0)
    {
      return -1;
    }
  

  /* We must check if the user wants that the computation be made for
     the whole sequence or for the selection region of the polymer sequence
     only, or for the interval that is defined with the start_pos and 
     end_pos entry widgets.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "whole_sequence_checkbutton");
  g_assert (widget != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      start = 0;
      end = len;
    }
  else
    {
      /* We should check what the start_pos and end_pos entries say:
       */
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  start = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == libpolyxmass_globals_strtoi (help, &start, 10))
	    start = 0;
	  else
	    {
	      /* Remember that the user "counts" monomers starting at 1,
		 and we need to count starting at 0, because we use indexes
		 and not positions. So decrement by one the value entered
		 by the user.
	      */
	      start--;
	    }
	}
      
      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  end = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == libpolyxmass_globals_strtoi (help, &end, 10))
	    end = 0;
	  else
	    {
	      /* Remember that for the *end_idx, the computations need
		 to have it incremented by one with respect to the real
		 last monomer index. So, contrary to start_idx, we do not
		 change its value.
		 
		 end;
	      */
	    }
	}

      /* Now make some final checks:
       */

      if (start > len)
	start = len;

      if (start < 0)
	start = 0;
      
      if (end > len)
	end = len;

      if (end < 0)
	end = 0;
    }

  *start_idx = start;
  *end_idx = end;
  
  return TRUE;
}


void
polyxedit_pka_ph_pi_wnd_sequence_region_button (GtkWidget *button,
						gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  PxmEditCtxt *editctxt = NULL;
  
  gchar *help = NULL;
  
  gint start = 0;
  gint end = 0;
  
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  
  /* Prepare the variables from the sequence editor window, so that
     we can put them immediately in the start/end positions's entry
     widgets.
  */
  if (FALSE != polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  &start, &end))
    {
      /* Remember that due to the handling of the polymer sequence
	 displayed in the sequence editor, the end value that is 
	 returned is indeed corresponding to the real index of the last
	 selected monomer + 1, so care should be taken not to outbound
	 the polymer sequence array.
      */
      
      /* We show the (start + 1) value because we want to show
	 a position and not an index, while the end needs not be
	 incremented because it is by itself already incremented.
      */      
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      help = g_strdup_printf ("%d", start + 1);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      widget =  g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      help = g_strdup_printf ("%d", end);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);
    }
  else
    {
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);
      
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  return;
}


/* Returns -1 if an error occurs. Returns the number of pka that were
   processed.
 */
gint
polyxedit_pka_ph_pi_wnd_compute_net_charge_button (GtkWidget *button,
						   gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *ph_widget = NULL;
  GtkWidget *charges_widget = NULL;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolymer *polymer = NULL;
  PxmEditCtxt *editctxt = NULL;
  PxmCalcOpt *calcopt = NULL;

  gint processed_chemgroups = 0;

  gchar *help = NULL;
  
  gdouble ph = 0;
  gdouble *positive_charges = NULL;
  gdouble *negative_charges = NULL;
    
  /* We are asked to compute the net charge of the polymer sequence
     (or its interval) for the pH currently displayed in the
     corresponding spinbutton widget.
  */
  g_assert (window != NULL);
  
  /* Note that the charges holders (gdouble *) are allocated when the
     window is set up, so we first have to get the pointers out of the
     window ! Of course we have to reset them to 0 prior to performing
     the new computation.
  */
  negative_charges = g_object_get_data (G_OBJECT (window),
					"negative_charges");
  g_assert (negative_charges != NULL);
  *negative_charges = 0;
  positive_charges = g_object_get_data (G_OBJECT (window),
					"positive_charges");
  g_assert (positive_charges != NULL);
  *positive_charges = 0;


  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  


  ph_widget =  g_object_get_data (G_OBJECT (window), 
				  "hydrogen_potential_spinbutton");
  g_assert (ph_widget != NULL);
  
  ph = gtk_spin_button_get_value (GTK_SPIN_BUTTON (ph_widget));

  if (ph < 0 || ph > 14)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("pH value is not acceptable."),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return -1;
    }
  

  /* Duplicate the editctxt's calcopt (because its start_idx, end_idx
     members are going to be modified during the cleavage operation):
   */
  calcopt = pxmchem_calcopt_dup (editctxt->calcopt);

  /* Get the sequence interval for which the computation should be
     performed.
   */
  if (FALSE ==  
      polyxedit_pka_ph_pi_wnd_get_sequence_interval (window,
						     &calcopt->start_idx,
						     &calcopt->end_idx))
  {
    polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				      _("Failed getting sequence interval."),
				      POLYXMASS_MEDI_MSG_TIMEOUT);

    pxmchem_calcopt_free (calcopt);
    
    return -1;
  }
  
  /* At this point we have all the requisites to work on the net charge
     computation.
  */
  processed_chemgroups =
    polyxedit_acidobasic_polseq_compute_net_charge (editctxt, ph,
						    calcopt,
						    negative_charges,
						    positive_charges);
  if (-1 == processed_chemgroups)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed computing net charge."),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      
      pxmchem_calcopt_free (calcopt);
      
      return -1;
    }

  /*
  debug_printf (("The charges are: "
		 "negative = %lf ; positive = %lf ; net = %lf\n"
		 "Number of processed pkas = %d\n",
		 *negative_charges, *positive_charges, 
		 *positive_charges + *negative_charges,
		 processed_chemgroups));
  */
  
  charges_widget = g_object_get_data (G_OBJECT (window), 
				      "negative_charges_entry");
  g_assert (charges_widget != NULL);
  
  help = g_strdup_printf ("%lf", *negative_charges);
  
  gtk_entry_set_text (GTK_ENTRY (charges_widget), help);
  
  g_free (help);

  charges_widget = g_object_get_data (G_OBJECT (window), 
				      "positive_charges_entry");
  g_assert (charges_widget != NULL);
  
  help = g_strdup_printf ("%lf", *positive_charges);
  
  gtk_entry_set_text (GTK_ENTRY (charges_widget), help);
  
  g_free (help);

  charges_widget = g_object_get_data (G_OBJECT (window), 
				      "net_charge_entry");
  g_assert (charges_widget != NULL);
  
  help = g_strdup_printf ("%lf", *positive_charges + *negative_charges);
  
  gtk_entry_set_text (GTK_ENTRY (charges_widget), help);
  
  g_free (help);

  pxmchem_calcopt_free (calcopt);

  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				    _("Operation completed."),
				    POLYXMASS_NORM_MSG_TIMEOUT);

  return processed_chemgroups;
}








gint
polyxedit_pka_ph_pi_wnd_compute_pi_button (GtkWidget *button,
					   gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *messages_entry = NULL;

  PxmEditCtxt *editctxt = NULL;
  PxmCalcOpt *calcopt = NULL;

  gint processed_chemgroups = 0;
  gint iteration = 0;

  gdouble tested_ph = 0;
  
  gdouble charge_first = 0;
  gdouble charge_third = 0;

  gdouble *positive_charges = NULL;
  gdouble *negative_charges = NULL;

  GtkWidget *widget = NULL;

  gchar *help = NULL;

  

  

  /* We are asked to compute the isoelectric point for the current
     polymer sequence (or a region of it). 

     The way we do this is start by setting pH to 7.0, compute the net
     charge of the sequence at this pH and then set the pH to
     higher/lower value if the charge is farther away (positive or
     negative) than 0.05.
  */

  g_assert (window != NULL);


  /* Note that the charges holders (gdouble *) are allocated when the
     window is set up, so we first have to get the pointers out of the
     window ! Of course we have to reset them to 0 prior to performing
     the new computation.
  */
  negative_charges = g_object_get_data (G_OBJECT (window),
					"negative_charges");
  g_assert (negative_charges != NULL);
  *negative_charges = 0;
  
  positive_charges = g_object_get_data (G_OBJECT (window),
					"positive_charges");
  g_assert (positive_charges != NULL);
  *positive_charges = 0;
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  /* Duplicate the editctxt's calcopt (because its start_idx, end_idx
     members are going to be modified during the cleavage operation):
   */
  calcopt = pxmchem_calcopt_dup (editctxt->calcopt);

  /* Get the sequence interval for which the computation should be
     performed.
   */
  if (FALSE ==  
      polyxedit_pka_ph_pi_wnd_get_sequence_interval (window,
						     &calcopt->start_idx,
						     &calcopt->end_idx))
  {
    polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				      _("Failed getting sequence interval."),
				      POLYXMASS_MEDI_MSG_TIMEOUT);

    pxmchem_calcopt_free (calcopt);
    
    return -1;
  }
  
  messages_entry = g_object_get_data (G_OBJECT (window), "messages_entry");
  g_assert (messages_entry != NULL);
    

  while (1)
    {

#if 0
      help = g_strdup_printf ("Current pH being tested: %lf", tested_ph);
      gtk_entry_set_text (GTK_ENTRY (messages_entry), help);
      debug_printf (("%s\n", help));
      g_free (help);
#endif

      processed_chemgroups =
	polyxedit_acidobasic_polseq_compute_net_charge (editctxt, tested_ph,
							calcopt,
							negative_charges,
							positive_charges);
      if (-1 == processed_chemgroups)
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					    _("Failed computing net charge."),
					    POLYXMASS_MEDI_MSG_TIMEOUT);
	  
	  pxmchem_calcopt_free (calcopt);
	  
	  return -1;
	}

      /* Note that if the 0.001 tested_ph step is enough to switch
	 the net charge from one excess value to another excess value
	 in the opposite direction, we'll enter an infinite loop.

	 The evidence for such loop is that every other two measures,
	 the net charge of the polymer sequence will be the same.

	 Here we test this so that we can break the loop.
      */
      iteration++;
      
      if (iteration == 1)
	{
	  charge_first = (*positive_charges + *negative_charges);
	}
      else if (iteration == 3)
	{
	  charge_third = (*positive_charges + *negative_charges);
	  
	  if (charge_third == charge_first)
	    {
	      /*
	      debug_printf (("Exiting loop with special case\n"));
	      */
	      break;
	    }
	    
	  iteration = 0;
	  
	  charge_first = (*positive_charges + *negative_charges);
	}
      
            
      /* At this point we have to test the net charge:
       */
      if ((*positive_charges + *negative_charges) >= -0.1
	  && (*positive_charges + *negative_charges) <= 0.1)
	break;
      
      if ((*positive_charges + *negative_charges) > 0)
	{
	  /* The tested_ph is too acidic.
	   */
	  tested_ph += 0.01;
	  
	  *positive_charges = 0;
	  *negative_charges = 0;
	  
	  continue;
	}
      
      if ((*positive_charges + *negative_charges) < 0)
	{
	  /* The tested_ph is too basic.
	   */
	  tested_ph -= 0.01;
	  
	  *positive_charges = 0;
	  *negative_charges = 0;

	  continue;
	}
    }
  
  /* At this point, we should have computed the pI.
   */
  widget = g_object_get_data (G_OBJECT (window), 
				      "negative_charges_entry");
  g_assert (widget != NULL);
  
  help = g_strdup_printf ("%lf", *negative_charges);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  
  g_free (help);


  widget = g_object_get_data (G_OBJECT (window), 
				      "positive_charges_entry");
  g_assert (widget != NULL);
  
  help = g_strdup_printf ("%lf", *positive_charges);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  
  g_free (help);


  widget = g_object_get_data (G_OBJECT (window), 
				      "net_charge_entry");
  g_assert (widget != NULL);
  
  help = g_strdup_printf ("%lf", *positive_charges + *negative_charges);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  
  g_free (help);


  widget = g_object_get_data (G_OBJECT (window), 
			      "isoelectric_point_entry");
  g_assert (widget != NULL);
  
  help = g_strdup_printf ("%lf", tested_ph);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);

  g_free (help);
  

  widget = g_object_get_data (G_OBJECT (window), 
			      "treated_pka_number_entry");
  g_assert (widget != NULL);
  
  help = g_strdup_printf ("%d", processed_chemgroups);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  
  g_free (help);
  
  
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				    _("Computation finished."),
				    POLYXMASS_MEDI_MSG_TIMEOUT);
  
  pxmchem_calcopt_free (calcopt);
  
  return processed_chemgroups;
}








/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_pka_ph_pi_wnd_really_close (GtkWidget *window)
{
  gint count = 0;

  
  g_assert (window != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
   
  gtk_widget_destroy (window);
}


gint
polyxedit_pka_ph_pi_wnd_delete_event  (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  help = g_strdup_printf ("pka_ph_pi_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT
		     (editctxt->seqeditorctxt->sequence_editor_wnd), 
		     help, NULL);
  g_free (help);
    
  return TRUE;
}


gint
polyxedit_pka_ph_pi_wnd_destroy_event  (GtkWidget *window, 
					 GdkEventAny *event, 
					 gpointer data)
{
  return FALSE;
}

