/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-seqed-widget-kbd.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxmass-ui-code-completions.h"
#include "polyxedit-ui-masses-display-wnd.h"



/* We are dealing with key strokes on the keyboard. These key strokes
   do happen while the focus is on the sequence canvas that belongs to
   the seqed_widget PxmSeqedWidget object. Remember that this widget
   is for a number of uses : in the polymer sequence editor, in the
   find/replace utility, in polyxcalc... It MUST know, at least at the
   moment a key stroke is performed in it, the PxmPolchemdefCtxt and
   the PxmPolymer. Otherwise it cannot perform any task. Remember that
   the PxmSeqedWidget class has these two pointers. Just access them.
*/


gboolean
polyxmass_seqed_widget_kbd_key_press_event (GtkWidget *widget,
					    GdkEventKey *event,
					    gpointer data)
{
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  GtkMenu *canvas_menu = NULL;
  
  gchar *key_name = NULL;
  
  /* Often, key combinations like Shift + arrow down are use to
   * perform actions like select a sequence stretch. When a key is
   * pressed it is thus necessary to make sure if it is either a
   * shift, ctrl or alt key or a simple ascii character key (an
   * alphabetic character). If a control character is being type and
   * caught here, we have to set to true the corresponding globabl
   * variable ('kbd_alt_down' for example). Each time a character key is
   * caught here, we check if a control key was down at that precise
   * time, so that it becomes clear what the user wanted to do.
   */

  switch (gdk_event_key->keyval)
    {
    case GDK_Shift_L:
    case GDK_Shift_R:
      kbd_shift_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Control_L:
    case GDK_Control_R:
      kbd_control_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Alt_L:
    case GDK_Alt_R:
      kbd_alt_down = TRUE;
      return TRUE;     
      break;

    case GDK_Meta_L:
    case GDK_Meta_R:
      kbd_meta_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Left:
      return polyxmass_seqed_widget_kbd_left_handler (widget);
      break;
	  
    case GDK_Right:
      return polyxmass_seqed_widget_kbd_right_handler (widget);
      break;
	  
    case GDK_Up:
      return polyxmass_seqed_widget_kbd_up_handler (widget);
      break;
	  
    case GDK_Down:
      return polyxmass_seqed_widget_kbd_down_handler (widget);
      break;
	  
    case GDK_Home:
      return polyxmass_seqed_widget_kbd_home_handler (widget);
      break;
	  
    case GDK_End:
      return polyxmass_seqed_widget_kbd_end_handler (widget);
      break;
	  
    case GDK_Page_Up:
      return polyxmass_seqed_widget_kbd_page_up_handler (widget);
      break;
	  
    case GDK_Page_Down:
      return polyxmass_seqed_widget_kbd_page_down_handler (widget);
      break;
    }
      
  if (GDK_Escape == gdk_event_key->keyval)
    {
      if (FALSE == 
	  polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget))
	{
	  g_error (_("%s@%d: failed to delete last char from elab code\n"),
		   __FILE__, __LINE__);
	      
	  return FALSE;
	}
	  
      return TRUE;
    }
      
  if (GDK_Tab == gdk_event_key->keyval)
    {
      /* TAB makes the completion. The function that is called
       * below will check if the current elab_code is unambiguous
       * and if so will evaluate it. If not, a list of all
       * available completions will be displayed for information
       * to the user.
       */
      if (FALSE == 
	  polyxmass_seqed_widget_kbd_check_complete_elab_code (widget))
	{
	  g_error (_("%s@%d: failed to check complete elab code\n"),
		   __FILE__, __LINE__);
	  
	  return FALSE;
	}
	  
      return TRUE;
    }
	    
  if (GDK_Return == gdk_event_key->keyval)
    {
      return polyxmass_seqed_widget_kbd_enter_handler (widget);
    }
      
  if (GDK_BackSpace == gdk_event_key->keyval)
    {
      return polyxmass_seqed_widget_kbd_backspace_handler (widget);
    }
      
  if (GDK_Delete == gdk_event_key->keyval)
    {
      return polyxmass_seqed_widget_kbd_delete_handler (widget);
    }

  if (GDK_F10 == gdk_event_key->keyval)
    {
      /* That key should elicit the appearance of the contextual menu
	 on the seqed_widget's canvas. The menu (if any) will be
	 popped up at the precise location of the mouse
	 cursor. Remember that the seqed_widget, for which we are
	 dealing with the keyboard key strokes, might not have a
	 contextual menu associated with it. This is why we do not
	 assert that 'canvas_menu' is not NULL !!!
      */
      canvas_menu = 
	(GtkMenu *) g_object_get_data (G_OBJECT (PXM_SEQED_WIDGET (widget)->canvas), 
				       "canvas_menu");
      if (canvas_menu != NULL)
	/* The 1 below, as sixth param is to tell that the first button
	   of the mouse is going to be used to trigger the "activate"
	   signal of the menu item to be selected.
	*/
	gtk_menu_popup (canvas_menu, NULL, NULL, NULL, NULL, 
			1, 
			gdk_event_key->time);
    }
  

  /* Now, check if the key that was pressed is for an alpha character.
   * If not, return, otherwise we ought to start analyzing it.
   */

  /* First make a duplicate of the key name (that could be either 'a'
     or 'A' or 'Enter') for our use:
  */
  key_name = g_strdup (gdk_keyval_name (gdk_event_key->keyval));
  
  /* Now we can start doing work on it:
     ----------------------------------
     
     First, check that the string is only one character long (that is
     we are keying in a single letter !)
     
     Second, check that the character is indeed a alphabetic
     non-numerical character
     
  */
  
  if (strlen (key_name) > 1)
    {
      g_free (key_name);
      
      return TRUE;
    }
  
  
  if (0 == isalpha (key_name [0]))
    {
      g_free (key_name);
      
      return TRUE;
    }
  
      
  /* At this point we know that we can deal with the character,
     now. We pass the pointer to *this widget so that handling
     will be constrained to the proper widget!
  */
  if (FALSE == polyxmass_seqed_widget_kbd_alpha_handler (key_name,
							 widget))
    {
      g_error (_("%s@%d: failed to handle alpha character\n"),
	       __FILE__, __LINE__);
	  
      return FALSE;
    }

  /* Now we can free the allocated copy of the key' name, because
     we do not need it anymore.
  */

  /* The whole sequence.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_WHOLE_SEQ);
      
      
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_key_release_event (GtkWidget *widget,
					      GdkEventKey *event,
					      gpointer data)
{
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  
  switch (gdk_event_key->keyval)
    {
    case GDK_Shift_L:
    case GDK_Shift_R:
      kbd_shift_down = FALSE;

      /* If the shift was because the user wanted to select a sequence
       * element, now, the selection would be stopped.
       */
      kbd_selecting = FALSE;
      break;
	  
    case GDK_Control_L:
    case GDK_Control_R:
      kbd_control_down = FALSE;
      break;
	  
    case GDK_Alt_L:
    case GDK_Alt_R:
      kbd_alt_down = FALSE;
      break;

    case GDK_Meta_L:
    case GDK_Meta_R:
      kbd_meta_down = FALSE;
      break;
    }
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_alpha_handler (gchar *key_name,
					  GtkWidget *widget)
{
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  PxmMonomer *monomer = NULL;
  
  GPtrArray *refGPA = NULL;

  gchar *err = NULL;

  gint complete = 0;
  

  /* The array to receive all the mononers that have a code
   * completing a given partial monomer code.
   */
  GPtrArray *fillGPA = NULL;
  
  
  /* We MUST have a valid pointer to the seqed_widget.
   */
  g_assert (widget != NULL);
  
  /* We MUST have a valid pointer to the polymer chemistry definition
     context !
  */
  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  refGPA = polchemdefctxt->polchemdef->monomerGPA;
  


  /**************************** ATTENTION ********************************/
  /* The 'fillGPA array of monomer pointers will be fed with pointers
     that reference monomers in the polymer definition context, and
     NOT allocated monomers for our consumption. THUS you'll see that
     the 'fillGPA' array is always g_ptr_array_free () and not
     pxmchem_monomer_GPA_free (). The latter function frees the array
     without freeing the monomer that are pointed to by its items,
     while the second frees everything. Since we ARE NOT the owners of
     the monomers that are pointed to by the items in 'fillGPA' we
     must not use the function pxmchem_monomer_GPA_free () to free the
     array only and not its contents !
  */

  fillGPA = g_ptr_array_new ();

  /* First of all we must ensure that no Ctrl Key is being pressed, otherwise
     we are not actually editing the polymer sequence by adding into it
     monomer codes, but we are editing it using 3 possibilities:

     1. Ctrl + A (select all)

     2. Ctrl + C (copy selection in the clipboard)

     3. Ctrl + X (remove selection and put in the clipboard)

     4. Ctrl + V (paste selection from the clipboard to the insertion
     point)
  */

  /* 
     debug_printf (("key_name is %s\n", key_name));
  */

  if ('C' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_clipboard_copy (widget);
       
       return TRUE;
    }
  
  if ('c' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_clipboard_copy (widget);
       
       return TRUE;
    }

  if ('X' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_cut (widget);
       
       return TRUE;
    }
  
  if ('x' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_cut (widget);
       
       return TRUE;
    }
  
  if ('V' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_clipboard_paste (widget);
       
       return TRUE;
    }
  
  if ('v' == key_name [0] && kbd_control_down == TRUE)
    {
       polyxmass_seqed_widget_clipboard_clipboard_paste (widget);
       
       return TRUE;
    }
  
  if ('A' == key_name [0] && kbd_control_down == TRUE)
    return polyxmass_seqed_widget_select_sequence (widget, -1, -1);
  if ('a' == key_name [0] && kbd_control_down == TRUE)
    return polyxmass_seqed_widget_select_sequence (widget, -1, -1);


  /* Ok, some keys can be used with a Ctrl key down, but these are
     limited to the ones above. If Ctrl is down, we now return because
     we do not want that monomer codes be considered if Ctrl is
     down. Same for the Alt key.
  */
  if (kbd_control_down == TRUE || kbd_alt_down == TRUE)
    return TRUE;
    
  /* We are allowed more than one character per monomer code. So the
   * rule is that the first character of a monomer code must be
   * UPPERcase, while the remaining characters must be lowerCASE. The
   * total number of characters is governed by the
   * polchemdefctxt->polchemdef->codelen variable.
   */  

  /* In the seqed_widget we have three variables that are there to
   * store monomer code elements during the parsing of newly typed
   * monomer codes:
   * 
   * elab_code - where the code that is being elaborated is stored.
   *
   * eval_code - where a code that is apparently completed is stored
   * for its further evaluation as a valid monomer code.
   *
   * kb_typed_chars - where the number of characters being typed to
   * make a monomer code is stored.
   */
  if (0 != isupper (key_name [0]))
    {
      /* The character is uppercase, by necessity this character
       * starts a new monomer code. That means that, if there were an
       * elab_code, we first have to evaluate it. After the
       * evaluation, we reset all variables and feed the elab_code
       * variable this uppercase character so as to start fresh with
       * the parsing of a new monomer code.
       *
       * We can know if a code was elaborating because if so the
       * kb_typed_chars variable should be different than 0.
       */
      if (PXM_SEQED_WIDGET (widget)->kb_typed_chars > 0)
	{
	  /* Some characters were typed already. The new uppercase
	   * character that we have read is starting a new monomer
	   * code, which also means that it is ending the elab_code.
	   */
	  strcpy (PXM_SEQED_WIDGET (widget)->eval_code, 
		  PXM_SEQED_WIDGET (widget)->elab_code);
	  
	  if (FALSE == polyxmass_seqed_widget_kbd_evaluate_code 
	      (PXM_SEQED_WIDGET (widget)->eval_code,
	       PXM_SEQED_WIDGET (widget)->last_point_1_idx,
	       widget))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     PXM_SEQED_WIDGET (widget)->eval_code);

	      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->
					     error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	     
	      /* The elab_code is found erroneous, so we give the user
	       * a chance to modify it so that it complies with the
	       * codes of the polymer chemistry definition with which
	       * we are working. To do that we remove the last
	       * character from the elaborating code, so that the user
	       * may change it into another character that completes a
	       * valid monomer code. When this is done, we just ask
	       * that the shortened elaborating monomer code be echoed
	       * to the proper GtkEntry widget.
	       */
	      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	      
	      /* Because the code that was elaborated is invalid, it
	       * makes no sense that we take into account the
	       * uppercase character that we were parsing in the first
	       * place, so we just return.
	       */

	      /* See ATTENTION notice above to understand why we do
		 not pxmchem_monomer_GPA_free () the array. In a word,
		 we do not own the monomers in this array.
	      */
	      g_ptr_array_free (fillGPA, TRUE);

	      return TRUE;
	    }
	  
	  /* If we are here, that means that the evaluation of the
	   * eval_code has been successful, so we have to do the
	   * following:
	   * 
	   * 1. emtpy the eval_code, so that we later can use it
	   fresh.
	   *
	   *
	   * 2. set to 0 the count of typed characters, since we have
	   finished dealing witht he previously elaborating code.
	  */
	  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);
	  
	  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
	}

      /* At this point we have finished dealing with the case that the
       * uppercase character is effectively ending the elaboration of
       * a monomer code. Because the uppercase character is starting a
       * new elaborated code, we just copy this character in the
       * elab_code at position 0 [which is the pointer of elab_code +
       * PXM_SEQED_WIDGET (widget)->kb_typed_chars (which is 0)]:
       */
      memset (PXM_SEQED_WIDGET (widget)->elab_code + 
	      PXM_SEQED_WIDGET (widget)->kb_typed_chars,
	      key_name [0], 1);
      
      /* Increment the count of typed characters.
       */
      PXM_SEQED_WIDGET (widget)->kb_typed_chars++;
    
      
      /* And now effectively deal with the uppercase character that
       * was parsed in the first place: check if this uppercase
       * character already characterizes with no ambiguity a monomer
       * (ie corresponds already to a unique monomer code).
       */
      complete = 
	pxmchem_monomer_get_ptrs_by_completing_code (PXM_SEQED_WIDGET (widget)->
						     elab_code,
						     refGPA, fillGPA);
      
      if (complete == 0)
	{
	  /* No one single monomer in refGPA does have a monomer code
	   * starting with the currently parsed uppercase letter. This
	   * is an error condition.
	   */
	  err = g_strdup_printf (_("code already invalid: '%s'"),
				 PXM_SEQED_WIDGET (widget)->elab_code);
	  
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      err);
	  
	  g_free (err);

	  polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      if (complete == 1)
	{
	  /* There is no degeneration here, the uppercase character
	   * that was entered corresponds without ambiguity to a
	   * single monomer so that we can go through all the
	   * evaluation process right now.
	   */
	  monomer = g_ptr_array_index (fillGPA, 0);
	  
	  if (FALSE == 
	      polyxmass_seqed_widget_kbd_evaluate_code (monomer->code,
							PXM_SEQED_WIDGET (widget)->
							last_point_1_idx,
							widget))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     monomer->code);
	      
	      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->
					     error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	      
	      /* See ATTENTION notice above to understand why we do
		 not pxmchem_monomer_GPA_free () the array. In a word,
		 we do not own the monomers in this array.
	      */
	      g_ptr_array_free (fillGPA, TRUE);
	  
	      return TRUE;
	    }

	  /* If we are here that means that we successfully evaluated
	   * the monomer->code, which means that we can reinitialize
	   * some variables for the next character to be entered to be
	   * handled freshly.
	   */
	  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
	  
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  

	  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
							   TRUE);

 
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");

	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && (PXM_SEQED_WIDGET (widget)->kb_typed_chars 
	      != polchemdefctxt->polchemdef->codelen))
	{
	  /* The monomer code that is represented by
	   * PXM_SEQED_WIDGET (widget)->elab_code corresponds to more than one valid
	   * monomer code. There is ambiguity, and there is still
	   * place for another discriminating character, so we just do
	   * nothing special here, awaiting for another character that
	   * may perform the discrimitation between all the possible
	   * monomer codes.
	   */
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && (PXM_SEQED_WIDGET (widget)->kb_typed_chars 
	      == polchemdefctxt->polchemdef->codelen))
	{
	  /* The code that is being elaborated has now the maximum
	   * number of authorized character. That means that we have
	   * to consider it for validity without waiting for any
	   * character more. Let's evaluate it.
	   */
	  strcpy (PXM_SEQED_WIDGET (widget)->eval_code, 
		  PXM_SEQED_WIDGET (widget)->elab_code);

	  if (FALSE == 
	      polyxmass_seqed_widget_kbd_evaluate_code (PXM_SEQED_WIDGET (widget)->
							eval_code,
							PXM_SEQED_WIDGET (widget)->
							last_point_1_idx,
							widget))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     PXM_SEQED_WIDGET (widget)->eval_code);
	      
	      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->
					     error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	      
	      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	      
	      /* See ATTENTION notice above to understand why we do
		 not pxmchem_monomer_GPA_free () the array. In a word,
		 we do not own the monomers in this array.
	      */
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }
	  
	  /* At this point we have successfully evaluated the eval_code,
	   * reinitialize the usual variables:
	   */
	  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
	  
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
							   TRUE);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
    }
  /* end of:
     if (0 != isupper (key_name [0]))
  */
  
  else /* character is lowercase */
    {
      /* We are pursuing the elaboration of a monomer code.
       */
      if  (PXM_SEQED_WIDGET (widget)->kb_typed_chars == 0)
	{
	  err = g_strdup_printf (_("cannot start a code with lowercase: '%s'"),
				 key_name);
	  
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      err);
	  
	  g_free (err);
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      /* Apparently we are pursuing effectively the elaboration of a
       * monomer code, add the character to the elab_code.
       */
      memset (PXM_SEQED_WIDGET (widget)->elab_code 
	      + PXM_SEQED_WIDGET (widget)->kb_typed_chars,
	      key_name [0], 1);
      
      /* Increment the count of typed characters.
       */
      PXM_SEQED_WIDGET (widget)->kb_typed_chars++;
      
      /* By the addition of this new character to the elab_code, we
       * may have gotten an unambiguous monomer code. Let us check
       * this immediately.
       */
      complete = 
	pxmchem_monomer_get_ptrs_by_completing_code (PXM_SEQED_WIDGET (widget)->
						     elab_code,
						     refGPA, fillGPA);
      
      if (complete == 0)
	{
	  /* No one single monomer in refGPA does have a monomer code
	   * resembling PXM_SEQED_WIDGET (widget)->elab_code. This is an error
	   * condition.
	   */
	  err = g_strdup_printf (_("code already invalid: '%s'"),
				 PXM_SEQED_WIDGET (widget)->elab_code);
	  
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      err);
	  
	  g_free (err);

	  polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete == 1)
	{
	  /* There is no degeneration here, the uppercase character
	   * that was entered corresponds without ambiguity to a
	   * single monomer so that we can go through all the
	   * evaluation process right now.
	   */
	  monomer = g_ptr_array_index (fillGPA, 0);
	  
	  if (FALSE == polyxmass_seqed_widget_kbd_evaluate_code (monomer->code,
								 PXM_SEQED_WIDGET (widget)->
								 last_point_1_idx,
								 widget))
	    {
	      err = g_strdup_printf (_("failed eval code: '%s'"),
				     monomer->code);
	      
	      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->
					     error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	      
	      /* See ATTENTION notice above to understand why we do
		 not pxmchem_monomer_GPA_free () the array. In a word,
		 we do not own the monomers in this array.
	      */
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }

	  /* If we are here that means that we successfully evaluated
	   * the monomer->code, which means that we can reinitialize
	   * some variables for the next character to be entered to be
	   * handled freshly.
	   */
	  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
	  
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
							   TRUE);

	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      if (complete > 1 
	  && (PXM_SEQED_WIDGET (widget)->kb_typed_chars 
	      != polchemdefctxt->polchemdef->codelen))
	{
	  /* The monomer code that is represented by
	   * PXM_SEQED_WIDGET (widget)->elab_code corresponds to more than one valid
	   * monomer code. There is ambiguity, and there is still
	   * place for another discriminating character, so we just do
	   * nothing special here, awaiting for another character that
	   * may perform the discrimitation between all the possible
	   * monomer codes.
	   */
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && (PXM_SEQED_WIDGET (widget)->kb_typed_chars 
	      == polchemdefctxt->polchemdef->codelen))
	{
	  /* The code that is being elaborated has now the maximum
	   * number of authorized character. That means that we have
	   * to consider it for validity without waiting for any
	   * character more. Let's evaluate it.
	   */
	  strcpy (PXM_SEQED_WIDGET (widget)->eval_code, PXM_SEQED_WIDGET (widget)->
		  elab_code);

	  if (FALSE == 
	      polyxmass_seqed_widget_kbd_evaluate_code (PXM_SEQED_WIDGET (widget)->
							eval_code,
							PXM_SEQED_WIDGET (widget)->
							last_point_1_idx,
							widget))
	    {
	      err = g_strdup_printf (_("failed eval code: '%s'"),
				     PXM_SEQED_WIDGET (widget)->eval_code);
	      
	      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->
					     error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	      
	      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
	      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	      
	      /* See ATTENTION notice above to understand why we do
		 not pxmchem_monomer_GPA_free () the array. In a word,
		 we do not own the monomers in this array.
	      */
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }
	  
	  /* At this point we have successfully evaluated the eval_code,
	   * reinitialize the usual variables:
	   */
	  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
	  
	  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
	  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
							   TRUE);

 
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			      "");
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
    }
  /* end of:
     else character is lowercase
  */
  
  return TRUE; 
}


gboolean
polyxmass_seqed_widget_kbd_backspace_handler (GtkWidget *widget)
{
  gint idx = -1;
  

  g_assert (widget != NULL);
  
  
  /* There are different situations here:
   * 
   * If there is an active selection in the seqed_widget, we should
   * destroy that selected oligomer first, and return right after
   * that.
   * 
   * If there is no active selection, then we have to remove the
   * monomer that is located prior to the current cursor position.
   */
  
  /*
    debug_printf (("last_point_1_idx = %d, sel_mnm_idx1 = %d, "
    "sel_mnm_idx2 = %d\n",
    PXM_SEQED_WIDGET (widget)->last_point_1_idx,
    PXM_SEQED_WIDGET (widget)->sel_mnm_idx1, PXM_SEQED_WIDGET (widget)->sel_mnm_idx2));
  */
  
  if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
							     NULL,
							     NULL))
    {
      /* There is no selection. We are thus ready to perform the removal
       * of a monomer in the sequence editor, if the cursor is not located
       * left of the beginning of the sequence.
       */
  
      /* Imagine we have the following sequence:
       * 
       * A R N ^ D D E Q, with the cursor being locate where ^ is.
       *
       * In this situation, the variable PXM_SEQED_WIDGET
       * (widget)->last_point_1_idx == 3, which is actually the index
       * of the monomer located at the right of the 'N'. If we issue a
       * backspace now, what we want to remove is not the monomer of
       * index 3, but the monomer at the left of the cursor position,
       * which is 'N', thus monomer 3-1 = 2. So we should issue a
       * removal order with an index of value (PXM_SEQED_WIDGET
       * (widget)->last_point_1_idx - 1).
       * 
       * But what if the cursor is at the following position:
       * 
       * ^ A R N D D E Q 
       *
       * It would not make sense to send a removal order at all:
       */
      if (PXM_SEQED_WIDGET (widget)->last_point_1_idx == 0)
	return TRUE;
      
      idx = PXM_SEQED_WIDGET (widget)->last_point_1_idx - 1;
      
      polyxmass_seqed_widget_remove_monomer (widget, idx);
      
      /* Each time a monicon is removed (towards the left, lesser
       * indexes) from the GPtrArray, the last_point_1_idx is
       * decremented by one unit BUT only if the removed monomer is
       * not removed from the first (index 0) GPtrArray index !!!
       */
      if (PXM_SEQED_WIDGET (widget)->last_point_1_idx > 0)
	PXM_SEQED_WIDGET (widget)->last_point_1_idx--;
      
      polyxmass_seqed_widget_redraw_sequence (widget);
      
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
						       TRUE);
      
    }
  /* end of:
     if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 == -1 
     || PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 == -1)
  */
  
  else
    {
      /* An oligomer is currently selected.
       */
      if (-1 == polyxmass_seqed_widget_remove_selected_oligomer (widget))
	{
	  g_critical (_("%s@%d: failed to remove the selected oligomer\n"),
		      __FILE__, __LINE__);
	  
	  return FALSE;
	}
      
      polyxmass_seqed_widget_redraw_sequence (widget);
      
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget),  
						       TRUE);

    }

  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_BOTH);

  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_delete_handler (GtkWidget *widget)
{
  gint idx = -1;
  gint length = 0;
  
  PxmPolymer *polymer = NULL;
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  

  /* There are different situations here:
   * 
   * If there is an active selection in the sequence editor, we should
   * destroy that selected oligomer first, and return after that.
   * 
   * If there is no active selection, then we have to remove the
   * monomer that is located after the current cursor position.
   */
  
  if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
							     NULL,
							     NULL))
    {
      /* There is no selection. We are thus ready to perform the removal
       * of a monomer in the sequence editor, if the cursor is not located
       * right of the last monomer in the sequence.
       */
  
      /* Imagine we have the following sequence:
       * 
       * A R N ^ D D E Q, with the cursor being locate where ^ is.
       *
       * In this situation, the variable PXM_SEQED_WIDGET (widget)->last_point_1_idx ==
       * 3, which is actually the index of the monomer located at the
       * right of the 'N'. If we issue a delete now, what we want to
       * remove is precisely the monomer of index 3. So we should
       * issue a removal order with an index of value
       * PXM_SEQED_WIDGET (widget)->last_point_1_idx.
       * 
       * But what if the cursor is at the following position:
       * 
       * A R N D D E Q ^
       *
       * It would not make sense to send a removal order, because we
       * would be asking to remove a monomer at index 7 which does not
       * exist !
       */
      if (PXM_SEQED_WIDGET (widget)->last_point_1_idx >= length)
	return TRUE;
      
      idx = PXM_SEQED_WIDGET (widget)->last_point_1_idx;
      
      polyxmass_seqed_widget_remove_monomer (widget, idx);
      
      /* In the present case we do not need to modify the
       * PXM_SEQED_WIDGET (widget)->last_point_1_idx because we
       * removed a monomer right of the cursor.
       */

      polyxmass_seqed_widget_redraw_sequence (widget);
      
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
						       TRUE);

    }  
  else
    {
      /* An oligomer is currently selected.
       */
      if (-1 == polyxmass_seqed_widget_remove_selected_oligomer (widget))
	{
	  g_error (_("%s@%d: failed to remove the selected oligomer\n"),
		   __FILE__, __LINE__);
	  
	  return FALSE;
	}

      polyxmass_seqed_widget_redraw_sequence (widget);
      
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
						       TRUE);

    }
  


  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_BOTH);

  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_right_handler (GtkWidget *widget)
{
  gint length = 0;
  
  PxmRect rect;
  
  PxmPolymer *polymer = NULL;
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  

  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx >= length)
    return TRUE;

  /* Be careful that if the user is pressing the right arrow key in
   * combination with a shift down, that means that she wants to
   * select a portion of polymer sequence.
   */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
	
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
	  
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx++;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;

	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx++;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1 
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* And now increment the last pointed monomer index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx++;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, 
						  COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
  
  if (rect.y1 > (PXM_SEQED_WIDGET (widget)->canvas_y_offset 
		 + PXM_SEQED_WIDGET (widget)->sw_height 
		 - PXM_SEQED_WIDGET (widget)->monicon_size))
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset +
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }

  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);
  

  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
      
  /* In case there was a selection going on, let's inform that this is 
     going on, so that the selected sequence is available in the
     PRIMARY selection.
  */
  polyxmass_seqed_widget_clipboard_primary_copy (widget);  
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_left_handler (GtkWidget *widget)
{
  gint length = 0;
  
  PxmPolymer *polymer = NULL;
  
  PxmRect rect;
  
  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  

  

  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx <=0)
    return TRUE;

  /* Be careful that if the user is pressing the right arrow key in
   * combination with a shift down, that means that she wants to
   * select a portion of polymer sequence.
   */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
	  

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx--;

	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx--;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1 
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* And now just decrement the last pointed monomer index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx--;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, 
						  COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
						 canvas),
				   &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
				   &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
  
  if (rect.y1 < (PXM_SEQED_WIDGET (widget)->canvas_y_offset 
		 + PXM_SEQED_WIDGET (widget)->monicon_size))
    {
      /* We are near the top of viewable sequence, so we ask
       * for a scroll to happen towards lower monomer indexes
       * (scroll upwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset -
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);
  


  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
      
  /* In case there was a selection going on, let's inform that this is 
     going on, so that the selected sequence is available in the
     PRIMARY selection.
  */
  polyxmass_seqed_widget_clipboard_primary_copy (widget);

  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_home_handler (GtkWidget *widget)
{
  gint length = 0;
  gint line = 0;
  gint target_idx = 0;
    
  PxmPolymer *polymer = NULL;
  
  PxmRect rect;
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  
  
  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx <=0)
    return TRUE;


  /* There are two cases:

  1. if the control key is not pressed, the target position is the
  beginning of the current line. 

  2. if the control key is pressed, the target position is the 
  beginning of the whole sequence.
  */

  if (kbd_control_down == TRUE)
    target_idx = 0;
  else
    {
      /* Calculate the line number (position not index) on which the cursor
	 is currently located.
      */
      line = 
	((PXM_SEQED_WIDGET (widget)->last_point_1_idx -1) / 
	 PXM_SEQED_WIDGET (widget)->monicons_per_line) + 1;
      
      target_idx = PXM_SEQED_WIDGET (widget)->monicons_per_line * (line - 1);
    }
  

  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the beginning of the
     current line.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  	    	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* We are changing the last point index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);

  /*
    debug_printf (("cursor y : %f\n", PXM_SEQED_WIDGET (widget)->cursor->y));
  */

  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_ensure_selection_and_cursor_visible 
    (PXM_SEQED_WIDGET (widget)->sw, NULL, widget);
  
  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
      
      
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_end_handler (GtkWidget *widget)
{
  gint length = 0;
  gint line = 0;
  gint target_idx = 0;
    
  PxmRect rect;

  PxmPolymer *polymer = NULL;
  
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  
  
  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx >= length)
    return TRUE;


  /* There are two cases:

  1. if the control key is not pressed, the target position is the
  end of the current line. 

  2. if the control key is pressed, the target position is the 
  end of the whole sequence.
  */

  if (kbd_control_down == TRUE)
    target_idx = length;
  else
    {
      /* Calculate the line number (position not index) on which the cursor
	 is currently located.
      */
      line = 
	((PXM_SEQED_WIDGET (widget)->last_point_1_idx -1) 
	 / PXM_SEQED_WIDGET (widget)->monicons_per_line) + 1;
      
      target_idx = PXM_SEQED_WIDGET (widget)->monicons_per_line * line;
    }
  

  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the end of the current
     line.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1 
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* We are changing the last point index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);

  /*
    debug_printf (("cursor y : %f\n", PXM_SEQED_WIDGET (widget)->cursor->y));
  */

  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_ensure_selection_and_cursor_visible
    (PXM_SEQED_WIDGET (widget)->sw, NULL, widget);
  
      
  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
      
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_up_handler (GtkWidget *widget)
{
  gint target_idx = -1;
  gint length = 0;
    
  PxmRect rect;
  
  PxmPolymer *polymer = NULL;
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  
  
  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx <=0)
    return TRUE;

  /* We want to draw the cursor one line above the one where it sits
     right now, ideally in the exact same column. That means going
     back in the sequence by monicons_per_line positions.
  */

  /* Calculate the number of monicons per line:
   */
  PXM_SEQED_WIDGET (widget)->monicons_per_line = 
    (PXM_SEQED_WIDGET (widget)->sw_width
     - PXM_SEQED_WIDGET (widget)->left_margin)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  target_idx = 
    PXM_SEQED_WIDGET (widget)->last_point_1_idx 
    - PXM_SEQED_WIDGET (widget)->monicons_per_line;

  if (target_idx < 0)
    return TRUE;
  
  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the line above, ideally
     in the exact same column. That means going back in the sequence
     by monicons_per_line positions.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1 
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* We are changing the last point index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, 
						  COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
  
  if (rect.y1 < (PXM_SEQED_WIDGET (widget)->canvas_y_offset 
		 + PXM_SEQED_WIDGET (widget)->monicon_size))
    {
      /* We are near the top of viewable sequence, so we ask
       * for a scroll to happen towards lower monomer indexes
       * (scroll upwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset -
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);
  
  /*
    debug_printf (("cursor y : %f\n", PXM_SEQED_WIDGET (widget)->cursor->y));
  */



  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_down_handler (GtkWidget *widget)
{
  gint target_idx = -1;
  gint length = 0;
  
  PxmRect rect;
  
  PxmPolymer *polymer = NULL;
  

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  

  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx >= length)
    return TRUE;

  /* We want to draw the cursor one line below the one where it sits
   * right now, ideally in the exact same column. That means going
   forward in the sequence by monicons_per_line positions.
  */

  /* Calculate the number of monicons per line:
   */
  PXM_SEQED_WIDGET (widget)->monicons_per_line = 
    (PXM_SEQED_WIDGET (widget)->sw_width
     - PXM_SEQED_WIDGET (widget)->left_margin)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  target_idx = PXM_SEQED_WIDGET (widget)->last_point_1_idx 
    + PXM_SEQED_WIDGET (widget)->monicons_per_line;

  if (target_idx > length)
    target_idx = length;
  
  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the line below, ideally
     in the exact same column. That means going forward in the sequence
     by monicons_per_line positions.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1 
	  && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1)
	polyxmass_seqed_widget_remove_selection_polygon (widget);
      
      /* We are changing the last point index.
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						      last_point_1_idx,
						      widget,
						      &rect,
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, 
						  COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
  
  if (rect.y1 > (PXM_SEQED_WIDGET (widget)->canvas_y_offset 
		 + PXM_SEQED_WIDGET (widget)->sw_height 
		 - PXM_SEQED_WIDGET (widget)->monicon_size))
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset +
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);
  
  /*
    debug_printf (("cursor y : %f\n", PXM_SEQED_WIDGET (widget)->cursor->y));
  */

  /* The selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_SELEC_SEQ);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_page_up_handler (GtkWidget *widget)
{
  gint length = 0; 
  gint y_new_scroll = 0;
  gint count_vis_lines = 0;
  gint target_idx = -1;
    
  PxmRect rect;

  PxmPolymer *polymer = NULL;
  


  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  
  
  /* There are two cases:
     
  1. No shift key is down, in which case we only make a scroll of the
  page. That scroll won't change the insertion point.

  2. kbd_shift_down is true, in which case we select the sequence portion
  that goes from "here" to the same column, a page up.
  */

  /* If we want to scroll one page up, that means that we want to
     display some lines that currently are hidden on top of the
     window. But how many lines? The exact number of lines that
     are currently displayed, which is calculatable:
  */
  count_vis_lines = PXM_SEQED_WIDGET (widget)->sw_height 
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  /* Calculate the number of monicons per line:
   */
  PXM_SEQED_WIDGET (widget)->monicons_per_line = 
    (PXM_SEQED_WIDGET (widget)->sw_width
     - PXM_SEQED_WIDGET (widget)->left_margin)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  

  if (kbd_shift_down == FALSE)
    {
      /* Get the scroll offsets. The y scroll offset is the number of
	 pixels from start of the sequence picture to the first
	 visible area. For example, if four rows of monomers are
	 hidden above the currently visible sequence, that each
	 monicon has a size of 100 pixels, we'll get a y scroll offset
	 of 400.
      */
      gnome_canvas_get_scroll_offsets 
	(GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	 &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	 &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
      
      /* Since we are scrolling upwards, that means that we want to
	 diminish the amount of hidden pixels above the visible region
	 of the window. That means that we want to decrement this
	 number of pixels by the amount of (count_vis_lines *
	 PXM_SEQED_WIDGET (widget)->monicon_size).
      */
      y_new_scroll = PXM_SEQED_WIDGET (widget)->canvas_y_offset - 
	(count_vis_lines * PXM_SEQED_WIDGET (widget)->monicon_size);
  
      if (y_new_scroll < 0)
	y_new_scroll = 0;
  
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      y_new_scroll);
      
      
      /* The selection does not change and cursor point neither, so do
	 not ask that masses be recalculated.
      */
      return TRUE;
    }
  else 
    /* (kbd_shift_down == TRUE)
     */
    {
      /* The selection should go from current cursor point to the same
	 column, one page up. That is the number of visibles lines
	 upwards.
      */
      target_idx = PXM_SEQED_WIDGET (widget)->last_point_1_idx - 
	(count_vis_lines * PXM_SEQED_WIDGET (widget)->monicons_per_line) ;
      
      if (target_idx < 0)
	target_idx = 0;


      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);


      /* The way we scroll the window depends on the manner in which
	 the selection is performed: 

	 1. if PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 <
	 PXM_SEQED_WIDGET (widget)->sel_mnm_idx2, then that means that
	 we are selecting from top to bottom, which is that we want to
	 have the bottom horizontal line of the selection drawn at the
	 bottom of the visible area of the canvas (Note the
	 COORDSYS_SW param to the pixel_coord function call !).

	 2. if PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 >
	 PXM_SEQED_WIDGET (widget)->sel_mnm_idx2, then that means that
	 we are selecting from bottom to top, which means that we want
	 to display the upper horizontal line of the selection polygon
	 at the top of the visible area of the canvas (Note the
	 COORDSYS_NW param to the pixel_coord function call !).
      */

      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  < PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	{
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_SW, 
							  COORDSYS_SW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
	  
	  y_new_scroll = rect.y1 
	    - (count_vis_lines * PXM_SEQED_WIDGET (widget)->monicon_size);
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     PXM_SEQED_WIDGET (widget)->canvas_x_offset,
	     y_new_scroll);
	}
      
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	{
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
	  
	  y_new_scroll = rect.y1;
	  
	  if (y_new_scroll < 0)
	    y_new_scroll = 0;
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     PXM_SEQED_WIDGET (widget)->canvas_x_offset,
	     y_new_scroll);
	}
      
      /* Finally we draw the cursor to the new cursor point location !
       */
      polyxmass_seqed_widget_draw_cursor (widget);
      
      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  



      /* The selection.
       */
      polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
							  PXM_MASSCALC_TARGET_SELEC_SEQ);
 
      return TRUE;
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_page_down_handler (GtkWidget *widget)
{
  gint length = 0;
  gint count_max_pixels = 0;
  gint y_new_scroll = 0;
  gint count_lines = 0;
  gint count_vis_lines = 0;
  gint target_idx = -1;
      
  PxmRect rect;

  PxmPolymer *polymer = NULL;
  


  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;
  
  
  /* There are two cases:
     
  1. No shift key is down, in which case we only make a scroll of the
  page. That scroll won't change the insertion point.

  2. kbd_shift_down is true, in which case we select the sequence portion
  that goes from "here" to the same column, a page down.
  */

  /* If we want to scroll one page down, that means that we want to
     display some lines that currently are hidden on bottom of the
     window. But how many lines? The exact number of lines that
     are currently displayed, which is calculatable:
  */
  count_vis_lines = 
    PXM_SEQED_WIDGET (widget)->sw_height 
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  /* Calculate the number of monicons per line:
   */
  PXM_SEQED_WIDGET (widget)->monicons_per_line = 
    (PXM_SEQED_WIDGET (widget)->sw_width
     - PXM_SEQED_WIDGET (widget)->left_margin)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  

  if (kbd_shift_down == FALSE)
    {
      /* Get the scroll offsets. The y scroll offset is the number of
	 pixels from start of the sequence picture to the first
	 visible area. For example, if four rows of monomers are
	 hidden above the currently visible sequence, that each
	 monicon has a size of 100 pixels, we'll get a y scroll offset
	 of 400.
      */
      gnome_canvas_get_scroll_offsets 
	(GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	 &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	 &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
      
      /* Since we are scrolling downwards, that means that we want to
	 increase the amount of hidden pixels above the visible region
	 of the window. That means that we want to increment this
	 number of pixels by the amount of (count_vis_lines *
	 PXM_SEQED_WIDGET (widget)->monicon_size).
      */
      y_new_scroll = PXM_SEQED_WIDGET (widget)->canvas_y_offset + 
	(count_vis_lines * PXM_SEQED_WIDGET (widget)->monicon_size);
  
      /* Test if the y_new_scroll position is not greater than the
	 number of pixels of the entire sequence picture!
      */
      count_lines = (length - 1) / PXM_SEQED_WIDGET (widget)->monicons_per_line;
      
      count_max_pixels = count_lines * PXM_SEQED_WIDGET (widget)->monicon_size;
      
      if (y_new_scroll > count_max_pixels)
	y_new_scroll = count_max_pixels;
      
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      y_new_scroll);
      
      
      /* The selection does not change and cursor point neither, so do
	 not ask that masses be recalculated.
      */
      return TRUE;
    }
  else 
    /* (kbd_shift_down == TRUE)
     */
    {
      /* The selection should go from current cursor point to the same
	 column, one page down. That is the number of visibles lines
	 downwards.
      */
      target_idx = PXM_SEQED_WIDGET (widget)->last_point_1_idx + 
	(count_vis_lines * PXM_SEQED_WIDGET (widget)->monicons_per_line) ;
      
      if (target_idx > length)
	target_idx = length;


      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
								 NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	  */	  
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  PXM_SEQED_WIDGET (widget)->last_point_1_idx = target_idx;

	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_NW, 
							  COORDSYS_NW);

	  PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x2;
	  PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxmass_seqed_widget_draw_sel_polygon (widget);


      /* The way we scroll the window depends on the manner in which
	 the selection is performed: 

	 1. if PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 <
	 PXM_SEQED_WIDGET (widget)->sel_mnm_idx2, then that means that
	 we are selecting from top to bottom, which is that we want to
	 have the bottom horizontal line of the selection drawn at the
	 bottom of the visible area of the canvas (Note the
	 COORDSYS_SW param to the pixel_coord function call !).

	 2. if PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 >
	 PXM_SEQED_WIDGET (widget)->sel_mnm_idx2, then that means that
	 we are selecting from bottom to top, which means that we want
	 to display the upper horizontal line of the selection polygon
	 at the top of the visible area of the canvas (Note the
	 COORDSYS_NW param to the pixel_coord function call !).
      */

      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  < PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	{
	  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
							  last_point_1_idx,
							  widget,
							  &rect,
							  COORDSYS_SW, 
							  COORDSYS_SW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
	  
	  y_new_scroll = rect.y1 
	    - (count_vis_lines * PXM_SEQED_WIDGET (widget)->monicon_size);
        
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     PXM_SEQED_WIDGET (widget)->canvas_x_offset,
	     y_new_scroll);
	}
      
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	{
	  polyxmass_seqed_widget_monicon_get_pixel_coord 
	    (PXM_SEQED_WIDGET (widget)->
	     last_point_1_idx,
	     widget,
	     &rect,
	     COORDSYS_NW, 
	     COORDSYS_NW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     &PXM_SEQED_WIDGET (widget)->canvas_x_offset, 
	     &PXM_SEQED_WIDGET (widget)->canvas_y_offset);
	  
	  y_new_scroll = rect.y1;

	  if (y_new_scroll < 0)
	    y_new_scroll = 0;
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     PXM_SEQED_WIDGET (widget)->canvas_x_offset,
	     y_new_scroll);
	}
      
      /* Finally we draw the cursor to the new cursor point location !
       */
      polyxmass_seqed_widget_draw_cursor (widget);
      
      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);  



      /* The selection.
       */
      polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
							  PXM_MASSCALC_TARGET_SELEC_SEQ);
 
      return TRUE;
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_enter_handler (GtkWidget *widget)
{

  g_assert (widget != NULL);
  
  polyxmass_seqed_widget_kbd_force_complete_elab_code (widget);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_del_last_char_elab_code (GtkWidget *widget)
{
  g_assert (widget != NULL);
  
  
  /* We have to go back one character in the currently elaborating
   * code.  To do this, we need to have at least one character in the
   * elab_code!
   */
  if (PXM_SEQED_WIDGET (widget)->kb_typed_chars == 0)
    return TRUE;
  
  /* Calculate the index where we will erase the character by putting
   * over it a 'x\0' character.
   */
  /* Imagine that 1 character was in elab_code (ie . PXM_SEQED_WIDGET
   * (widget)->kb_typed_chars = 1). That would mean that
   * (PXM_SEQED_WIDGET (widget)->elab_code + 0) contains that
   * character. And this is precisely the slot that we want to
   * anihilate.
   * 
   * So the logic is that if we have type n characters, then we want to 
   * anihilate the (n-1) slot.
   *
   * Since we also want to decrement PXM_SEQED_WIDGET
   * (widget)->kb_typed_chars, so that it still reflects correctly the
   * number of characters that have been typed in the elab_code, we
   * could use the C construct that first decrements the variable, and
   * next uses it.
   */
  memset (PXM_SEQED_WIDGET (widget)->elab_code 
	  + (--PXM_SEQED_WIDGET (widget)->kb_typed_chars), '\x0', 1);

  /* To maintain the systematic consistency between the contents of 
   * PXM_SEQED_WIDGET (widget)->elab_code and what is displayed in the elab_code_entry
   * let's call the function below:
   */
  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_echo_elab_code (GtkWidget *widget)
{
  g_assert (widget != NULL);

  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->elab_code_entry),
		      PXM_SEQED_WIDGET (widget)->elab_code);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_check_complete_elab_code (GtkWidget *widget)
{
  /* The array to receive all the mononers that have a code
   * completing a given partial monomer code.
   */
  GPtrArray *fillGPA = NULL;
  
  gint complete = 0;
  
  PxmMonomer *monomer = NULL;
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  g_assert (widget != NULL);
  
  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  /**************************** ATTENTION ********************************/
  /* The 'fillGPA array of monomer pointers will be fed with pointers
     that reference monomers in the polymer definition context, and
     NOT allocated monomers for our consumption. THUS you'll see that
     the 'fillGPA' array is always g_ptr_array_free () and not
     pxmchem_monomer_GPA_free (). The latter function frees the array
     without freeing the monomer that are pointed to by its items,
     while the second frees everything. Since we ARE NOT the owners of
     the monomers that are pointed to by the items in 'fillGPA' we
     must not use the function pxmchem_monomer_GPA_free () to free the
     array only and not its contents !
  */
  fillGPA = g_ptr_array_new ();
  


  /* Let us say that we have 3-lett monomer codes in the polymer
   * definition, of which Gly, Gl, G. We have entered G and this
   * function is called. What will happen here is that we'll first
   * check how many codes are available as codes which comply with the
   * first G chracter.  This function calls
   * pxmchem_monomer_get_ptrs_by_completing_code () name of called
   * function and gets back the pointers to Gly, Gl, and G in a
   * GPtrArray (not allocated monomers, only a reference to them in
   * the polchemdef->monomerGPA!).  Now imagine we have entered Gl and
   * this function is called. We would get Gl and Gly back. If we had
   * previously entered Gly, and this function was called, then we
   * would have gotten only one monomer code as an available code to
   * comply with Gly: Gly (!). So we just would evaluate this code. In
   * the two former cases, we would have displayed a list with the
   * available completions.
   */
  strcpy (PXM_SEQED_WIDGET (widget)->eval_code, PXM_SEQED_WIDGET (widget)->elab_code);
  
  complete = 
    pxmchem_monomer_get_ptrs_by_completing_code (PXM_SEQED_WIDGET (widget)->eval_code, 
						 polchemdef->monomerGPA, 
						 fillGPA);
  if (complete == 0)
    {
      g_error (_("%s@%d: code is already invalid: '%s'\n"),
	       __FILE__, __LINE__, PXM_SEQED_WIDGET (widget)->eval_code);
      
      /* See ATTENTION notice above to understand why we do
	 not pxmchem_monomer_GPA_free () the array. In a word,
	 we do not own the monomers in this array.
      */
      g_ptr_array_free (fillGPA, TRUE);
      
      return FALSE;
    }
  
  if (complete == 1)
    {
      /* Only one monomer can complete with current elab_code, so that
       * it is easy: we just evaluate it now.
       */
      monomer = g_ptr_array_index (fillGPA, 0);
      
      if (FALSE == polyxmass_seqed_widget_kbd_evaluate_code (monomer->code,
							     PXM_SEQED_WIDGET (widget)->
							     last_point_1_idx,
							     widget))
	{
	  g_error (_("%s@%d: failed to evaluate code: '%s'\n"),
		   __FILE__, __LINE__, monomer->code);
	  
	  /* See ATTENTION notice above to understand why we do
	     not pxmchem_monomer_GPA_free () the array. In a word,
	     we do not own the monomers in this array.
	  */
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return FALSE;
	}
      
      /* Now that we have successfully evaluated monomer->code, we
       * reinitialize some variables for next typed character.
       */
      memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
	      polchemdefctxt->polchemdef->codelen + 1);
      
      memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
	      polchemdefctxt->polchemdef->codelen + 1);
      
      PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
      
      polyxmass_seqed_widget_kbd_echo_elab_code (widget);
    }
  
  if (complete > 1)
    {
      /* There are more than one monomer that have a code that is 
       * compatible with eval_code. So we should display these monomers
       * in a graphical list packed with a button for each monomer so
       * that the user may see them and click on one of them to finally
       * make a "graphical" completion.
       */
      polyxmass_code_completions_mnm_code_show_list (PXM_SEQED_WIDGET (widget), fillGPA);
      
      /* See ATTENTION notice above to understand why we do
	 not pxmchem_monomer_GPA_free () the array. In a word,
	 we do not own the monomers in this array.
      */
      g_ptr_array_free (fillGPA, TRUE);
    }
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_kbd_force_complete_elab_code (GtkWidget *widget)
{
  PxmMonomer *monomer = NULL;
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  gchar *err = NULL;

  
  g_assert (widget != NULL);
  
  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  /* Let us say that we have a 3-letter monomer codes polymer
   * definition, of which "Gly", "Gl" and "G". We have entered "G" and
   * this function is called. What we want here to happen is that if
   * "G" corresponds to a valid monomer code it be evaluated without
   * bothering with other available completions.
   */
  strcpy (PXM_SEQED_WIDGET (widget)->eval_code,
	  PXM_SEQED_WIDGET (widget)->elab_code);
  
  monomer = 
    pxmchem_monomer_get_ptr_by_code (PXM_SEQED_WIDGET (widget)->eval_code,
				     polchemdef->monomerGPA);
  
  if (monomer == NULL)
    {
      /* No one single monomer in refGPA does have a monomer code
       * resembling PXM_SEQED_WIDGET (widget)->elab_code. This is an error
       * condition.
       */
      err = g_strdup_printf (_("code already invalid: '%s'"),
			     PXM_SEQED_WIDGET (widget)->eval_code);
      
      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			  err);
      
      g_free (err);
      
      polyxmass_seqed_widget_kbd_del_last_char_elab_code (widget);
      polyxmass_seqed_widget_kbd_echo_elab_code (widget);

      return FALSE;
    }

  /* Apparently we have a monomer, so just evaluate its code.
   */
  if (FALSE ==
      polyxmass_seqed_widget_kbd_evaluate_code (monomer->code,
						PXM_SEQED_WIDGET (widget)->
						last_point_1_idx,
						widget))
    {
      err = g_strdup_printf (_("failed eval code: '%s'"),
			     monomer->code);
      
      gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
			  err);
      
      g_free (err);
      
      return FALSE;
    }

  /* If we are here that means that we successfully evaluated
   * the monomer->code, which means that we can reinitialize
   * some variables for the next character to be entered to be
   * handled freshly.
   */
  memset (PXM_SEQED_WIDGET (widget)->eval_code, '\x0', 
	  polchemdef->codelen + 1);
  
  memset (PXM_SEQED_WIDGET (widget)->elab_code, '\x0', 
	  polchemdef->codelen + 1);
  
  PXM_SEQED_WIDGET (widget)->kb_typed_chars = 0;
  
  polyxmass_seqed_widget_kbd_echo_elab_code (widget);
	  
  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (widget), 
						   TRUE);
	  
  /* If there was an error message, just erase it.
   */
  gtk_entry_set_text (GTK_ENTRY (PXM_SEQED_WIDGET (widget)->error_code_entry),
		      "");

  return TRUE;
}



gboolean
polyxmass_seqed_widget_kbd_evaluate_code (gchar *code, gint idx,
					  GtkWidget *widget)
{
  PxmMonomer *monomer = NULL;
  
  PxmRect rect;
  
  GPtrArray *refGPA = NULL;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  


  g_assert (code != NULL);
  g_assert (widget != NULL);
  
  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  refGPA = polchemdef->monomerGPA;
  g_assert (refGPA != NULL);
  

  if (-1 == pxmchem_monomer_get_index_by_code (code, refGPA))
    {
      g_critical (_("%s@%d: monomer of code: '%s' is not known\n"),
		  __FILE__, __LINE__, code);
      
      return FALSE;
    }
  
  if (TRUE == polyxmass_seqed_widget_get_selection_indices (widget,
							    NULL,
							    NULL))
    {
      /*
	debug_printf (("selection to be removed is [%d-%d]\n",
	PXM_SEQED_WIDGET (widget)->sel_mnm_idx1, PXM_SEQED_WIDGET
	(widget)->sel_mnm_idx2));
      */

      /* The function below returns the location of the cursor after the
       * operation or -1 if an error occurred.
       */
      if (-1 == polyxmass_seqed_widget_remove_selected_oligomer (widget))
	{
	  g_error (_("%s@%d: failed to remove the selected oligomer\n"),
		   __FILE__, __LINE__);
	}
    }
  
  /* Create a new monomer from its code.
   */
  monomer = pxmchem_monomer_new_by_code (code, refGPA);

  /*
    debug_printf (("new code is %s\n", monomer->code));
  */
    
  /* Integrate the newly created monomer into the polymer sequence's
   * GPtrArray:
   */

  /*
    debug_printf (("last point idx before integration of monomer: %d\n", 
    PXM_SEQED_WIDGET (widget)->last_point_1_idx));
  */


  if (FALSE == polyxmass_seqed_widget_integrate_monomer_at_idx (monomer, 
								idx,
								widget,
								FALSE))
    {
      g_critical (_("%s@%d: failed to integrate new monomer: '%s' in sequence\n"),
		  __FILE__, __LINE__, monomer->code);
      
      pxmchem_monomer_free (monomer);
      
      return FALSE;
    }
  
  PXM_SEQED_WIDGET (widget)->last_point_1_idx++;
  
  /*
    debug_printf (("last point idx after integration of monomer: %d\n", 
    PXM_SEQED_WIDGET (widget)->last_point_1_idx));
  */

  /* We should make sure that the insertion point is systematically
     visible, even if we are inserting data on the lowest part of the
     sequence editor window.
  */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, COORDSYS_NW);

  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
						 canvas),
				   &(PXM_SEQED_WIDGET (widget)->canvas_x_offset),
				   &(PXM_SEQED_WIDGET (widget)->canvas_y_offset));
  
  if (rect.y1 
      >= PXM_SEQED_WIDGET (widget)->canvas_y_offset 
      + PXM_SEQED_WIDGET (widget)->monicon_size 
      + PXM_SEQED_WIDGET (widget)->sw_height
      - - PXM_SEQED_WIDGET (widget)->monicon_size)
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset +
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }
  
  polyxmass_seqed_widget_redraw_sequence (widget);
  
  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (widget),
						      PXM_MASSCALC_TARGET_BOTH);

  return TRUE;
}



