/*
 * ctlportslave.c  Command line interface to running portslaves.
 *      Basically this is to accomodate the pmmon
 *      program, but it might have some more uses too.
 *
 * Version:    ctlportslave  1.01  22-Nov-1997  miquels@cistron.nl
 *           1.2.0  1999-01-22  dcinege@psychosis.com
 *
 */
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <utmp.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <time.h>
#include <errno.h>

#include "server.h"

#define SESS 1
#define WHO 2
#define DETAIL 3
#define QUIET 10

static int do_show(int argc, const char **argv);
static int do_c_show(int argc, const char **argv);
static int do_reset(int argc, const char **argv);
static int do_quit(int unused_argc, const char **unused_argv);
static int do_help(int unused_argc, const char **unused_argv);
static int do_help_finger(void);
static int read_utmp(void);
static char* get_passwd(char* p);

struct utmp *lines;
int max_port_ever = 0;
int port_array_size = 0;
const char *eol = "\n";
int finger_prog = false;
char fpasswd[256];
char rpasswd[256];
char passwd_in[256];

#ifndef BIGUTMP
char eth0_classB[32];
#endif

int sock_fd;
extern int traffic_stats(int sockfd, struct in_addr ina, int *in, int *out);
extern int idle_stat(char *dev);

/*
 *  Header for the 2 formats: sessions and who.
 */
const char * const hdr_sess =
"Port User            Host/Inet/Dest   Type    Dir Status         Start   Idle";
const char * const sep_sess =
"---- --------------- ---------------- ------- --- ------------- ------ ------";
#ifdef PRINT_IDLE
const char * const idle_sess = 
"S%-3d -               -                Log/Net In  IDLE               0      0%s";
#endif
const char * const hdr_who =
"Port User         Host/Inet/Dest  Type    Stat Start        In       Out  Idle";
const char * const sep_who =
"---- ------------ --------------- ------- ---- ----- --------- --------- -----";
#ifdef PRINT_IDLE
const char * const idle_who =
"S%-3d -            -               -       IDLE     0         0         0     0%s%";
#endif
const char * const hdr_detail =
"Port User            Host/Inet/Dest   Type    Status";
const char * const sep_detail =
"---------------------------------------------------------------------------";
#ifdef PRINT_IDLE
const char * const idle_detail =
"S%-3d -               -                        IDLE%s%s%s";
#endif

/*
 *  Commands we support.
 */
struct commands {
  const char *cmd;
  int (*fun)(int, const char **);
} commands[] = {
  { "show",  do_show   },
  { "s",     do_show   },
  { "cshow", do_c_show },
  { "c",     do_c_show },
  { "reset", do_reset  },
  { "r",     do_reset  },
  { "help",  do_help   },
  { "h",     do_help   },
  { "?",     do_help   },
  { "exit",  do_quit   },
  { "e",     do_quit   },
  { "quit",  do_quit   },
  { "q",     do_quit   },
  { NULL,    NULL      },
};


/*
 *  Return extended type.
 */
const char *xtype(char t)
{
  if (t == 'x')  return("Network");
  if (t == 'E')  return("Telnet");
  if (t == 'R')  return("Rlogin");
  if (t == 'L')  return("Local");
  if (t == 'X')  return("Local");
  if (t == 'C')  return("CSLIP");
  if (t == 'S')  return("SLIP");
  if (t == 'P')  return("PPP");
  if (t == 'A')  return("APPP");
  if (t == 'H')  return("SSH");  
  if (t == 'I')  return("Logon");

  return ("unknown");
}

/*
 *  Return address.
 */
static char *address(struct utmp *ut, struct in_addr *ina)
{
#ifdef __linux__
  ina->s_addr = ut->ut_addr;
  return(inet_ntoa(*ina));
#else

  static char buf[256];
  char *p;
  int i = 0;
  ina->s_addr = 0;

#  ifndef BIGUTMP
  if ((p = strchr(ut->ut_host, ':')) == NULL || p[1] == 0)
    return("");
  p++;
  if (p[0]) p++;
  snprintf(buf, sizeof (buf), "%s%s", eth0_classB, p + 1);
#  else
  p = ut->ut_host + 6;
  while (*p && *p != ':') {
    buf[i++] = *p;
    p++;
  }
  buf[i] = '\0';
  if (*p == 0)
    return("");
#  endif

  ina->s_addr = inet_addr(buf);
  return buf;

#endif

}

static void got_hup(int sig)
{
}

/*
 *  Show logged in users continuously updating.
 */
static int do_c_show(int argc, const char **argv)
{
  struct sigaction sa, old_sa;
  struct stat stat_buf;
  int rc;
  time_t last;

  last = time(NULL);
  rc = do_show(argc, argv);
  if(rc)
    return rc;
  sa.sa_handler = got_hup;
  sa.sa_flags = SA_ONESHOT;
  if(sigaction(SIGINT, &sa, &old_sa))
    return 1;
  while(1)
  {
    struct timespec req;
    req.tv_sec = 1;
    req.tv_nsec = 0;
    if(nanosleep(&req, NULL))
      break;
    if(stat(_PATH_UTMP, &stat_buf))
      break;
    if(stat_buf.st_mtime > last)
    {
      last = time(NULL);
      rc = do_show(argc, argv);
      if(rc)
        break;
    }
  }
  sigaction(SIGINT, &old_sa, NULL);
  return rc;
}

/*
 *  Show logged in users. If arg is `sessions', use the
 *  portmaster format. If arg is `who', use our own format.
 */
static int do_show(int argc, const char **argv)
{
  int style = 0;
  int i;
  const char *type = NULL, *p;
  char name[UT_NAMESIZE];
  char show_status[UT_HOSTSIZE];
  const char *addr = NULL;
  struct in_addr ina;
  time_t now;
  int tm;
  int in, out, idle;

  if(argc > 1)
  {
    switch (argv[1][0])
    {
      case 's':  style = SESS;  break;
      case 'w':  style = WHO;  break;
      case 'd':  style = DETAIL;  break;
      default:
        printf("Error: usage: show sessions,who,detailed\n");
        return 1;
    }
  }
  
  read_utmp();
  now = time(NULL);

  switch(style)
  {
    case SESS:  printf("%s%s%s%s", hdr_sess, eol, sep_sess, eol);  break;
    case WHO:  printf("%s%s%s%s", hdr_who, eol, sep_who, eol);    break;
    case DETAIL:  printf("%s%s%s%s", hdr_detail, eol, sep_detail, eol);  break;
  }

  for(i = 0; i <= max_port_ever; i++)
  {

    p = strchr(lines[i].ut_host, ':');
    if(p == NULL || lines[i].ut_name[0] == 0)
    {
#ifdef PRINT_IDLE
      if(!finger_prog)
      {
        switch (style)
        {
          case SESS:  printf(idle_sess, i, eol);  break;
          case WHO:  printf(idle_who, i, eol);  break;
          case DETAIL:  printf(idle_detail, i, eol, sep_detail, eol);  break;
        }
      }
#endif
      continue;
    }

    strncpy(name, lines[i].ut_name, UT_NAMESIZE);
    name[UT_NAMESIZE - 1] = 0;
    tm = (now - lines[i].ut_time) / 60;
    p++;

    ina.s_addr = 0;
    if(strchr("xERSCPLXAI", *p))
    {
      
      if(strchr("I", *p))
      {
        addr = "";
        strncpy(show_status, lines[i].ut_host + 6, UT_HOSTSIZE - 6);
        show_status[UT_HOSTSIZE - 6] = 0;      
        
      }
      else
      {
        addr = address(&lines[i], &ina);
        strncpy(show_status, "ESTABLISHED", UT_HOSTSIZE);
        show_status[UT_HOSTSIZE - 1] = '\0';
      }
        
      type = xtype(*p);
#ifdef BIGUTMP
/* Override with new values */
      if (style != WHO)
      {
        strncpy(show_status, lines[i].ut_host + strlen(addr) + 7, UT_HOSTSIZE);
        show_status[UT_HOSTSIZE - 1] = '\0';
      }
#endif
    }
    

    in = out = idle = 0;
    if(style >= WHO && ina.s_addr)
      traffic_stats(sock_fd, ina, &in, &out);
    idle = idle_stat(lines[i].ut_line);
    if(idle > 0)
      idle /= 60;

    switch (style)
    {
      case SESS:
        printf("S%-3d %-15.15s %-16.16s %-7.7s In  %-13.13s %6d %6d%s",
                    i, name, addr, type, show_status, tm, idle, eol);
        break;
      case WHO:  
        printf("S%-3d %-12.12s %-15.15s %-7.7s %-4.4s %5d %9d %9d %5d%s",
                    i, name, addr, type, show_status, tm, in, out, idle, eol);
        break;
      case DETAIL:  
        printf("S%-3d %-15.15s %-16.16s %-7.7s %-28.28s%s",
                    i, name, addr, type, show_status, eol);
        printf(" IOT: %10d %10d %7d KB        Start:%6d  Idle:%5d%s",
                    in, out,  (in + out) / 1024, tm, idle, eol);
        printf("%s%s", sep_detail, eol);
        break;
    }
  }
  return 0;
}

/*
 *  Reset a terminal line (send SIGHUP)
 */
static int do_reset(int argc, const char **argv)
{
  int port;
  
  if(argc < 2 || (argv[1][0] != 's' && argv[1][0] != 'S'))
  {
    printf("Error: Usage: reset Port_Name\n");
    return 0;
  }
  port = atoi(argv[1] + 1);
  
  read_utmp();
  
  if(lines[port].ut_pid > 0)
  {
    printf("Resetting port S%d (line %s, pid %d)\n",
      port, lines[port].ut_line, lines[port].ut_pid);
    kill(lines[port].ut_pid, SIGHUP);
  }
  else
  {
    printf("Port S%d does not appear to be active. Can not reset...\n", port);
  }
  return 0;
}

/*
 *  Exit from this program.
 */
static int do_quit(int unused_argc, const char **unused_argv)
{
  return -1;
}

/*
 *  Pretend we're the finger daemon.
 */
static int do_finger(void)
{
  char buf[256];
  char port[16];
  const char *args[3] = {NULL, "sessions", NULL};  /* set default style */
  char *p;

  eol = "\r\n";

  memset(passwd_in,'\0',sizeof(passwd_in));
    
  fgets(buf, sizeof(buf) / 2, stdin);  /* Just paranoid about an overflow. */

  switch (buf[0])
  {
    case 'h':
      do_help_finger();
      return 1;
    case 'r':
      if (strcmp(rpasswd, "") == 0 )
      {
        printf("Reset access disabled. (no password defined)\n");
        break;
      }
      
      p = get_passwd(buf);
      if(strcmp(rpasswd,passwd_in) != 0)
      {
        printf("Permission denied.\n");
        break;
      }  

      memset(port,'\0',sizeof(port));
      strncpy(port,++p,3);
      
      args[0] = "reset"; args[1] = port; args[2] = NULL;
      do_reset(2,args);
    break;
    case 's':  args[1] = "sessions";  goto def; break;
    case 'w':  args[1] = "who";  goto def; break;
    case 'd':  args[1] = "detailed";  goto def; break;
    default:
def:    
      if(strcmp(fpasswd,"") != 0)
      {
        get_passwd(buf);
        if(strcmp(fpasswd,passwd_in) != 0)
        {
          printf("Permission denied.\n");
          return 1;
        }
      }  
        
      args[0] = "show"; args[2] = NULL;
      do_show(2, args);
      break;
  }
  return 0;
}


static char* get_passwd(char *p)
{
  int len = 0, i = 0;
  while(*p >= Fascii && *p <= Lascii)
  {    /* strlen up to non-escape chars */
    p++;
    len++;
  }
  if (len <= 2) goto bad_usage;
  p -= len;

  while((*++p != ':') && (--len > 0));
  if(len <= 1)
    goto bad_usage;
  p++; len--;

  while(*p != ':' &&  len > 1)
  {
    passwd_in[i++] = *p++;
    len--;
  }
  return (p);

bad_usage:
  printf("Permission denied.\n");
  exit(1);
}



/*
 *  Short help text.
 */
static int do_help(int unused_argc, const char **unused_argv)
{
  printf("\nCtlportslave v%s help:\n\n", PORTSLAVE_VERSION);
  printf("  show  <sessions|who|detailed>\n");
  printf("  cshow <sessions|who|detailed>\n");
  printf("  reset Sxx\n");
  printf("  exit\n");
  printf("  quit\n\n");
  return 0;
}

static int do_help_finger(void)
{
  printf("\nCtlportslave v%s as finger help:\n\n", PORTSLAVE_VERSION);
  printf("  <sessions|who|detailed>[:password]@  show active ports\n");
  printf("  reset:password:Sxx@      reset port Sxx\n");
  return 0;
}


/*
 *  Decipher portslave port from ut_host.
 */
static int ps_port(char *data)
{
  if(isdigit(data[0]) && isdigit(data[1]) && isdigit(data[2]) && data[3] == ':')
    return (atoi(data));
  return -1;
}

/*
 *  Read the portslave UTMP file..
 */
static int read_utmp(void)
{
  struct utmp *ut;
  int port;

  setutent();
  if(port_array_size)
  {
    memset(lines, 0, sizeof(struct utmp) * port_array_size);
  }
  else
  {
      port_array_size = 16;
      lines = xmalloc(sizeof(struct utmp) * port_array_size);
  }
  while((ut = getutent()) != NULL)
  {
    if((port = ps_port(ut->ut_host)) < 0)
      continue;
    if(port > max_port_ever)
      max_port_ever = port;
    if(port >= port_array_size)
    {
      int new_array_size = port_array_size * 2;
      struct utmp *new_arr;
      if(new_array_size <= port)
        new_array_size = port + 1;
      new_arr = xmalloc(sizeof(struct utmp) * new_array_size);
      memcpy(new_arr, lines, sizeof(struct utmp) * port_array_size);
      free(lines);
      port_array_size = new_array_size;
      lines = new_arr;
    }
    if(ut->ut_type != LOGIN_PROCESS && ut->ut_type != USER_PROCESS)
      continue;
    if(kill(ut->ut_pid, 0))
    {
      if(errno == ESRCH)
        continue;
    }
    lines[port] = *ut;
  }
  endutent();

  return 0;
}


int main(int argc, char **argv)
{
  char ps_hostname[128];
  char buf[64];
  const char *args[16];
  char *p=0;
  int i, n;
  bool ps_quit = false;

#ifndef __linux__ 
#ifndef BIGUTMP  
  struct ifreq ifr;
  struct sockaddr_in *sin;
#endif
#endif


  while(( i = getopt(argc, argv, "+fp:r:")) != -1)
  {
    switch(i)
    {
      case 'f':
        finger_prog = true;
      break;
      case 'p':
        strncpy(fpasswd, optarg, sizeof(fpasswd));
        fpasswd[sizeof(fpasswd) - 1] = '\0';
      break;
      case 'r': 
        strncpy(rpasswd, optarg, sizeof(rpasswd));
        rpasswd[sizeof(rpasswd) - 1] = '\0';
      break;
      case '?':
        exit (1);
    }  
  }

  /*
   *  See if we we're called as `fingerd'.
   */
  if((strstr(argv[0], "finger")) || finger_prog)
  {
    while (argc > 1)
    {    /* Clear displayed command line */
      p = argv[--argc];
      while (*p) *p++ = '\0';
    }
    finger_prog = true;
  }


  /*
   *  Find out our ethernet address. We only need
   *  the first two octets. This is because that's missing
   *  from the information in the utmp file...
   */
  if((sock_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    perror("socket");
    exit(1);
  }
  
#ifndef __linux__ 
#ifndef BIGUTMP  
  memset(&ifr, 0, sizeof(ifr));
  strcpy(ifr.ifr_name, "eth0");
  ifr.ifr_addr.sa_family = AF_INET;
  if(ioctl(sock_fd, SIOCGIFADDR, &ifr) < 0)
  {
    perror("SIOCGIFADDR(eth0)");
    exit(1);
  }
  
  sin = (struct sockaddr_in *)&(ifr.ifr_addr);
  strcpy(eth0_classB, inet_ntoa(sin->sin_addr));
  /*printf("ether is at %s\n", eth0_classB);*/
  p = eth0_classB;
  i = 0;
  while(*p && i < 2)
    if (*p++ == '.') i++;
  *p = 0;
#endif
#endif

  if(finger_prog)
    exit(do_finger());

  /*
   *  Do some network/socket initialization.
   */
  gethostname(ps_hostname, sizeof(ps_hostname));

  /*
   *  Command Loop.
   */
  while(!ps_quit)
  {
    printf("%s> ", ps_hostname);
    fflush(stdout);
    if(fgets(buf, sizeof(buf), stdin) == NULL)
    {
      printf("\n");
      break;
    }
    i = 0;
    p = strtok(buf, " \t\r\n");
    while(p && i < 15)
    {
      args[i++] = p;
      p = strtok(NULL, " \t\r\n");
    }
    args[i] = NULL;
    if (args[0] == NULL)
      continue;
    for(n = 0; commands[n].cmd; n++)
    {
      if(strcmp(args[0], commands[n].cmd) == 0)
      {
        if(commands[n].fun(i, args) < 0)
          ps_quit = true;
        break;
      }
    }
    if(commands[n].cmd == NULL)
      printf("Invalid command.\n");
  }
  printf("Goodbye....\n");

  return 0;
}

/*
 * Malloc and die if failed.
 * this is for ctlp-subs
 */
void *xmalloc(int size)
{
  void *p;
 
  if((p = malloc(size)) == NULL)
  {
    syslog(LOG_ERR, "Virtual memory exhausted.\n");
    exit(1);
  }
  memset(p, 0, size);
  return p;
}
