/*
 * getty.c  This is the part that talks with the modem,
 *    does the login stuff etc.
 *
 * Version:  @(#)getty.c  1.34  13-Jan-1998  MvS.
 *
 */

#include "server.h"
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <errno.h>
#include <netdb.h>
#include <arpa/inet.h>

#define GL_DO_ECHO 1
#define GL_ECHO2 2
#define GL_ECHO4 4
#define GL_ECHO8 8

/*
 *  Turn CR translation on or off.
 */
static void crtrans(bool inon, bool outon)
{
  struct termios tty;

  tcgetattr(0, &tty);
  if(inon)
    tty.c_iflag |= ICRNL|IGNCR;
  else
    tty.c_iflag &= ~(ICRNL|IGNCR);
  if(outon)
    tty.c_oflag |= ONLCR|OPOST;
  else
    tty.c_oflag &= ~(ONLCR|OPOST);
  tcsetattr(0, TCSANOW, &tty);
}

/*
 *  Disassociate from our controlling tty.
 */
static void disass_ctty()
{
  pid_t pid;
  int fl;

  /*
   *  First let go of our old tty.
   *  This looks like black magic. It is ;)
   */
  pid = getpid();
#ifndef __ELF__
  /*
   *  a.out systems generally don't have getsid()
   */
  setpgid(0, getpgid(getppid()));
  setsid();
#else
  if(getsid(pid) != pid)
  {
    if(setpgid(0, getpgid(getppid())) < 0)
      nsyslog(LOG_WARNING, "setpgid: %m");
    if(setsid() < 0)
      nsyslog(LOG_WARNING, "setsid: %m");
  }
#endif
  if((fl = open("/dev/tty", O_RDWR)) >= 0)
  {
    signal(SIGHUP, SIG_IGN);
    if(ioctl(fl, TIOCNOTTY) < 0)
      nsyslog(LOG_WARNING, "ioctl TIOCNOTTY: %m");
    signal(SIGHUP, SIG_DFL);
    close(fl);
  }
}

static void got_hup_exit(int sig)
{
  nsyslog(LOG_ERR, "Got Hangup - exiting");
  exit(1);
}

/*
 *  Run a getty on a port.
 */
int getty(struct auth *ai, LOCATION location)
{
  char banner [1024];
  CHAT_ERR rc;
  struct sigaction sa;

  switch(lineconf.protocol)
  {
  case P_SOCKET_SERVER:
    if(location != eLocBuf && location != eLocSocSrv)
      return -1;
  break;
  case P_SOCKET_SSH:
    if(location != eLocBuf && location != eLocSocket)
      return -1;
  break;
  }

  if(location != eLocSocket)
    disass_ctty();
  if(location == eLocModem)
  {
    close(0);
    close(1);
    close(2);
  }

  /*
   *  First lock or wait if the port is busy.
   */
  if(!lineconf.lockfile)
  {
    if(lockfile(&lineconf.lockfile, lineconf.lockdir, lineconf.tty, location))
    {
      nsyslog(LOG_ERR, "Can't create lock");
      return -1;
    }
  }
  dolock_and_wait(10);
  if(serial_open(location) < 0)
  {
    unlink(lineconf.lockfile);
    return -1;
  }
  if(ioctl(0, TIOCSCTTY, 1) < 0)
    nsyslog(LOG_WARNING, "TIOCSCTTY: %m");
  sa.sa_handler = got_hup_exit;
  sa.sa_flags = SA_RESTART;
  sigaction(SIGHUP, &sa, NULL);
  crtrans(false, false);
  rc = chat(1, lineconf.initchat, ai);
  if(rc != eNoErr)
  {
    crtrans(true, true);
    switch(rc)
    {
    case eInterrupt:
      nsyslog(LOG_ERR, "initchat failed - interrupted");
    break;
    case eTimeOut:
      nsyslog(LOG_ERR, "initchat failed - timeout");
    break;
    case eAbort:
    default:
    break;
    }
    serial_close();
    unlink(lineconf.lockfile);
    return -1;
  }
  crtrans(true, true);

  /*
   *  Delay just a little bit and flush junk.
   */

  xusleep(250000);
  tcflush(0, TCIOFLUSH);

  /*
   *  Show the banner and say hello.
   */
  if(lineconf.issue && lineconf.issue[0])
  {
    expand_format(banner, sizeof (banner), lineconf.issue, ai);
    fprintf(stdout, "%s\n", banner);
    fflush(stdout);
  }

  return 0;
}

typedef enum
{
  eGotAlarm = -2,
  eGotEOF = -1,
  eGotSuccess = 0,
  eGotPPP = 1
} GET_LINE_TYPE;

static GET_LINE_TYPE get_line(bool ppp, const char *prompt, int echo
    , char *buffer, int size, int flag_telnet_cyclades);

/*
 *  Imitate a modem on a port.
 */
int emumodem(struct auth *ai, LOCATION location)
{
  char banner [1024];
  char buf[128];

  disass_ctty();

  if(!lineconf.lockfile)
  {
    if(lockfile(&lineconf.lockfile, lineconf.lockdir, lineconf.tty, location))
    {
      nsyslog(LOG_ERR, "Can't create lock");
      return -1;
    }
  }
  close(0);
  close(1);
  close(2);

  /*
   *  First lock or wait if the port is busy.
   */
  dolock_and_wait(10);
  if(serial_open(location) < 0)
  {
    unlink(lineconf.lockfile);
    return -1;
  }

  /*
   *  Now force the new tty into becoming our
   *  controlling tty. This will also kill other
   *  processes hanging on this tty.
   */
  if(ioctl(0, TIOCSCTTY, 1) < 0)
    nsyslog(LOG_WARNING, "TIOCSCTTY: %m");

  /*
   *  Loop, and emulate a modem.
   */
  crtrans(false, false);
  printf("\r\nNO CARRIER\r\n");
  while(1)
  {
    fflush(stdout);
    get_line(false, "", GL_DO_ECHO, buf, sizeof(buf), 0);
    if(strncasecmp(buf, "ATD", 3) == 0)
    {
      xsleep(3);
      printf("CONNECT 115200\r\n");
      snprintf (ai->conn_info, sizeof(ai->conn_info), "115200/DIRECT");
      fflush(stdout);
      break;
    }
    if(strncasecmp(buf, "AT&V", 4) == 0)
    {
      printf("OK - Portslave Modem emulator\r\n");
      continue;
    }
    if(strncasecmp(buf, "AT", 2) == 0)
    {
      printf("OK\r\n");
      continue;
    }
    if(buf[0]) printf("ERROR\r\n");
  }
  crtrans(true, true);

  /*
   *  Delay just a little bit and flush junk.
   */
  xusleep(250000);
  tcflush(0, TCIOFLUSH);

  /*
   *  Show the banner and say hello.
   */
  if(lineconf.issue && lineconf.issue[0])
  {
    expand_format(banner, sizeof (banner), lineconf.issue, ai);
    fprintf(stdout, "%s\n", banner);
    fflush(stdout);
  }

  return 0;
}


/*
 *  Alarm handler for the login timeout.
 */
static bool got_alrm;
static void alrm_handler()
{
  got_alrm = true;
}

/*
 *  We've got a connection. Now let the user login.
 */
int do_login(struct auth *ai)
{
  /* r is ret from get_line(), -1 on EOF, 0 on success, 1 on AutoPPP. */
  GET_LINE_TYPE r = 0;
  const char *p = lineconf.prompt ? lineconf.prompt : "login: ";
  char prompt[1024];
  int to = 60;
  int flag = lineconf.protocol == P_SOCKET_SERVER;

  got_alrm = false;
  signal(SIGALRM, alrm_handler);
  if(lineconf.dcd == 0) to = 0;

  expand_format(prompt, sizeof (prompt), p, ai);

  alarm(to);
  crtrans(false, true);

  do
  {
    r = get_line(true, prompt, GL_DO_ECHO | GL_ECHO2, ai->login, sizeof(ai->login), flag);
  } while(r == eGotSuccess && ai->login[0] == '\0');

  alarm(0);
  crtrans(true, true);

  if(r == eGotSuccess && ai->login[0] == '\0')
    return 2;

  if(r == eGotSuccess)
    nsyslog(LOG_INFO, "Detected login for %s", ai->login);
  else if(r != eGotPPP)
    nsyslog(LOG_INFO, "Login timed out / quit");

  if(r == eGotPPP)
  {
    strncpy(ai->login, "AutoPPP", sizeof(ai->login)-1);
    ai->login [sizeof(ai->login)-1] = '\0';
    ai->passwd[0] = '\0';
    return 0;
  }

  if(lineconf.locallogins && ai->login[0] == '!')
    return 0;

  alarm(to);
  crtrans(false, true);
  if(r != eGotSuccess ||
     (r = get_line(true, "Password: ", GL_ECHO4, ai->passwd, sizeof(ai->passwd), flag)) != eGotSuccess)
  {
    if(r == eGotPPP)
    {
      strncpy(ai->login, "AutoPPP", sizeof(ai->login)-1);
      ai->login [sizeof(ai->login)-1] = '\0';
      ai->passwd[0] = '\0';
      r = 0;
    }
  }
  alarm(0);
  crtrans(true, true);
   
  return r;
}

/*
 * This part is for the automatic PPP detection.
 * get_line() can check for it.
 */
const char * const PPP1 = "\x7e\xff\x03\xc0\x21";
const int PPP1_LEN = 5;
const char * const PPP2 = "\x7e\xff\x7d\x23\xc0\x21";
const int PPP2_LEN = 6;
const char * const PPP3 = "\x7e\x7d\xdf\x7d\x23\xc0\x21";
const int PPP3_LEN = 7;
static int pppseen(int ch)
{
  static char buf[8];

  if(ch == -1)
  {
    memset(buf, 0, sizeof(buf));
    return 0;
  }

  memmove(buf, buf + 1, sizeof(buf) - 1);
  buf[sizeof(buf) - 2] = ch;
  if(!strcmp(buf + (sizeof(buf) - 1 - PPP1_LEN), PPP1)
     || !strcmp(buf + (sizeof(buf) - 1 - PPP2_LEN), PPP2)
     || !strcmp(buf + (sizeof(buf) - 1 - PPP3_LEN), PPP3) )
    return 1;

  return 0;
}

/*
 * Read and echo a line.
 *
 * Returns -1 on EOF, 0 on success and 1 on PPP detect.
 */
static GET_LINE_TYPE get_line(bool ppp, const char *prompt, int echo
    , char *buffer, int size, int flag_telnet_cyclades)
{
  int pos = 0;
  unsigned char c;
  bool echonl = true;
  static bool crseen = false;
  int len;
  GET_LINE_TYPE rc = eGotSuccess;
  struct sigaction sa;

  memset(&sa, 0, sizeof(sa));

  if(ppp) pppseen(-1);

  if(prompt && prompt[0])
  {
    len = strlen(prompt);
    write(STDOUT_FILENO, prompt, len);
    if(prompt[len - 1] != ' ')
      write(STDOUT_FILENO, " ", 1);
  }
  buffer[0] = 0;

  if(echo & (GL_ECHO2|GL_ECHO4) )
  {
    got_alrm = false;
    sa.sa_handler = alrm_handler;
    sigaction(SIGALRM,  &sa, NULL);
    unblock(SIGALRM);
  }

  while(1)
  {
    if(echo & GL_ECHO8)
      alarm(60);

    if(read(STDIN_FILENO, &c, 1) != 1)
    {
      rc = eGotEOF;
      if(got_alrm)
      {
        rc = eGotAlarm;
        write(STDOUT_FILENO, "\r\n", 2);
      }
      goto ret_back;
    }

    if(echo & GL_ECHO2)
      echo |= GL_ECHO8;

#ifdef CYCLADES
    if(flag_telnet)
    {
      if(!HandleChar(STDOUT_FILENO, DeviceFd, c))
        continue;
    }
#endif

    if(ppp && pppseen(c))
    {
      buffer[0] = 0;
      rc = eGotPPP;
      goto ret_back;
    }

    if(c == '\n' && crseen)
    {
      crseen = false;
      continue;
    }
    crseen = false;

    switch(c)
    {
      case '\n':
      case '\r':
        if(c == '\r') crseen = true;
        buffer[pos] = 0;
        if(echonl) write(STDOUT_FILENO, "\r\n", 2);
        rc = eGotSuccess;
        goto ret_back;
      case 4: /* Control-D */
        if(pos == 0)
        {
          if(echonl)
            write(STDOUT_FILENO, "\r\n", 2);
          rc = eGotEOF;
        }
      break;
      case 5: /* Control-E */
        echo ^= ~GL_DO_ECHO;
        echonl = false;
      break;
      case 127:
      case 8: /* Backspace. */
        if(pos == 0)
        {
          write(STDOUT_FILENO, "\007", 1);
          break;
        }
        if(echo & GL_DO_ECHO)
          write(STDOUT_FILENO, "\010 \010", 3);
        pos--;
      break;
      default:
        if(c < Fascii || c > Lascii || pos >= size - 1)
        {
          write(STDOUT_FILENO, "\007", 1);
          break;
        }
        if(echo & GL_DO_ECHO)
          write(STDOUT_FILENO, &c, 1);
        buffer[pos++] = c;
      break;
    }
  }
  rc = eGotEOF;
ret_back:
  alarm(0);
  if(echo & (GL_ECHO2 | GL_ECHO4) )
    signal(SIGALRM, SIG_DFL);
  buffer[pos] = 0;
  return rc;
}
