/*
 * radclient  A client-side implementation of the RADIUS protocol.
 *
 * Version:  @(#)radclient.c  1.43  08-Nov-1997  miquels@cistron.nl
 *
 */
#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>
#include <signal.h>

#include "radius.h"
#include "server.h"

#define ONE_UCHAR(ptr,offset) (*((unsigned char *)(ptr)+(offset)))

#define MSB_UINT4(ptr) ((ONE_UCHAR((ptr),0) << 24) + \
      (ONE_UCHAR((ptr),1) << 16) + \
      (ONE_UCHAR((ptr),2) << 8) + \
      (ONE_UCHAR((ptr),3)))

extern void radius_md5_calc(unsigned char *, unsigned char *, unsigned int);

/*
 *  This is an "attribute".
 */
typedef struct _attr_
{
  unsigned char type;
  unsigned char len;
  union {
    char str[254];
    unsigned int num;
  } val;
  unsigned char pwtype;
  struct _attr_ *next;
} ATTR;

/*
 *  This is a complete request or reply.
 */
typedef struct
{
  unsigned char code;
  unsigned char id;
  unsigned char auth_vector[AUTH_VECTOR_LEN];
  time_t timestamp;
  time_t lastsent;
  ATTR *list;
  unsigned char secret[100];
} RADPKT;

static int update_framed_route(const struct auth *ai, bool islogin);
static int update_filter_id(const struct auth *ai, bool islogin);

static int open_create_flock(const char * const file)
{
  int fd, i;

  if((fd = open(file, O_RDWR|O_CREAT, 0644)) < 0)
  {
    nsyslog(LOG_ERR, "%s: %m", file);
    return -1;
  }
  for(i = 0; i < 10; i++)
  {
    if(i > 0) xusleep(200000);
    if(flock(fd, LOCK_EX) == 0)
      break;
  }
  if(i == 10)
  {
    nsyslog(LOG_ERR, "rad_id: failed to lock %s\n", RAD_ID_FILE);
    close(fd);
    return -1;
  }
  return fd;
}

static bool read_int(int fd, int *num)
{
  int n;
  char id[12];

  n = read(fd, id, sizeof(id) - 1);
  if(n < 0) n = 0;
  id[n] = 0;
  *num = strtol(id, NULL, 10);
  return (errno == ERANGE);
}

static bool write_int(int fd, int num)
{
  char id[12];
  int len;

  snprintf(id, sizeof (id), "%d\n", num);
  len = strlen(id);
  return (ftruncate(fd, 0L) || lseek(fd, 0L, SEEK_SET) != 0
    || write(fd, id, len) != len);
}

static void close_unlock(int fd)
{
  flock(fd, LOCK_UN);
  close(fd);
}

/*
 *  Create a new 8 bit packet id.
 */
static int rad_id()
{
  int fd, num;

  if( (fd = open_create_flock(RAD_ID_FILE) ) < 0 )
    exit(1);

  if(read_int(fd, &num))
    num = 0;
  num = (num + 1) & 0xff;
  if(write_int(fd, num))
  {
    nsyslog(LOG_ERR, "Can't write RADIUS ID");
    close_unlock(fd);
    exit(1);
  }

  close_unlock(fd);

  return num;
}

/*
 *  Create a random authentication vector.  We use /dev/urandom
 *  for this under Linux if possible for good security.
 *  Should we use /dev/random for even better security?
 */
static void rad_vector(unsigned char *auth_vector)
{
  int  randno;
  int  i;

  int fd = open("/dev/urandom", O_RDONLY);
  if(fd == -1 || AUTH_VECTOR_LEN != read(fd, auth_vector, AUTH_VECTOR_LEN) )
  {
    nsyslog(LOG_INFO, "Can't open /dev/urandom, using rand() instead.");
    srand(time(0) + getpid());
    for(i = 0; i < AUTH_VECTOR_LEN; )
    {
      randno = rand();
      memcpy(auth_vector, &randno, sizeof(int));
      auth_vector += sizeof(int);
      i += sizeof(int);
    }
  }
  if(fd != -1)
    close(fd);
}


/*
 *  Build an attribute.
 *  Integers are passed through without conversion...
 *  Always returns a pointer to the new attribute, never fails.
 */
static ATTR *rad_addattr_real(ATTR *list, int type, int val_len, unsigned int num, const char *str)
{
  ATTR *a, *r;

  /*
   * Ask for space.
   */
  r = (ATTR *)xmalloc(sizeof(ATTR));

  /*
   * Find end of the list (if any) and add us to it.
   */
  for(a = list; a && a->next; a = a->next)
    ;
  if(a) a->next = r;

  /*
   * Fill out fields.
   */
  if(str)
  {
    r->pwtype = PW_TYPE_STRING;
    r->len = val_len >= 0 ? val_len : strlen(str);
    memcpy(r->val.str, str, r->len);
  }
  else
  {
    r->pwtype = PW_TYPE_INTEGER;
    r->len = val_len > 0 ? val_len : 4;
    r->val.num = num;
  }
  r->len += 2;
  r->type = type;

  return r;
}

/*
 *  Build a request.
 *  Writes to *pkt to update the ID and the last sent time.
 */
static int rad_buildreq(char *buf, int bufsz, RADPKT *pkt)
{
  char *tmp = buf;
  ATTR *a;
  int attr_len = 0;
  int tot_len;
  int tm;
  time_t now;

  /*
   *  Find out how much space we need for the value pairs.
   */
  for(a = pkt->list; a; a = a->next)
    attr_len += a->len;
  tot_len = attr_len + AUTH_HDR_LEN;
  if(pkt->timestamp) tot_len += 6;  
  if(tot_len > bufsz)
  {
    nsyslog(LOG_ERR, "rad_buildreq: buffer overflow\n");
    return -1;
  }
  memset(tmp, 0, tot_len);

  /*
   *  See if we're going to use a new timestamp -
   *  if so, we need a new ID.
   */
  if(pkt->timestamp)
  {
    time(&now);
    if(pkt->lastsent != now)
    {
      pkt->id = rad_id();
      pkt->lastsent = now;
    }
  }

  /*
   *  Fill in header fields.
   */
  *tmp++ = pkt->code;
  *tmp++ = pkt->id;
  *(unsigned short *)tmp = htons(tot_len);
  tmp += 2;
  memcpy(tmp, pkt->auth_vector, AUTH_VECTOR_LEN);
  tmp += AUTH_VECTOR_LEN;

  /*
   *  Build complete request.
   */
  for(a = pkt->list; a; a = a->next)
  {
    tmp[0] = a->type;
    tmp[1] = a->len;
    if (a->pwtype == PW_TYPE_STRING)
      memcpy(tmp + 2, a->val.str, a->len - 2);
    else
      memcpy(tmp + 2, &(a->val.num), 4);
    tmp += a->len;
  }

  /*
   *  And add a time stamp.
   */
  if(pkt->timestamp)
  {
    *tmp++ = PW_ACCT_DELAY_TIME;
    *tmp++ = 6;
    tm = htonl(now - pkt->timestamp);
    memcpy(tmp, &tm, 4);
  }

  return tot_len;
}

/*
 *  Free a list of attributes.
 */
static void rad_attrfree(ATTR *l)
{
  ATTR *next;

  while(l != NULL)
  {
    next = l->next;
    free(l);
    l = next;
  }
}

/* Calls rad_addattr_real() and frees the data on error.  Simplifies the error
   handling code. */
static ATTR *rad_addattr(RADPKT *pkt, int type, int val_len, unsigned int num, const char *str)
{
  ATTR *rc = rad_addattr_real(pkt->list, type, val_len, num, str);
  if(!rc)
  {
    if(pkt->list)
      rad_attrfree(pkt->list);
    free(pkt);
    return NULL;
  }
  if(!pkt->list)
    pkt->list = rc;
  return rc;
}

/* Add the basic data to an accounting or authorization packet */
static bool rad_add_base_data(RADPKT *pkt, const struct auth *ai)
{
  /*
   *  Now we finish off with the client id (our ip address),
   *  and the port id (tty number).
   */
  if(rad_addattr(pkt, PW_NAS_IP_ADDRESS, 0, lineconf.loc_host, NULL) == NULL)
    return 1;
#ifdef HAVE_IPV6
  if(lineconf.use_v6)
  {
    if(rad_addattr(pkt, PW_NAS_IPV6_ADDRESS, sizeof(lineconf.loc_host6), 0, (char *)&lineconf.loc_host6) == NULL)
      return 1;
  }
#endif
  if(rad_addattr(pkt, PW_NAS_PORT_ID, 0, htonl(ai->nasport), NULL) == NULL)
    return 1;
  if(rad_addattr(pkt, PW_NAS_PORT_TYPE, 0, htonl(ai->porttype), NULL) == NULL)
    return 1;
  /*
   *  Add connection info on login AND logout. That's what
   *  a portmaster does too..
   */
  if(ai->conn_info[0])
  {
    if(rad_addattr(pkt, PW_CONNECT_INFO, -1, 0, ai->conn_info) == NULL)
      return 1;
  }
  if(ai->cli_src[0])
  {
    if(rad_addattr(pkt, PW_CALLING_STATION_ID, -1, 0, ai->cli_src) == NULL)
      return 1;
  }
  if(ai->cli_dst[0])
  {
    if(rad_addattr(pkt, PW_CALLED_STATION_ID, -1, 0, ai->cli_dst) == NULL)
      return 1;
  }

  /*
   *  We have to add a unique identifier, the same
   *  for login as for logout.
   */
  if(rad_addattr(pkt, PW_ACCT_SESSION_ID, -1, 0, ai->acct_session_id) == NULL)
    return 1;

  return 0;
}

/*
 *  Build an authentication request.
 */
static RADPKT *rad_buildauth(const struct auth *ai, int ppp)
{
  RADPKT *pkt;
  char *secret_key;
  unsigned int secret_len;
  int pass_len;
  int padded_len;
  unsigned char  md5buf[256];
  unsigned char passtmp[256];
  int i, j;

  pkt = xmalloc(sizeof(RADPKT));

  /*
   *  Build the header.
   */
  pkt->code = PW_AUTHENTICATION_REQUEST;
  pkt->id   = rad_id();
  rad_vector(pkt->auth_vector);

  /*
   *  Add the login name.
   */
  pkt->list = NULL;
  if(rad_addattr(pkt, PW_USER_NAME, -1, 0, ai->login) == NULL)
    return NULL;

  if(ai->called_station)
  {
    if(!rad_addattr(pkt, PW_CALLED_STATION_ID, -1, 0, ai->called_station))
      return NULL;
  }
  if(ai->calling_station)
  {
    if(!rad_addattr(pkt, PW_CALLING_STATION_ID, -1, 0, ai->calling_station))
      return NULL;
  }

  /*
   *  And the password - this one is hard.
   */
  pass_len = strlen(ai->passwd); 
  memset(passtmp, 0, sizeof(passtmp));
  secret_key = lineconf.secret;
  secret_len = strlen(secret_key);
  secret_len = MIN(secret_len, sizeof(md5buf) - AUTH_VECTOR_LEN);
  strncpy((char *)md5buf, secret_key, secret_len);
  memcpy(md5buf + secret_len, pkt->auth_vector, AUTH_VECTOR_LEN);
  radius_md5_calc(passtmp, md5buf, secret_len + AUTH_VECTOR_LEN);

  for(i = 0; ai->passwd[i] && i < AUTH_PASS_LEN; i++)
    passtmp[i] ^= ai->passwd[i];
  while((i < pass_len) && ai->passwd[i])
  {
    memcpy(md5buf + secret_len,passtmp + i - AUTH_PASS_LEN,AUTH_PASS_LEN);
    radius_md5_calc(passtmp + i,md5buf,secret_len + AUTH_PASS_LEN);
    for(j = AUTH_PASS_LEN;j && ai->passwd [i];--j,++i)
      passtmp[i] ^= ai->passwd[i];
  }
  if(lineconf.logpassword)
    nsyslog(LOG_DEBUG, "passwd/secret: %s/%s", ai->passwd, secret_key);
  padded_len = (pass_len+(AUTH_VECTOR_LEN-1)) & ~(AUTH_VECTOR_LEN-1);
  
  if(padded_len == 0)
    padded_len = AUTH_VECTOR_LEN;
    
  if(rad_addattr(pkt, PW_PASSWORD, padded_len, 0, (char *)passtmp) == NULL)
    return NULL;

  if(rad_add_base_data(pkt, ai))
    return NULL;


  /*
   *  We add the protocol type if this is PPP.
   */
  if(ppp)
  {
    if(rad_addattr(pkt, PW_FRAMED_PROTOCOL, 0, htonl(PW_PPP), NULL) == NULL)
      return NULL;
    if(rad_addattr(pkt, PW_SERVICE_TYPE, 0, htonl(PW_FRAMED_USER), NULL) == NULL)
      return NULL;
  }

  return pkt;
}

/*
 *  Build an accounting request.
 */
static RADPKT *rad_buildacct(const struct auth *ai, bool islogin)
{
  RADPKT *pkt;
  int i, s, p, c;
  unsigned int h, a;

  pkt = xmalloc(sizeof(RADPKT));

  /*
   *  Build the header.
   */
  pkt->code = PW_ACCOUNTING_REQUEST;;
  pkt->id   = rad_id();
  memset(pkt->auth_vector, 0, AUTH_VECTOR_LEN);
  strncpy((char *)pkt->secret, lineconf.secret, sizeof(pkt->secret));
  pkt->secret[sizeof(pkt->secret) - 1] = '\0';

  pkt->timestamp = time(NULL);

  /*
   *  Tell the server what kind of request this is.
   */
  i = islogin ? PW_STATUS_START : PW_STATUS_STOP;
  pkt->list = NULL;
  if(rad_addattr(pkt, PW_ACCT_STATUS_TYPE, 0, htonl(i), NULL) == NULL)
    return NULL;

  /*
   *  Add the login name.
   */
  if(ai->login && ai->login[0])
  {
    if(rad_addattr(pkt, PW_USER_NAME, -1, 0, ai->login) == NULL)
      return NULL;
  }

  if(rad_add_base_data(pkt, ai))
    return NULL;

  if(!islogin)
  {
    /*
     *  Add input and output octets.
     */
    if(ai->traffic.sent_bytes || ai->traffic.recv_bytes)
    {
      if(rad_addattr(pkt, PW_ACCT_OUTPUT_OCTETS, 0, htonl(ai->traffic.sent_bytes), NULL) == NULL)
        return NULL;
      if(rad_addattr(pkt, PW_ACCT_INPUT_OCTETS, 0, htonl(ai->traffic.recv_bytes), NULL) == NULL)
        return NULL;
    }
    /*
     *  Add input and output packets.
     */
    if(ai->traffic.sent_pkts || ai->traffic.recv_pkts)
    {
      if(rad_addattr(pkt, PW_ACCT_OUTPUT_PACKETS, 0, htonl(ai->traffic.sent_pkts), NULL) == NULL)
        return NULL;
      if(rad_addattr(pkt, PW_ACCT_INPUT_PACKETS, 0, htonl(ai->traffic.recv_pkts), NULL) == NULL)
        return NULL;
    }
  }

  /*
   *  Include session time if this is a logout.
   */
  if(!islogin && rad_addattr(pkt, PW_ACCT_SESSION_TIME, 0,
      htonl(time(NULL) - ai->start), NULL) == NULL)
    return NULL;

  /*
   *  We'll have to add some more fields here:
   *  - service type
   *  - login service
   *  - framed protocol
   *  - framed IP address
   *  - framed compression
   *  - login IP host
   *
   */
  i = -1;
  s = -1;
  p = -1;
  c = -1;
  h = 0;
  a = 0;
  switch(ai->proto)
  {
    case P_SHELL:
      s = PW_SHELL_USER;
    break;
    case P_TELNET:
      s = PW_LOGIN_USER;
      i = PW_TELNET;
      h = ai->address;
    break;
    case P_RLOGIN:
      s = PW_LOGIN_USER;
      i = PW_RLOGIN;
      h = ai->address;
    break;
    case P_SSH:
    case P_SSH2:
      s = PW_LOGIN_USER;
      i = PW_SSH;
      h = ai->address;
    break;      
    case P_TCPCLEAR:
    case P_TCPLOGIN:
      s = PW_LOGIN_USER;
      i = PW_TCP_CLEAR;
      h = ai->address;
    break;
    case P_PPP:
    case P_PPP_ONLY:
    case P_SLIP:
    case P_CSLIP:
      s = PW_FRAMED_USER;
      a = ai->address;
    break;
  }
  switch(ai->proto)
  {
    case P_PPP:
    case P_PPP_ONLY:
      p = PW_PPP;
      c = PW_VAN_JACOBSEN_TCP_IP;
    break;
    case P_SLIP:
      p = PW_SLIP;
    break;
    case P_CSLIP:
      p = PW_SLIP;
      c = PW_VAN_JACOBSEN_TCP_IP;
  }
  if(s > 0 && rad_addattr(pkt, PW_SERVICE_TYPE, 0, htonl(s), NULL) == NULL)
    return NULL;
  if(i >= 0 && rad_addattr(pkt, PW_LOGIN_SERVICE, 0, htonl(i), NULL) == NULL)
    return NULL;

  if(p >= 0 && rad_addattr(pkt, PW_FRAMED_PROTOCOL, 0, htonl(p), NULL) == NULL)
    return NULL;

  if(a > 0 && rad_addattr(pkt, PW_FRAMED_IP_ADDRESS, 0, a, NULL) == NULL)
    return NULL;

  if(c >= 0 && rad_addattr(pkt, PW_FRAMED_COMPRESSION, 0, htonl(c), NULL) == NULL)
    return NULL;

  if(h > 0 && rad_addattr(pkt, PW_LOGIN_IP_HOST, 0, h, NULL) == NULL)
    return NULL;

  return pkt;
}

#define DEFAULT_TRIES 5
#define TOTAL_TRIES 30

static int bind_local(const struct sockaddr *target)
{
  struct sockaddr_in salocal;
#ifdef HAVE_IPV6
  struct sockaddr_in6 salocal6;
#endif
  in_port_t *port_ptr, port;
  struct sockaddr *sa;
  int sock_size;
  int sock_fd;

  /*
   *  We have to bind the socket in order to
   *  receive an answer back...
   */
  sock_fd = socket(target->sa_family, SOCK_DGRAM, 0);
  if(sock_fd < 0)
  {
    nsyslog(LOG_ERR, "socket: %m");
    return -1;
  }

#ifdef HAVE_IPV6
  if(target->sa_family == AF_INET6)
  {
    memset(&salocal6, 0, sizeof(salocal6));
    salocal6.sin6_family = AF_INET6;
    salocal6.sin6_addr = in6addr_any;
    sa = (struct sockaddr *)&salocal6;
    sock_size = sizeof(salocal6);
  }
  else
#endif
  {
    memset(&salocal, 0, sizeof(salocal));
    salocal.sin_family = AF_INET;
    salocal.sin_addr.s_addr = INADDR_ANY;
    sa = (struct sockaddr *)&salocal;
    sock_size = sizeof(salocal);
  }
  port_ptr = get_port_ptr(sa);
  port = 1025;
  do {
    port++;
    *port_ptr = htons(port);
  } while ((bind(sock_fd, sa, sock_size) < 0) && port < 64000);

  if(port >= 64000)
  {
    close(sock_fd);
    nsyslog(LOG_ERR, "bind: %m");
    return -1;
  }
  return sock_fd;
}

/* Return -1 for timeout, -2 for serious error (can't create a socket),
   otherwise return the length of data received. */
static int rad_send(const struct sockaddr *host1
  , const struct sockaddr *host2, time_t rad_timeout
  , char *recvbuf, int size, RADPKT *req, bool acctpkt, int try)
{
  char sendbuf[4096];
  int sendlen;
  struct sockaddr *sa_replied = NULL;
  int sock_fd, i;
  socklen_t len;
  const struct sockaddr *remote;
  int ret;
  struct pollfd read_fd;

  if(try == 0)
    try = DEFAULT_TRIES;

  sendlen = rad_buildreq(sendbuf, sizeof(sendbuf), req);
  if(sendlen < 0)
  {
    return (-2);
  }

  if(!host1 && host2)
  {
    host1 = host2;
    host2 = NULL;
  }
  remote = host1;

  /* if an acct pkt, create the authenticator */

  if(acctpkt)
  {
    i = strlen((char *)req->secret);
    memcpy(&sendbuf[sendlen], req->secret, i);
    radius_md5_calc((unsigned char *)&sendbuf[4], (unsigned char *)sendbuf, sendlen + i);
  }

  sock_fd = bind_local(remote);
  if(sock_fd == -1)
    return -2;

  /*
   *  Try at most 10 times. We time out after specified number of
   *  seconds, and after 5 tries we switch to an alternate server,
   *  and keep switching after every timeout.
   */
  for(i = 0; i < TOTAL_TRIES; i++)
  {
    if(i && i % try == 0)
    {
      nsyslog(LOG_INFO, "radius@%s not responding", dotted_sa(remote, 1));
      if(host2 && remote != host2)
        remote = host2;
      else
        remote = host1;
    }

    /*
     *  Connect to the radius server, and
     *  send the request.
     */
    sendto(sock_fd, sendbuf, sendlen, 0, remote, sa_len(remote));

    /*
     *  Now wait for an answer.
     */
    read_fd.fd = sock_fd;
    read_fd.events = POLLIN;
    read_fd.revents = 0;
    switch (ret = poll(&read_fd, 1, rad_timeout * 1000))
    {
      case -1:
        nsyslog(LOG_DEBUG, "rad_send: poll: %d(%m)", errno);
        close(sock_fd);
        return -1;
      case 0:
        nsyslog(LOG_DEBUG, "radius@%s timed out in poll", dotted_sa(remote, 1));
      break;
      case 1:
      break;
      default:
        nsyslog(LOG_DEBUG, "rad_send: poll returned %d", ret);
        close(sock_fd);
        return -1;
    }
    if(ret == 1)
    {
#ifdef HAVE_IPV6
      if(remote->sa_family == AF_INET6)
        len = sizeof(struct sockaddr_in6);
      else
#endif
        len = sizeof(struct sockaddr_in);
      sa_replied = xmalloc(len);
      ret = recvfrom(sock_fd, recvbuf, size, 0, (struct sockaddr *)sa_replied, &len);
      if(ret == -1)
      {
        nsyslog(LOG_DEBUG, "rad_send: recvfrom: %d(%m)", errno);
      }
      else
      {
        if(ret == 0)
          nsyslog (LOG_DEBUG, "rad_send: recvfrom: read 0 bytes");
        else
          break;
      }
    }
  }
  close(sock_fd);

  if(i == TOTAL_TRIES)
  {
    if(host2)
    {
      char *name1 = xstrdup(dotted_sa(host1, 1));
      snprintf(sendbuf, sizeof (sendbuf)
        ,"Radius servers %s and %s not responding", name1, dotted_sa(host2, 1));
      free(name1);
    }
    else
    {
      snprintf(sendbuf, sizeof (sendbuf)
        , "Radius server %s not responding", dotted_sa(host1, 1));
    }
    nsyslog(LOG_ERR, sendbuf);
    return -1;
  }
  if(i > 2 && host2)
    nsyslog(LOG_INFO, "radius@%s replied eventually", dotted_sa(sa_replied, 1));
  free(sa_replied);

  return len;
}

static void free_arrray(char **messages, int num)
{
  int i;
  for(i = 0; i < num; i++)
  {
    if(messages[i])
      free(messages[i]);
    messages[i] = NULL;
  }
}

/*
 *  set environment variable name with a representation of the array val of
 *  the number of strings represented by num
 */
int setenv_from_rad(const char *name, const char **val, unsigned int num)
{
  unsigned int i;
  int size = 0;
  char *buf;
  int rc = 0;

  if(num == 0)
    return 0;

  for(i = 0; i < num; i++)
    size += strlen(val[i]);

  buf = xmalloc(size + num);
  for(i = 0; i < num; i++)
  {
    strcat(buf, val[i]);
    if(i != num - 1)
      strcat(buf, "#");
  }
  if(setenv(name, buf, 1))
  {
    nsyslog(LOG_ERR, "Can't set environment variable %s.", name);
    rc = -1;
  }

  free(buf);
  return rc;
}

/*
 *  get data from environment variable name and put it in the array val of
 *  the maximum number of strings represented by max, too many strings cause
 *  an error.  Puts the number of strings in num.
 */
int getenv_from_rad(const char *name, char **val, unsigned int max, unsigned int *num)
{
  char *buf, *ptr, *next;

  *num = 0;
  buf = getenv(name);
  if(!buf)
    return 0;
  ptr = buf;
  next = buf;
  while(next)
  {
    if(*num >= max)
    {
      nsyslog(LOG_ERR, "Error extracting variable %s.", name);
      return -1;
    }
    next = strchr(ptr, '#');
    if(!next)
    {
      val[*num] = xstrdup(ptr);
    }
    else
    {
      val[*num] = xmalloc(next - ptr + 1);
      memcpy(val[*num], ptr, next - ptr);
      val[*num][next - ptr] = '\0';
    }
    (*num)++;
    ptr = next;
  }
  return 0;
}

/*
 *  Ask a radius server to authenticate us,
 *
 *      ppp parameter is whether the protocol is PPP.
 *      for adding PPP attribute to packet
 */
int rad_client(struct auth *ai, bool ppp)
{
  int ret, len;
  bool islogin = false;
  bool isframed = false;
  AUTH_HDR *reply;
  RADPKT *req;
  unsigned int a_len, a_type;
  unsigned char *a_val;
  char recvbuf[4096];
  char *ptr = recvbuf, *maxptr;
  int oldproto;
  int def_auth_port;
  struct servent *svp;
  struct realm_def *realm_ptr;
  struct sockaddr *realm_authhost1 = NULL, *realm_authhost2 = NULL;

  if((ai->passwd[0] == '\0') && (lineconf.radnullpass == 0))
    return (-1);

  /* Get radauth port, or use default */
  svp = getservbyname ("radius", "udp");
  if(svp == NULL)
    def_auth_port = PW_AUTH_UDP_PORT;
  else
    def_auth_port = ntohs(svp->s_port);
  
  /*
   *  Set the message to some error in case we fail.
   */
  if(ai->message[0])
    free(ai->message[0]);
  ai->message[0] = xstrdup("Internal error\r\n");
  ai->msn = 1;

        /*
         *      First, build the request.
         */
  if ((req = rad_buildauth(ai, ppp)) == NULL) 
    return -1;

  /*
   *  Now connect to the server.
   */
  if((realm_ptr = ckrealm(ai)) != NULL)
  {
    realm_authhost1 = realm_ptr->authhost1;
    realm_authhost2 = realm_ptr->authhost2;
  }
  else
  {
    realm_authhost1 = lineconf.authhost1;
    realm_authhost2 = lineconf.authhost2;
  }

  if(realm_authhost1 && !*get_port_ptr(realm_authhost1))
    *get_port_ptr(realm_authhost1) = htons(def_auth_port);
  if(realm_authhost2 && !*get_port_ptr(realm_authhost2))
    *get_port_ptr(realm_authhost2) = htons(def_auth_port);
  len = rad_send(realm_authhost1, realm_authhost2, lineconf.radtimeout,
      recvbuf, sizeof(recvbuf), req, false, lineconf.radretries);
      
  rad_attrfree(req->list);
  free(req);

  if(len < 0)
  {
    if(len == -1)
    {
      if(ai->message[0])
        free(ai->message[0]);
      ai->message[0] = xstrdup("RADIUS server not responding.\r\n");
      ai->msn = 1;
    }
    return -1;
  }

  reply = (AUTH_HDR *)ptr;
  nsyslog(LOG_DEBUG, "Got a reply, code %d, length %d", reply->code, ntohs(reply->length));
  ret = (reply->code == PW_AUTHENTICATION_ACK) ? 0 : -1;
  free_arrray(ai->message, MAX_RADIUS_MESSAGES);
  ai->msn = 0;
  free_arrray(ai->filterid, MAX_FILTERS);
  ai->fln = 0;
  oldproto = ai->proto;
  ai->proto = 0;

  /*
   *  Put the reply into our auth struct.
   */
  maxptr = ptr + ntohs(reply->length);
  ptr += 20;

  while(ptr < maxptr)
  {
    a_type = ptr[0];
    a_len = ptr[1];
    a_val = (unsigned char *)&ptr[2];
    ptr += a_len;
#if 0 /* XXX - Debug */
    nsyslog(LOG_DEBUG, "Attribute type %d, length %d", a_type, a_len);
#endif
    if (a_type == 0 || a_len < 2) break;
    a_len -= 2;
    switch(a_type)
    {
      case PW_SERVICE_TYPE:
        /* Framed or login. */
        switch(MSB_UINT4(a_val))
        {
          case PW_SHELL_USER:
            ai->proto = P_SHELL;
          break;
          case PW_LOGIN_USER:
            islogin = true;
          break;
          case PW_FRAMED_USER:
            isframed = true;
          break;
          default:
          break;
        }
      break;
      case PW_LOGIN_SERVICE:
        islogin = true;
        switch(MSB_UINT4(a_val))
        {
          case PW_TELNET:
            ai->proto = P_TELNET;
          break;
          case PW_RLOGIN:
            ai->proto = P_RLOGIN;
          break;
          case PW_SSH:
            ai->proto = P_SSH;
          break;            
          case PW_TCP_CLEAR:
            ai->proto = P_TCPCLEAR;
          break;
          case PW_PORTMASTER:
          default:
            islogin = false;
            /* Unknown to us. */
          break;
        }
      break;
      case PW_LOGIN_IP_HOST:
        ai->address = htonl(MSB_UINT4(a_val));
      break;
#ifdef HAVE_IPV6
      case PW_LOGIN_IPV6_HOST:
        ai->address = htonl(MSB_UINT4(a_val));
      break;
#endif
      case PW_LOGIN_PORT:
        ai->loginport = MSB_UINT4(a_val);
      break;
      case PW_NAS_PORT_LIMIT:
        ai->port_limit = MSB_UINT4(a_val);
      break;
      case PW_FRAMED_IP_ADDRESS:
        isframed = true;
        if((unsigned)MSB_UINT4(a_val) != 0xFFFFFFFE)
          ai->address = htonl (MSB_UINT4 (a_val));
      break;
      case PW_FRAMED_IP_NETMASK:
        ai->netmask = htonl (MSB_UINT4 (a_val));
      break;
      case PW_FRAMED_MTU:
        ai->mtu = MSB_UINT4 (a_val);
      break;
      case PW_IDLE_TIMEOUT:
        ai->idletime = MSB_UINT4 (a_val);
      break;
      case PW_SESSION_TIMEOUT:
        ai->sessiontime = MSB_UINT4 (a_val);
      break;        
      case PW_FRAMED_COMPRESSION:
        if (MSB_UINT4 (a_val) != PW_VAN_JACOBSEN_TCP_IP)
          break;
        if (ai->proto == 0 || ai->proto == P_SLIP)
          ai->proto = P_CSLIP;
      break;
      case PW_FRAMED_PROTOCOL:
        isframed = true;
        if (MSB_UINT4 (a_val) == PW_PPP)
          ai->proto = P_PPP;
        else if (ai->proto == 0)
          ai->proto = P_SLIP;
      break;
      case PW_FILTER_ID:
        if(ai->fln > MAX_FILTERS)
          break;      
        len = a_len;
        ai->filterid[ai->fln] = xmalloc(len + 1);
        memcpy(ai->filterid[ai->fln], a_val, len);
        ai->filterid[ai->fln][len] = 0;
        ai->fln++;
      break;
      case PW_FRAMED_ROUTE:
        if(ai->frn >= MAX_FRAMED_ROUTES)
          break;
        len = a_len;
        ai->framed_route[ai->frn] = xmalloc(len + 1);
        memcpy(ai->framed_route[ai->frn], a_val, len);
        ai->framed_route[ai->frn][len] = 0;
        ai->frn++;
      break;        
      case PW_REPLY_MESSAGE:
        if(ai->msn >= MAX_RADIUS_MESSAGES)
          break;
        len = a_len;
        ai->message[ai->msn] = xmalloc(len + 1);
        memcpy(ai->message[ai->msn], a_val, len);
        ai->message[ai->msn][len] = 0;
        ai->msn++;
      break;
#ifdef PORTSLAVE_CALLBACK
      case PW_CALLBACK_NUMBER:
        len = a_len;
        if(len == 0)
        {
          ai->cb_allowed = (1 << CB_CONF_USER);
        }
        else
        {
          ai->cb_allowed = (1 << CB_CONF_ADMIN);
          ai->cb_number = xmalloc(len + 1);
          memcpy(ai->cb_number,a_val,len);
          ai->cb_number[len] = 0;
        }
      break;
      case PW_VENDOR_SPECIFIC:
        len = a_len;
        if(len > 6)
        {
          unsigned long lvalue;
          unsigned long vid;
          char *attr;
          memcpy(&lvalue, a_val, 4);
          vid = ntohl(lvalue);
          if(vid == 9)
          { /* Cisco */
            if(((int)a_val[4] == 1) && ((int)a_val[5] >= 24) )
            {
              unsigned int vlen=(int)a_val[5];
              if(strncmp(&a_val[6],"lcp:callback-dialstring",24) == 0)
              {
                if(vlen==24)
                { /* empty string, user defined */
                  ai->cb_allowed = (1 << CB_CONF_USER);
                }
                else
                { /* admin defined number */
                  ai->cb_allowed = (1 << CB_CONF_ADMIN);
                  ai->cb_number = xmalloc(vlen-24+1);
                  memcpy(ai->cb_number,&a_val[30],vlen-24);
                  ai->cb_number[vlen-24]=0;
                }
              }
            }
          }
        }
      break;
#endif
    }
  }

  if(isframed && ai->address == 0 && lineconf.rem_host)
    ai->address = lineconf.rem_host;

  if(islogin && ai->address == 0 && lineconf.host)
    ai->address = lineconf.host;

  if(ai->proto == 0)
    ai->proto = oldproto;

  if(ret == 0)
  {
    ai->start = time(NULL);
  }
  else
  {
    nsyslog(LOG_INFO, "authentication failed (%s/%s) %s",
      ai->login, ai->passwd, ai->message[0] ? ai->message[0] : "");
  }

  return ret;
}

/*
 *  Send accounting info to a RADIUS server.
 */
static int real_rad_acct(const struct auth *ai, bool islogin)
{
  int len;
  AUTH_HDR *reply;
  RADPKT *req;
  char recvbuf[4096];
  int def_acct_port;
  struct servent *svp;
  struct realm_def *realm_ptr;
  struct sockaddr *realm_accthost1 = NULL;
  struct sockaddr *realm_accthost2 = NULL;

  /* Get radius-acct port, or use default */
  svp = getservbyname ("radius-acct", "udp");
  if(svp == NULL)
    def_acct_port = PW_ACCT_UDP_PORT;
  else
    def_acct_port = ntohs(svp->s_port);

  /*
   *  Update utmp/wtmp (ctlportslave)
   */
  update_utmp(lineconf.stripnames ? "%L" : "%l", lineconf.utmpfrom, ai, lineconf.syswtmp);

  /*
   *  Update Framed-Routes
   */
  update_framed_route(ai, islogin);

  /*
   *  Run filter backend
   */
  update_filter_id(ai, islogin);
  
        /*
         *      First, build the request.
         */
  if((req = rad_buildacct(ai, islogin)) == NULL)
    return -1;

  /*
   *  Now connect to the server.
   */
  if((realm_ptr = ckrealm(ai)) != NULL)
  {
    realm_accthost1 = realm_ptr->accthost1;
    realm_accthost2 = realm_ptr->accthost2;
  }
  else
  {
    realm_accthost1 = lineconf.accthost1;
    realm_accthost2 = lineconf.accthost2;
  }

  if(!realm_accthost1 && !realm_accthost2)
    return 0;

  if(realm_accthost1 && !*get_port_ptr(realm_accthost1))
    *get_port_ptr(realm_accthost1) = htons(def_acct_port);
  if(realm_accthost2 && !*get_port_ptr(realm_accthost2))
    *get_port_ptr(realm_accthost2) = htons(def_acct_port);
  len = rad_send(realm_accthost1, realm_accthost2, lineconf.radtimeout,
      recvbuf, sizeof(recvbuf), req, true, lineconf.radretries);
      
  rad_attrfree(req->list);
  free(req);

  if(len < 0)
    return -1;

  reply = (AUTH_HDR *)recvbuf;
  nsyslog(LOG_DEBUG, "Got a reply, code %d, length %d", reply->code, ntohs(reply->length));

  return 0;
}

int rad_acct(const struct auth *ai, bool islogin)
{
  int rc;

  if(!ai->do_acct)
    return 0;

   /*
    *      While messing around with accounting, we have to
    *      block, but not ignore, SIGHUP and friends.
    */
  block(SIGHUP);
  block(SIGTERM);
  rc = real_rad_acct(ai, islogin);
  unblock(SIGTERM);
  unblock(SIGHUP);
  return rc;
}

/*
 *     Check for realm in the login name.
 *  Returns a non-const pointer so that the calling code can change the
 *   port number.  Ugly but I've got other things to fix.
 */
struct realm_def *ckrealm(const struct auth *ai)
{
  struct realm_def *realm;
  int l, n;

  l = strlen(ai->login);
  for(realm = lineconf.realm; realm != NULL; realm = realm->next)
  {
    n = strlen(realm->name);
    if(realm->prefix && strncmp(ai->login, realm->name, n) == 0)
      break;
    
    if(!realm->prefix)
    {
      if(l < n)  continue;
      if(strcmp(ai->login + l - n, realm->name) == 0)  break;
    }
  }    

  return realm;
}

/*
 * Invoke 'route' command to update "Framed-Route".
 */
static int update_framed_route(const struct auth *ai, bool islogin)
{
  char cmd[1024], *buf = NULL;
  char *net, *gw, *metric;
  const char *net_or_host;
  int x, rc;

  if(ai->frn == 0) return 0;
  
  if(islogin)
  {
    nsyslog(LOG_INFO, "Adding routes: %d.", ai->frn);
    x = 0;
  }
  else
  {
    nsyslog(LOG_INFO, "Deleting routes: %d.", ai->frn);  
    x = ai->frn - 1;
  }
    
  while(x < (int)ai->frn && x >= 0)
  {  
    if(buf) free(buf);
    buf = xstrdup(ai->framed_route[x]);
    net = strtok (buf, " ");
    if(strlen(net) > 3 && !strcmp("/32", &net[strlen(net) - 3]))
    {
      net_or_host = "-host";
      net[strlen(net) - 3] = '\0';
    }
    else
    {
      net_or_host = "-net";
    }
    gw = strtok ((char *) 0, " ");
    if(!gw || !strcmp(gw, "0.0.0.0"))
      gw = xstrdup(dotted(ai->address));
    else
      gw = xstrdup(gw);
    metric = strtok ((char *) 0, " ");

    if(metric)
    {
      snprintf(cmd, sizeof(cmd) - 1, "exec %s %s %s %s gw %s metric %s >/dev/null 2>&1",
      PATH_ROUTE, islogin ? "add" : "del", net_or_host, net, gw, metric);
    }
    else
    {
      snprintf(cmd, sizeof(cmd) - 1, "exec %s %s %s %s gw %s >/dev/null 2>&1",
      PATH_ROUTE, islogin ? "add" : "del", net_or_host, net, gw);
    }
    free(gw);
    rc = system(cmd);
    if(rc)
      nsyslog(LOG_ERR, "Command \"%s\" returned %d", cmd, rc);
    
    islogin ? x++ : x--;    /* FIFO add, LIFO del */
  } 

  if(buf) free(buf);
  return 0;
}

/*
 * Invoke the filter script defined by "Framed-Filter-Id".
 */
static int update_filter_id(const struct auth *ai, bool islogin)
{
  char cmd[1024];
  int x;
  
  if(ai->fln == 0) return 0;
  
  if(islogin)
  {
    nsyslog(LOG_INFO, "Starting filters: %d.", ai->fln);
    x = 0;
  }
  else
  {
    nsyslog(LOG_INFO, "Stopping filters: %d.", ai->fln);
    x = ai->fln - 1;
  }
  /* Format: path/script <start:stop> <remote ip> <local ip> <remote netmask> */
  while(x < (int)ai->fln && x >= 0)
  {  
    char *addr_str;
    char *localip_str;
    char *netmask_str;

    addr_str = xstrdup(dotted(ai->address));
    localip_str = xstrdup(dotted(ai->localip));
    netmask_str = xstrdup(dotted(ai->netmask));
    snprintf(cmd, sizeof(cmd)-1,"exec %s/%s %s %s %s %s >/dev/null 2>&1",
      lineconf.filterdir, ai->filterid[x], islogin ? "start" : "stop",
      addr_str, localip_str, netmask_str);
    free(addr_str);
    free(localip_str);
    free(netmask_str);
    system(cmd);

    islogin ? x++ : x--;    /* FIFO add, LIFO del */
  }

  return 0;
}
