/*
 *  ppmtoagafb -- Display PPM or PGM images on the AGA frame buffer
 *
 *   Copyright 1996 by Geert Uytterhoeven
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING in the main directory of the Linux
 *  distribution for more details.
 */


/*
 *  Known bugs:
 *
 *    - ppmtoagafb must be setuid root, but this is a security problem:
 *      everybody can view root's private pictures :-)
 *
 *  To do:
 *
 *    - allow multiple images to be read from stdin
 */


#include <stdio.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/fb.h>
#include <linux/kd.h>
#include <linux/vt.h>
#include <signal.h>
#include <ppm.h>


    /*
     *  Command Line Options
     */

static const char *ProgramName;
static char **Files;
static int NumFiles = 0;


    /*
     *  Console and Frame Buffer
     */

static int ActiveVT = -1;
static int ConsoleFd = 0;
static const char *FrameBufferName = NULL;
static int FrameBufferFD = -1;
static caddr_t FrameBufferBits = (caddr_t)-1;
static size_t FrameBufferSize;
u_int DisplayWidth, DisplayHeight;
u_long NextLine, NextPlane;


    /*
     *  Color Maps
     */

static u_short Grey[256];
static struct fb_cmap GreyMap = {
    0, 256, Grey, Grey, Grey
};

static u_short Red[64];
static u_short Green[64];
static u_short Blue[64];
static struct fb_cmap ColorMap = {
    0, 64, Red, Green, Blue
};


    /*
     *  Conversion to HAM8
     */

static u_char RedConv[64] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40,
    0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
    0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0xc0, 0xc0, 0xc0,
    0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0,
};
static u_char GreenConv[64] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10,
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
};
static u_char BlueConv[64] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x0c, 0x0c, 0x0c,
    0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
};


    /*
     *  Function Prototypes
     */

static void Die(const char *fmt, ...) __attribute__ ((noreturn));
static void Warn(const char *fmt, ...);
static void SigHandler(int signo) __attribute__ ((noreturn));
static void VTRequest(int signo) __attribute__ ((noreturn));
static void Usage(void) __attribute__ ((noreturn));
static void OpenFB(void);
static void SetFBMode(int color);
static void CloseFB(void);
static void Chunky2Planar(u_char chunky[32], u_long *fb);
int main(int argc, char *argv[]);


    /*
     *  Print an Error Message and Exit
     */

static void Die(const char *fmt, ...)
{
    va_list ap;

    fflush(stdout);
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);

    CloseFB();

    exit(1);
}


    /*
     *  Print a Warning Message
     */

static void Warn(const char *fmt, ...)
{
    va_list ap;

    fflush(stdout);
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
}


    /*
     *  Signal Handler
     */

static void SigHandler(int signo)
{
    signal(signo, SIG_IGN);
    Die("Caught signal %d. Exiting\n", signo);
}


    /*
     *  Handler for the Virtual Terminal Request
     */

static void VTRequest(int signo)
{
    Die("VTRequest: Exiting\n");
}


    /*
     *  Open the Frame Buffer
     */

static void OpenFB(void)
{
    int i, fd, vtno, r, g, b;
    struct fb_fix_screeninfo fix;
    struct fb_var_screeninfo var;
    char vtname[11];
    struct vt_stat vts;
    struct vt_mode VT;

    if (geteuid())
	Die("%s must be suid root\n", ProgramName);
    if ((fd = open("/dev/console", O_WRONLY,0)) < 0)
	Die("Cannot open /dev/console: %s\n", strerror(errno));
    if (ioctl(fd, VT_OPENQRY, &vtno) < 0 || vtno == -1)
	Die("Cannot find a free VT\n");
    close(fd);
    sprintf(vtname,"/dev/tty%d", vtno);	/* /dev/tty1-64 */
    if ((ConsoleFd = open(vtname, O_RDWR|O_NDELAY, 0)) < 0)
	Die("Cannot open %s: %s\n", vtname, strerror(errno));
    /*
     * Linux doesn't switch to an active vt after the last close of a vt,
     * so we do this ourselves by remembering which is active now.
     */
    if (ioctl(ConsoleFd, VT_GETSTATE, &vts) == 0)
	ActiveVT = vts.v_active;
    /*
     * Detach from the controlling tty to avoid char loss
     */
    if ((i = open("/dev/tty", O_RDWR)) >= 0) {
	ioctl(i, TIOCNOTTY, 0);
	close(i);
    }

    /*
     * now get the VT
     */
    if (ioctl(ConsoleFd, VT_ACTIVATE, vtno) != 0)
	Warn("ioctl VT_ACTIVATE: %s\n", strerror(errno));
    if (ioctl(ConsoleFd, VT_WAITACTIVE, vtno) != 0)
	Warn("ioctl VT_WAITACTIVE: %s\n", strerror(errno));

    if (ioctl(ConsoleFd, VT_GETMODE, &VT) < 0)
	Die("ioctl VT_GETMODE: %s\n", strerror(errno));
    signal(SIGUSR1, VTRequest);
    VT.mode = VT_PROCESS;
    VT.relsig = SIGUSR1;
    VT.acqsig = SIGUSR1;
    if (ioctl(ConsoleFd, VT_SETMODE, &VT) < 0)
	Die("ioctl VT_SETMODE: %s\n", strerror(errno));

    /*
     *  Switch to Graphics Mode and Open the Frame Buffer Device
     */

    if (ioctl(ConsoleFd, KDSETMODE, KD_GRAPHICS) < 0)
	Die("ioctl KDSETMODE KD_GRAPHICS: %s\n", strerror(errno));
    if ((FrameBufferFD = open(FrameBufferName, O_RDWR)) < 0)
	Die("Cannot open %s: %s\n", FrameBufferName, strerror(errno));
    if (ioctl(FrameBufferFD, FBIOGET_VSCREENINFO, &var))
	Die("ioctl FBIOGET_VSCREENINFO: %s\n", strerror(errno));

    for (i = 0; i < 256; i++)
	Grey[i] = i*0x0101;
    i = 0;
    for (r = 0x0000; r < 0x10000; r += 0x5555)
	for (g = 0x0000; g < 0x10000; g += 0x5555)
	    for (b = 0x0000; b < 0x10000; b += 0x5555) {
		Red[i] = r;
		Green[i] = g;
		Blue[i] = b;
		i++;
	    }
    var.xres_virtual = var.xres;
    var.yres_virtual = var.yres;
    var.xoffset = 0;
    var.yoffset = 0;
    var.bits_per_pixel = 8;
    var.vmode &= ~FB_VMODE_YWRAP;
    if (ioctl(FrameBufferFD, FBIOPUT_VSCREENINFO, &var))
	Die("ioctl FBIOPUT_VSCREENINFO: %s\n", strerror(errno));
    if (ioctl(FrameBufferFD, FBIOGET_FSCREENINFO, &fix))
	Die("ioctl FBIOGET_FSCREENINFO: %s\n", strerror(errno));
    DisplayWidth = var.xres;
    DisplayHeight = var.yres;
    switch (fix.type) {
	case FB_TYPE_PACKED_PIXELS:
	    Die("Packed pixels are not supported\n");
	    break;
	case FB_TYPE_PLANES:
	    if (fix.line_length)
		NextLine = fix.line_length;
	    else
		NextLine = var.xres_virtual>>3;
	    NextPlane = NextLine*var.yres_virtual;
	    break;
	case FB_TYPE_INTERLEAVED_PLANES:
	    if (fix.line_length)
		NextLine = fix.line_length;
	    else
		NextLine = fix.type_aux;
	    NextPlane = NextLine/var.bits_per_pixel;
	    break;
	default:
	    Die("Unknown frame buffer type %s\n", fix.type);
	    break;
    }
    FrameBufferSize = fix.smem_len;
    FrameBufferBits = (caddr_t)mmap(0, FrameBufferSize, PROT_READ | PROT_WRITE,
				    MAP_SHARED, FrameBufferFD, 0);
    if (FrameBufferBits == (caddr_t)-1)
	Die("mmap: %s\n", strerror(errno));
}


    /*
     *  Set the Color Mode of the Frame Buffer
     */

static void SetFBMode(int color)
{
    struct fb_var_screeninfo var;
    struct fb_cmap *cmap;

    if (ioctl(FrameBufferFD, FBIOGET_VSCREENINFO, &var))
	Die("ioctl FBIOGET_VSCREENINFO: %s\n", strerror(errno));
    if (color) {
	var.nonstd = FB_NONSTD_HAM;
	cmap = &ColorMap;
    } else {
	var.nonstd = 0;
	cmap = &GreyMap;
    }
    if (ioctl(FrameBufferFD, FBIOPUT_VSCREENINFO, &var))
	Die("ioctl FBIOPUT_VSCREENINFO: %s\n", strerror(errno));
    if (ioctl(FrameBufferFD, FBIOPUTCMAP, cmap))
	Die("ioctl FBIOPUTCMAP: %s\n", strerror(errno));
}


    /*
     *  Close the Frame Buffer
     */

static void CloseFB(void)
{
    struct vt_mode VT;

    if (FrameBufferBits != (caddr_t)-1) {
	munmap(FrameBufferBits, FrameBufferSize);
	FrameBufferBits = (caddr_t)-1;
    }
    if (FrameBufferFD != -1) {
	close(FrameBufferFD);
	FrameBufferFD = -1;
    }
    if (ConsoleFd) {
	ioctl(ConsoleFd, KDSETMODE, KD_TEXT);
	if (ioctl(ConsoleFd, VT_GETMODE, &VT) != -1) {
	    VT.mode = VT_AUTO;
	    ioctl(ConsoleFd, VT_SETMODE, &VT);
	}
	if (ActiveVT >= 0) {
	    ioctl(ConsoleFd, VT_ACTIVATE, ActiveVT);
	    ActiveVT = -1;
	}
	close(ConsoleFd);
	ConsoleFd = 0;
    }
}


void ConvertPGMRow(pixel *pixelrow, pixval maxval, int cols, u_long *fb)
{
    pixel p;
    int i, j;
    u_char chunky[32];

    for (i = 0; i < cols; i++) {
	PPM_DEPTH(p, pixelrow[i], maxval, 255);
	chunky[i%32] = PPM_GETR(p);
	if (i%32 == 31)
	    Chunky2Planar(chunky, fb++);
    }
    if (i%32) {
	while (i%32) {
	    chunky[i%32] = 0;
	    i++;
	}
	Chunky2Planar(chunky, fb++);
    }
    while (i < DisplayWidth) {
	u_long *fb2 = fb;
	for (j = 0; j < 8; j++) {
	    *fb2 = 0;
	    fb2 += NextPlane/sizeof(u_long);
	}
	i += 32;
	fb++;
    }
}


void ConvertPPMRow(pixel *pixelrow, pixval maxval, int cols, u_long *fb)
{
    pixel p;
    int red, green, blue, lastred, lastgreen, lastblue, val;
    int dred, dgreen, dblue;
    int idx, i, j;
    u_char chunky[32];

    for (i = 0; i < cols; i++) {
	PPM_DEPTH(p, pixelrow[i], maxval, 63);
	red = PPM_GETR(p);
	green = PPM_GETG(p);
	blue = PPM_GETB(p);
	if (!i) {
	    val = RedConv[red]|GreenConv[green]|BlueConv[blue];
	    lastred = Blue[val]>>10;
	    lastgreen = Red[val]>>10;
	    lastblue = Green[val]>>10;
	    idx = 0;
	} else {
	    dred = red-lastred;
	    if (dred < 0)
		dred = -dred;
	    dgreen = green-lastgreen;
	    if (dgreen < 0)
		dgreen = -dgreen;
	    dblue = blue-lastblue;
	    if (dblue < 0)
		dblue = -dblue;
	    if (dblue > dgreen)
		if (dblue > dred)
		    val = ((lastblue = blue)<<2)+1;
		else
		    val = ((lastred = red)<<2)+2;
	    else if (dgreen > dred)
		val = ((lastgreen = green)<<2)+3;
	    else
		val = ((lastred = red)<<2)+2;
	}
	chunky[i%32] = val;
	if (i%32 == 31)
	    Chunky2Planar(chunky, fb++);
    }
    if (i%32) {
	while (i%32) {
	    chunky[i%32] = 0;
	    i++;
	}
	Chunky2Planar(chunky, fb++);
    }
    while (i < DisplayWidth) {
	u_long *fb2 = fb;
	for (j = 0; j < 8; j++) {
	    *fb2 = 0;
	    fb2 += NextPlane/sizeof(u_long);
	}
	i += 32;
	fb++;
    }
}


void FillBlackRow(u_long *fb)
{
    int i;

    for (i = 0; i < 8; i++) {
	memset(fb, 0, DisplayWidth/8);
	fb += NextPlane/sizeof(u_long);
    }
}


void DisplayImage(u_char *buffer, int color)
{
    u_char *fb = (u_char *)FrameBufferBits;
    u_int i, j;

    SetFBMode(color);
    for (i = 0; i < DisplayHeight; i++) {
	u_char *src = buffer;
	u_char *dst = fb;
	for (j = 0; j < 8; j++) {
	    memcpy(dst, src, DisplayWidth/8);
	    src += NextPlane;
	    dst += NextPlane;
	}
	buffer += NextLine;
	fb += NextLine;
    }
}


void Chunky2Planar(u_char chunky[32], u_long *fb)
{
    int i;
    u_char mask;
    u_long val, planemask;

    for (mask = 1; mask; mask <<= 1) {
	val = 0;
	for (planemask = 0x80000000, i = 0; planemask; planemask >>= 1, i++)
	    if (chunky[i] & mask)
		val |= planemask;
	*fb = val;
	fb += NextPlane/sizeof(u_long);
    }
}


    /*
     *  Read an Image
     */

int ReadImage(FILE *fp, u_long *buffer)
{
    int cols, rows, format, type, imcols, imrows, i, has_color;
    pixval maxval;
    pixel *pixelrow;

    ppm_readppminit(fp, &cols, &rows, &maxval, &format);
    type = PPM_FORMAT_TYPE(format);
    switch (type) {
	case PBM_FORMAT:
	    Warn("PBM is not supported\n");
	    return(0);
	    break;

	case PGM_FORMAT:
	    has_color = 0;
	    break;

	case PPM_FORMAT:
	    has_color = 1;
	    break;

	default:
	    Warn("Unknown pixel format %d\n", format);
	    return(0);
    }
    imcols = DisplayWidth < cols ? DisplayWidth : cols;
    imrows = DisplayHeight < rows ? DisplayHeight : rows;
    pixelrow = ppm_allocrow(cols);
    for (i = 0; i < imrows; i++) {
	ppm_readppmrow(fp, pixelrow, cols, maxval, format);
	if (has_color)
	    ConvertPPMRow(pixelrow, maxval, imcols, buffer);
	else
	    ConvertPGMRow(pixelrow, maxval, imcols, buffer);
	buffer += NextLine/sizeof(u_long);
    }
    for (; i < DisplayHeight; i++) {
	FillBlackRow(buffer);
	buffer += NextLine/sizeof(u_long);
    }
    ppm_freerow(pixelrow);
    return(has_color ? 1 : -1);
}


    /*
     *  Print the Usage Template and Exit
     */

static void Usage(void)
{
    Die("Usage: %s [options] [filenames]\n\n"
        "Valid options are:\n"
        "    -h, --help             : Display this usage information\n"
        "    -f, --frame-buffer     : Frame buffer device to use\n\n",
        ProgramName);
}


int main(int argc, char *argv[])
{
    FILE *fp;
    int res, first = 1;
    u_long *buffer;

    ProgramName = argv[0];
    Files = argv+1;
    while (--argc > 0) {
	argv++;
	if (!strcmp(argv[0], "-h") || !strcmp(argv[0], "--help"))
	    Usage();
	else if (!strcmp(argv[0], "-f") || !strcmp(argv[0], "--frame-buffer"))
	    if (argc-- > 1 && !FrameBufferName) {
		FrameBufferName = argv[1];
		argv++;
	    } else
		Usage();
	else
	    Files[NumFiles++] = argv[0];
    }
    if (!FrameBufferName && !(FrameBufferName = getenv("FRAMEBUFFER")))
	FrameBufferName = "/dev/fb0current";

    signal(SIGSEGV, SigHandler);
    signal(SIGILL, SigHandler);
    signal(SIGFPE, SigHandler);
    signal(SIGBUS, SigHandler);
    signal(SIGXCPU, SigHandler);
    signal(SIGXFSZ, SigHandler);

    signal(SIGALRM, SigHandler);

    if (NumFiles)
	OpenFB();
    buffer = malloc(DisplayWidth*DisplayHeight);
    for (; NumFiles--; Files++) {
	if (!(fp = fopen(*Files, "r"))) {
	    Warn("Can't open file `%s'\n", *Files);
	    continue;
	}
	res = ReadImage(fp, buffer);
	if (!res)
	    goto NextImage;
	if (first)
	    first = 0;
	else {
#if 0
	    alarm(5);
	    getchar();
	    alarm(0);
#else
	    sleep(5);
#endif
	}
	DisplayImage((u_char *)buffer, res > 0);
	if (!NumFiles)
	    sleep(5);
NextImage:
	fclose(fp);
    }
    CloseFB();

    return(0);
}
