#include <tcl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

#include <config.h>
#include <support.h>
#include <xcio.h>
#include <account.h>
#include <version.h>
#include "info.h"
#include "const.h"

extern int	PPxPSetup(int *, char **);
extern int	PPxPRead(int, unsigned int, struct xcio_s *);
extern int	PPxPCommand(int, xcmd_t, int, char **);
extern char *	PPxPEnvGet(int, u_int8_t);
extern int	PPxPEnvRequest(int, int, char **);
extern int	PPxPRequest(int, u_int8_t);
extern xcmd_t	PPxPCommandType(char *);
extern void	PPxPUpdateRequest(int);
extern void	PPxPAutoUpdate(int, bool_t);
extern int	PPxPwdRequest(int, char *);
extern int	PPxPwdSet(int, char *, char *, char *);
extern int	PPxPListupRequest(int);
extern time_t	AccountLoad(char *, time_t, time_t, int, void (*)());

typedef struct {
    int fd;
    struct xcio_s xcio;
    int xid;
    char *name;		/* command name */
    Tcl_Channel ch;
    struct keyval_s *pppinfo;
    struct keyval_s *pwdinfo;
    struct keyval_s *coninfo;
#ifdef DEBUG
    int debug;
#endif
} PPxP;

#if DEBUG
#define MSG(ppxp, msg) if((ppxp)->debug){ printf msg; }
#else
#define MSG(ppxp, msg)
#endif

/**********************************************************************/
static void
argcerror(Tcl_Interp *interp, char *argv0, char *usage)
{
    Tcl_AppendResult(
	interp,
	"wrong # args: should be \"", argv0, " ", usage, "\"",
	(char *)NULL
	);
}

/************************************************************/
static void
cmd_close(PPxP *ppxp, Tcl_Interp *interp)
{
    if(ppxp->ch != NULL){
	Tcl_UnregisterChannel(interp, ppxp->ch);
    }
    close(ppxp->fd);
    ppxp->ch = NULL;
    ppxp->fd = -1;
	
}

static int
cmd_filehandle(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    if(ppxp->fd < 0){
	Tcl_SetResult(
	    interp,
	    "PPxP::filehandle: "
	    "Invalid file descriptor has been set to ppxp.",
	    TCL_VOLATILE
	    );
	return(TCL_ERROR);
    }
    if(ppxp->ch == NULL){
#if TCL_MAJOR_VERSION >= 8
	ppxp->ch = Tcl_MakeFileChannel(
	    (ClientData)ppxp->fd, TCL_READABLE
	    );
#else
	ppxp->ch = Tcl_MakeFileChannel(
	    (ClientData)ppxp->fd, (ClientData)ppxp->fd,
	    TCL_READABLE
	    );
#endif
	if(ppxp->ch == NULL){
	    Tcl_SetResult(
		interp,
		"PPxP::filehandle: "
		"Failed to create file channel.\n",
		TCL_VOLATILE
		);
	    return(TCL_ERROR);
	}
	Tcl_RegisterChannel(interp, ppxp->ch);
    }
    Tcl_SetResult(interp, Tcl_GetChannelName(ppxp->ch), TCL_VOLATILE);

    return(TCL_OK);
}

static void
info_append(Tcl_DString *result, struct keyval_s *info,
	    struct xcio_s *xc, struct ppxpinfo_ops *ops)
{
    int n, i;

    n = ops->parse(info, xc->buf, xc->len);
    for(i = 0; i < n; i++){
	Tcl_DStringStartSublist(result);
	Tcl_DStringAppendElement(result, info[i].keyval[0]);
	Tcl_DStringAppendElement(result, info[i].keyval[1]);
	Tcl_DStringEndSublist(result);
    }
    return;
}

static void
parse_env(Tcl_DString *result, char *buf, int len)
{
    char name[64];
    char *envv[7];
    int envc, i;

    envc = DecodeArgs(envv, buf, len, 7);
    if(*envv[2] == '\0'){
	sprintf(name, "%s", envv[0]);
    }else{
	sprintf(name, "%s.%s", envv[2], envv[0]);
    }
    Tcl_DStringAppendElement(result, name);
    Tcl_DStringAppendElement(result, envv[1]);
    for(i = 3; i <= 5; i++){
	Tcl_DStringAppendElement(result, envv[i]);
    }
    FreeArgs(envc, envv);
}

static void
parse_listup(Tcl_DString *result, char *buf, int len)
{
    char num[8];
    if((unsigned char)buf[0] == XLABEL_COMMAND)
	sprintf(num, "CMD.%d", buf[1]);
    else
	sprintf(num, "%d.%d", buf[0], buf[1]);
    Tcl_DStringAppendElement(result, num);
    Tcl_DStringAppendElement(result, &buf[3]);
}

static int
cmd_read(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    int xid, type, r;
    Tcl_DString result;
    struct xcio_s *xc = &ppxp->xcio;

    if(argc > 2){
	if(Tcl_GetInt(interp, argv[2], &xid) == TCL_ERROR){
	    Tcl_AppendResult(
		interp,
		"PPxP::read: Can't get ID from ",
		argv[1], (char *)NULL
		);
	    return(TCL_ERROR);
	}
    }else{
	xid = XID_ANY;
    }
    if((r = PPxPRead(ppxp->fd, xid, xc)) < 0){
	Tcl_AppendResult(
	    interp, "PPxP::read: Read failed.", (char *)NULL
	    );
	cmd_close(ppxp, interp);
	return(TCL_ERROR);
    }else if(r == 0){
	sprintf(interp->result, "%d", 0);
	return(TCL_OK);
    }
    sprintf(interp->result, "%d", xc->type);
    if(argc < 2){
	return(TCL_OK);
    }
    Tcl_DStringInit(&result);

    switch(xc->type & XCIO_MASK){
    case XCIO_WAIT:
	break;
    case XCIO_RETURN:
    {
	char buf[16];
	sprintf(buf, "%d", (int)xc->buf[0]);
	Tcl_DStringAppend(&result, buf, strlen(buf));
	MSG(ppxp, ("PPxP:read:XCIO_RETURN:%s\n", result));
    }
	break;
    case XCIO_S_OUT:
	Tcl_DStringAppend(&result, xc->buf, xc->len);
	break;
    case XCIO_MESSAGE:
	Tcl_DStringAppend(&result, xc->buf, xc->len);
	break;
    case XCIO_ENV_SET:
	parse_env(&result, xc->buf, xc->len);
	MSG(ppxp, ("PPxP:read:XCIO_ENV_SET:%s\n", result));
	break;
    case XCIO_UP_ENVS:
	parse_env(&result, xc->buf, xc->len);
	MSG(ppxp, ("PPxP:read:XCIO_UP_ENVS:%s\n", result));
	break;
    case XCIO_LISTUP:
	parse_listup(&result, xc->buf, xc->len);
	break;
    case XCIO_PWD_SET:
	if(!ppxp->pwdinfo)
	    ppxp->pwdinfo = PPxP_PwdInfo.new();
	info_append(&result, ppxp->pwdinfo, xc, &PPxP_PwdInfo);
	break;
    case XCIO_CONSOLES:
	if(!ppxp->coninfo)
	    ppxp->coninfo = PPxP_ConInfo.new();
	info_append(&result, ppxp->coninfo, xc, &PPxP_ConInfo);
	break;
    case XCIO_UP_INFO:
	if(!ppxp->pppinfo)
	    ppxp->pppinfo = PPxP_pppInfo.new();
	info_append(&result, ppxp->pppinfo, xc, &PPxP_pppInfo);
	break;
    default:
    {
	char buf[128];
	Tcl_ResetResult(interp);
	sprintf(buf, "(type=%d,id=%d,len=%d)",
		xc->type, xc->xid, xc->len);
	Tcl_AppendResult(
	    interp,
	    "PPxP::read1: Unknown return code from ppxpd ",
	    buf, (char *)NULL
	    );
	return(TCL_ERROR);
    }
    }
    if(!Tcl_SetVar(interp, argv[1], Tcl_DStringValue(&result), 0)){
	return(TCL_ERROR);
    }
    Tcl_DStringFree(&result);

    return(TCL_OK);
}

static int
cmd_nread(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    Tcl_DString nresult;
    struct xcio_s *xc = &ppxp->xcio;
    char *var, *val;

    if(argc < 2){
	Tcl_AppendResult(
	    interp,
	    "Wrong # args.",
	    (char *)NULL);
	return(TCL_ERROR);
    }
    var = argv[1];
    Tcl_DStringInit(&nresult);

    for(;;){
	if(cmd_read(ppxp, interp, argc, argv) == TCL_ERROR){
	    return(TCL_ERROR);
	}
	val = Tcl_GetVar(interp, var, 0);
	Tcl_DStringAppendElement(&nresult, val);
	if(xc->type & XCIO_LAST){
	    goto output;
	}else{
	    switch(xc->type & XCIO_MASK){
	    case XCIO_RETURN:
	    case XCIO_UP_INFO:
		goto output;
	    }
	}
    }
    return(TCL_ERROR);
output:
    if(!Tcl_SetVar(interp, var, Tcl_DStringValue(&nresult), 0)){
	return(TCL_ERROR);
    }
    Tcl_DStringResult(interp, &nresult);
    return(TCL_OK);
}

static int
cmd_console(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    ppxp->xid = PPxPRequest(ppxp->fd, XCIO_CONSOLES);
    sprintf(interp->result, "%d", ppxp->xid);
    return(TCL_OK);
}

static int
cmd_env(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    ppxp->xid = PPxPEnvRequest(ppxp->fd, argc - 1, argv + 1);
    sprintf(interp->result, "%d", ppxp->xid);
    return(TCL_OK);
}

static int
cmd_listup(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    ppxp->xid = PPxPListupRequest(ppxp->fd);
    sprintf(interp->result, "%d", ppxp->xid);
    return(TCL_OK);
}

static int
cmd_update(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    PPxPUpdateRequest(ppxp->fd);

    ppxp->xid = XID_UPDATE;
    sprintf(interp->result, "%d", XID_UPDATE);

    return(TCL_OK);
}

static int
cmd_autoupdate(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    bool_t sw;
    int r;

    if(argc > 1){
	int i;
	if((r = Tcl_GetBoolean(interp, argv[1], &i)) != TCL_OK)
	    return(r);
	sw = i ? TRUE : FALSE;
    }else{
	sw = TRUE;
    }
    PPxPAutoUpdate(ppxp->fd, sw);

    ppxp->xid = XID_UPDATE;
    sprintf(interp->result, "%d", XID_UPDATE);

    return(TCL_OK);
}

static int
cmd_passwd(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    int xid;
    int r = TCL_ERROR;

    if(argc < 2){
	xid = PPxPwdRequest(ppxp->fd, NULL);
	r = TCL_OK;
    }else if(argc < 3){
	xid = PPxPwdRequest(ppxp->fd, argv[1]);
	r = TCL_OK;
    }else if(argc < 4){
	xid = -1;
	r = TCL_ERROR;
    }else{
	if(*argv[1] == '\0') argv[1] = NULL;
	xid = PPxPwdSet(ppxp->fd, argv[1], argv[2], argv[3]);
	r = TCL_OK;
    }
    ppxp->xid = xid;
    sprintf(interp->result, "%d", xid);
    return(r);
}

static int
cmd_input(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    struct xcio_s xc;

    xc.len = strlen(argv[1]);
    memcpy(xc.buf, argv[1], xc.len);
    xc.type = XCIO_S_IN;
    xc.xid = 1;
    return(XcioWrite(ppxp->fd, &xc) > 0 ? TCL_OK : TCL_ERROR);
}

#ifdef DEBUG
static int
cmd_debug(PPxP *ppxp, Tcl_Interp *interp, int argc, char **argv)
{
    ppxp->debug = ppxp->debug ? 0 : 1;
    return(TCL_OK);
}
#endif

static struct ppxpcmd_s {
    const char *name;
    int (*proc)(PPxP *, Tcl_Interp *, int, char **);
    int argc;
} builtin_list[] = {
    { "filehandle",	cmd_filehandle,	1 },
    { "read",		cmd_read,	2 },
    /* { "nread",	cmd_nread,	1 }, */
    { "console",	cmd_console,	1 },
    { "env",		cmd_env,	1 },
    { "listup",		cmd_listup,	1 },
    { "update",		cmd_update,	1 },
    { "autoupdate",	cmd_autoupdate,	1 },
    { "passwd",		cmd_passwd,	1 },
    { "input",		cmd_input,	2 },
#ifdef DEBUG
    { "debug",		cmd_debug,	1 },
#endif
    { NULL, NULL }
};

/************************************************************/
static int
builtin_request(PPxP *ppxp, Tcl_Interp *interp, int argc, char *argv[])
{
    struct ppxpcmd_s *c;
    char *cmd = argv[0];

    for(c = builtin_list; c->name; c++){
	if(strcmp(cmd, c->name) == 0){
	    break;
	}
    }
    if(!c->name || !c->proc || argc < c->argc){
	return(TCL_ERROR);
    }
    MSG(ppxp, ("PPxP:builtin_request:%s\n", cmd));

    return((*c->proc)(ppxp, interp, argc, argv));
}

static int
command_request(PPxP *ppxp, Tcl_Interp *interp, int argc, char *argv[])
{
    xcmd_t xcmd;
    int ret;

    if((xcmd = PPxPCommandType(argv[0])) >= XCMD_MAX){
	return(TCL_ERROR);
    }
    ppxp->xid = PPxPCommand(ppxp->fd, xcmd, argc - 1, argv + 1);
    MSG(ppxp, ("PPxP:command_request:xcmd=%d,xid=%d\n"));

    if(xcmd == XCMD_BYE || xcmd == XCMD_QUIT){
	/* close(ppxp->fd); */
	cmd_close(ppxp, interp);
	Tcl_DeleteCommand(interp, ppxp->name);
	ret = TCL_OK;
    }else if(ppxp->xid < 0){
	ret = TCL_ERROR;
    }else{
	sprintf(interp->result, "%d", ppxp->xid);
	ret = TCL_OK;
    }
    return(ret);
}

static int
RunCmd(ClientData clientData, Tcl_Interp *interp,
       int argc, char *argv[])
{
    int ret;
    PPxP *ppxp = (PPxP *)clientData;

    if(argc < 2){
	argcerror(interp, argv[0], "command");
	return(TCL_ERROR);
    }
    if((ret = builtin_request(ppxp, interp, argc - 1, argv + 1))
       == TCL_ERROR){
	ret = command_request(ppxp, interp, argc - 1, argv + 1);
    }
    return(ret);
}

/**********************************************************************/

static void
DeletePPxP(ClientData clientData)
{
    PPxP *ppxpPtr = (PPxP *)clientData;

    free(ppxpPtr->name);
    if(ppxpPtr->pppinfo) free(ppxpPtr->pppinfo);
    if(ppxpPtr->pwdinfo) free(ppxpPtr->pwdinfo);
    if(ppxpPtr->coninfo) free(ppxpPtr->coninfo);

    free((char *)clientData);
}

static int
SetupCmd(ClientData clientData, Tcl_Interp *interp,
	 int argc, char *argv[])
{
    PPxP *ppxpPtr;
    int fd;
    char *argv0, *tmpv0, name[64];

    tmpv0 = argv[0];
    argv[0] = Tcl_GetVar(interp, "argv0", TCL_GLOBAL_ONLY);
    fd = PPxPSetup(&argc, argv);

    if(fd < 0){
	interp->result = "Failed";
	return(TCL_ERROR);
    }
    if((ppxpPtr = (PPxP *)malloc(sizeof(PPxP))) == NULL){
	interp->result = "Can't malloc";
	return(TCL_ERROR);
    }
    memset(ppxpPtr, 0, sizeof(PPxP));
    ppxpPtr->fd = fd;
    ppxpPtr->ch = NULL;
    ppxpPtr->xid = -1;
#ifdef DEBUG
    ppxpPtr->debug = 0;
#endif
    sprintf(name, "ppxp%d", fd);
    Tcl_SetResult(interp, name, TCL_VOLATILE);
    ppxpPtr->name = strdup(name);

    ppxpPtr->pppinfo = NULL;
    ppxpPtr->pwdinfo = NULL;
    ppxpPtr->coninfo = NULL;

    Tcl_CreateCommand(interp, interp->result, RunCmd,
		      (ClientData)ppxpPtr, DeletePPxP);

    return(TCL_OK);
}

/**********************************************************************/

static char *account_varname;
static char *account_command;
static Tcl_Interp *account_interp;

#define ACCSETVAR(elem, val) \
(Tcl_SetVar2(account_interp, account_varname, elem, val, 0))

static void
AccountCallback(struct accrec_s *on, struct accrec_s *off, time_t dif)
{
    char buf[32];
    struct passwd *pw;

    pw = (on && on->name[0]) ? getpwuid(on->uid) : NULL;
    if(on){
	ACCSETVAR("name", on->name);
	ACCSETVAR("user", (pw && pw->pw_name) ? pw->pw_name : "");
	sprintf(buf, "%d", (int)on->pid);
	ACCSETVAR("pid", buf);
	sprintf(buf, "%ld", (long)on->time.sec);
	ACCSETVAR("ontime", buf);
    }else{
	ACCSETVAR("name", "");
	ACCSETVAR("user", "");
	ACCSETVAR("pid", "");
	ACCSETVAR("ontime", "");
    }
    if(off){
	sprintf(buf, "%ld", (long)off->time.sec);
	ACCSETVAR("offtime", buf);
    }else{
	ACCSETVAR("offtime", "");
    }
    sprintf(buf, "%ld", (long)dif);
    ACCSETVAR("time", buf);

    Tcl_Eval(account_interp, account_command);
}

#define ACCOUNT_USAGE "varname ?-name name? ?-start start? ?-end end? ?-command varname command?"

static int
AccountLoadCmd(ClientData client, Tcl_Interp *interp, int argc, char *argv[])
{
    int ret, i;
    time_t total;
    char buf[32];

    char *name = NULL;
    int ssec = 0;		/* time_t */
    int esec = 0;		/* time_t */
    int nmax = 0;

    if(argc < 2){
	argcerror(interp, argv[0], ACCOUNT_USAGE);
	return(TCL_ERROR);
    }
    account_varname = NULL;
    account_command = NULL;
    account_interp = interp;

    ret = TCL_OK;
    for(i = 1; i < argc; i++){
	if(strcmp(argv[i], "-name") == 0){
	    if(++i > argc){
		ret = TCL_ERROR;
		break;
	    }
	    name = strdup(argv[i]);
	}else if(strcmp(argv[i], "-start") == 0){
	    if(++i > argc){
		ret = TCL_ERROR;
		break;
	    }
	    ret = Tcl_GetInt(interp, argv[i], &ssec);
	}else if(strcmp(argv[i], "-end") == 0){
	    if(++i > argc){
		ret = TCL_ERROR;
		break;
	    }
	    ret = Tcl_GetInt(interp, argv[i], &esec);
	}else if(strcmp(argv[i], "-command") == 0){
	    if(++i > argc){
		ret = TCL_ERROR;
		break;
	    }
	    account_varname = strdup(argv[i]);
	    if(++i > argc){
		ret = TCL_ERROR;
		break;
	    }
	    account_command = strdup(argv[i]);
	}else{
	    ret = TCL_ERROR;
	    break;
	}
    }
    if(ret == TCL_OK){
	total = AccountLoad(name, ssec, esec, nmax,
			    account_command ? AccountCallback : NULL);
    }else{
	total = (time_t)-1;
    }
    if(account_command) free(account_command);
    if(account_varname) free(account_varname);
    if(name) free(name);

    sprintf(buf, "%ld", (long)total);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);

    return(ret);
}

/**********************************************************************/

static int
XcioTypeCmd(ClientData client, Tcl_Interp *interp, int argc, char *argv[])
{
    int n;
    char *s;

    if(argc != 2){
	argcerror(interp, argv[0], "type");
	return(TCL_ERROR);
    }
    if(Tcl_GetInt(interp, argv[1], &n) != TCL_OK){
	return(TCL_ERROR);
    }
    s = PPxP_XcioStr(n);

    if(s)
	sprintf(interp->result, "%s", s);
    else
	sprintf(interp->result, "");
    return(TCL_OK);
}

static int
XcioLastCmd(ClientData client, Tcl_Interp *interp, int argc, char *argv[])
{
    int n;

    if(argc != 2){
	argcerror(interp, argv[0], "type");
	return(TCL_ERROR);
    }
    if(Tcl_GetInt(interp, argv[1], &n) != TCL_OK){
	return(TCL_ERROR);
    }
    if(n & XCIO_LAST)
	sprintf(interp->result, "1");
    else
	sprintf(interp->result, "0");
    return(TCL_OK);
}

/**********************************************************************/

#ifndef PREFIX
#define PREFIX "PPxP"
#endif

int
PPxP_Init(Tcl_Interp *interp)
{
    struct ppxp_const_s *p;
    char name[64], val[256];

    Tcl_CreateCommand(interp, PREFIX "Setup", SetupCmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateCommand(interp, PREFIX "AccountLoad", AccountLoadCmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateCommand(interp, "XcioType", XcioTypeCmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateCommand(interp, "XcioLast", XcioLastCmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);

    for(p = PPxP_Name2Const; p->name; p++){
	sprintf(name, PREFIX "_%s", p->name);
	sprintf(val, "%d", p->val);
	Tcl_SetVar(interp, name, val, TCL_GLOBAL_ONLY);
    }
    DirNameInit(getuid());
    if(usrPPxP){
	Tcl_SetVar(interp, PREFIX "_UsrPath", usrPPxP,
		   TCL_GLOBAL_ONLY);
    }
    if(sysPPxP){
	Tcl_SetVar(interp, PREFIX "_SysPath", sysPPxP,
		   TCL_GLOBAL_ONLY);
    }
    Tcl_PkgProvide(interp, "PPxP", VERSION);
    return(TCL_OK);
}

/* dummy */
int
Ppxp_Init(Tcl_Interp *interp)
{
    return(PPxP_Init(interp));
}
