/*****
*
* Copyright (C) 1999,2000, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <stdarg.h>
#include <sys/types.h>
#include <inttypes.h>
#include <netinet/in.h>

#include <libprelude/list.h>
#include <libprelude/idmef-tree.h>
#include <libprelude/prelude-log.h>
#include <libprelude/extract.h>
#include "nethdr.h"
#include "optparse.h"
#include "passive-os-fingerprint.h"

#define MAX_OPTS_LEN 40

#ifndef IPOPT_SECURITY
 #define IPOPT_SECURITY 130
#endif

#ifndef IPOPT_RA
 #define IPOPT_RA 148
#endif


/*
 * From rfc793 :
 * Options may occupy space at the end of the TCP header and are a
 * multiple of 8 bits in length.  All options are included in the
 * checksum.  An option may begin on any octet boundary.  There are two
 * cases for the format of an option:
 *
 *  Case 1:  A single octet of option-kind.
 *
 *  Case 2:  An octet of option-kind, an octet of option-length, and
 *           the actual option-data octets.
 *
 * The option-length counts the two octets of option-kind and
 * option-length as well as the option-data octets.
 *
 * Note that the list of options may be shorter than the data offset
 * field might imply.  The content of the header beyond the
 * End-of-Option option must be header padding (i.e., zero).
 *
 * A TCP must implement all options.
 */


static char *buf;
static size_t bsize;
extern pof_host_data_t pof_host_data;



static int printopt(const char *comment, ...) 
{
        int ret;
        va_list va;

        va_start(va, comment);
        ret = vsnprintf(buf, bsize, comment, va);
        va_end(va);

        if ( ret >= bsize || ret < 0 )
                return -1;
        
        buf += ret;
        bsize -= ret;

        return 0;
}



/*
 * Dump tcp options and their value.
 */
static int tcp_optval(unsigned char *optbuf, int opt, int datalen) 
{
        int i;
        
        switch (opt) {
                
        case TCPOPT_MAXSEG:
                printopt("mss %u", extract_uint16(optbuf));
                pof_host_data.mss = extract_uint16(optbuf);
                break;
                
        case TCPOPT_WSCALE:
                printopt("wscale %u", *optbuf);
                pof_host_data.wscale = *optbuf;
                break;

        case TCPOPT_SACK_PERMITTED:
                printopt("sackOK");
                pof_host_data.sackok = 1;
                break;
                
        case TCPOPT_SACK:
                if ( datalen % 8 != 0 )
                        printopt("malformed sack");
                else {
                        uint32_t s, e;

                        printopt("sack %d", datalen / 8 );
                        for ( i = 0; i < datalen; i += 8 ) {
                                s = extract_uint32(optbuf + i);
                                e = extract_uint32(optbuf + i + 4);
                        }
                        
                }
                break;
                                
        case TCPOPT_ECHO:
                printopt("echo %u", extract_uint32(optbuf));
                break;
                              
        case TCPOPT_ECHOREPLY:
                printopt("echoreply %u", extract_uint32(optbuf));
                break;

        case TCPOPT_TIMESTAMP:
                pof_host_data.timestamp = 1;
                printopt("timestamp %u %u",
                         extract_uint32(optbuf), extract_uint32(optbuf + 4));
                break;
                
        case TCPOPT_CC:
                printopt("cc %u", extract_uint32(optbuf));
                break;

        case TCPOPT_CCNEW:
                printopt("ccnew %u", extract_uint32(optbuf));
                break;
                
        case TCPOPT_CCECHO:
                printopt("ccecho %u", extract_uint32(optbuf));
                break;

        default:
                printopt("opt-%d:", opt);
                break;

        }

        return -1;
}



/*
 * Dump Ip options and their value.
 */
static int ip_optval(unsigned char *optbuf, int opt, int datalen)
{
        int optlen = datalen + 2;

        switch (opt) {
                                
        case IPOPT_TIMESTAMP:
                printopt("ts");
                break;
                
        case IPOPT_SECURITY:
                printopt("security{%d}", optlen);
                break;
                
        case IPOPT_RR:
                printopt("rr");
                break;
                
        case IPOPT_SSRR:
                printopt("ssrr");
                break;
                
        case IPOPT_LSRR:
                printopt("lsrr");
                break;

        case IPOPT_RA:
                if (datalen != 2)
                        printopt("ra{%d}", optlen);
                else if (optbuf[0] || optbuf[1])
                        printopt("ra{%d.%d}", optbuf[0], optbuf[1]);
                break;
                
        default:
                printopt("ipopt-%d{%d}", opt, optlen);
                break;
        }
        
        return -1;
}



/*
 * Verify if the option 'opt' is one of
 * the 1 byte only option (nop || eol).
 */
static int is_1byte_option(int opt) 
{
        if ( opt == TCPOPT_NOP ) {
                printopt("nop");
                pof_host_data.nop = 1;
                return 0;
        }

        else if (opt == TCPOPT_EOL) {
                printopt("eol");
                return 0;
        }

        return -1;
}



/*
 * Verify that an option is valid :
 * - verify that this option len is > 2 bytes.
 * - verify that this option len is < than our total option len.
 * - do some bound check on our option buffer, to avoid going out of bound.
 */
static int is_option_valid(unsigned char *optbuf, int optlen, int totlen) 
{        
        if ( optlen < 2 ) {
                printopt("options is not \"nop\" or \"eol\" so option len (%d) "
                         "should be >= 2.", optlen);
                return -1;
        }
                
        if ( optlen > totlen ) {
                printopt("option len (%d) is > remaining total options len (%d).",
                         optlen, totlen);
                return -1;
        }

        /*
         * This check should never be reached because
         * of the optlen > totlen test.
         */
        if ( (optbuf + (optlen - 2)) > (optbuf + (totlen - 2) ) ) {
                printopt("options buffer seem to be truncated (%p > %p).",
                         (optbuf + (optlen - 2)),  (optbuf + (totlen - 2)));
                return -1;
        }

        return 0;
}



/*
 * Verify that our total options len is big enough
 * to contain a len byte, which mean totlen must be
 * >= 2 (1 byte for optkind, and 1 for optlen).
 */
static int is_len_byte_ok(int totlen) 
{
        if ( totlen < 2 ) {
                printopt("not \"nop\" or \"eol\", "
                         "but no space remaining for option len byte"
                         "in option buffer.");
                return -1;
        }
        
        return 0;
}



/*
 * Walk options in 'optbuf' of total len 'totlen',
 * the callback function optval should point on a function
 * printing tcp or ip options, depending on the kind of header
 * theses options are from.
 */
static int walk_options(unsigned char *optbuf, int totlen,
                        int (*optval)(unsigned char *optbuf, int opt, int optlen)) 
{
        int opt, optlen, ret;
        
        do {
                opt = *optbuf++;

                if ( is_1byte_option(opt) == 0 )
                        totlen -= 1;
                else {
                        if ( is_len_byte_ok(totlen) < 0 )
                                return -1;

                        optlen = *optbuf++;
                        
                        ret = is_option_valid(optbuf, optlen, totlen);
                        if ( ret < 0 )
                                return -1;
                        
                        optval(optbuf, opt, optlen - 2);
                        totlen -= optlen;
                        optbuf += optlen - 2;

                }
                
                assert(totlen >= 0);
                
                if ( totlen > 0 )
                        printopt(",");
                
        } while ( totlen != 0 );

        return 0;
}



/*
 *
 */
const char *tcp_optdump(unsigned char *optbuf, size_t optlen)
{
        static char buffer[1024];
        
        buf = buffer;
        bsize = sizeof(buffer);
        
        if ( optlen > MAX_OPTS_LEN ) {
                printopt("total option len (%d) > maximum option len (%d).",
                         optlen, MAX_OPTS_LEN);
                return buffer;
        }
        
        walk_options(optbuf, optlen, tcp_optval);

        return buffer;
}



/*
 *
 */
const char *ip_optdump(unsigned char *optbuf, size_t optlen)
{
        static char buffer[1024];
        
        buf = buffer;
        bsize = sizeof(buffer);

        if ( optlen > MAX_OPTS_LEN ) {
                printopt("total option len (%d) > maximum option len (%d).",
                         optlen, MAX_OPTS_LEN);
                return buffer;
        }

        walk_options(optbuf, optlen, ip_optval);
        
        return buffer;
}
