/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: slidenode.cc,v $
 * Revision 1.4  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.3  2001/04/27 14:43:46  slash
 * Fixed the determination of the child-node, so libxml-2.3.* can be used.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:32  slash
 * Initial release.
 *
 */

#include <stdlib.h>
#include <stdio.h>

#include "main.h"
#include "outputfilter.h"
#include "titlenode.h"
#include "textnode.h"
#include "codenode.h"
#include "imagenode.h"
#include "itemizenode.h"
#include "enumeratenode.h"
#include "gridnode.h"
#include "slidenode.h"
#include "sectionnode.h"
#include "presentation.h"
#include "parameters.h"


// The following macro is used to provide an easy handling,
// if the apropriate property isn't given in the XML-file.
#define LATEX_  (latex_ != -1 ? 0 : 1)


vector<SlideNode*> SlideNode::All_Slides_;
char SlideNode::Filename_Of_First_Slide_[64] = "";
char SlideNode::Filename_Of_Last_Slide_[64] = "";
const char *SlideNode::Section_Title_ = NULL;
char SlideNode::Section_Filename_[64] = "";
int SlideNode::Section_Written_In_Outline_ = 0;

//---------------------------------------------------------------------------
SlideNode::SlideNode(xmlNodePtr node) : Node()
{
    All_Slides_.push_back(this);
    sequence_number_ = All_Slides_.size();
    number_of_subslides_ = -1;
    latex_ = -1;

    // Extract the properties of the slide-tag.
    title_ = (char*)xmlGetProp(node, (xmlChar*)"title");
    note_  = (char*)xmlGetProp(node, (xmlChar*)"note");
    char *latex = (char*)xmlGetProp(node, (xmlChar*)"latex");
    if (latex) {
        if      (!strcmp(latex, "yes")) latex_ = 1;
        else if (!strcmp(latex, "no"))  latex_ = 0;
    }

    // Extract the environment
    for (node = node->xmlChildrenNode; node; node = node->next) {
        EnvironmentNode *environment = NULL;

        if (node->type == XML_ELEMENT_NODE) {
            if (!strcmp((char*)node->name,"title")) {
                environment = new TitleNode(node);
            } else if (!strcmp((char*)node->name,"itemize")) {
                environment = new ItemizeNode(node);
            } else if (!strcmp((char*)node->name,"enumerate")) {
                environment = new EnumerateNode(node);
            } else if (!strcmp((char*)node->name,"image")) {
                environment = new ImageNode(node);
            } else if (!strcmp((char*)node->name,"code")) {
                environment = new CodeNode(node);
            } else if (!strcmp((char*)node->name,"text")) {
                environment = new TextNode(node);
            } else if (!strcmp((char*)node->name,"grid")) {
                environment = new GridNode(node);
            } else if (node->type != XML_COMMENT_NODE) {
                Number_Of_Errors_++;
            }
        }

        if (environment) {
            if (environment->isHTMLStepped()) {
                if ((signed)environment->getNumberOfHTMLSteps() > number_of_subslides_) {
                    number_of_subslides_ = environment->getNumberOfHTMLSteps();
                }
            }
            child_nodes_.push_back(environment);
        }
    }
}

//---------------------------------------------------------------------------
SlideNode::~SlideNode()
{
}


//---------------------------------------------------------------------------
void SlideNode::writeHTML(int parameter) const
{
    if (number_of_subslides_ == -1) {
        // No subslides. There's just one slide to write.
        writeSubSlideHTML(-1);

    } else {
        // There are "number_of_subslides_" subslides to write.
        for (int i=1; i<=number_of_subslides_; i++) {
            writeSubSlideHTML(i);
        }
    }
}

//---------------------------------------------------------------------------
void SlideNode::writeLaTeX() const
{
    if (LATEX_ == 0)  return;

    if (Parameters_.isOutlineEnabled()) {
        if (Section_Title_ != NULL) {
            if (Section_Written_In_Outline_ == 0) {
                Output_ << "\\section{";
                OutputFilter::writeLaTeX(Section_Title_, 1);
                Output_ << "}" << endl;
                Section_Written_In_Outline_ = 1;
            }
            Output_ << "\\subsection{";
            OutputFilter::writeLaTeX(title_, 1);
            Output_ << "}" << endl;
        } else {
            Output_ << "\\section{";
            OutputFilter::writeLaTeX(title_, 1);
            Output_ << "}" << endl << endl;
        }
    } else {
        Output_ << "\\begin{slide}" << endl;
        if (Section_Title_ != NULL) {
            Output_ << "\\slideheading{";
            OutputFilter::writeLaTeX(Section_Title_, 1);
            Output_ << "}" << endl
                    << "\\slidesubheading{";
            OutputFilter::writeLaTeX(title_, 1);
            Output_ << "}" << endl;
        } else {
            Output_ << "\\slideheading{";
            OutputFilter::writeLaTeX(title_, 1);
            Output_ << "}" << endl;
        }
    }
    
    if (!child_nodes_.empty()) {
        child_nodes_[0]->writeLaTeX();
    }

    if (Parameters_.isOutlineEnabled()) {
        Output_ << endl << endl;
    } else {
        Output_ << "\\end{slide}" << endl << endl;
    }
}

//---------------------------------------------------------------------------
void SlideNode::writeXML() const
{
    Output_ << "<slide"
            << " title=\"" << title_ << "\"";
    if (note_) {
        Output_ << " note=\"" << (note_ ? note_ : "") << "\"";
    }
    if (latex_ != -1) {
        Output_ << " latex=\"" << (latex_ == 1 ? "yes" : "no") << "\"";
    }
    Output_ << ">" << endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</slide>" << endl << endl;
}


//---------------------------------------------------------------------------
void SlideNode::initFilenameOfFirstAndLastSlide()
{
    // Initialize the filename of the first slide.
    strcpy(Filename_Of_First_Slide_,
           All_Slides_[0]->number_of_subslides_ > 0 ? "01_1.html" : "01.html");

    // Initialize the filename of the last slide.
    SlideNode *last_slide = All_Slides_.back();
    if (last_slide->number_of_subslides_ > 0) {
        sprintf(Filename_Of_Last_Slide_, "%02d_%d.html",
                last_slide->sequence_number_,
                last_slide->number_of_subslides_);
    } else {
        sprintf(Filename_Of_Last_Slide_, "%02d.html",
                last_slide->sequence_number_);
    }
}


//---------------------------------------------------------------------------
void SlideNode::writeTOCEntry() const
{
    char filename[64];
    sprintf(filename,
            number_of_subslides_ > 1 ? "%02d_1.html" : "%02d.html",
            sequence_number_);

    Output_ << "<A href=\"" << filename << "\">" << title_ << "</A>";
}


//---------------------------------------------------------------------------
void SlideNode::writeSubSlideHTML(const int sub_slide) const
{
    // Determine the filename of this (sub)slide.
    char filename[64];
    if (sub_slide > 0) {
        sprintf(filename, "%02d_%d.html", sequence_number_, sub_slide);
    } else {
        sprintf(filename, "%02d.html", sequence_number_);
    }

    // If we are inside of a section, the filename of the first slide
    // must be stored.
    if ((Section_Title_ != NULL) && (Section_Filename_[0] == '\0')) {
        strcpy(Section_Filename_, filename);
    }


    // Determine the filename of the previous slide.
    char filename_of_prev_slide[64] = "";

    if (sub_slide > 1) {
        // This is a subslide and there is at least one previous subslide.
        sprintf(filename_of_prev_slide, "%02d_%d.html",
                sequence_number_, sub_slide-1);

    } else if (sequence_number_ > 1) {
        // Get the previous slide.
        SlideNode *prev_slide = All_Slides_[sequence_number_-2];

        if (prev_slide->number_of_subslides_ == -1) {
            // The previous slide doesn't have any subslides.
            sprintf(filename_of_prev_slide, "%02d.html", sequence_number_-1);

        } else {
            // The previous slide has subslides. Take the filename of the last.
            sprintf(filename_of_prev_slide, "%02d_%d.html", sequence_number_-1,
                    prev_slide->number_of_subslides_);
        }
    }


    // Determine the filename of the next slide.
    char filename_of_next_slide[64] = "";

    if ((sub_slide > 0) && (sub_slide < number_of_subslides_)) {
        // This is a subslide and there is at least one subslide left.
        sprintf(filename_of_next_slide, "%02d_%d.html",
                sequence_number_, sub_slide+1);

    } else if (sequence_number_ < All_Slides_.size()) {
        // Get the next slide.
        SlideNode *next_slide = All_Slides_[sequence_number_];

        if (next_slide->number_of_subslides_ == -1) {
            // The next slide doesn't have any subslides.
            sprintf(filename_of_next_slide, "%02d.html", sequence_number_+1);

        } else {
            // The next slide has subslides. Take the filename of the first.
            sprintf(filename_of_next_slide, "%02d_1.html", sequence_number_+1);
        }
    }


    // Write the slide to the HTML-file.
    Output_.open(filename);
    writeHTMLHeader(filename_of_prev_slide, filename_of_next_slide);
    if (!child_nodes_.empty()) {
        child_nodes_[0]->writeHTML(sub_slide);
    }
    writeHTMLFooter(filename_of_prev_slide, filename_of_next_slide);
    Output_.close();
}


//---------------------------------------------------------------------------
void SlideNode::writeHTMLHeader(const char *prev_slide,
                                const char *next_slide) const
{
    // Start with the HTML-header.
    Output_ << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">" << endl
            << "<HTML>" << endl
            << "<HEAD>" << endl
            << "<META"
            << " name=\"Generator\""
            << " content=\"PresTiMeL " << VERSION << "\""
            << ">" << endl
            << "<TITLE>";
    OutputFilter::writeHTML(title_, 1);
    Output_ << "</TITLE>" << endl;

    // Link to the Table of Contents.
    Output_ << "<LINK"
            << " href=\"toc.html\""
            << " rel=\"contents\""
            << " title=\"Contents\""
            << ">" << endl;

    // Link to the first slide.
    if (strlen(prev_slide) > 0) {
        Output_ << "<LINK"
                << " href=\"" << prev_slide << "\""
                << " rel=\"previous\""
                << " title=\"Previous slide\""
                << ">" << endl;
    }

    // Link to the next slide.
    if (strlen(next_slide) > 0) {
        Output_ << "<LINK"
                << " href=\"" << next_slide << "\""
                << " rel=\"next\""
                << " title=\"Next slide\""
                << ">" << endl;
    }

    // Link the stylesheet-file.
    Output_ << "<LINK"
            << " href=\"styles.css\""
            << " rel=\"stylesheet\""
            << " type=\"text/css\""
            << ">" << endl;

    // End of HTML-header. Start with the body.
    Output_ << "</HEAD>" << endl
            << "<BODY class=\""
            << (Parameters_.isSlideBackgroundEnabled() ? "graphic" : "color")
            << "\">" << endl;

    // Write the navigation-buttons, if they should be on the top.
    if (Parameters_.isTopNavigationEnabled()) {
        writeHTMLNavigationButtons(prev_slide, next_slide);
    }
    
    // Write the Header for the slide.
    Output_ << "<DIV class=\"header\">"
            << "<TABLE"
            << " summary=\"header\""
            << " align=\"center\""
            << " border=\"0\"";
    if (Parameters_.isHeaderBackgroundEnabled()) {
        Output_ << " width=\"" << Parameters_.header_->width_ << "\""
                << " height=\"" << Parameters_.header_->height_ << "\""
                << ">" << endl;
    } else {
        Output_ << " width=\"100%\""
                << ">" << endl;
    }

    Output_ << "<TR><TD";
    if (Parameters_.isHeaderBackgroundEnabled()) {
        Output_ << " class=\"header\"";
    }
    Output_ << ">" << endl;

    // If the slide is inside of a section, write the section-header.
    if (Section_Title_ != NULL) {
        if (strlen(Section_Title_) > 0) {
            Output_ << "<H2><A class=\"sectionheader\" href=\"";
            Output_ << (Section_Filename_ == NULL ?
                        "toc.html" : Section_Filename_) << "\">";
            Output_ << Section_Title_ << "</A></H2>" << endl;
        } else {
            // If the section doesn't have a title, we use the title
            // of the first slide as section-title.
            Section_Title_ = title_;
        }
    }

    // Write the title of the slide.
    Output_ << "<H1><A class=\"header\" href=\"toc.html\">";
    OutputFilter::writeHTML(title_, 1);
    Output_ << "</A></H1>" << endl
            << "</TD></TR>" << endl
            << "</TABLE></DIV>" << endl;

    // Write the HTML-framework for sloppy buttons, if enabled.
    // If the navigation-buttons are on the top, we can skip it.
    if (!Parameters_.isTopNavigationEnabled() &&
        Parameters_.isSloppyButtonsEnabled()) {
        Output_ << "<TABLE"
                << " summary=\"content\""
                << " width=\"100%\""
                << " height=\"65%\""
                << " border=\"0\""
                << "><TR valign=\"top\"><TD>" << endl;
    }
}


//---------------------------------------------------------------------------
void SlideNode::writeHTMLNavigationButtons(const char *prev_slide,
                                           const char *next_slide) const
{
    Output_ << "<TABLE"
            << " summary=\"navigation-buttons\""
            << " border=\"0\""
            << " width=\"100%\""
            << "><TR>" << endl;

    // This is the cell for the buttons on the left side.
    Output_ << "<TD align=\"left\" width=\"50%\">" << endl;

    // Write the first-button.
    if ((strlen(prev_slide) > 0) &&
        Parameters_.isFirstAndLastButtonEnabled()) {
        Output_ << "<A rel=\"start\" href=\""
                << Filename_Of_First_Slide_ << "\">"
                << "<IMG"
                << " src=\"" << BUTTON_FIRST << "\""
                << " alt=\"\""
                << " border=\"0\"";
        if (Parameters_.first_) {
            Output_ << " width=\"" << Parameters_.first_->width_ << "\""
                    << " height=\"" << Parameters_.first_->height_ << "\"";
        }
        Output_ << "></A>" << endl;

        Output_ << "&nbsp;&nbsp;&nbsp;" << endl;
    }

    // Write the previous-button.
    if (strlen(prev_slide) > 0) {
        Output_ << "<A rel=\"prev\" href=\"" << prev_slide << "\">"
                << "<IMG"
                << " src=\"" << BUTTON_PREV << "\""
                << " alt=\"\""
                << " border=\"0\"";
        if (Parameters_.prev_) {
            Output_ << " width=\"" << Parameters_.prev_->width_ << "\""
                    << " height=\"" << Parameters_.prev_->height_ << "\"";
        }
        Output_ << "></A>" << endl;
    }

    // This is the cell for the buttons on the right side.
    Output_ << "</TD><TD align=\"right\" width=\"50%\">" << endl;

    // Write the note-button.
    if (note_ && Parameters_.isNoteButtonEnabled()) {
        Output_ << "<A rel=\"help\" href=\"" << note_ << "\">"
                << "<IMG"
                << " src=\"" << BUTTON_NOTE << "\""
                << " alt=\"\""
                << " border=\"0\"";
        if (Parameters_.note_) {
            Output_ << " width=\"" << Parameters_.note_->width_ << "\""
                    << " height=\"" << Parameters_.note_->height_ << "\"";
        }
        Output_ << "></A>" << endl;

        Output_ << "&nbsp;&nbsp;&nbsp;" << endl;
    }

    // Write the next-button.
    if (strlen(next_slide) > 0) {
        Output_ << "<A rel=\"next\" href=\"" << next_slide << "\">"
                << "<IMG"
                << " src=\"" << BUTTON_NEXT << "\""
                << " alt=\"\""
                << " border=\"0\"";
        if (Parameters_.next_) {
            Output_ << " width=\"" << Parameters_.next_->width_ << "\""
                    << " height=\"" << Parameters_.next_->height_ << "\"";
        }
        Output_ << "></A>" << endl;
    }

    // Write the last-button.
    if ((strlen(next_slide) > 0) &&
        Parameters_.isFirstAndLastButtonEnabled()) {
        Output_ << "&nbsp;&nbsp;&nbsp;" << endl;

        Output_ << "<A href=\""
                << Filename_Of_Last_Slide_ << "\">"
                << "<IMG"
                << " src=\"" << BUTTON_LAST << "\""
                << " alt=\"\""
                << " border=\"0\"";
        if (Parameters_.last_) {
            Output_ << " width=\"" << Parameters_.last_->width_ << "\""
                    << " height=\"" << Parameters_.last_->height_ << "\"";
        }
        Output_ << "></A>" << endl;;
    }

    Output_ << "</TD>" << endl
            << "</TR></TABLE>" << endl;
}


//---------------------------------------------------------------------------
void SlideNode::writeHTMLFooter(const char *prev_slide,
                                const char *next_slide) const
{
    // Write the navigation-buttons, if they should be on the bottom.
    if (!Parameters_.isTopNavigationEnabled()) {
        if (Parameters_.isSloppyButtonsEnabled()) {
            Output_ << "</TD></TR></TABLE>" << endl;
        } else {
            Output_ << "<DIV class=\"footer\">" << endl;
        }

        writeHTMLNavigationButtons(prev_slide,next_slide);

        if (!Parameters_.isSloppyButtonsEnabled()) {
            Output_ << "</DIV>" << endl;
        }
    }

    Output_ << "</BODY>" << endl
            << "</HTML>" << endl;
}
