/***************************************************************************
                          job_synchronizewithdatabase.cpp  -  description
                             -------------------
    begin                : Wed May 9 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
As an reminder, here are the ctor parameters when called from gui_sourcer.cpp 
for  "Harddisk Operations"  PHF Jan 2003
============================================================================

1 "Synchronize (prefer Database)"
UPDATE_AND_SYNCHRONIZE, PREFER_INTERN_CHANGED
2 "Synchronize (prefer Sources)" 
UPDATE_AND_SYNCHRONIZE, PREFER_EXTERN_CHANGED
--------------------
3 "Update + Read Tags"
UPDATE_AND_READ_EXTERN_CHANGED_TRACKS, ONLY_NOT_INTERN_CHANGED
4 "Update + Read Tags (force)"
UPDATE_AND_READ_EXTERN_CHANGED_TRACKS, FORCE_REREAD
--------------------
5 "Update + Write Tags"
UPDATE_AND_WRITE_INTERN_CHANGED_TRACKS, ONLY_NOT_EXTERN_CHANGED
6 "Update + Write Tags (force)"
UPDATE_AND_WRITE_INTERN_CHANGED_TRACKS, FORCE_WRITE
--------------------
7 "Update Only"
UPDATE_ONLY, NO_PARAMETER

Following actions are taken with the 3 main files map, depending of the menu item.
INTERN CHANGED = File content ( tags)  modified using prokyon3.
   1,2,5,6 :WRITE TAGS
EXTERN CHANGED = Filename time stamp differs from the one sored in prokyon3 DB.
   1,2,3,4 READ TAGS
DOUBLE CHANGED = Files changed both internally and externally
   2,4 APPENDED TO EXTERN CHANGES
   1,6 APPENDED TO INTERN CHANGES
   3,5,7 DISCARDED
*********************************************************************/


#include "job_synchronizewithdatabase.h"

#include "job_disconnectfromdatabase.h"
#include "job_querydatabase.h"

#include "configuration.h"
#include "jobmanager.h"
#include "database.h"
#include "datadispatcher.h"

#include "gui.h"

#include <qptrlist.h>
#include <qstringlist.h>
#include <qptrqueue.h>
#include <qdir.h>

#include <iostream>

#ifdef EMBEDDED 
  #include <mysql.h>
#endif

using namespace std;

Job_SynchronizeWithDatabase::Job_SynchronizeWithDatabase(int _modus, int _parameter)
  : mediumType(MEDIUM_HARDDISK), modus(_modus), parameter(_parameter), drive(0), share(0) {}

Job_SynchronizeWithDatabase::Job_SynchronizeWithDatabase(LVI_CDDrive *_drive, int _modus, int _parameter)
  : mediumType(MEDIUM_CDROM), modus(_modus), parameter(_parameter), drive(_drive), share(0) {}

Job_SynchronizeWithDatabase::Job_SynchronizeWithDatabase(LVI_SMBShare *_share, int _modus, int _parameter)
  : mediumType(MEDIUM_SMB), modus(_modus), parameter(_parameter), drive(0), share(_share) {}

Job_SynchronizeWithDatabase::Job_SynchronizeWithDatabase(LVI_NFSExport *_Export, int _modus, int _parameter)
  : mediumType(MEDIUM_NFS), modus(_modus), parameter(_parameter), drive(0), Export(_Export) {}

int Job_SynchronizeWithDatabase::getMediumID()
{
  switch(mediumType) {
  case MEDIUM_HARDDISK: return 0;
  case MEDIUM_CDROM: return drive->getCdID();
  case MEDIUM_SMB: return share->getMediumID();
  case MEDIUM_NFS: return Export->getMediumID();
  default: return -1; // should never happen
  }
}

void Job_SynchronizeWithDatabase::run()
{
  int error;

  if (verbose==8) qWarning( "    >run() tread:%p type:%s HANDLE:%p", this, JOB_CODE[type()], currentThread() );

#ifdef EMBEDDED 
  mysql_thread_init(); 
#endif

  // save listview's selection  for later restore 
  QString artistBuf = QString::null;

  if ( QListViewItem* lvi = gui->getSelector()->getTabArtist()->selectedItem() )
    if ( dynamic_cast<QListViewItem*>( lvi->parent() ) ) 
      artistBuf = lvi->parent()->text(0); 
    else 
      artistBuf = lvi->text(0); 
 
  app->lock();
  datadispatcher->eventAddEditLock();
  switch(mediumType) {
  case MEDIUM_HARDDISK: datadispatcher->eventStartedHarddiskSynchronizing(); break;
  case MEDIUM_CDROM:    datadispatcher->eventStartedCDROMSynchronizing(drive); break;
  case MEDIUM_SMB:      datadispatcher->eventStartedSMBSynchronizing(share); break;
  case MEDIUM_NFS:      datadispatcher->eventStartedNFSSynchronizing(Export); break;
  }
  app->unlock();
  
  QPtrQueue<QDir>	directoryQueue;
  
  QStringList userDirectories;

  QList<TRACK> *storedFiles;
  QList<TRACK> *localFiles         = new QList<TRACK>;
  QList<TRACK> *externChangedFiles = new QList<TRACK>;
  QList<TRACK> *internChangedFiles = new QList<TRACK>;
  QList<TRACK> *doubleChangedFiles = new QList<TRACK>;
  
  // get directories for the medium
  switch(mediumType) {
  case MEDIUM_HARDDISK:
    config->lock();
    userDirectories = config->getUserDirectories();
    config->unlock();
    for(QStringList::Iterator it = userDirectories.begin(); it != userDirectories.end(); ++it) directoryQueue.enqueue(new QDir(*it));
    break;
  case MEDIUM_CDROM:    directoryQueue.enqueue(new QDir(drive->getPath())); break;
  case MEDIUM_SMB:      directoryQueue.enqueue(new QDir(share->getPath())); break;
  case MEDIUM_NFS:      directoryQueue.enqueue(new QDir(Export->getPath())); break;
  }

  QString previousStatusMode = gui->getStatusInfo( 5 );
  qApp->lock();
  gui->setStatusInfo( " Mode: Sync.... ", 5);
  qApp->unlock();

  // get all files from medium
  while(!directoryQueue.isEmpty()) {
    QDir *curr = directoryQueue.dequeue();
    localFiles = getFileList(curr, true, localFiles);
    delete curr;
  }


  // Remove duplicates entry if same files are found twice in path 
  QMap<QString,TRACK*> DBByPath;
  QPtrListIterator<TRACK> it( *localFiles );
  TRACK* cur;

  while ( (cur = it.current()) != 0 ) {
    ++it; 
    if ( DBByPath.contains( cur->path + cur->filename) ) {
      delete( cur ); //autoDelete false by default
      localFiles->remove( cur);  // delete current item and update iterator
    } else { 
      DBByPath[cur->path + cur->filename] = cur;
    }
  }
  
  for(TRACK *curr = localFiles->first(); curr != 0; curr = localFiles->next()) {
    switch(mediumType) {
    case MEDIUM_CDROM: curr->path.replace(0, drive->getPath().length(), ""); break;
    case MEDIUM_SMB  : curr->path.replace(0, share->getPath().length(), ""); break;
    case MEDIUM_NFS  : curr->path.replace(0, Export->getPath().length(), ""); break;
    }
  }

  // get all files from database
  database->lock();
  storedFiles = database->getTracksByMedium(getMediumID());
  error = database->getError();
  database->unlock();

  // *********************************************
  // *** SYNCHRONISATION harddisk <-> database ***
  // *********************************************
  
  // *** FIRST: unchanged filenames ***
  
  // --- look for not moved files (changed or unchanged)
  if (verbose>1) qWarning(" synchwithdatabase:--- look for not moved files (changed or unchanged)");

  if(!error) {

    QMap<QString,TRACK*> DBByAbsFile;
    for(TRACK *curr = storedFiles->first(); curr != 0; curr = storedFiles->next())
      DBByAbsFile[curr->path + curr->filename] = curr;
    
    TRACK *curr = localFiles->first();
    while(curr != 0) {
      QString key = curr->path + curr->filename;
      if(DBByAbsFile.contains(key)) {
	TRACK *dbentry = DBByAbsFile[key];
	storedFiles->remove(dbentry);
	if(curr->lastModified != dbentry->lastModified || dbentry->hasChanged) {
	  if(!dbentry->hasChanged) externChangedFiles->append(dbentry);
	  else if(curr->lastModified == dbentry->lastModified) internChangedFiles->append(dbentry);
	  else doubleChangedFiles->append(dbentry);
	} else delete dbentry;
	if(curr == localFiles->getLast()) {
	  localFiles->remove();
	  delete curr;
	  curr = 0;
	} else {
	  localFiles->remove();
	  delete curr;
	  curr = localFiles->current();
	}
      } else {
	curr = localFiles->next();
      }
    }
    
    // --- look for moved files (changed or unchanged)
    if ( verbose > 1 ) qWarning( " synchwithdatabase:--- look for moved files (changed or unchanged)");

    QMap<QString,QList<TRACK>*> DBByFile;
    
    for(TRACK *curr = storedFiles->first(); curr != 0; curr = storedFiles->next()) {
      QString key = curr->filename;
      if(!DBByFile.contains(key)) DBByFile[key] = new QList<TRACK>;
      DBByFile[key]->append(curr);
    }
    
    QMap<QString,QList<TRACK>*> LocalByFile;
    
    for(TRACK *curr = localFiles->first(); curr != 0; curr = localFiles->next()) {
      QString key = curr->filename;
      if(!LocalByFile.contains(key)) LocalByFile[key] = new QList<TRACK>;
      LocalByFile[key]->append(curr);
    }
    
    QMap<QString,QList<TRACK>*>::Iterator it = LocalByFile.begin();
    for(; it != LocalByFile.end(); ++it) {
      QString key = it.key();
      if(DBByFile.contains(key)) {
	QList<TRACK> *list   = LocalByFile[key];
	QList<TRACK> *dblist = DBByFile[key];
	TRACK *foundLocal = 0, *foundDB = 0;
	bool foundTwice = false;
	for(TRACK *curr = list->first(); curr != 0; curr = list->next()) {
	  for(TRACK *dbcurr = dblist->first(); dbcurr != 0; dbcurr = dblist->next()) {
	    if(curr->lastModified == dbcurr->lastModified) {
	      if(foundLocal == 0) {
		foundLocal = curr;
		foundDB    = dbcurr;
	      } else foundTwice = true;
	    }
	  }
	}
	if(list->count() == 1 && dblist->count() == 1) {
	  foundLocal = list->first();
	  foundDB = dblist->first();
	}
	if(!foundTwice && foundLocal != 0) {
	  list->remove(foundLocal); localFiles->remove(foundLocal);
	  dblist->remove(foundDB); storedFiles->remove(foundDB);
	  foundDB->path = foundLocal->path;
	  if(foundLocal->lastModified != foundDB->lastModified || foundDB->hasChanged) {
	    if(!foundDB->hasChanged) externChangedFiles->append(foundDB);
	    else if(foundLocal->lastModified == foundDB->lastModified) internChangedFiles->append(foundDB);
	    else doubleChangedFiles->append(foundDB);
	  }
	  else externChangedFiles->append(foundDB);
	}
	if(dblist->count() == 0) DBByFile.remove(key);
      }
    }
    
    // *** SECOND: changed filenames ***
    
    // --- look for renamed file names (by lastModified and filesize)
    if ( verbose > 1 ) qWarning(" synchwithdatabase:--- look for renamed file names (by lastModified and filesize)");

    QMap<QString,TRACK*> DBByModified;
    QMap<QString,int> droplist;
    for(TRACK *curr = storedFiles->first(); curr != 0; curr = storedFiles->next()) {
      QString key( "%1%2" );
      key.arg( curr->lastModified.toString().local8Bit().data() ).arg( curr->size );
      if(DBByModified.contains(key)) {
	droplist[key] = 0;
	DBByModified.remove(key);
      } else if(!droplist.contains(key)) DBByModified[key] = curr;
    }
    
    droplist.clear();
    
    QMap<QString,TRACK*> LocalByModified;
    for(TRACK *curr = localFiles->first(); curr != 0; curr = localFiles->next()) {
      QString key( "%1%2" );
      key.arg( curr->lastModified.toString().local8Bit().data() ).arg( curr->size );
      if(LocalByModified.contains(key)) {
	droplist[key] = 0;
	LocalByModified.remove(key);
      } else if(!droplist.contains(key)) LocalByModified[key] = curr;
    }
    
    QMap<QString,TRACK*>::Iterator it2 = LocalByModified.begin();
    for(; it2 != LocalByModified.end(); ++it2) {
      QString key = it2.key();
      if(DBByModified.contains(key)) {
	TRACK *local = it2.data();
	TRACK *stored = DBByModified[key];
	localFiles->remove(local);
	storedFiles->remove(stored);
	stored->filename = local->filename;
	stored->path = local->path;
	externChangedFiles->append(stored);
	DBByModified.remove(key);
      }
    }
    
  }

  // --- append doublechanged files to externchanged or internchanged
  if (verbose>1) qWarning(" synchwithdatabase:--- append doublechanged files to externchanged or internchanged");

  if(parameter == FORCE_REREAD || parameter == PREFER_EXTERN_CHANGED) {
    TRACK *curr;
    for(curr = doubleChangedFiles->first(); curr != 0; curr = doubleChangedFiles->next()) externChangedFiles->append(curr);
    delete doubleChangedFiles;
  } else if(parameter == FORCE_WRITE || parameter == PREFER_INTERN_CHANGED) {
    TRACK *curr;
    for(curr = doubleChangedFiles->first(); curr != 0; curr = doubleChangedFiles->next()) internChangedFiles->append(curr);
    delete doubleChangedFiles;
  } else {
    TRACK *curr;
    for(curr = doubleChangedFiles->first(); curr != 0; curr = doubleChangedFiles->next()) delete curr;
    delete doubleChangedFiles;
  }

  if ( verbose>1 ) {
    qWarning( "dumpListTracks( storedFiles )" );
    dumpListTracks( storedFiles );
    qWarning( "dumpListTracks( localFiles )" );
    dumpListTracks( localFiles );
    qWarning( "dumpListTracks( externChangedFiles )" );
    dumpListTracks( externChangedFiles );
    qWarning( "dumpListTracks( internChangedFiles )" );
    dumpListTracks( internChangedFiles );
  };

  // *** FOURTH: apply to database & GUI ***
  // --- complete new files
  if (verbose>1) qWarning(" synchwithdatabase:--- complete new files");

  if(!error) {

    for(TRACK *curr = localFiles->first(); curr != 0; curr = localFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path = drive->getPath() + curr->path; break;
      case MEDIUM_SMB  : curr->path = share->getPath() + curr->path; break;
      case MEDIUM_NFS  : curr->path = Export->getPath() + curr->path; break;
      }
    }

    localFiles = readTags(localFiles, mediumType == MEDIUM_SMB || mediumType == MEDIUM_NFS); 

    for(TRACK *curr = localFiles->first(); curr != 0; curr = localFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path.replace(0, drive->getPath().length(), ""); break;
      case MEDIUM_SMB  : curr->path.replace(0, share->getPath().length(), ""); break;
      case MEDIUM_NFS  : curr->path.replace(0, Export->getPath().length(), ""); break;
      }
    }
    
    database->lock();
    switch(mediumType) {
    case MEDIUM_HARDDISK: database->appendTracks(localFiles, MEDIUM_HARDDISK, "", "", 0, false, 0); break;
    case MEDIUM_CDROM:    database->appendTracks(localFiles, MEDIUM_CDROM, "", "", 0, false, drive->getCdID()); break;
    case MEDIUM_SMB:      database->appendTracks(localFiles, MEDIUM_SMB, "", "", 0, false, share->getMediumID()); break;
    case MEDIUM_NFS:      database->appendTracks(localFiles, MEDIUM_NFS, "", "", 0, false, Export->getMediumID()); break;
    }
    error = database->getError();
    database->unlock();
  }

  // --- extern changed files
  if (verbose>1) qWarning(" synchwithdatabase:--- extern changed files");

  if(!error && (modus == UPDATE_AND_READ_EXTERN_CHANGED_TRACKS || modus == UPDATE_AND_SYNCHRONIZE)) {

    for(TRACK *curr = externChangedFiles->first(); curr != 0; curr = externChangedFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path = drive->getPath() + curr->path; break;
      case MEDIUM_SMB  : curr->path = share->getPath() + curr->path; break;
      case MEDIUM_NFS  : curr->path = Export->getPath() + curr->path; break;
      }
    }

    externChangedFiles = readTagsOnly(externChangedFiles);
    
    for(TRACK *curr = externChangedFiles->first(); curr != 0; curr = externChangedFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path.replace(0, drive->getPath().length(), ""); break;
      case MEDIUM_SMB  : curr->path.replace(0, share->getPath().length(), ""); break;
      case MEDIUM_NFS  : curr->path.replace(0, Export->getPath().length(), ""); break;
      }
    }

    database->lock();
    database->updateTracks(externChangedFiles);
    error = database->getError();
    database->unlock();
  }

  // --- intern changed files
  if (verbose>1) qWarning(" synchwithdatabase:--- intern changed files");

  if(!error && (modus == UPDATE_AND_WRITE_INTERN_CHANGED_TRACKS || modus == UPDATE_AND_SYNCHRONIZE)) {

    for(TRACK *curr = internChangedFiles->first(); curr != 0; curr = internChangedFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path = drive->getPath() + curr->path; break;
      case MEDIUM_SMB  : curr->path = share->getPath() + curr->path; break;
      case MEDIUM_NFS  : curr->path = Export->getPath() + curr->path; break;
      }
    }

    internChangedFiles = writeTags(internChangedFiles);
    
    for(TRACK *curr = internChangedFiles->first(); curr != 0; curr = internChangedFiles->next()) {
      switch(mediumType) {
      case MEDIUM_CDROM: curr->path.replace(0, drive->getPath().length(), ""); break;
      case MEDIUM_SMB  : curr->path.replace(0, share->getPath().length(), ""); break;
      case MEDIUM_NFS  : curr->path.replace(0, Export->getPath().length(), ""); break;
      }
    }

    database->lock();
    database->updateTracks(internChangedFiles);
    error = database->getError();
    database->unlock();
  }

  // --- deleted files
  if (verbose>1) qWarning(" synchwithdatabase:--- deleted files");

  if(!error) {
    database->lock();
    database->deleteTracks(storedFiles);
    error = database->getError();
    database->unlock();
  }

  // --- apply changes to selector (GUI)
  if (verbose>1) qWarning(" synchwithdatabase:--- apply changes to selector (GUI)");

  if(!error) {
    database->lock();
    app->lock();

    // Quick fix when directory is changed in configuration.cpp
    gui->getSelector()->clear();
    datadispatcher->eventNewArtistAlbumBasis(database->getArtistAlbumBasis(), false);
    datadispatcher->eventNewFavouritesBasis(database->getFavouritesBasis());
    datadispatcher->eventNewAlbumBasis(database->getAlbumBasis());
    datadispatcher->eventNewLocalAlbumDeltaBasis(database->getLocalAlbumDeltaBasis());
    datadispatcher->eventNewMediumBasis(database->getMediumBasis());
    datadispatcher->eventNewPlaylistBasis(database->getPlaylistBasis());
    datadispatcher->eventNewPlaylistTracksBasis(database->getPlaylistTracksBasis(0),0);
    // This will reset the deltamap.
    delete database->getArtistDelta(); 

    datadispatcher->eventNewTrackListing(database->getLastQuery());
    datadispatcher->eventNewPlaylistTracksBasis(database->getPlaylistTracksBasis(0),0);
    int ID = gui->getPlaylisting()->getDisplayedPlaylistID();
    if(ID > 0) datadispatcher->eventNewPlaylistTracksBasis(database->getPlaylistTracksBasis(ID),ID);

    error = database->getError();

    if(!error) switch(mediumType) {
    case MEDIUM_HARDDISK: datadispatcher->eventStoppedHarddiskSynchronizing(); break;
    case MEDIUM_CDROM:    datadispatcher->eventStoppedCDROMSynchronizing(drive); break;
    case MEDIUM_SMB:      datadispatcher->eventStoppedSMBSynchronizing(share); break;
    case MEDIUM_NFS:      datadispatcher->eventStoppedNFSSynchronizing(Export); break;
    }
    datadispatcher->eventRemoveEditLock();

    app->unlock();
    database->unlock();
  }
  
  // *** FIFTH: clean up memory ***

  // -- error?
  if (verbose>1) qWarning(" synchwithdatabase:--- error?");

  if(error) {
    app->lock();
    switch(mediumType) {
    case MEDIUM_HARDDISK: datadispatcher->eventStoppedHarddiskSynchronizing(); break;
    case MEDIUM_CDROM:    datadispatcher->eventStoppedCDROMSynchronizing(drive); break;
    case MEDIUM_SMB:      datadispatcher->eventStoppedSMBSynchronizing(share); break;
    case MEDIUM_NFS:      datadispatcher->eventStoppedNFSSynchronizing(Export); break;
    }
    app->unlock();
    jobman->lock();
    jobman->addJob(new Job_DisconnectFromDatabase());
    jobman->unlock();
  }
  
  // --- deallocate memory
  if (verbose>1) qWarning(" synchwithdatabase:--- deallocate memory");

  TRACK *curr;
  for(curr = localFiles->first(); curr != 0; curr = localFiles->next()) delete curr;
  for(curr = externChangedFiles->first(); curr != 0; curr = externChangedFiles->next()) delete curr;
  for(curr = internChangedFiles->first(); curr != 0; curr = internChangedFiles->next()) delete curr;
  if(storedFiles != 0) for(curr = storedFiles->first(); curr != 0; curr = storedFiles->next()) delete curr;

  delete localFiles;
  delete externChangedFiles;
  delete internChangedFiles;
  if(storedFiles != 0) delete storedFiles;

  qApp->lock();

  // restore listview selection 
  LVI_Album* lvi1 = 0;
  QListView* listArtist = gui->getSelector()->getTabArtist();
  if (!artistBuf.isNull() ) { 
    lvi1 = dynamic_cast<LVI_Album*>( listArtist->findItem ( artistBuf, 0) ); 
    if (lvi1) {
      listArtist->setCurrentItem( lvi1 );
      listArtist->setSelected( lvi1, true );
    }
  }

  gui->setStatusInfo( previousStatusMode, 5 );
  qApp->unlock();

  jobman->lock();
  jobman->jobDone(this);
  jobman->unlock();
#ifdef EMBEDDED 
  mysql_thread_end(); 
#endif

  if (verbose==8) qWarning( "    >exit() thread:%p type:%s HANDLE: %p", this, JOB_CODE[type()], currentThread() );

  exit();
}

Job_SynchronizeWithDatabase::~Job_SynchronizeWithDatabase() {}
