// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "label.h"
#include "prim.h"
#include <sstream>
#include <string>
#include <ostream>
#include <boost/python/class.hpp>
#include <boost/python/return_internal_reference.hpp>
#include GL_INCLUDE

namespace visual {

Label::Label()
	: pos( &mtx, 0,0,0),
	space(0),
	xoffset(0),
	yoffset(0),
	border(5),
	font_description( ""),
	font_height(0),
	box_enabled(true),
	line_enabled(true),
	font(0),
	textWidth(0.0),
	lineColor(1,1,1),
	opacity(0.66)
{
	text.push_back( "");
}

Label::Label( const Label& other)
	: DisplayObject( other), pos( &mtx, other.pos.x, other.pos.y, other.pos.z),
	space( other.space), 
	xoffset( other.xoffset),
	yoffset( other.yoffset),
	border( other.border),
	font_description( other.font_description),
	font_height( other.font_height),
	box_enabled( other.box_enabled),
	line_enabled( other.line_enabled),
	font( 0),
	textWidth( other.textWidth),
	lineColor( other.lineColor),
	opacity( other.opacity),
	text( other.text)
{
}

Label::~Label() 
{
	if (font) 
		font->release();
}

std::string
Label::get_text()
{
	if (text[0] == std::string( ""))
		return std::string( "");
	std::ostringstream ret;
	ret << text[0];
	for ( text_iterator i = text.begin()+1; i != text.end(); i++) {
		ret << '\n' << *i;
	}
	return ret.str();
}

void 
Label::refreshCache() 
{
	// No cache for class Label.
}

void 
Label::glRender( rView& view) 
{
	if (!font) {
		font = view.cx.getFont(font_description.c_str(), font_height);
		if (!font) 
			return;
	}

	glNewList( view.createSortList(), GL_COMPILE );
	glDisable(GL_DEPTH_TEST);

	view.ext_point( pos);

	double xpix = 2.0 / view.cx.width(); 
	double ypix = 2.0 / view.cx.height();

	// alignv is the *screen space* direction pointing
	//   from pos to the origin of the text
	vector alignv( xoffset * xpix, yoffset * ypix );
 
	double wInverse = 1.0 / view.wct.w( pos);
	vector proj_pos = (view.wct * pos) * wInverse;
	vector start = proj_pos;

	if (space) {
		double cube[6][3] = { {1,0,0}, {-1,0,0}, {0,1,0},{0,-1,0}, {0,0,1},{0,0,-1} };
		double r2 = 0;
		for(int f=0;f<6;f++) {
			vector v = vector(cube[f])*space + pos;
			v = view.wct*v/view.wct.w(v) - proj_pos;
			double m = (v.x*v.x / xpix/xpix) + (v.y*v.y / ypix/ypix);
			if (m > r2) 
				r2=m;
		}

		vector spacev = alignv.norm() * std::sqrt(r2);  // in pixels
		start = start + vector( spacev.x * xpix, spacev.y * ypix);
	}

	vector end = start + alignv;

	double borderx = border * 2 / view.cx.width();
	double bordery = border * 2 / view.cx.height();
	if (textWidth == 0.0) {
		for (text_iterator i = text.begin(); i != text.end(); ++i) {
			if (font->getWidth(i->c_str()) > textWidth) {
				textWidth = font->getWidth(i->c_str());
			}
		}
	}
	
	double width = borderx + textWidth;
	double descent = font->descent() + bordery;
	double ascent = font->ascent() + bordery;
	double height = ascent + descent;         

	vector origin;
	if ( !alignv ) {
		// center align, center vertically
		origin = end + vector( -(width-borderx)/2, (descent-ascent)/2 );
	} 
	else if ( std::fabs(alignv.x) > std::fabs(alignv.y) ) {
		// left or right align, center vertically
		if (alignv.x > 0)
			origin = end + vector(borderx, (descent-ascent)/2);
		else
			origin = end + vector(-width, (descent-ascent)/2);
	} 
	else {
		// center align, top or bottom vertically
		if (alignv.y > 0)
			origin = end + vector( -(width-borderx)/2, descent);
		else
			origin = end + vector( -(width-borderx)/2, -ascent);
	}

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glShadeModel(GL_FLAT);

	if (opacity) {
		glBegin( GL_POLYGON);
		glColor4f( 0, 0, 0, opacity);
		glVertex3d( origin.x - borderx
			, origin.y - descent 
			, origin.z);
		glVertex3d( origin.x + width
			, origin.y - descent
			, origin.z);

		glVertex3d( origin.x + width
			, origin.y + ascent + ((text.size()-1) * height)
			, origin.z);
		glVertex3d( origin.x - borderx
			, origin.y + ascent + ((text.size()-1) * height)
			, origin.z);
		glEnd();
	}

	rgb line_color( lineColor);
	if (view.anaglyph) {
		if (view.coloranaglyph)
			line_color = lineColor.unsaturate();
		else {
			float gray = lineColor.grayscale();
			line_color.r = line_color.g = line_color.b = gray;
		}
	}

	if (box_enabled) {
		glBegin( GL_LINE_LOOP);
		glColor3f( line_color.r, line_color.g, line_color.b);
		glVertex3d( origin.x - borderx
			, origin.y - descent
			, origin.z);
		glVertex3d( origin.x + width
			, origin.y - descent
			, origin.z);

		glVertex3d( origin.x + width
			, origin.y + ascent + ((text.size()-1) * height)
			, origin.z);
		glVertex3d( origin.x - borderx
			, origin.y + ascent + ((text.size()-1) * height)
			, origin.z);
		glEnd();
	}

	if (line_enabled) {
		glBegin( GL_LINES);
		glColor4f( line_color.r, line_color.g, line_color.b, 1);
		glVertex3d( start.x, start.y, start.z);
		glVertex3d( end.x, end.y, end.z);
		glEnd();
	}

	glColor3f( color.r, color.g, color.b);
	for (size_t i = 0; i < text.size(); i++) {
		glRasterPos3d( origin.x
			, origin.y + ((text.size()-1)*height) - (i*height)
			, origin.z );
		font->draw(text[i].c_str());
	}

	glEnable( GL_DEPTH_TEST);
	glEndList();
}

void
Label::set_color( rgb c)
{
	write_lock l( mtx);
	this->color =  c;
}

void
Label::set_green( const float& green)
{
	write_lock L( mtx);
	this->color.g = green;
}

void
Label::set_blue( const float& blue)
{
	write_lock L(mtx);
	this->color.b = blue;
}

void
Label::set_red( const float& red)
{
	write_lock L(mtx);
	this->color.r = red;
}

void
Label::set_pos( const vector& v)
{
	pos = v;
}

void
Label::set_xoffset( const double& offset)
{
	write_lock L( mtx);
	xoffset = offset;
}

void
Label::set_yoffset( const double& offset)
{
	write_lock L( mtx);
	yoffset = offset;
}

void
Label::set_border( const double& b)
{
	write_lock L( mtx);
	border = b;
}

void
Label::set_box( bool box)
{
	write_lock L( mtx);
	box_enabled = box;
}

void
Label::set_line( bool line)
{
	write_lock L( mtx);
	line_enabled = line;
}

void
Label::set_space( const double& s)
{
	write_lock L( mtx);
	space = s;
}

void
Label::set_opacity( const double& o)
{
	write_lock L( mtx);
	opacity = o;
}

void
Label::set_height( const double& h)
{
	write_lock L( mtx);
	font_height = h;
	if (font) { 
		font->release(); 
		font = 0; 
	}
}

void
Label::set_linecolor( rgb c)
{
	write_lock L( mtx);
	lineColor = c;
}

void
Label::set_font( const std::string& s)
{
	write_lock L( mtx);
	font_description = s;
	if (font) {
		font->release();
		font = 0;
	}
}

void
Label::set_x( const double& x)
{
	pos.set_x( x);
}

void
Label::set_y( const double& y)
{
	pos.set_y( y);
}

void
Label::set_z( const double& z)
{
	pos.set_z( z);
}


// This function parses an incoming Python multi-line string.  See the member variable 
//   documentation for 'text'.
void
Label::set_text( const std::string& new_text)
{
	write_lock L( mtx);
	textWidth = 0.0;

	text.clear();
	std::istringstream buf( new_text);
	std::string line;
	while (std::getline( buf, line)) {;
		text.push_back( line);
	}
}

void
label_init_type()
{
	using namespace boost::python;
	
	// Label shares several properties with Primitive, but it is *not* derived from
	//   Primitive.
	class_< Label, bases<DisplayObject>, boost::shared_ptr<Label> >( "label")
		.def( init<const Label&>())
		.add_property( "color", &Label::get_color, &Label::set_color)
			.add_property( "red", &Label::get_red, &Label::set_red)
			.add_property( "green", &Label::get_green, &Label::set_green)
			.add_property( "blue", &Label::get_blue, &Label::set_blue)
		.def( "_get_pos", &Label::get_pos, return_internal_reference<>())
		.def( "_set_pos", &Label::set_pos)
			.add_property( "x", &Label::get_x, &Label::set_x)
			.add_property( "y", &Label::get_y, &Label::set_y)
			.add_property( "z", &Label::get_z, &Label::set_z)
		.add_property( "height", &Label::get_height, &Label::set_height)
		.add_property( "xoffset", &Label::get_xoffset, &Label::set_xoffset)
		.add_property( "yoffset", &Label::get_yoffset, &Label::set_yoffset)
		.add_property( "opacity", &Label::get_opacity, &Label::set_opacity)
		.add_property( "border", &Label::get_border, &Label::set_border)
		.add_property( "box", &Label::has_box, &Label::set_box)
		.add_property( "line", &Label::has_line, &Label::set_line)
		.add_property( "linecolor", &Label::get_linecolor, &Label::set_linecolor)
		.add_property( "font", &Label::get_font, &Label::set_font)
		.add_property( "text", &Label::get_text, &Label::set_text)
		.add_property( "space", &Label::get_space, &Label::set_space)
		// .def( self_ns::str(self))
		;
}

} // !namespace visual
