#ifndef VISUAL_DISPLAYLIST_H
#define VISUAL_DISPLAYLIST_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "cvisual.h"
#include <boost/weak_ptr.hpp>
#include <boost/python/object.hpp>
#include "vcache.h"
#include "tmatrix.h"
#include "light.h"
#include "color.h"

#include "display.h"

namespace visual {

struct rView 
{
	tmatrix wct;
	lighting lights;
	struct glContext& cx;

	tmatrix fwt;   // frame-to-world (for extent calculations)
	double min_extent[3];
	double max_extent[3];
	bool anaglyph;
	bool coloranaglyph;

	std::vector<int> sortlists;

	rView( const tmatrix& _wct, const lighting& _lights,
	       const tmatrix& _fwt, struct glContext& _cx,
	       const vector& mine, const vector& maxe,
		   const bool _anaglyph, const bool _coloranaglyph);
	~rView();

	void ext_set(const vector& mine, const vector& maxe);
	void ext_brect(const tmatrix& mwt, vector bmin, vector bmax);
	void ext_brect(const tmatrix& mft, const double *b);
	void ext_point(vector v);
	void ext_sphere(vector v, double size);
	void ext_circle(vector p, vector n, double r);

	void absorb_local( rView& local );

	int createSortList();
	// returns an OpenGL display list number that can be used to store
	//   rendering commands to be rendered after all opaque objects have
	//   been drawn.  At some point there may be depth sorting support as
	//   well.
	// Usage:
	//   glNewList( view.createSortList(), GL_COMPILE);
	//   ...
	//   glEndList();
};

class frame;

// This class implements the minimum subset of information required by all
// drawable objects in Visual.
class DisplayObject : public Cache
{
 public:
	DisplayObject();
	DisplayObject( const DisplayObject& other);
	virtual ~DisplayObject();

	// Called by gldevice to render the object.
	virtual void glRender(rView&);
	
	// Used to determine if a vector coming from the screen position camera
	// in the direction of ray intersects this object, and returns the distance
	// between camera and the point of intersection.
	virtual double rayIntersect( const vector &camera, const vector &ray);


	// Used by frame to provide an extra coordinate transformation for the frame's
	// children.  It must return an orthogonal matrix.
	virtual tmatrix getChildTransform();

	// Return the frame which owns this object, possibly NULL.
	boost::shared_ptr<frame> getParent();
	
	// Return, to Python, the frame which owns this object, possibly PyNONE.
	boost::python::object py_get_parent();
	
	// Make this object a child of a particular frame.
	void setParent( boost::shared_ptr<frame> m_frame);
	
	// Returns the PyObject* corrisponding to this particular object.  This is
	// used to return a pointer to an already-wrapped object while retaining its
	// identity.  visual::Display uses this to return a heterogeneous list of 
	// objects without having to perform explicit downcasts (it only sees 
	// DisplayObjects)
	boost::python::object getObject() { return self; }

	// Initialize or reset this DisplayObject to a particular OpenGL display.
	void set_display( boost::shared_ptr<Display> d);
	
	// Get the display that this object is being rendered into.
	boost::shared_ptr<Display> get_display() const;
	boost::python::object py_get_display() const;

	// This class performs final initialization of the DisplayObject that cannot
	// be performed within the constructor.
	void
	py_complete_init( 
		boost::shared_ptr<DisplayObject> cpp_self, 
		boost::python::object py_self,
		bool _visible,
		boost::shared_ptr<Display> _display,
		boost::shared_ptr<frame> _parent);
	
	// Export a lock/unlock interface to Python for new rendering primitives.
	// Not used at this time.
	inline void py_read_lock() { mtx.sync_lock(); }
	inline void py_write_lock() { mtx.count_lock(); }
	inline void py_unlock() { mtx.sync_unlock(); }
	
	// Get/set the visibility property of this DisplayObject.
	bool get_visible() const;
	void set_visible( bool vis);

 protected:
	bool visible;
	rgb color;
	boost::shared_ptr<Display> display;
	boost::shared_ptr<frame> parent;

	// The PyObject* for this object.
	boost::python::object self;
	// operate on the linked list held by display.
	boost::weak_ptr<DisplayObject> weak_this;

 private:
	// Encapsulates the code required to insert and remove one of these buggars
	// from the displaylist, including working aroud some complex locking symantics.
	void insert();
	void remove();

	friend class Display;
	friend class GLDevice;
};

} // !namespace visual


#endif // !VISUAL_DISPLAYLIST_H
