// xgl2: Windowing environment-specific code for Gtkmm2 and GtkGLextmm.

#ifndef VISUAL_XGL2_H
#define VISUAL_XGL2_H

#include "glcontext.h"
#include "vector.h"
#include <gtkmm/window.h>
#include <gtkmm/gl/drawingarea.h>
#include <gdkmm/gl/font.h>
#include <queue>
#include <boost/scoped_ptr.hpp>

namespace visual {

#if !USE_NEWFONT
struct xgl2Font : glFont 
{
public:
	//from glFont
	virtual double getWidth(const char *text);
	virtual double ascent();
	virtual double descent();
	virtual void draw( const char *text);
	virtual void release();
  
	//used by xglContext
	xgl2Font( struct glContext& cx, const char *name, double size);
	virtual ~xgl2Font();
	void addref() { refcount++; }
  
private:
	struct glContext& cx;
	int listBase;
	Glib::RefPtr<Pango::Font> font;
	int refcount;
	// Cache metrics for the font.
	Pango::FontMetrics metrics;
};
#else
struct xgl2Font : glFont
{
 private:
	Glib::RefPtr<Pango::Context> ft2_context;
	boost::shared_ptr<Gtk::Widget> parent_widget;
	std::size_t font_size;
	
 public:
	struct layout : glFont::layout
	{
	 private:
	 	// The width and height of the text run in pixels
		std::size_t pixel_extent_width;
		std::size_t pixel_extent_height;
		// The texture id for this object
		GLuint texture_id;
		// The width and height of the texture.  These will be equal to the
		// extents, rounded up to the next power of 2.
		std::size_t tex_width;
		std::size_t tex_height;
		layout();
	 
	 public:
	 	// In practice, only xgl2Font can ever call this constructor
	 	layout( std::string str, std::size_t font_size, Gtk::Widget* widget, 
			Glib::RefPtr<Pango::Context> ft2_context);
	 
		virtual ~layout();
		
		// Render the run or paragraph of text, such that in GL coordinates, the
		// center of the bounding box is at <0, 0, 0>, and the text is rendered
		// in the x-y plane.  The texture state is not preserved.  The text is
		// rendered using the current fragment color.
		virtual void gl_render() const;
		
		// Returns the size of the bounding box in the x-y plane for the layed
		// out run of text.  The z-componant of the return is always zero
		virtual vector get_extent() const;
		
	};
	
	xgl2Font( boost::shared_ptr<Gtk::Widget> parent_widget, 
		Glib::RefPtr<Pango::Context> ft2_context, std::size_t font_size);
	virtual ~xgl2Font();
	// Compute a layout for a run of text, including any newlines.  The text is
	// always left-justified.  An OpenGL rendering context must be active when
	// invoking this function.  The texture state is not preserved.
	virtual boost::shared_ptr<glFont::layout> lay_out( const std::string& str);
};

#endif

class xgl2Context : public glContext,
	                public SigC::Object
{
public:
	virtual void lockMouse();
	virtual void unlockMouse();
	virtual void showMouse();
	virtual void hideMouse();
	virtual int getMouseButtons();
	virtual int getMouseButtonsChanged();
	
	// Returns change in position in pixels.
	virtual vector getMouseDelta();
	
	// Returns the mouse position as a fraction of the window width.
	virtual vector getMousePos();
	
	// Actually only returns a single key code.  NOT compatable with non-latin languages.
	virtual std::string getKeys();
	
	// These functions only have meaning for mouse events
	// return 1 if present, 0 if not.
	virtual int getShiftKey();
	virtual int getAltKey();
	virtual int getCtrlKey();

	xgl2Context();
	virtual ~xgl2Context();

	virtual bool initWindow( const char* title, int x, int y, int width, int height, int flags);
	virtual bool changeWindow( const char* title, int x, int y, int width, int height, int flags);
	virtual bool isOpen();
	virtual void cleanup();
  
	virtual void makeCurrent();
	virtual void makeNotCurrent();
	virtual void swapBuffers();
	
	virtual vector origin();
	virtual vector corner();
	virtual int width();
	virtual int height();

	virtual std::string lastError();
    
    #if !USE_NEWFONT
    // Make sure to call ->release() on the returned pointer.
	virtual glFont* getFont(const char* description, double size);
	#else
	virtual boost::shared_ptr<glFont> getFont( double size);
	#endif

private:
#if USE_NEWFONT
	Glib::RefPtr<Pango::Context> ft2_context;
#endif
	// The scoped pointers allevieate the need to use cleanup().
	boost::scoped_ptr<Gtk::Window> m_window;
	boost::shared_ptr<Gtk::GL::DrawingArea> m_area;
	
	// Current state of the window's width, in pixels.
	// See documentation for Gtk::Window::get_size().
	std::string error_message;
	
	// Maintain the state of any extended keys: alt, ctrl, shift.
	// SHIFT_PRESSED = 1 << 3
	// CTRL_PRESSED =  1 << 4
	// ALT_PRESSED =   1 << 5
	// Any other flag is invalid.
	int extKeyState;
  
	// Maintains the state of the currently pressed buttons:
	// LEFT_CLICK =   1 << 0
	// RIGHT_CLICK =  1 << 1
	// MIDDLE_CLICK = 1 << 2
	int buttonState;
	int buttonsChanged;
	
	// Maintain the state of the mouse for Python's benefit.
	// These values store the mouse postion in pixels relative to the upper-right corner
	// of the window.
	vector mousePos;
	vector oldMousePos;
	
	// A historical list of the keys the user has pressed.
	// This will fail for non-latin character sets.
	std::queue<std::string> key_queue;

	// Callback functions for X events.  These override the base class virtual functions,
	// and then pass the signal up to the parent signal handler.
	bool on_configure_event( GdkEventConfigure* event);
 	bool on_mouse_motion_event( GdkEventMotion* event);
	bool on_delete_event( GdkEventAny* event);
	bool on_button_press_event( GdkEventButton* event);
	bool on_button_release_event( GdkEventButton* event);
	bool on_key_press_event( GdkEventKey* key); 
};

} // !namespace visual


#endif // !XGL2_H
