// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "box.h"
#include GL_INCLUDE

namespace visual {

vector
box::getScale()
{
	return vector(axis.mag(),height,width);
}

double
box::rayIntersect( const vector& camera, const vector& ray)
{
	if (degenerate)
		return 0.0;
	vector delta(camera.x - mwt[0][3], camera.y - mwt[1][3], camera.z - mwt[2][3]);
	vector vR0 = wlt.times_v(delta);
	vector vRd = wlt.times_v(ray).norm();

	double R0[] = {vR0.x, vR0.y, vR0.z};
	double Rd[] = {vRd.x, vRd.y, vRd.z};
	double sc[] = {scale.x*0.5,scale.y*0.5,scale.z*0.5};
	double tnear = -1e300, tfar = 1e300;

	for(int a=0;a<3;a++) {
		if (Rd[a] == 0) {
			if (R0[a] < -sc[a] || R0[a] > sc[a])
				return 0.0;
		}
		else {
			double xi = 1.0 / Rd[a];
			double t1 = (-sc[a] - R0[a]) * xi;
			double t2 = (sc[a] - R0[a]) * xi;
			if (t1>t2) {
				if (t2 > tnear)
					tnear = t2;
				if (t1 < tfar)
					tfar = t1;
			}
			else {
				if (t1 > tnear)
					tnear = t1;
				if (t2 < tfar)
					tfar = t2;
			}
			if (tnear > tfar)
				return 0.0;
			if (tfar < 0)
				return 0.0;
		}
	}
	if (tnear<0)
		return tfar;
	else
		return tnear;
}

void
box::glRender( rView& view)
{
	if (degenerate)
		return;

	static double brect[] = { 0.5,  0.5,  0.5,
	                         -0.5, -0.5, -0.5};
	view.ext_brect(mwt, brect);

	tmatrix mct(mwt,view.wct);

	// Calculate the lighting on each of the faces of the cube.  Since
	//   the normals lie along the axes, all of the dot products 
	//   conveniently degenerate.
	float L[6];
	for(int i=0;i<6;i++) 
		L[i] = view.lights.ambient;
	for(int l=0;l<view.lights.n_lights;l++) {
		vector lt = wlt * view.lights.L[l];
		if (lt.x > 0)
			L[3]+=lt.x;
		else
			L[2]-=lt.x;
		if (lt.y > 0)
			L[4]+=lt.y;
		else
			L[1]-=lt.y;
		if (lt.z > 0)
			L[5]+=lt.z;
		else
			L[0]-=lt.z;
	}

	// The following static arrays contain the vertex and
	// face data for a cube.  For symmetry with sphere and
	// cylinder, this could be wrapped in a box_model
	// object.
	static vertex projected[8];
	static int s[6][4] = { {1,3,0,2},
	                       {0,4,1,5},
	                       {0,2,4,6},
	                       {5,7,1,3},
	                       {6,2,7,3},
	                       {4,6,5,7} };
	static float v[8][3] = { {-0.5,-0.5,-0.5},
	                         { 0.5,-0.5,-0.5},
	                         {-0.5, 0.5,-0.5},
	                         { 0.5, 0.5,-0.5},
	                         {-0.5,-0.5, 0.5},
	                         { 0.5,-0.5, 0.5},
	                         {-0.5, 0.5, 0.5},
	                         { 0.5, 0.5, 0.5} };

	// Projection and rendering
	for(int corner=0; corner<8; corner++)
		mct.project(v[corner], projected[corner]);

	glEnableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glVertexPointer(4, GL_DOUBLE, sizeof(vertex), &projected[0].x);
	glShadeModel(GL_FLAT);

	for(int side=0; side<6; side++) {
		float c = L[side];
		glColor3f(c*color.r,c*color.g,c*color.b);
		glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, s[side]);
	}
}


} // !namespace visual
