// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "ellipsoid.h"
#include "sphmodel.h"
#include GL_INCLUDE
#include "exceptions.h"

namespace visual {

void
ellipsoid::set_width( double w)
{
	write_lock L(mtx);
	width = w;
}

void
ellipsoid::set_height( double h)
{
	write_lock L(mtx);
	height = h;
}

void
ellipsoid::set_length( double length)
{
	if (!axis) 
		throw python::ZeroDivisionError("Degenerate primitive");
	axis = axis * (length / axis.mag());
}

double
ellipsoid::get_length()
{
	return axis.mag();
}
	
vector 
ellipsoid::get_size()
{
	// Correct for the fact that we're using a sphere model,
	// but box-like attributes (so length of cigar is ellipsoid.length,
	// but we use sphere rendering, where radius is half the length)
	vector rscale = getScale();
	return 2.0 * rscale;
}
	
void 
ellipsoid::set_size( vector v)
{
	axis = axis.norm() * v.x;
	write_lock L(mtx);
	height = v.y;
	width = v.z;
}
	
vector 
ellipsoid::getScale()
{
	// Correct for the fact that we're using a sphere model,
	// but box-like attributes (so length of cigar is ellipsoid.length
	// but we use sphere rendering, where radius is half the length)
	return vector( 0.5*axis.mag(), 0.5*height, 0.5*width);
}

void 
ellipsoid::glRender( rView& view)
{
	if (degenerate) 
		return;

	double s_size = scale.x;
	if (scale.y > s_size)
		s_size = scale.y;
	if (scale.z > s_size)
		s_size = scale.z;

	view.ext_sphere( mwt * vector(0,0,0), 0.5*s_size );

	lighting lt(view.lights, wlt);
	tmatrix mct(mwt, view.wct);

	// Level-of-detail heuristic.
	// TODO: Figure out how this should actually work!
	vector a = mct*vector(0,0,0) / mct.w(vector(0,0,0));
	vector b = mct*vector(0.5,0,0) / mct.w(vector(0.5,0,0));
	vector c = mct*vector(0,0.5,0) / mct.w(vector(0,0.5,0));
	vector d = mct*vector(0,0,0.5) / mct.w(vector(0,0,0.5));
	float size = std::sqrt((a-b).mag2() + (a-c).mag2() + (a-d).mag2());
	int n;
	if (size < 0.02) 
		n = 0;
	else if (size < 0.125) 
		n = 1;
	else n = 2;

	sph_model& model = sph_model::get(n);

	float red = color.r;
	float green = color.g;
	float blue = color.b;

	// Projection and lighting loop
	vertex *pr = model.proj;
	float *col = model.color;
	float *mv = model.verts;

	for(int v=0; v<model.nverts; v++) {
		double illum = lt.illuminate(mv[0], mv[1], mv[2]);

		col[0] = red *illum;
		col[1] = green *illum;
		col[2] = blue *illum;
		col[3] = 0.5;
        
        mct.project(mv, *pr);
        ++pr;
		mv += 3;
		col += 4;
	}
  
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glVertexPointer(4, GL_DOUBLE, sizeof(vertex), &model.proj[0].x);
	glColorPointer(4, GL_FLOAT, 4*sizeof(float), model.color);
	glShadeModel(GL_SMOOTH);

	glDrawElements(GL_TRIANGLES, model.ni, GL_UNSIGNED_INT, model.indices);
}
	
double
ellipsoid::rayIntersect( const vector& camera, const vector& ray)
{
	if (degenerate) 
		return 0.0;
	vector delta(camera.x - mwt[0][3], camera.y - mwt[1][3], camera.z - mwt[2][3]);
	vector vR0 = wlt.times_v(delta);
	vector vRd = wlt.times_v(ray).norm();

	double R0[] = { vR0.x, vR0.y, vR0.z };
	double Rd[] = { vRd.x, vRd.y, vRd.z };
	double sc[] = { scale.x, scale.y, scale.z };
	double tnear = -1e300;
	double tfar = 1e300;

	for(int a=0;a<3;a++) {
		if (Rd[a] == 0) {
			if (R0[a] < -sc[a] || R0[a] > sc[a]) 
				return 0.0;
		} 
		else {
			double xi = 1.0 / Rd[a];
			double t1 = (-sc[a] - R0[a]) * xi;
			double t2 = (sc[a] - R0[a]) * xi;
			if (t1>t2) {
				if (t2 > tnear) 
					tnear = t2;
				if (t1 < tfar)  
					tfar = t1;
			} 
			else {
				if (t1 > tnear) 
					tnear = t1;
				if (t2 < tfar) 
					tfar = t2;
			}
			if (tnear > tfar) 
				return 0.0;
			if (tfar < 0) 
				return 0.0;
		}
	}
	if (tnear<0) 
		return tfar; 
	else 
		return tnear;
}
	
} // !namespace visual
