#include "syncprocessor.h"
#include "qsstvglobal.h"
#ifndef QT_NO_DEBUG
#include "scope/scopeview.h"
#endif
#include "dsp/filterparam.h"
#include "configparams.h"
#include "dispatcher.h"
#include <QApplication>
#include <QDebug>

/*!
  \class syncProcessor Filters the incoming signal using a narrow bandpassfilter centered around 1200 Hz

  The basspand filter is wide enough to detect the 1100Hz-1300Hz used during retarce, carrying the VisCode.
  Event generated: verticalRetraceEvent
  <br> isInSync: test for sync status \sa getSyncPosition()


*/


ssenitivity sensitivityArray[3]= /**< TODO */
{
  {0.70,  0.60, 500, 5},
  {0.65,  0.50, 200, 7},
  {0.60,  0.45, 100, 10}

};

/**
 * @brief
 *
 * @param parent
 */
syncProcessor::syncProcessor(QObject *parent) :  QObject(parent)
{
  syncFilter=NULL;
}

/**
 * @brief
 *
 */
void syncProcessor::init()
{
  freqPtr=syncFilter->filteredDataPtr();
  syncVolumePtr=syncFilter->volumePtr();
  rxVolumePtr=rxFilter->volumePtr();
  sampleCounter=0;
  syncArrayIndex=0;
  shadowSyncArrayIndex=0;
  syncState=SYNCOFF;
  syncFound=false;
  slantAdjustLine=6;
  signalQuality=0;
  modifiedClock=rxClock/SUBSAMPLINGRATIO;
  newClock=false;
  idxStart=M1;
  idxEnd=FAX480;
  volumeOffCounter=0;
  syncDeviation=SYNCDEVIATION;
  retraceDetected=false;
  if(sstvModeIndex!=0)
    {
      idxEnd=idxStart=(esstvMode)(sstvModeIndex-1);
    }
#ifndef QT_NO_DEBUG
  scopeViewerSync->setCurveName("SYNC VOL",SCDATA1);
  scopeViewerSync->setCurveName("Sync State",SCDATA2);
  scopeViewerSync->setCurveName("RX VOL",SCDATA3);
  scopeViewerSync->setCurveName("SYNC Freq",SCDATA4);
  scopeViewerSync->setAxisTitles("Samples","int","State");
#endif
  displaySyncEvent* ce;
  ce = new displaySyncEvent(0,0);
  QApplication::postEvent(dispatcherPtr, ce);

}

#ifndef QT_NO_DEBUG
void syncProcessor::setOffset(unsigned int dataScopeOffset)
{
  xOffset=dataScopeOffset;
  scopeViewerSync->setOffset(xOffset);
}
#endif

/**
 * @brief
 *
 */
void syncProcessor::process()
{
  if(!syncFound) syncQuality=0;

//  addToLog(QString("sync samplecounter:%1").arg(sampleCounter),LOGSYNC1);

  extractSync();
  #ifndef QT_NO_DEBUG
  scopeViewerSync->addData1(syncVolumePtr,sampleCounter,RXSTRIPE);
  scopeViewerSync->addData2(syncStateBuffer,sampleCounter,RXSTRIPE);
  scopeViewerSync->addData3(rxVolumePtr,sampleCounter,RXSTRIPE);
  scopeViewerSync->addData4(freqPtr,sampleCounter,RXSTRIPE);
#endif
  sampleCounter+=RXSTRIPE;
}

/**
 * @brief
 *
 */
void syncProcessor::extractSync()
{
  int i;
  int syncDurationCount=0;

  DSPFLOAT syncAvgFreq=0;

  for(i=0;i<RXSTRIPE;i++)
    {
      if(rxVolumePtr[i]<sensitivityArray[squelch-1].minVolume)
        {
          addToLog(QString("volume off %1").arg(rxVolumePtr[i]),LOGSYNC1);
          syncState=SYNCOFF;
          volumeOffCounter++;
        }
      else
        {
          if(volumeOffCounter>0) volumeOffCounter--;
          if(syncState==SYNCOFF)
            {
              if(syncVolumePtr[i]>(sensitivityArray[squelch-1].switchOn*rxVolumePtr[i]))
                {
                   addToLog(QString("volume switch on %1").arg(rxVolumePtr[i]),LOGSYNC1);
                  syncState=SYNCON;
                  visCode=0;
                  visCounter=0;
                  visAvg=0.;
                  syncDurationCount=0;
                  syncAvgFreq=1200;
                  syncArray[syncArrayIndex].start=sampleCounter+i;
                }

            }
          else
            {
              if(syncVolumePtr[i]<(sensitivityArray[squelch-1].switchOff*rxVolumePtr[i]))
                {
                  addToLog(QString("volume switch off %1").arg(rxVolumePtr[i]),LOGSYNC1);
                  syncState=SYNCOFF;
                  syncArray[syncArrayIndex].end=sampleCounter+i;
                  syncArray[syncArrayIndex].width=syncArray[syncArrayIndex].end-syncArray[syncArrayIndex].start;
                  syncArray[syncArrayIndex].freq=syncAvgFreq;

                  if(syncArray[syncArrayIndex].width>(MINSYNCWIDTH*SAMPLERATE))
                    {
                      if(syncArray[syncArrayIndex].width>(RETRACEWIDTH*SAMPLERATE))
                        {
                          syncArray[syncArrayIndex].retrace=true;
                          modeDetect(true);
                         }
                      else
                        {
                          visCode=0;
                          syncArray[syncArrayIndex].retrace=false;
                          modeDetect(false);

                        }

                      incrementSyncArray();
                    }
                }
              else // we are still detecting a sync
                {
                  visCounter++;
                  syncDurationCount++;
                  if((syncDurationCount>10) && (syncDurationCount<50))
                    {
                      syncAvgFreq=syncAvgFreq+0.1*(freqPtr[i]-syncAvgFreq);
                    }
                  if(visCounter>100)
                    {
                      visAvg+=freqPtr[i];
                    }
                  if(visCounter==200)
                    {
                      visAvg/=100.;
                      visCode=visCode>>1;
                      if(visAvg<=1200.) visCode=visCode | 0X100;
                      addToLog(QString("visfreq %1, counter %2, code=%3").arg(visAvg).arg(visCounter).arg(visCode),LOGSYNC2);
                     }
                  if(visCounter > (unsigned int)(0.030*SAMPLERATE)) //reset visCounter per bit
                    {
                      visCounter=0;
                      visAvg=0.;
                    }
                }
            }
        }
      syncStateBuffer[i]=(unsigned char)syncState;
    }
  if(syncFound)
    {
      if(volumeOffCounter>(samplesPerLine*sensitivityArray[squelch-1].syncLost))
        {
          restart();
        }
    }
}



/**
 * @brief
 *
 */
void syncProcessor::modeDetect(bool atRetrace)
{
//  bool done;
  shadowSyncArray[shadowSyncArrayIndex]=syncArray[syncArrayIndex];
  dumpSyncArray(false,syncArrayIndex);
  if(!atRetrace) modeDetectByLine();
  else
    {
      if(syncFound==true)
        {
          //  we have a retrace during image capturing
//          verticalRetraceEvent* ce;
//          ce = new 	verticalRetraceEvent();
//          ce->waitFor(&done);
//          QApplication::postEvent(dispatcherPtr, ce);
//          addToLog("Vertical retrace while imagecapturing",LOGSYNC1);
          retraceDetected=true;
          return;
        }
      moveToTop(syncArrayIndex);
      if((mode=lookupVIS(visCode))!=NOTVALID)
        {
          syncFound=true;
          syncQuality=4;
          syncPosition=syncArray[0].end; /// this always indicates a valid retrace
          syncArrayIndex=0;
          addToLog(QString("modeDetect: after retrace syncpos:=%1").arg(syncPosition),LOGSYNC1);
          samplesPerLine=lineLength(mode,modifiedClock);
          syncWd=syncWidth(mode,modifiedClock);
          addToLog(QString("syncProcessor:modeDetect Viscode used: %1,mode=%2").arg(QString::number(visCode,16)).arg(mode),LOGSYNC1);
//          logfile->addToAux(QString("Index\tStart\tEnd\tClosestLine\tRetrace\t%1").arg(samplesPerLine));
          return;
        }
      else if(visCode!=0)
        {
          syncFound=false;
          addToLog(QString("syncProcessor:modeDetect Viscode rejected: %1").arg(QString::number(visCode,16)),LOGSYNC1);
        }
    }
}


/**
 * @brief use sync interval to detect mode
 *
 * The syncArrayIndex is pointing to the last valid entry in the array (i.e. it is post incremented)
 * We need a minimal number of syncs to determine the mode.
 *
 */

void syncProcessor::modeDetectByLine()
{
  int i;
  esstvMode selectedMode;
  unsigned int highest=0;
  selectedMode=NOTVALID;
  if(syncArrayIndex<3) return; // we need enough sync entries
  for(i=idxStart;i<idxEnd;i++)
   {
     modeArray[i].clear();
     getMatchingSyncPulse((esstvMode) i);
     if((modeArray[i].match>highest) && (modeArray[i].match>=3))
       {
         if(modeArray[i].consecutiveSyncs<3) continue;
         highest=modeArray[i].match;
         selectedMode=(esstvMode)i;
       }
   }
  if(syncFound==true)
    {
      trackSyncs(selectedMode);
      return;
    }
  if(selectedMode==NOTVALID) return;
  if(modeArray[selectedMode].consecutiveSyncs<3) return;
  // we have a good match

  //dumpSyncArray();
  mode=selectedMode;
  samplesPerLine=lineLength(mode,modifiedClock);
  syncWd=syncWidth(mode,modifiedClock);
  //find first occurence
  moveToTop(modeArray[(int)selectedMode].firstSyncIndex);
  syncPosition=syncArray[0].end;
//  logfile->addToAux(QString("Index\tStart\tEnd\tClosestLine\tRetrace\t%1").arg(samplesPerLine));
  cleanupSyncArray(selectedMode);

  syncFound=true; // must be after cleaunupSyncArray
  syncQuality=4;;
  addToLog(QString("syncFound at: %1 mode=%2").arg(syncPosition).arg(selectedMode),LOGSYNC1);
}


void syncProcessor::getMatchingSyncPulse(esstvMode idx)
{
  int i;
  bool adjecent=true;
  DSPFLOAT ratio,length,syncW;
  unsigned int offset;
  unsigned int pos;
  int start,result;
  ssyncArray *ptr;
  int syncIndex;
  if(syncFound)
    {
      ptr=shadowSyncArray;
      syncIndex=shadowSyncArrayIndex;
    }
  else
    {
      ptr=syncArray;
      syncIndex=syncArrayIndex;
    }
  length=lineLength(idx,modifiedClock);
  syncW=syncWidth(idx,modifiedClock);
  offset=ptr[syncIndex].end- (int)rint(syncW/2);
  start=syncIndex-1;
  if(offset<=length) return;
  pos=offset-length;
  modeArray[idx].clear();
  ratio=1000;
  for(i=0;i<=syncIndex;i++)
    {
      ptr[i].inUse=false;
    }
  while(1)
    {
      if((result=matchingSync(start,pos,syncW,syncFound))>=0)
        {
          pos=ptr[result].end-(int)rint(syncW/2);
          start=result-1;
          if(ptr[result].spurious<2)
            {
              modeArray[idx].match++;
              modeArray[idx].firstSyncIndex=result;
              ptr[result].inUse=1;
              if(adjecent==true) modeArray[idx].consecutiveSyncs++;
            }
         }
      else
        {
          adjecent=false;
        }
      modeArray[idx].lineNumber++;
      if(pos<length) break;
      if(pos<ptr[0].start)break;
      pos-=length;
    }

  if( modeArray[idx].consecutiveSyncs>=3)
    {
      ratio=((double)(ptr[syncIndex].end-ptr[modeArray[idx].firstSyncIndex].end))/(length*(double) modeArray[idx].match);
      ratio*=(double) modeArray[idx].match/(double) modeArray[idx].consecutiveSyncs;
    }
  modeArray[idx].ratio=ratio;
}



/**
 * @brief check if we find a syncpulse at position pos
 *
 * @param start index of last position in syncArray
 * @param pos  samplecounter value of the sync
 * @param syncWidth
 * @return int  index of syncArray where we found the sync at position pos, -1 if not found
 */
int syncProcessor::matchingSync(int start, unsigned int pos, DSPFLOAT syncWidth, bool shadow)
{
  int i;
  int spurious=0;
  DSPFLOAT sw;
  ssyncArray *ptr;
  if(shadow) ptr=shadowSyncArray;
  else ptr=syncArray;
  for(i=start;i>=0;i--)
    {
      if(i>=SYNCARRAYLENGTH)
        {
          qDebug() << "index error";
        }
      sw=(DSPFLOAT)(ptr[i].end-ptr[i].start);
      if(ptr[i].retrace==true)
        {
          if((pos>ptr[i].end-MAXSYNCWIDTH*modifiedClock) && (pos<ptr[i].end)) return i;
        }
      else if((syncWidth>(1.2*sw))||(syncWidth<(0.8*sw)))
        {
          continue;
        }
      else if((pos>ptr[i].start) && (pos<ptr[i].end))
        {
          ptr[i].spurious=spurious;
          return i;
        }
      if(pos>ptr[i].end) return -1;
      spurious++;
    }
  return -1;
}

/*!
returns 0 if not in Sync
returns 1 if in Sync without a retrace
returns 2 if in Sync and retrace -- the syncPosition is at the end of the retrace
*/
int syncProcessor::isInSync()
{
  int result=0;
  if(syncFound)
    {
      result++;
      if(syncArray[0].retrace==true)
        {
          result++;
        }
    }

  return result;
}


/**
 * @brief
 *
 */
void syncProcessor::incrementSyncArray()
{
  if (syncArrayIndex==(SYNCARRAYLENGTH-1))
    {
      memmove(syncArray,&syncArray[1],sizeof(ssyncArray)*(SYNCARRAYLENGTH-1));
    }
  else syncArrayIndex++;
  //copy syncArray to shadowSyncArray
  {
    if (shadowSyncArrayIndex==(SYNCARRAYLENGTH-1))
      {
        memmove(shadowSyncArray,&shadowSyncArray[1],sizeof(ssyncArray)*(SYNCARRAYLENGTH-1));
      }
    else shadowSyncArrayIndex++;
  }
}

/*!  shift up the syncArray so that syncArray[index] is at the top.
*/
void syncProcessor::moveToTop(int index)
{
  if(index==0) return;
  memmove(syncArray,syncArray+index,sizeof(ssyncArray)*(syncArrayIndex+1-index));
  syncArrayIndex-=index;
  shadowSyncArrayIndex=0; // reset shadowIndex
}


void syncProcessor::cleanupSyncArray(esstvMode modeIndex)
{
  unsigned int i,stpos;
  getMatchingSyncPulse(modeIndex);
  for(i=0;i<syncArrayIndex;) // keep last one
  {
    if(!syncArray[i].inUse)
    {
      deleteSyncEntry(i);
    }
    else
    {
      syncArray[i].lineNumber=(int) round(((double)(syncArray[i].end-syncArray[0].end))/samplesPerLine);
      i++;
    }
   }
  syncArray[syncArrayIndex].lineNumber=(int) round(((double)(syncArray[syncArrayIndex].end-syncArray[0].end))/samplesPerLine);
  for(i=1;i<=syncArrayIndex;i++)
    {
      if(syncArray[0].retrace) stpos=syncArray[i-1].end-syncWidth(modeIndex,modifiedClock)/2;
      else stpos=syncArray[i-1].start +(syncArray[i-1].end-syncArray[i-1].start)/2;
      syncArray[i].length=(syncArray[i].start +(syncArray[i].end-syncArray[i].start)/2)-stpos;
    }
  dumpSyncArray(false,0);
}



void syncProcessor::deleteSyncEntry(int index)
{
  memmove(syncArray+index,syncArray+index+1,sizeof(ssyncArray)*(syncArrayIndex-index));
  syncArrayIndex--;
//  if(syncQuality>0) syncQuality--;
}

/*!
 The modeDetect tries to find the sstv mode based on the line length. If there are a number (based on the squelch) of consecutive correct lines the syncFound and the mode are set. The syncArray is the searched for the first valid line length and the pointer is set at the end of the  sync pulse.
*/



void syncProcessor::dumpSyncArray(bool withShadow, int start)
{
  int st=start;
  if(start==0)
    {
      addToLog(QString("index=%1 start=%2 end=%3 width=%4 linelength=n/a  frequency=%5 volume=%6")
               .arg(0).arg(syncArray[0].start).arg(syncArray[0].end)
               .arg(syncArray[0].end-syncArray[0].start).arg(syncArray[0].freq).arg(rxVolumePtr[0]),LOGSYNC2);
      st=1;
    }

  for(uint i=st;i<=syncArrayIndex;i++)
    {
      addToLog(QString("index=%1 start=%2 end=%3 width=%4 linelength=%5 frequency=%6 volume=%7")
               .arg(i).arg(syncArray[i].start).arg(syncArray[i].end)
               .arg(syncArray[i].end-syncArray[i].start).arg(syncArray[i].end-syncArray[i-1].end)
               .arg(syncArray[i].freq).arg(rxVolumePtr[0]),LOGSYNC2);
    }
  if(withShadow)
    {
      if(start==0)
        {
          addToLog(QString("syncProcessor: shadowSyncArrayDump index=%1 start=%2 end=%3 width=%4 linelength=n/a  frequency=%5 volume=%6")
                   .arg(0).arg(shadowSyncArray[0].start).arg(shadowSyncArray[0].end)
                   .arg(shadowSyncArray[0].end-shadowSyncArray[0].start).arg(shadowSyncArray[0].freq).arg(rxVolumePtr[0]),LOGSYNC2);
                st=1;
        }
      for(uint i=st;i<=shadowSyncArrayIndex;i++)
        {
          addToLog(QString("syncProcessor: shadowSyncArrayDump index=%1 start=%2 end=%3 width=%4 linelength=%5 frequency=%6 volume=%7")
                   .arg(i).arg(shadowSyncArray[i].start).arg(shadowSyncArray[i].end)
                   .arg(shadowSyncArray[i].end-shadowSyncArray[i].start).arg(shadowSyncArray[i].end-shadowSyncArray[i-1].end)
                   .arg(shadowSyncArray[i].freq).arg(rxVolumePtr[0]),LOGSYNC2);
        }
    }
}

int syncProcessor::getSignalQuality()
{
  if(!syncFound)
    {
      syncQuality=0;
      lostLines=0;
      return 0;
    }
  if(syncArrayIndex<7)
    {
      syncQuality=4;
    }
  if((sampleCounter-syncArray[syncArrayIndex-1].end)>(samplesPerLine*sensitivityArray[squelch-1].syncLost))
    {
      syncQuality=0;
    }

  if(lostLines!=0)
    {
      syncQuality--;
      if(lostLines>3) syncQuality--;
      if(lostLines>5) syncQuality--;
      if(lostLines>7) syncQuality--;
      if(syncQuality<0) syncQuality=0;
      addToLog(QString("lostLines %1, sq:=%2").arg(lostLines).arg(syncQuality),LOGSYNC1);
      lostLines=0;
    }
  else
    {
      addToLog(QString("sq:=%2").arg(syncQuality),LOGSYNC1);
      if(syncQuality<10) syncQuality++;
    }
  return syncQuality;
}


void syncProcessor::trackSyncs(esstvMode selectedMode)
{
  double temp;
  double tempr;

  if((selectedMode!=mode)&&(selectedMode!=NOTVALID))
    {
//      qDebug() << "new mode detected";
   //   dumpSyncArray();
    }
  temp=(double)(syncArray[syncArrayIndex].end-syncArray[syncArrayIndex-1].end);
  tempr =round(temp/samplesPerLine)*samplesPerLine+0.00000001; // make it non-zero to avoid divide by zero
  temp=temp/tempr;
  if(fabs(1-temp)>SYNCDEVIATION)
    {
//      addToLog(QString("delete sync entry %1").arg(syncArrayIndex),LOGSYNC1);
      deleteSyncEntry(syncArrayIndex);
      return;
    }
  ssyncArray *ptr=&syncArray[syncArrayIndex];
  addToLog(QString("SyncArrayIndex %1").arg(syncArrayIndex),LOGSYNC1);

  if(syncArrayIndex>0)
    {
      ptr->length=ptr->end-syncArray[syncArrayIndex-1].end;
    }
  else  ptr->length=0;
  syncArray[syncArrayIndex].lineNumber=(int) round(((double)(syncArray[syncArrayIndex].end-syncArray[0].end))/samplesPerLine);
  lostLines=syncArray[syncArrayIndex].lineNumber-syncArray[syncArrayIndex-1].lineNumber-1;
  if(getSignalQuality()<3)
    {
      restart();
      return;
    }
  if(!autoSlantAdjust) return;
  if ((mode>=AVT24) && (mode <= AVT94)) return;
  if (syncArray[syncArrayIndex].lineNumber<slantAdjustLine) return;
  slantAdjust();
}


void syncProcessor::restart()
{
//  bool done=false;

//  //      dumpSyncArray(false,0);
//  syncLostEvent* ce;
//  ce = new 	syncLostEvent();
//  ce->waitFor(&done);
//  QApplication::postEvent(dispatcherPtr, ce);
//  while(!done) { qApp->processEvents();}
  addToLog("sync lost",LOGSYNC1);
  syncFound=false;
}

void syncProcessor::regression(DSPFLOAT &a,DSPFLOAT &b,int start, int end)
{
  /* calculate linear regression
    formula x=a+by
    b=sum((x[i]-xm)*(y[i]-ym))/sum((y[i]-ym)*(y[i]-ym))
    a=xm-b*ym
  */
  int i,j;

  DSPFLOAT sum_x,sum_y,sum_xx,sum_xy;
  int count=end-start+1;
  sum_x=sum_y=sum_xx=sum_xy=a=b=0;
  for(j=1,i=start;i<=end;i++,j++)
    {
      slantArray[j].y= (DSPFLOAT)(syncArray[i].end-syncArray[start-1].end);
      slantArray[j].x= round(slantArray[j].y/samplesPerLine)*samplesPerLine;
      sum_x+=slantArray[j].x;
      sum_y+=slantArray[j].y;
      sum_xx+=slantArray[j].x*slantArray[j].x;
      sum_xy+=slantArray[j].x*slantArray[j].y;
    }
  b=((count)*sum_xy-(sum_x*sum_y))/((count)*sum_xx-(sum_x*sum_x));
  a=sum_y/(count)-(b*sum_x)/(count);
}



bool syncProcessor::slantAdjust()
{
  DSPFLOAT a,b;
  if ((mode>=AVT24) && (mode <= AVT94)) return true;
  if(mode==NOTVALID) return true;
//  logfile->addToAux(QString("%1\t%2\t%3\t%4")
//                    .arg(syncArrayIndex)
//                    .arg(syncArray[syncArrayIndex].start)
//                    .arg(syncArray[syncArrayIndex].end)
//                    .arg(syncArray[syncArrayIndex].retrace)
//                    );

  if (((mode==S1)||(mode==S2)||(mode==SDX))&&(syncArray[0].retrace==true))
    {
      regression(a,b,2,syncArrayIndex);
    }
  else regression(a,b,1,syncArrayIndex);
  addToLog(QString("step:%1 a=%2 b=%3, modified rxclock=%4").arg(syncArrayIndex).arg(a).arg(b).arg(modifiedClock*b),LOGSLANT);
  slantAdjustLine+=7;
  if((fabs(1.-b)>0.00001)||(fabs(a)>1))
    {
      newClock=true;
      modifiedClock*=b;
      samplesPerLine=lineLength(mode,modifiedClock); //recalculate the samples per line
      addToLog("new clock accepted",LOGSLANT);
      syncPosition=syncArray[0].end+(long)round(a);
      addToLog(QString("slantAdjust: modified  syncpos:=%1").arg(syncPosition),LOGSLANT);
      syncDeviation=SYNCDEVIATION/3.;
    }
  return true;
}





