/*
 * zgetdump - Tool for copying and converting System z dumps
 *
 * Option parsing
 *
 * Copyright IBM Corp. 2001, 2010
 * Author(s): Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "zgetdump.h"
#include "zt_common.h"

/*
 * Text for --help option
 */
static char help_text[] =
"Usage: zgetdump [OPTIONS] [DUMPDEV] [DIR]\n"
"\n"
"The zgetdump tool takes as source a dump device or dump file (DUMPDEV)\n"
"and writes its contents to standard output, which you can redirect to a\n"
"specific file. Alternatively  you  can  also  mount the dump content.\n"
"Because zgetdump is able to read and write different dump formats, it\n"
"can  be used to convert a dump from one format to another. zgetdump can\n"
"also verify if a dump is valid or check, whether a DASD device contains\n"
"a valid dump tool.\n"
"\n"
"-h, --help       Print this help, then exit.\n"
"-v, --version    Print version information, then exit.\n"
"-m, --mount      Mount dump to mount point DIR\n"
"-u, --umount     Unmount dump from mount point DIR\n"
"-f, --fmt <FMT>  Specify target dump format FMT (\"elf\" or \"s390\")\n"
"-i, --info       Print dump information\n"
"-d, --device     Print dump device information\n";

static const char copyright_str[] = "Copyright IBM Corp. 2001, 2010";

/*
 * Initialize default settings
 */
static void init_defaults(void)
{
	g.prog_name = "zgetdump";
	g.opts.action = ZG_ACTION_STDOUT;
	g.opts.fmt = "s390";
	dfo_set(g.opts.fmt);
}

/*
 * Print "help" hint
 */
static void print_usage_exit(void)
{
	STDERR("Try '%s --help' for more information.\n", g.prog_name);
	zg_exit(1);
}

/*
 * Print help text
 */
static void print_help_exit(void)
{
	STDOUT(help_text);
	zg_exit(0);
}

/*
 * Print version information
 */
static void print_version_exit(void)
{
	STDOUT("%s: Tool for copying and converting dumps version %s\n",
	       g.prog_name, RELEASE_STRING);
	STDOUT("%s\n", copyright_str);
	zg_exit(0);
}

/*
 * Set "--fmt" option
 */
static void fmt_set(const char *fmt)
{
	if (dfo_set(fmt) != 0)
		ERR_EXIT("Invalid target format \"%s\" specified", fmt);
	g.opts.fmt_specified = 1;
	g.opts.fmt = fmt;
}

/*
 * Set mount point
 */
static void mount_point_set(const char *mount_point)
{
	g.opts.mount_point = zg_strdup(mount_point);
}

/*
 * Set device
 */
static void device_set(const char *path)
{
	g.opts.device = zg_strdup(path);
}

/*
 * Set FUSE debug options
 */
static void argv_fuse_set(char **argv, int argc)
{
	int i;

	g.opts.argv_fuse = argv;
	g.opts.argc_fuse = argc;

	STDERR_PR("Fuse Options: ");
	for (i = 0; i < argc; i++)
		STDERR("%s ", g.opts.argv_fuse[i]);
	STDERR("\n");
}

/*
 * Set action
 */
static void action_set(enum zg_action action)
{
	if (g.opts.action_specified)
		ERR_EXIT("Please specifiy only one of the \"-i\", \"-d\", "
			 "\"-m\" or \"-u\" option");
	g.opts.action = action;
	g.opts.action_specified = 1;
}

/*
 * Verify option combinations
 */
static void verify_opts(void)
{
	if (!g.opts.fmt_specified)
		return;

	if (g.opts.action == ZG_ACTION_DUMP_INFO)
		ERR_EXIT("The \"--fmt\" option cannot be specified "
			 "together with \"--info\"");
	if (g.opts.action == ZG_ACTION_DEVICE_INFO)
		ERR_EXIT("The \"--fmt\" option cannot be specified "
			 "together with \"--device\"");
	if (g.opts.action == ZG_ACTION_UMOUNT)
		ERR_EXIT("The \"--fmt\" option cannot be specified "
			 "together with \"--umount\"");
}

/*
 * Parse positional arguments
 */
static void parse_pos_args(char *argv[], int argc)
{
	int pos_args = argc - optind;

	switch (g.opts.action) {
	case ZG_ACTION_STDOUT:
	case ZG_ACTION_DUMP_INFO:
	case ZG_ACTION_DEVICE_INFO:
		if (pos_args == 0)
			ERR_EXIT("No device or dump specified");
		if (pos_args > 1 && !g.opts.debug_specified)
			ERR_EXIT("Too many positional paramters specified");
		device_set(argv[optind]);
		break;
	case ZG_ACTION_MOUNT:
		if (pos_args == 0)
			ERR_EXIT("No dump specified");
		if (pos_args == 1)
			ERR_EXIT("No mount point specified");
		if (pos_args > 2 && !g.opts.debug_specified)
			ERR_EXIT("Too many positional paramters specified");
		device_set(argv[optind]);
		mount_point_set(argv[optind + 1]);
		if (g.opts.debug_specified && pos_args > 2)
			argv_fuse_set(&argv[optind + 2], pos_args - 2);
		break;
	case ZG_ACTION_UMOUNT:
		if (pos_args == 0)
			ERR_EXIT("No mount point specified");
		mount_point_set(argv[optind]);
		break;
	}
}

/*
 * Main command line parsing function
 */
void opts_parse(int argc, char *argv[])
{
	int opt, idx;
	static struct option long_opts[] = {
		{"help",    no_argument,       NULL, 'h'},
		{"version", no_argument,       NULL, 'v'},
		{"info",    no_argument,       NULL, 'i'},
		{"device",  no_argument,       NULL, 'd'},
		{"mount",   no_argument,       NULL, 'm'},
		{"umount",  no_argument,       NULL, 'u'},
		{"fmt",     required_argument, NULL, 'f'},
		{"debug",   no_argument,       NULL, 'X'},
		{0,         0,                 0,     0 }
	};
	static const char optstr[] = "hvidmuf:X";

	init_defaults();
	while ((opt = getopt_long(argc, argv, optstr, long_opts, &idx)) != -1) {
		switch (opt) {
		case 'h':
			print_help_exit();
		case 'v':
			print_version_exit();
		case 'i':
			action_set(ZG_ACTION_DUMP_INFO);
			break;
		case 'd':
			action_set(ZG_ACTION_DEVICE_INFO);
			break;
		case 'm':
			action_set(ZG_ACTION_MOUNT);
			break;
		case 'u':
			action_set(ZG_ACTION_UMOUNT);
			break;
		case 'f':
			fmt_set(optarg);
			break;
		case 'X':
			g.opts.debug_specified = 1;
			break;
		default:
			print_usage_exit();
		}
	}
	parse_pos_args(argv, argc);
	verify_opts();
}
