//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2,er,mu,zonals)
// Lyddane orbit propagation analytical model
//
// Calling Sequence
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2 [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the mean orbital elements <b>mean_kep_t2</b> 
// and the osculating elements <b>osc_kep_t2</b>
// at time <b>t2</b> given the mean elements 
// <b>mean_kep_t1</b> at time <b>t1</b>.</p>
// <p>Zonals coefficients up to J5 are taken into account.</p>
// <p>There can be 1 or N initial times, and 1 or N final times. </p>
// <p></p></listitem>
// <listitem>
// <p>The orbital elements are the following:</p>  
// <p><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></p>
// <p></p></listitem>
// <listitem>
// <p>Warnings :</p>
// <p> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</p>
// <p> - This function does not work for eccentricities higher than 0.9</p></listitem>
// </itemizedlist>
//
// Parameters
// t1: Initial time [days] (1x1 or 1xN)
// mean_kep_t1: Mean orbital elements at time t1 (6x1 or 6xN).  
// t2: Final time [days] (1xN or 1x1).  
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonal coefficients J1 to Jn (troncated to J5) to be used (default is %CL_j1jn(1:5)) (Nz x 1)
// mean_kep_t2: Mean orbital elements at t2 (6xN)
// osc_kep_t2: Osculating orbital elements at t2 (6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1) CNES - MSLIB FORTRAN 90, Volume E (me_lyddane)
//
// See also
// CL_ex_meanLyddane
// CL_ex_lyddaneMan
//
// Examples
// t1 = 12584;
// mean_kep_t1 = [7078e3;7.9e-3;CL_deg2rad([97.2; 89; 125; 0])];
// t2 = 12587:0.2:12588;
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2)
//

//
// l'eccentricite doit appartenir a [ 0. , 0.9 [
// le domaine d'erreur est          < 0. et >= 0.9
//
// l'inclinaison  doit appartenir a [ 0. , pi ]

// initialisations

// Declarations:
if(~exists('%CL_criticalInc1')) then global %CL_criticalInc1; end;
if(~exists('%CL_criticalInc2')) then global %CL_criticalInc2; end;
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:


lhs = argn(1);
compute_osc_t2 = %f;
if (lhs == 2); compute_osc_t2 = %t; end

if (lhs > 2)
  CL__error('Too many output argumments');
end

if ~exists('er','local'); er=%CL_eqRad; end
if ~exists('mu','local'); mu=%CL_mu; end
if ~exists('zonals','local') 
   zonals = %CL_j1jn(1:5);
else
   zonals = [matrix(zonals, 1, -1), zeros(1,5)]; 
end

// valeur charnire de l'excentricit
eccel = 0.05;


// controle des donnees
// demi grand axe
if or(mean_kep_t1(1,:) <= 0)
  CL__error("Semi major axis < or = 0");
end

// eccentricite
if or(mean_kep_t1(2,:) < 0)
  CL__error("Negative eccentricity");
end

// inclinaison
if or(mean_kep_t1(3,:) < 0 | mean_kep_t1(3,:) > %pi)
  CL__error("Inclination out of range");
end

// inclinaison proche de l'inclinaison critique.
// obtenu experimentalement : au dela ca plante
eps_i_critique = 4e-5; 
if (abs(mean_kep_t1(3,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(mean_kep_t1(3,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("Inclination too close to critical value");
end

// Empty final times
if (t2 == []) 
  mean_kep_t2 = [];
  osc_kep_t2 = [];
  return; // *** RETURN ***
end

// check arguments sizes
[t1, mean_kep_t1, N1] = CL__checkInputs(t1,1, mean_kep_t1, 6); 
[t2, N2] = CL__checkInputs(t2,1); // check nb of rows == 1

if ~(N1 == 1 | N2 == 1 | N1 == N2)
   CL__error('Wrong size of input arguments'); 
end
N = max(N1, N2);

mean_kep_t2 = zeros(6, N);
if (compute_osc_t2); osc_kep_t2 = zeros(6, N); end


// DEBUT DU MODELE
// ===============

// -------------------------------------------------
// Calcul d'argument du perigee, raan et anm seculaire
// -------------------------------------------------
function [po1,go1,am1] = pom_raan_anm_secu()
  // determination argument du perigee po1 :
  h22 = -35 + 24*tt1 + 25*(tt1.^2) + (90-192*tt1-126*(tt1.^2)) .* (c1.^2)+(385+360*tt1+45*(tt1.^2)) .* (c1.^4);
  h41 = 21 - 9*(tt1.^2)+(-270 + 126*(tt1.^2)) .* (c1.^2) + (385 - 189*(tt1.^2)) .* (c1.^4);
  po1 = mean_kep_t1(4,:) + ant .* (1.5 * h2 .* (-dc + h2 .* h22/16) + (5/16) * h4 .* h41); 

  // determination ascension droite du noeud go1 :
  h22 = (-5 + 12*tt1 + 9 * (tt1.^2)) .* c1 + (-35 -36*tt1 -5*(tt1.^2)) .* (c1.^3);
  h41 = (5 - 3*(tt1.^2)) .* c1 .* (3 - 7*(c1.^2));
  go1 = mean_kep_t1(5,:) + ant .* (3 * h2 .* (-c1 + h2 .* h22/8) + 5 * h4 .* h41/4);    

  // determination anomalie moyenne am1 :
  h22 = -15 + 16*tt1 + 25*(tt1.^2) + (30 - 96*tt1 - 90*(tt1.^2)) .* (c1.^2) + (105 + 144*tt1 + 25 *(tt1.^2)) .* (c1.^4);
  h41 = (3 - 30*(c1.^2) + 35*(c1.^4));
  am1 = mean_kep_t1(6,:) + ant .* (1 + 1.5 * h2 .* (tt1 .* (-1 + 3*(c1.^2)) + h2 .* tt1 .* h22/16) + 15 * h4 .* tt1 .* e2 .* h41/16);
endfunction


// Passage aux unites utilisees par l'algorithme : km
// --------------------------------------------------
KM2M = 1000; 
er = er / KM2M;
mean_kep_t1(1,:) = mean_kep_t1(1,:) / KM2M;
mu = mu * 1.e-9;


// constantes liees a 1/2 grand-axe, eccentricite et inclinaison
// ============================================================
q = er ./ mean_kep_t1(1,:); 

g2 = zonals(2) * (q^2) / 2;

e2 = mean_kep_t1(2,:).^2
tt1 = sqrt(1-e2);

h2 = g2 ./ (tt1.^4);
h3 = -zonals(3) * (q^3) ./ (tt1.^6);
h4 = -zonals(4) * (q^4) * (3/8) ./ (tt1.^8);
h5 = -zonals(5) * (q^5) ./ (tt1.^10);

ant = (t2-t1) .* sqrt(mu ./ mean_kep_t1(1,:)) ./ mean_kep_t1(1,:) * 86400.;

c1 = cos(mean_kep_t1(3,:));
dc = 1 - 5 * (c1.^2);

s1 = sin(mean_kep_t1(3,:));

r1 = 4 + 3 * e2;
r2 = 4 + 9 * e2;
r4 = e2 .* (c1.^6) ./ dc.^2;


// calcul des termes seculaires en J2,J2 * * 2,J4
// ===========================================
[po1,go1,am1] = pom_raan_anm_secu();

// Sortie des elements moyens a la date t2
// =======================================
mean_kep_t2(1,:) = mean_kep_t1(1,:) * KM2M; 
mean_kep_t2(2,:) = mean_kep_t1(2,:);
mean_kep_t2(3,:) = mean_kep_t1(3,:);
// seuls pom, gom et M sont modifies
mean_kep_t2(4,:) = po1;
mean_kep_t2(5,:) = go1;
mean_kep_t2(6,:) = am1;
am1 = CL_rMod(am1,2 * %pi);

// FIN DU MODELE POUR LES MOYENS
//==============================


// MODELE POUR LES OSCULATEURS
//============================
if (compute_osc_t2)

  cdai = cos(0.5 * mean_kep_t1(3,:));
  sdai = sin(0.5 * mean_kep_t1(3,:));

  //  calcul des termes a longues periodes en J2,J3,J4,J5
  //  ===================================================

  cc = 2 * (c1.^2) ./ dc;

  q0 = 16 * ((c1.^2) ./ dc) + 40 * ((c1.^4) ./ dc.^2);

  a = 0.5 * (1- 5 .* cc) .* s1 .* h2 - 5 * (1- cc) .* s1 .* h4/(3 .* h2);
  b = ((tt1.^2) .* h3 + (5/16) * r1 .* (3 * (1- cc) .* s1.^2 - 2) ./ (tt1.^2) .* h5) ./ h2;
  c = (35/32) * h5 .* ((1- 2 * cc) .* s1.^2) ./ (3 * h2);

  dai2 = mean_kep_t1(2,:) .* (-a .* cos(2 * po1) + mean_kep_t1(2,:) .* c .* sin(3 * po1));
  dex2 = 0.25 * (-(tt1.^2) .* dai2 + b .* sin(po1)) .* s1;
  dai2 = 0.25 * (dai2 - b .* sin(po1) ./ (tt1.^2)) .* (mean_kep_t1(2,:) .* c1);

  edm2 = - h3 - (5/16) * r2 .* h5 .* (3 * (1- cc) .* s1.^2 - 2);
  edm2 = edm2 .* cos(po1) ./ h2 + mean_kep_t1(2,:) .* a .* sin(2 * po1);
  edm2 = edm2 + e2 .* c .* cos(3 * po1);
  edm2 = 0.25 * edm2 .* (tt1.^3) .* s1;

  a = -0.5 * (11+5 * q0) .* h2 + 5 * (3+q0) .* h4 ./ (3 * h2);
  a = (mean_kep_t1(2,:) .* s1) .* a .* sin(2 * po1);
  b = (5/8) * r1 .* (0.5 * (3 .* (1- cc) .* s1.^2 - 2) + 3 * (s1.^2) .* (3+q0));
  b = (h3 + b .* h5) .* cos(po1) ./ h2;
  c = (35/16) * (0.5 * ((1- 2 * cc) .* s1.^2) + (s1.^2) .* (5+2 * q0)) .* h5 ./ (9 * h2);
  c = e2 .* c .* cos(3 * po1);

  sidg2 = 0.25 * (mean_kep_t1(2,:) .* c1) .* (a+b-c);

  a = c1 ./ (1+c1);
  b = tt1 ./ (1+tt1);
  c = tt1 + 1 ./ (1+tt1);

  som1 = -(c+2.5) .* (1- 5 .* cc) .* s1.^2 - c1 .* (11+5 * q0) - 11 * (c1.^2);
  som1 = (som1 + 2) .* e2 + 200 * r4;
  som2 = (c+2.5) .* (1- cc) .* s1.^2 + c1 .* (3+q0) + 3 * (c1.^2);
  som2 = (som2 -2) .* e2 - 40 * r4;
  som3 = (a+c) .* (mean_kep_t1(2,:) .* s1);
  som4 = (a + tt1 .* b) .* r1;
  som4 = som4 + e2 .* (9 + 6 * c) +20;
  som4 = som4 .* (3 * (1- cc) .* s1.^2 - 2) + 6 * r1 .* c1 .* (1-c1) .* (3+q0);
  som4 = (mean_kep_t1(2,:) .* s1) .* som4;
  som5 = -0.5 * ((1- 2 * cc) .* s1.^2) .* (a + 3 * c + 2);
  som5 = (som5 - c1 .* (1-c1) .* (5+2 * q0)) .* (mean_kep_t1(2,:) .* s1) .* e2;

  pgm2 = 0.125 * (h2 .* som1 + 10 * h4 .* som2 ./ (3 * h2)) .* sin(2 * po1);
  pgm2 = pgm2 + (0.25 * h3 .* som3 + (5/64) * h5 .* som4) .* cos(po1) ./ h2;
  pgm2 = pgm2 + (35/64) * h5 .* som5 .* cos(3 * po1) ./ (9 * h2);
  pgm2 = pgm2 + (po1+go1+am1);

  // calcul des termes a courtes periodes en J2
  // ==========================================
  am2 = am1;
  ecc2 = mean_kep_t1(2,:);
  po2 = po1;

  x = zeros(am1);
  y = zeros(am1);
  ede2 = zeros(am1);
  gom2 = zeros(am1);
  aux = zeros(am1);
  auxi = zeros(am1);
  mkt12 = mean_kep_t1(2,:);
  
  // recherche conditions sur excentricite
  I = find(mean_kep_t1(2,:) > eccel);
    // Si on a un seul bulletin d'entree : 
	// on etend certaines variables pour que tout ait la meme dimension
	
  if (size(mean_kep_t1,2) == 1 & I <> [])
    I = 1:length(t2);
    mkt12 = mkt12 * ones(t2);
    sdai = sdai * ones(t2);
    cdai = cdai * ones(t2);
  end
    
  ede2(I) = mkt12(I) + dex2(I);
  x(I) = ede2(I) .* cos(am1(I)) - edm2(I) .* sin(am1(I));
  y(I) = ede2(I) .* sin(am1(I)) + edm2(I) .* cos(am1(I));
  am2(I) = CL__sc2angle(x(I),y(I));
    
  ecc2(I) = sqrt(x(I).^2 + y(I).^2);
  aux(I) = sdai(I) + 0.5 * dai2(I) .* cdai(I);
  auxi(I) = 0.5 * sidg2(I) ./ cdai(I);
  x(I)=aux(I) .* cos(go1(I)) - auxi(I) .* sin(go1(I));
  y(I)=aux(I) .* sin(go1(I)) + auxi(I) .* cos(go1(I));

  gom2(I) = CL__sc2angle(x(I),y(I));

  po2(I) = pgm2(I) - gom2(I) - am2(I);
  // -- end I -- 
  
  e = CL_kp_M2E(ecc2,am2);
  v = CL__sc2angle(cos(e) - ecc2 , sqrt(1 - ecc2 .* ecc2) .* sin(e));
  xc = (3 * (1+mean_kep_t1(2,:) .* cos(v)) + e2 .* cos(v).^2) .* cos(v);

  dax3 = mean_kep_t1(2,:) .* (-1+3 * (c1.^2)) .* (xc+mean_kep_t1(2,:) .* c) + (1 + mean_kep_t1(2,:) .* xc) .* (3 * (s1.^2) .* cos(2 * po2+2 * v));
  dax3 = mean_kep_t1(1,:) .* dax3 .* g2 ./ (tt1.^6);

  de1 = ((-1+3 * (c1.^2)) .* (xc+mean_kep_t1(2,:) .* c) + (3 * (s1.^2) .* cos(2 * po2+2 * v)) .* (xc+mean_kep_t1(2,:))) ./ (tt1.^6);
  de2 = cos(2 * po2+3 .* v) + 3 * cos(2 * po2+v);

  dex3 = 0.5 * (tt1.^2) .* (de1 .* g2 - de2 .* (s1.^2) .* h2);

  dai3 = (mean_kep_t1(2,:) .* de2 + 3 * cos(2 * po2+2 * v)) / 2;
  dai3 = c1 .* s1 .* dai3 .* h2;

  a1 = 1+mean_kep_t1(2,:) .* cos(v);
  a2 = a1 .* (1+a1) ./ (tt1.^2);
  d1 = 2 * (3 * (c1.^2)-1) .* (1+a2) .* sin(v);
  d1 = d1+(s1.^2) .* (3 * (1-a2) .* sin(2 * po2+v) + (3 * a2+1) .* sin(2 * po2+3 * v));

  d21 = 6 * (v + mean_kep_t1(2,:) .* sin(v) - am2);
  d22 = 3 * sin(2 * po2+2 * v) + mean_kep_t1(2,:) .* (3 .* sin(2 * po2+v) + sin(2 * po2+3 * v));

  a1 = 5 * (c1.^2) - 2 * c1 -1;

  dpgm3 = mean_kep_t1(2,:) .* b .* tt1 .* d1 + a1 .* d21 + (2-a1) .* d22;
  dpgm3 = 0.25 * dpgm3 .* h2;

  edm = edm2 + - 0.25 * (tt1.^3) .* d1 .* h2;
  sidg = sidg2 + 0.5 * c1 .* s1 .* (d22-d21) .* h2;

  ede = mean_kep_t1(2,:) + dex2 + dex3;

  am3 = CL__sc2angle(ede .* cos(am1) - edm .* sin(am1) , ede .* sin(am1) + edm .* cos(am1));

  sidi = sdai + 0.5 * (dai2 + dai3) .* cdai
  sidg = 0.5 * sidg ./ cdai  

  go3 = CL__sc2angle(sidi .* cos(go1) - sidg .* sin(go1) , sidi .* sin(go1) + sidg .* cos(go1))

  // FIN DU MODELE POUR LES OSCULATEURS
  //===================================
  osc_kep_t2(1,:) = (mean_kep_t1(1,:) + dax3) .* KM2M; 
  osc_kep_t2(2,:) = sqrt(ede .* ede + edm .* edm);
  osc_kep_t2(3,:) = 2 * asin(sqrt(sidi .* sidi + sidg .* sidg));
  osc_kep_t2(4,:) = pgm2 + dpgm3 - go3 - am3;
  osc_kep_t2(5,:) = go3;
  osc_kep_t2(6,:) = am3;

  osc_kep_t2(4:6,:) = CL_rMod(osc_kep_t2(4:6,:), mean_kep_t2(4:6,:)-%pi, mean_kep_t2(4:6,:)+%pi); 

end 


endfunction


