/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to calculate
 *
 * @author Stephane MANKOWSKI
 */
#include "skgcalculatorplugin.h"
#include "skgcalculatorpluginwidget.h"
#include "skginterestboardwidget.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kaboutdata.h>


/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGCalculatorPluginFactory, registerPlugin<SKGCalculatorPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGCalculatorPluginFactory("skrooge_calculator", "skrooge_calculator"))

SKGCalculatorPlugin::SKGCalculatorPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
    SKGTRACEIN(10, "SKGCalculatorPlugin::SKGCalculatorPlugin");
}

SKGCalculatorPlugin::~SKGCalculatorPlugin()
{
    SKGTRACEIN(10, "SKGCalculatorPlugin::~SKGCalculatorPlugin");
    m_currentBankDocument = NULL;
}

bool SKGCalculatorPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEIN(10, "SKGCalculatorPlugin::setupActions");
    Q_UNUSED(iArgument);
    m_currentBankDocument = qobject_cast<SKGDocumentBank*>(iDocument);
    if (m_currentBankDocument == NULL) return false;

    setComponentData(KGlobal::mainComponent());
    setXMLFile("../skrooge_calculator/skrooge_calculator.rc");

    //Create yours actions here
    return true;
}

int SKGCalculatorPlugin::getNbDashboardWidgets()
{
    return 1;
}

QString SKGCalculatorPlugin::getDashboardWidgetTitle(int iIndex)
{
    if (iIndex == 0) return i18nc("The estimated amount of money earned through interests on a remunerated account", "Estimated interest");
    return "";
}

SKGWidget* SKGCalculatorPlugin::getDashboardWidget(int iIndex)
{
    if (iIndex == 0) return new SKGInterestBoardWidget(m_currentBankDocument);
    return NULL;
}

SKGTabPage* SKGCalculatorPlugin::getWidget()
{
    SKGTRACEIN(10, "SKGCalculatorPlugin::getWidget");
    return new SKGCalculatorPluginWidget(m_currentBankDocument);
}

QString SKGCalculatorPlugin::title() const
{
    return toolTip();
}

QString SKGCalculatorPlugin::icon() const
{
    return "view-expenses-categories";
}

QString SKGCalculatorPlugin::toolTip() const
{
    return i18nc("Compute financial simulations for various situations (interests...)", "Simulations");
}

int SKGCalculatorPlugin::getOrder() const
{
    return 100;
}

QStringList SKGCalculatorPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... you can use the calculator for many things</p>"));
    return output;
}

bool SKGCalculatorPlugin::isInPagesChooser() const
{
    return true;
}
SKGAdviceList SKGCalculatorPlugin::advices() const
{
    SKGTRACEIN(10, "SKGCalculatorPlugin::advices");
    SKGAdviceList output;

    //Search investment accounts without interest rate
    SKGObjectBase::SKGListSKGObjectBase accounts;
    m_currentBankDocument->getObjects("account", "t_type='I' AND NOT EXISTS (SELECT 1 FROM interest WHERE interest.rd_account_id=account.id)", accounts);
    int nb = accounts.count();
    for (int i = 0; i < nb; ++i) {
        SKGAccountObject account = accounts.at(i);
        SKGAdvice ad;
        ad.setUUID("skgcalculatorplugin_nointerest|" % account.getName());
        ad.setPriority(3);
        ad.setShortMessage(i18nc("User did not define an interest rate on an investment account", "No interest rate defined for account '%1'", account.getName()));
        ad.setLongMessage(i18nc("User did not define an interest rate on an investment account", "Your investment account '%1' doesn't have interest rate defined", account.getName()));
        QStringList autoCorrections;
        autoCorrections.push_back(i18nc("Link allowing user to open a new tab for defining interests parameters", "Open interest page"));
        ad.setAutoCorrections(autoCorrections);
        output.push_back(ad);
    }
    return output;
}

SKGError SKGCalculatorPlugin::executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) const
{
    if (m_currentBankDocument && iAdviceIdentifier.startsWith(QLatin1String("skgcalculatorplugin_nointerest|"))) {
        //Get parameters
        QString account = iAdviceIdentifier.right(iAdviceIdentifier.length() - 31);

        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent(m_currentBankDocument->getParameter("SKGCALCULATOR_DEFAULT_PARAMETERS"));
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
            root = doc.createElement("parameters");
            doc.appendChild(root);
        }

        root.setAttribute("currentPage", "0");
        root.setAttribute("account", account);
        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge calculator plugin"), -1, doc.toString());
        return SKGError();
    }

    return SKGInterfacePlugin::executeAdviceCorrection(iAdviceIdentifier, iSolution);
}
#include "skgcalculatorplugin.moc"
