-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler.Conversions.ToString)
procedure UncondDependency (Err_Num          : in     Error_Types.NumericError;
                            With_Explanation : in     Boolean;
                            E_Str            : in out E_Strings.T) is
   Err_Type : ErrorHandler.Full_Depend_Err_Type;

   procedure UncondDependencyExpl (E_Str : in out E_Strings.T)
   --# global in Err_Type;
   --# derives E_Str from *,
   --#                    Err_Type;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an Append_String for the
   --      explanatory text that is delineated by --! comments.

   procedure Append_Explanation
   --# global in     Err_Type;
   --#        in     With_Explanation;
   --#        in out E_Str;
   --# derives E_Str from *,
   --#                    Err_Type,
   --#                    With_Explanation;
   is
      Explanation_String : E_Strings.T := E_Strings.Empty_String;
   begin
      if With_Explanation then
         -- we need to at least look for an explanation
         UncondDependencyExpl (E_Str => Explanation_String);
         if E_Strings.Get_Length (E_Str => Explanation_String) > 0 then
            -- there actually is one
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Prefix);
            E_Strings.Append_Examiner_String (E_Str1 => E_Str,
                                              E_Str2 => Explanation_String);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Postfix);
         end if;
      end if;
   end Append_Explanation;

begin
   Err_Type := ErrorHandler.Dependency_Err_Type'Val (Err_Num.ErrorNum - Error_Types.UncondDependencyErrorOffset);
   case Err_Type is

      -- HTML Directives
      --! <NameFormat> <"flow-"><Name>
      --! <ErrorFormat> <"!!! Flow Error : "><Name><" : "><Error>

      when ErrorHandler.Not_Used_New =>
         --! <Name> 50
         Append_Export_Var (E_Str, Err_Num.Name2, Err_Num.Scope, True);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not derived from the imported value(s) of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The item before &quot;is not derived ...&quot; is an export or function return value and the item(s)
         --! after are imports of the subprogram.  The message indicates that a dependency, stated in the dependency
         --! relation (derives annotation) or implied by the function signature is not present in the code.
         --! The absence of a stated dependency
         --! is always an error in either code or annotation.

      when ErrorHandler.Not_Used =>
         --! <Name> 50
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The imported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not used in the derivation of ");
         Append_Export_Var (E_Str, Err_Num.Name2, Err_Num.Scope, False);
         --! The variable XXX, which appears in the dependency relation of a
         --! procedure subprogram, as an import from which the export YYY is derived,
         --! is not used in the code for that purpose. YYY may be a function return value.
         --! This version of the message has been retained for backward compatibility.

      when ErrorHandler.Not_Used_Continue =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ", ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when ErrorHandler.Ineff_Init =>
         --! <Name> 53
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The package initialization of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is ineffective");
         --! Here XXX is an own variable of a package, initialized in the package
         --! initialization.
         --! The message states that XXX is updated elsewhere, before being read.

      when ErrorHandler.Ineff_Local_Init =>
         --! <Name> 54
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The initialization at declaration of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is ineffective");
         --! Issued if the value assigned to a variable at declaration cannot
         --! affect the final value of any exported variable of the subprogram in
         --! which it occurs because, for example, it is overwritten before it is used.

      when ErrorHandler.Policy_Violation =>
         --! <Name> 57
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Information flow from ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " to ");
         Append_Export_Var (E_Str, Err_Num.Name2, Err_Num.Scope, True);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " violates the selected information flow policy");
         --! Issued if safety or security policy checking is enabled and the specified
         --! dependency relation contains a relationship in which the flow of information
         --! from state or input to state or output violates the selected policy.

      when others =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "UNKNOWN ERROR NUMBER PASSED TO Uncond_Dependency");
   end case;
   Append_Explanation;
   if Err_Type /= ErrorHandler.Not_Used_Continue and Err_Type /= ErrorHandler.Not_Used_New then
      E_Strings.Append_String (E_Str => E_Str,
                               Str   => ".");
   end if;
end UncondDependency;
