/*

 srs_init.c -- populating the SPATIAL_REF_SYS table

 version 3.0, 2011 July 20

 Author: Sandro Furieri a.furieri@lqt.it

 -----------------------------------------------------------------------------
 
 Version: MPL 1.1/GPL 2.0/LGPL 2.1
 
 The contents of this file are subject to the Mozilla Public License Version
 1.1 (the "License"); you may not use this file except in compliance with
 the License. You may obtain a copy of the License at
 http://www.mozilla.org/MPL/
 
Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
for the specific language governing rights and limitations under the
License.

The Original Code is the SpatiaLite library

The Initial Developer of the Original Code is Alessandro Furieri
 
Portions created by the Initial Developer are Copyright (C) 2008
the Initial Developer. All Rights Reserved.

Contributor(s):

Alternatively, the contents of this file may be used under the terms of
either the GNU General Public License Version 2 or later (the "GPL"), or
the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
in which case the provisions of the GPL or the LGPL are applicable instead
of those above. If you wish to allow use of your version of this file only
under the terms of either the GPL or the LGPL, and not to allow others to
use your version of this file under the terms of the MPL, indicate your
decision by deleting the provisions above and replace them with the notice
and other provisions required by the GPL or the LGPL. If you do not delete
the provisions above, a recipient may use your version of this file under
the terms of any one of the MPL, the GPL or the LGPL.
 
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef SPL_AMALGAMATION		/* spatialite-amalgamation */
#include <spatialite/sqlite3.h>
#else
#include <sqlite3.h>
#endif

#ifdef _WIN32
#define strcasecmp	_stricmp
#endif /* not WIN32 */

#include <spatialite.h>

struct epsg_defs
{
    int srid;
    char *auth_name;
    int auth_srid;
    char *ref_sys_name;
    char *proj4text;
    char *srs_wkt;
    struct epsg_defs *next;
};

static void
free_epsg_def (struct epsg_defs *ptr)
{
/* memory cleanup - destroying an EPSG def item */
    if (ptr->auth_name)
	free (ptr->auth_name);
    if (ptr->ref_sys_name)
	free (ptr->ref_sys_name);
    if (ptr->proj4text)
	free (ptr->proj4text);
    if (ptr->srs_wkt)
	free (ptr->srs_wkt);
    free (ptr);
}

static struct epsg_defs *
add_epsg_def (struct epsg_defs **first, struct epsg_defs **last, int srid,
	      const char *auth_name, int auth_srid, const char *ref_sys_name)
{
/* appending an EPSG def to the list */
    int len;
    struct epsg_defs *p = malloc (sizeof (struct epsg_defs));
    if (!p)
	return NULL;
    p->srid = srid;
    p->auth_name = NULL;
    p->auth_srid = auth_srid;
    p->ref_sys_name = NULL;
    p->proj4text = NULL;
    p->srs_wkt = NULL;
    p->next = NULL;
    if (auth_name)
      {
	  len = strlen (auth_name);
	  if (len > 0)
	    {
		p->auth_name = malloc (len + 1);
		if (p->auth_name == NULL)
		    goto error;
		strcpy (p->auth_name, auth_name);
	    }
      }
    if (ref_sys_name)
      {
	  len = strlen (ref_sys_name);
	  if (len > 0)
	    {
		p->ref_sys_name = malloc (len + 1);
		if (p->ref_sys_name == NULL)
		    goto error;
		strcpy (p->ref_sys_name, ref_sys_name);
	    }
      }
    if (*first == NULL)
	*first = p;
    if (*last != NULL)
	(*last)->next = p;
    *last = p;
    return p;
  error:
    free_epsg_def (p);
    return NULL;
}

static void
add_proj4text (struct epsg_defs *p, int count, const char *text)
{
/* creating the PROJ4TEXT string */
    int len;
    int olen;
    char *string;
    if (text == NULL)
	return;
    len = strlen (text);
    if (!count)
      {
	  p->proj4text = malloc (len + 1);
	  if (p->proj4text == NULL)
	      return;
	  strcpy (p->proj4text, text);
	  return;
      }
    if (p->proj4text == NULL)
	return;
    olen = strlen (p->proj4text);
    string = malloc (len + olen + 1);
    if (string == NULL)
	return;
    strcpy (string, p->proj4text);
    free (p->proj4text);
    p->proj4text = string;
    strcat (p->proj4text, text);
}

static void
add_srs_wkt (struct epsg_defs *p, int count, const char *text)
{
/* creating the SRS_WKT string */
    int len;
    int olen;
    char *string;
    if (text == NULL)
	return;
    len = strlen (text);
    if (!count)
      {
	  p->srs_wkt = malloc (len + 1);
	  if (p->srs_wkt == NULL)
	      return;
	  strcpy (p->srs_wkt, text);
	  return;
      }
    if (p->srs_wkt == NULL)
	return;
    olen = strlen (p->srs_wkt);
    string = malloc (len + olen + 1);
    if (string == NULL)
	return;
    strcpy (string, p->srs_wkt);
    free (p->srs_wkt);
    p->srs_wkt = string;
    strcat (p->srs_wkt, text);
}

static void
initialize_epsg_00 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2000, "epsg", 2000,
		      "Anguilla 1957 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1, "_0=400000 +y_0=0 +ellps=clrk80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Anguilla 1957 / British West Indies Grid\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Anguilla 1957\",DATUM[\"Anguilla_1957\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7012\"]],AUTHORITY[\"EPSG\",\"6600\"]],PRIMEM[\"Gre");
    add_srs_wkt (p, 4,
		 "enwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4600\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 9,
		 "dian\",-62],PARAMETER[\"scale_factor\",0.9995],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",400000],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2000\"],AXIS[\"Easting\",EAST],A");
    add_srs_wkt (p, 12, "XIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2001, "epsg", 2001,
		      "Antigua 1943 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1, "_0=400000 +y_0=0 +ellps=clrk80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Antigua 1943 / British West Indies Grid\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"Antigua 1943\",DATUM[\"Antigua_1943\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7012\"]],AUTHORITY[\"EPSG\",\"6601\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4601\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 7,
		 ",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETE");
    add_srs_wkt (p, 8,
		 "R[\"latitude_of_origin\",0],PARAMETER[\"central_meridian");
    add_srs_wkt (p, 9,
		 "\",-62],PARAMETER[\"scale_factor\",0.9995],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_easting\",400000],PARAMETER[\"false_northing\",0],A");
    add_srs_wkt (p, 11,
		 "UTHORITY[\"EPSG\",\"2001\"],AXIS[\"Easting\",EAST],AXIS[");
    add_srs_wkt (p, 12, "\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2002, "epsg", 2002,
		      "Dominica 1945 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1,
		   "_0=400000 +y_0=0 +ellps=clrk80 +towgs84=725,685,536,0,0,");
    add_proj4text (p, 2, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Dominica 1945 / British West Indies Grid\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Dominica 1945\",DATUM[\"Dominica_1945\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7012\"]],TOWGS84[725,685,536,0,0,0,0],AUTHORITY[\"E");
    add_srs_wkt (p, 4,
		 "PSG\",\"6602\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 5,
		 "\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHOR");
    add_srs_wkt (p, 6,
		 "ITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4602\"]],UN");
    add_srs_wkt (p, 7,
		 "IT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[");
    add_srs_wkt (p, 8,
		 "\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 9,
		 ",0],PARAMETER[\"central_meridian\",-62],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",0.9995],PARAMETER[\"false_easting\",400000],P");
    add_srs_wkt (p, 11,
		 "ARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2002");
    add_srs_wkt (p, 12, "\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2003, "epsg", 2003,
		      "Grenada 1953 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1,
		   "_0=400000 +y_0=0 +ellps=clrk80 +towgs84=72,213.7,93,0,0,");
    add_proj4text (p, 2, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Grenada 1953 / British West Indies Grid\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"Grenada 1953\",DATUM[\"Grenada_1953\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7012\"]],TOWGS84[72,213.7,93,0,0,0,0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"6603\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 5,
		 "\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY");
    add_srs_wkt (p, 6,
		 "[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4603\"]],UNIT[");
    add_srs_wkt (p, 7,
		 "\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"T");
    add_srs_wkt (p, 8,
		 "ransverse_Mercator\"],PARAMETER[\"latitude_of_origin\",0");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-62],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9995],PARAMETER[\"false_easting\",400000],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2003\"");
    add_srs_wkt (p, 12, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2004, "epsg", 2004,
		      "Montserrat 1958 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1,
		   "_0=400000 +y_0=0 +ellps=clrk80 +towgs84=174,359,365,0,0,");
    add_proj4text (p, 2, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Montserrat 1958 / British West Indies Grid\",GE");
    add_srs_wkt (p, 1,
		 "OGCS[\"Montserrat 1958\",DATUM[\"Montserrat_1958\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7012\"]],TOWGS84[174,359,365,0,0,0,0],AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"6604\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 5,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 6,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4604\"");
    add_srs_wkt (p, 7,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 8,
		 "TION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_or");
    add_srs_wkt (p, 9,
		 "igin\",0],PARAMETER[\"central_meridian\",-62],PARAMETER[");
    add_srs_wkt (p, 10,
		 "\"scale_factor\",0.9995],PARAMETER[\"false_easting\",400");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12,
		 "\"2004\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2005, "epsg", 2005,
		      "St. Kitts 1955 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1, "_0=400000 +y_0=0 +ellps=clrk80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"St. Kitts 1955 / British West Indies Grid\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"St. Kitts 1955\",DATUM[\"St_Kitts_1955\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7012\"]],AUTHORITY[\"EPSG\",\"6605\"]],PRIMEM[\"G");
    add_srs_wkt (p, 4,
		 "reenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree");
    add_srs_wkt (p, 5,
		 "\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"4605\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",-62],PARAMETER[\"scale_factor\",0.9995],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_easting\",400000],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 11,
		 ",0],AUTHORITY[\"EPSG\",\"2005\"],AXIS[\"Easting\",EAST],");
    add_srs_wkt (p, 12, "AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2006, "epsg", 2006,
		      "St. Lucia 1955 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1,
		   "_0=400000 +y_0=0 +ellps=clrk80 +towgs84=-149,128,296,0,0");
    add_proj4text (p, 2, ",0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"St. Lucia 1955 / British West Indies Grid\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"St. Lucia 1955\",DATUM[\"St_Lucia_1955\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7012\"]],TOWGS84[-149,128,296,0,0,0,0],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6606\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4606\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",-62],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",0.9995],PARAMETER[\"false_easting\",40000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12,
		 "2006\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2007, "epsg", 2007,
		      "St. Vincent 45 / British West Indies Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-62 +k=0.9995000000000001 +x");
    add_proj4text (p, 1,
		   "_0=400000 +y_0=0 +ellps=clrk80 +towgs84=195.671,332.517,");
    add_proj4text (p, 2, "274.607,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"St. Vincent 45 / British West Indies Grid\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"St. Vincent 1945\",DATUM[\"St_Vincent_1945\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7012\"]],TOWGS84[195.671,332.517,274.607,0,0,");
    add_srs_wkt (p, 4,
		 "0,0],AUTHORITY[\"EPSG\",\"6607\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 5,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 6,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"4607\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 8,
		 "01\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"l");
    add_srs_wkt (p, 9,
		 "atitude_of_origin\",0],PARAMETER[\"central_meridian\",-6");
    add_srs_wkt (p, 10,
		 "2],PARAMETER[\"scale_factor\",0.9995],PARAMETER[\"false_");
    add_srs_wkt (p, 11,
		 "easting\",400000],PARAMETER[\"false_northing\",0],AUTHOR");
    add_srs_wkt (p, 12,
		 "ITY[\"EPSG\",\"2007\"],AXIS[\"Easting\",EAST],AXIS[\"Nor");
    add_srs_wkt (p, 13, "thing\",NORTH]]");
    p = add_epsg_def (first, last, 2008, "epsg", 2008,
		      "NAD27(CGQ77) / SCoPQ zone 2");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-55.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 2\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-55.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2008\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2009, "epsg", 2009,
		      "NAD27(CGQ77) / SCoPQ zone 3");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-58.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 3\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-58.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2009\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2010, "epsg", 2010,
		      "NAD27(CGQ77) / SCoPQ zone 4");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-61.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 4\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-61.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2010\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2011, "epsg", 2011,
		      "NAD27(CGQ77) / SCoPQ zone 5");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-64.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 5\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-64.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2011\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2012, "epsg", 2012,
		      "NAD27(CGQ77) / SCoPQ zone 6");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-67.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 6\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-67.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2012\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2013, "epsg", 2013,
		      "NAD27(CGQ77) / SCoPQ zone 7");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-70.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 7\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-70.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2013\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2014, "epsg", 2014,
		      "NAD27(CGQ77) / SCoPQ zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-73.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 8\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-73.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2014\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2015, "epsg", 2015,
		      "NAD27(CGQ77) / SCoPQ zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-76.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 9\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-76.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2015\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2016, "epsg", 2016,
		      "NAD27(CGQ77) / SCoPQ zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-79.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / SCoPQ zone 10\",GEOGCS[\"NAD27(C");
    add_srs_wkt (p, 1,
		 "GQ77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHERO");
    add_srs_wkt (p, 2,
		 "ID[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIME");
    add_srs_wkt (p, 4,
		 "M[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"d");
    add_srs_wkt (p, 5,
		 "egree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]");
    add_srs_wkt (p, 6,
		 "],AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORI");
    add_srs_wkt (p, 7,
		 "TY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-79.5],PARAMETER[\"scale_factor\",0.9999],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"false_easting\",304800],PARAMETER[\"false_nor");
    add_srs_wkt (p, 11,
		 "thing\",0],AUTHORITY[\"EPSG\",\"2016\"],AXIS[\"X\",EAST]");
    add_srs_wkt (p, 12, ",AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2017, "epsg", 2017,
		      "NAD27(76) / MTM zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-73.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 8\",GEOGCS[\"NAD27(76)\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clark");
    add_srs_wkt (p, 2,
		 "e 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 7,
		 ",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETE");
    add_srs_wkt (p, 8,
		 "R[\"latitude_of_origin\",0],PARAMETER[\"central_meridian");
    add_srs_wkt (p, 9,
		 "\",-73.5],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2017\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2018, "epsg", 2018,
		      "NAD27(76) / MTM zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-76.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 9\",GEOGCS[\"NAD27(76)\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clark");
    add_srs_wkt (p, 2,
		 "e 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 7,
		 ",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETE");
    add_srs_wkt (p, 8,
		 "R[\"latitude_of_origin\",0],PARAMETER[\"central_meridian");
    add_srs_wkt (p, 9,
		 "\",-76.5],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2018\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2019, "epsg", 2019,
		      "NAD27(76) / MTM zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-79.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 10\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-79.5],PARAMETER[\"scale_factor\",0.9999],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",304800],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2019\"],AXIS[\"X\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Y\",NORTH]]");
    p = add_epsg_def (first, last, 2020, "epsg", 2020,
		      "NAD27(76) / MTM zone 11");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-82.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 11\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-82.5],PARAMETER[\"scale_factor\",0.9999],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",304800],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2020\"],AXIS[\"Easting\",EAST],A");
    add_srs_wkt (p, 12, "XIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2021, "epsg", 2021,
		      "NAD27(76) / MTM zone 12");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-81 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 12\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-81],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2021\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2022, "epsg", 2022,
		      "NAD27(76) / MTM zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-84 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 13\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-84],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2022\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2023, "epsg", 2023,
		      "NAD27(76) / MTM zone 14");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-87 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 14\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-87],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2023\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2024, "epsg", 2024,
		      "NAD27(76) / MTM zone 15");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-90 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 15\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-90],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2024\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2025, "epsg", 2025,
		      "NAD27(76) / MTM zone 16");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-93 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 16\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-93],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2025\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2026, "epsg", 2026,
		      "NAD27(76) / MTM zone 17");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-96 +k=0.9999 +x_0=304800 +y");
    add_proj4text (p, 1, "_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / MTM zone 17\",GEOGCS[\"NAD27(76)\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Clar");
    add_srs_wkt (p, 2,
		 "ke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-96],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",304800],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2026\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2027, "epsg", 2027,
		      "NAD27(76) / UTM zone 15N");
    add_proj4text (p, 0, "+proj=utm +zone=15 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / UTM zone 15N\",GEOGCS[\"NAD27(76)\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-93],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2027\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2028, "epsg", 2028,
		      "NAD27(76) / UTM zone 16N");
    add_proj4text (p, 0, "+proj=utm +zone=16 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / UTM zone 16N\",GEOGCS[\"NAD27(76)\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-87],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2028\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2029, "epsg", 2029,
		      "NAD27(76) / UTM zone 17N");
    add_proj4text (p, 0, "+proj=utm +zone=17 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / UTM zone 17N\",GEOGCS[\"NAD27(76)\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-81],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2029\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2030, "epsg", 2030,
		      "NAD27(76) / UTM zone 18N");
    add_proj4text (p, 0, "+proj=utm +zone=18 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(76) / UTM zone 18N\",GEOGCS[\"NAD27(76)\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1927_1976\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7008\"]],AUTHORITY[\"EPSG\",\"6608\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4608\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-75],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2030\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2031, "epsg", 2031,
		      "NAD27(CGQ77) / UTM zone 17N");
    add_proj4text (p, 0, "+proj=utm +zone=17 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / UTM zone 17N\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-81],PARAMETER[\"scale_factor\",0.9996],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",500000],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",0],AUTHORITY[\"EPSG\",\"2031\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2032, "epsg", 2032,
		      "NAD27(CGQ77) / UTM zone 18N");
    add_proj4text (p, 0, "+proj=utm +zone=18 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / UTM zone 18N\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-75],PARAMETER[\"scale_factor\",0.9996],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",500000],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",0],AUTHORITY[\"EPSG\",\"2032\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2033, "epsg", 2033,
		      "NAD27(CGQ77) / UTM zone 19N");
    add_proj4text (p, 0, "+proj=utm +zone=19 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / UTM zone 19N\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-69],PARAMETER[\"scale_factor\",0.9996],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",500000],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",0],AUTHORITY[\"EPSG\",\"2033\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2034, "epsg", 2034,
		      "NAD27(CGQ77) / UTM zone 20N");
    add_proj4text (p, 0, "+proj=utm +zone=20 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / UTM zone 20N\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-63],PARAMETER[\"scale_factor\",0.9996],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",500000],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",0],AUTHORITY[\"EPSG\",\"2034\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2035, "epsg", 2035,
		      "NAD27(CGQ77) / UTM zone 21N");
    add_proj4text (p, 0, "+proj=utm +zone=21 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / UTM zone 21N\",GEOGCS[\"NAD27(CG");
    add_srs_wkt (p, 1,
		 "Q77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIMEM");
    add_srs_wkt (p, 4,
		 "[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"de");
    add_srs_wkt (p, 5,
		 "gree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]]");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",-57],PARAMETER[\"scale_factor\",0.9996],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",500000],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",0],AUTHORITY[\"EPSG\",\"2035\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2036, "epsg", 2036,
		      "NAD83(CSRS98) / New Brunswick Stereo (deprecated)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=46.5 +lon_0=-66.5 +k=0.999912 +x_0=2");
    add_proj4text (p, 1,
		   "500000 +y_0=7500000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / New Brunswick Stereo (deprecate");
    add_srs_wkt (p, 1,
		 "d)\",GEOGCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spa");
    add_srs_wkt (p, 2,
		 "tial_Reference_System\",SPHEROID[\"GRS 1980\",6378137,29");
    add_srs_wkt (p, 3,
		 "8.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,");
    add_srs_wkt (p, 4,
		 "0,0,0,0],AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwic");
    add_srs_wkt (p, 5,
		 "h\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01");
    add_srs_wkt (p, 6,
		 "74532925199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 8,
		 "9001\"]],PROJECTION[\"Oblique_Stereographic\"],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"latitude_of_origin\",46.5],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 10,
		 "an\",-66.5],PARAMETER[\"scale_factor\",0.999912],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_easting\",2500000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 12,
		 "\",7500000],AUTHORITY[\"EPSG\",\"2036\"],AXIS[\"Northing");
    add_srs_wkt (p, 13, "\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2037, "epsg", 2037,
		      "NAD83(CSRS98) / UTM zone 19N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=19 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 19N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-69],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2037\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2038, "epsg", 2038,
		      "NAD83(CSRS98) / UTM zone 20N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=20 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 20N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-63],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2038\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2039, "epsg", 2039,
		      "Israel / Israeli TM Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31.73439361111111 +lon_0=35.204516944");
    add_proj4text (p, 1,
		   "44445 +k=1.0000067 +x_0=219529.584 +y_0=626907.39 +ellps");
    add_proj4text (p, 2, "=GRS80 +towgs84=-48,55,52,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Israel / Israeli TM Grid\",GEOGCS[\"Israel\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"Israel\",SPHEROID[\"GRS 1980\",6378137,298.2572221");
    add_srs_wkt (p, 2,
		 "01,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[-48,55,52,0,0,0");
    add_srs_wkt (p, 3,
		 ",0],AUTHORITY[\"EPSG\",\"6141\"]],PRIMEM[\"Greenwich\",0");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532");
    add_srs_wkt (p, 5,
		 "9251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 6,
		 "G\",\"4141\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"900");
    add_srs_wkt (p, 7,
		 "1\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"la");
    add_srs_wkt (p, 8,
		 "titude_of_origin\",31.73439361111111],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",35.20451694444445],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",1.0000067],PARAMETER[\"false_easting\",219529.584],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",626907.39],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12,
		 ",\"2039\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORT");
    add_srs_wkt (p, 13, "H]]");
    p = add_epsg_def (first, last, 2040, "epsg", 2040,
		      "Locodjo 1965 / UTM zone 30N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=30 +ellps=clrk80 +towgs84=-125,53,467,0,");
    add_proj4text (p, 1, "0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Locodjo 1965 / UTM zone 30N\",GEOGCS[\"Locodjo ");
    add_srs_wkt (p, 1,
		 "1965\",DATUM[\"Locodjo_1965\",SPHEROID[\"Clarke 1880 (RG");
    add_srs_wkt (p, 2,
		 "S)\",6378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[-125,53,467,0,0,0,0],AUTHORITY[\"EPSG\",\"6142\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4142\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-3],PARAMETER[\"scale_factor\",0.99");
    add_srs_wkt (p, 10,
		 "96],PARAMETER[\"false_easting\",500000],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2040\"],AXIS[\"East");
    add_srs_wkt (p, 12, "ing\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2041, "epsg", 2041,
		      "Abidjan 1987 / UTM zone 30N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=30 +ellps=clrk80 +towgs84=-124.76,53,466");
    add_proj4text (p, 1, ".79,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Abidjan 1987 / UTM zone 30N\",GEOGCS[\"Abidjan ");
    add_srs_wkt (p, 1,
		 "1987\",DATUM[\"Abidjan_1987\",SPHEROID[\"Clarke 1880 (RG");
    add_srs_wkt (p, 2,
		 "S)\",6378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[-124.76,53,466.79,0,0,0,0],AUTHORITY[\"EPSG\",\"6");
    add_srs_wkt (p, 4,
		 "143\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901");
    add_srs_wkt (p, 5,
		 "\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPS");
    add_srs_wkt (p, 6,
		 "G\",\"9122\"]],AUTHORITY[\"EPSG\",\"4143\"]],UNIT[\"metr");
    add_srs_wkt (p, 7,
		 "e\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transve");
    add_srs_wkt (p, 8,
		 "rse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARA");
    add_srs_wkt (p, 9,
		 "METER[\"central_meridian\",-3],PARAMETER[\"scale_factor\"");
    add_srs_wkt (p, 10,
		 ",0.9996],PARAMETER[\"false_easting\",500000],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_northing\",0],AUTHORITY[\"EPSG\",\"2041\"],AXIS[\"");
    add_srs_wkt (p, 12, "Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2042, "epsg", 2042,
		      "Locodjo 1965 / UTM zone 29N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=29 +ellps=clrk80 +towgs84=-125,53,467,0,");
    add_proj4text (p, 1, "0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Locodjo 1965 / UTM zone 29N\",GEOGCS[\"Locodjo ");
    add_srs_wkt (p, 1,
		 "1965\",DATUM[\"Locodjo_1965\",SPHEROID[\"Clarke 1880 (RG");
    add_srs_wkt (p, 2,
		 "S)\",6378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[-125,53,467,0,0,0,0],AUTHORITY[\"EPSG\",\"6142\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4142\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-9],PARAMETER[\"scale_factor\",0.99");
    add_srs_wkt (p, 10,
		 "96],PARAMETER[\"false_easting\",500000],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2042\"],AXIS[\"East");
    add_srs_wkt (p, 12, "ing\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2043, "epsg", 2043,
		      "Abidjan 1987 / UTM zone 29N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=29 +ellps=clrk80 +towgs84=-124.76,53,466");
    add_proj4text (p, 1, ".79,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Abidjan 1987 / UTM zone 29N\",GEOGCS[\"Abidjan ");
    add_srs_wkt (p, 1,
		 "1987\",DATUM[\"Abidjan_1987\",SPHEROID[\"Clarke 1880 (RG");
    add_srs_wkt (p, 2,
		 "S)\",6378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[-124.76,53,466.79,0,0,0,0],AUTHORITY[\"EPSG\",\"6");
    add_srs_wkt (p, 4,
		 "143\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901");
    add_srs_wkt (p, 5,
		 "\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPS");
    add_srs_wkt (p, 6,
		 "G\",\"9122\"]],AUTHORITY[\"EPSG\",\"4143\"]],UNIT[\"metr");
    add_srs_wkt (p, 7,
		 "e\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transve");
    add_srs_wkt (p, 8,
		 "rse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARA");
    add_srs_wkt (p, 9,
		 "METER[\"central_meridian\",-9],PARAMETER[\"scale_factor\"");
    add_srs_wkt (p, 10,
		 ",0.9996],PARAMETER[\"false_easting\",500000],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_northing\",0],AUTHORITY[\"EPSG\",\"2043\"],AXIS[\"");
    add_srs_wkt (p, 12, "Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2044, "epsg", 2044,
		      "Hanoi 1972 / Gauss-Kruger zone 18");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=18500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=-17.51,-108.32,-62.39,0,0,0,0 +u");
    add_proj4text (p, 2, "nits=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hanoi 1972 / Gauss-Kruger zone 18\",GEOGCS[\"Ha");
    add_srs_wkt (p, 1,
		 "noi 1972\",DATUM[\"Hanoi_1972\",SPHEROID[\"Krassowsky 19");
    add_srs_wkt (p, 2,
		 "40\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TOWGS84");
    add_srs_wkt (p, 3,
		 "[-17.51,-108.32,-62.39,0,0,0,0],AUTHORITY[\"EPSG\",\"614");
    add_srs_wkt (p, 4,
		 "7\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4147\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"central_meridian\",105],PARAMETER[\"scale_factor\",");
    add_srs_wkt (p, 10,
		 "1],PARAMETER[\"false_easting\",18500000],PARAMETER[\"fal");
    add_srs_wkt (p, 11,
		 "se_northing\",0],AUTHORITY[\"EPSG\",\"2044\"],AXIS[\"X\"");
    add_srs_wkt (p, 12, ",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2045, "epsg", 2045,
		      "Hanoi 1972 / Gauss-Kruger zone 19");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=19500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=-17.51,-108.32,-62.39,0,0,0,0 +u");
    add_proj4text (p, 2, "nits=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hanoi 1972 / Gauss-Kruger zone 19\",GEOGCS[\"Ha");
    add_srs_wkt (p, 1,
		 "noi 1972\",DATUM[\"Hanoi_1972\",SPHEROID[\"Krassowsky 19");
    add_srs_wkt (p, 2,
		 "40\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TOWGS84");
    add_srs_wkt (p, 3,
		 "[-17.51,-108.32,-62.39,0,0,0,0],AUTHORITY[\"EPSG\",\"614");
    add_srs_wkt (p, 4,
		 "7\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4147\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"central_meridian\",111],PARAMETER[\"scale_factor\",");
    add_srs_wkt (p, 10,
		 "1],PARAMETER[\"false_easting\",19500000],PARAMETER[\"fal");
    add_srs_wkt (p, 11,
		 "se_northing\",0],AUTHORITY[\"EPSG\",\"2045\"],AXIS[\"X\"");
    add_srs_wkt (p, 12, ",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2056, "epsg", 2056, "CH1903+ / LV95");
    add_proj4text (p, 0,
		   "+proj=somerc +lat_0=46.95240555555556 +lon_0=7.439583333");
    add_proj4text (p, 1,
		   "333333 +k_0=1 +x_0=2600000 +y_0=1200000 +ellps=bessel +t");
    add_proj4text (p, 2,
		   "owgs84=674.374,15.056,405.346,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"CH1903+ / LV95\",GEOGCS[\"CH1903+\",DATUM[\"CH1");
    add_srs_wkt (p, 1,
		 "903\",SPHEROID[\"Bessel 1841\",6377397.155,299.1528128,A");
    add_srs_wkt (p, 2,
		 "UTHORITY[\"EPSG\",\"7004\"]],TOWGS84[674.374,15.056,405.");
    add_srs_wkt (p, 3,
		 "346,0,0,0,0],AUTHORITY[\"EPSG\",\"6150\"]],PRIMEM[\"Gree");
    add_srs_wkt (p, 4,
		 "nwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",");
    add_srs_wkt (p, 5,
		 "0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHOR");
    add_srs_wkt (p, 6,
		 "ITY[\"EPSG\",\"4150\"]],UNIT[\"metre\",1,AUTHORITY[\"EPS");
    add_srs_wkt (p, 7,
		 "G\",\"9001\"]],PROJECTION[\"Hotine_Oblique_Mercator\"],P");
    add_srs_wkt (p, 8,
		 "ARAMETER[\"latitude_of_center\",46.95240555555556],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"longitude_of_center\",7.439583333333333],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"azimuth\",90],PARAMETER[\"rectified_grid_angle\",90]");
    add_srs_wkt (p, 11,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 12,
		 "\",2600000],PARAMETER[\"false_northing\",1200000],AUTHOR");
    add_srs_wkt (p, 13,
		 "ITY[\"EPSG\",\"2056\"],AXIS[\"Y\",EAST],AXIS[\"X\",NORTH");
    add_srs_wkt (p, 14, "]]");
    p = add_epsg_def (first, last, 2057, "epsg", 2057,
		      "Rassadiran / Nakhl e Taqi");
    add_proj4text (p, 0,
		   "+proj=omerc +lat_0=27.51882880555555 +lonc=52.6035391666");
    add_proj4text (p, 1,
		   "6667 +alpha=0.5716611944444444 +k=0.999895934 +x_0=65837");
    add_proj4text (p, 2,
		   "7.437 +y_0=3044969.194 +ellps=intl +towgs84=-133.63,-157");
    add_proj4text (p, 3, ".5,-158.62,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Rassadiran / Nakhl e Taqi\",GEOGCS[\"Rassadiran");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"Rassadiran\",SPHEROID[\"International 1924\",");
    add_srs_wkt (p, 2,
		 "6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],TOWGS84[-133.6");
    add_srs_wkt (p, 3,
		 "3,-157.5,-158.62,0,0,0,0],AUTHORITY[\"EPSG\",\"6153\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4153\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Hotine_Oblique_");
    add_srs_wkt (p, 8,
		 "Mercator\"],PARAMETER[\"latitude_of_center\",27.51882880");
    add_srs_wkt (p, 9,
		 "555555],PARAMETER[\"longitude_of_center\",52.60353916666");
    add_srs_wkt (p, 10,
		 "667],PARAMETER[\"azimuth\",0.5716611944444444],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"rectified_grid_angle\",0.5716611944444444],PARAMETER[");
    add_srs_wkt (p, 12,
		 "\"scale_factor\",0.999895934],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 13,
		 ",658377.437],PARAMETER[\"false_northing\",3044969.194],A");
    add_srs_wkt (p, 14,
		 "UTHORITY[\"EPSG\",\"2057\"],AXIS[\"Easting\",EAST],AXIS[");
    add_srs_wkt (p, 15, "\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2058, "epsg", 2058,
		      "ED50(ED77) / UTM zone 38N");
    add_proj4text (p, 0, "+proj=utm +zone=38 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50(ED77) / UTM zone 38N\",GEOGCS[\"ED50(ED77)");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Datum_1950_1977\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6154\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4154\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",45],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",0.9996],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11,
		 "EPSG\",\"2058\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2059, "epsg", 2059,
		      "ED50(ED77) / UTM zone 39N");
    add_proj4text (p, 0, "+proj=utm +zone=39 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50(ED77) / UTM zone 39N\",GEOGCS[\"ED50(ED77)");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Datum_1950_1977\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6154\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4154\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",51],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",0.9996],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11,
		 "EPSG\",\"2059\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2060, "epsg", 2060,
		      "ED50(ED77) / UTM zone 40N");
    add_proj4text (p, 0, "+proj=utm +zone=40 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50(ED77) / UTM zone 40N\",GEOGCS[\"ED50(ED77)");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Datum_1950_1977\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6154\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4154\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",57],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",0.9996],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11,
		 "EPSG\",\"2060\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2061, "epsg", 2061,
		      "ED50(ED77) / UTM zone 41N");
    add_proj4text (p, 0, "+proj=utm +zone=41 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50(ED77) / UTM zone 41N\",GEOGCS[\"ED50(ED77)");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Datum_1950_1977\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6154\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4154\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",63],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",0.9996],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11,
		 "EPSG\",\"2061\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2062, "epsg", 2062,
		      "Madrid 1870 (Madrid) / Spain");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=40 +lat_0=40 +lon_0=0 +k_0=0.9988085293");
    add_proj4text (p, 1,
		   " +x_0=600000 +y_0=600000 +a=6378298.3 +b=6356657.1426695");
    add_proj4text (p, 2, "61 +pm=madrid +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Madrid 1870 (Madrid) / Spain\",GEOGCS[\"Madrid ");
    add_srs_wkt (p, 1,
		 "1870 (Madrid)\",DATUM[\"Madrid_1870_Madrid\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Struve 1860\",6378298.3,294.73,AUTHORITY[\"EPSG\",\"7028");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6903\"]],PRIMEM[\"Madrid\",-3.");
    add_srs_wkt (p, 4,
		 "687938888888889,AUTHORITY[\"EPSG\",\"8905\"]],UNIT[\"deg");
    add_srs_wkt (p, 5,
		 "ree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],");
    add_srs_wkt (p, 6,
		 "AUTHORITY[\"EPSG\",\"4903\"]],UNIT[\"metre\",1,AUTHORITY");
    add_srs_wkt (p, 7,
		 "[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert_Conformal_Coni");
    add_srs_wkt (p, 8,
		 "c_1SP\"],PARAMETER[\"latitude_of_origin\",40],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",0],PARAMETER[\"scale_factor\",0.998");
    add_srs_wkt (p, 10,
		 "8085293],PARAMETER[\"false_easting\",600000],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_northing\",600000],AUTHORITY[\"EPSG\",\"2062\"],AX");
    add_srs_wkt (p, 12, "IS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2063, "epsg", 2063,
		      "Dabola 1981 / UTM zone 28N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=28 +a=6378249.2 +b=6356515 +towgs84=-23,");
    add_proj4text (p, 1, "259,-9,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Dabola 1981 / UTM zone 28N (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"Conakry 1905\",DATUM[\"Conakry_1905\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1880 (IGN)\",6378249.2,293.4660212936265,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7011\"]],TOWGS84[-23,259,-9,0,0,0,0],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6315\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4315\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",-15],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",0.9996],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12,
		 "2063\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2064, "epsg", 2064,
		      "Dabola 1981 / UTM zone 29N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=29 +a=6378249.2 +b=6356515 +towgs84=-23,");
    add_proj4text (p, 1, "259,-9,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Dabola 1981 / UTM zone 29N (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"Conakry 1905\",DATUM[\"Conakry_1905\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1880 (IGN)\",6378249.2,293.4660212936265,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7011\"]],TOWGS84[-23,259,-9,0,0,0,0],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6315\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4315\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",-9],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",0.9996],PARAMETER[\"false_easting\",500000");
    add_srs_wkt (p, 11,
		 "],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 12,
		 "064\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2065, "epsg", 2065,
		      "S-JTSK (Ferro) / Krovak");
    add_proj4text (p, 0,
		   "+proj=krovak +lat_0=49.5 +lon_0=42.5 +alpha=30.288139722");
    add_proj4text (p, 1,
		   "22222 +k=0.9999 +x_0=0 +y_0=0 +ellps=bessel +pm=ferro +u");
    add_proj4text (p, 2, "nits=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"S-JTSK (Ferro) / Krovak\",GEOGCS[\"S-JTSK (Ferr");
    add_srs_wkt (p, 1,
		 "o)\",DATUM[\"S_JTSK_Ferro\",SPHEROID[\"Bessel 1841\",637");
    add_srs_wkt (p, 2,
		 "7397.155,299.1528128,AUTHORITY[\"EPSG\",\"7004\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6818\"]],PRIMEM[\"Ferro\",-17.6666666666");
    add_srs_wkt (p, 4,
		 "6667,AUTHORITY[\"EPSG\",\"8909\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4818\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Krovak\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "center\",49.5],PARAMETER[\"longitude_of_center\",42.5],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"azimuth\",30.28813972222222],PARAMETER[\"pseu");
    add_srs_wkt (p, 10,
		 "do_standard_parallel_1\",78.5],PARAMETER[\"scale_factor\"");
    add_srs_wkt (p, 11,
		 ",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[\"fals");
    add_srs_wkt (p, 12,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2065\"],AXIS[\"X\",");
    add_srs_wkt (p, 13, "SOUTH],AXIS[\"Y\",WEST]]");
    p = add_epsg_def (first, last, 2066, "epsg", 2066,
		      "Mount Dillon / Tobago Grid");
    add_proj4text (p, 0,
		   "+proj=cass +lat_0=11.25217861111111 +lon_0=-60.686008888");
    add_proj4text (p, 1,
		   "88889 +x_0=37718.66159325 +y_0=36209.91512952 +a=6378293");
    add_proj4text (p, 2,
		   ".645208759 +b=6356617.987679838 +to_meter=0.201166195164");
    add_proj4text (p, 3, " +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Mount Dillon / Tobago Grid\",GEOGCS[\"Mount Dil");
    add_srs_wkt (p, 1,
		 "lon\",DATUM[\"Mount_Dillon\",SPHEROID[\"Clarke 1858\",63");
    add_srs_wkt (p, 2,
		 "78293.645208759,294.2606763692569,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "007\"]],AUTHORITY[\"EPSG\",\"6157\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4157\"]],UNIT[\"Clarke's link\",0.201166195164,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9039\"]],PROJECTION[\"Cassini_Soldn");
    add_srs_wkt (p, 8,
		 "er\"],PARAMETER[\"latitude_of_origin\",11.25217861111111");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-60.68600888888889],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"false_easting\",187500],PARAMETER[\"false_north");
    add_srs_wkt (p, 11,
		 "ing\",180000],AUTHORITY[\"EPSG\",\"2066\"],AXIS[\"Eastin");
    add_srs_wkt (p, 12, "g\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2067, "epsg", 2067,
		      "Naparima 1955 / UTM zone 20N");
    add_proj4text (p, 0, "+proj=utm +zone=20 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Naparima 1955 / UTM zone 20N\",GEOGCS[\"Naparim");
    add_srs_wkt (p, 1,
		 "a 1955\",DATUM[\"Naparima_1955\",SPHEROID[\"Internationa");
    add_srs_wkt (p, 2,
		 "l 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6158\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"415");
    add_srs_wkt (p, 6,
		 "8\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",-63],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",0.9996],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 10,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 11,
		 "\",\"2067\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NO");
    add_srs_wkt (p, 12, "RTH]]");
    p = add_epsg_def (first, last, 2068, "epsg", 2068, "ELD79 / Libya zone 5");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=9 +k=0.9999 +x_0=200000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 5\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",9],PARAMETER[\"s");
    add_srs_wkt (p, 9,
		 "cale_factor\",0.9999],PARAMETER[\"false_easting\",200000");
    add_srs_wkt (p, 10,
		 "],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 11, "068\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2069, "epsg", 2069, "ELD79 / Libya zone 6");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=11 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 6\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",11],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2069\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2070, "epsg", 2070, "ELD79 / Libya zone 7");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=13 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 7\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",13],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2070\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2071, "epsg", 2071, "ELD79 / Libya zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 8\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",15],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2071\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2072, "epsg", 2072, "ELD79 / Libya zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=17 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 9\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",17],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2072\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2073, "epsg", 2073, "ELD79 / Libya zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=19 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 10\",GEOGCS[\"ELD79\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"European_Libyan_Datum_1979\",SPHEROID[\"International ");
    add_srs_wkt (p, 2,
		 "1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_or");
    add_srs_wkt (p, 8,
		 "igin\",0],PARAMETER[\"central_meridian\",19],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2073\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2074, "epsg", 2074, "ELD79 / Libya zone 11");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 11\",GEOGCS[\"ELD79\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"European_Libyan_Datum_1979\",SPHEROID[\"International ");
    add_srs_wkt (p, 2,
		 "1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_or");
    add_srs_wkt (p, 8,
		 "igin\",0],PARAMETER[\"central_meridian\",21],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2074\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2075, "epsg", 2075, "ELD79 / Libya zone 12");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=23 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 12\",GEOGCS[\"ELD79\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"European_Libyan_Datum_1979\",SPHEROID[\"International ");
    add_srs_wkt (p, 2,
		 "1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_or");
    add_srs_wkt (p, 8,
		 "igin\",0],PARAMETER[\"central_meridian\",23],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2075\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2076, "epsg", 2076, "ELD79 / Libya zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=25 +k=0.9999 +x_0=200000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / Libya zone 13\",GEOGCS[\"ELD79\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"European_Libyan_Datum_1979\",SPHEROID[\"International ");
    add_srs_wkt (p, 2,
		 "1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_or");
    add_srs_wkt (p, 8,
		 "igin\",0],PARAMETER[\"central_meridian\",25],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9999],PARAMETER[\"false_easting\",20000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2076\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2077, "epsg", 2077, "ELD79 / UTM zone 32N");
    add_proj4text (p, 0, "+proj=utm +zone=32 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / UTM zone 32N\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",9],PARAMETER[\"s");
    add_srs_wkt (p, 9,
		 "cale_factor\",0.9996],PARAMETER[\"false_easting\",500000");
    add_srs_wkt (p, 10,
		 "],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 11,
		 "077\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2078, "epsg", 2078, "ELD79 / UTM zone 33N");
    add_proj4text (p, 0, "+proj=utm +zone=33 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / UTM zone 33N\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",15],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9996],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11,
		 "2078\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2079, "epsg", 2079, "ELD79 / UTM zone 34N");
    add_proj4text (p, 0, "+proj=utm +zone=34 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / UTM zone 34N\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",21],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9996],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11,
		 "2079\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2080, "epsg", 2080, "ELD79 / UTM zone 35N");
    add_proj4text (p, 0, "+proj=utm +zone=35 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / UTM zone 35N\",GEOGCS[\"ELD79\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "European_Libyan_Datum_1979\",SPHEROID[\"International 19");
    add_srs_wkt (p, 2,
		 "24\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",27],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9996],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11,
		 "2080\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2081, "epsg", 2081,
		      "Chos Malal 1914 / Argentina zone 2");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-90 +lon_0=-69 +k=1 +x_0=2500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Chos Malal 1914 / Argentina zone 2\",GEOGCS[\"C");
    add_srs_wkt (p, 1,
		 "hos Malal 1914\",DATUM[\"Chos_Malal_1914\",SPHEROID[\"In");
    add_srs_wkt (p, 2,
		 "ternational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6160\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4160\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "01\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"l");
    add_srs_wkt (p, 8,
		 "atitude_of_origin\",-90],PARAMETER[\"central_meridian\",");
    add_srs_wkt (p, 9,
		 "-69],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_eas");
    add_srs_wkt (p, 10,
		 "ting\",2500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2081\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2082, "epsg", 2082,
		      "Pampa del Castillo / Argentina zone 2");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-90 +lon_0=-69 +k=1 +x_0=2500000 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=intl +towgs84=27.5,14,186.4,0,0,0,0 +units=m +");
    add_proj4text (p, 2, "no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pampa del Castillo / Argentina zone 2\",GEOGCS[");
    add_srs_wkt (p, 1,
		 "\"Pampa del Castillo\",DATUM[\"Pampa_del_Castillo\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"International 1924\",6378388,297,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7022\"]],TOWGS84[27.5,14,186.4,0,0,0,0],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6161\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4161\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",-90],PARAMETER[\"central_meridian\",-69],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",2500000],P");
    add_srs_wkt (p, 11,
		 "ARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2082");
    add_srs_wkt (p, 12, "\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2083, "epsg", 2083,
		      "Hito XVIII 1963 / Argentina zone 2");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-90 +lon_0=-69 +k=1 +x_0=2500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hito XVIII 1963 / Argentina zone 2\",GEOGCS[\"H");
    add_srs_wkt (p, 1,
		 "ito XVIII 1963\",DATUM[\"Hito_XVIII_1963\",SPHEROID[\"In");
    add_srs_wkt (p, 2,
		 "ternational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6254\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4254\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "01\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"l");
    add_srs_wkt (p, 8,
		 "atitude_of_origin\",-90],PARAMETER[\"central_meridian\",");
    add_srs_wkt (p, 9,
		 "-69],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_eas");
    add_srs_wkt (p, 10,
		 "ting\",2500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2083\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2084, "epsg", 2084,
		      "Hito XVIII 1963 / UTM zone 19S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=19 +south +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hito XVIII 1963 / UTM zone 19S\",GEOGCS[\"Hito ");
    add_srs_wkt (p, 1,
		 "XVIII 1963\",DATUM[\"Hito_XVIII_1963\",SPHEROID[\"Intern");
    add_srs_wkt (p, 2,
		 "ational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6254\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4254\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 8,
		 "ude_of_origin\",0],PARAMETER[\"central_meridian\",-69],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",10000000],AUT");
    add_srs_wkt (p, 11,
		 "HORITY[\"EPSG\",\"2084\"],AXIS[\"Easting\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2085, "epsg", 2085,
		      "NAD27 / Cuba Norte (deprecated)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=22.35 +lat_0=22.35 +lon_0=-81 +k_0=0.99");
    add_proj4text (p, 1,
		   "993602 +x_0=500000 +y_0=280296.016 +ellps=clrk66 +datum=");
    add_proj4text (p, 2, "NAD27 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27 / Cuba Norte (deprecated)\",GEOGCS[\"NAD2");
    add_srs_wkt (p, 1,
		 "7\",DATUM[\"North_American_Datum_1927\",SPHEROID[\"Clark");
    add_srs_wkt (p, 2,
		 "e 1866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7008\"]],AUTHORITY[\"EPSG\",\"6267\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4267\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 7,
		 ",\"9001\"]],PROJECTION[\"Lambert_Conformal_Conic_1SP\"],");
    add_srs_wkt (p, 8,
		 "PARAMETER[\"latitude_of_origin\",22.35],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",-81],PARAMETER[\"scale_factor\",0.9999360");
    add_srs_wkt (p, 10,
		 "2],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 11,
		 "_northing\",280296.016],AUTHORITY[\"EPSG\",\"2085\"],AXI");
    add_srs_wkt (p, 12, "S[\"Y\",NORTH],AXIS[\"X\",EAST]]");
    p = add_epsg_def (first, last, 2086, "epsg", 2086,
		      "NAD27 / Cuba Sur (deprecated)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=20.71666666666667 +lat_0=20.71666666666");
    add_proj4text (p, 1,
		   "667 +lon_0=-76.83333333333333 +k_0=0.99994848 +x_0=50000");
    add_proj4text (p, 2,
		   "0 +y_0=229126.939 +ellps=clrk66 +datum=NAD27 +units=m +n");
    add_proj4text (p, 3, "o_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27 / Cuba Sur (deprecated)\",GEOGCS[\"NAD27\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1927\",SPHEROID[\"Clarke 1");
    add_srs_wkt (p, 2,
		 "866\",6378206.4,294.9786982139006,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "008\"]],AUTHORITY[\"EPSG\",\"6267\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4267\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Lambert_Conformal_Conic_1SP\"],PAR");
    add_srs_wkt (p, 8,
		 "AMETER[\"latitude_of_origin\",20.71666666666667],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"central_meridian\",-76.83333333333333],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",0.99994848],PARAMETER[\"false_easting\",5");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",229126.939],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2086\"],AXIS[\"Y\",NORTH],AXIS[\"X\",EAST]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2087, "epsg", 2087, "ELD79 / TM 12 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=12 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ELD79 / TM 12 NE\",GEOGCS[\"ELD79\",DATUM[\"Eur");
    add_srs_wkt (p, 1,
		 "opean_Libyan_Datum_1979\",SPHEROID[\"International 1924\"");
    add_srs_wkt (p, 2,
		 ",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 3,
		 "PSG\",\"6159\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4159\"]],UN");
    add_srs_wkt (p, 6,
		 "IT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[");
    add_srs_wkt (p, 7,
		 "\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 8,
		 ",0],PARAMETER[\"central_meridian\",12],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",0.9996],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2087\"");
    add_srs_wkt (p, 11, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2088, "epsg", 2088, "Carthage / TM 11 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=11 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +a=6378249.2 +b=6356515 +datum=carthage +units=m +no");
    add_proj4text (p, 2, "_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Carthage / TM 11 NE\",GEOGCS[\"Carthage\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Carthage\",SPHEROID[\"Clarke 1880 (IGN)\",6378249.2,2");
    add_srs_wkt (p, 2,
		 "93.4660212936265,AUTHORITY[\"EPSG\",\"7011\"]],AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"6223\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4223\"]");
    add_srs_wkt (p, 6,
		 "],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECT");
    add_srs_wkt (p, 7,
		 "ION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_ori");
    add_srs_wkt (p, 8,
		 "gin\",0],PARAMETER[\"central_meridian\",11],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "scale_factor\",0.9996],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11,
		 "2088\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2089, "epsg", 2089,
		      "Yemen NGN96 / UTM zone 38N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=38 +ellps=WGS84 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Yemen NGN96 / UTM zone 38N\",GEOGCS[\"Yemen NGN");
    add_srs_wkt (p, 1,
		 "96\",DATUM[\"Yemen_National_Geodetic_Network_1996\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7030\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6163\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4163\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",45],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",0.9996],PARAMETER[\"false_easting\",500000],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2089\"],AXIS");
    add_srs_wkt (p, 12, "[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2090, "epsg", 2090,
		      "Yemen NGN96 / UTM zone 39N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=39 +ellps=WGS84 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Yemen NGN96 / UTM zone 39N\",GEOGCS[\"Yemen NGN");
    add_srs_wkt (p, 1,
		 "96\",DATUM[\"Yemen_National_Geodetic_Network_1996\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7030\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6163\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4163\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",51],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",0.9996],PARAMETER[\"false_easting\",500000],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2090\"],AXIS");
    add_srs_wkt (p, 12, "[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2091, "epsg", 2091,
		      "South Yemen / Gauss Kruger zone 8 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=8500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=-76,-138,67,0,0,0,0 +units=m +no_d");
    add_proj4text (p, 2, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"South Yemen / Gauss Kruger zone 8 (deprecated)\"");
    add_srs_wkt (p, 1,
		 ",GEOGCS[\"South Yemen\",DATUM[\"South_Yemen\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[-76,-138,67,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6164\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4164\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",45],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",1],PARAMETER[\"false_easting\",8500000],PARAMETER[\"f");
    add_srs_wkt (p, 11,
		 "alse_northing\",0],AUTHORITY[\"EPSG\",\"2091\"],AXIS[\"X");
    add_srs_wkt (p, 12, "\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2092, "epsg", 2092,
		      "South Yemen / Gauss Kruger zone 9 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=9500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=-76,-138,67,0,0,0,0 +units=m +no_d");
    add_proj4text (p, 2, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"South Yemen / Gauss Kruger zone 9 (deprecated)\"");
    add_srs_wkt (p, 1,
		 ",GEOGCS[\"South Yemen\",DATUM[\"South_Yemen\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[-76,-138,67,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6164\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4164\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",51],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",1],PARAMETER[\"false_easting\",9500000],PARAMETER[\"f");
    add_srs_wkt (p, 11,
		 "alse_northing\",0],AUTHORITY[\"EPSG\",\"2092\"],AXIS[\"X");
    add_srs_wkt (p, 12, "\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2093, "epsg", 2093,
		      "Hanoi 1972 / GK 106 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=106 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=-17.51,-108.32,-62.39,0,0,0,0 +uni");
    add_proj4text (p, 2, "ts=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hanoi 1972 / GK 106 NE\",GEOGCS[\"Hanoi 1972\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"Hanoi_1972\",SPHEROID[\"Krassowsky 1940\",637824");
    add_srs_wkt (p, 2,
		 "5,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TOWGS84[-17.51,-10");
    add_srs_wkt (p, 3,
		 "8.32,-62.39,0,0,0,0],AUTHORITY[\"EPSG\",\"6147\"]],PRIME");
    add_srs_wkt (p, 4,
		 "M[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"d");
    add_srs_wkt (p, 5,
		 "egree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]");
    add_srs_wkt (p, 6,
		 "],AUTHORITY[\"EPSG\",\"4147\"]],UNIT[\"metre\",1,AUTHORI");
    add_srs_wkt (p, 7,
		 "TY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"centra");
    add_srs_wkt (p, 9,
		 "l_meridian\",106],PARAMETER[\"scale_factor\",1],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_easting\",500000],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 11,
		 ",0],AUTHORITY[\"EPSG\",\"2093\"],AXIS[\"X\",NORTH],AXIS[");
    add_srs_wkt (p, 12, "\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2094, "epsg", 2094, "WGS 72BE / TM 106 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=106 +k=0.9996 +x_0=500000 +y");
    add_proj4text (p, 1,
		   "_0=0 +ellps=WGS72 +towgs84=0,0,1.9,0,0,0.814,-0.38 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"WGS 72BE / TM 106 NE\",GEOGCS[\"WGS 72BE\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"WGS_1972_Transit_Broadcast_Ephemeris\",SPHEROID[\"WG");
    add_srs_wkt (p, 2,
		 "S 72\",6378135,298.26,AUTHORITY[\"EPSG\",\"7043\"]],TOWG");
    add_srs_wkt (p, 3,
		 "S84[0,0,1.9,0,0,0.814,-0.38],AUTHORITY[\"EPSG\",\"6324\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4324\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",106],PARAMETER[\"scale_factor\",0.9");
    add_srs_wkt (p, 10,
		 "996],PARAMETER[\"false_easting\",500000],PARAMETER[\"fal");
    add_srs_wkt (p, 11,
		 "se_northing\",0],AUTHORITY[\"EPSG\",\"2094\"],AXIS[\"Eas");
    add_srs_wkt (p, 12, "ting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2095, "epsg", 2095, "Bissau / UTM zone 28N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=28 +ellps=intl +towgs84=-173,253,27,0,0,");
    add_proj4text (p, 1, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Bissau / UTM zone 28N\",GEOGCS[\"Bissau\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Bissau\",SPHEROID[\"International 1924\",6378388,297,");
    add_srs_wkt (p, 2,
		 "AUTHORITY[\"EPSG\",\"7022\"]],TOWGS84[-173,253,27,0,0,0,");
    add_srs_wkt (p, 3,
		 "0],AUTHORITY[\"EPSG\",\"6165\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4165\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",0],PARAMETER[\"central_meridian\",-15]");
    add_srs_wkt (p, 9,
		 ",PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_ea");
    add_srs_wkt (p, 10,
		 "sting\",500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2095\"],AXIS[\"Easting\",EAST],AXIS[\"North");
    add_srs_wkt (p, 12, "ing\",NORTH]]");
    p = add_epsg_def (first, last, 2096, "epsg", 2096,
		      "Korean 1985 / Korea East Belt");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=38 +lon_0=129 +k=1 +x_0=200000 +y_0=5");
    add_proj4text (p, 1, "00000 +ellps=bessel +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Korean 1985 / Korea East Belt\",GEOGCS[\"Korean");
    add_srs_wkt (p, 1,
		 " 1985\",DATUM[\"Korean_Datum_1985\",SPHEROID[\"Bessel 18");
    add_srs_wkt (p, 2,
		 "41\",6377397.155,299.1528128,AUTHORITY[\"EPSG\",\"7004\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6162\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4162\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",38],PARAMETER[\"central_meridian\",129");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",200000],PARAMETER[\"false_northing\",500000],AUTHORI");
    add_srs_wkt (p, 11,
		 "TY[\"EPSG\",\"2096\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]");
    add_srs_wkt (p, 12, "]");
    p = add_epsg_def (first, last, 2097, "epsg", 2097,
		      "Korean 1985 / Korea Central Belt");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=38 +lon_0=127 +k=1 +x_0=200000 +y_0=5");
    add_proj4text (p, 1, "00000 +ellps=bessel +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Korean 1985 / Korea Central Belt\",GEOGCS[\"Kor");
    add_srs_wkt (p, 1,
		 "ean 1985\",DATUM[\"Korean_Datum_1985\",SPHEROID[\"Bessel");
    add_srs_wkt (p, 2,
		 " 1841\",6377397.155,299.1528128,AUTHORITY[\"EPSG\",\"700");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6162\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4162\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",38],PARAMETER[\"central_meridian\",");
    add_srs_wkt (p, 9,
		 "127],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_eas");
    add_srs_wkt (p, 10,
		 "ting\",200000],PARAMETER[\"false_northing\",500000],AUTH");
    add_srs_wkt (p, 11,
		 "ORITY[\"EPSG\",\"2097\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EA");
    add_srs_wkt (p, 12, "ST]]");
    p = add_epsg_def (first, last, 2098, "epsg", 2098,
		      "Korean 1985 / Korea West Belt");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=38 +lon_0=125 +k=1 +x_0=200000 +y_0=5");
    add_proj4text (p, 1, "00000 +ellps=bessel +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Korean 1985 / Korea West Belt\",GEOGCS[\"Korean");
    add_srs_wkt (p, 1,
		 " 1985\",DATUM[\"Korean_Datum_1985\",SPHEROID[\"Bessel 18");
    add_srs_wkt (p, 2,
		 "41\",6377397.155,299.1528128,AUTHORITY[\"EPSG\",\"7004\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6162\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4162\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",38],PARAMETER[\"central_meridian\",125");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_eastin");
    add_srs_wkt (p, 10,
		 "g\",200000],PARAMETER[\"false_northing\",500000],AUTHORI");
    add_srs_wkt (p, 11,
		 "TY[\"EPSG\",\"2098\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]");
    add_srs_wkt (p, 12, "]");
    p = add_epsg_def (first, last, 2099, "epsg", 2099,
		      "Qatar 1948 / Qatar Grid");
    add_proj4text (p, 0,
		   "+proj=cass +lat_0=25.38236111111111 +lon_0=50.7613888888");
    add_proj4text (p, 1,
		   "8889 +x_0=100000 +y_0=100000 +ellps=helmert +units=m +no");
    add_proj4text (p, 2, "_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Qatar 1948 / Qatar Grid\",GEOGCS[\"Qatar 1948\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"Qatar_1948\",SPHEROID[\"Helmert 1906\",6378200,");
    add_srs_wkt (p, 2,
		 "298.3,AUTHORITY[\"EPSG\",\"7020\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "6286\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 4,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 5,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4286\"]],UNIT[\"met");
    add_srs_wkt (p, 6,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Cassin");
    add_srs_wkt (p, 7,
		 "i_Soldner\"],PARAMETER[\"latitude_of_origin\",25.3823611");
    add_srs_wkt (p, 8,
		 "1111111],PARAMETER[\"central_meridian\",50.7613888888888");
    add_srs_wkt (p, 9,
		 "9],PARAMETER[\"false_easting\",100000],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_northing\",100000],AUTHORITY[\"EPSG\",\"2099\"],AXIS[\"");
    add_srs_wkt (p, 11, "Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2100, "epsg", 2100, "GGRS87 / Greek Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=GRS80 +datum=GGRS87 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"GGRS87 / Greek Grid\",GEOGCS[\"GGRS87\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "Greek_Geodetic_Reference_System_1987\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",TOWGS84[-199.87,74.79,246.62,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6121\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4121\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"central_meridian\",24],PARAMETER[\"scale_fact");
    add_srs_wkt (p, 10,
		 "or\",0.9996],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2100\"],AX");
    add_srs_wkt (p, 12, "IS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2101, "epsg", 2101,
		      "Lake / Maracaibo Grid M1");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=10.16666666666667 +lat_0=10.16666666666");
    add_proj4text (p, 1,
		   "667 +lon_0=-71.60561777777777 +k_0=1 +x_0=0 +y_0=-52684.");
    add_proj4text (p, 2, "972 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Lake / Maracaibo Grid M1\",GEOGCS[\"Lake\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"Lake\",SPHEROID[\"International 1924\",6378388,297,A");
    add_srs_wkt (p, 2,
		 "UTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6249\"");
    add_srs_wkt (p, 3,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 4,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4249\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 6,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert_Confo");
    add_srs_wkt (p, 7,
		 "rmal_Conic_1SP\"],PARAMETER[\"latitude_of_origin\",10.16");
    add_srs_wkt (p, 8,
		 "666666666667],PARAMETER[\"central_meridian\",-71.6056177");
    add_srs_wkt (p, 9,
		 "7777777],PARAMETER[\"scale_factor\",1],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_easting\",0],PARAMETER[\"false_northing\",-52684.972],A");
    add_srs_wkt (p, 11,
		 "UTHORITY[\"EPSG\",\"2101\"],AXIS[\"X\",EAST],AXIS[\"Y\",");
    add_srs_wkt (p, 12, "NORTH]]");
    p = add_epsg_def (first, last, 2102, "epsg", 2102, "Lake / Maracaibo Grid");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=10.16666666666667 +lat_0=10.16666666666");
    add_proj4text (p, 1,
		   "667 +lon_0=-71.60561777777777 +k_0=1 +x_0=200000 +y_0=14");
    add_proj4text (p, 2, "7315.028 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Lake / Maracaibo Grid\",GEOGCS[\"Lake\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "Lake\",SPHEROID[\"International 1924\",6378388,297,AUTHO");
    add_srs_wkt (p, 2,
		 "RITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6249\"]],P");
    add_srs_wkt (p, 3,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 4,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 5,
		 "2\"]],AUTHORITY[\"EPSG\",\"4249\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert_Conforma");
    add_srs_wkt (p, 7,
		 "l_Conic_1SP\"],PARAMETER[\"latitude_of_origin\",10.16666");
    add_srs_wkt (p, 8,
		 "666666667],PARAMETER[\"central_meridian\",-71.6056177777");
    add_srs_wkt (p, 9,
		 "7777],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_ea");
    add_srs_wkt (p, 10,
		 "sting\",200000],PARAMETER[\"false_northing\",147315.028]");
    add_srs_wkt (p, 11,
		 ",AUTHORITY[\"EPSG\",\"2102\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",NORTH]]");
    p = add_epsg_def (first, last, 2103, "epsg", 2103,
		      "Lake / Maracaibo Grid M3");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=10.16666666666667 +lat_0=10.16666666666");
    add_proj4text (p, 1,
		   "667 +lon_0=-71.60561777777777 +k_0=1 +x_0=500000 +y_0=44");
    add_proj4text (p, 2, "7315.028 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Lake / Maracaibo Grid M3\",GEOGCS[\"Lake\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"Lake\",SPHEROID[\"International 1924\",6378388,297,A");
    add_srs_wkt (p, 2,
		 "UTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6249\"");
    add_srs_wkt (p, 3,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 4,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4249\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 6,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert_Confo");
    add_srs_wkt (p, 7,
		 "rmal_Conic_1SP\"],PARAMETER[\"latitude_of_origin\",10.16");
    add_srs_wkt (p, 8,
		 "666666666667],PARAMETER[\"central_meridian\",-71.6056177");
    add_srs_wkt (p, 9,
		 "7777777],PARAMETER[\"scale_factor\",1],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_easting\",500000],PARAMETER[\"false_northing\",447315.0");
    add_srs_wkt (p, 11,
		 "28],AUTHORITY[\"EPSG\",\"2103\"],AXIS[\"X\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Y\",NORTH]]");
    p = add_epsg_def (first, last, 2104, "epsg", 2104,
		      "Lake / Maracaibo La Rosa Grid");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=10.16666666666667 +lat_0=10.16666666666");
    add_proj4text (p, 1,
		   "667 +lon_0=-71.60561777777777 +k_0=1 +x_0=-17044 +y_0=-2");
    add_proj4text (p, 2, "3139.97 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Lake / Maracaibo La Rosa Grid\",GEOGCS[\"Lake\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"Lake\",SPHEROID[\"International 1924\",6378388,");
    add_srs_wkt (p, 2,
		 "297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6");
    add_srs_wkt (p, 3,
		 "249\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901");
    add_srs_wkt (p, 4,
		 "\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"9122\"]],AUTHORITY[\"EPSG\",\"4249\"]],UNIT[\"metr");
    add_srs_wkt (p, 6,
		 "e\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert");
    add_srs_wkt (p, 7,
		 "_Conformal_Conic_1SP\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 8,
		 ",10.16666666666667],PARAMETER[\"central_meridian\",-71.6");
    add_srs_wkt (p, 9,
		 "0561777777777],PARAMETER[\"scale_factor\",1],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",-17044],PARAMETER[\"false_northing\",-23");
    add_srs_wkt (p, 11,
		 "139.97],AUTHORITY[\"EPSG\",\"2104\"],AXIS[\"X\",EAST],AX");
    add_srs_wkt (p, 12, "IS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2105, "epsg", 2105,
		      "NZGD2000 / Mount Eden 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-36.87972222222222 +lon_0=174.7641666");
    add_proj4text (p, 1,
		   "666667 +k=0.9999 +x_0=400000 +y_0=800000 +ellps=GRS80 +t");
    add_proj4text (p, 2, "owgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Mount Eden 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-36.879722222");
    add_srs_wkt (p, 9,
		 "22222],PARAMETER[\"central_meridian\",174.7641666666667]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_ea");
    add_srs_wkt (p, 11,
		 "sting\",400000],PARAMETER[\"false_northing\",800000],AUT");
    add_srs_wkt (p, 12,
		 "HORITY[\"EPSG\",\"2105\"],AXIS[\"Northing\",NORTH],AXIS[");
    add_srs_wkt (p, 13, "\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2106, "epsg", 2106,
		      "NZGD2000 / Bay of Plenty 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-37.76111111111111 +lon_0=176.4661111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Bay of Plenty 2000\",GEOGCS[\"NZGD20");
    add_srs_wkt (p, 1,
		 "00\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"616");
    add_srs_wkt (p, 4,
		 "7\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",-37.761111");
    add_srs_wkt (p, 9,
		 "11111111],PARAMETER[\"central_meridian\",176.46611111111");
    add_srs_wkt (p, 10,
		 "11],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",400000],PARAMETER[\"false_northing\",800000],AUTHO");
    add_srs_wkt (p, 12,
		 "RITY[\"EPSG\",\"2106\"],AXIS[\"Northing\",NORTH],AXIS[\"");
    add_srs_wkt (p, 13, "Easting\",EAST]]");
    p = add_epsg_def (first, last, 2107, "epsg", 2107,
		      "NZGD2000 / Poverty Bay 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-38.62444444444444 +lon_0=177.8855555");
    add_proj4text (p, 1,
		   "555556 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Poverty Bay 2000\",GEOGCS[\"NZGD2000");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-38.624444444");
    add_srs_wkt (p, 9,
		 "44444],PARAMETER[\"central_meridian\",177.8855555555556]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2107\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2108, "epsg", 2108,
		      "NZGD2000 / Hawkes Bay 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-39.65083333333333 +lon_0=176.6736111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Hawkes Bay 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-39.650833333");
    add_srs_wkt (p, 9,
		 "33333],PARAMETER[\"central_meridian\",176.6736111111111]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2108\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2109, "epsg", 2109,
		      "NZGD2000 / Taranaki 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-39.13555555555556 +lon_0=174.2277777");
    add_proj4text (p, 1,
		   "777778 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Taranaki 2000\",GEOGCS[\"NZGD2000\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",-39.13555555555");
    add_srs_wkt (p, 9,
		 "556],PARAMETER[\"central_meridian\",174.2277777777778],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",400000],PARAMETER[\"false_northing\",800000],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2109\"],AXIS[\"Northing\",NORTH],AXIS[\"Easti");
    add_srs_wkt (p, 13, "ng\",EAST]]");
    p = add_epsg_def (first, last, 2110, "epsg", 2110,
		      "NZGD2000 / Tuhirangi 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-39.51222222222222 +lon_0=175.64 +k=1");
    add_proj4text (p, 1,
		   " +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0");
    add_proj4text (p, 2, ",0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Tuhirangi 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-39.512222222");
    add_srs_wkt (p, 9,
		 "22222],PARAMETER[\"central_meridian\",175.64],PARAMETER[");
    add_srs_wkt (p, 10,
		 "\"scale_factor\",1],PARAMETER[\"false_easting\",400000],");
    add_srs_wkt (p, 11,
		 "PARAMETER[\"false_northing\",800000],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12,
		 "\"2110\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2111, "epsg", 2111,
		      "NZGD2000 / Wanganui 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-40.24194444444444 +lon_0=175.4880555");
    add_proj4text (p, 1,
		   "555555 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Wanganui 2000\",GEOGCS[\"NZGD2000\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",-40.24194444444");
    add_srs_wkt (p, 9,
		 "444],PARAMETER[\"central_meridian\",175.4880555555555],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",400000],PARAMETER[\"false_northing\",800000],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2111\"],AXIS[\"Northing\",NORTH],AXIS[\"Easti");
    add_srs_wkt (p, 13, "ng\",EAST]]");
    p = add_epsg_def (first, last, 2112, "epsg", 2112,
		      "NZGD2000 / Wairarapa 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-40.92527777777777 +lon_0=175.6472222");
    add_proj4text (p, 1,
		   "222222 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Wairarapa 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-40.925277777");
    add_srs_wkt (p, 9,
		 "77777],PARAMETER[\"central_meridian\",175.6472222222222]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2112\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2113, "epsg", 2113,
		      "NZGD2000 / Wellington 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-41.3011111111111 +lon_0=174.77638888");
    add_proj4text (p, 1,
		   "88889 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84");
    add_proj4text (p, 2, "=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Wellington 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-41.301111111");
    add_srs_wkt (p, 9,
		 "1111],PARAMETER[\"central_meridian\",174.7763888888889],");
    add_srs_wkt (p, 10,
		 "PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",400000],PARAMETER[\"false_northing\",800000],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2113\"],AXIS[\"Northing\",NORTH],AXIS[\"Easti");
    add_srs_wkt (p, 13, "ng\",EAST]]");
    p = add_epsg_def (first, last, 2114, "epsg", 2114,
		      "NZGD2000 / Collingwood 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-40.71472222222223 +lon_0=172.6719444");
    add_proj4text (p, 1,
		   "444444 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Collingwood 2000\",GEOGCS[\"NZGD2000");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-40.714722222");
    add_srs_wkt (p, 9,
		 "22223],PARAMETER[\"central_meridian\",172.6719444444444]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2114\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2115, "epsg", 2115,
		      "NZGD2000 / Nelson 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-41.27444444444444 +lon_0=173.2991666");
    add_proj4text (p, 1,
		   "666667 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Nelson 2000\",GEOGCS[\"NZGD2000\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],P");
    add_srs_wkt (p, 4,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 5,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 6,
		 "2\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 7,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merca");
    add_srs_wkt (p, 8,
		 "tor\"],PARAMETER[\"latitude_of_origin\",-41.274444444444");
    add_srs_wkt (p, 9,
		 "44],PARAMETER[\"central_meridian\",173.2991666666667],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "400000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2115\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting");
    add_srs_wkt (p, 13, "\",EAST]]");
    p = add_epsg_def (first, last, 2116, "epsg", 2116,
		      "NZGD2000 / Karamea 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-41.28972222222222 +lon_0=172.1088888");
    add_proj4text (p, 1,
		   "888889 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Karamea 2000\",GEOGCS[\"NZGD2000\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",-41.28972222222");
    add_srs_wkt (p, 9,
		 "222],PARAMETER[\"central_meridian\",172.1088888888889],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",400000],PARAMETER[\"false_northing\",800000],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2116\"],AXIS[\"Northing\",NORTH],AXIS[\"Easti");
    add_srs_wkt (p, 13, "ng\",EAST]]");
    p = add_epsg_def (first, last, 2117, "epsg", 2117,
		      "NZGD2000 / Buller 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-41.81055555555555 +lon_0=171.5811111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Buller 2000\",GEOGCS[\"NZGD2000\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],P");
    add_srs_wkt (p, 4,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 5,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 6,
		 "2\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 7,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merca");
    add_srs_wkt (p, 8,
		 "tor\"],PARAMETER[\"latitude_of_origin\",-41.810555555555");
    add_srs_wkt (p, 9,
		 "55],PARAMETER[\"central_meridian\",171.5811111111111],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "400000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2117\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting");
    add_srs_wkt (p, 13, "\",EAST]]");
    p = add_epsg_def (first, last, 2118, "epsg", 2118, "NZGD2000 / Grey 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-42.33361111111111 +lon_0=171.5497222");
    add_proj4text (p, 1,
		   "222222 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Grey 2000\",GEOGCS[\"NZGD2000\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 198");
    add_srs_wkt (p, 2,
		 "0\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],");
    add_srs_wkt (p, 3,
		 "TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],PRI");
    add_srs_wkt (p, 4,
		 "MEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",-42.33361111111111]");
    add_srs_wkt (p, 9,
		 ",PARAMETER[\"central_meridian\",171.5497222222222],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",400");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"EP");
    add_srs_wkt (p, 12,
		 "SG\",\"2118\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\"");
    add_srs_wkt (p, 13, ",EAST]]");
    p = add_epsg_def (first, last, 2119, "epsg", 2119, "NZGD2000 / Amuri 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-42.68888888888888 +lon_0=173.01 +k=1");
    add_proj4text (p, 1,
		   " +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0");
    add_proj4text (p, 2, ",0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Amuri 2000\",GEOGCS[\"NZGD2000\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",-42.6888888888888");
    add_srs_wkt (p, 9,
		 "8],PARAMETER[\"central_meridian\",173.01],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",400000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",800000],AUTHORITY[\"EPSG\",\"21");
    add_srs_wkt (p, 12,
		 "19\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2120, "epsg", 2120,
		      "NZGD2000 / Marlborough 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-41.54444444444444 +lon_0=173.8019444");
    add_proj4text (p, 1,
		   "444444 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Marlborough 2000\",GEOGCS[\"NZGD2000");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-41.544444444");
    add_srs_wkt (p, 9,
		 "44444],PARAMETER[\"central_meridian\",173.8019444444444]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2120\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2121, "epsg", 2121,
		      "NZGD2000 / Hokitika 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-42.88611111111111 +lon_0=170.9797222");
    add_proj4text (p, 1,
		   "222222 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Hokitika 2000\",GEOGCS[\"NZGD2000\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",-42.88611111111");
    add_srs_wkt (p, 9,
		 "111],PARAMETER[\"central_meridian\",170.9797222222222],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",400000],PARAMETER[\"false_northing\",800000],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2121\"],AXIS[\"Northing\",NORTH],AXIS[\"Easti");
    add_srs_wkt (p, 13, "ng\",EAST]]");
    p = add_epsg_def (first, last, 2122, "epsg", 2122,
		      "NZGD2000 / Okarito 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-43.11 +lon_0=170.2608333333333 +k=1 ");
    add_proj4text (p, 1,
		   "+x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0,");
    add_proj4text (p, 2, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Okarito 2000\",GEOGCS[\"NZGD2000\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",-43.11],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"central_meridian\",170.2608333333333],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",1],PARAMETER[\"false_easting\",400000],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"false_northing\",800000],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 12,
		 "122\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2123, "epsg", 2123,
		      "NZGD2000 / Jacksons Bay 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-43.97777777777778 +lon_0=168.6061111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Jacksons Bay 2000\",GEOGCS[\"NZGD200");
    add_srs_wkt (p, 1,
		 "0\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-43.977777777");
    add_srs_wkt (p, 9,
		 "77778],PARAMETER[\"central_meridian\",168.6061111111111]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2123\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2124, "epsg", 2124,
		      "NZGD2000 / Mount Pleasant 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-43.59055555555556 +lon_0=172.7269444");
    add_proj4text (p, 1,
		   "444445 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Mount Pleasant 2000\",GEOGCS[\"NZGD2");
    add_srs_wkt (p, 1,
		 "000\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID");
    add_srs_wkt (p, 2,
		 "[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"616");
    add_srs_wkt (p, 4,
		 "7\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",-43.590555");
    add_srs_wkt (p, 9,
		 "55555556],PARAMETER[\"central_meridian\",172.72694444444");
    add_srs_wkt (p, 10,
		 "45],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",400000],PARAMETER[\"false_northing\",800000],AUTHO");
    add_srs_wkt (p, 12,
		 "RITY[\"EPSG\",\"2124\"],AXIS[\"Northing\",NORTH],AXIS[\"");
    add_srs_wkt (p, 13, "Easting\",EAST]]");
    p = add_epsg_def (first, last, 2125, "epsg", 2125,
		      "NZGD2000 / Gawler 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-43.74861111111111 +lon_0=171.3605555");
    add_proj4text (p, 1,
		   "555555 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Gawler 2000\",GEOGCS[\"NZGD2000\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],P");
    add_srs_wkt (p, 4,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 5,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 6,
		 "2\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 7,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merca");
    add_srs_wkt (p, 8,
		 "tor\"],PARAMETER[\"latitude_of_origin\",-43.748611111111");
    add_srs_wkt (p, 9,
		 "11],PARAMETER[\"central_meridian\",171.3605555555555],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "400000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2125\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting");
    add_srs_wkt (p, 13, "\",EAST]]");
    p = add_epsg_def (first, last, 2126, "epsg", 2126,
		      "NZGD2000 / Timaru 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-44.40194444444445 +lon_0=171.0572222");
    add_proj4text (p, 1,
		   "222222 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Timaru 2000\",GEOGCS[\"NZGD2000\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],P");
    add_srs_wkt (p, 4,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 5,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 6,
		 "2\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 7,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merca");
    add_srs_wkt (p, 8,
		 "tor\"],PARAMETER[\"latitude_of_origin\",-44.401944444444");
    add_srs_wkt (p, 9,
		 "45],PARAMETER[\"central_meridian\",171.0572222222222],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "400000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2126\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting");
    add_srs_wkt (p, 13, "\",EAST]]");
    p = add_epsg_def (first, last, 2127, "epsg", 2127,
		      "NZGD2000 / Lindis Peak 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-44.735 +lon_0=169.4675 +k=1 +x_0=400");
    add_proj4text (p, 1,
		   "000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +uni");
    add_proj4text (p, 2, "ts=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Lindis Peak 2000\",GEOGCS[\"NZGD2000");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-44.735],PARA");
    add_srs_wkt (p, 9,
		 "METER[\"central_meridian\",169.4675],PARAMETER[\"scale_f");
    add_srs_wkt (p, 10,
		 "actor\",1],PARAMETER[\"false_easting\",400000],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"false_northing\",800000],AUTHORITY[\"EPSG\",\"2127\"]");
    add_srs_wkt (p, 12, ",AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2128, "epsg", 2128,
		      "NZGD2000 / Mount Nicholas 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-45.13277777777778 +lon_0=168.3986111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Mount Nicholas 2000\",GEOGCS[\"NZGD2");
    add_srs_wkt (p, 1,
		 "000\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID");
    add_srs_wkt (p, 2,
		 "[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"616");
    add_srs_wkt (p, 4,
		 "7\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",-45.132777");
    add_srs_wkt (p, 9,
		 "77777778],PARAMETER[\"central_meridian\",168.39861111111");
    add_srs_wkt (p, 10,
		 "11],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",400000],PARAMETER[\"false_northing\",800000],AUTHO");
    add_srs_wkt (p, 12,
		 "RITY[\"EPSG\",\"2128\"],AXIS[\"Northing\",NORTH],AXIS[\"");
    add_srs_wkt (p, 13, "Easting\",EAST]]");
    p = add_epsg_def (first, last, 2129, "epsg", 2129,
		      "NZGD2000 / Mount York 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-45.56361111111111 +lon_0=167.7386111");
    add_proj4text (p, 1,
		   "111111 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Mount York 2000\",GEOGCS[\"NZGD2000\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-45.563611111");
    add_srs_wkt (p, 9,
		 "11111],PARAMETER[\"central_meridian\",167.7386111111111]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting");
    add_srs_wkt (p, 11,
		 "\",400000],PARAMETER[\"false_northing\",800000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2129\"],AXIS[\"Northing\",NORTH],AXIS[\"Eas");
    add_srs_wkt (p, 13, "ting\",EAST]]");
    p = add_epsg_def (first, last, 2130, "epsg", 2130,
		      "NZGD2000 / Observation Point 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-45.81611111111111 +lon_0=170.6283333");
    add_proj4text (p, 1,
		   "333333 +k=1 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs8");
    add_proj4text (p, 2, "4=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Observation Point 2000\",GEOGCS[\"NZ");
    add_srs_wkt (p, 1,
		 "GD2000\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6167\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",-45.81");
    add_srs_wkt (p, 9,
		 "611111111111],PARAMETER[\"central_meridian\",170.6283333");
    add_srs_wkt (p, 10,
		 "333333],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_");
    add_srs_wkt (p, 11,
		 "easting\",400000],PARAMETER[\"false_northing\",800000],A");
    add_srs_wkt (p, 12,
		 "UTHORITY[\"EPSG\",\"2130\"],AXIS[\"Northing\",NORTH],AXI");
    add_srs_wkt (p, 13, "S[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2131, "epsg", 2131,
		      "NZGD2000 / North Taieri 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-45.86138888888889 +lon_0=170.2825 +k");
    add_proj4text (p, 1,
		   "=0.99996 +x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84=0");
    add_proj4text (p, 2, ",0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / North Taieri 2000\",GEOGCS[\"NZGD200");
    add_srs_wkt (p, 1,
		 "0\",DATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",-45.861388888");
    add_srs_wkt (p, 9,
		 "88889],PARAMETER[\"central_meridian\",170.2825],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"scale_factor\",0.99996],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "400000],PARAMETER[\"false_northing\",800000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2131\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting");
    add_srs_wkt (p, 13, "\",EAST]]");
    p = add_epsg_def (first, last, 2132, "epsg", 2132, "NZGD2000 / Bluff 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=-46.6 +lon_0=168.3427777777778 +k=1 +");
    add_proj4text (p, 1,
		   "x_0=400000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0,0");
    add_proj4text (p, 2, ",0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / Bluff 2000\",GEOGCS[\"NZGD2000\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",-46.6],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",168.3427777777778],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",1],PARAMETER[\"false_easting\",400000],PARAME");
    add_srs_wkt (p, 11,
		 "TER[\"false_northing\",800000],AUTHORITY[\"EPSG\",\"2132");
    add_srs_wkt (p, 12, "\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2133, "epsg", 2133,
		      "NZGD2000 / UTM zone 58S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=58 +south +ellps=GRS80 +towgs84=0,0,0,0,");
    add_proj4text (p, 1, "0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / UTM zone 58S\",GEOGCS[\"NZGD2000\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "central_meridian\",165],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 11,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2133\"],AXIS[");
    add_srs_wkt (p, 12, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2134, "epsg", 2134,
		      "NZGD2000 / UTM zone 59S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=59 +south +ellps=GRS80 +towgs84=0,0,0,0,");
    add_proj4text (p, 1, "0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / UTM zone 59S\",GEOGCS[\"NZGD2000\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "central_meridian\",171],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 11,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2134\"],AXIS[");
    add_srs_wkt (p, 12, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2135, "epsg", 2135,
		      "NZGD2000 / UTM zone 60S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=60 +south +ellps=GRS80 +towgs84=0,0,0,0,");
    add_proj4text (p, 1, "0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / UTM zone 60S\",GEOGCS[\"NZGD2000\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"New_Zealand_Geodetic_Datum_2000\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6167\"]],");
    add_srs_wkt (p, 4,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 5,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 6,
		 "22\"]],AUTHORITY[\"EPSG\",\"4167\"]],UNIT[\"metre\",1,AU");
    add_srs_wkt (p, 7,
		 "THORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merc");
    add_srs_wkt (p, 8,
		 "ator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "central_meridian\",177],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 11,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2135\"],AXIS[");
    add_srs_wkt (p, 12, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2136, "epsg", 2136,
		      "Accra / Ghana National Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=4.666666666666667 +lon_0=-1 +k=0.9997");
    add_proj4text (p, 1,
		   "5 +x_0=274319.7391633579 +y_0=0 +a=6378300 +b=6356751.68");
    add_proj4text (p, 2, "9189189 +to_meter=0.3047997101815088 +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Accra / Ghana National Grid\",GEOGCS[\"Accra\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"Accra\",SPHEROID[\"War Office\",6378300,296,AUTH");
    add_srs_wkt (p, 2,
		 "ORITY[\"EPSG\",\"7029\"]],AUTHORITY[\"EPSG\",\"6168\"]],");
    add_srs_wkt (p, 3,
		 "PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNI");
    add_srs_wkt (p, 4,
		 "T[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"91");
    add_srs_wkt (p, 5,
		 "22\"]],AUTHORITY[\"EPSG\",\"4168\"]],UNIT[\"Gold Coast f");
    add_srs_wkt (p, 6,
		 "oot\",0.3047997101815088,AUTHORITY[\"EPSG\",\"9094\"]],P");
    add_srs_wkt (p, 7,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 8,
		 "of_origin\",4.666666666666667],PARAMETER[\"central_merid");
    add_srs_wkt (p, 9,
		 "ian\",-1],PARAMETER[\"scale_factor\",0.99975],PARAMETER[");
    add_srs_wkt (p, 10,
		 "\"false_easting\",900000],PARAMETER[\"false_northing\",0");
    add_srs_wkt (p, 11,
		 "],AUTHORITY[\"EPSG\",\"2136\"],AXIS[\"Easting\",EAST],AX");
    add_srs_wkt (p, 12, "IS[\"Northing\",NORTH]]");
#endif /* full EPSG initialization enabled/disabled */
    if (first == last)
	last = first;		/* suppressing stupid compiler warnings */
}

static void
initialize_epsg_01 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2137, "epsg", 2137, "Accra / TM 1 NW");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-1 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +a=6378300 +b=6356751.689189189 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Accra / TM 1 NW\",GEOGCS[\"Accra\",DATUM[\"Accr");
    add_srs_wkt (p, 1,
		 "a\",SPHEROID[\"War Office\",6378300,296,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 2,
		 "\",\"7029\"]],AUTHORITY[\"EPSG\",\"6168\"]],PRIMEM[\"Gre");
    add_srs_wkt (p, 3,
		 "enwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 4,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"4168\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 7,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 8,
		 "dian\",-1],PARAMETER[\"scale_factor\",0.9996],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"false_easting\",500000],PARAMETER[\"false_northing\",0");
    add_srs_wkt (p, 10,
		 "],AUTHORITY[\"EPSG\",\"2137\"],AXIS[\"Easting\",EAST],AX");
    add_srs_wkt (p, 11, "IS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2138, "epsg", 2138,
		      "NAD27(CGQ77) / Quebec Lambert");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=60 +lat_2=46 +lat_0=44 +lon_0=-68.5 +x_");
    add_proj4text (p, 1, "0=0 +y_0=0 +ellps=clrk66 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27(CGQ77) / Quebec Lambert\",GEOGCS[\"NAD27(");
    add_srs_wkt (p, 1,
		 "CGQ77)\",DATUM[\"North_American_Datum_1927_CGQ77\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"Clarke 1866\",6378206.4,294.9786982139006,AUTHORIT");
    add_srs_wkt (p, 3,
		 "Y[\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"6609\"]],PRIM");
    add_srs_wkt (p, 4,
		 "EM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4609\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Lambert_Conformal_C");
    add_srs_wkt (p, 8,
		 "onic_2SP\"],PARAMETER[\"standard_parallel_1\",60],PARAME");
    add_srs_wkt (p, 9,
		 "TER[\"standard_parallel_2\",46],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 10,
		 "origin\",44],PARAMETER[\"central_meridian\",-68.5],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_easting\",0],PARAMETER[\"false_northing\",0");
    add_srs_wkt (p, 12,
		 "],AUTHORITY[\"EPSG\",\"2138\"],AXIS[\"X\",EAST],AXIS[\"Y");
    add_srs_wkt (p, 13, "\",NORTH]]");
    p = add_epsg_def (first, last, 2139, "epsg", 2139,
		      "NAD83(CSRS98) / SCoPQ zone 2 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-55.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / SCoPQ zone 2 (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-55.5],");
    add_srs_wkt (p, 10,
		 "PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_eas");
    add_srs_wkt (p, 11,
		 "ting\",304800],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 12,
		 "[\"EPSG\",\"2139\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2140, "epsg", 2140,
		      "NAD83(CSRS98) / MTM zone 3 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-58.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 3 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-58.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2140\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2141, "epsg", 2141,
		      "NAD83(CSRS98) / MTM zone 4 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-61.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 4 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-61.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2141\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2142, "epsg", 2142,
		      "NAD83(CSRS98) / MTM zone 5 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-64.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 5 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-64.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2142\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2143, "epsg", 2143,
		      "NAD83(CSRS98) / MTM zone 6 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-67.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 6 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-67.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2143\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2144, "epsg", 2144,
		      "NAD83(CSRS98) / MTM zone 7 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-70.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 7 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-70.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2144\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2145, "epsg", 2145,
		      "NAD83(CSRS98) / MTM zone 8 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-73.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 8 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-73.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2145\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2146, "epsg", 2146,
		      "NAD83(CSRS98) / MTM zone 9 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-76.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 9 (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refer");
    add_srs_wkt (p, 2,
		 "ence_System\",SPHEROID[\"GRS 1980\",6378137,298.25722210");
    add_srs_wkt (p, 3,
		 "1,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "9433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-76.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2146\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2147, "epsg", 2147,
		      "NAD83(CSRS98) / MTM zone 10 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-79.5 +k=0.9999 +x_0=304800 ");
    add_proj4text (p, 1,
		   "+y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_");
    add_proj4text (p, 2, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / MTM zone 10 (deprecated)\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Refe");
    add_srs_wkt (p, 2,
		 "rence_System\",SPHEROID[\"GRS 1980\",6378137,298.2572221");
    add_srs_wkt (p, 3,
		 "01,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 5,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 6,
		 "99433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-79.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9999],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",304800],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2147\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH");
    add_srs_wkt (p, 13, "]]");
    p = add_epsg_def (first, last, 2148, "epsg", 2148,
		      "NAD83(CSRS98) / UTM zone 21N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=21 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 21N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-57],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2148\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2149, "epsg", 2149,
		      "NAD83(CSRS98) / UTM zone 18N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=18 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 18N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-75],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2149\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2150, "epsg", 2150,
		      "NAD83(CSRS98) / UTM zone 17N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=17 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 17N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-81],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_easti");
    add_srs_wkt (p, 11,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2150\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2151, "epsg", 2151,
		      "NAD83(CSRS98) / UTM zone 13N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=13 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 13N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-105],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2151\"],AXIS[\"Easting\",EAST],AXIS[\"Northin");
    add_srs_wkt (p, 13, "g\",NORTH]]");
    p = add_epsg_def (first, last, 2152, "epsg", 2152,
		      "NAD83(CSRS98) / UTM zone 12N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=12 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 12N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-111],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2152\"],AXIS[\"Easting\",EAST],AXIS[\"Northin");
    add_srs_wkt (p, 13, "g\",NORTH]]");
    p = add_epsg_def (first, last, 2153, "epsg", 2153,
		      "NAD83(CSRS98) / UTM zone 11N (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=11 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / UTM zone 11N (deprecated)\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"NAD83(CSRS98)\",DATUM[\"NAD83_Canadian_Spatial_Ref");
    add_srs_wkt (p, 2,
		 "erence_System\",SPHEROID[\"GRS 1980\",6378137,298.257222");
    add_srs_wkt (p, 3,
		 "101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6140\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"4140\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 8,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 9,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",-117],P");
    add_srs_wkt (p, 10,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 11,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2153\"],AXIS[\"Easting\",EAST],AXIS[\"Northin");
    add_srs_wkt (p, 13, "g\",NORTH]]");
    p = add_epsg_def (first, last, 2154, "epsg", 2154, "RGF93 / Lambert-93");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=49 +lat_2=44 +lat_0=46.5 +lon_0=3 +x_0=");
    add_proj4text (p, 1,
		   "700000 +y_0=6600000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"RGF93 / Lambert-93\",GEOGCS[\"RGF93\",DATUM[\"R");
    add_srs_wkt (p, 1,
		 "eseau_Geodesique_Francais_1993\",SPHEROID[\"GRS 1980\",6");
    add_srs_wkt (p, 2,
		 "378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS");
    add_srs_wkt (p, 3,
		 "84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6171\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4171\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Lambert_Conformal_Conic_2");
    add_srs_wkt (p, 8,
		 "SP\"],PARAMETER[\"standard_parallel_1\",49],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "standard_parallel_2\",44],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 10,
		 "\",46.5],PARAMETER[\"central_meridian\",3],PARAMETER[\"f");
    add_srs_wkt (p, 11,
		 "alse_easting\",700000],PARAMETER[\"false_northing\",6600");
    add_srs_wkt (p, 12,
		 "000],AUTHORITY[\"EPSG\",\"2154\"],AXIS[\"X\",EAST],AXIS[");
    add_srs_wkt (p, 13, "\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2155, "epsg", 2155,
		      "American Samoa 1962 / American Samoa Lambert (deprecated)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=-14.26666666666667 +lat_0=-14.266666666");
    add_proj4text (p, 1,
		   "66667 +lon_0=170 +k_0=1 +x_0=152400.3048006096 +y_0=0 +e");
    add_proj4text (p, 2,
		   "llps=clrk66 +towgs84=-115,118,426,0,0,0,0 +units=us-ft +");
    add_proj4text (p, 3, "no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"American Samoa 1962 / American Samoa Lambert (d");
    add_srs_wkt (p, 1,
		 "eprecated)\",GEOGCS[\"American Samoa 1962\",DATUM[\"Amer");
    add_srs_wkt (p, 2,
		 "ican_Samoa_1962\",SPHEROID[\"Clarke 1866\",6378206.4,294");
    add_srs_wkt (p, 3,
		 ".9786982139006,AUTHORITY[\"EPSG\",\"7008\"]],TOWGS84[-11");
    add_srs_wkt (p, 4,
		 "5,118,426,0,0,0,0],AUTHORITY[\"EPSG\",\"6169\"]],PRIMEM[");
    add_srs_wkt (p, 5,
		 "\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"deg");
    add_srs_wkt (p, 6,
		 "ree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"4169\"]],UNIT[\"US survey foot\",0.");
    add_srs_wkt (p, 8,
		 "3048006096012192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTIO");
    add_srs_wkt (p, 9,
		 "N[\"Lambert_Conformal_Conic_1SP\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 10,
		 "of_origin\",-14.26666666666667],PARAMETER[\"central_meri");
    add_srs_wkt (p, 11,
		 "dian\",170],PARAMETER[\"scale_factor\",1],PARAMETER[\"fa");
    add_srs_wkt (p, 12,
		 "lse_easting\",500000],PARAMETER[\"false_northing\",0],AU");
    add_srs_wkt (p, 13,
		 "THORITY[\"EPSG\",\"2155\"],AXIS[\"X\",EAST],AXIS[\"Y\",N");
    add_srs_wkt (p, 14, "ORTH]]");
    p = add_epsg_def (first, last, 2156, "epsg", 2156,
		      "NAD83(HARN) / UTM zone 59S (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=59 +south +ellps=GRS80 +units=m +no_defs");
    add_proj4text (p, 1, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(HARN) / UTM zone 59S (deprecated)\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"NAD83(HARN)\",DATUM[\"NAD83_High_Accuracy_Regional_N");
    add_srs_wkt (p, 2,
		 "etwork\",SPHEROID[\"GRS 1980\",6378137,298.257222101,AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"7019\"]],AUTHORITY[\"EPSG\",\"6152\"]]");
    add_srs_wkt (p, 4,
		 ",PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UN");
    add_srs_wkt (p, 5,
		 "IT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 6,
		 "122\"]],AUTHORITY[\"EPSG\",\"4152\"]],UNIT[\"metre\",1,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "central_meridian\",171],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 11,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2156\"],AXIS[");
    add_srs_wkt (p, 12, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2157, "epsg", 2157,
		      "IRENET95 / Irish Transverse Mercator");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=53.5 +lon_0=-8 +k=0.99982 +x_0=600000");
    add_proj4text (p, 1,
		   " +y_0=750000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=");
    add_proj4text (p, 2, "m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"IRENET95 / Irish Transverse Mercator\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "IRENET95\",DATUM[\"IRENET95\",SPHEROID[\"GRS 1980\",6378");
    add_srs_wkt (p, 2,
		 "137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[");
    add_srs_wkt (p, 3,
		 "0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6173\"]],PRIMEM[\"Gr");
    add_srs_wkt (p, 4,
		 "eenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4173\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",53.5],PARAMETER[\"central_m");
    add_srs_wkt (p, 9,
		 "eridian\",-8],PARAMETER[\"scale_factor\",0.99982],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",600000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 11,
		 "\",750000],AUTHORITY[\"EPSG\",\"2157\"],AXIS[\"Easting\"");
    add_srs_wkt (p, 12, ",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2158, "epsg", 2158,
		      "IRENET95 / UTM zone 29N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=29 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"IRENET95 / UTM zone 29N\",GEOGCS[\"IRENET95\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"IRENET95\",SPHEROID[\"GRS 1980\",6378137,298.2572");
    add_srs_wkt (p, 2,
		 "22101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,");
    add_srs_wkt (p, 3,
		 "0],AUTHORITY[\"EPSG\",\"6173\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4173\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",0],PARAMETER[\"central_meridian\",-9],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_eas");
    add_srs_wkt (p, 10,
		 "ting\",500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2158\"],AXIS[\"Easting\",EAST],AXIS[\"Northi");
    add_srs_wkt (p, 12, "ng\",NORTH]]");
    p = add_epsg_def (first, last, 2159, "epsg", 2159,
		      "Sierra Leone 1924 / New Colony Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=6.666666666666667 +lon_0=-12 +k=1 +x_");
    add_proj4text (p, 1,
		   "0=152399.8550907544 +y_0=0 +a=6378300 +b=6356751.6891891");
    add_proj4text (p, 2, "89 +to_meter=0.3047997101815088 +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Sierra Leone 1924 / New Colony Grid\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Sierra Leone 1924\",DATUM[\"Sierra_Leone_Colony_1924\",S");
    add_srs_wkt (p, 2,
		 "PHEROID[\"War Office\",6378300,296,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7029\"]],AUTHORITY[\"EPSG\",\"6174\"]],PRIMEM[\"Greenwic");
    add_srs_wkt (p, 4,
		 "h\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01");
    add_srs_wkt (p, 5,
		 "745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[");
    add_srs_wkt (p, 6,
		 "\"EPSG\",\"4174\"]],UNIT[\"Gold Coast foot\",0.304799710");
    add_srs_wkt (p, 7,
		 "1815088,AUTHORITY[\"EPSG\",\"9094\"]],PROJECTION[\"Trans");
    add_srs_wkt (p, 8,
		 "verse_Mercator\"],PARAMETER[\"latitude_of_origin\",6.666");
    add_srs_wkt (p, 9,
		 "666666666667],PARAMETER[\"central_meridian\",-12],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12,
		 "2159\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2160, "epsg", 2160,
		      "Sierra Leone 1924 / New War Office Grid");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=6.666666666666667 +lon_0=-12 +k=1 +x_");
    add_proj4text (p, 1,
		   "0=243839.7681452071 +y_0=182879.8261089053 +a=6378300 +b");
    add_proj4text (p, 2,
		   "=6356751.689189189 +to_meter=0.3047997101815088 +no_defs");
    add_proj4text (p, 3, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Sierra Leone 1924 / New War Office Grid\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"Sierra Leone 1924\",DATUM[\"Sierra_Leone_Colony_1924");
    add_srs_wkt (p, 2,
		 "\",SPHEROID[\"War Office\",6378300,296,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7029\"]],AUTHORITY[\"EPSG\",\"6174\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4174\"]],UNIT[\"Gold Coast foot\",0.304799");
    add_srs_wkt (p, 7,
		 "7101815088,AUTHORITY[\"EPSG\",\"9094\"]],PROJECTION[\"Tr");
    add_srs_wkt (p, 8,
		 "ansverse_Mercator\"],PARAMETER[\"latitude_of_origin\",6.");
    add_srs_wkt (p, 9,
		 "666666666666667],PARAMETER[\"central_meridian\",-12],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",8");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",600000],AUTHORITY[\"");
    add_srs_wkt (p, 12,
		 "EPSG\",\"2160\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\"");
    add_srs_wkt (p, 13, ",NORTH]]");
    p = add_epsg_def (first, last, 2161, "epsg", 2161,
		      "Sierra Leone 1968 / UTM zone 28N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=28 +ellps=clrk80 +towgs84=-88,4,101,0,0,");
    add_proj4text (p, 1, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Sierra Leone 1968 / UTM zone 28N\",GEOGCS[\"Sie");
    add_srs_wkt (p, 1,
		 "rra Leone 1968\",DATUM[\"Sierra_Leone_1968\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7012\"]],TOWGS84[-88,4,101,0,0,0,0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"6175\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4175\"]],UNIT");
    add_srs_wkt (p, 7,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 8,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 9,
		 "0],PARAMETER[\"central_meridian\",-15],PARAMETER[\"scale");
    add_srs_wkt (p, 10,
		 "_factor\",0.9996],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2161\"");
    add_srs_wkt (p, 12, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2162, "epsg", 2162,
		      "Sierra Leone 1968 / UTM zone 29N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=29 +ellps=clrk80 +towgs84=-88,4,101,0,0,");
    add_proj4text (p, 1, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Sierra Leone 1968 / UTM zone 29N\",GEOGCS[\"Sie");
    add_srs_wkt (p, 1,
		 "rra Leone 1968\",DATUM[\"Sierra_Leone_1968\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Clarke 1880 (RGS)\",6378249.145,293.465,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7012\"]],TOWGS84[-88,4,101,0,0,0,0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"6175\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4175\"]],UNIT");
    add_srs_wkt (p, 7,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 8,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 9,
		 "0],PARAMETER[\"central_meridian\",-9],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9996],PARAMETER[\"false_easting\",500000],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2162\"");
    add_srs_wkt (p, 12, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2163, "epsg", 2163, "unnamed");
    add_proj4text (p, 0,
		   "+proj=laea +lat_0=45 +lon_0=-100 +x_0=0 +y_0=0 +a=637099");
    add_proj4text (p, 1, "7 +b=6370997 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"unnamed\",GEOGCS[\"unnamed ellipse\",DATUM[\"un");
    add_srs_wkt (p, 1,
		 "known\",SPHEROID[\"unnamed\",6370997,0]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 2,
		 "ich\",0],UNIT[\"degree\",0.0174532925199433]],PROJECTION");
    add_srs_wkt (p, 3,
		 "[\"Lambert_Azimuthal_Equal_Area\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 4,
		 "of_center\",45],PARAMETER[\"longitude_of_center\",-100],");
    add_srs_wkt (p, 5,
		 "PARAMETER[\"false_easting\",0],PARAMETER[\"false_northin");
    add_srs_wkt (p, 6,
		 "g\",0],UNIT[\"Meter\",1],AUTHORITY[\"EPSG\",\"2163\"]]");
    p = add_epsg_def (first, last, 2164, "epsg", 2164,
		      "Locodjo 1965 / TM 5 NW");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-5 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=clrk80 +towgs84=-125,53,467,0,0,0,0 +units=m ");
    add_proj4text (p, 2, "+no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Locodjo 1965 / TM 5 NW\",GEOGCS[\"Locodjo 1965\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"Locodjo_1965\",SPHEROID[\"Clarke 1880 (RGS)\",6");
    add_srs_wkt (p, 2,
		 "378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],TOWGS84");
    add_srs_wkt (p, 3,
		 "[-125,53,467,0,0,0,0],AUTHORITY[\"EPSG\",\"6142\"]],PRIM");
    add_srs_wkt (p, 4,
		 "EM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4142\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",-5],PARAMETER[\"scale_factor\",0.9996],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"false_easting\",500000],PARAMETER[\"false_nort");
    add_srs_wkt (p, 11,
		 "hing\",0],AUTHORITY[\"EPSG\",\"2164\"],AXIS[\"Easting\",");
    add_srs_wkt (p, 12, "EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2165, "epsg", 2165,
		      "Abidjan 1987 / TM 5 NW");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-5 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=clrk80 +towgs84=-124.76,53,466.79,0,0,0,0 +un");
    add_proj4text (p, 2, "its=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Abidjan 1987 / TM 5 NW\",GEOGCS[\"Abidjan 1987\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"Abidjan_1987\",SPHEROID[\"Clarke 1880 (RGS)\",6");
    add_srs_wkt (p, 2,
		 "378249.145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],TOWGS84");
    add_srs_wkt (p, 3,
		 "[-124.76,53,466.79,0,0,0,0],AUTHORITY[\"EPSG\",\"6143\"]");
    add_srs_wkt (p, 4,
		 "],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],U");
    add_srs_wkt (p, 5,
		 "NIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4143\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-5],PARAMETER[\"scale_factor\",0.99");
    add_srs_wkt (p, 10,
		 "96],PARAMETER[\"false_easting\",500000],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2165\"],AXIS[\"East");
    add_srs_wkt (p, 12, "ing\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2166, "epsg", 2166,
		      "Pulkovo 1942(83) / Gauss Kruger zone 3 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=9 +k=1 +x_0=3500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / Gauss Kruger zone 3 (depreca");
    add_srs_wkt (p, 1,
		 "ted)\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_");
    add_srs_wkt (p, 2,
		 "83\",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORIT");
    add_srs_wkt (p, 3,
		 "Y[\"EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIM");
    add_srs_wkt (p, 4,
		 "EM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",9],PARAMETER[\"scale_factor\",1],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_easting\",3500000],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 11,
		 ",0],AUTHORITY[\"EPSG\",\"2166\"],AXIS[\"X\",NORTH],AXIS[");
    add_srs_wkt (p, 12, "\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2167, "epsg", 2167,
		      "Pulkovo 1942(83) / Gauss Kruger zone 4 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=12 +k=1 +x_0=4500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / Gauss Kruger zone 4 (depreca");
    add_srs_wkt (p, 1,
		 "ted)\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_");
    add_srs_wkt (p, 2,
		 "83\",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORIT");
    add_srs_wkt (p, 3,
		 "Y[\"EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIM");
    add_srs_wkt (p, 4,
		 "EM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",12],PARAMETER[\"scale_factor\",1],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_easting\",4500000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 11,
		 "\",0],AUTHORITY[\"EPSG\",\"2167\"],AXIS[\"X\",NORTH],AXI");
    add_srs_wkt (p, 12, "S[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2168, "epsg", 2168,
		      "Pulkovo 1942(83) / Gauss Kruger zone 5 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=1 +x_0=5500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / Gauss Kruger zone 5 (depreca");
    add_srs_wkt (p, 1,
		 "ted)\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_");
    add_srs_wkt (p, 2,
		 "83\",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORIT");
    add_srs_wkt (p, 3,
		 "Y[\"EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIM");
    add_srs_wkt (p, 4,
		 "EM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"");
    add_srs_wkt (p, 5,
		 "degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"");
    add_srs_wkt (p, 6,
		 "]],AUTHORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",15],PARAMETER[\"scale_factor\",1],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_easting\",5500000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 11,
		 "\",0],AUTHORITY[\"EPSG\",\"2168\"],AXIS[\"X\",NORTH],AXI");
    add_srs_wkt (p, 12, "S[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2169, "epsg", 2169,
		      "Luxembourg 1930 / Gauss");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=49.83333333333334 +lon_0=6.1666666666");
    add_proj4text (p, 1,
		   "66667 +k=1 +x_0=80000 +y_0=100000 +ellps=intl +towgs84=-");
    add_proj4text (p, 2,
		   "193,13.7,-39.3,-0.41,-2.933,2.688,0.43 +units=m +no_defs");
    add_proj4text (p, 3, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Luxembourg 1930 / Gauss\",GEOGCS[\"Luxembourg 1");
    add_srs_wkt (p, 1,
		 "930\",DATUM[\"Luxembourg_1930\",SPHEROID[\"International");
    add_srs_wkt (p, 2,
		 " 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],TOWGS8");
    add_srs_wkt (p, 3,
		 "4[-193,13.7,-39.3,-0.41,-2.933,2.688,0.43],AUTHORITY[\"E");
    add_srs_wkt (p, 4,
		 "PSG\",\"6181\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 5,
		 "\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHOR");
    add_srs_wkt (p, 6,
		 "ITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4181\"]],UN");
    add_srs_wkt (p, 7,
		 "IT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[");
    add_srs_wkt (p, 8,
		 "\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 9,
		 ",49.83333333333334],PARAMETER[\"central_meridian\",6.166");
    add_srs_wkt (p, 10,
		 "666666666667],PARAMETER[\"scale_factor\",1],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_easting\",80000],PARAMETER[\"false_northing\",1000");
    add_srs_wkt (p, 12,
		 "00],AUTHORITY[\"EPSG\",\"2169\"],AXIS[\"X\",NORTH],AXIS[");
    add_srs_wkt (p, 13, "\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2170, "epsg", 2170,
		      "MGI / Slovenia Grid (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=0.9999 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=bessel +datum=hermannskogel +units=m +no_defs");
    add_proj4text (p, 2, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"MGI / Slovenia Grid (deprecated)\",GEOGCS[\"MGI");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"Militar_Geographische_Institute\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Bessel 1841\",6377397.155,299.1528128,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7004\"]],TOWGS84[577.326,90.129,463.919,5.137,1.474,5");
    add_srs_wkt (p, 4,
		 ".297,2.4232],AUTHORITY[\"EPSG\",\"6312\"]],PRIMEM[\"Gree");
    add_srs_wkt (p, 5,
		 "nwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",");
    add_srs_wkt (p, 6,
		 "0.0174532925199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORI");
    add_srs_wkt (p, 7,
		 "TY[\"EPSG\",\"4312\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 8,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 9,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 10,
		 "an\",15],PARAMETER[\"scale_factor\",0.9999],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 12,
		 "AUTHORITY[\"EPSG\",\"2170\"],AXIS[\"X\",NORTH],AXIS[\"Y\"");
    add_srs_wkt (p, 13, ",EAST]]");
    p = add_epsg_def (first, last, 2171, "epsg", 2171,
		      "Pulkovo 1942(58) / Poland zone I (deprecated)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=50.625 +lon_0=21.08333333333333 +k=0");
    add_proj4text (p, 1,
		   ".9998 +x_0=4637000 +y_0=5647000 +ellps=krass +units=m +n");
    add_proj4text (p, 2, "o_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(58) / Poland zone I (deprecated)\"");
    add_srs_wkt (p, 1,
		 ",GEOGCS[\"Pulkovo 1942(58)\",DATUM[\"Pulkovo_1942_58\",S");
    add_srs_wkt (p, 2,
		 "PHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6179\"]],PRIMEM[\"G");
    add_srs_wkt (p, 4,
		 "reenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree");
    add_srs_wkt (p, 5,
		 "\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"4179\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Oblique_Stereographic\"],");
    add_srs_wkt (p, 8,
		 "PARAMETER[\"latitude_of_origin\",50.625],PARAMETER[\"cen");
    add_srs_wkt (p, 9,
		 "tral_meridian\",21.08333333333333],PARAMETER[\"scale_fac");
    add_srs_wkt (p, 10,
		 "tor\",0.9998],PARAMETER[\"false_easting\",4637000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",5647000],AUTHORITY[\"EPSG\",\"21");
    add_srs_wkt (p, 12, "71\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2172, "epsg", 2172,
		      "Pulkovo 1942(58) / Poland zone II");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=53.00194444444445 +lon_0=21.50277777");
    add_proj4text (p, 1,
		   "777778 +k=0.9998 +x_0=4603000 +y_0=5806000 +ellps=krass ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(58) / Poland zone II\",GEOGCS[\"Pu");
    add_srs_wkt (p, 1,
		 "lkovo 1942(58)\",DATUM[\"Pulkovo_1942_58\",SPHEROID[\"Kr");
    add_srs_wkt (p, 2,
		 "assowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6179\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4179\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Oblique_Stereographic\"],PARAMETER[\"l");
    add_srs_wkt (p, 8,
		 "atitude_of_origin\",53.00194444444445],PARAMETER[\"centr");
    add_srs_wkt (p, 9,
		 "al_meridian\",21.50277777777778],PARAMETER[\"scale_facto");
    add_srs_wkt (p, 10,
		 "r\",0.9998],PARAMETER[\"false_easting\",4603000],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",5806000],AUTHORITY[\"EPSG\",\"2172");
    add_srs_wkt (p, 12, "\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2173, "epsg", 2173,
		      "Pulkovo 1942(58) / Poland zone III");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=53.58333333333334 +lon_0=17.00833333");
    add_proj4text (p, 1,
		   "333333 +k=0.9998 +x_0=3501000 +y_0=5999000 +ellps=krass ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(58) / Poland zone III\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942(58)\",DATUM[\"Pulkovo_1942_58\",SPHEROID[\"K");
    add_srs_wkt (p, 2,
		 "rassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6179\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4179\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "01\"]],PROJECTION[\"Oblique_Stereographic\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",53.58333333333334],PARAMETER[\"cent");
    add_srs_wkt (p, 9,
		 "ral_meridian\",17.00833333333333],PARAMETER[\"scale_fact");
    add_srs_wkt (p, 10,
		 "or\",0.9998],PARAMETER[\"false_easting\",3501000],PARAME");
    add_srs_wkt (p, 11,
		 "TER[\"false_northing\",5999000],AUTHORITY[\"EPSG\",\"217");
    add_srs_wkt (p, 12, "3\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2174, "epsg", 2174,
		      "Pulkovo 1942(58) / Poland zone IV");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=51.67083333333333 +lon_0=16.67222222");
    add_proj4text (p, 1,
		   "222222 +k=0.9998 +x_0=3703000 +y_0=5627000 +ellps=krass ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(58) / Poland zone IV\",GEOGCS[\"Pu");
    add_srs_wkt (p, 1,
		 "lkovo 1942(58)\",DATUM[\"Pulkovo_1942_58\",SPHEROID[\"Kr");
    add_srs_wkt (p, 2,
		 "assowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6179\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4179\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Oblique_Stereographic\"],PARAMETER[\"l");
    add_srs_wkt (p, 8,
		 "atitude_of_origin\",51.67083333333333],PARAMETER[\"centr");
    add_srs_wkt (p, 9,
		 "al_meridian\",16.67222222222222],PARAMETER[\"scale_facto");
    add_srs_wkt (p, 10,
		 "r\",0.9998],PARAMETER[\"false_easting\",3703000],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",5627000],AUTHORITY[\"EPSG\",\"2174");
    add_srs_wkt (p, 12, "\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2175, "epsg", 2175,
		      "Pulkovo 1942(58) / Poland zone V");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=18.95833333333333 +k=0.99998");
    add_proj4text (p, 1,
		   "3 +x_0=237000 +y_0=-4700000 +ellps=krass +units=m +no_de");
    add_proj4text (p, 2, "fs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(58) / Poland zone V\",GEOGCS[\"Pul");
    add_srs_wkt (p, 1,
		 "kovo 1942(58)\",DATUM[\"Pulkovo_1942_58\",SPHEROID[\"Kra");
    add_srs_wkt (p, 2,
		 "ssowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6179\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4179\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",0],PARAMETER[\"central_meridian\",18.9");
    add_srs_wkt (p, 9,
		 "5833333333333],PARAMETER[\"scale_factor\",0.999983],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"false_easting\",237000],PARAMETER[\"false_northi");
    add_srs_wkt (p, 11,
		 "ng\",-4700000],AUTHORITY[\"EPSG\",\"2175\"],AXIS[\"X\",N");
    add_srs_wkt (p, 12, "ORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2176, "epsg", 2176,
		      "ETRS89 / Poland CS2000 zone 5");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=0.999923 +x_0=5500000 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Poland CS2000 zone 5\",GEOGCS[\"ETRS89");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Terrestrial_Reference_System_1989\",");
    add_srs_wkt (p, 2,
		 "SPHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",15],PARAMETER[\"scale_factor\",0.999923],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",5500000],PARAMETER[\"false_northin");
    add_srs_wkt (p, 11,
		 "g\",0],AUTHORITY[\"EPSG\",\"2176\"],AXIS[\"x\",NORTH],AX");
    add_srs_wkt (p, 12, "IS[\"y\",EAST]]");
    p = add_epsg_def (first, last, 2177, "epsg", 2177,
		      "ETRS89 / Poland CS2000 zone 6");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=18 +k=0.999923 +x_0=6500000 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Poland CS2000 zone 6\",GEOGCS[\"ETRS89");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Terrestrial_Reference_System_1989\",");
    add_srs_wkt (p, 2,
		 "SPHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",18],PARAMETER[\"scale_factor\",0.999923],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",6500000],PARAMETER[\"false_northin");
    add_srs_wkt (p, 11,
		 "g\",0],AUTHORITY[\"EPSG\",\"2177\"],AXIS[\"x\",NORTH],AX");
    add_srs_wkt (p, 12, "IS[\"y\",EAST]]");
    p = add_epsg_def (first, last, 2178, "epsg", 2178,
		      "ETRS89 / Poland CS2000 zone 7");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=0.999923 +x_0=7500000 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Poland CS2000 zone 7\",GEOGCS[\"ETRS89");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Terrestrial_Reference_System_1989\",");
    add_srs_wkt (p, 2,
		 "SPHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",21],PARAMETER[\"scale_factor\",0.999923],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",7500000],PARAMETER[\"false_northin");
    add_srs_wkt (p, 11,
		 "g\",0],AUTHORITY[\"EPSG\",\"2178\"],AXIS[\"x\",NORTH],AX");
    add_srs_wkt (p, 12, "IS[\"y\",EAST]]");
    p = add_epsg_def (first, last, 2179, "epsg", 2179,
		      "ETRS89 / Poland CS2000 zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=0.999923 +x_0=8500000 ");
    add_proj4text (p, 1, "+y_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Poland CS2000 zone 8\",GEOGCS[\"ETRS89");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"European_Terrestrial_Reference_System_1989\",");
    add_srs_wkt (p, 2,
		 "SPHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",24],PARAMETER[\"scale_factor\",0.999923],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",8500000],PARAMETER[\"false_northin");
    add_srs_wkt (p, 11,
		 "g\",0],AUTHORITY[\"EPSG\",\"2179\"],AXIS[\"x\",NORTH],AX");
    add_srs_wkt (p, 12, "IS[\"y\",EAST]]");
    p = add_epsg_def (first, last, 2180, "epsg", 2180, "ETRS89 / Poland CS92");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=19 +k=0.9993 +x_0=500000 +y_");
    add_proj4text (p, 1, "0=-5300000 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Poland CS92\",GEOGCS[\"ETRS89\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"European_Terrestrial_Reference_System_1989\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"Greenwic");
    add_srs_wkt (p, 4,
		 "h\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01");
    add_srs_wkt (p, 5,
		 "745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[");
    add_srs_wkt (p, 6,
		 "\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 7,
		 "\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER");
    add_srs_wkt (p, 8,
		 "[\"latitude_of_origin\",0],PARAMETER[\"central_meridian\"");
    add_srs_wkt (p, 9,
		 ",19],PARAMETER[\"scale_factor\",0.9993],PARAMETER[\"fals");
    add_srs_wkt (p, 10,
		 "e_easting\",500000],PARAMETER[\"false_northing\",-530000");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2180\"],AXIS[\"x\",NORTH],AXIS[\"");
    add_srs_wkt (p, 12, "y\",EAST]]");
    p = add_epsg_def (first, last, 2188, "epsg", 2188,
		      "Azores Occidental 1939 / UTM zone 25N");
    add_proj4text (p, 0, "+proj=utm +zone=25 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Azores Occidental 1939 / UTM zone 25N\",GEOGCS[");
    add_srs_wkt (p, 1,
		 "\"Azores Occidental 1939\",DATUM[\"Azores_Occidental_Isl");
    add_srs_wkt (p, 2,
		 "ands_1939\",SPHEROID[\"International 1924\",6378388,297,");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6182\"");
    add_srs_wkt (p, 4,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 5,
		 "UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "9122\"]],AUTHORITY[\"EPSG\",\"4182\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Me");
    add_srs_wkt (p, 8,
		 "rcator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-33],PARAMETER[\"scale_factor\",0.9");
    add_srs_wkt (p, 10,
		 "996],PARAMETER[\"false_easting\",500000],PARAMETER[\"fal");
    add_srs_wkt (p, 11,
		 "se_northing\",0],AUTHORITY[\"EPSG\",\"2188\"],AXIS[\"Eas");
    add_srs_wkt (p, 12, "ting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2189, "epsg", 2189,
		      "Azores Central 1948 / UTM zone 26N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=26 +ellps=intl +towgs84=-104,167,-38,0,0");
    add_proj4text (p, 1, ",0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Azores Central 1948 / UTM zone 26N\",GEOGCS[\"A");
    add_srs_wkt (p, 1,
		 "zores Central 1948\",DATUM[\"Azores_Central_Islands_1948");
    add_srs_wkt (p, 2,
		 "\",SPHEROID[\"International 1924\",6378388,297,AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"7022\"]],TOWGS84[-104,167,-38,0,0,0,0],AUTHO");
    add_srs_wkt (p, 4,
		 "RITY[\"EPSG\",\"6183\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 6,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"418");
    add_srs_wkt (p, 7,
		 "3\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 8,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 9,
		 "_origin\",0],PARAMETER[\"central_meridian\",-27],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",0.9996],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12,
		 "\",\"2189\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NO");
    add_srs_wkt (p, 13, "RTH]]");
    p = add_epsg_def (first, last, 2190, "epsg", 2190,
		      "Azores Oriental 1940 / UTM zone 26N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=26 +ellps=intl +towgs84=-203,141,53,0,0,");
    add_proj4text (p, 1, "0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Azores Oriental 1940 / UTM zone 26N\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Azores Oriental 1940\",DATUM[\"Azores_Oriental_Islands_1");
    add_srs_wkt (p, 2,
		 "940\",SPHEROID[\"International 1924\",6378388,297,AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"7022\"]],TOWGS84[-203,141,53,0,0,0,0],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6184\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "184\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",-27],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",0.9996],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12,
		 "G\",\"2190\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",N");
    add_srs_wkt (p, 13, "ORTH]]");
    p = add_epsg_def (first, last, 2191, "epsg", 2191,
		      "Madeira 1936 / UTM zone 28N (deprecated)");
    add_proj4text (p, 0, "+proj=utm +zone=28 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Madeira 1936 / UTM zone 28N (deprecated)\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Madeira 1936\",DATUM[\"Madeira_1936\",SPHEROID[\"In");
    add_srs_wkt (p, 2,
		 "ternational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6185\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "2925199433,AUTHORITY[\"EPSG\",\"9108\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 6,
		 "G\",\"4185\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"900");
    add_srs_wkt (p, 7,
		 "1\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"la");
    add_srs_wkt (p, 8,
		 "titude_of_origin\",0],PARAMETER[\"central_meridian\",-15");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_e");
    add_srs_wkt (p, 10,
		 "asting\",500000],PARAMETER[\"false_northing\",0],AUTHORI");
    add_srs_wkt (p, 11,
		 "TY[\"EPSG\",\"2191\"],AXIS[\"Easting\",EAST],AXIS[\"Nort");
    add_srs_wkt (p, 12, "hing\",NORTH]]");
    p = add_epsg_def (first, last, 2192, "epsg", 2192,
		      "ED50 / France EuroLambert");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=46.8 +lat_0=46.8 +lon_0=2.3372291666666");
    add_proj4text (p, 1,
		   "67 +k_0=0.99987742 +x_0=600000 +y_0=2200000 +ellps=intl ");
    add_proj4text (p, 2, "+units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / France EuroLambert\",GEOGCS[\"ED50\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"European_Datum_1950\",SPHEROID[\"International 1924");
    add_srs_wkt (p, 2,
		 "\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],U");
    add_srs_wkt (p, 6,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 7,
		 "[\"Lambert_Conformal_Conic_1SP\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 8,
		 "f_origin\",46.8],PARAMETER[\"central_meridian\",2.337229");
    add_srs_wkt (p, 9,
		 "166666667],PARAMETER[\"scale_factor\",0.99987742],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",600000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 11,
		 "\",2200000],AUTHORITY[\"EPSG\",\"2192\"],AXIS[\"X\",EAST");
    add_srs_wkt (p, 12, "],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2193, "epsg", 2193,
		      "NZGD2000 / New Zealand Transverse Mercator 2000");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=173 +k=0.9996 +x_0=1600000 +");
    add_proj4text (p, 1,
		   "y_0=10000000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=");
    add_proj4text (p, 2, "m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NZGD2000 / New Zealand Transverse Mercator 2000");
    add_srs_wkt (p, 1,
		 "\",GEOGCS[\"NZGD2000\",DATUM[\"New_Zealand_Geodetic_Datu");
    add_srs_wkt (p, 2,
		 "m_2000\",SPHEROID[\"GRS 1980\",6378137,298.257222101,AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"6167\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 5,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 6,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4167");
    add_srs_wkt (p, 7,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 8,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 9,
		 "origin\",0],PARAMETER[\"central_meridian\",173],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"scale_factor\",0.9996],PARAMETER[\"false_easting\",1");
    add_srs_wkt (p, 11,
		 "600000],PARAMETER[\"false_northing\",10000000],AUTHORITY");
    add_srs_wkt (p, 12,
		 "[\"EPSG\",\"2193\"],AXIS[\"Northing\",NORTH],AXIS[\"East");
    add_srs_wkt (p, 13, "ing\",EAST]]");
    p = add_epsg_def (first, last, 2194, "epsg", 2194,
		      "American Samoa 1962 / American Samoa Lambert (deprecated)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=-14.26666666666667 +lat_0=-14.266666666");
    add_proj4text (p, 1,
		   "66667 +lon_0=-170 +k_0=1 +x_0=152400.3048006096 +y_0=0 +");
    add_proj4text (p, 2,
		   "ellps=clrk66 +towgs84=-115,118,426,0,0,0,0 +units=us-ft ");
    add_proj4text (p, 3, "+no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"American Samoa 1962 / American Samoa Lambert (d");
    add_srs_wkt (p, 1,
		 "eprecated)\",GEOGCS[\"American Samoa 1962\",DATUM[\"Amer");
    add_srs_wkt (p, 2,
		 "ican_Samoa_1962\",SPHEROID[\"Clarke 1866\",6378206.4,294");
    add_srs_wkt (p, 3,
		 ".9786982139006,AUTHORITY[\"EPSG\",\"7008\"]],TOWGS84[-11");
    add_srs_wkt (p, 4,
		 "5,118,426,0,0,0,0],AUTHORITY[\"EPSG\",\"6169\"]],PRIMEM[");
    add_srs_wkt (p, 5,
		 "\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"deg");
    add_srs_wkt (p, 6,
		 "ree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],");
    add_srs_wkt (p, 7,
		 "AUTHORITY[\"EPSG\",\"4169\"]],UNIT[\"US survey foot\",0.");
    add_srs_wkt (p, 8,
		 "3048006096012192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTIO");
    add_srs_wkt (p, 9,
		 "N[\"Lambert_Conformal_Conic_1SP\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 10,
		 "of_origin\",-14.26666666666667],PARAMETER[\"central_meri");
    add_srs_wkt (p, 11,
		 "dian\",-170],PARAMETER[\"scale_factor\",1],PARAMETER[\"f");
    add_srs_wkt (p, 12,
		 "alse_easting\",500000],PARAMETER[\"false_northing\",0],A");
    add_srs_wkt (p, 13,
		 "UTHORITY[\"EPSG\",\"2194\"],AXIS[\"X\",EAST],AXIS[\"Y\",");
    add_srs_wkt (p, 14, "NORTH]]");
    p = add_epsg_def (first, last, 2195, "epsg", 2195,
		      "NAD83(HARN) / UTM zone 2S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=2 +south +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(HARN) / UTM zone 2S\",GEOGCS[\"NAD83(HARN");
    add_srs_wkt (p, 1,
		 ")\",DATUM[\"NAD83_High_Accuracy_Regional_Network\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],AUTHORITY[\"EPSG\",\"6152\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4152\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-171],PARAMETER[\"scale_factor\",0.9996],PARAMETER[");
    add_srs_wkt (p, 10,
		 "\"false_easting\",500000],PARAMETER[\"false_northing\",1");
    add_srs_wkt (p, 11,
		 "0000000],AUTHORITY[\"EPSG\",\"2195\"],AXIS[\"Easting\",E");
    add_srs_wkt (p, 12, "AST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2196, "epsg", 2196,
		      "ETRS89 / Kp2000 Jutland");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=9.5 +k=0.99995 +x_0=200000 +");
    add_proj4text (p, 1, "y_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Kp2000 Jutland\",GEOGCS[\"ETRS89\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"European_Terrestrial_Reference_System_1989\",SPHERO");
    add_srs_wkt (p, 2,
		 "ID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",9.5],PARAMETER[\"scale_factor\",0.99995],PARAMETER[");
    add_srs_wkt (p, 10,
		 "\"false_easting\",200000],PARAMETER[\"false_northing\",0");
    add_srs_wkt (p, 11,
		 "],AUTHORITY[\"EPSG\",\"2196\"],AXIS[\"Easting\",EAST],AX");
    add_srs_wkt (p, 12, "IS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2197, "epsg", 2197,
		      "ETRS89 / Kp2000 Zealand");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=12 +k=0.99995 +x_0=500000 +y");
    add_proj4text (p, 1, "_0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Kp2000 Zealand\",GEOGCS[\"ETRS89\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"European_Terrestrial_Reference_System_1989\",SPHERO");
    add_srs_wkt (p, 2,
		 "ID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",12],PARAMETER[\"scale_factor\",0.99995],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2197\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2198, "epsg", 2198,
		      "ETRS89 / Kp2000 Bornholm");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=1 +x_0=900000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / Kp2000 Bornholm\",GEOGCS[\"ETRS89\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"European_Terrestrial_Reference_System_1989\",SPHER");
    add_srs_wkt (p, 2,
		 "OID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",15],PARAMETER[\"scale_factor\",1],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_easting\",900000],PARAMETER[\"false_northing\",0],AUTHO");
    add_srs_wkt (p, 11,
		 "RITY[\"EPSG\",\"2198\"],AXIS[\"Easting\",EAST],AXIS[\"No");
    add_srs_wkt (p, 12, "rthing\",NORTH]]");
    p = add_epsg_def (first, last, 2199, "epsg", 2199,
		      "Albanian 1987 / Gauss Kruger zone 4 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=4500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Albanian 1987 / Gauss Kruger zone 4 (deprecated");
    add_srs_wkt (p, 1,
		 ")\",GEOGCS[\"Albanian 1987\",DATUM[\"Albanian_1987\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7024\"]],AUTHORITY[\"EPSG\",\"6191\"]],PRIMEM[\"Gre");
    add_srs_wkt (p, 4,
		 "enwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4191\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 9,
		 "dian\",21],PARAMETER[\"scale_factor\",1],PARAMETER[\"fal");
    add_srs_wkt (p, 10,
		 "se_easting\",4500000],PARAMETER[\"false_northing\",0],AU");
    add_srs_wkt (p, 11,
		 "THORITY[\"EPSG\",\"2199\"],AXIS[\"X\",NORTH],AXIS[\"Y\",");
    add_srs_wkt (p, 12, "EAST]]");
    p = add_epsg_def (first, last, 2200, "epsg", 2200,
		      "ATS77 / New Brunswick Stereographic (ATS77)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=46.5 +lon_0=-66.5 +k=0.999912 +x_0=3");
    add_proj4text (p, 1,
		   "00000 +y_0=800000 +a=6378135 +b=6356750.304921594 +units");
    add_proj4text (p, 2, "=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / New Brunswick Stereographic (ATS77)\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"ATS77\",DATUM[\"Average_Terrestrial_System_1977\"");
    add_srs_wkt (p, 2,
		 ",SPHEROID[\"Average Terrestrial System 1977\",6378135,29");
    add_srs_wkt (p, 3,
		 "8.257,AUTHORITY[\"EPSG\",\"7041\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6122\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4122\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Obliqu");
    add_srs_wkt (p, 8,
		 "e_Stereographic\"],PARAMETER[\"latitude_of_origin\",46.5");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-66.5],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",0.999912],PARAMETER[\"false_easting\",300000]");
    add_srs_wkt (p, 11,
		 ",PARAMETER[\"false_northing\",800000],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12,
		 ",\"2200\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAS");
    add_srs_wkt (p, 13, "T]]");
    p = add_epsg_def (first, last, 2201, "epsg", 2201, "REGVEN / UTM zone 18N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=18 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"REGVEN / UTM zone 18N\",GEOGCS[\"REGVEN\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Red_Geodesica_Venezolana\",SPHEROID[\"GRS 1980\",6378");
    add_srs_wkt (p, 2,
		 "137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[");
    add_srs_wkt (p, 3,
		 "0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6189\"]],PRIMEM[\"Gr");
    add_srs_wkt (p, 4,
		 "eenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4189\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 9,
		 "dian\",-75],PARAMETER[\"scale_factor\",0.9996],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",500000],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2201\"],AXIS[\"Easting\",EAST],A");
    add_srs_wkt (p, 12, "XIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2202, "epsg", 2202, "REGVEN / UTM zone 19N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=19 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"REGVEN / UTM zone 19N\",GEOGCS[\"REGVEN\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Red_Geodesica_Venezolana\",SPHEROID[\"GRS 1980\",6378");
    add_srs_wkt (p, 2,
		 "137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[");
    add_srs_wkt (p, 3,
		 "0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6189\"]],PRIMEM[\"Gr");
    add_srs_wkt (p, 4,
		 "eenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4189\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 9,
		 "dian\",-69],PARAMETER[\"scale_factor\",0.9996],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",500000],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2202\"],AXIS[\"Easting\",EAST],A");
    add_srs_wkt (p, 12, "XIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2203, "epsg", 2203, "REGVEN / UTM zone 20N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=20 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +");
    add_proj4text (p, 1, "units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"REGVEN / UTM zone 20N\",GEOGCS[\"REGVEN\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Red_Geodesica_Venezolana\",SPHEROID[\"GRS 1980\",6378");
    add_srs_wkt (p, 2,
		 "137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],TOWGS84[");
    add_srs_wkt (p, 3,
		 "0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6189\"]],PRIMEM[\"Gr");
    add_srs_wkt (p, 4,
		 "eenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\"");
    add_srs_wkt (p, 5,
		 ",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"4189\"]],UNIT[\"metre\",1,AUTHORITY[\"EP");
    add_srs_wkt (p, 7,
		 "SG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARA");
    add_srs_wkt (p, 8,
		 "METER[\"latitude_of_origin\",0],PARAMETER[\"central_meri");
    add_srs_wkt (p, 9,
		 "dian\",-63],PARAMETER[\"scale_factor\",0.9996],PARAMETER");
    add_srs_wkt (p, 10,
		 "[\"false_easting\",500000],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 11,
		 "0],AUTHORITY[\"EPSG\",\"2203\"],AXIS[\"Easting\",EAST],A");
    add_srs_wkt (p, 12, "XIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2204, "epsg", 2204, "NAD27 / Tennessee");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=35.25 +lat_2=36.41666666666666 +lat_0=3");
    add_proj4text (p, 1,
		   "4.66666666666666 +lon_0=-86 +x_0=609601.2192024384 +y_0=");
    add_proj4text (p, 2,
		   "30480.06096012192 +ellps=clrk66 +datum=NAD27 +units=us-f");
    add_proj4text (p, 3, "t +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD27 / Tennessee\",GEOGCS[\"NAD27\",DATUM[\"No");
    add_srs_wkt (p, 1,
		 "rth_American_Datum_1927\",SPHEROID[\"Clarke 1866\",63782");
    add_srs_wkt (p, 2,
		 "06.4,294.9786982139006,AUTHORITY[\"EPSG\",\"7008\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6267\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "267\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_");
    add_srs_wkt (p, 8,
		 "Conic_2SP\"],PARAMETER[\"standard_parallel_1\",35.25],PA");
    add_srs_wkt (p, 9,
		 "RAMETER[\"standard_parallel_2\",36.41666666666666],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"latitude_of_origin\",34.66666666666666],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"central_meridian\",-86],PARAMETER[\"false_easting\",2");
    add_srs_wkt (p, 12,
		 "000000],PARAMETER[\"false_northing\",100000],AUTHORITY[\"");
    add_srs_wkt (p, 13, "EPSG\",\"2204\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2205, "epsg", 2205,
		      "NAD83 / Kentucky North");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=37.96666666666667 +lat_2=38.96666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=37.5 +lon_0=-84.25 +x_0=500000 +y_0=0 +ellps=");
    add_proj4text (p, 2, "GRS80 +datum=NAD83 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Kentucky North\",GEOGCS[\"NAD83\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",637");
    add_srs_wkt (p, 2,
		 "8137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[\"standa");
    add_srs_wkt (p, 8,
		 "rd_parallel_1\",37.96666666666667],PARAMETER[\"standard_");
    add_srs_wkt (p, 9,
		 "parallel_2\",38.96666666666667],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 10,
		 "origin\",37.5],PARAMETER[\"central_meridian\",-84.25],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_easting\",500000],PARAMETER[\"false_nort");
    add_srs_wkt (p, 12,
		 "hing\",0],AUTHORITY[\"EPSG\",\"2205\"],AXIS[\"X\",EAST],");
    add_srs_wkt (p, 13, "AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2206, "epsg", 2206,
		      "ED50 / 3-degree Gauss-Kruger zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=9500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 9\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",27],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",950");
    add_srs_wkt (p, 10,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2206\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2207, "epsg", 2207,
		      "ED50 / 3-degree Gauss-Kruger zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=10500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 10\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",30],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",105");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2207\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2208, "epsg", 2208,
		      "ED50 / 3-degree Gauss-Kruger zone 11");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=11500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 11\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",33],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",115");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2208\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2209, "epsg", 2209,
		      "ED50 / 3-degree Gauss-Kruger zone 12");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=12500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 12\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",36],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",125");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2209\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2210, "epsg", 2210,
		      "ED50 / 3-degree Gauss-Kruger zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=13500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 13\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",39],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",135");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2210\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2211, "epsg", 2211,
		      "ED50 / 3-degree Gauss-Kruger zone 14");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=14500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 14\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",42],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",145");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2211\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2212, "epsg", 2212,
		      "ED50 / 3-degree Gauss-Kruger zone 15");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=15500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / 3-degree Gauss-Kruger zone 15\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "ED50\",DATUM[\"European_Datum_1950\",SPHEROID[\"Internat");
    add_srs_wkt (p, 2,
		 "ional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6230\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4230\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 8,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",45],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",155");
    add_srs_wkt (p, 10,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 11, ",\"2212\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2213, "epsg", 2213, "ETRS89 / TM 30 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=0.9996 +x_0=500000 +y_");
    add_proj4text (p, 1, "0=0 +ellps=GRS80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ETRS89 / TM 30 NE\",GEOGCS[\"ETRS89\",DATUM[\"E");
    add_srs_wkt (p, 1,
		 "uropean_Terrestrial_Reference_System_1989\",SPHEROID[\"G");
    add_srs_wkt (p, 2,
		 "RS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"701");
    add_srs_wkt (p, 3,
		 "9\"]],AUTHORITY[\"EPSG\",\"6258\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4258\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",3");
    add_srs_wkt (p, 9,
		 "0],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "easting\",500000],PARAMETER[\"false_northing\",0],AUTHOR");
    add_srs_wkt (p, 11,
		 "ITY[\"EPSG\",\"2213\"],AXIS[\"Easting\",EAST],AXIS[\"Nor");
    add_srs_wkt (p, 12, "thing\",NORTH]]");
    p = add_epsg_def (first, last, 2214, "epsg", 2214,
		      "Douala 1948 / AOF west (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=10.5 +k=0.999 +x_0=1000000 +");
    add_proj4text (p, 1,
		   "y_0=1000000 +ellps=intl +towgs84=-206.1,-174.7,-87.7,0,0");
    add_proj4text (p, 2, ",0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Douala 1948 / AOF west (deprecated)\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Douala 1948\",DATUM[\"Douala_1948\",SPHEROID[\"Internati");
    add_srs_wkt (p, 2,
		 "onal 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[-206.1,-174.7,-87.7,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6192\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4192\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",10.5],PARAMETER[\"scale_fact");
    add_srs_wkt (p, 10,
		 "or\",0.999],PARAMETER[\"false_easting\",1000000],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",1000000],AUTHORITY[\"EPSG\",\"2214");
    add_srs_wkt (p, 12, "\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2215, "epsg", 2215,
		      "Manoca 1962 / UTM zone 32N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=32 +a=6378249.2 +b=6356515 +towgs84=-70.");
    add_proj4text (p, 1, "9,-151.8,-41.4,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Manoca 1962 / UTM zone 32N\",GEOGCS[\"Manoca 19");
    add_srs_wkt (p, 1,
		 "62\",DATUM[\"Manoca_1962\",SPHEROID[\"Clarke 1880 (IGN)\"");
    add_srs_wkt (p, 2,
		 ",6378249.2,293.4660212936265,AUTHORITY[\"EPSG\",\"7011\"");
    add_srs_wkt (p, 3,
		 "]],TOWGS84[-70.9,-151.8,-41.4,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6193\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4193\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"central_meridian\",9],PARAMETER[\"scale_facto");
    add_srs_wkt (p, 10,
		 "r\",0.9996],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2215\"],AXI");
    add_srs_wkt (p, 12, "S[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2216, "epsg", 2216,
		      "Qornoq 1927 / UTM zone 22N");
    add_proj4text (p, 0, "+proj=utm +zone=22 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Qornoq 1927 / UTM zone 22N\",GEOGCS[\"Qornoq 19");
    add_srs_wkt (p, 1,
		 "27\",DATUM[\"Qornoq_1927\",SPHEROID[\"International 1924");
    add_srs_wkt (p, 2,
		 "\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"6194\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4194\"]],U");
    add_srs_wkt (p, 6,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 7,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 8,
		 "\",0],PARAMETER[\"central_meridian\",-51],PARAMETER[\"sc");
    add_srs_wkt (p, 9,
		 "ale_factor\",0.9996],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"22");
    add_srs_wkt (p, 11,
		 "16\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2217, "epsg", 2217,
		      "Qornoq 1927 / UTM zone 23N");
    add_proj4text (p, 0, "+proj=utm +zone=23 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Qornoq 1927 / UTM zone 23N\",GEOGCS[\"Qornoq 19");
    add_srs_wkt (p, 1,
		 "27\",DATUM[\"Qornoq_1927\",SPHEROID[\"International 1924");
    add_srs_wkt (p, 2,
		 "\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"6194\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4194\"]],U");
    add_srs_wkt (p, 6,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 7,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 8,
		 "\",0],PARAMETER[\"central_meridian\",-45],PARAMETER[\"sc");
    add_srs_wkt (p, 9,
		 "ale_factor\",0.9996],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"22");
    add_srs_wkt (p, 11,
		 "17\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2219, "epsg", 2219, "ATS77 / UTM zone 19N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=19 +a=6378135 +b=6356750.304921594 +unit");
    add_proj4text (p, 1, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / UTM zone 19N\",GEOGCS[\"ATS77\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "Average_Terrestrial_System_1977\",SPHEROID[\"Average Ter");
    add_srs_wkt (p, 2,
		 "restrial System 1977\",6378135,298.257,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7041\"]],AUTHORITY[\"EPSG\",\"6122\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4122\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-69],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2219\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2220, "epsg", 2220, "ATS77 / UTM zone 20N");
    add_proj4text (p, 0,
		   "+proj=utm +zone=20 +a=6378135 +b=6356750.304921594 +unit");
    add_proj4text (p, 1, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / UTM zone 20N\",GEOGCS[\"ATS77\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "Average_Terrestrial_System_1977\",SPHEROID[\"Average Ter");
    add_srs_wkt (p, 2,
		 "restrial System 1977\",6378135,298.257,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7041\"]],AUTHORITY[\"EPSG\",\"6122\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 4,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 5,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 6,
		 "TY[\"EPSG\",\"4122\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 7,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 8,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 9,
		 "an\",-63],PARAMETER[\"scale_factor\",0.9996],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2220\"],AXIS[\"Easting\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2222, "epsg", 2222,
		      "NAD83 / Arizona East (ft)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-110.1666666666667 +k=0.999");
    add_proj4text (p, 1,
		   "9 +x_0=213360 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft");
    add_proj4text (p, 2, " +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Arizona East (ft)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",31],PARAMETER[\"central_meridian\",-110.1");
    add_srs_wkt (p, 9,
		 "666666666667],PARAMETER[\"scale_factor\",0.9999],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_easting\",700000],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 11,
		 ",0],AUTHORITY[\"EPSG\",\"2222\"],AXIS[\"X\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Y\",NORTH]]");
    p = add_epsg_def (first, last, 2223, "epsg", 2223,
		      "NAD83 / Arizona Central (ft)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-111.9166666666667 +k=0.999");
    add_proj4text (p, 1,
		   "9 +x_0=213360 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft");
    add_proj4text (p, 2, " +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Arizona Central (ft)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 8,
		 "ude_of_origin\",31],PARAMETER[\"central_meridian\",-111.");
    add_srs_wkt (p, 9,
		 "9166666666667],PARAMETER[\"scale_factor\",0.9999],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_easting\",700000],PARAMETER[\"false_northing");
    add_srs_wkt (p, 11,
		 "\",0],AUTHORITY[\"EPSG\",\"2223\"],AXIS[\"X\",EAST],AXIS");
    add_srs_wkt (p, 12, "[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2224, "epsg", 2224,
		      "NAD83 / Arizona West (ft)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-113.75 +k=0.999933333 +x_0");
    add_proj4text (p, 1,
		   "=213360 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft +no_d");
    add_proj4text (p, 2, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Arizona West (ft)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",31],PARAMETER[\"central_meridian\",-113.7");
    add_srs_wkt (p, 9,
		 "5],PARAMETER[\"scale_factor\",0.999933333],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_easting\",700000],PARAMETER[\"false_northing\",0],A");
    add_srs_wkt (p, 11,
		 "UTHORITY[\"EPSG\",\"2224\"],AXIS[\"X\",EAST],AXIS[\"Y\",");
    add_srs_wkt (p, 12, "NORTH]]");
    p = add_epsg_def (first, last, 2225, "epsg", 2225,
		      "NAD83 / California zone 1 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.66666666666666 +lat_2=40 +lat_0=39.3");
    add_proj4text (p, 1,
		   "3333333333334 +lon_0=-122 +x_0=2000000.0001016 +y_0=5000");
    add_proj4text (p, 2,
		   "00.0001016001 +ellps=GRS80 +datum=NAD83 +units=us-ft +no");
    add_proj4text (p, 3, "_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 1 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",41.");
    add_srs_wkt (p, 9,
		 "66666666666666],PARAMETER[\"standard_parallel_2\",40],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"latitude_of_origin\",39.33333333333334],PARAME");
    add_srs_wkt (p, 11,
		 "TER[\"central_meridian\",-122],PARAMETER[\"false_easting");
    add_srs_wkt (p, 12,
		 "\",6561666.667],PARAMETER[\"false_northing\",1640416.667");
    add_srs_wkt (p, 13,
		 "],AUTHORITY[\"EPSG\",\"2225\"],AXIS[\"X\",EAST],AXIS[\"Y");
    add_srs_wkt (p, 14, "\",NORTH]]");
    p = add_epsg_def (first, last, 2226, "epsg", 2226,
		      "NAD83 / California zone 2 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=39.83333333333334 +lat_2=38.33333333333");
    add_proj4text (p, 1,
		   "334 +lat_0=37.66666666666666 +lon_0=-122 +x_0=2000000.00");
    add_proj4text (p, 2,
		   "01016 +y_0=500000.0001016001 +ellps=GRS80 +datum=NAD83 +");
    add_proj4text (p, 3, "units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 2 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",39.");
    add_srs_wkt (p, 9,
		 "83333333333334],PARAMETER[\"standard_parallel_2\",38.333");
    add_srs_wkt (p, 10,
		 "33333333334],PARAMETER[\"latitude_of_origin\",37.6666666");
    add_srs_wkt (p, 11,
		 "6666666],PARAMETER[\"central_meridian\",-122],PARAMETER[");
    add_srs_wkt (p, 12,
		 "\"false_easting\",6561666.667],PARAMETER[\"false_northin");
    add_srs_wkt (p, 13,
		 "g\",1640416.667],AUTHORITY[\"EPSG\",\"2226\"],AXIS[\"X\"");
    add_srs_wkt (p, 14, ",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2227, "epsg", 2227,
		      "NAD83 / California zone 3 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=38.43333333333333 +lat_2=37.06666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=36.5 +lon_0=-120.5 +x_0=2000000.0001016 +y_0=");
    add_proj4text (p, 2,
		   "500000.0001016001 +ellps=GRS80 +datum=NAD83 +units=us-ft");
    add_proj4text (p, 3, " +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 3 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",38.");
    add_srs_wkt (p, 9,
		 "43333333333333],PARAMETER[\"standard_parallel_2\",37.066");
    add_srs_wkt (p, 10,
		 "66666666667],PARAMETER[\"latitude_of_origin\",36.5],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"central_meridian\",-120.5],PARAMETER[\"false_eas");
    add_srs_wkt (p, 12,
		 "ting\",6561666.667],PARAMETER[\"false_northing\",1640416");
    add_srs_wkt (p, 13,
		 ".667],AUTHORITY[\"EPSG\",\"2227\"],AXIS[\"X\",EAST],AXIS");
    add_srs_wkt (p, 14, "[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2228, "epsg", 2228,
		      "NAD83 / California zone 4 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=37.25 +lat_2=36 +lat_0=35.3333333333333");
    add_proj4text (p, 1,
		   "4 +lon_0=-119 +x_0=2000000.0001016 +y_0=500000.000101600");
    add_proj4text (p, 2, "1 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 4 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",37.");
    add_srs_wkt (p, 9,
		 "25],PARAMETER[\"standard_parallel_2\",36],PARAMETER[\"la");
    add_srs_wkt (p, 10,
		 "titude_of_origin\",35.33333333333334],PARAMETER[\"centra");
    add_srs_wkt (p, 11,
		 "l_meridian\",-119],PARAMETER[\"false_easting\",6561666.6");
    add_srs_wkt (p, 12,
		 "67],PARAMETER[\"false_northing\",1640416.667],AUTHORITY[");
    add_srs_wkt (p, 13,
		 "\"EPSG\",\"2228\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2229, "epsg", 2229,
		      "NAD83 / California zone 5 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=35.46666666666667 +lat_2=34.03333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=33.5 +lon_0=-118 +x_0=2000000.0001016 +y_0=50");
    add_proj4text (p, 2,
		   "0000.0001016001 +ellps=GRS80 +datum=NAD83 +units=us-ft +");
    add_proj4text (p, 3, "no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 5 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",35.");
    add_srs_wkt (p, 9,
		 "46666666666667],PARAMETER[\"standard_parallel_2\",34.033");
    add_srs_wkt (p, 10,
		 "33333333333],PARAMETER[\"latitude_of_origin\",33.5],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"central_meridian\",-118],PARAMETER[\"false_easti");
    add_srs_wkt (p, 12,
		 "ng\",6561666.667],PARAMETER[\"false_northing\",1640416.6");
    add_srs_wkt (p, 13,
		 "67],AUTHORITY[\"EPSG\",\"2229\"],AXIS[\"X\",EAST],AXIS[\"");
    add_srs_wkt (p, 14, "Y\",NORTH]]");
    p = add_epsg_def (first, last, 2230, "epsg", 2230,
		      "NAD83 / California zone 6 (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=33.88333333333333 +lat_2=32.78333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=32.16666666666666 +lon_0=-116.25 +x_0=2000000");
    add_proj4text (p, 2,
		   ".0001016 +y_0=500000.0001016001 +ellps=GRS80 +datum=NAD8");
    add_proj4text (p, 3, "3 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / California zone 6 (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",33.");
    add_srs_wkt (p, 9,
		 "88333333333333],PARAMETER[\"standard_parallel_2\",32.783");
    add_srs_wkt (p, 10,
		 "33333333333],PARAMETER[\"latitude_of_origin\",32.1666666");
    add_srs_wkt (p, 11,
		 "6666666],PARAMETER[\"central_meridian\",-116.25],PARAMET");
    add_srs_wkt (p, 12,
		 "ER[\"false_easting\",6561666.667],PARAMETER[\"false_nort");
    add_srs_wkt (p, 13,
		 "hing\",1640416.667],AUTHORITY[\"EPSG\",\"2230\"],AXIS[\"");
    add_srs_wkt (p, 14, "X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2231, "epsg", 2231,
		      "NAD83 / Colorado North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=40.78333333333333 +lat_2=39.71666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=39.33333333333334 +lon_0=-105.5 +x_0=914401.8");
    add_proj4text (p, 2,
		   "288036576 +y_0=304800.6096012192 +ellps=GRS80 +datum=NAD");
    add_proj4text (p, 3, "83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Colorado North (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",40.78333");
    add_srs_wkt (p, 9,
		 "333333333],PARAMETER[\"standard_parallel_2\",39.71666666");
    add_srs_wkt (p, 10,
		 "666667],PARAMETER[\"latitude_of_origin\",39.333333333333");
    add_srs_wkt (p, 11,
		 "34],PARAMETER[\"central_meridian\",-105.5],PARAMETER[\"f");
    add_srs_wkt (p, 12,
		 "alse_easting\",3000000],PARAMETER[\"false_northing\",100");
    add_srs_wkt (p, 13,
		 "0000],AUTHORITY[\"EPSG\",\"2231\"],AXIS[\"X\",EAST],AXIS");
    add_srs_wkt (p, 14, "[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2232, "epsg", 2232,
		      "NAD83 / Colorado Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=39.75 +lat_2=38.45 +lat_0=37.8333333333");
    add_proj4text (p, 1,
		   "3334 +lon_0=-105.5 +x_0=914401.8288036576 +y_0=304800.60");
    add_proj4text (p, 2,
		   "96012192 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_proj4text (p, 3, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Colorado Central (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Confor");
    add_srs_wkt (p, 8,
		 "mal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",39.75");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"standard_parallel_2\",38.45],PARAMETER[\"l");
    add_srs_wkt (p, 10,
		 "atitude_of_origin\",37.83333333333334],PARAMETER[\"centr");
    add_srs_wkt (p, 11,
		 "al_meridian\",-105.5],PARAMETER[\"false_easting\",300000");
    add_srs_wkt (p, 12,
		 "0],PARAMETER[\"false_northing\",1000000],AUTHORITY[\"EPS");
    add_srs_wkt (p, 13, "G\",\"2232\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2233, "epsg", 2233,
		      "NAD83 / Colorado South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=38.43333333333333 +lat_2=37.23333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=36.66666666666666 +lon_0=-105.5 +x_0=914401.8");
    add_proj4text (p, 2,
		   "288036576 +y_0=304800.6096012192 +ellps=GRS80 +datum=NAD");
    add_proj4text (p, 3, "83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Colorado South (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",38.43333");
    add_srs_wkt (p, 9,
		 "333333333],PARAMETER[\"standard_parallel_2\",37.23333333");
    add_srs_wkt (p, 10,
		 "333333],PARAMETER[\"latitude_of_origin\",36.666666666666");
    add_srs_wkt (p, 11,
		 "66],PARAMETER[\"central_meridian\",-105.5],PARAMETER[\"f");
    add_srs_wkt (p, 12,
		 "alse_easting\",3000000],PARAMETER[\"false_northing\",100");
    add_srs_wkt (p, 13,
		 "0000],AUTHORITY[\"EPSG\",\"2233\"],AXIS[\"X\",EAST],AXIS");
    add_srs_wkt (p, 14, "[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2234, "epsg", 2234,
		      "NAD83 / Connecticut (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.86666666666667 +lat_2=41.2 +lat_0=40");
    add_proj4text (p, 1,
		   ".83333333333334 +lon_0=-72.75 +x_0=304800.6096012192 +y_");
    add_proj4text (p, 2,
		   "0=152400.3048006096 +ellps=GRS80 +datum=NAD83 +units=us-");
    add_proj4text (p, 3, "ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Connecticut (ftUS)\",GEOGCS[\"NAD83\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_");
    add_srs_wkt (p, 8,
		 "Conic_2SP\"],PARAMETER[\"standard_parallel_1\",41.866666");
    add_srs_wkt (p, 9,
		 "66666667],PARAMETER[\"standard_parallel_2\",41.2],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"latitude_of_origin\",40.83333333333334],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"central_meridian\",-72.75],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 12,
		 ",1000000],PARAMETER[\"false_northing\",500000],AUTHORITY");
    add_srs_wkt (p, 13,
		 "[\"EPSG\",\"2234\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2235, "epsg", 2235,
		      "NAD83 / Delaware (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=38 +lon_0=-75.41666666666667 +k=0.999");
    add_proj4text (p, 1,
		   "995 +x_0=200000.0001016002 +y_0=0 +ellps=GRS80 +datum=NA");
    add_proj4text (p, 2, "D83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Delaware (ftUS)\",GEOGCS[\"NAD83\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",63");
    add_srs_wkt (p, 2,
		 "78137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercator\"");
    add_srs_wkt (p, 8,
		 "],PARAMETER[\"latitude_of_origin\",38],PARAMETER[\"centr");
    add_srs_wkt (p, 9,
		 "al_meridian\",-75.41666666666667],PARAMETER[\"scale_fact");
    add_srs_wkt (p, 10,
		 "or\",0.999995],PARAMETER[\"false_easting\",656166.667],P");
    add_srs_wkt (p, 11,
		 "ARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2235");
    add_srs_wkt (p, 12, "\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2236, "epsg", 2236,
		      "NAD83 / Florida East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=24.33333333333333 +lon_0=-81 +k=0.999");
    add_proj4text (p, 1,
		   "941177 +x_0=200000.0001016002 +y_0=0 +ellps=GRS80 +datum");
    add_proj4text (p, 2, "=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Florida East (ftUS)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercato");
    add_srs_wkt (p, 8,
		 "r\"],PARAMETER[\"latitude_of_origin\",24.33333333333333]");
    add_srs_wkt (p, 9,
		 ",PARAMETER[\"central_meridian\",-81],PARAMETER[\"scale_f");
    add_srs_wkt (p, 10,
		 "actor\",0.999941177],PARAMETER[\"false_easting\",656166.");
    add_srs_wkt (p, 11,
		 "667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2236\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2237, "epsg", 2237,
		      "NAD83 / Florida West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=24.33333333333333 +lon_0=-82 +k=0.999");
    add_proj4text (p, 1,
		   "941177 +x_0=200000.0001016002 +y_0=0 +ellps=GRS80 +datum");
    add_proj4text (p, 2, "=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Florida West (ftUS)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercato");
    add_srs_wkt (p, 8,
		 "r\"],PARAMETER[\"latitude_of_origin\",24.33333333333333]");
    add_srs_wkt (p, 9,
		 ",PARAMETER[\"central_meridian\",-82],PARAMETER[\"scale_f");
    add_srs_wkt (p, 10,
		 "actor\",0.999941177],PARAMETER[\"false_easting\",656166.");
    add_srs_wkt (p, 11,
		 "667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2237\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2238, "epsg", 2238,
		      "NAD83 / Florida North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=30.75 +lat_2=29.58333333333333 +lat_0=2");
    add_proj4text (p, 1,
		   "9 +lon_0=-84.5 +x_0=600000 +y_0=0 +ellps=GRS80 +datum=NA");
    add_proj4text (p, 2, "D83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Florida North (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",30.75],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"standard_parallel_2\",29.58333333333333],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"latitude_of_origin\",29],PARAMETER[\"central_mer");
    add_srs_wkt (p, 11,
		 "idian\",-84.5],PARAMETER[\"false_easting\",1968500],PARA");
    add_srs_wkt (p, 12,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2238\"]");
    add_srs_wkt (p, 13, ",AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2239, "epsg", 2239,
		      "NAD83 / Georgia East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=30 +lon_0=-82.16666666666667 +k=0.999");
    add_proj4text (p, 1,
		   "9 +x_0=200000.0001016002 +y_0=0 +ellps=GRS80 +datum=NAD8");
    add_proj4text (p, 2, "3 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Georgia East (ftUS)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercato");
    add_srs_wkt (p, 8,
		 "r\"],PARAMETER[\"latitude_of_origin\",30],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",-82.16666666666667],PARAMETER[\"scale_f");
    add_srs_wkt (p, 10,
		 "actor\",0.9999],PARAMETER[\"false_easting\",656166.667],");
    add_srs_wkt (p, 11,
		 "PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"223");
    add_srs_wkt (p, 12, "9\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2240, "epsg", 2240,
		      "NAD83 / Georgia West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=30 +lon_0=-84.16666666666667 +k=0.999");
    add_proj4text (p, 1,
		   "9 +x_0=699999.9998983998 +y_0=0 +ellps=GRS80 +datum=NAD8");
    add_proj4text (p, 2, "3 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Georgia West (ftUS)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercato");
    add_srs_wkt (p, 8,
		 "r\"],PARAMETER[\"latitude_of_origin\",30],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",-84.16666666666667],PARAMETER[\"scale_f");
    add_srs_wkt (p, 10,
		 "actor\",0.9999],PARAMETER[\"false_easting\",2296583.333]");
    add_srs_wkt (p, 11,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"22");
    add_srs_wkt (p, 12, "40\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2241, "epsg", 2241,
		      "NAD83 / Idaho East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=41.66666666666666 +lon_0=-112.1666666");
    add_proj4text (p, 1,
		   "666667 +k=0.9999473679999999 +x_0=200000.0001016002 +y_0");
    add_proj4text (p, 2, "=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Idaho East (ftUS)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",41.66666666666666],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",-112.1666666666667],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",0.999947368],PARAMETER[\"false_eas");
    add_srs_wkt (p, 11,
		 "ting\",656166.667],PARAMETER[\"false_northing\",0],AUTHO");
    add_srs_wkt (p, 12,
		 "RITY[\"EPSG\",\"2241\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORT");
    add_srs_wkt (p, 13, "H]]");
    p = add_epsg_def (first, last, 2242, "epsg", 2242,
		      "NAD83 / Idaho Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=41.66666666666666 +lon_0=-114 +k=0.99");
    add_proj4text (p, 1,
		   "99473679999999 +x_0=500000.0001016001 +y_0=0 +ellps=GRS8");
    add_proj4text (p, 2, "0 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Idaho Central (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",41.66666666666666");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-114],PARAMETER[\"scale");
    add_srs_wkt (p, 10,
		 "_factor\",0.999947368],PARAMETER[\"false_easting\",16404");
    add_srs_wkt (p, 11,
		 "16.667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2242\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2243, "epsg", 2243,
		      "NAD83 / Idaho West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=41.66666666666666 +lon_0=-115.75 +k=0");
    add_proj4text (p, 1,
		   ".999933333 +x_0=800000.0001016001 +y_0=0 +ellps=GRS80 +d");
    add_proj4text (p, 2, "atum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Idaho West (ftUS)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHOR");
    add_srs_wkt (p, 7,
		 "ITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercator");
    add_srs_wkt (p, 8,
		 "\"],PARAMETER[\"latitude_of_origin\",41.66666666666666],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",-115.75],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",0.999933333],PARAMETER[\"false_easting\",2624");
    add_srs_wkt (p, 11,
		 "666.667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2243\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2244, "epsg", 2244,
		      "NAD83 / Indiana East (ftUS) (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=37.5 +lon_0=-85.66666666666667 +k=0.9");
    add_proj4text (p, 1,
		   "99966667 +x_0=99999.99989839978 +y_0=249364.9987299975 +");
    add_proj4text (p, 2, "ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Indiana East (ftUS) (deprecated)\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"NAD83\",DATUM[\"North_American_Datum_1983\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7019\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4269\"]],UNIT[\"US survey foot\",0.30480060");
    add_srs_wkt (p, 7,
		 "96012192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",37.5");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-85.66666666666667],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",0.999966667],PARAMETER[\"false_e");
    add_srs_wkt (p, 11,
		 "asting\",328083.333],PARAMETER[\"false_northing\",818125");
    add_srs_wkt (p, 12,
		 "],AUTHORITY[\"EPSG\",\"2244\"],AXIS[\"X\",EAST],AXIS[\"Y");
    add_srs_wkt (p, 13, "\",NORTH]]");
    p = add_epsg_def (first, last, 2245, "epsg", 2245,
		      "NAD83 / Indiana West (ftUS) (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=37.5 +lon_0=-87.08333333333333 +k=0.9");
    add_proj4text (p, 1,
		   "99966667 +x_0=900000 +y_0=249364.9987299975 +ellps=GRS80");
    add_proj4text (p, 2, " +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Indiana West (ftUS) (deprecated)\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"NAD83\",DATUM[\"North_American_Datum_1983\",SPHEROI");
    add_srs_wkt (p, 2,
		 "D[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7019\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 4,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 5,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"4269\"]],UNIT[\"US survey foot\",0.30480060");
    add_srs_wkt (p, 7,
		 "96012192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",37.5");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-87.08333333333333],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",0.999966667],PARAMETER[\"false_e");
    add_srs_wkt (p, 11,
		 "asting\",2952750],PARAMETER[\"false_northing\",818125],A");
    add_srs_wkt (p, 12,
		 "UTHORITY[\"EPSG\",\"2245\"],AXIS[\"X\",EAST],AXIS[\"Y\",");
    add_srs_wkt (p, 13, "NORTH]]");
    p = add_epsg_def (first, last, 2246, "epsg", 2246,
		      "NAD83 / Kentucky North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=37.96666666666667 +lat_2=38.96666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=37.5 +lon_0=-84.25 +x_0=500000.0001016001 +y_");
    add_proj4text (p, 2, "0=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Kentucky North (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",37.96666");
    add_srs_wkt (p, 9,
		 "666666667],PARAMETER[\"standard_parallel_2\",38.96666666");
    add_srs_wkt (p, 10,
		 "666667],PARAMETER[\"latitude_of_origin\",37.5],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"central_meridian\",-84.25],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 12,
		 ",1640416.667],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 13,
		 "\"EPSG\",\"2246\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2247, "epsg", 2247,
		      "NAD83 / Kentucky South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=37.93333333333333 +lat_2=36.73333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=36.33333333333334 +lon_0=-85.75 +x_0=500000.0");
    add_proj4text (p, 2,
		   "001016001 +y_0=500000.0001016001 +ellps=GRS80 +datum=NAD");
    add_proj4text (p, 3, "83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Kentucky South (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",37.93333");
    add_srs_wkt (p, 9,
		 "333333333],PARAMETER[\"standard_parallel_2\",36.73333333");
    add_srs_wkt (p, 10,
		 "333333],PARAMETER[\"latitude_of_origin\",36.333333333333");
    add_srs_wkt (p, 11,
		 "34],PARAMETER[\"central_meridian\",-85.75],PARAMETER[\"f");
    add_srs_wkt (p, 12,
		 "alse_easting\",1640416.667],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 13,
		 ",1640416.667],AUTHORITY[\"EPSG\",\"2247\"],AXIS[\"X\",EA");
    add_srs_wkt (p, 14, "ST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2248, "epsg", 2248,
		      "NAD83 / Maryland (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=39.45 +lat_2=38.3 +lat_0=37.66666666666");
    add_proj4text (p, 1,
		   "666 +lon_0=-77 +x_0=399999.9998983998 +y_0=0 +ellps=GRS8");
    add_proj4text (p, 2, "0 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Maryland (ftUS)\",GEOGCS[\"NAD83\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",63");
    add_srs_wkt (p, 2,
		 "78137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_Con");
    add_srs_wkt (p, 8,
		 "ic_2SP\"],PARAMETER[\"standard_parallel_1\",39.45],PARAM");
    add_srs_wkt (p, 9,
		 "ETER[\"standard_parallel_2\",38.3],PARAMETER[\"latitude_");
    add_srs_wkt (p, 10,
		 "of_origin\",37.66666666666666],PARAMETER[\"central_merid");
    add_srs_wkt (p, 11,
		 "ian\",-77],PARAMETER[\"false_easting\",1312333.333],PARA");
    add_srs_wkt (p, 12,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2248\"]");
    add_srs_wkt (p, 13, ",AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2249, "epsg", 2249,
		      "NAD83 / Massachusetts Mainland (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=42.68333333333333 +lat_2=41.71666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=41 +lon_0=-71.5 +x_0=200000.0001016002 +y_0=7");
    add_proj4text (p, 2,
		   "50000 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Massachusetts Mainland (ftUS)\",GEOGCS[");
    add_srs_wkt (p, 1,
		 "\"NAD83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 3,
		 "19\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012");
    add_srs_wkt (p, 7,
		 "192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_C");
    add_srs_wkt (p, 8,
		 "onformal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",");
    add_srs_wkt (p, 9,
		 "42.68333333333333],PARAMETER[\"standard_parallel_2\",41.");
    add_srs_wkt (p, 10,
		 "71666666666667],PARAMETER[\"latitude_of_origin\",41],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"central_meridian\",-71.5],PARAMETER[\"false_eas");
    add_srs_wkt (p, 12,
		 "ting\",656166.667],PARAMETER[\"false_northing\",2460625]");
    add_srs_wkt (p, 13,
		 ",AUTHORITY[\"EPSG\",\"2249\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 14, ",NORTH]]");
    p = add_epsg_def (first, last, 2250, "epsg", 2250,
		      "NAD83 / Massachusetts Island (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.48333333333333 +lat_2=41.28333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=41 +lon_0=-70.5 +x_0=500000.0001016001 +y_0=0");
    add_proj4text (p, 2, " +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Massachusetts Island (ftUS)\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "NAD83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"G");
    add_srs_wkt (p, 2,
		 "RS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"701");
    add_srs_wkt (p, 3,
		 "9\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012");
    add_srs_wkt (p, 7,
		 "192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_C");
    add_srs_wkt (p, 8,
		 "onformal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",");
    add_srs_wkt (p, 9,
		 "41.48333333333333],PARAMETER[\"standard_parallel_2\",41.");
    add_srs_wkt (p, 10,
		 "28333333333333],PARAMETER[\"latitude_of_origin\",41],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"central_meridian\",-70.5],PARAMETER[\"false_eas");
    add_srs_wkt (p, 12,
		 "ting\",1640416.667],PARAMETER[\"false_northing\",0],AUTH");
    add_srs_wkt (p, 13,
		 "ORITY[\"EPSG\",\"2250\"],AXIS[\"X\",EAST],AXIS[\"Y\",NOR");
    add_srs_wkt (p, 14, "TH]]");
    p = add_epsg_def (first, last, 2251, "epsg", 2251,
		      "NAD83 / Michigan North (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=47.08333333333334 +lat_2=45.48333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=44.78333333333333 +lon_0=-87 +x_0=7999999.999");
    add_proj4text (p, 2,
		   "968001 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft +no_de");
    add_proj4text (p, 3, "fs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Michigan North (ft)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER");
    add_srs_wkt (p, 8,
		 "[\"standard_parallel_1\",47.08333333333334],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "standard_parallel_2\",45.48333333333333],PARAMETER[\"lat");
    add_srs_wkt (p, 10,
		 "itude_of_origin\",44.78333333333333],PARAMETER[\"central");
    add_srs_wkt (p, 11,
		 "_meridian\",-87],PARAMETER[\"false_easting\",26246719.16");
    add_srs_wkt (p, 12,
		 "],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 13, "251\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2252, "epsg", 2252,
		      "NAD83 / Michigan Central (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=45.7 +lat_2=44.18333333333333 +lat_0=43");
    add_proj4text (p, 1,
		   ".31666666666667 +lon_0=-84.36666666666666 +x_0=5999999.9");
    add_proj4text (p, 2,
		   "99976001 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft +no_");
    add_proj4text (p, 3, "defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Michigan Central (ft)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER");
    add_srs_wkt (p, 8,
		 "[\"standard_parallel_1\",45.7],PARAMETER[\"standard_para");
    add_srs_wkt (p, 9,
		 "llel_2\",44.18333333333333],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 10,
		 "in\",43.31666666666667],PARAMETER[\"central_meridian\",-");
    add_srs_wkt (p, 11,
		 "84.36666666666666],PARAMETER[\"false_easting\",19685039.");
    add_srs_wkt (p, 12,
		 "37],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 13, "2252\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2253, "epsg", 2253,
		      "NAD83 / Michigan South (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=43.66666666666666 +lat_2=42.1 +lat_0=41");
    add_proj4text (p, 1,
		   ".5 +lon_0=-84.36666666666666 +x_0=3999999.999984 +y_0=0 ");
    add_proj4text (p, 2, "+ellps=GRS80 +datum=NAD83 +units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Michigan South (ft)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER");
    add_srs_wkt (p, 8,
		 "[\"standard_parallel_1\",43.66666666666666],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "standard_parallel_2\",42.1],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 10,
		 "in\",41.5],PARAMETER[\"central_meridian\",-84.3666666666");
    add_srs_wkt (p, 11,
		 "6666],PARAMETER[\"false_easting\",13123359.58],PARAMETER");
    add_srs_wkt (p, 12,
		 "[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2253\"],AXIS");
    add_srs_wkt (p, 13, "[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2254, "epsg", 2254,
		      "NAD83 / Mississippi East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=29.5 +lon_0=-88.83333333333333 +k=0.9");
    add_proj4text (p, 1,
		   "9995 +x_0=300000.0000000001 +y_0=0 +ellps=GRS80 +datum=N");
    add_proj4text (p, 2, "AD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Mississippi East (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",29.5],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"central_meridian\",-88.83333333333333],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",0.99995],PARAMETER[\"false_easting\",98425");
    add_srs_wkt (p, 11,
		 "0.0000000002],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2254\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2255, "epsg", 2255,
		      "NAD83 / Mississippi West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=29.5 +lon_0=-90.33333333333333 +k=0.9");
    add_proj4text (p, 1,
		   "9995 +x_0=699999.9998983998 +y_0=0 +ellps=GRS80 +datum=N");
    add_proj4text (p, 2, "AD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Mississippi West (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",29.5],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"central_meridian\",-90.33333333333333],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",0.99995],PARAMETER[\"false_easting\",22965");
    add_srs_wkt (p, 11,
		 "83.333],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2255\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2256, "epsg", 2256, "NAD83 / Montana (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=49 +lat_2=45 +lat_0=44.25 +lon_0=-109.5");
    add_proj4text (p, 1,
		   " +x_0=599999.9999976 +y_0=0 +ellps=GRS80 +datum=NAD83 +u");
    add_proj4text (p, 2, "nits=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Montana (ft)\",GEOGCS[\"NAD83\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "North_American_Datum_1983\",SPHEROID[\"GRS 1980\",637813");
    add_srs_wkt (p, 2,
		 "7,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHORITY[");
    add_srs_wkt (p, 3,
		 "\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 4,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 5,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269\"]]");
    add_srs_wkt (p, 6,
		 ",UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[\"stan");
    add_srs_wkt (p, 8,
		 "dard_parallel_1\",49],PARAMETER[\"standard_parallel_2\",");
    add_srs_wkt (p, 9,
		 "45],PARAMETER[\"latitude_of_origin\",44.25],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "central_meridian\",-109.5],PARAMETER[\"false_easting\",1");
    add_srs_wkt (p, 11,
		 "968503.937],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 12, "EPSG\",\"2256\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2257, "epsg", 2257,
		      "NAD83 / New Mexico East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-104.3333333333333 +k=0.999");
    add_proj4text (p, 1,
		   "909091 +x_0=165000 +y_0=0 +ellps=GRS80 +datum=NAD83 +uni");
    add_proj4text (p, 2, "ts=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New Mexico East (ftUS)\",GEOGCS[\"NAD83");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",31],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-104.3333333333333],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",0.999909091],PARAMETER[\"false_easting\",541");
    add_srs_wkt (p, 11,
		 "337.5],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2257\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2258, "epsg", 2258,
		      "NAD83 / New Mexico Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-106.25 +k=0.9999 +x_0=5000");
    add_proj4text (p, 1,
		   "00.0001016001 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=us");
    add_proj4text (p, 2, "-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New Mexico Central (ftUS)\",GEOGCS[\"NA");
    add_srs_wkt (p, 1,
		 "D83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_M");
    add_srs_wkt (p, 8,
		 "ercator\"],PARAMETER[\"latitude_of_origin\",31],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"central_meridian\",-106.25],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",0.9999],PARAMETER[\"false_easting\",1640416.667],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2258\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2259, "epsg", 2259,
		      "NAD83 / New Mexico West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=31 +lon_0=-107.8333333333333 +k=0.999");
    add_proj4text (p, 1,
		   "916667 +x_0=830000.0001016001 +y_0=0 +ellps=GRS80 +datum");
    add_proj4text (p, 2, "=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New Mexico West (ftUS)\",GEOGCS[\"NAD83");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",31],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-107.8333333333333],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",0.999916667],PARAMETER[\"false_easting\",272");
    add_srs_wkt (p, 11,
		 "3091.667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EP");
    add_srs_wkt (p, 12, "SG\",\"2259\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2260, "epsg", 2260,
		      "NAD83 / New York East (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=38.83333333333334 +lon_0=-74.5 +k=0.9");
    add_proj4text (p, 1,
		   "999 +x_0=150000 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=");
    add_proj4text (p, 2, "us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New York East (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",38.83333333333334");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"central_meridian\",-74.5],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",0.9999],PARAMETER[\"false_easting\",492125],P");
    add_srs_wkt (p, 11,
		 "ARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2260");
    add_srs_wkt (p, 12, "\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2261, "epsg", 2261,
		      "NAD83 / New York Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=40 +lon_0=-76.58333333333333 +k=0.999");
    add_proj4text (p, 1,
		   "9375 +x_0=249999.9998983998 +y_0=0 +ellps=GRS80 +datum=N");
    add_proj4text (p, 2, "AD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New York Central (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 8,
		 "cator\"],PARAMETER[\"latitude_of_origin\",40],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"central_meridian\",-76.58333333333333],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",0.9999375],PARAMETER[\"false_easting\",82020");
    add_srs_wkt (p, 11,
		 "8.3330000002],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 12,
		 "\"EPSG\",\"2261\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2262, "epsg", 2262,
		      "NAD83 / New York West (ftUS)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=40 +lon_0=-78.58333333333333 +k=0.999");
    add_proj4text (p, 1,
		   "9375 +x_0=350000.0001016001 +y_0=0 +ellps=GRS80 +datum=N");
    add_proj4text (p, 2, "AD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New York West (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",40],PARAMETER[\"c");
    add_srs_wkt (p, 9,
		 "entral_meridian\",-78.58333333333333],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9999375],PARAMETER[\"false_easting\",1148291.");
    add_srs_wkt (p, 11,
		 "667],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2262\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2263, "epsg", 2263,
		      "NAD83 / New York Long Island (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.03333333333333 +lat_2=40.66666666666");
    add_proj4text (p, 1,
		   "666 +lat_0=40.16666666666666 +lon_0=-74 +x_0=300000.0000");
    add_proj4text (p, 2,
		   "000001 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no");
    add_proj4text (p, 3, "_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / New York Long Island (ftUS)\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "NAD83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"G");
    add_srs_wkt (p, 2,
		 "RS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"701");
    add_srs_wkt (p, 3,
		 "9\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012");
    add_srs_wkt (p, 7,
		 "192,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_C");
    add_srs_wkt (p, 8,
		 "onformal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",");
    add_srs_wkt (p, 9,
		 "41.03333333333333],PARAMETER[\"standard_parallel_2\",40.");
    add_srs_wkt (p, 10,
		 "66666666666666],PARAMETER[\"latitude_of_origin\",40.1666");
    add_srs_wkt (p, 11,
		 "6666666666],PARAMETER[\"central_meridian\",-74],PARAMETE");
    add_srs_wkt (p, 12,
		 "R[\"false_easting\",984250.0000000002],PARAMETER[\"false");
    add_srs_wkt (p, 13,
		 "_northing\",0],AUTHORITY[\"EPSG\",\"2263\"],AXIS[\"X\",E");
    add_srs_wkt (p, 14, "AST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2264, "epsg", 2264,
		      "NAD83 / North Carolina (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=36.16666666666666 +lat_2=34.33333333333");
    add_proj4text (p, 1,
		   "334 +lat_0=33.75 +lon_0=-79 +x_0=609601.2192024384 +y_0=");
    add_proj4text (p, 2, "0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / North Carolina (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",36.16666");
    add_srs_wkt (p, 9,
		 "666666666],PARAMETER[\"standard_parallel_2\",34.33333333");
    add_srs_wkt (p, 10,
		 "333334],PARAMETER[\"latitude_of_origin\",33.75],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"central_meridian\",-79],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 12,
		 "2000000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 13, "G\",\"2264\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2265, "epsg", 2265,
		      "NAD83 / North Dakota North (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=48.73333333333333 +lat_2=47.43333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=47 +lon_0=-100.5 +x_0=599999.9999976 +y_0=0 +");
    add_proj4text (p, 2, "ellps=GRS80 +datum=NAD83 +units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / North Dakota North (ft)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "02\"]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAM");
    add_srs_wkt (p, 8,
		 "ETER[\"standard_parallel_1\",48.73333333333333],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"standard_parallel_2\",47.43333333333333],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "latitude_of_origin\",47],PARAMETER[\"central_meridian\",");
    add_srs_wkt (p, 11,
		 "-100.5],PARAMETER[\"false_easting\",1968503.937],PARAMET");
    add_srs_wkt (p, 12,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2265\"],AX");
    add_srs_wkt (p, 13, "IS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2266, "epsg", 2266,
		      "NAD83 / North Dakota South (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=47.48333333333333 +lat_2=46.18333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=45.66666666666666 +lon_0=-100.5 +x_0=599999.9");
    add_proj4text (p, 2,
		   "999976 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=ft +no_de");
    add_proj4text (p, 3, "fs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / North Dakota South (ft)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "02\"]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAM");
    add_srs_wkt (p, 8,
		 "ETER[\"standard_parallel_1\",47.48333333333333],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"standard_parallel_2\",46.18333333333333],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "latitude_of_origin\",45.66666666666666],PARAMETER[\"cent");
    add_srs_wkt (p, 11,
		 "ral_meridian\",-100.5],PARAMETER[\"false_easting\",19685");
    add_srs_wkt (p, 12,
		 "03.937],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 13, "\",\"2266\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2267, "epsg", 2267,
		      "NAD83 / Oklahoma North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=36.76666666666667 +lat_2=35.56666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=35 +lon_0=-98 +x_0=600000 +y_0=0 +ellps=GRS80");
    add_proj4text (p, 2, " +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Oklahoma North (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",36.76666");
    add_srs_wkt (p, 9,
		 "666666667],PARAMETER[\"standard_parallel_2\",35.56666666");
    add_srs_wkt (p, 10,
		 "666667],PARAMETER[\"latitude_of_origin\",35],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "central_meridian\",-98],PARAMETER[\"false_easting\",1968");
    add_srs_wkt (p, 12,
		 "500],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 13, "\"2267\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2268, "epsg", 2268,
		      "NAD83 / Oklahoma South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=35.23333333333333 +lat_2=33.93333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=33.33333333333334 +lon_0=-98 +x_0=600000 +y_0");
    add_proj4text (p, 2, "=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Oklahoma South (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",35.23333");
    add_srs_wkt (p, 9,
		 "333333333],PARAMETER[\"standard_parallel_2\",33.93333333");
    add_srs_wkt (p, 10,
		 "333333],PARAMETER[\"latitude_of_origin\",33.333333333333");
    add_srs_wkt (p, 11,
		 "34],PARAMETER[\"central_meridian\",-98],PARAMETER[\"fals");
    add_srs_wkt (p, 12,
		 "e_easting\",1968500],PARAMETER[\"false_northing\",0],AUT");
    add_srs_wkt (p, 13,
		 "HORITY[\"EPSG\",\"2268\"],AXIS[\"X\",EAST],AXIS[\"Y\",NO");
    add_srs_wkt (p, 14, "RTH]]");
    p = add_epsg_def (first, last, 2269, "epsg", 2269,
		      "NAD83 / Oregon North (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=46 +lat_2=44.33333333333334 +lat_0=43.6");
    add_proj4text (p, 1,
		   "6666666666666 +lon_0=-120.5 +x_0=2500000.0001424 +y_0=0 ");
    add_proj4text (p, 2, "+ellps=GRS80 +datum=NAD83 +units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Oregon North (ft)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[");
    add_srs_wkt (p, 8,
		 "\"standard_parallel_1\",46],PARAMETER[\"standard_paralle");
    add_srs_wkt (p, 9,
		 "l_2\",44.33333333333334],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 10,
		 ",43.66666666666666],PARAMETER[\"central_meridian\",-120.");
    add_srs_wkt (p, 11,
		 "5],PARAMETER[\"false_easting\",8202099.738],PARAMETER[\"");
    add_srs_wkt (p, 12,
		 "false_northing\",0],AUTHORITY[\"EPSG\",\"2269\"],AXIS[\"");
    add_srs_wkt (p, 13, "X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2270, "epsg", 2270,
		      "NAD83 / Oregon South (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=44 +lat_2=42.33333333333334 +lat_0=41.6");
    add_proj4text (p, 1,
		   "6666666666666 +lon_0=-120.5 +x_0=1500000.0001464 +y_0=0 ");
    add_proj4text (p, 2, "+ellps=GRS80 +datum=NAD83 +units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Oregon South (ft)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[");
    add_srs_wkt (p, 8,
		 "\"standard_parallel_1\",44],PARAMETER[\"standard_paralle");
    add_srs_wkt (p, 9,
		 "l_2\",42.33333333333334],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 10,
		 ",41.66666666666666],PARAMETER[\"central_meridian\",-120.");
    add_srs_wkt (p, 11,
		 "5],PARAMETER[\"false_easting\",4921259.843],PARAMETER[\"");
    add_srs_wkt (p, 12,
		 "false_northing\",0],AUTHORITY[\"EPSG\",\"2270\"],AXIS[\"");
    add_srs_wkt (p, 13, "X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2271, "epsg", 2271,
		      "NAD83 / Pennsylvania North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.95 +lat_2=40.88333333333333 +lat_0=4");
    add_proj4text (p, 1,
		   "0.16666666666666 +lon_0=-77.75 +x_0=600000 +y_0=0 +ellps");
    add_proj4text (p, 2, "=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Pennsylvania North (ftUS)\",GEOGCS[\"NA");
    add_srs_wkt (p, 1,
		 "D83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",41.");
    add_srs_wkt (p, 9,
		 "95],PARAMETER[\"standard_parallel_2\",40.88333333333333]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"latitude_of_origin\",40.16666666666666],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"central_meridian\",-77.75],PARAMETER[\"false_ea");
    add_srs_wkt (p, 12,
		 "sting\",1968500],PARAMETER[\"false_northing\",0],AUTHORI");
    add_srs_wkt (p, 13,
		 "TY[\"EPSG\",\"2271\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]");
    add_srs_wkt (p, 14, "]");
    p = add_epsg_def (first, last, 2272, "epsg", 2272,
		      "NAD83 / Pennsylvania South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=40.96666666666667 +lat_2=39.93333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=39.33333333333334 +lon_0=-77.75 +x_0=600000 +");
    add_proj4text (p, 2,
		   "y_0=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Pennsylvania South (ftUS)\",GEOGCS[\"NA");
    add_srs_wkt (p, 1,
		 "D83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS");
    add_srs_wkt (p, 2,
		 " 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",40.");
    add_srs_wkt (p, 9,
		 "96666666666667],PARAMETER[\"standard_parallel_2\",39.933");
    add_srs_wkt (p, 10,
		 "33333333333],PARAMETER[\"latitude_of_origin\",39.3333333");
    add_srs_wkt (p, 11,
		 "3333334],PARAMETER[\"central_meridian\",-77.75],PARAMETE");
    add_srs_wkt (p, 12,
		 "R[\"false_easting\",1968500],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 13,
		 ",0],AUTHORITY[\"EPSG\",\"2272\"],AXIS[\"X\",EAST],AXIS[\"");
    add_srs_wkt (p, 14, "Y\",NORTH]]");
    p = add_epsg_def (first, last, 2273, "epsg", 2273,
		      "NAD83 / South Carolina (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=34.83333333333334 +lat_2=32.5 +lat_0=31");
    add_proj4text (p, 1,
		   ".83333333333333 +lon_0=-81 +x_0=609600 +y_0=0 +ellps=GRS");
    add_proj4text (p, 2, "80 +datum=NAD83 +units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / South Carolina (ft)\",GEOGCS[\"NAD83\",");
    add_srs_wkt (p, 1,
		 "DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER");
    add_srs_wkt (p, 8,
		 "[\"standard_parallel_1\",34.83333333333334],PARAMETER[\"");
    add_srs_wkt (p, 9,
		 "standard_parallel_2\",32.5],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 10,
		 "in\",31.83333333333333],PARAMETER[\"central_meridian\",-");
    add_srs_wkt (p, 11,
		 "81],PARAMETER[\"false_easting\",2000000],PARAMETER[\"fal");
    add_srs_wkt (p, 12,
		 "se_northing\",0],AUTHORITY[\"EPSG\",\"2273\"],AXIS[\"X\"");
    add_srs_wkt (p, 13, ",EAST],AXIS[\"Y\",NORTH]]");
#endif /* full EPSG initialization enabled/disabled */
    if (first == last)
	last = first;		/* suppressing stupid compiler warnings */
}

static void
initialize_epsg_02 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2274, "epsg", 2274,
		      "NAD83 / Tennessee (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=36.41666666666666 +lat_2=35.25 +lat_0=3");
    add_proj4text (p, 1,
		   "4.33333333333334 +lon_0=-86 +x_0=600000 +y_0=0 +ellps=GR");
    add_proj4text (p, 2, "S80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Tennessee (ftUS)\",GEOGCS[\"NAD83\",DAT");
    add_srs_wkt (p, 1,
		 "UM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",6");
    add_srs_wkt (p, 2,
		 "378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"426");
    add_srs_wkt (p, 6,
		 "9\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHORI");
    add_srs_wkt (p, 7,
		 "TY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_Co");
    add_srs_wkt (p, 8,
		 "nic_2SP\"],PARAMETER[\"standard_parallel_1\",36.41666666");
    add_srs_wkt (p, 9,
		 "666666],PARAMETER[\"standard_parallel_2\",35.25],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"latitude_of_origin\",34.33333333333334],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "central_meridian\",-86],PARAMETER[\"false_easting\",1968");
    add_srs_wkt (p, 12,
		 "500],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 13, "\"2274\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2275, "epsg", 2275,
		      "NAD83 / Texas North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=36.18333333333333 +lat_2=34.65 +lat_0=3");
    add_proj4text (p, 1,
		   "4 +lon_0=-101.5 +x_0=200000.0001016002 +y_0=999999.99989");
    add_proj4text (p, 2,
		   "83998 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Texas North (ftUS)\",GEOGCS[\"NAD83\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_");
    add_srs_wkt (p, 8,
		 "Conic_2SP\"],PARAMETER[\"standard_parallel_1\",36.183333");
    add_srs_wkt (p, 9,
		 "33333333],PARAMETER[\"standard_parallel_2\",34.65],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"latitude_of_origin\",34],PARAMETER[\"central_meri");
    add_srs_wkt (p, 11,
		 "dian\",-101.5],PARAMETER[\"false_easting\",656166.667],P");
    add_srs_wkt (p, 12,
		 "ARAMETER[\"false_northing\",3280833.333],AUTHORITY[\"EPS");
    add_srs_wkt (p, 13, "G\",\"2275\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2276, "epsg", 2276,
		      "NAD83 / Texas North Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=33.96666666666667 +lat_2=32.13333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=31.66666666666667 +lon_0=-98.5 +x_0=600000 +y");
    add_proj4text (p, 2,
		   "_0=2000000.0001016 +ellps=GRS80 +datum=NAD83 +units=us-f");
    add_proj4text (p, 3, "t +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Texas North Central (ftUS)\",GEOGCS[\"N");
    add_srs_wkt (p, 1,
		 "AD83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4269\"]],UNIT[\"US survey foot\",0.30480060960121");
    add_srs_wkt (p, 7,
		 "92,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Co");
    add_srs_wkt (p, 8,
		 "nformal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",3");
    add_srs_wkt (p, 9,
		 "3.96666666666667],PARAMETER[\"standard_parallel_2\",32.1");
    add_srs_wkt (p, 10,
		 "3333333333333],PARAMETER[\"latitude_of_origin\",31.66666");
    add_srs_wkt (p, 11,
		 "666666667],PARAMETER[\"central_meridian\",-98.5],PARAMET");
    add_srs_wkt (p, 12,
		 "ER[\"false_easting\",1968500],PARAMETER[\"false_northing");
    add_srs_wkt (p, 13,
		 "\",6561666.667],AUTHORITY[\"EPSG\",\"2276\"],AXIS[\"X\",");
    add_srs_wkt (p, 14, "EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2277, "epsg", 2277,
		      "NAD83 / Texas Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=31.88333333333333 +lat_2=30.11666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=29.66666666666667 +lon_0=-100.3333333333333 +");
    add_proj4text (p, 2,
		   "x_0=699999.9998983998 +y_0=3000000 +ellps=GRS80 +datum=N");
    add_proj4text (p, 3, "AD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Texas Central (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",31.88333");
    add_srs_wkt (p, 9,
		 "333333333],PARAMETER[\"standard_parallel_2\",30.11666666");
    add_srs_wkt (p, 10,
		 "666667],PARAMETER[\"latitude_of_origin\",29.666666666666");
    add_srs_wkt (p, 11,
		 "67],PARAMETER[\"central_meridian\",-100.3333333333333],P");
    add_srs_wkt (p, 12,
		 "ARAMETER[\"false_easting\",2296583.333],PARAMETER[\"fals");
    add_srs_wkt (p, 13,
		 "e_northing\",9842500.000000002],AUTHORITY[\"EPSG\",\"227");
    add_srs_wkt (p, 14, "7\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2278, "epsg", 2278,
		      "NAD83 / Texas South Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=30.28333333333333 +lat_2=28.38333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=27.83333333333333 +lon_0=-99 +x_0=600000 +y_0");
    add_proj4text (p, 2,
		   "=3999999.9998984 +ellps=GRS80 +datum=NAD83 +units=us-ft ");
    add_proj4text (p, 3, "+no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Texas South Central (ftUS)\",GEOGCS[\"N");
    add_srs_wkt (p, 1,
		 "AD83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GR");
    add_srs_wkt (p, 2,
		 "S 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4269\"]],UNIT[\"US survey foot\",0.30480060960121");
    add_srs_wkt (p, 7,
		 "92,AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Co");
    add_srs_wkt (p, 8,
		 "nformal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",3");
    add_srs_wkt (p, 9,
		 "0.28333333333333],PARAMETER[\"standard_parallel_2\",28.3");
    add_srs_wkt (p, 10,
		 "8333333333333],PARAMETER[\"latitude_of_origin\",27.83333");
    add_srs_wkt (p, 11,
		 "333333333],PARAMETER[\"central_meridian\",-99],PARAMETER");
    add_srs_wkt (p, 12,
		 "[\"false_easting\",1968500],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 13,
		 ",13123333.333],AUTHORITY[\"EPSG\",\"2278\"],AXIS[\"X\",E");
    add_srs_wkt (p, 14, "AST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2279, "epsg", 2279,
		      "NAD83 / Texas South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=27.83333333333333 +lat_2=26.16666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=25.66666666666667 +lon_0=-98.5 +x_0=300000.00");
    add_proj4text (p, 2,
		   "00000001 +y_0=5000000.0001016 +ellps=GRS80 +datum=NAD83 ");
    add_proj4text (p, 3, "+units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Texas South (ftUS)\",GEOGCS[\"NAD83\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\"");
    add_srs_wkt (p, 2,
		 ",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUT");
    add_srs_wkt (p, 3,
		 "HORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 4,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 5,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 6,
		 "269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTHO");
    add_srs_wkt (p, 7,
		 "RITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal_");
    add_srs_wkt (p, 8,
		 "Conic_2SP\"],PARAMETER[\"standard_parallel_1\",27.833333");
    add_srs_wkt (p, 9,
		 "33333333],PARAMETER[\"standard_parallel_2\",26.166666666");
    add_srs_wkt (p, 10,
		 "66667],PARAMETER[\"latitude_of_origin\",25.6666666666666");
    add_srs_wkt (p, 11,
		 "7],PARAMETER[\"central_meridian\",-98.5],PARAMETER[\"fal");
    add_srs_wkt (p, 12,
		 "se_easting\",984250.0000000002],PARAMETER[\"false_northi");
    add_srs_wkt (p, 13,
		 "ng\",16404166.667],AUTHORITY[\"EPSG\",\"2279\"],AXIS[\"X");
    add_srs_wkt (p, 14, "\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2280, "epsg", 2280,
		      "NAD83 / Utah North (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=41.78333333333333 +lat_2=40.71666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=40.33333333333334 +lon_0=-111.5 +x_0=500000.0");
    add_proj4text (p, 2,
		   "001504 +y_0=999999.9999960001 +ellps=GRS80 +datum=NAD83 ");
    add_proj4text (p, 3, "+units=ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Utah North (ft)\",GEOGCS[\"NAD83\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",63");
    add_srs_wkt (p, 2,
		 "78137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "standard_parallel_1\",41.78333333333333],PARAMETER[\"sta");
    add_srs_wkt (p, 9,
		 "ndard_parallel_2\",40.71666666666667],PARAMETER[\"latitu");
    add_srs_wkt (p, 10,
		 "de_of_origin\",40.33333333333334],PARAMETER[\"central_me");
    add_srs_wkt (p, 11,
		 "ridian\",-111.5],PARAMETER[\"false_easting\",1640419.948");
    add_srs_wkt (p, 12,
		 "],PARAMETER[\"false_northing\",3280839.895],AUTHORITY[\"");
    add_srs_wkt (p, 13, "EPSG\",\"2280\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2281, "epsg", 2281,
		      "NAD83 / Utah Central (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=40.65 +lat_2=39.01666666666667 +lat_0=3");
    add_proj4text (p, 1,
		   "8.33333333333334 +lon_0=-111.5 +x_0=500000.0001504 +y_0=");
    add_proj4text (p, 2,
		   "1999999.999992 +ellps=GRS80 +datum=NAD83 +units=ft +no_d");
    add_proj4text (p, 3, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Utah Central (ft)\",GEOGCS[\"NAD83\",DA");
    add_srs_wkt (p, 1,
		 "TUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",");
    add_srs_wkt (p, 2,
		 "6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTH");
    add_srs_wkt (p, 3,
		 "ORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 4,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 5,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 6,
		 "69\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[");
    add_srs_wkt (p, 8,
		 "\"standard_parallel_1\",40.65],PARAMETER[\"standard_para");
    add_srs_wkt (p, 9,
		 "llel_2\",39.01666666666667],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 10,
		 "in\",38.33333333333334],PARAMETER[\"central_meridian\",-");
    add_srs_wkt (p, 11,
		 "111.5],PARAMETER[\"false_easting\",1640419.948],PARAMETE");
    add_srs_wkt (p, 12,
		 "R[\"false_northing\",6561679.79],AUTHORITY[\"EPSG\",\"22");
    add_srs_wkt (p, 13, "81\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2282, "epsg", 2282,
		      "NAD83 / Utah South (ft)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=38.35 +lat_2=37.21666666666667 +lat_0=3");
    add_proj4text (p, 1,
		   "6.66666666666666 +lon_0=-111.5 +x_0=500000.0001504 +y_0=");
    add_proj4text (p, 2,
		   "2999999.999988 +ellps=GRS80 +datum=NAD83 +units=ft +no_d");
    add_proj4text (p, 3, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Utah South (ft)\",GEOGCS[\"NAD83\",DATU");
    add_srs_wkt (p, 1,
		 "M[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980\",63");
    add_srs_wkt (p, 2,
		 "78137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4269");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"foot\",0.3048,AUTHORITY[\"EPSG\",\"9002\"]],");
    add_srs_wkt (p, 7,
		 "PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "standard_parallel_1\",38.35],PARAMETER[\"standard_parall");
    add_srs_wkt (p, 9,
		 "el_2\",37.21666666666667],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 10,
		 "\",36.66666666666666],PARAMETER[\"central_meridian\",-11");
    add_srs_wkt (p, 11,
		 "1.5],PARAMETER[\"false_easting\",1640419.948],PARAMETER[");
    add_srs_wkt (p, 12,
		 "\"false_northing\",9842519.685],AUTHORITY[\"EPSG\",\"228");
    add_srs_wkt (p, 13, "2\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2283, "epsg", 2283,
		      "NAD83 / Virginia North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=39.2 +lat_2=38.03333333333333 +lat_0=37");
    add_proj4text (p, 1,
		   ".66666666666666 +lon_0=-78.5 +x_0=3500000.0001016 +y_0=2");
    add_proj4text (p, 2,
		   "000000.0001016 +ellps=GRS80 +datum=NAD83 +units=us-ft +n");
    add_proj4text (p, 3, "o_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Virginia North (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",39.2],PA");
    add_srs_wkt (p, 9,
		 "RAMETER[\"standard_parallel_2\",38.03333333333333],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"latitude_of_origin\",37.66666666666666],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"central_meridian\",-78.5],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 12,
		 ",11482916.667],PARAMETER[\"false_northing\",6561666.667]");
    add_srs_wkt (p, 13,
		 ",AUTHORITY[\"EPSG\",\"2283\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 14, ",NORTH]]");
    p = add_epsg_def (first, last, 2284, "epsg", 2284,
		      "NAD83 / Virginia South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=37.96666666666667 +lat_2=36.76666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=36.33333333333334 +lon_0=-78.5 +x_0=3500000.0");
    add_proj4text (p, 2,
		   "001016 +y_0=999999.9998983998 +ellps=GRS80 +datum=NAD83 ");
    add_proj4text (p, 3, "+units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Virginia South (ftUS)\",GEOGCS[\"NAD83\"");
    add_srs_wkt (p, 1,
		 ",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1980");
    add_srs_wkt (p, 2,
		 "\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]],A");
    add_srs_wkt (p, 3,
		 "UTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 5,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 6,
		 "4269\"]],UNIT[\"US survey foot\",0.3048006096012192,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conformal");
    add_srs_wkt (p, 8,
		 "_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",37.96666");
    add_srs_wkt (p, 9,
		 "666666667],PARAMETER[\"standard_parallel_2\",36.76666666");
    add_srs_wkt (p, 10,
		 "666667],PARAMETER[\"latitude_of_origin\",36.333333333333");
    add_srs_wkt (p, 11,
		 "34],PARAMETER[\"central_meridian\",-78.5],PARAMETER[\"fa");
    add_srs_wkt (p, 12,
		 "lse_easting\",11482916.667],PARAMETER[\"false_northing\"");
    add_srs_wkt (p, 13,
		 ",3280833.333],AUTHORITY[\"EPSG\",\"2284\"],AXIS[\"X\",EA");
    add_srs_wkt (p, 14, "ST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2285, "epsg", 2285,
		      "NAD83 / Washington North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=48.73333333333333 +lat_2=47.5 +lat_0=47");
    add_proj4text (p, 1,
		   " +lon_0=-120.8333333333333 +x_0=500000.0001016001 +y_0=0");
    add_proj4text (p, 2, " +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Washington North (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Confor");
    add_srs_wkt (p, 8,
		 "mal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",48.73");
    add_srs_wkt (p, 9,
		 "333333333333],PARAMETER[\"standard_parallel_2\",47.5],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"latitude_of_origin\",47],PARAMETER[\"central_m");
    add_srs_wkt (p, 11,
		 "eridian\",-120.8333333333333],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 12,
		 ",1640416.667],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 13,
		 "\"EPSG\",\"2285\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2286, "epsg", 2286,
		      "NAD83 / Washington South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=47.33333333333334 +lat_2=45.83333333333");
    add_proj4text (p, 1,
		   "334 +lat_0=45.33333333333334 +lon_0=-120.5 +x_0=500000.0");
    add_proj4text (p, 2,
		   "001016001 +y_0=0 +ellps=GRS80 +datum=NAD83 +units=us-ft ");
    add_proj4text (p, 3, "+no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Washington South (ftUS)\",GEOGCS[\"NAD8");
    add_srs_wkt (p, 1,
		 "3\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 1");
    add_srs_wkt (p, 2,
		 "980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]");
    add_srs_wkt (p, 3,
		 "],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292");
    add_srs_wkt (p, 5,
		 "51994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Confor");
    add_srs_wkt (p, 8,
		 "mal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",47.33");
    add_srs_wkt (p, 9,
		 "333333333334],PARAMETER[\"standard_parallel_2\",45.83333");
    add_srs_wkt (p, 10,
		 "333333334],PARAMETER[\"latitude_of_origin\",45.333333333");
    add_srs_wkt (p, 11,
		 "33334],PARAMETER[\"central_meridian\",-120.5],PARAMETER[");
    add_srs_wkt (p, 12,
		 "\"false_easting\",1640416.667],PARAMETER[\"false_northin");
    add_srs_wkt (p, 13,
		 "g\",0],AUTHORITY[\"EPSG\",\"2286\"],AXIS[\"X\",EAST],AXI");
    add_srs_wkt (p, 14, "S[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2287, "epsg", 2287,
		      "NAD83 / Wisconsin North (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=46.76666666666667 +lat_2=45.56666666666");
    add_proj4text (p, 1,
		   "667 +lat_0=45.16666666666666 +lon_0=-90 +x_0=600000 +y_0");
    add_proj4text (p, 2, "=0 +ellps=GRS80 +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Wisconsin North (ftUS)\",GEOGCS[\"NAD83");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Confor");
    add_srs_wkt (p, 8,
		 "mal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",46.76");
    add_srs_wkt (p, 9,
		 "666666666667],PARAMETER[\"standard_parallel_2\",45.56666");
    add_srs_wkt (p, 10,
		 "666666667],PARAMETER[\"latitude_of_origin\",45.166666666");
    add_srs_wkt (p, 11,
		 "66666],PARAMETER[\"central_meridian\",-90],PARAMETER[\"f");
    add_srs_wkt (p, 12,
		 "alse_easting\",1968500],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 13,
		 "AUTHORITY[\"EPSG\",\"2287\"],AXIS[\"X\",EAST],AXIS[\"Y\"");
    add_srs_wkt (p, 14, ",NORTH]]");
    p = add_epsg_def (first, last, 2288, "epsg", 2288,
		      "NAD83 / Wisconsin Central (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=45.5 +lat_2=44.25 +lat_0=43.83333333333");
    add_proj4text (p, 1,
		   "334 +lon_0=-90 +x_0=600000 +y_0=0 +ellps=GRS80 +datum=NA");
    add_proj4text (p, 2, "D83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Wisconsin Central (ftUS)\",GEOGCS[\"NAD");
    add_srs_wkt (p, 1,
		 "83\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS ");
    add_srs_wkt (p, 2,
		 "1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192");
    add_srs_wkt (p, 7,
		 ",AUTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Conf");
    add_srs_wkt (p, 8,
		 "ormal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",45.");
    add_srs_wkt (p, 9,
		 "5],PARAMETER[\"standard_parallel_2\",44.25],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "latitude_of_origin\",43.83333333333334],PARAMETER[\"cent");
    add_srs_wkt (p, 11,
		 "ral_meridian\",-90],PARAMETER[\"false_easting\",1968500]");
    add_srs_wkt (p, 12,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"22");
    add_srs_wkt (p, 13, "88\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2289, "epsg", 2289,
		      "NAD83 / Wisconsin South (ftUS)");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=44.06666666666667 +lat_2=42.73333333333");
    add_proj4text (p, 1,
		   "333 +lat_0=42 +lon_0=-90 +x_0=600000 +y_0=0 +ellps=GRS80");
    add_proj4text (p, 2, " +datum=NAD83 +units=us-ft +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83 / Wisconsin South (ftUS)\",GEOGCS[\"NAD83");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"North_American_Datum_1983\",SPHEROID[\"GRS 19");
    add_srs_wkt (p, 2,
		 "80\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"7019\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6269\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4269\"]],UNIT[\"US survey foot\",0.3048006096012192,A");
    add_srs_wkt (p, 7,
		 "UTHORITY[\"EPSG\",\"9003\"]],PROJECTION[\"Lambert_Confor");
    add_srs_wkt (p, 8,
		 "mal_Conic_2SP\"],PARAMETER[\"standard_parallel_1\",44.06");
    add_srs_wkt (p, 9,
		 "666666666667],PARAMETER[\"standard_parallel_2\",42.73333");
    add_srs_wkt (p, 10,
		 "333333333],PARAMETER[\"latitude_of_origin\",42],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"central_meridian\",-90],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 12,
		 "1968500],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 13, "G\",\"2289\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    p = add_epsg_def (first, last, 2290, "epsg", 2290,
		      "ATS77 / Prince Edward Isl. Stereographic (ATS77)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=47.25 +lon_0=-63 +k=0.999912 +x_0=70");
    add_proj4text (p, 1,
		   "0000 +y_0=400000 +a=6378135 +b=6356750.304921594 +units=");
    add_proj4text (p, 2, "m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / Prince Edward Isl. Stereographic (ATS77");
    add_srs_wkt (p, 1,
		 ")\",GEOGCS[\"ATS77\",DATUM[\"Average_Terrestrial_System_");
    add_srs_wkt (p, 2,
		 "1977\",SPHEROID[\"Average Terrestrial System 1977\",6378");
    add_srs_wkt (p, 3,
		 "135,298.257,AUTHORITY[\"EPSG\",\"7041\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 4,
		 "SG\",\"6122\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4122\"]],UNIT");
    add_srs_wkt (p, 7,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 8,
		 "Oblique_Stereographic\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 9,
		 ",47.25],PARAMETER[\"central_meridian\",-63],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",0.999912],PARAMETER[\"false_easting\",700");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",400000],AUTHORITY[\"EP");
    add_srs_wkt (p, 12,
		 "SG\",\"2290\"],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2291, "epsg", 2291,
		      "NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=47.25 +lon_0=-63 +k=0.999912 +x_0=40");
    add_proj4text (p, 1,
		   "0000 +y_0=800000 +a=6378135 +b=6356750.304921594 +units=");
    add_proj4text (p, 2, "m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / Prince Edward Isl. Stereographi");
    add_srs_wkt (p, 1,
		 "c (NAD83) (deprecated)\",GEOGCS[\"ATS77\",DATUM[\"Averag");
    add_srs_wkt (p, 2,
		 "e_Terrestrial_System_1977\",SPHEROID[\"Average Terrestri");
    add_srs_wkt (p, 3,
		 "al System 1977\",6378135,298.257,AUTHORITY[\"EPSG\",\"70");
    add_srs_wkt (p, 4,
		 "41\"]],AUTHORITY[\"EPSG\",\"6122\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 5,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 6,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 7,
		 "PSG\",\"4122\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 8,
		 "001\"]],PROJECTION[\"Oblique_Stereographic\"],PARAMETER[");
    add_srs_wkt (p, 9,
		 "\"latitude_of_origin\",47.25],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 10,
		 "an\",-63],PARAMETER[\"scale_factor\",0.999912],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"false_easting\",400000],PARAMETER[\"false_northing\",");
    add_srs_wkt (p, 12,
		 "800000],AUTHORITY[\"EPSG\",\"2291\"],AXIS[\"E(X)\",EAST]");
    add_srs_wkt (p, 13, ",AXIS[\"N(Y)\",NORTH]]");
    p = add_epsg_def (first, last, 2292, "epsg", 2292,
		      "NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated)");
    add_proj4text (p, 0,
		   "+proj=sterea +lat_0=47.25 +lon_0=-63 +k=0.999912 +x_0=40");
    add_proj4text (p, 1,
		   "0000 +y_0=800000 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +un");
    add_proj4text (p, 2, "its=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"NAD83(CSRS98) / Prince Edward Isl. Stereographi");
    add_srs_wkt (p, 1,
		 "c (NAD83) (deprecated)\",GEOGCS[\"NAD83(CSRS98)\",DATUM[");
    add_srs_wkt (p, 2,
		 "\"NAD83_Canadian_Spatial_Reference_System\",SPHEROID[\"G");
    add_srs_wkt (p, 3,
		 "RS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\",\"701");
    add_srs_wkt (p, 4,
		 "9\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"6140\"");
    add_srs_wkt (p, 5,
		 "]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],");
    add_srs_wkt (p, 6,
		 "UNIT[\"degree\",0.0174532925199433,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9108\"]],AUTHORITY[\"EPSG\",\"4140\"]],UNIT[\"metre\",1,");
    add_srs_wkt (p, 8,
		 "AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Oblique_Stere");
    add_srs_wkt (p, 9,
		 "ographic\"],PARAMETER[\"latitude_of_origin\",47.25],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"central_meridian\",-63],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 11,
		 "\",0.999912],PARAMETER[\"false_easting\",400000],PARAMET");
    add_srs_wkt (p, 12,
		 "ER[\"false_northing\",800000],AUTHORITY[\"EPSG\",\"2292\"");
    add_srs_wkt (p, 13, "],AXIS[\"E(X)\",EAST],AXIS[\"N(Y)\",NORTH]]");
    p = add_epsg_def (first, last, 2294, "epsg", 2294,
		      "ATS77 / MTM Nova Scotia zone 4");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-61.5 +k=0.9999 +x_0=4500000");
    add_proj4text (p, 1,
		   " +y_0=0 +a=6378135 +b=6356750.304921594 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / MTM Nova Scotia zone 4\",GEOGCS[\"ATS77");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"Average_Terrestrial_System_1977\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Average Terrestrial System 1977\",6378135,298.257,AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"7041\"]],AUTHORITY[\"EPSG\",\"6122\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4122\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",-61.5],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "9],PARAMETER[\"false_easting\",4500000],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2294\"],AXIS[\"East");
    add_srs_wkt (p, 12, "ing\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2295, "epsg", 2295,
		      "ATS77 / MTM Nova Scotia zone 5");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-64.5 +k=0.9999 +x_0=5500000");
    add_proj4text (p, 1,
		   " +y_0=0 +a=6378135 +b=6356750.304921594 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ATS77 / MTM Nova Scotia zone 5\",GEOGCS[\"ATS77");
    add_srs_wkt (p, 1,
		 "\",DATUM[\"Average_Terrestrial_System_1977\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Average Terrestrial System 1977\",6378135,298.257,AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"7041\"]],AUTHORITY[\"EPSG\",\"6122\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4122\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",-64.5],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 10,
		 "9],PARAMETER[\"false_easting\",5500000],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_northing\",0],AUTHORITY[\"EPSG\",\"2295\"],AXIS[\"East");
    add_srs_wkt (p, 12, "ing\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2308, "epsg", 2308, "Batavia / TM 109 SE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=109 +k=0.9996 +x_0=500000 +y");
    add_proj4text (p, 1, "_0=10000000 +ellps=bessel +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Batavia / TM 109 SE\",GEOGCS[\"Batavia\",DATUM[");
    add_srs_wkt (p, 1,
		 "\"Batavia\",SPHEROID[\"Bessel 1841\",6377397.155,299.152");
    add_srs_wkt (p, 2,
		 "8128,AUTHORITY[\"EPSG\",\"7004\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "6211\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 4,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 5,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4211\"]],UNIT[\"met");
    add_srs_wkt (p, 6,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 7,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 8,
		 "AMETER[\"central_meridian\",109],PARAMETER[\"scale_facto");
    add_srs_wkt (p, 9,
		 "r\",0.9996],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",10000000],AUTHORITY[\"EPSG\",\"2308");
    add_srs_wkt (p, 11, "\"],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2309, "epsg", 2309, "WGS 84 / TM 116 SE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=116 +k=0.9996 +x_0=500000 +y");
    add_proj4text (p, 1,
		   "_0=10000000 +ellps=WGS84 +datum=WGS84 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"WGS 84 / TM 116 SE\",GEOGCS[\"WGS 84\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "WGS_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563,AUT");
    add_srs_wkt (p, 2,
		 "HORITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]]");
    add_srs_wkt (p, 3,
		 ",PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UN");
    add_srs_wkt (p, 4,
		 "IT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 5,
		 "122\"]],AUTHORITY[\"EPSG\",\"4326\"]],UNIT[\"metre\",1,A");
    add_srs_wkt (p, 6,
		 "UTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 7,
		 "cator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "central_meridian\",116],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2309\"],AXIS[");
    add_srs_wkt (p, 11, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2310, "epsg", 2310, "WGS 84 / TM 132 SE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=0.9996 +x_0=500000 +y");
    add_proj4text (p, 1,
		   "_0=10000000 +ellps=WGS84 +datum=WGS84 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"WGS 84 / TM 132 SE\",GEOGCS[\"WGS 84\",DATUM[\"");
    add_srs_wkt (p, 1,
		 "WGS_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563,AUT");
    add_srs_wkt (p, 2,
		 "HORITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]]");
    add_srs_wkt (p, 3,
		 ",PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UN");
    add_srs_wkt (p, 4,
		 "IT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 5,
		 "122\"]],AUTHORITY[\"EPSG\",\"4326\"]],UNIT[\"metre\",1,A");
    add_srs_wkt (p, 6,
		 "UTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mer");
    add_srs_wkt (p, 7,
		 "cator\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "central_meridian\",132],PARAMETER[\"scale_factor\",0.999");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"false_easting\",500000],PARAMETER[\"false");
    add_srs_wkt (p, 10,
		 "_northing\",10000000],AUTHORITY[\"EPSG\",\"2310\"],AXIS[");
    add_srs_wkt (p, 11, "\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2311, "epsg", 2311, "WGS 84 / TM 6 NE");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=6 +k=0.9996 +x_0=500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=WGS84 +datum=WGS84 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"WGS 84 / TM 6 NE\",GEOGCS[\"WGS 84\",DATUM[\"WG");
    add_srs_wkt (p, 1,
		 "S_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHO");
    add_srs_wkt (p, 2,
		 "RITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]],P");
    add_srs_wkt (p, 3,
		 "RIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT");
    add_srs_wkt (p, 4,
		 "[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"912");
    add_srs_wkt (p, 5,
		 "2\"]],AUTHORITY[\"EPSG\",\"4326\"]],UNIT[\"metre\",1,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Merca");
    add_srs_wkt (p, 7,
		 "tor\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"c");
    add_srs_wkt (p, 8,
		 "entral_meridian\",6],PARAMETER[\"scale_factor\",0.9996],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"false_easting\",500000],PARAMETER[\"false_no");
    add_srs_wkt (p, 10,
		 "rthing\",0],AUTHORITY[\"EPSG\",\"2311\"],AXIS[\"Easting\"");
    add_srs_wkt (p, 11, ",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2312, "epsg", 2312, "Garoua / UTM zone 33N");
    add_proj4text (p, 0, "+proj=utm +zone=33 +ellps=clrk80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Garoua / UTM zone 33N\",GEOGCS[\"Garoua\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Garoua\",SPHEROID[\"Clarke 1880 (RGS)\",6378249.145,2");
    add_srs_wkt (p, 2,
		 "93.465,AUTHORITY[\"EPSG\",\"7012\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"6197\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8");
    add_srs_wkt (p, 4,
		 "901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 5,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4197\"]],UNIT[\"m");
    add_srs_wkt (p, 6,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 7,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],P");
    add_srs_wkt (p, 8,
		 "ARAMETER[\"central_meridian\",15],PARAMETER[\"scale_fact");
    add_srs_wkt (p, 9,
		 "or\",0.9996],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2312\"],AX");
    add_srs_wkt (p, 11, "IS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2313, "epsg", 2313,
		      "Kousseri / UTM zone 33N");
    add_proj4text (p, 0, "+proj=utm +zone=33 +ellps=clrk80 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Kousseri / UTM zone 33N\",GEOGCS[\"Kousseri\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"Kousseri\",SPHEROID[\"Clarke 1880 (RGS)\",6378249");
    add_srs_wkt (p, 2,
		 ".145,293.465,AUTHORITY[\"EPSG\",\"7012\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 3,
		 "PSG\",\"6198\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4198\"]],UN");
    add_srs_wkt (p, 6,
		 "IT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[");
    add_srs_wkt (p, 7,
		 "\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\"");
    add_srs_wkt (p, 8,
		 ",0],PARAMETER[\"central_meridian\",15],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",0.9996],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2313\"");
    add_srs_wkt (p, 11, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2314, "epsg", 2314,
		      "Trinidad 1903 / Trinidad Grid (ftCla)");
    add_proj4text (p, 0,
		   "+proj=cass +lat_0=10.44166666666667 +lon_0=-61.333333333");
    add_proj4text (p, 1,
		   "33334 +x_0=86501.46392052001 +y_0=65379.0134283 +a=63782");
    add_proj4text (p, 2,
		   "93.645208759 +b=6356617.987679838 +to_meter=0.3047972654");
    add_proj4text (p, 3, " +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Trinidad 1903 / Trinidad Grid (ftCla)\",GEOGCS[");
    add_srs_wkt (p, 1,
		 "\"Trinidad 1903\",DATUM[\"Trinidad_1903\",SPHEROID[\"Cla");
    add_srs_wkt (p, 2,
		 "rke 1858\",6378293.645208759,294.2606763692569,AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"7007\"]],AUTHORITY[\"EPSG\",\"6302\"]],PRIME");
    add_srs_wkt (p, 4,
		 "M[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"d");
    add_srs_wkt (p, 5,
		 "egree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]");
    add_srs_wkt (p, 6,
		 "],AUTHORITY[\"EPSG\",\"4302\"]],UNIT[\"Clarke's foot\",0");
    add_srs_wkt (p, 7,
		 ".3047972654,AUTHORITY[\"EPSG\",\"9005\"]],PROJECTION[\"C");
    add_srs_wkt (p, 8,
		 "assini_Soldner\"],PARAMETER[\"latitude_of_origin\",10.44");
    add_srs_wkt (p, 9,
		 "166666666667],PARAMETER[\"central_meridian\",-61.3333333");
    add_srs_wkt (p, 10,
		 "3333334],PARAMETER[\"false_easting\",283800],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_northing\",214500],AUTHORITY[\"EPSG\",\"2314\"],AX");
    add_srs_wkt (p, 12, "IS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2315, "epsg", 2315,
		      "Campo Inchauspe / UTM zone 19S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=19 +south +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Campo Inchauspe / UTM zone 19S\",GEOGCS[\"Campo");
    add_srs_wkt (p, 1,
		 " Inchauspe\",DATUM[\"Campo_Inchauspe\",SPHEROID[\"Intern");
    add_srs_wkt (p, 2,
		 "ational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6221\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4221\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 8,
		 "ude_of_origin\",0],PARAMETER[\"central_meridian\",-69],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",10000000],AUT");
    add_srs_wkt (p, 11,
		 "HORITY[\"EPSG\",\"2315\"],AXIS[\"Easting\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2316, "epsg", 2316,
		      "Campo Inchauspe / UTM zone 20S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=20 +south +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Campo Inchauspe / UTM zone 20S\",GEOGCS[\"Campo");
    add_srs_wkt (p, 1,
		 " Inchauspe\",DATUM[\"Campo_Inchauspe\",SPHEROID[\"Intern");
    add_srs_wkt (p, 2,
		 "ational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6221\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4221\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 8,
		 "ude_of_origin\",0],PARAMETER[\"central_meridian\",-63],P");
    add_srs_wkt (p, 9,
		 "ARAMETER[\"scale_factor\",0.9996],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",10000000],AUT");
    add_srs_wkt (p, 11,
		 "HORITY[\"EPSG\",\"2316\"],AXIS[\"Easting\",EAST],AXIS[\"");
    add_srs_wkt (p, 12, "Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2317, "epsg", 2317, "PSAD56 / ICN Regional");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=9 +lat_2=3 +lat_0=6 +lon_0=-66 +x_0=100");
    add_proj4text (p, 1, "0000 +y_0=1000000 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"PSAD56 / ICN Regional\",GEOGCS[\"PSAD56\",DATUM");
    add_srs_wkt (p, 1,
		 "[\"Provisional_South_American_Datum_1956\",SPHEROID[\"In");
    add_srs_wkt (p, 2,
		 "ternational 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022");
    add_srs_wkt (p, 3,
		 "\"]],AUTHORITY[\"EPSG\",\"6248\"]],PRIMEM[\"Greenwich\",");
    add_srs_wkt (p, 4,
		 "0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453");
    add_srs_wkt (p, 5,
		 "29251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"4248\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"90");
    add_srs_wkt (p, 7,
		 "01\"]],PROJECTION[\"Lambert_Conformal_Conic_2SP\"],PARAM");
    add_srs_wkt (p, 8,
		 "ETER[\"standard_parallel_1\",9],PARAMETER[\"standard_par");
    add_srs_wkt (p, 9,
		 "allel_2\",3],PARAMETER[\"latitude_of_origin\",6],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"central_meridian\",-66],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",1000000],PARAMETER[\"false_northing\",1000000],AUTHORIT");
    add_srs_wkt (p, 12,
		 "Y[\"EPSG\",\"2317\"],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH]]");
    add_srs_wkt (p, 13, "");
    p = add_epsg_def (first, last, 2318, "epsg", 2318,
		      "Ain el Abd / Aramco Lambert");
    add_proj4text (p, 0,
		   "+proj=lcc +lat_1=17 +lat_2=33 +lat_0=25.08951 +lon_0=48 ");
    add_proj4text (p, 1, "+x_0=0 +y_0=0 +ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Ain el Abd / Aramco Lambert\",GEOGCS[\"Ain el A");
    add_srs_wkt (p, 1,
		 "bd\",DATUM[\"Ain_el_Abd_1970\",SPHEROID[\"International ");
    add_srs_wkt (p, 2,
		 "1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],AUTHORI");
    add_srs_wkt (p, 3,
		 "TY[\"EPSG\",\"6204\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,");
    add_srs_wkt (p, 5,
		 "AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4204\"");
    add_srs_wkt (p, 6,
		 "]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJEC");
    add_srs_wkt (p, 7,
		 "TION[\"Lambert_Conformal_Conic_2SP\"],PARAMETER[\"standa");
    add_srs_wkt (p, 8,
		 "rd_parallel_1\",17],PARAMETER[\"standard_parallel_2\",33");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"latitude_of_origin\",25.08951],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "central_meridian\",48],PARAMETER[\"false_easting\",0],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2318\"");
    add_srs_wkt (p, 12, "],AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2319, "epsg", 2319, "ED50 / TM27");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM27\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",27],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2319\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2320, "epsg", 2320, "ED50 / TM30");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM30\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",30],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2320\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2321, "epsg", 2321, "ED50 / TM33");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM33\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",33],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2321\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2322, "epsg", 2322, "ED50 / TM36");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM36\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",36],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2322\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2323, "epsg", 2323, "ED50 / TM39");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM39\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",39],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2323\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2324, "epsg", 2324, "ED50 / TM42");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM42\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",42],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2324\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2325, "epsg", 2325, "ED50 / TM45");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"ED50 / TM45\",GEOGCS[\"ED50\",DATUM[\"European_");
    add_srs_wkt (p, 1,
		 "Datum_1950\",SPHEROID[\"International 1924\",6378388,297");
    add_srs_wkt (p, 2,
		 ",AUTHORITY[\"EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6230");
    add_srs_wkt (p, 3,
		 "\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]");
    add_srs_wkt (p, 4,
		 "],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4230\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",45],PARAMETER[\"scale_factor\",1");
    add_srs_wkt (p, 9,
		 "],PARAMETER[\"false_easting\",500000],PARAMETER[\"false_");
    add_srs_wkt (p, 10,
		 "northing\",0],AUTHORITY[\"EPSG\",\"2325\"],AXIS[\"X\",NO");
    add_srs_wkt (p, 11, "RTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2326, "epsg", 2326,
		      "Hong Kong 1980 Grid System");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=22.31213333333334 +lon_0=114.17855555");
    add_proj4text (p, 1,
		   "55556 +k=1 +x_0=836694.05 +y_0=819069.8 +ellps=intl +tow");
    add_proj4text (p, 2,
		   "gs84=-162.619,-276.959,-161.764,0.067753,-2.24365,-1.158");
    add_proj4text (p, 3, "83,-1.09425 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Hong Kong 1980 Grid System\",GEOGCS[\"Hong Kong");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Hong_Kong_1980\",SPHEROID[\"Internationa");
    add_srs_wkt (p, 2,
		 "l 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],TOWGS");
    add_srs_wkt (p, 3,
		 "84[-162.619,-276.959,-161.764,0.067753,-2.24365,-1.15883");
    add_srs_wkt (p, 4,
		 ",-1.09425],AUTHORITY[\"EPSG\",\"6611\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 5,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 6,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"4611\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 8,
		 ",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"latitude_of_origin\",22.31213333333334],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "central_meridian\",114.1785555555556],PARAMETER[\"scale_");
    add_srs_wkt (p, 11,
		 "factor\",1],PARAMETER[\"false_easting\",836694.05],PARAM");
    add_srs_wkt (p, 12,
		 "ETER[\"false_northing\",819069.8],AUTHORITY[\"EPSG\",\"2");
    add_srs_wkt (p, 13,
		 "326\"],AXIS[\"Northing\",NORTH],AXIS[\"Easting\",EAST]]");
    p = add_epsg_def (first, last, 2327, "epsg", 2327,
		      "Xian 1980 / Gauss-Kruger zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=13500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 13\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",75],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",13500000],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2327\"],A");
    add_srs_wkt (p, 11, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2328, "epsg", 2328,
		      "Xian 1980 / Gauss-Kruger zone 14");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=14500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 14\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",81],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",14500000],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2328\"],A");
    add_srs_wkt (p, 11, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2329, "epsg", 2329,
		      "Xian 1980 / Gauss-Kruger zone 15");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=15500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 15\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",87],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",15500000],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2329\"],A");
    add_srs_wkt (p, 11, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2330, "epsg", 2330,
		      "Xian 1980 / Gauss-Kruger zone 16");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=16500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 16\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",93],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",16500000],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2330\"],A");
    add_srs_wkt (p, 11, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2331, "epsg", 2331,
		      "Xian 1980 / Gauss-Kruger zone 17");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=17500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 17\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",99],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",17500000],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2331\"],A");
    add_srs_wkt (p, 11, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2332, "epsg", 2332,
		      "Xian 1980 / Gauss-Kruger zone 18");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=18500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 18\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",105],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",18500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2332\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2333, "epsg", 2333,
		      "Xian 1980 / Gauss-Kruger zone 19");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=19500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 19\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",111],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",19500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2333\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2334, "epsg", 2334,
		      "Xian 1980 / Gauss-Kruger zone 20");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=20500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 20\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",117],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",20500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2334\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2335, "epsg", 2335,
		      "Xian 1980 / Gauss-Kruger zone 21");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=21500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 21\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",123],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",21500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2335\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2336, "epsg", 2336,
		      "Xian 1980 / Gauss-Kruger zone 22");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=22500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 22\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",129],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",22500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2336\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2337, "epsg", 2337,
		      "Xian 1980 / Gauss-Kruger zone 23");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=23500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger zone 23\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",135],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",23500000],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2337\"],");
    add_srs_wkt (p, 11, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2338, "epsg", 2338,
		      "Xian 1980 / Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 75E\",GEOGCS[\"Xian");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",63781");
    add_srs_wkt (p, 2,
		 "40,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",75],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2338\"],AXI");
    add_srs_wkt (p, 11, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2339, "epsg", 2339,
		      "Xian 1980 / Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 81E\",GEOGCS[\"Xian");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",63781");
    add_srs_wkt (p, 2,
		 "40,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",81],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2339\"],AXI");
    add_srs_wkt (p, 11, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2340, "epsg", 2340,
		      "Xian 1980 / Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 87E\",GEOGCS[\"Xian");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",63781");
    add_srs_wkt (p, 2,
		 "40,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",87],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2340\"],AXI");
    add_srs_wkt (p, 11, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2341, "epsg", 2341,
		      "Xian 1980 / Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 93E\",GEOGCS[\"Xian");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",63781");
    add_srs_wkt (p, 2,
		 "40,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",93],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2341\"],AXI");
    add_srs_wkt (p, 11, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2342, "epsg", 2342,
		      "Xian 1980 / Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 99E\",GEOGCS[\"Xian");
    add_srs_wkt (p, 1,
		 " 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",63781");
    add_srs_wkt (p, 2,
		 "40,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",99],PARAMETER[\"scale_");
    add_srs_wkt (p, 9,
		 "factor\",1],PARAMETER[\"false_easting\",500000],PARAMETE");
    add_srs_wkt (p, 10,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2342\"],AXI");
    add_srs_wkt (p, 11, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2343, "epsg", 2343,
		      "Xian 1980 / Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 105E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",105],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2343\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2344, "epsg", 2344,
		      "Xian 1980 / Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 111E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",111],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2344\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2345, "epsg", 2345,
		      "Xian 1980 / Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 117E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",117],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2345\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2346, "epsg", 2346,
		      "Xian 1980 / Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 123E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",123],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2346\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2347, "epsg", 2347,
		      "Xian 1980 / Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 129E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",129],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2347\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2348, "epsg", 2348,
		      "Xian 1980 / Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / Gauss-Kruger CM 135E\",GEOGCS[\"Xia");
    add_srs_wkt (p, 1,
		 "n 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 1980\",6378");
    add_srs_wkt (p, 2,
		 "140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 5,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610\"]],UNIT");
    add_srs_wkt (p, 6,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 7,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 8,
		 "0],PARAMETER[\"central_meridian\",135],PARAMETER[\"scale");
    add_srs_wkt (p, 9,
		 "_factor\",1],PARAMETER[\"false_easting\",500000],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2348\"],AX");
    add_srs_wkt (p, 11, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2349, "epsg", 2349,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 25");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=25500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 25\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",75],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",255000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2349\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2350, "epsg", 2350,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 26");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=26500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 26\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",78],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",265000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2350\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2351, "epsg", 2351,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 27");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=27500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 27\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",81],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",275000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2351\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2352, "epsg", 2352,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 28");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=28500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 28\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",84],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",285000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2352\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2353, "epsg", 2353,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 29");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=29500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 29\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",87],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",295000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2353\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2354, "epsg", 2354,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 30");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=30500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 30\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",90],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",305000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2354\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2355, "epsg", 2355,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 31");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=31500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 31\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",93],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",315000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2355\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2356, "epsg", 2356,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 32");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=32500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 32\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",96],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",325000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2356\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2357, "epsg", 2357,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 33");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=33500000 +y_0=0");
    add_proj4text (p, 1, " +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 33\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",99],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"scale_factor\",1],PARAMETER[\"false_easting\",335000");
    add_srs_wkt (p, 10,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2357\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2358, "epsg", 2358,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 34");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=34500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 34\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",102],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",34500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2358\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2359, "epsg", 2359,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 35");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=35500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 35\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",105],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",35500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2359\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2360, "epsg", 2360,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 36");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=36500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 36\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",108],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",36500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2360\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2361, "epsg", 2361,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 37");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=37500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 37\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",111],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",37500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2361\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2362, "epsg", 2362,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 38");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=38500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 38\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",114],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",38500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2362\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2363, "epsg", 2363,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 39");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=39500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 39\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",117],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",39500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2363\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2364, "epsg", 2364,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 40");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 40\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",120],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",40500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2364\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2365, "epsg", 2365,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 41");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=41500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 41\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",123],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",41500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2365\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2366, "epsg", 2366,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 42");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=42500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 42\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",126],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",42500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2366\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2367, "epsg", 2367,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 43");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=43500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 43\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",129],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",43500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2367\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2368, "epsg", 2368,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 44");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=44500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 44\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",132],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",44500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2368\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2369, "epsg", 2369,
		      "Xian 1980 / 3-degree Gauss-Kruger zone 45");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=45500000 +y_0=");
    add_proj4text (p, 1, "0 +a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger zone 45\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",135],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",45500");
    add_srs_wkt (p, 10,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 11, "\"2369\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2370, "epsg", 2370,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 75E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",75],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "70\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2371, "epsg", 2371,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 78E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 78E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",78],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "71\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2372, "epsg", 2372,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 81E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",81],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "72\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2373, "epsg", 2373,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 84E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 84E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",84],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "73\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2374, "epsg", 2374,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 87E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",87],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "74\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2375, "epsg", 2375,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 90E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 90E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",90],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "75\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2376, "epsg", 2376,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 93E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",93],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "76\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2377, "epsg", 2377,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 96E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 96E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",96],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "77\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2378, "epsg", 2378,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 99E\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 198");
    add_srs_wkt (p, 2,
		 "0\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHOR");
    add_srs_wkt (p, 3,
		 "ITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORITY");
    add_srs_wkt (p, 4,
		 "[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328");
    add_srs_wkt (p, 5,
		 ",AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4610");
    add_srs_wkt (p, 6,
		 "\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJ");
    add_srs_wkt (p, 7,
		 "ECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_");
    add_srs_wkt (p, 8,
		 "origin\",0],PARAMETER[\"central_meridian\",99],PARAMETER");
    add_srs_wkt (p, 9,
		 "[\"scale_factor\",1],PARAMETER[\"false_easting\",500000]");
    add_srs_wkt (p, 10,
		 ",PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"23");
    add_srs_wkt (p, 11, "78\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2379, "epsg", 2379,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 102E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 102E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",102],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2379\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2380, "epsg", 2380,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 105E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",105],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2380\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2381, "epsg", 2381,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 108E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 108E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",108],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2381\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2382, "epsg", 2382,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 111E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",111],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2382\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2383, "epsg", 2383,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 114E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 114E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",114],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2383\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2384, "epsg", 2384,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 117E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",117],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2384\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2385, "epsg", 2385,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 120E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 120E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",120],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2385\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2386, "epsg", 2386,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 123E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",123],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2386\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2387, "epsg", 2387,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 126E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 126E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",126],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2387\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2388, "epsg", 2388,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 129E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",129],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2388\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2389, "epsg", 2389,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 132E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 132E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",132],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2389\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2390, "epsg", 2390,
		      "Xian 1980 / 3-degree Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+a=6378140 +b=6356755.288157528 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Xian 1980 / 3-degree Gauss-Kruger CM 135E\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"Xian 1980\",DATUM[\"Xian_1980\",SPHEROID[\"Xian 19");
    add_srs_wkt (p, 2,
		 "80\",6378140,298.257,AUTHORITY[\"EPSG\",\"7049\"]],AUTHO");
    add_srs_wkt (p, 3,
		 "RITY[\"EPSG\",\"6610\"]],PRIMEM[\"Greenwich\",0,AUTHORIT");
    add_srs_wkt (p, 4,
		 "Y[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199432");
    add_srs_wkt (p, 5,
		 "8,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"461");
    add_srs_wkt (p, 6,
		 "0\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PRO");
    add_srs_wkt (p, 7,
		 "JECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_of");
    add_srs_wkt (p, 8,
		 "_origin\",0],PARAMETER[\"central_meridian\",135],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 10,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 11, "2390\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2391, "epsg", 2391, "KKJ / Finland zone 1");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=1500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"KKJ / Finland zone 1\",GEOGCS[\"KKJ\",DATUM[\"K");
    add_srs_wkt (p, 1,
		 "artastokoordinaattijarjestelma_1966\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6123\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4123\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",21],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",1");
    add_srs_wkt (p, 10,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 11, "\",\"2391\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2392, "epsg", 2392, "KKJ / Finland zone 2");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=1 +x_0=2500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"KKJ / Finland zone 2\",GEOGCS[\"KKJ\",DATUM[\"K");
    add_srs_wkt (p, 1,
		 "artastokoordinaattijarjestelma_1966\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6123\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4123\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",24],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",2");
    add_srs_wkt (p, 10,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 11, "\",\"2392\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2393, "epsg", 2393,
		      "KKJ / Finland Uniform Coordinate System");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=3500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"KKJ / Finland Uniform Coordinate System\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"KKJ\",DATUM[\"Kartastokoordinaattijarjestelma_1966\"");
    add_srs_wkt (p, 2,
		 ",SPHEROID[\"International 1924\",6378388,297,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7022\"]],AUTHORITY[\"EPSG\",\"6123\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4123\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",27],PARAMETER[\"scale_factor\",1],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_easting\",3500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2393\"],AXIS[\"X\",NORTH],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",EAST]]");
    p = add_epsg_def (first, last, 2394, "epsg", 2394, "KKJ / Finland zone 4");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=4500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=intl +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"KKJ / Finland zone 4\",GEOGCS[\"KKJ\",DATUM[\"K");
    add_srs_wkt (p, 1,
		 "artastokoordinaattijarjestelma_1966\",SPHEROID[\"Interna");
    add_srs_wkt (p, 2,
		 "tional 1924\",6378388,297,AUTHORITY[\"EPSG\",\"7022\"]],");
    add_srs_wkt (p, 3,
		 "AUTHORITY[\"EPSG\",\"6123\"]],PRIMEM[\"Greenwich\",0,AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251");
    add_srs_wkt (p, 5,
		 "994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 6,
		 "\"4123\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]");
    add_srs_wkt (p, 7,
		 "],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitu");
    add_srs_wkt (p, 8,
		 "de_of_origin\",0],PARAMETER[\"central_meridian\",30],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",4");
    add_srs_wkt (p, 10,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 11, "\",\"2394\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2395, "epsg", 2395,
		      "South Yemen / Gauss-Kruger zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=8500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=-76,-138,67,0,0,0,0 +units=m +no_d");
    add_proj4text (p, 2, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"South Yemen / Gauss-Kruger zone 8\",GEOGCS[\"So");
    add_srs_wkt (p, 1,
		 "uth Yemen\",DATUM[\"South_Yemen\",SPHEROID[\"Krassowsky ");
    add_srs_wkt (p, 2,
		 "1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TOWGS");
    add_srs_wkt (p, 3,
		 "84[-76,-138,67,0,0,0,0],AUTHORITY[\"EPSG\",\"6164\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4164\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",45],PARAMETER[\"scale_factor\",1],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_easting\",8500000],PARAMETER[\"false_northi");
    add_srs_wkt (p, 11,
		 "ng\",0],AUTHORITY[\"EPSG\",\"2395\"],AXIS[\"X\",NORTH],A");
    add_srs_wkt (p, 12, "XIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2396, "epsg", 2396,
		      "South Yemen / Gauss-Kruger zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=9500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=-76,-138,67,0,0,0,0 +units=m +no_d");
    add_proj4text (p, 2, "efs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"South Yemen / Gauss-Kruger zone 9\",GEOGCS[\"So");
    add_srs_wkt (p, 1,
		 "uth Yemen\",DATUM[\"South_Yemen\",SPHEROID[\"Krassowsky ");
    add_srs_wkt (p, 2,
		 "1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TOWGS");
    add_srs_wkt (p, 3,
		 "84[-76,-138,67,0,0,0,0],AUTHORITY[\"EPSG\",\"6164\"]],PR");
    add_srs_wkt (p, 4,
		 "IMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[");
    add_srs_wkt (p, 5,
		 "\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122");
    add_srs_wkt (p, 6,
		 "\"]],AUTHORITY[\"EPSG\",\"4164\"]],UNIT[\"metre\",1,AUTH");
    add_srs_wkt (p, 7,
		 "ORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercat");
    add_srs_wkt (p, 8,
		 "or\"],PARAMETER[\"latitude_of_origin\",0],PARAMETER[\"ce");
    add_srs_wkt (p, 9,
		 "ntral_meridian\",51],PARAMETER[\"scale_factor\",1],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"false_easting\",9500000],PARAMETER[\"false_northi");
    add_srs_wkt (p, 11,
		 "ng\",0],AUTHORITY[\"EPSG\",\"2396\"],AXIS[\"X\",NORTH],A");
    add_srs_wkt (p, 12, "XIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2397, "epsg", 2397,
		      "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=9 +k=1 +x_0=3500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3");
    add_srs_wkt (p, 1,
		 "\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_83\"");
    add_srs_wkt (p, 2,
		 ",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",9],PARAMETER[\"scale_factor\",1],PARAMETER[\"fa");
    add_srs_wkt (p, 10,
		 "lse_easting\",3500000],PARAMETER[\"false_northing\",0],A");
    add_srs_wkt (p, 11,
		 "UTHORITY[\"EPSG\",\"2397\"],AXIS[\"X\",NORTH],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",EAST]]");
    p = add_epsg_def (first, last, 2398, "epsg", 2398,
		      "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=12 +k=1 +x_0=4500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4");
    add_srs_wkt (p, 1,
		 "\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_83\"");
    add_srs_wkt (p, 2,
		 ",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",12],PARAMETER[\"scale_factor\",1],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_easting\",4500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2398\"],AXIS[\"X\",NORTH],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",EAST]]");
    p = add_epsg_def (first, last, 2399, "epsg", 2399,
		      "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=1 +x_0=5500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5");
    add_srs_wkt (p, 1,
		 "\",GEOGCS[\"Pulkovo 1942(83)\",DATUM[\"Pulkovo_1942_83\"");
    add_srs_wkt (p, 2,
		 ",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7024\"]],AUTHORITY[\"EPSG\",\"6178\"]],PRIMEM[\"");
    add_srs_wkt (p, 4,
		 "Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degre");
    add_srs_wkt (p, 5,
		 "e\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AU");
    add_srs_wkt (p, 6,
		 "THORITY[\"EPSG\",\"4178\"]],UNIT[\"metre\",1,AUTHORITY[\"");
    add_srs_wkt (p, 7,
		 "EPSG\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PA");
    add_srs_wkt (p, 8,
		 "RAMETER[\"latitude_of_origin\",0],PARAMETER[\"central_me");
    add_srs_wkt (p, 9,
		 "ridian\",15],PARAMETER[\"scale_factor\",1],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_easting\",5500000],PARAMETER[\"false_northing\",0],");
    add_srs_wkt (p, 11,
		 "AUTHORITY[\"EPSG\",\"2399\"],AXIS[\"X\",NORTH],AXIS[\"Y\"");
    add_srs_wkt (p, 12, ",EAST]]");
    p = add_epsg_def (first, last, 2400, "epsg", 2400,
		      "RT90 2.5 gon W (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15.80827777777778 +k=1 +x_0=");
    add_proj4text (p, 1, "1500000 +y_0=0 +ellps=bessel +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"RT90 2.5 gon W (deprecated)\",GEOGCS[\"RT90\",D");
    add_srs_wkt (p, 1,
		 "ATUM[\"Rikets_koordinatsystem_1990\",SPHEROID[\"Bessel 1");
    add_srs_wkt (p, 2,
		 "841\",6377397.155,299.1528128,AUTHORITY[\"EPSG\",\"7004\"");
    add_srs_wkt (p, 3,
		 "]],AUTHORITY[\"EPSG\",\"6124\"]],PRIMEM[\"Greenwich\",0,");
    add_srs_wkt (p, 4,
		 "AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329");
    add_srs_wkt (p, 5,
		 "251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 6,
		 "\",\"4124\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001");
    add_srs_wkt (p, 7,
		 "\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"lat");
    add_srs_wkt (p, 8,
		 "itude_of_origin\",0],PARAMETER[\"central_meridian\",15.8");
    add_srs_wkt (p, 9,
		 "0827777777778],PARAMETER[\"scale_factor\",1],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "false_easting\",1500000],PARAMETER[\"false_northing\",0]");
    add_srs_wkt (p, 11,
		 ",AUTHORITY[\"EPSG\",\"2400\"],AXIS[\"X\",NORTH],AXIS[\"Y");
    add_srs_wkt (p, 12, "\",EAST]]");
    p = add_epsg_def (first, last, 2401, "epsg", 2401,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 25");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=25500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 25\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",7");
    add_srs_wkt (p, 9,
		 "5],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",25500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2401\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2402, "epsg", 2402,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 26");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=26500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 26\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",7");
    add_srs_wkt (p, 9,
		 "8],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",26500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2402\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2403, "epsg", 2403,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 27");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=27500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 27\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "1],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",27500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2403\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2404, "epsg", 2404,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 28");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=28500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 28\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "4],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",28500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2404\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2405, "epsg", 2405,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 29");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=29500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 29\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "7],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",29500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2405\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2406, "epsg", 2406,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 30");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=30500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 30\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "0],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",30500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2406\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2407, "epsg", 2407,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 31");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=31500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 31\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "3],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",31500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2407\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2408, "epsg", 2408,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 32");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=32500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 32\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",32500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2408\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2409, "epsg", 2409,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 33");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=33500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 33\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "9],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",33500000],PARAMETER[\"false_northing\",0],AUTHORITY");
    add_srs_wkt (p, 11,
		 "[\"EPSG\",\"2409\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2410, "epsg", 2410,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 34");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=34500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 34\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "02],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",34500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2410\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2411, "epsg", 2411,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 35");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=35500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 35\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "05],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",35500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2411\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2412, "epsg", 2412,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 36");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=36500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 36\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "08],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",36500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2412\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2413, "epsg", 2413,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 37");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=37500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 37\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "11],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",37500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2413\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2414, "epsg", 2414,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 38");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=38500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 38\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "14],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",38500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2414\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
#endif /* full EPSG initialization enabled/disabled */
    if (first == last)
	last = first;		/* suppressing stupid compiler warnings */
}

static void
initialize_epsg_03 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2415, "epsg", 2415,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 39");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=39500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 39\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "17],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",39500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2415\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2416, "epsg", 2416,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 40");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 40\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "20],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",40500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2416\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2417, "epsg", 2417,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 41");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=41500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 41\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "23],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",41500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2417\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2418, "epsg", 2418,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 42");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=42500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 42\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "26],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",42500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2418\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2419, "epsg", 2419,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 43");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=43500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 43\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "29],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",43500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2419\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2420, "epsg", 2420,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 44");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=44500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 44\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "32],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",44500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2420\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2421, "epsg", 2421,
		      "Beijing 1954 / 3-degree Gauss-Kruger zone 45");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=45500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger zone 45\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "35],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",45500000],PARAMETER[\"false_northing\",0],AUTHORIT");
    add_srs_wkt (p, 11,
		 "Y[\"EPSG\",\"2421\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    add_srs_wkt (p, 12, "");
    p = add_epsg_def (first, last, 2422, "epsg", 2422,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 75E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",7");
    add_srs_wkt (p, 9,
		 "5],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2422\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2423, "epsg", 2423,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 78E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 78E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",7");
    add_srs_wkt (p, 9,
		 "8],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2423\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2424, "epsg", 2424,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 81E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "1],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2424\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2425, "epsg", 2425,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 84E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 84E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "4],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2425\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2426, "epsg", 2426,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 87E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",8");
    add_srs_wkt (p, 9,
		 "7],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2426\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2427, "epsg", 2427,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 90E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 90E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "0],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2427\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2428, "epsg", 2428,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 93E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "3],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2428\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2429, "epsg", 2429,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 96E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 96E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2429\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2430, "epsg", 2430,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 99E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich\"");
    add_srs_wkt (p, 4,
		 ",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745");
    add_srs_wkt (p, 5,
		 "329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"E");
    add_srs_wkt (p, 6,
		 "PSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9");
    add_srs_wkt (p, 7,
		 "001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",9");
    add_srs_wkt (p, 9,
		 "9],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easti");
    add_srs_wkt (p, 10,
		 "ng\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"");
    add_srs_wkt (p, 11, "EPSG\",\"2430\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2431, "epsg", 2431,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 102E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 102E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "02],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2431\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2432, "epsg", 2432,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 105E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "05],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2432\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2433, "epsg", 2433,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 108E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 108E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "08],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2433\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2434, "epsg", 2434,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 111E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "11],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2434\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2435, "epsg", 2435,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 114E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 114E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "14],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2435\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2436, "epsg", 2436,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 117E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "17],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2436\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2437, "epsg", 2437,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 120E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 120E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "20],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2437\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2438, "epsg", 2438,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 123E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "23],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2438\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2439, "epsg", 2439,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 126E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 126E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "26],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2439\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2440, "epsg", 2440,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 129E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "29],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2440\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2441, "epsg", 2441,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 132E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 132E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "32],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2441\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2442, "epsg", 2442,
		      "Beijing 1954 / 3-degree Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Beijing 1954 / 3-degree Gauss-Kruger CM 135E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Beijing 1954\",DATUM[\"Beijing_1954\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],AUTHORITY[\"EPSG\",\"6214\"]],PRIMEM[\"Greenwich");
    add_srs_wkt (p, 4,
		 "\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017");
    add_srs_wkt (p, 5,
		 "45329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"4214\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"");
    add_srs_wkt (p, 8,
		 "latitude_of_origin\",0],PARAMETER[\"central_meridian\",1");
    add_srs_wkt (p, 9,
		 "35],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_east");
    add_srs_wkt (p, 10,
		 "ing\",500000],PARAMETER[\"false_northing\",0],AUTHORITY[");
    add_srs_wkt (p, 11,
		 "\"EPSG\",\"2442\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2443, "epsg", 2443,
		      "JGD2000 / Japan Plane Rectangular CS I");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=33 +lon_0=129.5 +k=0.9999 +x_0=0 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_proj4text (p, 2, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS I\",GEOGCS");
    add_srs_wkt (p, 1,
		 "[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 4,
		 "\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8");
    add_srs_wkt (p, 5,
		 "901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",33],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",129.5],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2443\"],AXI");
    add_srs_wkt (p, 12, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2444, "epsg", 2444,
		      "JGD2000 / Japan Plane Rectangular CS II");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=33 +lon_0=131 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS II\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",33],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",131],PARAMETER[\"scale_fa");
    add_srs_wkt (p, 10,
		 "ctor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"false_northing\",0],AUTHORITY[\"EPSG\",\"2444\"],AXIS[");
    add_srs_wkt (p, 12, "\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2445, "epsg", 2445,
		      "JGD2000 / Japan Plane Rectangular CS III");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=132.1666666666667 +k=0.9999");
    add_proj4text (p, 1,
		   " +x_0=0 +y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS III\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",36],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",132.1666666666667],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",0.9999],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2445\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2446, "epsg", 2446,
		      "JGD2000 / Japan Plane Rectangular CS IV");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=33 +lon_0=133.5 +k=0.9999 +x_0=0 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_proj4text (p, 2, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS IV\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",33],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",133.5],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2446\"],AXI");
    add_srs_wkt (p, 12, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2447, "epsg", 2447,
		      "JGD2000 / Japan Plane Rectangular CS V");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=134.3333333333333 +k=0.9999");
    add_proj4text (p, 1,
		   " +x_0=0 +y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS V\",GEOGCS");
    add_srs_wkt (p, 1,
		 "[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 4,
		 "\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8");
    add_srs_wkt (p, 5,
		 "901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",36],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",134.3333333333333],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",0.9999],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2447\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2448, "epsg", 2448,
		      "JGD2000 / Japan Plane Rectangular CS VI");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=136 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS VI\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",36],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",136],PARAMETER[\"scale_fa");
    add_srs_wkt (p, 10,
		 "ctor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"false_northing\",0],AUTHORITY[\"EPSG\",\"2448\"],AXIS[");
    add_srs_wkt (p, 12, "\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2449, "epsg", 2449,
		      "JGD2000 / Japan Plane Rectangular CS VII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=137.1666666666667 +k=0.9999");
    add_proj4text (p, 1,
		   " +x_0=0 +y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS VII\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",36],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",137.1666666666667],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",0.9999],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2449\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2450, "epsg", 2450,
		      "JGD2000 / Japan Plane Rectangular CS VIII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=138.5 +k=0.9999 +x_0=0 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_proj4text (p, 2, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS VIII\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",S");
    add_srs_wkt (p, 2,
		 "PHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"E");
    add_srs_wkt (p, 3,
		 "PSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 5,
		 "\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY");
    add_srs_wkt (p, 6,
		 "[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[");
    add_srs_wkt (p, 7,
		 "\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"T");
    add_srs_wkt (p, 8,
		 "ransverse_Mercator\"],PARAMETER[\"latitude_of_origin\",3");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"central_meridian\",138.5],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",0.9999],PARAMETER[\"false_easting\",0],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2450\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2451, "epsg", 2451,
		      "JGD2000 / Japan Plane Rectangular CS IX");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=36 +lon_0=139.8333333333333 +k=0.9999");
    add_proj4text (p, 1,
		   " +x_0=0 +y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS IX\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",36],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",139.8333333333333],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",0.9999],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2451\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2452, "epsg", 2452,
		      "JGD2000 / Japan Plane Rectangular CS X");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=40 +lon_0=140.8333333333333 +k=0.9999");
    add_proj4text (p, 1,
		   " +x_0=0 +y_0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +unit");
    add_proj4text (p, 2, "s=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS X\",GEOGCS");
    add_srs_wkt (p, 1,
		 "[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPHE");
    add_srs_wkt (p, 2,
		 "ROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 3,
		 "\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 4,
		 "\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8");
    add_srs_wkt (p, 5,
		 "901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",40],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",140.8333333333333],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",0.9999],PARAMETER[\"false_easting\"");
    add_srs_wkt (p, 11,
		 ",0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2452\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2453, "epsg", 2453,
		      "JGD2000 / Japan Plane Rectangular CS XI");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=44 +lon_0=140.25 +k=0.9999 +x_0=0 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XI\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",44],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",140.25],PARAMETER[\"scale");
    add_srs_wkt (p, 10,
		 "_factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2453\"],AX");
    add_srs_wkt (p, 12, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2454, "epsg", 2454,
		      "JGD2000 / Japan Plane Rectangular CS XII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=44 +lon_0=142.25 +k=0.9999 +x_0=0 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XII\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",44],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",142.25],PARAMETER[\"scale");
    add_srs_wkt (p, 10,
		 "_factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2454\"],AX");
    add_srs_wkt (p, 12, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2455, "epsg", 2455,
		      "JGD2000 / Japan Plane Rectangular CS XIII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=44 +lon_0=144.25 +k=0.9999 +x_0=0 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XIII\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",S");
    add_srs_wkt (p, 2,
		 "PHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"E");
    add_srs_wkt (p, 3,
		 "PSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 5,
		 "\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY");
    add_srs_wkt (p, 6,
		 "[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[");
    add_srs_wkt (p, 7,
		 "\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"T");
    add_srs_wkt (p, 8,
		 "ransverse_Mercator\"],PARAMETER[\"latitude_of_origin\",4");
    add_srs_wkt (p, 9,
		 "4],PARAMETER[\"central_meridian\",144.25],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",0.9999],PARAMETER[\"false_easting\",0],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2455\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2456, "epsg", 2456,
		      "JGD2000 / Japan Plane Rectangular CS XIV");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=26 +lon_0=142 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XIV\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",26],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",142],PARAMETER[\"scale_fa");
    add_srs_wkt (p, 10,
		 "ctor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"false_northing\",0],AUTHORITY[\"EPSG\",\"2456\"],AXIS[");
    add_srs_wkt (p, 12, "\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2457, "epsg", 2457,
		      "JGD2000 / Japan Plane Rectangular CS XV");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=26 +lon_0=127.5 +k=0.9999 +x_0=0 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_proj4text (p, 2, "");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XV\",GEOGC");
    add_srs_wkt (p, 1,
		 "S[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SPH");
    add_srs_wkt (p, 2,
		 "EROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPS");
    add_srs_wkt (p, 3,
		 "G\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",26],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",127.5],PARAMETER[\"scale_");
    add_srs_wkt (p, 10,
		 "factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETE");
    add_srs_wkt (p, 11,
		 "R[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2457\"],AXI");
    add_srs_wkt (p, 12, "S[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2458, "epsg", 2458,
		      "JGD2000 / Japan Plane Rectangular CS XVI");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=26 +lon_0=124 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XVI\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",26],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",124],PARAMETER[\"scale_fa");
    add_srs_wkt (p, 10,
		 "ctor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"false_northing\",0],AUTHORITY[\"EPSG\",\"2458\"],AXIS[");
    add_srs_wkt (p, 12, "\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2459, "epsg", 2459,
		      "JGD2000 / Japan Plane Rectangular CS XVII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=26 +lon_0=131 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XVII\",GEO");
    add_srs_wkt (p, 1,
		 "GCS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",S");
    add_srs_wkt (p, 2,
		 "PHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"E");
    add_srs_wkt (p, 3,
		 "PSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 4,
		 "\",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 5,
		 "\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY");
    add_srs_wkt (p, 6,
		 "[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[");
    add_srs_wkt (p, 7,
		 "\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"T");
    add_srs_wkt (p, 8,
		 "ransverse_Mercator\"],PARAMETER[\"latitude_of_origin\",2");
    add_srs_wkt (p, 9,
		 "6],PARAMETER[\"central_meridian\",131],PARAMETER[\"scale");
    add_srs_wkt (p, 10,
		 "_factor\",0.9999],PARAMETER[\"false_easting\",0],PARAMET");
    add_srs_wkt (p, 11,
		 "ER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2459\"],AX");
    add_srs_wkt (p, 12, "IS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2460, "epsg", 2460,
		      "JGD2000 / Japan Plane Rectangular CS XVIII");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=20 +lon_0=136 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XVIII\",GE");
    add_srs_wkt (p, 1,
		 "OGCS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",");
    add_srs_wkt (p, 2,
		 "SPHEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"");
    add_srs_wkt (p, 3,
		 "EPSG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 4,
		 "G\",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORIT");
    add_srs_wkt (p, 6,
		 "Y[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT");
    add_srs_wkt (p, 7,
		 "[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"");
    add_srs_wkt (p, 8,
		 "Transverse_Mercator\"],PARAMETER[\"latitude_of_origin\",");
    add_srs_wkt (p, 9,
		 "20],PARAMETER[\"central_meridian\",136],PARAMETER[\"scal");
    add_srs_wkt (p, 10,
		 "e_factor\",0.9999],PARAMETER[\"false_easting\",0],PARAME");
    add_srs_wkt (p, 11,
		 "TER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2460\"],A");
    add_srs_wkt (p, 12, "XIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2461, "epsg", 2461,
		      "JGD2000 / Japan Plane Rectangular CS XIX");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=26 +lon_0=154 +k=0.9999 +x_0=0 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"JGD2000 / Japan Plane Rectangular CS XIX\",GEOG");
    add_srs_wkt (p, 1,
		 "CS[\"JGD2000\",DATUM[\"Japanese_Geodetic_Datum_2000\",SP");
    add_srs_wkt (p, 2,
		 "HEROID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EP");
    add_srs_wkt (p, 3,
		 "SG\",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 4,
		 ",\"6612\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 5,
		 "8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"");
    add_srs_wkt (p, 6,
		 "EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4612\"]],UNIT[\"m");
    add_srs_wkt (p, 7,
		 "etre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Tran");
    add_srs_wkt (p, 8,
		 "sverse_Mercator\"],PARAMETER[\"latitude_of_origin\",26],");
    add_srs_wkt (p, 9,
		 "PARAMETER[\"central_meridian\",154],PARAMETER[\"scale_fa");
    add_srs_wkt (p, 10,
		 "ctor\",0.9999],PARAMETER[\"false_easting\",0],PARAMETER[");
    add_srs_wkt (p, 11,
		 "\"false_northing\",0],AUTHORITY[\"EPSG\",\"2461\"],AXIS[");
    add_srs_wkt (p, 12, "\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2462, "epsg", 2462,
		      "Albanian 1987 / Gauss-Kruger zone 4");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=4500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Albanian 1987 / Gauss-Kruger zone 4\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Albanian 1987\",DATUM[\"Albanian_1987\",SPHEROID[\"Krass");
    add_srs_wkt (p, 2,
		 "owsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]]");
    add_srs_wkt (p, 3,
		 ",AUTHORITY[\"EPSG\",\"6191\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 5,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"4191\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"");
    add_srs_wkt (p, 7,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 8,
		 "ude_of_origin\",0],PARAMETER[\"central_meridian\",21],PA");
    add_srs_wkt (p, 9,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 10,
		 "4500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 11, "G\",\"2462\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2463, "epsg", 2463,
		      "Pulkovo 1995 / Gauss-Kruger CM 21E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 21E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",21],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2463\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2464, "epsg", 2464,
		      "Pulkovo 1995 / Gauss-Kruger CM 27E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 27E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",27],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2464\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2465, "epsg", 2465,
		      "Pulkovo 1995 / Gauss-Kruger CM 33E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 33E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",33],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2465\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2466, "epsg", 2466,
		      "Pulkovo 1995 / Gauss-Kruger CM 39E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 39E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",39],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2466\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2467, "epsg", 2467,
		      "Pulkovo 1995 / Gauss-Kruger CM 45E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 45E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",45],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2467\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2468, "epsg", 2468,
		      "Pulkovo 1995 / Gauss-Kruger CM 51E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 51E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",51],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2468\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2469, "epsg", 2469,
		      "Pulkovo 1995 / Gauss-Kruger CM 57E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 57E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",57],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2469\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2470, "epsg", 2470,
		      "Pulkovo 1995 / Gauss-Kruger CM 63E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 63E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",63],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2470\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2471, "epsg", 2471,
		      "Pulkovo 1995 / Gauss-Kruger CM 69E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 69E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",69],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2471\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2472, "epsg", 2472,
		      "Pulkovo 1995 / Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 75E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",75],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2472\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2473, "epsg", 2473,
		      "Pulkovo 1995 / Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 81E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",81],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2473\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2474, "epsg", 2474,
		      "Pulkovo 1995 / Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 87E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",87],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2474\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2475, "epsg", 2475,
		      "Pulkovo 1995 / Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 93E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",93],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2475\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2476, "epsg", 2476,
		      "Pulkovo 1995 / Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 99E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",99],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2476\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2477, "epsg", 2477,
		      "Pulkovo 1995 / Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 105E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",105],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2477\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2478, "epsg", 2478,
		      "Pulkovo 1995 / Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 111E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",111],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2478\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2479, "epsg", 2479,
		      "Pulkovo 1995 / Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 117E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",117],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2479\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2480, "epsg", 2480,
		      "Pulkovo 1995 / Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 123E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",123],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2480\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2481, "epsg", 2481,
		      "Pulkovo 1995 / Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 129E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",129],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2481\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2482, "epsg", 2482,
		      "Pulkovo 1995 / Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 135E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",135],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2482\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2483, "epsg", 2483,
		      "Pulkovo 1995 / Gauss-Kruger CM 141E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=141 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 141E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",141],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2483\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2484, "epsg", 2484,
		      "Pulkovo 1995 / Gauss-Kruger CM 147E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=147 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 147E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",147],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2484\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2485, "epsg", 2485,
		      "Pulkovo 1995 / Gauss-Kruger CM 153E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=153 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 153E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",153],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2485\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2486, "epsg", 2486,
		      "Pulkovo 1995 / Gauss-Kruger CM 159E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=159 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 159E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",159],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2486\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2487, "epsg", 2487,
		      "Pulkovo 1995 / Gauss-Kruger CM 165E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=165 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 165E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",165],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2487\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2488, "epsg", 2488,
		      "Pulkovo 1995 / Gauss-Kruger CM 171E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=171 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 171E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",171],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2488\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2489, "epsg", 2489,
		      "Pulkovo 1995 / Gauss-Kruger CM 177E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=177 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 177E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",177],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2489\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2490, "epsg", 2490,
		      "Pulkovo 1995 / Gauss-Kruger CM 177W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-177 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 177W\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",-177],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2490\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2491, "epsg", 2491,
		      "Pulkovo 1995 / Gauss-Kruger CM 171W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-171 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / Gauss-Kruger CM 171W\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUTHORITY[");
    add_srs_wkt (p, 4,
		 "\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"E");
    add_srs_wkt (p, 5,
		 "PSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUT");
    add_srs_wkt (p, 6,
		 "HORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4200\"]]");
    add_srs_wkt (p, 7,
		 ",UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTI");
    add_srs_wkt (p, 8,
		 "ON[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_orig");
    add_srs_wkt (p, 9,
		 "in\",0],PARAMETER[\"central_meridian\",-171],PARAMETER[\"");
    add_srs_wkt (p, 10,
		 "scale_factor\",1],PARAMETER[\"false_easting\",500000],PA");
    add_srs_wkt (p, 11,
		 "RAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2491\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2492, "epsg", 2492,
		      "Pulkovo 1942 / Gauss-Kruger CM 9E (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=9 +k=1 +x_0=500000 +y_0=0 +e");
    add_proj4text (p, 1, "llps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 9E (deprecated)\"");
    add_srs_wkt (p, 1,
		 ",GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID");
    add_srs_wkt (p, 2,
		 "[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 3,
		 "7024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",9],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2492\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2493, "epsg", 2493,
		      "Pulkovo 1942 / Gauss-Kruger CM 15E (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=15 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 15E (deprecated)");
    add_srs_wkt (p, 1,
		 "\",GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHERO");
    add_srs_wkt (p, 2,
		 "ID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 3,
		 "\"7024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12]");
    add_srs_wkt (p, 4,
		 ",AUTHORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AU");
    add_srs_wkt (p, 5,
		 "THORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925");
    add_srs_wkt (p, 6,
		 "1994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 7,
		 ",\"4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"");
    add_srs_wkt (p, 8,
		 "]],PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latit");
    add_srs_wkt (p, 9,
		 "ude_of_origin\",0],PARAMETER[\"central_meridian\",15],PA");
    add_srs_wkt (p, 10,
		 "RAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2493\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2494, "epsg", 2494,
		      "Pulkovo 1942 / Gauss-Kruger CM 21E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 21E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",21],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2494\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2495, "epsg", 2495,
		      "Pulkovo 1942 / Gauss-Kruger CM 27E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 27E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",27],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2495\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2496, "epsg", 2496,
		      "Pulkovo 1942 / Gauss-Kruger CM 33E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 33E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",33],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2496\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2497, "epsg", 2497,
		      "Pulkovo 1942 / Gauss-Kruger CM 39E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 39E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",39],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2497\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2498, "epsg", 2498,
		      "Pulkovo 1942 / Gauss-Kruger CM 45E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 45E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",45],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2498\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2499, "epsg", 2499,
		      "Pulkovo 1942 / Gauss-Kruger CM 51E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 51E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",51],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2499\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2500, "epsg", 2500,
		      "Pulkovo 1942 / Gauss-Kruger CM 57E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 57E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",57],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2500\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2501, "epsg", 2501,
		      "Pulkovo 1942 / Gauss-Kruger CM 63E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 63E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",63],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2501\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2502, "epsg", 2502,
		      "Pulkovo 1942 / Gauss-Kruger CM 69E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 69E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",69],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2502\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2503, "epsg", 2503,
		      "Pulkovo 1942 / Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 75E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",75],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2503\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2504, "epsg", 2504,
		      "Pulkovo 1942 / Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 81E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",81],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2504\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2505, "epsg", 2505,
		      "Pulkovo 1942 / Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 87E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",87],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2505\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2506, "epsg", 2506,
		      "Pulkovo 1942 / Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 93E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",93],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2506\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2507, "epsg", 2507,
		      "Pulkovo 1942 / Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 99E\",GEOGCS[\"P");
    add_srs_wkt (p, 1,
		 "ulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassows");
    add_srs_wkt (p, 2,
		 "ky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],TO");
    add_srs_wkt (p, 3,
		 "WGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",99],PARAMETER[\"sca");
    add_srs_wkt (p, 10,
		 "le_factor\",1],PARAMETER[\"false_easting\",500000],PARAM");
    add_srs_wkt (p, 11,
		 "ETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2507\"],");
    add_srs_wkt (p, 12, "AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2508, "epsg", 2508,
		      "Pulkovo 1942 / Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 105E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",105],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2508\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2509, "epsg", 2509,
		      "Pulkovo 1942 / Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 111E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",111],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2509\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2510, "epsg", 2510,
		      "Pulkovo 1942 / Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 117E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",117],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2510\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2511, "epsg", 2511,
		      "Pulkovo 1942 / Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 123E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",123],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2511\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2512, "epsg", 2512,
		      "Pulkovo 1942 / Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 129E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",129],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2512\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2513, "epsg", 2513,
		      "Pulkovo 1942 / Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 135E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",135],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2513\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2514, "epsg", 2514,
		      "Pulkovo 1942 / Gauss-Kruger CM 141E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=141 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 141E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",141],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2514\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2515, "epsg", 2515,
		      "Pulkovo 1942 / Gauss-Kruger CM 147E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=147 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 147E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",147],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2515\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2516, "epsg", 2516,
		      "Pulkovo 1942 / Gauss-Kruger CM 153E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=153 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 153E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",153],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2516\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2517, "epsg", 2517,
		      "Pulkovo 1942 / Gauss-Kruger CM 159E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=159 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 159E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",159],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2517\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2518, "epsg", 2518,
		      "Pulkovo 1942 / Gauss-Kruger CM 165E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=165 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 165E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",165],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2518\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2519, "epsg", 2519,
		      "Pulkovo 1942 / Gauss-Kruger CM 171E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=171 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 171E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",171],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2519\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2520, "epsg", 2520,
		      "Pulkovo 1942 / Gauss-Kruger CM 177E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=177 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 177E\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",177],PARAMETER[\"sc");
    add_srs_wkt (p, 10,
		 "ale_factor\",1],PARAMETER[\"false_easting\",500000],PARA");
    add_srs_wkt (p, 11,
		 "METER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2520\"]");
    add_srs_wkt (p, 12, ",AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2521, "epsg", 2521,
		      "Pulkovo 1942 / Gauss-Kruger CM 177W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-177 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 177W\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",-177],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",1],PARAMETER[\"false_easting\",500000],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2521\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2522, "epsg", 2522,
		      "Pulkovo 1942 / Gauss-Kruger CM 171W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-171 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / Gauss-Kruger CM 171W\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"Krassow");
    add_srs_wkt (p, 2,
		 "sky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7024\"]],T");
    add_srs_wkt (p, 3,
		 "OWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTHORITY[\"");
    add_srs_wkt (p, 4,
		 "EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPS");
    add_srs_wkt (p, 5,
		 "G\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHO");
    add_srs_wkt (p, 6,
		 "RITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4284\"]],U");
    add_srs_wkt (p, 7,
		 "NIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION");
    add_srs_wkt (p, 8,
		 "[\"Transverse_Mercator\"],PARAMETER[\"latitude_of_origin");
    add_srs_wkt (p, 9,
		 "\",0],PARAMETER[\"central_meridian\",-171],PARAMETER[\"s");
    add_srs_wkt (p, 10,
		 "cale_factor\",1],PARAMETER[\"false_easting\",500000],PAR");
    add_srs_wkt (p, 11,
		 "AMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2522\"");
    add_srs_wkt (p, 12, "],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2523, "epsg", 2523,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 7");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=7500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 7\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",21],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",75000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2523\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2524, "epsg", 2524,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=1 +x_0=8500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 8\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",24],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",85000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2524\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2525, "epsg", 2525,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=9500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 9\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",27],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",95000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2525\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2526, "epsg", 2526,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=10500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 10\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",30],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",105");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2526\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2527, "epsg", 2527,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 11");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=11500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 11\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",33],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",115");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2527\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2528, "epsg", 2528,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 12");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=12500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 12\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",36],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",125");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2528\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2529, "epsg", 2529,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=13500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 13\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",39],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",135");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2529\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2530, "epsg", 2530,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 14");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=14500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 14\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",42],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",145");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2530\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2531, "epsg", 2531,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 15");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=15500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 15\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",45],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",155");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2531\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2532, "epsg", 2532,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 16");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=48 +k=1 +x_0=16500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 16\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",48],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",165");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2532\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2533, "epsg", 2533,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 17");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=17500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 17\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",51],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",175");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2533\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2534, "epsg", 2534,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 18");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=54 +k=1 +x_0=18500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 18\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",54],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",185");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2534\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2535, "epsg", 2535,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 19");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=19500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 19\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",57],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",195");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2535\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2536, "epsg", 2536,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 20");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=60 +k=1 +x_0=20500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 20\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",60],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",205");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2536\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2537, "epsg", 2537,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 21");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=21500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 21\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",63],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",215");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2537\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2538, "epsg", 2538,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 22");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=66 +k=1 +x_0=22500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 22\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",66],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",225");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2538\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2539, "epsg", 2539,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 23");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=23500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 23\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",69],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",235");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2539\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2540, "epsg", 2540,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 24");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=72 +k=1 +x_0=24500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 24\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",72],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",245");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2540\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2541, "epsg", 2541,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 25");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=25500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 25\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",75],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",255");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2541\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2542, "epsg", 2542,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 26");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=26500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 26\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",78],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",265");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2542\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
#endif /* full EPSG initialization enabled/disabled */
    if (first == last)
	last = first;		/* suppressing stupid compiler warnings */
}

static void
initialize_epsg_04 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2543, "epsg", 2543,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 27");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=27500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 27\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",81],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",275");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2543\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2544, "epsg", 2544,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 28");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=28500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 28\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",84],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",285");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2544\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2545, "epsg", 2545,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 29");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=29500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 29\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",87],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",295");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2545\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2546, "epsg", 2546,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 30");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=30500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 30\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",90],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",305");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2546\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2547, "epsg", 2547,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 31");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=31500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 31\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",93],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",315");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2547\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2548, "epsg", 2548,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 32");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=32500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 32\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",96],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",325");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2548\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2549, "epsg", 2549,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 33");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=33500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 33\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",99],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",335");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2549\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2550, "epsg", 2550,
		      "Samboja / UTM zone 50S (deprecated)");
    add_proj4text (p, 0,
		   "+proj=utm +zone=50 +south +ellps=bessel +towgs84=-404.78");
    add_proj4text (p, 1, ",685.68,45.47,0,0,0,0 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Samboja / UTM zone 50S (deprecated)\",GEOGCS[\"");
    add_srs_wkt (p, 1,
		 "Samboja\",DATUM[\"Samboja\",SPHEROID[\"Bessel 1841\",637");
    add_srs_wkt (p, 2,
		 "7397.155,299.1528128,AUTHORITY[\"EPSG\",\"7004\"]],TOWGS");
    add_srs_wkt (p, 3,
		 "84[-404.78,685.68,45.47,0,0,0,0],AUTHORITY[\"EPSG\",\"61");
    add_srs_wkt (p, 4,
		 "25\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 5,
		 "]],UNIT[\"degree\",0.0174532925199433,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 6,
		 ",\"9108\"]],AUTHORITY[\"EPSG\",\"4125\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 7,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 8,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 9,
		 "ER[\"central_meridian\",117],PARAMETER[\"scale_factor\",");
    add_srs_wkt (p, 10,
		 "0.9996],PARAMETER[\"false_easting\",500000],PARAMETER[\"");
    add_srs_wkt (p, 11,
		 "false_northing\",10000000],AUTHORITY[\"EPSG\",\"2550\"],");
    add_srs_wkt (p, 12, "AXIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2551, "epsg", 2551,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 34");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=34500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 34\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",102],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",34");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2551\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2552, "epsg", 2552,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 35");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=35500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 35\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",105],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",35");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2552\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2553, "epsg", 2553,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 36");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=36500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 36\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",108],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",36");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2553\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2554, "epsg", 2554,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 37");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=37500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 37\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",111],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",37");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2554\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2555, "epsg", 2555,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 38");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=38500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 38\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",114],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",38");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2555\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2556, "epsg", 2556,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 39");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=39500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 39\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",117],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",39");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2556\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2557, "epsg", 2557,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 40");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 40\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",120],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",40");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2557\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2558, "epsg", 2558,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 41");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=41500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 41\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",123],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",41");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2558\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2559, "epsg", 2559,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 42");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=42500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 42\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",126],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",42");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2559\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2560, "epsg", 2560,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 43");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=43500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 43\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",129],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",43");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2560\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2561, "epsg", 2561,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 44");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=44500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 44\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",132],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",44");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2561\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2562, "epsg", 2562,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 45");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=45500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 45\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",135],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",45");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2562\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2563, "epsg", 2563,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 46");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=138 +k=1 +x_0=46500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 46\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",138],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",46");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2563\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2564, "epsg", 2564,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 47");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=141 +k=1 +x_0=47500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 47\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",141],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",47");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2564\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2565, "epsg", 2565,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 48");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=144 +k=1 +x_0=48500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 48\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",144],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",48");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2565\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2566, "epsg", 2566,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 49");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=147 +k=1 +x_0=49500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 49\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",147],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",49");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2566\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2567, "epsg", 2567,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 50");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=150 +k=1 +x_0=50500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 50\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",150],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2567\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2568, "epsg", 2568,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 51");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=153 +k=1 +x_0=51500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 51\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",153],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",51");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2568\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2569, "epsg", 2569,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 52");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=156 +k=1 +x_0=52500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 52\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",156],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",52");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2569\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2570, "epsg", 2570,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 53");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=159 +k=1 +x_0=53500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 53\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",159],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",53");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2570\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2571, "epsg", 2571,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 54");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=162 +k=1 +x_0=54500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 54\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",162],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",54");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2571\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2572, "epsg", 2572,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 55");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=165 +k=1 +x_0=55500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 55\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",165],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",55");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2572\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2573, "epsg", 2573,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 56");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=168 +k=1 +x_0=56500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 56\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",168],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",56");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2573\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2574, "epsg", 2574,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 57");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=171 +k=1 +x_0=57500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 57\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",171],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",57");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2574\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2575, "epsg", 2575,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 58");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=174 +k=1 +x_0=58500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 58\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",174],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",58");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2575\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2576, "epsg", 2576,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 59");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=177 +k=1 +x_0=59500000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 59\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",177],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",59");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2576\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2577, "epsg", 2577,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=180 +k=1 +x_0=60000000 +y_0=");
    add_proj4text (p, 1, "0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 (d");
    add_srs_wkt (p, 1,
		 "eprecated)\",GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_194");
    add_srs_wkt (p, 2,
		 "2\",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"7024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0");
    add_srs_wkt (p, 4,
		 ".85,-0.12],AUTHORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenw");
    add_srs_wkt (p, 5,
		 "ich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.");
    add_srs_wkt (p, 6,
		 "01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORIT");
    add_srs_wkt (p, 7,
		 "Y[\"EPSG\",\"4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 8,
		 ",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAMETE");
    add_srs_wkt (p, 9,
		 "R[\"latitude_of_origin\",0],PARAMETER[\"central_meridian");
    add_srs_wkt (p, 10,
		 "\",180],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_");
    add_srs_wkt (p, 11,
		 "easting\",60000000],PARAMETER[\"false_northing\",0],AUTH");
    add_srs_wkt (p, 12,
		 "ORITY[\"EPSG\",\"2577\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EA");
    add_srs_wkt (p, 13, "ST]]");
    p = add_epsg_def (first, last, 2578, "epsg", 2578,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 61");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-177 +k=1 +x_0=61500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 61\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-177],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "1500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2578\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2579, "epsg", 2579,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 62");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-174 +k=1 +x_0=62500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 62\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-174],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "2500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2579\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2580, "epsg", 2580,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 63");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-171 +k=1 +x_0=63500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 63\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-171],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "3500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2580\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2581, "epsg", 2581,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger zone 64");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-168 +k=1 +x_0=64500000 +y_0");
    add_proj4text (p, 1, "=0 +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger zone 64\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-168],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "4500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2581\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2582, "epsg", 2582,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",21],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2582\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2583, "epsg", 2583,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",24],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2583\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2584, "epsg", 2584,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",27],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2584\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2585, "epsg", 2585,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",30],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2585\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2586, "epsg", 2586,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",33],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2586\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2587, "epsg", 2587,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",36],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2587\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2588, "epsg", 2588,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",39],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2588\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2589, "epsg", 2589,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",42],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2589\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2590, "epsg", 2590,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",45],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2590\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2591, "epsg", 2591,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=48 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",48],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2591\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2592, "epsg", 2592,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",51],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2592\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2593, "epsg", 2593,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=54 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",54],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2593\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2594, "epsg", 2594,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",57],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2594\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2595, "epsg", 2595,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=60 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",60],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2595\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2596, "epsg", 2596,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",63],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2596\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2597, "epsg", 2597,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=66 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",66],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2597\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2598, "epsg", 2598,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",69],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2598\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2599, "epsg", 2599,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=72 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",72],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2599\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2600, "epsg", 2600,
		      "Lietuvos Koordinoei Sistema 1994 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=0.9998 +x_0=500000 +y_");
    add_proj4text (p, 1,
		   "0=0 +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_def");
    add_proj4text (p, 2, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Lietuvos Koordinoei Sistema 1994 (deprecated)\"");
    add_srs_wkt (p, 1,
		 ",GEOGCS[\"LKS94\",DATUM[\"Lithuania_1994_ETRS89\",SPHERO");
    add_srs_wkt (p, 2,
		 "ID[\"GRS 1980\",6378137,298.257222101,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 3,
		 ",\"7019\"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 4,
		 "6126\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"890");
    add_srs_wkt (p, 5,
		 "1\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EP");
    add_srs_wkt (p, 6,
		 "SG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4669\"]],UNIT[\"met");
    add_srs_wkt (p, 7,
		 "re\",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transv");
    add_srs_wkt (p, 8,
		 "erse_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PAR");
    add_srs_wkt (p, 9,
		 "AMETER[\"central_meridian\",24],PARAMETER[\"scale_factor");
    add_srs_wkt (p, 10,
		 "\",0.9998],PARAMETER[\"false_easting\",500000],PARAMETER");
    add_srs_wkt (p, 11,
		 "[\"false_northing\",0],AUTHORITY[\"EPSG\",\"2600\"],AXIS");
    add_srs_wkt (p, 12, "[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2601, "epsg", 2601,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",75],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2601\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2602, "epsg", 2602,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",78],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2602\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2603, "epsg", 2603,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",81],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2603\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2604, "epsg", 2604,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",84],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2604\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2605, "epsg", 2605,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",87],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2605\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2606, "epsg", 2606,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",90],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2606\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2607, "epsg", 2607,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",93],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2607\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2608, "epsg", 2608,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",96],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2608\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2609, "epsg", 2609,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1, "ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AUTH");
    add_srs_wkt (p, 4,
		 "ORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHORI");
    add_srs_wkt (p, 5,
		 "TY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519943");
    add_srs_wkt (p, 6,
		 "28,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"42");
    add_srs_wkt (p, 7,
		 "84\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],PR");
    add_srs_wkt (p, 8,
		 "OJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_o");
    add_srs_wkt (p, 9,
		 "f_origin\",0],PARAMETER[\"central_meridian\",99],PARAMET");
    add_srs_wkt (p, 10,
		 "ER[\"scale_factor\",1],PARAMETER[\"false_easting\",50000");
    add_srs_wkt (p, 11,
		 "0],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2609\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2610, "epsg", 2610,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",102],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2610\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2611, "epsg", 2611,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",105],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2611\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2612, "epsg", 2612,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",108],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2612\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2613, "epsg", 2613,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",111],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2613\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2614, "epsg", 2614,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",114],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2614\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2615, "epsg", 2615,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",117],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2615\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2616, "epsg", 2616,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",120],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2616\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2617, "epsg", 2617,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",123],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2617\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2618, "epsg", 2618,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",126],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2618\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2619, "epsg", 2619,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",129],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2619\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2620, "epsg", 2620,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",132],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2620\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2621, "epsg", 2621,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",135],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2621\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2622, "epsg", 2622,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=138 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",138],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2622\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2623, "epsg", 2623,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=141 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",141],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2623\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2624, "epsg", 2624,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=144 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",144],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2624\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2625, "epsg", 2625,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=147 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",147],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2625\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2626, "epsg", 2626,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=150 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",150],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2626\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2627, "epsg", 2627,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=153 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",153],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2627\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2628, "epsg", 2628,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=156 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",156],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2628\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2629, "epsg", 2629,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=159 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",159],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2629\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2630, "epsg", 2630,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=162 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",162],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2630\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2631, "epsg", 2631,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=165 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",165],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2631\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2632, "epsg", 2632,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=168 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",168],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2632\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2633, "epsg", 2633,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=171 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",171],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2633\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2634, "epsg", 2634,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=174 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",174],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2634\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2635, "epsg", 2635,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=177 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",177],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2635\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2636, "epsg", 2636,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=180 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1, "+ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",180],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2636\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2637, "epsg", 2637,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-177 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-177],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",5");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2637\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2638, "epsg", 2638,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-174 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-174],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",5");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2638\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2639, "epsg", 2639,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-171 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-171],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",5");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2639\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2640, "epsg", 2640,
		      "Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-168 +k=1 +x_0=500000 +y_0=0");
    add_proj4text (p, 1, " +ellps=krass +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1942\",DATUM[\"Pulkovo_1942\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[23.9,-141.3,-80.9,-0,0.37,0.85,-0.12],AU");
    add_srs_wkt (p, 4,
		 "THORITY[\"EPSG\",\"6284\"]],PRIMEM[\"Greenwich\",0,AUTHO");
    add_srs_wkt (p, 5,
		 "RITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199");
    add_srs_wkt (p, 6,
		 "4328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4284\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-168],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",5");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2640\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2641, "epsg", 2641,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 7");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=7500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 7\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",21],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",7500");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2641\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2642, "epsg", 2642,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 8");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=1 +x_0=8500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 8\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",24],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",8500");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2642\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2643, "epsg", 2643,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 9");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=9500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 9\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",27],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",9500");
    add_srs_wkt (p, 11,
		 "000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",");
    add_srs_wkt (p, 12, "\"2643\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2644, "epsg", 2644,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 10");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=10500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 10\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",30],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",105");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2644\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2645, "epsg", 2645,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 11");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=11500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 11\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",33],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",115");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2645\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2646, "epsg", 2646,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 12");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=12500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 12\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",36],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",125");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2646\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2647, "epsg", 2647,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 13");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=13500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 13\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",39],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",135");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2647\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2648, "epsg", 2648,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 14");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=14500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 14\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",42],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",145");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2648\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2649, "epsg", 2649,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 15");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=15500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 15\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",45],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",155");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2649\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2650, "epsg", 2650,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 16");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=48 +k=1 +x_0=16500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 16\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",48],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",165");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2650\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2651, "epsg", 2651,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 17");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=17500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 17\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",51],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",175");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2651\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2652, "epsg", 2652,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 18");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=54 +k=1 +x_0=18500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 18\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",54],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",185");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2652\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2653, "epsg", 2653,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 19");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=19500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 19\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",57],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",195");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2653\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2654, "epsg", 2654,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 20");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=60 +k=1 +x_0=20500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 20\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",60],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",205");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2654\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2655, "epsg", 2655,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 21");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=21500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 21\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",63],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",215");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2655\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2656, "epsg", 2656,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 22");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=66 +k=1 +x_0=22500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 22\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",66],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",225");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2656\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2657, "epsg", 2657,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 23");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=23500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 23\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",69],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",235");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2657\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2658, "epsg", 2658,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 24");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=72 +k=1 +x_0=24500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 24\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",72],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",245");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2658\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2659, "epsg", 2659,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 25");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=25500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 25\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",75],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",255");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2659\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2660, "epsg", 2660,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 26");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=26500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 26\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",78],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",265");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2660\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2661, "epsg", 2661,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 27");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=27500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 27\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",81],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",275");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2661\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2662, "epsg", 2662,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 28");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=28500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 28\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",84],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",285");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2662\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2663, "epsg", 2663,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 29");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=29500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 29\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",87],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",295");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2663\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2664, "epsg", 2664,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 30");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=30500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 30\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",90],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",305");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2664\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2665, "epsg", 2665,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 31");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=31500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 31\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",93],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",315");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2665\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2666, "epsg", 2666,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 32");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=32500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 32\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",96],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",325");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2666\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2667, "epsg", 2667,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 33");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=33500000 +y_0=0");
    add_proj4text (p, 1,
		   " +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-");
    add_proj4text (p, 2, "0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 33\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",99],PARAM");
    add_srs_wkt (p, 10,
		 "ETER[\"scale_factor\",1],PARAMETER[\"false_easting\",335");
    add_srs_wkt (p, 11,
		 "00000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2667\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2668, "epsg", 2668,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 34");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=34500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 34\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",102],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",34");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2668\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2669, "epsg", 2669,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 35");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=35500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 35\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",105],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",35");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2669\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2670, "epsg", 2670,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 36");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=36500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 36\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",108],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",36");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2670\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
#endif /* full EPSG initialization enabled/disabled */
    if (first == last)
	last = first;		/* suppressing stupid compiler warnings */
}

static void
initialize_epsg_05 (struct epsg_defs **first, struct epsg_defs **last)
{
/* initializing the EPSG defs list */
#ifndef OMIT_EPSG		/* full EPSG initialization enabled */
    struct epsg_defs *p;
    p = add_epsg_def (first, last, 2671, "epsg", 2671,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 37");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=37500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 37\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",111],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",37");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2671\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2672, "epsg", 2672,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 38");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=38500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 38\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",114],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",38");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2672\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2673, "epsg", 2673,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 39");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=39500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 39\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",117],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",39");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2673\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2674, "epsg", 2674,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 40");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 40\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",120],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",40");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2674\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2675, "epsg", 2675,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 41");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=41500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 41\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",123],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",41");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2675\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2676, "epsg", 2676,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 42");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=42500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 42\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",126],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",42");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2676\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2677, "epsg", 2677,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 43");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=43500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 43\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",129],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",43");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2677\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2678, "epsg", 2678,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 44");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=44500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 44\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",132],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",44");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2678\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2679, "epsg", 2679,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 45");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=135 +k=1 +x_0=45500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 45\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",135],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",45");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2679\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2680, "epsg", 2680,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 46");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=138 +k=1 +x_0=46500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 46\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",138],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",46");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2680\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2681, "epsg", 2681,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 47");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=141 +k=1 +x_0=47500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 47\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",141],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",47");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2681\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2682, "epsg", 2682,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 48");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=144 +k=1 +x_0=48500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 48\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",144],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",48");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2682\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2683, "epsg", 2683,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 49");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=147 +k=1 +x_0=49500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 49\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",147],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",49");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2683\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2684, "epsg", 2684,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 50");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=150 +k=1 +x_0=50500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 50\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",150],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2684\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2685, "epsg", 2685,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 51");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=153 +k=1 +x_0=51500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 51\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",153],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",51");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2685\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2686, "epsg", 2686,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 52");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=156 +k=1 +x_0=52500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 52\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",156],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",52");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2686\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2687, "epsg", 2687,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 53");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=159 +k=1 +x_0=53500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 53\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",159],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",53");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2687\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2688, "epsg", 2688,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 54");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=162 +k=1 +x_0=54500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 54\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",162],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",54");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2688\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2689, "epsg", 2689,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 55");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=165 +k=1 +x_0=55500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 55\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",165],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",55");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2689\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2690, "epsg", 2690,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 56");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=168 +k=1 +x_0=56500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 56\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",168],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",56");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2690\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2691, "epsg", 2691,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 57");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=171 +k=1 +x_0=57500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 57\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",171],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",57");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2691\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2692, "epsg", 2692,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 58");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=174 +k=1 +x_0=58500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 58\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",174],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",58");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2692\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2693, "epsg", 2693,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 59");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=177 +k=1 +x_0=59500000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 59\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",177],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",59");
    add_srs_wkt (p, 11,
		 "500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG");
    add_srs_wkt (p, 12, "\",\"2693\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2694, "epsg", 2694,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 (deprecated)");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=180 +k=1 +x_0=60000000 +y_0=");
    add_proj4text (p, 1,
		   "0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,");
    add_proj4text (p, 2, "-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 (d");
    add_srs_wkt (p, 1,
		 "eprecated)\",GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_199");
    add_srs_wkt (p, 2,
		 "5\",SPHEROID[\"Krassowsky 1940\",6378245,298.3,AUTHORITY");
    add_srs_wkt (p, 3,
		 "[\"EPSG\",\"7024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,");
    add_srs_wkt (p, 4,
		 "0.16,-0.12],AUTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Green");
    add_srs_wkt (p, 5,
		 "wich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0");
    add_srs_wkt (p, 6,
		 ".01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORI");
    add_srs_wkt (p, 7,
		 "TY[\"EPSG\",\"4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG");
    add_srs_wkt (p, 8,
		 "\",\"9001\"]],PROJECTION[\"Transverse_Mercator\"],PARAME");
    add_srs_wkt (p, 9,
		 "TER[\"latitude_of_origin\",0],PARAMETER[\"central_meridi");
    add_srs_wkt (p, 10,
		 "an\",180],PARAMETER[\"scale_factor\",1],PARAMETER[\"fals");
    add_srs_wkt (p, 11,
		 "e_easting\",60000000],PARAMETER[\"false_northing\",0],AU");
    add_srs_wkt (p, 12,
		 "THORITY[\"EPSG\",\"2694\"],AXIS[\"X\",NORTH],AXIS[\"Y\",");
    add_srs_wkt (p, 13, "EAST]]");
    p = add_epsg_def (first, last, 2695, "epsg", 2695,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 61");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-177 +k=1 +x_0=61500000 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16");
    add_proj4text (p, 2, ",-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 61\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-177],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "1500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2695\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2696, "epsg", 2696,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 62");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-174 +k=1 +x_0=62500000 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16");
    add_proj4text (p, 2, ",-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 62\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-174],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "2500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2696\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2697, "epsg", 2697,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 63");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-171 +k=1 +x_0=63500000 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16");
    add_proj4text (p, 2, ",-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 63\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-171],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "3500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2697\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2698, "epsg", 2698,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger zone 64");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=-168 +k=1 +x_0=64500000 +y_0");
    add_proj4text (p, 1,
		   "=0 +ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16");
    add_proj4text (p, 2, ",-0.12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger zone 64\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",-168],PAR");
    add_srs_wkt (p, 10,
		 "AMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",6");
    add_srs_wkt (p, 11,
		 "4500000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPS");
    add_srs_wkt (p, 12, "G\",\"2698\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2699, "epsg", 2699,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=21 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",21],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2699\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2700, "epsg", 2700,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=24 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",24],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2700\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2701, "epsg", 2701,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=27 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",27],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2701\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2702, "epsg", 2702,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=30 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",30],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2702\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2703, "epsg", 2703,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=33 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",33],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2703\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2704, "epsg", 2704,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=36 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",36],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2704\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2705, "epsg", 2705,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=39 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",39],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2705\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2706, "epsg", 2706,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=42 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",42],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2706\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2707, "epsg", 2707,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=45 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",45],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2707\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2708, "epsg", 2708,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=48 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",48],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2708\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2709, "epsg", 2709,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=51 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",51],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2709\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2710, "epsg", 2710,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=54 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",54],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2710\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2711, "epsg", 2711,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=57 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",57],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2711\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2712, "epsg", 2712,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=60 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",60],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2712\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2713, "epsg", 2713,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=63 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",63],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2713\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2714, "epsg", 2714,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=66 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",66],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2714\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2715, "epsg", 2715,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=69 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",69],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2715\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2716, "epsg", 2716,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=72 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",72],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2716\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2717, "epsg", 2717,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",75],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2717\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2718, "epsg", 2718,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=78 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",78],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2718\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2719, "epsg", 2719,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=81 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",81],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2719\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2720, "epsg", 2720,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=84 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",84],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2720\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2721, "epsg", 2721,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=87 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",87],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2721\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2722, "epsg", 2722,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=90 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",90],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2722\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2723, "epsg", 2723,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=93 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",93],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2723\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2724, "epsg", 2724,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=96 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",96],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2724\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2725, "epsg", 2725,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=99 +k=1 +x_0=500000 +y_0=0 +");
    add_proj4text (p, 1,
		   "ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0.");
    add_proj4text (p, 2, "12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E\",G");
    add_srs_wkt (p, 1,
		 "EOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[\"");
    add_srs_wkt (p, 2,
		 "Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"702");
    add_srs_wkt (p, 3,
		 "4\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],AUT");
    add_srs_wkt (p, 4,
		 "HORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTHOR");
    add_srs_wkt (p, 5,
		 "ITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994");
    add_srs_wkt (p, 6,
		 "328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4");
    add_srs_wkt (p, 7,
		 "200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],P");
    add_srs_wkt (p, 8,
		 "ROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude_");
    add_srs_wkt (p, 9,
		 "of_origin\",0],PARAMETER[\"central_meridian\",99],PARAME");
    add_srs_wkt (p, 10,
		 "TER[\"scale_factor\",1],PARAMETER[\"false_easting\",5000");
    add_srs_wkt (p, 11,
		 "00],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 12, "2725\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2726, "epsg", 2726,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=102 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",102],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2726\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2727, "epsg", 2727,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=105 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",105],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2727\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2728, "epsg", 2728,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=108 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",108],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2728\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2729, "epsg", 2729,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",111],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2729\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2730, "epsg", 2730,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",114],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2730\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2731, "epsg", 2731,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",117],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2731\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2732, "epsg", 2732,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",120],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2732\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2733, "epsg", 2733,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",123],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2733\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2734, "epsg", 2734,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=126 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",126],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2734\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2735, "epsg", 2735,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=129 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",129],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2735\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2736, "epsg", 2736, "Tete / UTM zone 36S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=36 +south +ellps=clrk66 +units=m +no_def");
    add_proj4text (p, 1, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Tete / UTM zone 36S\",GEOGCS[\"Tete\",DATUM[\"T");
    add_srs_wkt (p, 1,
		 "ete\",SPHEROID[\"Clarke 1866\",6378206.4,294.97869821390");
    add_srs_wkt (p, 2,
		 "06,AUTHORITY[\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"61");
    add_srs_wkt (p, 3,
		 "27\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 4,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4127\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",33],PARAMETER[\"scale_factor\",0");
    add_srs_wkt (p, 9,
		 ".9996],PARAMETER[\"false_easting\",500000],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_northing\",10000000],AUTHORITY[\"EPSG\",\"2736\"],A");
    add_srs_wkt (p, 11, "XIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2737, "epsg", 2737, "Tete / UTM zone 37S");
    add_proj4text (p, 0,
		   "+proj=utm +zone=37 +south +ellps=clrk66 +units=m +no_def");
    add_proj4text (p, 1, "s");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Tete / UTM zone 37S\",GEOGCS[\"Tete\",DATUM[\"T");
    add_srs_wkt (p, 1,
		 "ete\",SPHEROID[\"Clarke 1866\",6378206.4,294.97869821390");
    add_srs_wkt (p, 2,
		 "06,AUTHORITY[\"EPSG\",\"7008\"]],AUTHORITY[\"EPSG\",\"61");
    add_srs_wkt (p, 3,
		 "27\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"");
    add_srs_wkt (p, 4,
		 "]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 5,
		 ",\"9122\"]],AUTHORITY[\"EPSG\",\"4127\"]],UNIT[\"metre\"");
    add_srs_wkt (p, 6,
		 ",1,AUTHORITY[\"EPSG\",\"9001\"]],PROJECTION[\"Transverse");
    add_srs_wkt (p, 7,
		 "_Mercator\"],PARAMETER[\"latitude_of_origin\",0],PARAMET");
    add_srs_wkt (p, 8,
		 "ER[\"central_meridian\",39],PARAMETER[\"scale_factor\",0");
    add_srs_wkt (p, 9,
		 ".9996],PARAMETER[\"false_easting\",500000],PARAMETER[\"f");
    add_srs_wkt (p, 10,
		 "alse_northing\",10000000],AUTHORITY[\"EPSG\",\"2737\"],A");
    add_srs_wkt (p, 11, "XIS[\"Easting\",EAST],AXIS[\"Northing\",NORTH]]");
    p = add_epsg_def (first, last, 2738, "epsg", 2738,
		      "Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E");
    add_proj4text (p, 0,
		   "+proj=tmerc +lat_0=0 +lon_0=132 +k=1 +x_0=500000 +y_0=0 ");
    add_proj4text (p, 1,
		   "+ellps=krass +towgs84=24.82,-131.21,-82.66,-0,-0,0.16,-0");
    add_proj4text (p, 2, ".12 +units=m +no_defs");
    add_srs_wkt (p, 0,
		 "PROJCS[\"Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E\",");
    add_srs_wkt (p, 1,
		 "GEOGCS[\"Pulkovo 1995\",DATUM[\"Pulkovo_1995\",SPHEROID[");
    add_srs_wkt (p, 2,
		 "\"Krassowsky 1940\",6378245,298.3,AUTHORITY[\"EPSG\",\"7");
    add_srs_wkt (p, 3,
		 "024\"]],TOWGS84[24.82,-131.21,-82.66,-0,-0,0.16,-0.12],A");
    add_srs_wkt (p, 4,
		 "UTHORITY[\"EPSG\",\"6200\"]],PRIMEM[\"Greenwich\",0,AUTH");
    add_srs_wkt (p, 5,
		 "ORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.017453292519");
    add_srs_wkt (p, 6,
		 "94328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"");
    add_srs_wkt (p, 7,
		 "4200\"]],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],");
    add_srs_wkt (p, 8,
		 "PROJECTION[\"Transverse_Mercator\"],PARAMETER[\"latitude");
    add_srs_wkt (p, 9,
		 "_of_origin\",0],PARAMETER[\"central_meridian\",132],PARA");
    add_srs_wkt (p, 10,
		 "METER[\"scale_factor\",1],PARAMETER[\"false_easting\",50");
    add_srs_wkt (p, 11,
		 "0000],PARAMETER[\"false_northing\",0],AUTHORITY[\"EPSG\"");
    add_srs_wkt (p, 12, ",\"2738\"],AXIS[\"X\",NORTH],AXIS[\"Y\",EAST]]");
    p = add_epsg_def (first, last, 2739, "epsg"