##############################################################################  
#   
# This software is released under the Zope Public License (ZPL) Version 1.0
#
# Copyright (c) Digital Creations.  All rights reserved.  
# Portions Copyright (c) 1999 by Butch Landingin.
# Portions Copyright (c) 2000-2001 by Chris Withers.
#   
##############################################################################  
     
__doc__ = """Squishdot - a web-based news publishing and discussion system"""
__version__='$Revision: 1.33 $'[11:-2]     
     
import sys
from time import time
from string import strip,join,atoi,replace
from urllib import unquote     

from zLOG import LOG, ERROR
from AccessControl import ClassSecurityInfo
from Globals import PersistentMapping, HTMLFile, MessageDialog, InitializeClass     
from Acquisition import aq_base
from BTrees.IOBTree import IOBTree
from intSet import intSet     
from DocumentTemplate.DT_Util import html_quote
import StructuredText

from OFS.Document import Document
from OFS.ObjectManager import REPLACEABLE
from Products.ZCatalog import ZCatalog

from stripogram import html2safehtml, html2text

from Utility import CRLF,tagRegex,file_path,Stack,sameday,doAddPosting,getitem,\
                    addDTML,addImage,addTable,addArt,addFile,addText,addDTML,addPythonScript
from SquishPermissions import ManageSquishdot,ModeratePostings,AddPostings,View
from Squishfile import Squishfile
from Article import Article

class SquishSite(ZCatalog.ZCatalog):     
    """A Squishdot Site is a self contained web-based news publishing and discussion system"""     
    meta_type  ='Squishdot Site'     
    description='Squishdot Site'     
     
    security = ClassSecurityInfo()
    security.setPermissionDefault(ManageSquishdot, ('Manager',))
    security.setPermissionDefault(ModeratePostings,('Manager',))
    security.setPermissionDefault(AddPostings,     ('Anonymous','Manager',))
    security.setPermissionDefault(View,            ('Anonymous','Manager',))

    icon       ='misc_/Squishdot/squishdot_img'     
    root       =1     
    squishlogo ='misc_/Squishdot/squishlogo'    
    
    _properties=({'id':'title', 'type':'string','mode':'w'},)     
     
    # Default encoding for old postings
    encoding = 'HTML'

    # Defaults
    mail_articles=0
    mail_replies=0
    level=-1
    thread=[]

    manage_options=({'label':'Contents', 'icon':icon, 'action':'manage_main', 'target':'manage_main'},     
                    {'label':'View', 'icon':'', 'action':'index_html', 'target':'manage_main'},     
                    {'label':'Postings', 'icon':'', 'action':'manage_postings', 'target':'manage_main'},     
                    {'label':'Moderation', 'icon':'', 'action':'manage_moderation', 'target':'manage_main'},
                    {'label':'Subjects', 'icon':'', 'action':'manage_subjects', 'target':'manage_main'},     
                    {'label':'Options', 'icon':'', 'action':'manage_editForm', 'target':'manage_main'},     
                    {'label':'Properties', 'icon':'', 'action':'manage_propertiesForm', 'target':'manage_main'},
                    {'label':'Security', 'icon':'', 'action':'manage_access', 'target':'manage_main'},
                    {'label':'Undo', 'icon':'', 'action':'manage_UndoForm', 'target':'manage_main'}
                    )     

    security.declareProtected(ModeratePostings, 'manage_postings')
    manage_postings   = HTMLFile('Squishdot_manage_postings', globals())

    security.declareProtected(ModeratePostings, 'manage_moderation')
    manage_moderation = HTMLFile('Squishdot_manage_moderation', globals())

    security.declareProtected(ModeratePostings, 'moderateArticle')
    moderateArticle   = HTMLFile('moderateArticle',globals())

    security.declareProtected(ManageSquishdot, 'manage_editForm')
    manage_editForm   = HTMLFile('Squishdot_editForm', globals())     

    security.declareProtected(ManageSquishdot, 'manage_subjects')
    manage_subjects   = HTMLFile('Squishdot_manage_subjects', globals())    

    security.declarePublic('About')
    About = HTMLFile('about',globals())
    
    security.declarePublic('Readme')
    Readme = Document(addText('README.txt','Readme'), __name__='Readme')    

    security.declarePublic('Credits')
    Credits = Document(addText('Credits.txt','Credits'), __name__='Credits')    

    security.declarePublic('License')
    License = Document(addText('Credits.txt','Credits'), __name__='License')
    
    security.declarePublic('Changes')
    Changes = Document(addText('Changes.txt','Changes'), __name__='Changes')
    
    security.declarePublic('Version')
    Version = Document(addText('version.txt','Version'), __name__='Version')
       
    security.declarePrivate('_buildIndexing')
    def _buildIndexing(self, id, title):
        # Initialise ZCatalog
        if not hasattr(self,'_catalog'):
            ZCatalog.ZCatalog.__init__(self, id, title)

        base = aq_base(self)
        if hasattr(base, 'addIndex'):
            # Zope 2.4
            addIndex = self.addIndex
        else:
            # Zope 2.3 and below
            addIndex = self._catalog.addIndex
        if hasattr(base, 'addColumn'):
            # Zope 2.4
            addColumn = self.addColumn
        else:
            # Zope 2.3 and below
            addColumn = self._catalog.addColumn

        # Set up the indexes
        # We get a 'title' TextIndex for free from ZCatalog.ZCatalog
        for (name,index_type) in [('author', 'FieldIndex'),
                                  ('textToSearch', 'TextIndex'),
                                  ('subject', 'FieldIndex'),
                                  ('reviewed', 'FieldIndex'),
                                  ('date', 'FieldIndex')]:
            try:
                addIndex(name,index_type)
            except:
                # do nothing if this index already exists
                if sys.exc_type!='Index Exists':
                    raise
                          
        # Add the meta data columns for search results
        for name in ['thread_path','author','date_posted','reply_cnt']:
            try:
                addColumn(name,'')
            except:
                # do nothing if this column already exists
                if sys.exc_type!='Column Exists':
                    raise
        # delete unnecessary and large meta data column
        self._catalog.delColumn('summary')
      
    security.declareProtected(ManageSquishdot, 'updateIndexing')
    def updateIndexing(self,REQUEST=None):
        """ A a ZCatalog and appropriate indexes to this SquishSite if they aren't there already """
        self._buildIndexing(self.id, self.title)
        self.recatalogPostings(REQUEST)

    security.declareProtected(ModeratePostings, 'recatalogPostings')
    def recatalogPostings(self,REQUEST=None):
        """ Clear the Catalog and then Index all the postings. """
        self._catalog.clear()
        for id in self.data.keys():
            obj = self.data[id].__of__(self)
            obj.index()
        if REQUEST is None:
            return
        else:
            return self.manage_main(self, REQUEST)

    # sort out the catalog if we're moved, copied or renamed
    security.declareProtected(ManageSquishdot, 'manage_afterAdd')
    def manage_afterAdd(self, item, container):
        try:
            self.recatalogPostings()            
        except AttributeError:
            self.updateIndexing()

    security.declarePrivate('__init__')
    def __init__(self, id, title, mhost, exp, moderated, max_itemlist, default_doc, mail_articles, mail_replies,parent):
        self._buildIndexing(id,title)
        
        # create a wrapped version of self, so we can get acquisition to happen properly
        wrapped_self = self.__of__(parent)

        t=time()     
        self.created  = t     
        self.modified = t     
        self.mail_host= mhost     
        self.expire   = exp
        self.mail_articles = mail_articles
        self.mail_replies = mail_replies
     
        self.moderated= 0     
        self.mod_comment = 0     
        if moderated == 'both':     
            self.moderated = 1     
            self.mod_comment= 1     
        else:     
            if moderated == 'articles':     
                self.moderated = 1     
                self.mod_comment=0     
     
        if max_itemlist:     
            self.max_itemlist=max_itemlist     
        else:     
            self.max_itemlist=30     

        self.data     =IOBTree()  # id -> Message     
        self.ids      =intSet() # ids of children
        
        self.subjects = PersistentMapping() # The List of subjects/categories   
     
        addPythonScript(self,'validatePosting','validatePosting')     
        addDTML(self,'rdf','Netscape RDF/RSS channel page',     
                'Squishdot_rdf_rss')     
    
        # set rss related properties          
        newprop = list(self._properties)
        newprop.append({'id':'rss_description', 'type': 'string', 'mode': 'wd'})     
        newprop.append({'id':'rss_title', 'type': 'string', 'mode': 'wd'})     
        newprop.append({'id':'rss_image_url', 'type': 'string', 'mode': 'wd'})     
        newprop.append({'id':'admin_address', 'type': 'string', 'mode': 'wd'})     
        self._properties = tuple(newprop)     
        self._updateProperty('rss_image_url','')      
        self._updateProperty('admin_address','squishdot@yahoo.com')     
        self._updateProperty('rss_description','Squishdot: Cool Dope on Zope')      
        self._updateProperty('rss_title',title)      
    
        addDTML(self,'mail_html','Notification Email',     
                'Squishdot_mail_html')
        
        if default_doc == 'plain':     
            addDTML(self,'showMessage','Show Message',     
                    'showMessage')     
            addDTML(self,'showError','Show Error',     
                    'showError')     
            addDTML(self,'index_html','Squishdot Homepage',     
                    'Squishdot_index_html')     
            addDTML(self,'posting_html','Posting',     
                    'Squishdot_posting_html')     
            addDTML(self,'searchForm','Search Form',     
                    'Squishdot_searchForm')     
            addDTML(self,'showSearchResults','Search Results',     
                    'Squishdot_searchResults')     
            addDTML(self,'addPostingForm','Add Posting Form',     
                    'addPostingForm')     
            addDTML(self,'previewPosting','Preview Posting',     
                    'Squishdot_previewForm')     
     
        if default_doc == 'demo1' or default_doc == 'demo2':     
            addDTML(self,'showMessage','Show Message',     
                    'demo/showMessage')     
            addDTML(self,'showError','Show Error',     
                    'demo/showError')     

            
            # Check whether TinyTable or TinyTablePlus is available
            # Use TinyTablePlus by preference
            all_meta_types=self.all_meta_types
            if callable(all_meta_types):
                all_meta_types = all_meta_types()

            table_add_method=''
            for meta_type in all_meta_types:
                if meta_type['name']=='TinyTable':
                    table_add_method='manage_addTinyTable'
                if meta_type['name']=='TinyTablePlus':
                    table_add_method='manage_addTinyTablePlus'
                    break

            # Add tables if a TinyTable product is installed
            if table_add_method:
                addTable(wrapped_self,table_add_method,'bottom_items','demo/bottom_items')
                addTable(wrapped_self,table_add_method,'leftbox_items','demo/leftbox_items')     
                addTable(wrapped_self,table_add_method,'rightbox_items','demo/rightbox_items')     
     
            addDTML(self,'index_html','Squishdot Homepage',     
                    'demo/Squishdot_index_html')     
            addDTML(self,'posting_html','Posting',     
                    'demo/Squishdot_posting_html')     
            addDTML(self,'searchForm','Search Form',     
                    'demo/Squishdot_searchForm')     
            addDTML(self,'showSearchResults','Search Results',     
                    'demo/Squishdot_searchResults')     
            addDTML(self,'addPostingForm','Add Posting Form',     
                    'demo/addPostingForm')     
      
            addDTML(self,'site_footer','Site Footer',     
                    'demo/site_footer')     
            addDTML(self,'site_header','Site Header',     
                    'demo/site_header')     
            addDTML(self,'previewPosting','Preview Posting',     
                    'demo/Squishdot_previewForm')     
     
            self.manage_addFolder('Images','Images')     
            self.manage_addFolder('TopicImages','Topic Images')     
            self.manage_addFolder('misc_methods','Misc. methods')     
            self.manage_addFolder('rightbox_methods','Right box methods')     
     
            curr_folder =  getattr(self,'misc_methods')     
            addDTML(curr_folder,'advertising','advertising','demo/misc_methods/advertising')     
            addDTML(curr_folder,'copyright_notice','copyright','demo/misc_methods/copyright')     
            addDTML(curr_folder,'quotation','quotation','demo/misc_methods/quotation')     
     
            curr_folder =  getattr(self,'rightbox_methods')     
            feat_method = addDTML(curr_folder,'features','Features','demo/rightbox_methods/features')     
            prev_method = addDTML(curr_folder,'prev_articles','Previous Articles','demo/rightbox_methods/prevarticles')     
            quik_method = addDTML(curr_folder,'quick_links','Quick Links','demo/rightbox_methods/quicklinks')     
            rfc_method = addDTML(curr_folder,'request_comments','Requests for Comments','demo/rightbox_methods/reqcomments')     
            rev_method = addDTML(curr_folder,'reviews','Reviews','demo/rightbox_methods/reviews')     
     
            curr_folder =  getattr(self,'Images')     
            addImage(curr_folder,'botshadow_img','demo/Images/botshadow.gif')     
            addImage(curr_folder,'roundedge_img','demo/Images/roundedge.gif')     
            addImage(curr_folder,'rtbotshadow_img','demo/Images/rtbotshadow.gif')     
            addImage(curr_folder,'rtshadow_img','demo/Images/rtshadow.gif')     
            addImage(curr_folder,'sitetitle_img','demo/Images/sitetitle.gif')     
            addImage(curr_folder,'bluepix_img','demo/Images/bluepix.gif')     
            addImage(curr_folder,'greenpix_img','demo/Images/greenpix.gif')     
            addImage(curr_folder,'advert_img','demo/Images/advert.gif')     
     
            curr_folder =  getattr(self,'TopicImages')     
            # note: build images      
            addImage(curr_folder,'dc_img',  'demo/TopicImages/dc.gif')     
            addImage(curr_folder,'zope_img','demo/TopicImages/zope.gif')     
            addImage(curr_folder,'dtml_img','demo/TopicImages/dtml.gif')     
            addImage(curr_folder,'help_img','demo/TopicImages/help.gif')     
            addImage(curr_folder,'squishdot_img','demo/TopicImages/squish.gif')     
            addImage(curr_folder,'zopedev_img','demo/TopicImages/zopedev.gif')     
            addImage(curr_folder,'zdp_img','demo/TopicImages/zdp.gif')     
            addImage(curr_folder,'zserver_img','demo/TopicImages/zserver.gif')     
            addImage(curr_folder,'dope_img','demo/TopicImages/dope.gif')     
     
            newprop = list(self._properties)
            newprop.append({'id':'color1', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'color2', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'color3', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'linedot_image', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'color4', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'color5', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'drop_shadow', 'type': 'int', 'mode': 'wd'})     
            newprop.append({'id':'round_edge', 'type': 'int', 'mode': 'wd'})     
            newprop.append({'id':'title_image', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'site_name', 'type': 'string', 'mode': 'wd'})     
            newprop.append({'id':'comment_spillover', 'type': 'int', 'mode': 'wd'})     
            newprop.append({'id':'prevday_cnt', 'type': 'int', 'mode': 'wd'})     
            newprop.append({'id':'admin_name', 'type': 'string', 'mode': 'wd'})     
            self._properties = tuple(newprop)     
            self._updateProperty('color1','#000000')      
            self._updateProperty('color2','#ffffff')      
            if default_doc == 'demo1':     
                self._updateProperty('color3','#006666')      
                self._updateProperty('linedot_image','Images/greenpix_img')      
            else:     
                self._updateProperty('color3','#0066cc')      
                self._updateProperty('linedot_image','Images/bluepix_img')      
            self._updateProperty('color4','#cccccc')      
            self._updateProperty('color5','#333333')      
            self._updateProperty('drop_shadow',1)      
            self._updateProperty('round_edge',1)      
            self._updateProperty('title_image','Images/sitetitle_img')      
            self._updateProperty('site_name','Squishdot')      
            self._updateProperty('comment_spillover',10)      
            self._updateProperty('prevday_cnt',7)      
            self._updateProperty('admin_address','squishdot@yahoo.com')     
            self._updateProperty('admin_name','the ZopeMeister')     
            self._updateProperty('rss_image_url','Images/sitetitle_img')      
     
            self.add_subject('Zope','TopicImages/zope_img')     
            self.add_subject('Zope DTML','TopicImages/dtml_img')     
            self.add_subject('Help','TopicImages/help_img')     
            self.add_subject('Digital Creations','TopicImages/dc_img')     
            self.add_subject('Squishdot','TopicImages/squishdot_img')     
            self.add_subject('Zope Development','TopicImages/zopedev_img')     
            self.add_subject('Zope Doc Project','TopicImages/zdp_img')     
            self.add_subject('ZServer','TopicImages/zserver_img')     
            self.add_subject('Zope Dopes','TopicImages/dope_img')     
     
     
            # Add sample content
            ids={}
            for i in range(10,0,-1):
                ids[i] = addArt(wrapped_self,'demo/messages/message%s.txt' % i)
                
            art8 = self.data[ids[8]]
            addFile(art8,'demo/messages/images.zip')

            # Complete templates
            raw = feat_method.raw % (str(ids[1]),str(ids[2]),str(ids[5]))     
            feat_method.raw = raw     
            raw = rfc_method.raw % str(ids[3])     
            rfc_method.raw = raw
            
       
    security.declarePublic('__len__')
    def __len__(self): return 1     
     
    security.declareProtected(View, '__getitem__')
    __getitem__ = getitem
     
    security.declareProtected(View, 'dupString')
    def dupString(self, dstr, count):     
        # ''' returns a duplicate of dstr by count times'''     
        return dstr * count     
    
    security.declareProtected(View, 'striptags')
    def striptags(self,s):    
        # ''' removes char sequences that invalidate RSS syntax'''     
        return tagRegex.sub("",s)
    
    security.declareProtected(View, 'html2text')
    def html2text(self,s):
        return html2text(s)
     
    security.declareProtected(View, 'html2safehtml')
    def html2safehtml(self,text):        
        return strip(html2safehtml(text,self.getProperty('valid_tags',('b', 'a', 'i', 'br', 'p', 'h3', 'ul', 'li', 'font', 'br'))))
    
    security.declareProtected(View, 'has_subjects')
    def has_subjects(self):     
        # ''' returns true if site has subject topics defined'''     
        return len(self.subjects)     
     
    security.declareProtected(ManageSquishdot, 'subject_count')
    def subject_count(self):     
        # ''' gives count of subject topics defined'''     
        return self.has_subjects()     
     
    security.declareProtected(ManageSquishdot, 'add_subject')
    def add_subject(self,subject='', imgurl='', REQUEST=None,RESPONSE=None):     
        """ add a subject topic """     
        if not subject:     
            return MessageDialog(title='Data Missing',     
                                 message='You must enter a subject!',     
                                 action=REQUEST['URL1']+'/manage_subjects',     
                                 target='manage_main'     
                                )     
        self.subjects[subject]= imgurl     
        self._p_changed = 1    
     
        if REQUEST:     
            return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])
     
    security.declareProtected(ManageSquishdot, 'delete_subjects')
    def delete_subjects(self,subjs=[],REQUEST=None):     
        """ delete a subject topic"""     
        for subj in subjs:     
            del self.subjects[subj]     
        self._p_changed = 1    
        if REQUEST:     
            return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])
     
    security.declareProtected(View, 'subjects_list')
    def subjects_list(self):     
        # ''' lists all subjects '''    
        subjects = self.subjects.keys()    
        subjects.sort()    
        return subjects    
     
    security.declareProtected(View, 'subject_image')
    def subject_image(self,subj):     
        # ''' returns an image url associated with the subject (can be an empty string)'''     
        if self.subjects.has_key(subj):     
            return self.subjects[subj]     
        else:     
            return ''     

    security.declarePrivate('setItem')
    def setItem(self,id,obj):     
        # Make sure the object we store is not wrapped with     
        # an acquisition wrapper, since we will be wrapping     
        # it again manually in __getitem__ when it is needed.
        bobj = getattr(obj,'aq_base',obj)
        self.ids.insert(id)     
        self.data[id]=bobj

        # index the posting
        obj.index()
     
    security.declarePrivate('delItem')
    def delItem(self,id):
        # get the item to delete
        data=self.data
        try:
            item=data[id].__of__(self)
        except KeyError:
            # The posting is already gone...
            return

        # make a list of ids to be deleted
        # (the id specified, and all it's child postings)
        ids=intSet()     
        ids.insert(id)     
        ids=item.sub_ids(ids)

        review_count = item.revsub
        reply_cnt = item.reply_cnt
        
        if item.reviewed:
            reply_cnt = reply_cnt + 1
        else:
            review_count = review_count + 1

        # If it's an article, delete it from the SquishSite's list of postings...
        if self.ids.has_key(id):
            self.ids.remove(id)
            
        # ...otherwise, things are a bit more complicated
        else:
            parent = data[item.thread[-1]]

            # remove it from it's parents list of ids
            parent.ids.remove(id)
            
            for i in item.thread:
                posting = data[i]

                # adjust the count of unreviewed replies for each posting in the thread
                posting.revsub = posting.revsub - review_count

                # adjust the count of reviewed replies for each posting in the thread
                posting.reply_cnt = posting.reply_cnt - reply_cnt
        
            # re-index the parent, so counts get reset
            data[item.thread[-1]].__of__(self).index()
            
        item=None        
        
        # Un-catalog the objects to be deleted.
        for id in ids:     
            obj = data[id]
            self.uncatalog_object(join(self.getPhysicalPath(),'/')+obj.thread_path()+'/'+`id`)

        for id in ids:     
            del data[id]     
     
    security.declarePrivate('createId')
    def createId(self):     
        id=int(time())     
        while self.data.has_key(id):     
            id=id+1     
        return id     
     
    security.declarePrivate('data_map')
    def data_map(self,ids):
        result=[]
        for id in ids:
            result.append(self.data[id].__of__(self))
        return result
    
    security.declareProtected(ModeratePostings, 'postingValues')
    def postingValues(self):     
        # """ return list of articles """
        articles = self.data_map(self.ids)
        articles.reverse()
        return articles
     
    security.declarePrivate('rev_id_list')
    def rev_id_list(self):     
        # """ returns reversed id list of reviewed articles  """     
        rlist = map(None,self.ids)     
        rlist = filter(lambda x,p=self : p.data[x].reviewed, rlist)     
        rlist.reverse()     
        return rlist     
     
    security.declareProtected(View, 'item_list')
    def item_list(self):     
        # """ returns latest articles  """                          
        currtime =   int(time())     
        return self.data_map(self.id_list(currtime))     
     
    security.declarePrivate('id_list')
    def id_list(self, currtime):     
        # ''' returns id list of latest articles at currtime '''     
        rlist = self.rev_id_list()     
        max = self.max_itemlist     
        min = max / 3     
        rlen = len(rlist)     
        if rlen <= min:     
            pass # take all elements     
        else:     
            today_list = self.date_id_list(currtime)  # take only items from today     
            tlen = len(today_list)     
            if tlen <= min:     
                rlist = rlist[:min] # take minimum elements even though some items came from yesterday     
            else:     
                if tlen >= max:     
                    rlist = today_list[:max]  # take maximum elements from today                        
                else:     
                    rlist = today_list # take entire list of items from today     
        return rlist     
             
    security.declarePrivate('other_id_list')
    def other_id_list(self, currtime):     
        # '''returns id list of articles exceeding max but still within current day'''      
        today_list = self.date_id_list(currtime) # take only items from today     
        tlen = len(today_list)     
        curr_list = self.id_list(currtime)  # take items displayed on main page     
        clen = len(curr_list)     
        if tlen > clen:     
            return today_list[clen:]  # return items from today not displayed on main page     
        else:     
            return []     
     
    security.declareProtected(View, 'other_list')
    def other_list(self):     
        # """returns articles exceeding max but still within current day"""     
        currtime =   int(time())     
        return self.data_map(self.other_id_list(currtime))     
     
    security.declarePrivate('date_id_list')
    def date_id_list(self, currtime):     
        # """returns list from sameday """     
        rlist = self.rev_id_list()     
        return filter(lambda x,p=currtime: sameday(x,p), rlist)     
             
    security.declareProtected(View, 'date_list')
    def date_list(self, day=0):     
        # """return list from day """     
        currtime = int(time())  - (86400 * day)     
        return self.data_map(self.date_id_list(currtime))     
     
    security.declarePrivate('site_id_list')
    def site_id_list(self, currtime):     
        # """ returns latest id list from currtime  """     
        ilist = self.id_list(currtime)     
        tdict = {}     
        tlist = []     
        ilen = len(ilist)     
        cnt = 0      
        while (cnt < ilen) and (len(tdict) < 5):     
            id = ilist[cnt]     
            item = self.data[id]     
            if tdict.has_key(item.subject):     
                pass     
            else:     
                tdict[item.subject] = id     
                tlist.append(id)     
            cnt = cnt + 1     
     
        return tlist     
     
    security.declareProtected(View, 'site_item_list')
    def site_item_list(self):     
        # """ returns latest articles from today """     
        currtime =   int(time())     
        return self.data_map(self.site_id_list(currtime))
            
     
    security.declareProtected(View, 'tpId')
    def tpId(self):     
        return self.id     
     
    security.declareProtected(View, 'tpURL')
    def tpURL(self):     
        return self.id     
     
    security.declareProtected(View, 'this')
    def this(self):     
        return self     
     
    security.declareProtected(View, 'site')
    def site(self):     
        return (self,)
     
    security.declareProtected(View, 'site_url')
    def site_url(self):    
        # """ url of the Squishdot main page """ 
        return self.absolute_url()
     
    security.declareProtected(View, 'has_items')
    def has_items(self):     
        return len(self.ids)     
     
    security.declareProtected(ModeratePostings, 'item_count')
    def item_count(self):     
        return len(self.data)     
     
    security.declareProtected('Add Squishdot Sites', 'mailhost_list')
    def mailhost_list(self):
        #  """ list of mail hosts """
         try:    return self.superValues(('Mail Host',))     
         except: return []     
     
    security.declareProtected(View, 'desc_items')
    def desc_items(self):     
        # """ articles in descending order """     
        mlist = []     
        mstack = Stack()     
        if self.has_items():     
            plist = []     
            for id in self.ids:     
                plist.append(id)     
            plist.reverse()     
            for id in plist:     
                mstack.push(id)     
            while not mstack.isEmpty():     
                item_id = mstack.pop()     
                item = self.data[item_id]     
                mlist.append(item)     
                if item.has_items():     
                    plist = []     
                    for id in item.ids:     
                        plist.append(id)     
                    plist.reverse()     
                    for id in plist:     
                        mstack.push(id)     
                        
        return map((lambda x, p=self: x.__of__(p)), mlist)     
           
    security.declarePrivate('expire_items')
    def expire_items(self):     
        # """ run to delete articles which are past their expiration (assuming that it has been set)"""     
        if self.expire:     
            d=self.data     
            t=int(time()-(self.expire * 86400.0))     
            ids=[]     
            for id in d.keys():     
                if d[id].modified < t:     
                    ids.append(id)     
            for id in ids:     
                try:    self.delItem(id)     
                except: pass     
        return ''     

    security.declareProtected(AddPostings, 'dummyPosting')
    def dummyPosting(self):
        """ returns a dummy posting for the previewPosting method """
        return Article(0,[],0,1).__of__(self)
    
    security.declareProtected(AddPostings, 'addPosting')
    def addPosting(self,file='',REQUEST=None,RESPONSE=None):     
        """ add an article """
        return doAddPosting(self,file,REQUEST,RESPONSE,
                            moderated='moderated',
                            message  ='Your article has been posted',
                            klass    =Article)
     
    security.declareProtected(View, 'search')
    def search(self,REQUEST):     
        """ fulfill a search request """
        # Massage REQUEST so old form works with new searching...        
        if REQUEST.has_key('body') and REQUEST['body']:
            REQUEST.set('textToSearch',REQUEST['body'])
        if REQUEST.has_key('op') and REQUEST['op']=='articles':
            REQUEST.set('meta_type','Article')
        REQUEST.set('reviewed',1)
    
        sr=self.__call__(REQUEST)     
        rc=len(sr)     
        return self.showSearchResults(self,REQUEST,search_results=sr,     
                                  result_count=rc)     
     
     
    security.declareProtected(ManageSquishdot, 'manage_edit')
    def manage_edit(self,exp,expire,moderated, max_itemlist,     
                    REQUEST=None, mail_articles=0, mail_replies=0,mailhost=''):     
        """ edit SquishDot options  """     
        if exp: expire=atoi(expire)     
        else:   expire=0     
        if mailhost:     
            try:    v=getattr(self, mailhost)     
            except:      
                return MessageDialog(title='Invalid Mail Host',     
                                     message='Cannot find the named mail host!',     
                                     action=REQUEST['URL']+'/manage_main'     
                                    )     
        self.mail_host=mailhost     
        self.expire   =expire
        self.mail_articles = mail_articles
        self.mail_replies = mail_replies
     
        if moderated == 'both':     
            self.moderated = 1     
            self.mod_comment= 1     
        else:     
            if moderated == 'articles':     
                self.moderated = 1     
                self.mod_comment=0     
            else:     
                if moderated == 'none':     
                    self.moderated = 0     
                    self.mod_comment = 0     
     
        if max_itemlist:     
            self.max_itemlist=max_itemlist     
        else:     
            self.max_itemlist=30
        return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])
     
    security.declareProtected(ModeratePostings, 'manage_delete')
    def manage_delete(self,ids=[],REQUEST=None):     
        """ delete selected articles from a Squishdot site """     
        ids=map(atoi, ids)     
        for id in ids:     
            self.delItem(id)
        if REQUEST is not None:
            return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])
        
    security.declareProtected(ModeratePostings, 'manage_review')
    def manage_review(self,ids=[],REQUEST=None):     
        """ approve selected articles from a Squishdot site """     
        ids=map(atoi, ids)
        for id in ids:
            self.set_reviewed(self.data[id].__of__(self))
        if REQUEST is not None:
            return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])

    security.declareProtected(ModeratePostings, 'moderation_process')
    def moderation_process(self, approve=None, delete=None, REQUEST=None):
        """ do the processing for things from the moderation tab """
        if approve:
            self.manage_review(REQUEST['approve'],REQUEST)
        if delete:
            self.manage_delete(REQUEST['delete'],REQUEST)
        if REQUEST is not None:
            return REQUEST.RESPONSE.redirect(REQUEST['HTTP_REFERER'])

    security.declareProtected(AddPostings, 'suggest_author')
    def suggest_author(self):    
        author = ''    
        try: author = self.REQUEST.cookies['_suggest_author']    
        except: pass    
        if author : author = unquote(author)    
        return author or None    
    
    security.declareProtected(AddPostings, 'suggest_email')
    def suggest_email (self):    
        email  = ''    
        try: email  = self.REQUEST.cookies['_suggest_email']    
        except: pass    
        if email  : email  = unquote(email)    
        return email  or None    
                    
    # Used for Bruce Perens' Batch Moderation Code
    # Indicate that posting has been reviewed
    security.declarePrivate('set_reviewed')
    def set_reviewed(self, item):
        item.reviewed=1
        # re-catalog the item now that it's reviewed status has changed.
        item.index()
        for t in item.thread:
            obj=self.data[t]
            if obj.revsub > 0: obj.revsub=obj.revsub-1
            obj.reply_cnt = obj.reply_cnt+1

    # Returns true if there are un-reviewed postings in this site
    security.declareProtected(ModeratePostings, 'unmoderated_postings')
    def unmoderated_postings(self):
        return len(self.searchResults({'reviewed' : 1 })) < len(self.data)        

    # Searchable interface     
    security.declareProtected(View, '__call__')
    def __call__(self, REQUEST=None, internal=0, **kw):        
        brains = apply(self.searchResults,(REQUEST,),kw)
	if internal:
	    return map(lambda x: x.getObject(), brains)
	return brains

    # This renders the raw document text into stuff that can be displayed with a dtml-var
    # This method is overidable by placing a script/method called 'render' in the Squishdot Site object
    
    security.declareProtected(AddPostings, 'render')
    def render(self,storedLines,format):
        if not storedLines:
            return ''
        if   format=='Plain':
            return join(map(html_quote,storedLines),'<BR>\n')
        elif format=='HTML':
            return join(storedLines,'\n')
        elif format=='STX':
            return str(StructuredText.html_with_references(replace(join(storedLines,'\n'),'\r',''), level=3))
        return 'Invalid Format!'
    
    render__replaceable__ = REPLACEABLE
        
    security.declarePrivate('sendEmail')
    def sendEmail(self,msg,address,REQUEST,manage_notify=None):
        # sends an email to the address using the mailhost, if there is one
        try:
            if self.mail_host:
                mhost=getattr(self,self.mail_host)
                mail =self.mail_html(self, REQUEST, newItem=msg, email=address, manage_notify=manage_notify)
                mhost.send(mail)
        except:
            # something bad happened :-(
            LOG('Squishdot',
                ERROR,
                'Error sending notification email',
                'URL: %s%s/%s' % (self.absolute_url(),msg.thread_path(),msg.id),
                error=sys.exc_info)
            
InitializeClass(SquishSite)





