/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dataFile.h"

#include "dataNode.h"
#include <visu_object.h>
#include <visu_tools.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolShade.h>

/**
 * SECTION:dataFile
 * @short_description: Adds a possibility to colorize nodes depending
 * on data read in an input file.
 *
 * <para>With this module, it is possible to colorize nodes depending
 * on data read in an input file. An input file can be associated to a
 * #VisuData object using dataFileSet_file(). Doing this, the
 * rendering is changed and nodes are colorized following a scheme
 * describe later. To turn off colorization without removing the data
 * file (for temporary turn off for instance), use
 * dataFileSet_used().</para>
 * <para>The input file must have the same numbers of uncommented
 * lines as there are nodes in the #VisuData associated with. If less
 * data is given, missing data are treaded as min values data. The
 * input data file can has as much column as desired. The colorization
 * is based on a linear color transformation. This transformation is
 * applied on color channel in RGB mode or in HSV mode. Resulting
 * color is given by : [resulting color vect] = [vectB] + [input
 * data][vectA], where [input data] are input data scaled to [0;1]. It
 * is possible to choose which column multiplies which color
 * channel.</para>
 * <para>Using dataFileApply_hideOnMinValue(), it is possible to hide
 * some nodes depending on given input data. If the @column argument
 * is -1, it unmask all nodes.</para>
 */

struct DataFile_struct
{
  /* Set to TRUE to colorize. */
  gboolean used;

  Shade *shade;

  /* Columns to be used. */
  int colUsed[3];
  int scaleUsed;

  /* Scheme to scale : manual or auto. */
  DataFileInputScaleId scaleType;

  /* Min and max values for manual scale. */
  float minMaxValues[2];

  /* Number of read columns. */
  int nbColumns;

  /* Values min and max read for each column. */
  float *readMinMax;

  /* File where data has been read. */
  gchar *file;
};
typedef struct DataFile_struct DataFile;

/* Cache pointers for data from currentVisuData.
   This cache is usefull to avoid too much g_object_get_data().
   This cache is refresh as soon as currentVisuData is changed. */
DataFile *cacheDataFile;
VisuData *cacheVisuData;

/* Default values */
#define DATAFILE_NB_COLUMN_DEFAULT  -4
#define DATAFILE_COLUMN_X  -3
#define DATAFILE_COLUMN_Y  -2
#define DATAFILE_COLUMN_Z  -1
#define DATAFILE_VISU_GLEXT_SCALE_TYPE_DEFAULT dataFile_normalize
#define DATAFILE_MIN_NORM_DEFAULT   -1.
#define DATAFILE_MAX_NORM_DEFAULT   +1.
#define DATAFILE_COLOR_TYPE_DEFAULT dataFile_hsv

#define DATAFILE_ID "dataColor_data"

/* Internal variables. */
static VisuDataNode *dataNode;
static GValue dataValue = {0, {{0}, {0}}};
static GQuark quark;

/* local methods. */
static void dataFileFree(gpointer data);
static DataFile* dataFileNew(VisuData *visuData);
static DataFile* _get_dataColor(VisuData *data, gboolean create, gboolean *new);

static void colorFromUserData(VisuData *visuData, float rgba[4],
			      VisuElement *ele, VisuNode* node);
static float radiusFromUserData(VisuData *visuData, VisuNode *node);
static float dataFileGet_valuesFromData(VisuData *visuData,
					int column, float fromVal);
static void freeData(gpointer obj, gpointer data);
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data);

int initDataFileModule()
{
  cacheDataFile = (DataFile*)0;
  cacheVisuData = (VisuData*)0;
  quark         =  g_quark_from_static_string("data_file");

  /* Register a new NodeData. */
  dataNode = VISU_DATA_NODE(visu_data_node_new(DATAFILE_ID, G_TYPE_FLOAT));
  visu_data_node_setLabel(dataNode, _("Colorisation data"));

  g_value_init(&dataValue, G_TYPE_POINTER);

  return 1;
}

/**
 * dataFileGet_quark:
 *
 * Internal routine for error handling.
 *
 * Returns: the #GQuark associated to errors related to colour data
 * files.
 */
GQuark dataFileGet_quark()
{
  return quark;
}

void dataFileReDraw(VisuData *data)
{
  visu_data_createAllNodes(data);
  VISU_ADD_REDRAW;
}


static DataFile* dataFileNew(VisuData *visuData)
{
  DataFile *dataFile;
  GList *lst;

  DBG_fprintf(stderr, "Visu dataFile: allocate data informations for"
	      " given VisuData object %p.\n", (gpointer)visuData);
  g_return_val_if_fail(visuData, (DataFile*)0);

  lst = toolShadeListGet();

  dataFile = g_malloc(sizeof(DataFile));
  dataFile->used            = FALSE;
  dataFile->file            = (gchar*)0;
  dataFile->readMinMax      = (float*)0;
  dataFile->nbColumns       = 0;
  dataFile->scaleType       = DATAFILE_VISU_GLEXT_SCALE_TYPE_DEFAULT;
  dataFile->minMaxValues[0] = DATAFILE_MIN_NORM_DEFAULT;
  dataFile->minMaxValues[1] = DATAFILE_MAX_NORM_DEFAULT;
  dataFile->shade           = (lst)?shadeCopy((Shade*)lst->data):(Shade*)0;
  dataFile->colUsed[0]      = DATAFILE_NB_COLUMN_DEFAULT;
  dataFile->colUsed[1]      = DATAFILE_NB_COLUMN_DEFAULT;
  dataFile->colUsed[2]      = DATAFILE_NB_COLUMN_DEFAULT;
  dataFile->scaleUsed       = DATAFILE_NB_COLUMN_DEFAULT;
  g_object_set_data_full(G_OBJECT(visuData), "dataColor_parameters",
			 (gpointer)dataFile, dataFileFree);
  DBG_fprintf(stderr, " | %p\n", (gpointer)dataFile);

  /* Set the cache. */
  cacheDataFile = dataFile;
  cacheVisuData = visuData;

  return dataFile;
}

static void dataFileFree(gpointer data)
{
  DataFile *dataFile;

  DBG_fprintf(stderr, "Data File : freeing dataFile.\n");
  dataFile = (DataFile*)data;

  if (dataFile->readMinMax)
    g_free(dataFile->readMinMax);
  if (dataFile->file)
    g_free(dataFile->file);
  if (dataFile->shade)
    shadeFree(dataFile->shade);
  g_free(dataFile);

  /* Update the cache. */
  if (dataFile == cacheDataFile)
    {
      cacheDataFile = (DataFile*)0;
      cacheVisuData = (VisuData*)0;
    }

}

void dataFileActivate(VisuData *visuData, gboolean status)
{
  g_return_if_fail(visuData);

  if (status)
    {
      visu_element_setUpdateNodesOnMaterialChange();
      visu_data_setColorFunc(visuData, colorFromUserData);
      visu_data_setNodeScalingFunc(visuData, radiusFromUserData);
    }
  else
    {
      visu_element_unsetUpdateNodesOnMaterialChange();
      visu_data_setColorFunc(visuData, (VisuDataColorFunc)0);
      visu_data_setNodeScalingFunc(visuData, (VisuDataScalingFunc)0);
    }
}

static void freeData(gpointer obj, gpointer data)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * GPOINTER_TO_INT(data), obj);
#else
  if (GPOINTER_TO_INT(data) > 0)
    g_free(obj);
#endif
}
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data)
{
  float *data;
  int nb;

  nb = GPOINTER_TO_INT(user_data);
#if GLIB_MINOR_VERSION > 9
  data = g_slice_alloc(sizeof(float) * nb);
#else
  data = g_malloc(sizeof(float) * nb);
#endif
  if (orig)
    memcpy(data, orig, sizeof(float) * nb);
  else
    memset(data, 0, sizeof(float) * nb);

  return (gpointer)data;
}

gboolean dataFileSet_file(VisuData *dataObj, const char* filename, gboolean *new,
			  GError **error)
{
  int i, nb, colUsed[3];
  GIOChannel *readFrom;
  GIOStatus status;
  GString *line;
  gsize term;
  int nAtome;
  float *data;
  gchar **dataRead;
  DataFile *dataFile;
  VisuDataIter iter;
  gboolean voidLine;
  VisuNodeArray *nodeArray;
  VisuNodeProperty *prop;
#if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
#endif

  DBG_fprintf(stderr, "Visu dataFile: set data file to '%s' for %p.\n",
	      filename, (gpointer)dataObj);
  g_return_val_if_fail(dataObj && filename, FALSE);
  g_return_val_if_fail(error && (*error) == (GError*)0 && new, FALSE);

  *new = TRUE;

  /* Test if this VisuData has already been colorized by testing
     the existence of the 'dataColor_used' property. If not, all required
     memory is allocated. */
  dataFile = g_object_get_data(G_OBJECT(dataObj), "dataColor_parameters");
  if (!dataFile)
    dataFile = dataFileNew(dataObj);
  else
    *new = FALSE;
  g_return_val_if_fail(dataFile, 0);

  /* erase previously stored dataFile informations. */
  dataFile->nbColumns = 0;
  for (i = 0; i < 3; i++)
    {
      colUsed[i] = dataFile->colUsed[i];
      dataFile->colUsed[i] = -1;
    }
  if (dataFile->file)
    {
      g_free(dataFile->file);
      dataFile->file = (gchar*)0;
    }
  if (dataFile->readMinMax)
    {
      g_free(dataFile->readMinMax);
      dataFile->readMinMax = (float*)0;
    }

  /* We clear previously stored user data from the current
     visuData structure. */
  nodeArray = visu_data_getNodeArray(dataObj);
  visu_node_array_freeProperty(nodeArray, DATAFILE_ID);
  prop = (VisuNodeProperty*)0;
  visu_data_node_setUsed(dataNode, dataObj, 0);

  readFrom = g_io_channel_new_file(filename, "r", error);
  if (!readFrom)
    return FALSE;

  DBG_fprintf(stderr, "Visu DataFile: begin reading file.\n");
#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  line = g_string_new("");
  nAtome = 0;
  visu_data_iterNew(dataObj, &iter);
  for (visu_data_iterStart(dataObj, &iter); iter.node;
       visu_data_iterNextNodeNumber(dataObj, &iter))
    {
      /* If the file is not finished, we read until a non-void line
	 and return this line. */
      voidLine = TRUE;
      do
	{
	  status = g_io_channel_read_line_string(readFrom, line, &term, error);
	  if (status == G_IO_STATUS_NORMAL)
	    {
	      g_strchug(line->str);
	      voidLine = (line->str[0] == '#' ||
			  line->str[0] == '!' ||
			  line->str[0] == '\0');
	    }
	}
      while (status == G_IO_STATUS_NORMAL && voidLine);
      if (status == G_IO_STATUS_ERROR)
	{
	  g_string_free(line, TRUE);
	  dataFileActivate(dataObj, FALSE);
	  dataFile->used = FALSE;
	  status = g_io_channel_shutdown(readFrom, FALSE, (GError**)0);
	  g_io_channel_unref(readFrom);
	  return FALSE;
	}

      /* If still not done, we test the number of columns. */
      if (dataFile->nbColumns == 0)
	{
	  if (!voidLine)
	    {
	      dataRead = g_strsplit_set(line->str, " \t;:\n", MAX_LINE_LENGTH);
	      DBG_fprintf(stderr, "Visu dataFile: looking for number of columns.\n");
	      dataFile->nbColumns = 0;
	      for (i = 0; dataRead[i]; i++)
		if (dataRead[i][0])
		  {
		    DBG_fprintf(stderr, "   | one token : '%s'\n", dataRead[i]);
		    dataFile->nbColumns += 1;
		  }
	      g_strfreev(dataRead);
	      DBG_fprintf(stderr, "Visu dataFile: detected number of columns : %d\n",
			  dataFile->nbColumns);
	    }
	  if (dataFile->nbColumns == 0 || voidLine)
	    {
	      *error = g_error_new(VISU_ERROR_DATA_FILE, DATA_FILE_ERROR_NO_COLUMN,
				   _("Can't find any column of"
				     " data in the given file.\n"));
	      g_string_free(line, TRUE);
	      dataFileActivate(dataObj, FALSE);
	      dataFile->used = FALSE;
	      status = g_io_channel_shutdown(readFrom, FALSE, (GError**)0);
	      g_io_channel_unref(readFrom);
	      return FALSE;
	    }
	  dataFile->readMinMax = g_malloc(sizeof(float) * 2 * (dataFile->nbColumns));
	  /* Add a node property with the number of columns. */
	  prop = visu_node_property_newPointer(nodeArray, DATAFILE_ID,
					     freeData, newOrCopyData,
					     GINT_TO_POINTER(dataFile->nbColumns));
	  /* Set the dimensions for the node property. */
	  visu_data_node_setUsed(dataNode, dataObj, dataFile->nbColumns);
	}

      /* We read the data from the line, if not void. */
      data = newOrCopyData((gconstpointer)0, GINT_TO_POINTER(dataFile->nbColumns));
      if (!voidLine)
	{
	  dataRead = g_strsplit_set(line->str, " \t;:\n", MAX_LINE_LENGTH);
	  nb = 0;
	  for (i = 0; dataRead[i] && nb < dataFile->nbColumns; i++)
	    {
	      if (1 == sscanf(dataRead[i], "%f", data + nb))
		{
		  DBG_fprintf(stderr, "   '%s' - '%f'",
			      dataRead[i], data[nb]);
		  nb += 1;
		}
	    }
	  DBG_fprintf(stderr, "\n");
	  g_strfreev(dataRead);
	}
      else
	{
	  if (!*error)
	    *error = g_error_new(VISU_ERROR_DATA_FILE, DATA_FILE_ERROR_MISSING_DATA,
				 _("There are more nodes than data.\n"));
	}
      /* Associates the values to the node. */
      g_value_set_pointer(&dataValue, data);
      visu_node_property_setValue(prop, iter.node, &dataValue);
      /* Check for minMax values for each column. */
      for (i = 0; i < dataFile->nbColumns; i++)
	{
	  if (nAtome == 0)
	    {
	      dataFile->readMinMax[i * 2 + 0] = data[i];
	      dataFile->readMinMax[i * 2 + 1] = data[i];
	    }
	  else
	    {
	      if (data[i] < dataFile->readMinMax[i * 2 + 0])
		dataFile->readMinMax[i * 2 + 0] = data[i];
	      if (data[i] > dataFile->readMinMax[i * 2 + 1])
		dataFile->readMinMax[i * 2 + 1] = data[i];
	    }
	}
      nAtome++;
    }
  status = g_io_channel_shutdown(readFrom, FALSE, (GError**)0);
  g_io_channel_unref(readFrom);
  g_string_free(line, TRUE);

  dataFile->file = g_strdup(filename);

  if (dataFile->nbColumns == 0)
    {
      *error = g_error_new(VISU_ERROR_DATA_FILE, DATA_FILE_ERROR_NO_COLUMN,
			   _("Can't find any columns with numbers.\n"
			     "Valid format are as much numbers as desired, separated"
			     " by any of the following characters : [ ;:\\t].\n"));
      dataFile->used = FALSE;
      return FALSE;
    }

  if (DEBUG)
    {
      DBG_fprintf(stderr, "Visu dataFile : min/max values :\n");
      for (i = 0; i < dataFile->nbColumns; i++)
	DBG_fprintf(stderr, "Visu dataFile :  col %d -> %f (min) and %f (max).\n",
		    i, dataFile->readMinMax[i * 2 + 0], dataFile->readMinMax[i * 2 + 1]);
    }
  for (i = 0; i < 3; i++)
    dataFile->colUsed[i] = (colUsed[i] >= dataFile->nbColumns)?-1:colUsed[i];

  dataFileActivate(dataObj, dataFile->used);
  cacheDataFile = dataFile;
  cacheVisuData = dataObj;

#if DEBUG == 1
  g_timer_stop(timer);
  fprintf(stderr, "Visu dataFile: data associated in %g micro-s.\n", g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif

  return TRUE;
}

static DataFile* _get_dataColor(VisuData *data, gboolean create, gboolean *new)
{
  DataFile *dataFile;

  g_return_val_if_fail(data, (DataFile*)0);

  DBG_fprintf(stderr, "Visu DataFile: get data for %p (%p %p).\n",
	      (gpointer)data, (gpointer)cacheVisuData, (gpointer)cacheDataFile);
  if (new)
    *new = FALSE;
  if (data == cacheVisuData)
    dataFile = cacheDataFile;
  else
    {
      dataFile = (DataFile*)g_object_get_data(G_OBJECT(data), "dataColor_parameters");
      cacheDataFile = dataFile;
      cacheVisuData = data;
    }
  if (!dataFile && create)
    {
      DBG_fprintf(stderr, "Visu DataFile: get data and create new.\n");
      dataFile = dataFileNew(data);
      if (new)
	*new = TRUE;
    }
  return dataFile;
}

/**
 * dataFileGet_fileSet:
 * @visuData: a #VisuData object.
 *
 * A set of data per node can be associated to a #VisuData. This
 * routine retrieves if @visuData has it or not.
 *
 * Since: 3.6
 *
 * Returns: TRUE if @visuData has colour data associated.
 */
gboolean dataFileGet_fileSet(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);

  if (dataFile && dataFile->file)
    return TRUE;
  else
    return FALSE;
}
gchar* dataFileGet_file(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);

  if (dataFile)
    return g_strdup(dataFile->file);
  else
    return (gchar*)0;
}
gboolean dataFileGet_fileMinMaxFromColumn(VisuData *visuData,
					  float minMax[2], int column)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);
  g_return_val_if_fail(dataFile, FALSE);
  g_return_val_if_fail(column >= 0 && column < dataFile->nbColumns, FALSE);

  minMax[0] = dataFile->readMinMax[column * 2 + 0];
  minMax[1] = dataFile->readMinMax[column * 2 + 1];
  return TRUE;
}

gboolean dataFileSet_used(VisuData *visuData, int val)
{
  DataFile *dataFile;
  gboolean new;

  DBG_fprintf(stderr, "Visu dataFile: set using flag for"
	      " dataFile module to %d .\n", val);

  dataFile = _get_dataColor(visuData, TRUE, &new);
  g_return_val_if_fail(dataFile, FALSE);

  if (val == dataFile->used)
    return FALSE;

  dataFile->used = val;

  dataFileActivate(visuData, dataFile->used);
  return !new;
}
gboolean dataFileGet_used(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);
  
  if (!dataFile)
    return FALSE;
  else
    return dataFile->used;
}

gboolean dataFileSet_scaleType(VisuData *visuData, DataFileInputScaleId scale)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);  
  g_return_val_if_fail(dataFile, FALSE);

  DBG_fprintf(stderr, "Visu dataFile: set the scale type to %d (previuosly %d).\n",
	      scale, dataFile->scaleType);
  if (scale == dataFile->scaleType)
    return FALSE;
  dataFile->scaleType = scale;

  return dataFile->used;
}
DataFileInputScaleId dataFileGet_scaleType(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return DATAFILE_VISU_GLEXT_SCALE_TYPE_DEFAULT;
  else
    return dataFile->scaleType;
}
gboolean dataFileSet_min(VisuData *visuData, float min)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);
  g_return_val_if_fail(dataFile, FALSE);
/*   g_return_val_if_fail(min < dataFile->minMaxValues[1], 0); */

  DBG_fprintf(stderr, "Visu dataFile: set the min value to %f (previuosly %f).\n",
	      min, dataFile->minMaxValues[0]);
  if (dataFile->minMaxValues[0] == min)
    return FALSE;
  dataFile->minMaxValues[0] = min;

  return dataFile->used;
}
gboolean dataFileSet_max(VisuData *visuData, float max)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);  
  g_return_val_if_fail(dataFile, FALSE);
/*   g_return_val_if_fail(max > dataFile->minMaxValues[0], 0); */

  DBG_fprintf(stderr, "Visu dataFile: set the max value to %f (previuosly %f).\n",
	      max, dataFile->minMaxValues[1]);
  if (dataFile->minMaxValues[1] == max)
    return FALSE;
  dataFile->minMaxValues[1] = max;

  return dataFile->used;
}
float dataFileGet_min(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return DATAFILE_MIN_NORM_DEFAULT;
  else
    return dataFile->minMaxValues[0];
}
float dataFileGet_max(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return DATAFILE_MAX_NORM_DEFAULT;
  else
    return dataFile->minMaxValues[1];
}

int dataFileGet_nbColumns(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return DATAFILE_NB_COLUMN_DEFAULT;
  else
    return dataFile->nbColumns;
}

gboolean dataFileSet_colUsed(VisuData *visuData, int val, int pos)
{
  DataFile *dataFile;

  g_return_val_if_fail(pos >= 0 && pos < 3, FALSE);

  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);  
  g_return_val_if_fail(dataFile, FALSE);

  DBG_fprintf(stderr, "Visu dataFile: channel %d uses column %d (previuosly %d).\n",
	      pos, val, dataFile->colUsed[pos]);
  g_return_val_if_fail(val < dataFile->nbColumns &&
		       val >= DATAFILE_NB_COLUMN_DEFAULT, FALSE);

  if (dataFile->colUsed[pos] == val)
    return FALSE;
  dataFile->colUsed[pos] = val;

  return dataFile->used;
}
int* dataFileGet_colUsed(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return (int*)0;
  else
    return dataFile->colUsed;
}
gboolean dataFileSet_scalingUsed(VisuData *visuData, int val)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);  
  g_return_val_if_fail(dataFile, FALSE);

  DBG_fprintf(stderr, "Visu dataFile: scaling uses column %d (previuosly %d).\n",
	      val, dataFile->scaleUsed);
  g_return_val_if_fail((val < dataFile->nbColumns &&
			val >= 0) || val == DATAFILE_NB_COLUMN_DEFAULT, FALSE);

  if (dataFile->scaleUsed == val)
    return FALSE;
  dataFile->scaleUsed = val;

  return dataFile->used;
}
int dataFileGet_scalingUsed(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);  

  if (!dataFile)
    return DATAFILE_NB_COLUMN_DEFAULT;
  else
    return dataFile->scaleUsed;
}
gboolean dataFileSet_shade(VisuData *visuData, Shade *shade)
{
  DataFile *dataFile;

  DBG_fprintf(stderr, "Visu DataFile: set shade.\n");
  dataFile = _get_dataColor(visuData, TRUE, (gboolean*)0);
  g_return_val_if_fail(dataFile, FALSE);

  shadeFree(dataFile->shade);
  dataFile->shade = shadeCopy(shade);

  return dataFile->used;
}
Shade* dataFileGet_shade(VisuData *visuData)
{
  DataFile *dataFile;

  dataFile = _get_dataColor(visuData, FALSE, (gboolean*)0);

  if (!dataFile)
    return (Shade*)0;
  else
    return dataFile->shade;
}
gboolean dataFileApply_hideOnMinValue(VisuData *visuData, int column, float value)
{
  gboolean redraw;
  float *storedValues;
  DataFile *dataFile;
  VisuDataIter iter;
  VisuNodeProperty *prop;
  GValue data = {0, {{0}, {0}}};

  g_return_val_if_fail(visuData, FALSE);

  if (column == -1)
    {
      DBG_fprintf(stderr, "Data File : unmask previously masked nodes.\n");
      return FALSE;
    }
  
  dataFile = (DataFile*)g_object_get_data(G_OBJECT(visuData), "dataColor_parameters");
  if (!dataFile)
    return FALSE;
  if (dataFile->nbColumns == 0)
    return FALSE;

  g_return_val_if_fail(column > DATAFILE_NB_COLUMN_DEFAULT && column < dataFile->nbColumns, FALSE);
  
  g_value_init(&data, G_TYPE_POINTER);
  prop = visu_node_array_getProperty(visu_data_getNodeArray(visuData), DATAFILE_ID);

  redraw = FALSE;
  visu_data_iterNew(visuData, &iter);
  for (visu_data_iterStartVisible(visuData, &iter); iter.node;
       visu_data_iterNextVisible(visuData, &iter))
    {
      visu_node_property_getValue(prop, iter.node, &data);
      storedValues = (float*)g_value_get_pointer(&data);
      g_return_val_if_fail(storedValues, FALSE);
      if (storedValues[column] < value) /* && -storedValues[column] < value) */
	redraw = visu_node_setVisibility(iter.node, FALSE) || redraw;
    }

  return TRUE;
}


/*******************/
/* Drawing methods */
/*******************/
/* In these methods, vectA and vectB (and others) are not retrieve from the visuData object for
   speed reason. We use instead global pointers dataFile_vectA/B that are
   pointing to the right place in the visuData object that needs to be rendered. */

/* Normalization method */
static float dataFileGet_valuesFromData(VisuData *visuData _U_,
					int column, float fromVal)
{
  float res;

  g_return_val_if_fail(cacheDataFile, 0.);
  switch (cacheDataFile->scaleType)
    {
    case dataFile_normalize:
      g_return_val_if_fail(column >= 0 && column < cacheDataFile->nbColumns, 0.);
      res = ( (fromVal - cacheDataFile->readMinMax[column * 2 + 0]) /
	      (cacheDataFile->readMinMax[column * 2 + 1] -
	       cacheDataFile->readMinMax[column * 2 + 0]) );
      DBG_fprintf(stderr, "Visu DataFile: normalise %f -> %f.\n", fromVal, res);
      return res;
    case dataFile_minMax:
      res = ( (fromVal - cacheDataFile->minMaxValues[0]) /
	      (cacheDataFile->minMaxValues[1] - cacheDataFile->minMaxValues[0]) );
      DBG_fprintf(stderr, "Visu DataFile: normalise %f -> %f.\n",
		  fromVal, CLAMP(res, 0., 1.));
      return CLAMP(res, 0., 1.);
    }
  return 0.;
}

/*******************/
/* ToolColor functions */
/*******************/
static void colorFromUserData(VisuData *visuData, float rgba[4],
			      VisuElement *ele, VisuNode* node)
{
  float val[3], red[3], coord[3], delta;
  int i;
  float *storedValues;
  gboolean useCoord, useData;

  g_return_if_fail(visuData && node && ele && rgba);

  useCoord = (cacheDataFile->colUsed[0] == DATAFILE_COLUMN_X ||
	      cacheDataFile->colUsed[0] == DATAFILE_COLUMN_Y ||
	      cacheDataFile->colUsed[0] == DATAFILE_COLUMN_Z ||
	      cacheDataFile->colUsed[1] == DATAFILE_COLUMN_X ||
	      cacheDataFile->colUsed[1] == DATAFILE_COLUMN_Y ||
	      cacheDataFile->colUsed[1] == DATAFILE_COLUMN_Z ||
	      cacheDataFile->colUsed[2] == DATAFILE_COLUMN_X ||
	      cacheDataFile->colUsed[2] == DATAFILE_COLUMN_Y ||
	      cacheDataFile->colUsed[2] == DATAFILE_COLUMN_Z);
  useData = (cacheDataFile->colUsed[0] >= 0 ||
	     cacheDataFile->colUsed[1] >= 0 ||
	     cacheDataFile->colUsed[2] >= 0);

  storedValues = (float*)0;
  if (useData)
    {
      visu_node_array_getPropertyValue(visu_data_getNodeArray(visuData), node,
				DATAFILE_ID, &dataValue);
      storedValues = (float*)g_value_get_pointer(&dataValue);
      g_return_if_fail(storedValues);
    }
  if (useCoord)
    {
      visu_data_getNodePosition(visuData, node, coord);
      visu_data_convertXYZtoBoxCoordinates(visuData, red, coord);
      if (cacheDataFile->scaleType == dataFile_minMax)
	{
	  delta = cacheDataFile->minMaxValues[1] - cacheDataFile->minMaxValues[0];
	  red[0] = CLAMP((red[0] - cacheDataFile->minMaxValues[0]) / delta, 0.f, 1.f);
	  red[1] = CLAMP((red[1] - cacheDataFile->minMaxValues[0]) / delta, 0.f, 1.f);
	  red[2] = CLAMP((red[2] - cacheDataFile->minMaxValues[0]) / delta, 0.f, 1.f);
	}
    }

  for (i = 0; i < 3; i++)
    if (cacheDataFile->colUsed[i] == DATAFILE_NB_COLUMN_DEFAULT)
      val[i] = 1.;
    else if (cacheDataFile->colUsed[i] == DATAFILE_COLUMN_X)
      val[i] = fModulo(red[0], 1);
    else if (cacheDataFile->colUsed[i] == DATAFILE_COLUMN_Y)
      val[i] = fModulo(red[1], 1);
    else if (cacheDataFile->colUsed[i] == DATAFILE_COLUMN_Z)
      val[i] = fModulo(red[2], 1);
    else
      val[i] = dataFileGet_valuesFromData(visuData, cacheDataFile->colUsed[i],
					  storedValues[cacheDataFile->colUsed[i]]);

  shadeGet_RGBFromValues(cacheDataFile->shade, rgba, val);
/*   fprintf(stderr, "%f %f %f  ->  %f %f %f %f\n", */
/* 	  val[0], val[1], val[2], rgba[0], rgba[1], rgba[2], rgba[3]); */
}
static float radiusFromUserData(VisuData *visuData, VisuNode *node)
{
  float *storedValues;

  if (cacheDataFile->scaleUsed == DATAFILE_NB_COLUMN_DEFAULT)
    return 1.f;

  storedValues = (float*)0;
  visu_node_array_getPropertyValue(visu_data_getNodeArray(visuData), node,
			    DATAFILE_ID, &dataValue);
  storedValues = (float*)g_value_get_pointer(&dataValue);
  return dataFileGet_valuesFromData(visuData, cacheDataFile->scaleUsed,
				    storedValues[cacheDataFile->scaleUsed]);

}
