/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "geometry.h"

#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include <opengl.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolConfigFile.h>
#include <coreTools/toolShade.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>
#include <visu_extension.h>
#include <visu_configFile.h>

/**
 * SECTION:geometry
 * @short_description: different routines to do high level geometry
 * studies on a box or a set of boxes.
 *
 * <para>The first possibility of the #geometry section is to make a
 * two by two difference node position difference between two
 * #VisuData objects. The #VisuNode positions are compared (number to number and
 * not closed equivalent to closed equivalent) and stored for
 * visualisation. The visualisation is done through small arrows
 * position on the currently visualised file.</para>
 *
 * Since: 3.5
 */

enum PathItem
  {
    PATH_ITEM_COORD,
    PATH_ITEM_DELTA
  };
struct Item_
{
  enum PathItem type;
  guint time;
  float dxyz[3];
  float energy;
};
typedef struct Path_
{
  guint nodeId;

  /* A per node translation. */
  float translation[3];

  guint size, nItems;
  struct Item_ *items;
} Path;
/**
 * VisuPathes:
 *
 * An opaque structure to save a set of pathes.
 *
 * Since: 3.6
 */
struct VisuPathes_
{
  /* The current time stamp. */
  guint time;

  /* The global translation. */
  float translation[3];

  /* The min/max for the possible energies along the pathes. */
  float minE, maxE;
  Shade *shade;

  /* A list of Path elements. */
  GList *lst;
};

#define GEOMETRY_PATH_ALLOC_SIZE 50
#define GEOMETRY_DIFF_ID        "geometry_diff"
#define GEOMETRY_DIFF_MINMAX_ID "geometry_diffMinMax"

#define FLAG_RESOURCE_ARROW "nodeDisplacement_arrow"
#define DESC_RESOURCE_ARROW "Describe the arrow to be drawn ; four floats (tail lg. tail rd. head lg. head rd., negative values means auto)"
static gboolean readArrow(gchar **lines, int nbLines, int position,
			  VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_RATIO_MIN "nodeDisplacement_minThreshold"
#define DESC_RESOURCE_RATIO_MIN "Choose the minimum value for drawn arrows in geometry differences ; float (ratio threshold if between -1 and 0)"
#define DEFT_RESOURCE_RATIO_MIN -0.1f
static gboolean readRatioMin(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_RATIO_STR "nodeDisplacement_lblThreshold"
#define DESC_RESOURCE_RATIO_STR "Choose the minimum value for labels in geometry differences ; float (ratio threshold if between -1 and 0)"
#define DEFT_RESOURCE_RATIO_STR -0.9f
static gboolean readRatioStr(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_MULT "nodeDisplacement_factor"
#define DESC_RESOURCE_MULT "Choose the factor to draw arrows in geometry differences ; float (negative means auto)"
#define DEFT_RESOURCE_MULT -1.f
static gboolean readMult(gchar **lines, int nbLines, int position,
			 VisuData *dataObj, GError **error);

static void exportResources(GString *data, VisuData *dataObj);

/* Local variables. */
static VisuExtension *extGeo = NULL;
static float ratioMin = DEFT_RESOURCE_RATIO_MIN;
static float ratioStr = DEFT_RESOURCE_RATIO_STR;
static float mult = DEFT_RESOURCE_MULT;
static float arrow[4] = {-1.f, -1.f, -1.f, -1.f};

/* Local methods. */
static void geometryCreate_extGeo();

/**********************/
/* Memory management. */
/**********************/
static void freeData(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * 6, obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data _U_)
{
  float *data;

#if GLIB_MINOR_VERSION > 9
  data = g_slice_alloc(sizeof(float) * 6);
#else
  data = g_malloc(sizeof(float) * 6);
#endif
  if (orig)
    memcpy(data, orig, sizeof(float) * 6);
  else
    memset(data, 0, sizeof(float) * 6);

  return (gpointer)data;
}

/*************************/
/* Calculation routines. */
/*************************/
/**
 * visu_geodiff_getPeriodicDistance:
 * @diff: a location to store the shift.
 * @data: a #VisuData object.
 * @node1: a #VisuNode object.
 * @node2: another #VisuNode object.
 *
 * Compute the shortest distance between @node1 and @node2 of @data
 * taking into account the periodic boundary conditions.
 *
 * Since: 3.5
 */
void visu_geodiff_getPeriodicDistance(float diff[3], VisuData *data,
                                      VisuNode *node1, VisuNode *node2)
{
  float red[3];
  VisuDataBoxBoundaries bc;

  diff[0] = node1->xyz[0] + node1->translation[0] -
    node2->xyz[0] - node2->translation[0];
  diff[1] = node1->xyz[1] + node1->translation[1] -
    node2->xyz[1] - node2->translation[1];
  diff[2] = node1->xyz[2] + node1->translation[2] -
    node2->xyz[2] - node2->translation[2];
  
  bc = visu_data_getBoundaryConditions(data);
  if (bc == VISU_DATA_BOX_FREE)
    return;

  visu_data_convertXYZtoBoxCoordinates(data, red, diff);
  if (bc != VISU_DATA_BOX_SURFACE_YZ)
    {
      while (red[0] >= 0.5f) red[0] -= 1.f;
      while (red[0] < -0.5f) red[0] += 1.f;
    }

  if (bc != VISU_DATA_BOX_SURFACE_ZX)
    {
      while (red[1] >= 0.5f) red[1] -= 1.f;
      while (red[1] < -0.5f) red[1] += 1.f;
    }

  if (bc != VISU_DATA_BOX_SURFACE_XY)
    {
      while (red[2] >= 0.5f) red[2] -= 1.f;
      while (red[2] < -0.5f) red[2] += 1.f;
    }
  visu_data_convertBoxCoordinatestoXYZ(data, diff, red);
}

/**
 * visu_geodiff_new:
 * @dataRef: a #VisuData object ;
 * @data: another #VisuData object.
 * @reorder: a boolean.
 *
 * Compare the position of each #VisuNode of @dataRef and @data,
 * the result is store as a property of @data and can be visualise
 * using visu_geodiff_setActive(). If @reorder is TRUE, the nodes in
 * @data will be modified to follow the ordering of @dataRef.
 *
 * Since: 3.5
 *
 * Returns: TRUE if a difference is possible (same number of nodes).
 */
gboolean visu_geodiff_new(VisuData *dataRef, VisuData *data, gboolean reorder)
{
  VisuNodeArray *nodesRef, *nodes;
  guint i;
  VisuNodeProperty *prop;
  VisuDataIter iter, iterRef;
  GValue diffValue = {0, {{0}, {0}}};
  float *diff, align[3];
  float *minMax;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataRef), FALSE);
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  DBG_fprintf(stderr, "Geometry: making a diff between %p and %p.\n",
	      (gpointer)dataRef, (gpointer)data);

  nodesRef = visu_data_getNodeArray(dataRef);
  nodes    = visu_data_getNodeArray(data);

  DBG_fprintf(stderr, " | %d - %d.\n", nodes->ntype, nodesRef->ntype);
  if (nodes->ntype != nodesRef->ntype)
    return FALSE;

  for (i = 0; i < nodes->ntype; i++)
    {
      DBG_fprintf(stderr, " | (%d) # %d - %d.\n",
                  i, nodes->numberOfStoredNodes[i], nodesRef->numberOfStoredNodes[i]);
      if (nodes->numberOfStoredNodes[i] != nodesRef->numberOfStoredNodes[i])
        return FALSE;
    }

  if (reorder)
    visu_data_reorder(data, dataRef);

  /* Ok, here the two files match with respect of number of nodes per
     element. */
  g_value_init(&diffValue, G_TYPE_POINTER);

  /* If the two files are free BC, we align the first atom. */
  align[0] = 0.f;
  align[1] = 0.f;
  align[2] = 0.f;
  if (visu_data_getBoundaryConditions(dataRef) == VISU_DATA_BOX_FREE &&
      visu_data_getBoundaryConditions(data) == VISU_DATA_BOX_FREE)
    {
      visu_geodiff_getPeriodicDistance(align, dataRef, visu_data_getNodeFromNumber(dataRef, 0),
                       visu_data_getNodeFromNumber(data, 0));
      DBG_fprintf(stderr, "Geometry: free BC bioxes, applying translation %gx%gx%g.\n",
                  align[0], align[1], align[2]);
    }

  /* We add a VisuData property to store the min max values. */
  minMax = g_malloc(sizeof(float) * 2);
  g_object_set_data_full(G_OBJECT(data), GEOMETRY_DIFF_MINMAX_ID,
			 (gpointer)minMax, g_free);
  minMax[0] = G_MAXFLOAT;
  minMax[1] = 0.f;

  /* Add a node property with the number of columns. */
  prop = visu_node_property_newPointer(nodes, GEOMETRY_DIFF_ID,
				     freeData, newOrCopyData, (gpointer)0);

  visu_data_iterNew(data, &iter);
  visu_data_iterNew(dataRef, &iterRef);
  for (visu_data_iterStartNumber(data, &iter), visu_data_iterStartNumber(dataRef, &iterRef);
       iter.node && iterRef.node;
       visu_data_iterNextNodeNumber(data, &iter), visu_data_iterNextNodeNumber(dataRef, &iterRef))
    {
      /* We read the data from the line, if not void. */
      diff = newOrCopyData((gconstpointer)0, (gpointer)0);
      visu_geodiff_getPeriodicDistance(diff, data, iter.node, iterRef.node);
      diff[0] += align[0];
      diff[1] += align[1];
      diff[2] += align[2];
      tool_matrix_cartesianToSpherical(diff + 3, diff);
      DBG_fprintf(stderr, "Geometry: diff atom %d, %9g %9g %9g (%9g %5g %5g).\n",
		  iter.node->number, diff[0], diff[1], diff[2],
		  diff[3], diff[4], diff[5]);
      minMax[0] = MIN(minMax[0], diff[3]);
      minMax[1] = MAX(minMax[1], diff[3]);
      
      /* Associates the values to the node. */
      g_value_set_pointer(&diffValue, diff);
      visu_node_property_setValue(prop, iter.node, &diffValue);
    }

  return TRUE;
}
/**
 * visu_pathes_new:
 * @translation: the current box translation (cartesian).
 *
 * Create a new #VisuPathes object.
 *
 * Since: 3.6
 *
 * Returns: the newly create object #VisuPathes, to be freed with
 * visu_pathes_free().
 */
VisuPathes* visu_pathes_new(float translation[3])
{
  VisuPathes *pathes;

  pathes = g_malloc(sizeof(VisuPathes));
  pathes->time = 0;
  pathes->translation[0] = translation[0];
  pathes->translation[1] = translation[1];
  pathes->translation[2] = translation[2];
  pathes->shade = (Shade*)0;
  pathes->minE = G_MAXFLOAT;
  pathes->maxE = -G_MAXFLOAT;
  pathes->lst = (GList*)0;

  return pathes;
}
static Path* newPath()
{
  Path *path;

  path = g_malloc(sizeof(Path));
  path->size    = GEOMETRY_PATH_ALLOC_SIZE;
  path->items   = g_malloc(sizeof(struct Item_) * path->size);
  path->nItems  = 0;

  return path;
}
static Path* addPathItem(Path *path, guint time, float xyz[3],
			 enum PathItem type, float energy)
{
  DBG_fprintf(stderr, "Visu Geometry: add a new item for path %p.\n",
	      (gpointer)path);

  if (!path)
    /* We create a new path for the given node id. */
    path = newPath();
  
  /* We reallocate if necessary. */
  if (path->nItems == path->size)
    {
      path->size  += GEOMETRY_PATH_ALLOC_SIZE;
      path->items  = g_realloc(path->items, sizeof(struct Item_) * path->size);
    }

  /* If previous type was PATH_ITEM_COORD and new is also
     PATH_ITEM_COORD we remove the previous one. */
  if (path->nItems > 0 && path->items[path->nItems - 1].type == PATH_ITEM_COORD &&
      type == PATH_ITEM_COORD)
    path->nItems -= 1;

  /* We add the new delta to the path. */
  path->items[path->nItems].time = time;
  path->items[path->nItems].type = type;
  path->items[path->nItems].dxyz[0] = xyz[0];
  path->items[path->nItems].dxyz[1] = xyz[1];
  path->items[path->nItems].dxyz[2] = xyz[2];
  path->items[path->nItems].energy  = (energy == G_MAXFLOAT)?-G_MAXFLOAT:energy;
  path->nItems += 1;

  return path;
}
/**
 * visu_pathes_addNodeStep:
 * @pathes: a set of pathes.
 * @time: the flag that give the number of expansion to update.
 * @nodeId: the node to expand the path of.
 * @xyz: the current position of the path.
 * @dxyz: the variation in the path.
 * @energy: the energy of the system.
 *
 * This routine expand the path for the given @nodeId at position @xyz
 * of @dxyz. The @energy value will be used only if
 * visu_pathes_setShade() is used with a non NULL #Shade. In that
 * case the @energy value will be used to colourise the provided path.
 *
 * Since: 3.6
 *
 * Returns: TRUE if a new path is started.
 */
gboolean visu_pathes_addNodeStep(VisuPathes *pathes, guint time, guint nodeId,
			  float xyz[3], float dxyz[3], float energy)
{
  GList *tmpLst;
  Path *path;
  gboolean new;

  /* Look for a Path with the good node id. */
  new = FALSE;
  for (tmpLst = pathes->lst; tmpLst && ((Path*)tmpLst->data)->nodeId != nodeId;
       tmpLst = g_list_next(tmpLst));
  if (!tmpLst)
    {
      /* We create a new path for the given node id. */
      path = addPathItem((Path*)0, time, xyz, PATH_ITEM_COORD, energy);
      path->nodeId         = nodeId;
      path->translation[0] = pathes->translation[0];
      path->translation[1] = pathes->translation[1];
      path->translation[2] = pathes->translation[2];
      pathes->lst   = g_list_prepend(pathes->lst, (gpointer)path);
      new = TRUE;
    }
  else
    path = (Path*)tmpLst->data;

  addPathItem(path, time, dxyz, PATH_ITEM_DELTA, energy);

  if (energy != G_MAXFLOAT)
    {
      pathes->minE = MIN(pathes->minE, energy);
      pathes->maxE = MAX(pathes->maxE, energy);
    }

  return new;
}
/**
 * visu_pathes_empty:
 * @pathes: a #VisuPathes object.
 *
 * Reinitialise internal values of a given @pathes.
 *
 * Since: 3.6
 */
void visu_pathes_empty(VisuPathes *pathes)
{
  GList *tmpLst;
  Path *path;

  g_return_if_fail(pathes);

  for (tmpLst = pathes->lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      path = (Path*)tmpLst->data;
      g_free(path->items);
      g_free(path);
    }
  g_list_free(pathes->lst);
  pathes->lst = (GList*)0;
  pathes->minE = G_MAXFLOAT;
  pathes->maxE = -G_MAXFLOAT;
  pathes->time = 0;
}
/**
 * visu_pathes_free:
 * @pathes: a #VisuPathes object.
 *
 * Free a set of pathes.
 *
 * Since: 3.6
 */
void visu_pathes_free(VisuPathes *pathes)
{
  visu_pathes_empty(pathes);
  g_free(pathes);
}
/**
 * visu_pathes_addFromDiff:
 * @data: a #VisuData object with a geometry difference (see
 * visu_geodiff_new()).
 * @pathes: the set of pathes to extend.
 *
 * This routine read the geometry difference hold in @data and add a
 * new step in the set of pathes. If new pathes are created, one
 * should call visu_pathes_setTranslation() to be sure that all
 * pathes are moved inside the box.
 *
 * Since: 3.6
 *
 * Returns: TRUE if new pathes have been added.
 */
gboolean visu_pathes_addFromDiff(VisuPathes *pathes, VisuData *data)
{
  GValue diffValue = {0, {{0}, {0}}};
  VisuNodeProperty *prop;
  VisuDataIter iter;
  float *diff, xyz[3];
  gdouble energy;
  gboolean new;

  new = FALSE;
  prop = visu_node_array_getProperty(visu_data_getNodeArray(data), GEOMETRY_DIFF_ID);
  g_return_val_if_fail(prop, FALSE);

  g_object_get(G_OBJECT(data), "totalEnergy", &energy, NULL);
  if (energy == G_MAXFLOAT)
    energy = pathes->minE;

  g_value_init(&diffValue, G_TYPE_POINTER);
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      visu_node_property_getValue(prop, iter.node, &diffValue);
      diff = (float*)g_value_get_pointer(&diffValue);
      if (diff[3] > 0.01f)
	{
	  xyz[0] = iter.node->xyz[0] - diff[0];
	  xyz[1] = iter.node->xyz[1] - diff[1];
	  xyz[2] = iter.node->xyz[2] - diff[2];
	  new = visu_pathes_addNodeStep(pathes, pathes->time, iter.node->number,
				 xyz, diff, (float)energy) || new;
	}
    }
  pathes->time += 1;

  return new;
}
/**
 * visu_pathes_getLength:
 * @pathes: a #VisuPathes object.
 *
 * Get the number of steps stored in a #VisuPathes.
 *
 * Since: 3.6
 *
 * Returns: the number of steps.
 */
guint visu_pathes_getLength(VisuPathes *pathes)
{
  g_return_val_if_fail(pathes, 0);

  return pathes->time;
}
/**
 * visu_pathes_setTranslation:
 * @pathes: a #VisuPathes object.
 * @cartCoord: three floats.
 *
 * Change the translation of the path, stored in cartesian
 * coordinates.
 *
 * Since: 3.6
 */
void visu_pathes_setTranslation(VisuPathes *pathes, float cartCoord[3])
{
  g_return_if_fail(pathes);

  pathes->translation[0] = cartCoord[0];
  pathes->translation[1] = cartCoord[1];
  pathes->translation[2] = cartCoord[2];
}
/**
 * visu_pathes_constrainInBox:
 * @pathes: a #VisuPathes object.
 * @data: a #VisuData object.
 *
 * Modify the corrdinates of the path nodes to contraint them in a box
 * (when applying translations for instance).
 *
 * Since: 3.6
 */
void visu_pathes_constrainInBox(VisuPathes *pathes, VisuData *data)
{
  double btx[3][3], xtb[3][3];
  float xyz[3], box[3], ext[3], t[3];
  GList *tmpLst;
  Path *path;

  g_return_if_fail(pathes && data);

  /* Create the transformation matrix. */
  xyz[0] = 1.f; xyz[1] = 0.f; xyz[2] = 0.f;
  visu_data_convertXYZtoBoxCoordinates(data, box, xyz);
  xtb[0][0] = (double)box[0]; xtb[1][0] = (double)box[1]; xtb[2][0] = (double)box[2];
  xyz[0] = 0.f; xyz[1] = 1.f; xyz[2] = 0.f;
  visu_data_convertXYZtoBoxCoordinates(data, box, xyz);
  xtb[0][1] = (double)box[0]; xtb[1][1] = (double)box[1]; xtb[2][1] = (double)box[2];
  xyz[0] = 0.f; xyz[1] = 0.f; xyz[2] = 1.f;
  visu_data_convertXYZtoBoxCoordinates(data, box, xyz);
  xtb[0][2] = (double)box[0]; xtb[1][2] = (double)box[1]; xtb[2][2] = (double)box[2];

  box[0] = 1.f; box[1] = 0.f; box[2] = 0.f;
  visu_data_convertBoxCoordinatestoXYZ(data, xyz, box);
  btx[0][0] = (double)xyz[0]; btx[1][0] = (double)xyz[1]; btx[2][0] = (double)xyz[2];
  box[0] = 0.f; box[1] = 1.f; box[2] = 0.f;
  visu_data_convertBoxCoordinatestoXYZ(data, xyz, box);
  btx[0][1] = (double)xyz[0]; btx[1][1] = (double)xyz[1]; btx[2][1] = (double)xyz[2];
  box[0] = 0.f; box[1] = 0.f; box[2] = 1.f;
  visu_data_convertBoxCoordinatestoXYZ(data, xyz, box);
  btx[0][2] = (double)xyz[0]; btx[1][2] = (double)xyz[1]; btx[2][2] = (double)xyz[2];

  visu_data_getExtension(data, ext);

  for(tmpLst = pathes->lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      path = (Path*)tmpLst->data;
      xyz[0] = path->items[0].dxyz[0] + pathes->translation[0];
      xyz[1] = path->items[0].dxyz[1] + pathes->translation[1];
      xyz[2] = path->items[0].dxyz[2] + pathes->translation[2];
      tool_matrix_constrainInBox(t, xyz, ext, xtb, btx);
      path->translation[0] = t[0] + pathes->translation[0];
      path->translation[1] = t[1] + pathes->translation[1];
      path->translation[2] = t[2] + pathes->translation[2];
    }
}
/**
 * visu_geodiff_hasData:
 * @data: a #VisuData object.
 *
 * A set coordinate differences can be associated to a #VisuData using visu_geodiff_new().
 *
 * Since: 3.6
 *
 * Returns: TRUE if the given @data has a set of differences associated.
 */
gboolean visu_geodiff_hasData(VisuData *data)
{
  return (g_object_get_data(G_OBJECT(data), GEOMETRY_DIFF_MINMAX_ID) != (gpointer)0);
}
/**
 * visu_geodiff_export:
 * @data: a #VisuData object.
 *
 * Create a string with differences of coordinates stored in @data in
 * cartesian coordinates.
 *
 * Since: 3.6
 *
 * Returns: a new string that should be freed after use.
 */
gchar* visu_geodiff_export(VisuData *data)
{
  GString *output;
  VisuNodeArray *nodes;
  VisuDataIter iter;
  GValue diffValue = {0, {{0}, {0}}};
  float *diff;
  gboolean start;
  VisuNodeProperty *prop;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (gchar*)0);
  
  nodes = visu_data_getNodeArray(data);
  prop = visu_node_array_getProperty(nodes, GEOMETRY_DIFF_ID);
  g_return_val_if_fail(prop, (gchar*)0);

  start = TRUE;
  output = g_string_new("#metaData: diff=[ \\\n");
  g_value_init(&diffValue, G_TYPE_POINTER);
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      if (!start)
	output = g_string_append(output, "; \\\n");
      visu_node_property_getValue(prop, iter.node, &diffValue);
      diff = (float*)g_value_get_pointer(&diffValue);

      g_string_append_printf(output, "# %12.8f; %12.8f; %12.8f",
			     diff[0], diff[1], diff[2]);
      start = FALSE;
    }
  output = g_string_append(output, " \\\n# ]\n");

  return g_string_free(output, FALSE);
}
/**
 * visu_pathes_pinPositions:
 * @pathes: a #VisuPathes object.
 * @data: a #VisuData object.
 *
 * Use the current positions of @data to extend @pathes.
 *
 * Since: 3.6
 */
void visu_pathes_pinPositions(VisuPathes *pathes, VisuData *data)
{
  VisuDataIter iter;
  GList *tmpLst;
  gdouble energy;

  g_return_if_fail(pathes && data);
  
  g_object_get(G_OBJECT(data), "totalEnergy", &energy, NULL);
  if (energy == G_MAXFLOAT)
    energy = pathes->minE;

  DBG_fprintf(stderr, "Visu Geometry: pin current positions.\n");
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      for (tmpLst = pathes->lst; tmpLst; tmpLst = g_list_next(tmpLst))
	if (((Path*)tmpLst->data)->nodeId == iter.node->number)
	  break;
      if (tmpLst)
	addPathItem((Path*)tmpLst->data, pathes->time,
		    iter.node->xyz, PATH_ITEM_COORD, (float)energy);
    }

  if (energy != G_MAXFLOAT)
    {
      pathes->minE = MIN(pathes->minE, (float)energy);
      pathes->maxE = MAX(pathes->maxE, (float)energy);
    }
}
/**
 * visu_pathes_setShade:
 * @pathes: a #VisuPathes object.
 * @shade: a #Shade object.
 *
 * Set the colourisation scheme for the path.
 *
 * Since: 3.6
 *
 * Returns: TRUE is the scheme is changed.
 */
gboolean visu_pathes_setShade(VisuPathes *pathes, Shade* shade)
{
  g_return_val_if_fail(pathes, FALSE);

  if (pathes->shade == shade)
    return FALSE;
  
  pathes->shade = shade;
  return TRUE;
}
/**
 * visu_pathes_getShade:
 * @pathes: a #VisuPathes object.
 *
 * The pathes are drawn with a colourisation scheme.
 *
 * Since: 3.6
 *
 * Returns: the #Shade used by the @pathes.
 */
Shade* visu_pathes_getShade(VisuPathes *pathes)
{
  g_return_val_if_fail(pathes, (Shade*)0);

  return pathes->shade;
}

/*********************/
/* Drawing routines. */
/*********************/
void geometryDraw(VisuData *data)
{
  VisuNodeArray *nodes;
  VisuDataIter iter;
  GValue diffValue = {0, {{0}, {0}}};
  float *diff, *minMax, xyz[3], rdT_, rdT, lgT, rdH_, rdH, lgH_, lgH, fact;
  float eleSize, rMult, sMult, r, s;
  VisuNodeProperty *prop;
  GLUquadricObj *obj;
  VisuElement *prevEle;
  char distStr[108];

  if (!visu_geodiff_hasData(data))
    return;

  obj = gluNewQuadric();

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  g_value_init(&diffValue, G_TYPE_POINTER);

  minMax = (float*)g_object_get_data(G_OBJECT(data), GEOMETRY_DIFF_MINMAX_ID);
  g_return_if_fail(minMax);

  openGLText_initFontList();

  /* We compute the maximum size of the arrow tail. */
  eleSize = visu_data_getAllElementExtens(data);

  nodes = visu_data_getNodeArray(data);
  prop = visu_node_array_getProperty(nodes, GEOMETRY_DIFF_ID);
  g_return_if_fail(prop);

  if (arrow[1] <= 0.f || mult <= 0.f)
    rdT_ = rdT = 0.2f * eleSize * ABS(mult);
  else
    rdT_ = rdT = arrow[1];
  if (arrow[2] <= 0.f || mult <= 0.f)
    lgH_ = lgH = 0.5f * eleSize * ABS(mult);
  else
    lgH_ = lgH = arrow[2];
  if (arrow[3] <= 0.f || mult <= 0.f)
    rdH_ = rdH = 0.3f * eleSize * ABS(mult);
  else
    rdH_ = rdH = arrow[3];
  if (mult <= 0.f)
    fact = eleSize * 4. * ABS(mult) / minMax[1];
  else
    fact = mult;
  if (ratioMin <= 0.f)
    {
      rMult = 1.f / minMax[1];
      r = -1.f;
    }
  else
    {
      rMult = 1.f;
      r = 1.f;
    }
  if (ratioStr <= 0.f)
    {
      sMult = 1.f / minMax[1];
      s = -1.f;
    }
  else
    {
      sMult = 1.f;
      s = 1.f;
    }

  prevEle = (VisuElement*)0;
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      if (!iter.node->rendered || !iter.element->rendered)
	continue;

      visu_node_property_getValue(prop, iter.node, &diffValue);
      diff = (float*)g_value_get_pointer(&diffValue);

      if (diff[3] * rMult > ratioMin * r)
	{
	  /* The size of the arrow... */
	  lgT = diff[3] * fact;
	  if (arrow[0] > 0.f && mult > 0.f)
	    {
	      rdT = rdT_ * diff[3] * fact;
	      lgH = lgH_ * diff[3] * fact;
	      rdH = rdH_ * diff[3] * fact;
	    }

	  /* We draw the arrow. */
	  visu_data_getNodePosition(data, iter.node, xyz);

	  glPushMatrix();
	  glTranslated(xyz[0], xyz[1], xyz[2]);
	  glRotated(diff[5], 0, 0, 1);  
	  glRotated(diff[4], 0, 1, 0); 

	  if (prevEle != iter.element)
	    {
	      openGLSet_highlightColor(iter.element->material,
				       iter.element->rgb, 1.f);
	      prevEle = iter.element;
	    }
	  visu_openGL_drawSmoothArrow(obj, -1, VISU_OPENGL_ARROW_CENTERED,
					   MAX(0.f, lgT - lgH), rdT, 10, FALSE,
					   MIN(lgH, lgT), rdH, 10, FALSE);

	  if (diff[3] * sMult > ratioStr * s)
	    {
	      glRasterPos3f(0.f, 0.f, 0.f);
	      sprintf(distStr, "%6.3f", diff[3]);
	      openGLText_drawChars(distStr, TEXT_NORMAL);
	    }

	  glPopMatrix();
	}
    }

  gluDeleteQuadric(obj);
}
static void drawPath(Path *path, Shade *shade, float min, float max)
{
  guint i;
  float xyz[3], rgba[4];

  g_return_if_fail(path);

/*   DBG_fprintf(stderr, "Geometry: draw path for node %d.\n", path->nodeId); */
  if (!shade)
    glColor3f(0.f, 0.f, 0.f);

  for (i = 0; i < path->nItems; i++)
    {
      if (path->items[i].type == PATH_ITEM_COORD)
	{
	  if (i > 0)
	    glEnd();
	  glBegin(GL_LINE_STRIP);
	  xyz[0] = path->items[i].dxyz[0] + path->translation[0];
	  xyz[1] = path->items[i].dxyz[1] + path->translation[1];
	  xyz[2] = path->items[i].dxyz[2] + path->translation[2];
	}
      else
	{
	  xyz[0] += path->items[i].dxyz[0];
	  xyz[1] += path->items[i].dxyz[1];
	  xyz[2] += path->items[i].dxyz[2];
	}
      if (shade)
	{
	  shadeGet_valueTransformedInRGB
	    (shade, rgba, CLAMP((path->items[i].energy - min) / (max - min), 0., 1.));
	  glColor3fv(rgba);
	}
      glVertex3fv(xyz);
    }
  glEnd();

  glEnable(GL_POINT_SMOOTH);
  glBegin(GL_POINTS);
  for (i = 0; i < path->nItems; i++)
    {
      if (path->items[i].type == PATH_ITEM_COORD)
	{
	  xyz[0] = path->items[i].dxyz[0] + path->translation[0];
	  xyz[1] = path->items[i].dxyz[1] + path->translation[1];
	  xyz[2] = path->items[i].dxyz[2] + path->translation[2];
	}
      else
	{
	  xyz[0] += path->items[i].dxyz[0];
	  xyz[1] += path->items[i].dxyz[1];
	  xyz[2] += path->items[i].dxyz[2];
	}
      if (shade)
	{
	  shadeGet_valueTransformedInRGB
	    (shade, rgba, CLAMP((path->items[i].energy - min) / (max - min), 0., 1.));
	  glColor3fv(rgba);
	}
      glVertex3fv(xyz);
    }
  glEnd();
  glDisable(GL_POINT_SMOOTH);
}
/**
 * visu_pathes_draw:
 * @pathes: a set of pathes.
 *
 * OpenGL calls to create the pathes.
 *
 * Since: 3.6
 */
void visu_pathes_draw(VisuPathes *pathes)
{
  GList *tmpLst;
  Shade *shade;
#define WIDTH 3.f

  if (ABS(pathes->maxE - pathes->minE) < 1e-6)
    shade = (Shade*)0;
  else
    shade = pathes->shade;

  glDisable(GL_LIGHTING);
  glDepthMask(0);
  glColor3f(0.f, 0.f, 0.f);
  glLineWidth(WIDTH);
  glPointSize(WIDTH);
  for (tmpLst = pathes->lst; tmpLst; tmpLst = g_list_next(tmpLst))
    drawPath((Path*)tmpLst->data, shade, pathes->minE, pathes->maxE);
  glDepthMask(1);
  glEnable(GL_LIGHTING);
}

/**
 * visu_geodiff_rebuildList:
 * @dataObj: a #VisuData containing a geometry diff.
 *
 * Create the OpenGL list holding the representation of a geometry
 * difference.
 *
 * Since: 3.5
 */
void visu_geodiff_rebuildList(VisuData *dataObj)
{
  DBG_fprintf(stderr, "Geometry: rebuilding diff for visuData %p.\n",
	      (gpointer)dataObj);

  if (!extGeo)
    geometryCreate_extGeo();

  glNewList(extGeo->objectListId, GL_COMPILE);
  geometryDraw(dataObj);
  glEndList();
}

static void geometryCreate_extGeo()
{
  int id;

  /* Get an OpenGL identifier to store the arrows and labels. */
  id     = visu_openGL_objectList_new(1);
  extGeo = visu_extension_new("geometry", "geometry",
			       "Different geometry representations",
			       id, visu_geodiff_rebuildList);
  visu_extension_setPriority(extGeo, VISU_EXTENSION_PRIORITY_NODE_DECORATIONS);
  extGeo->used = 0;
  visuExtensions_add(extGeo);
}

/**
 * visu_geodiff_setActive:
 * @value: a boolean.
 *
 * Visualise or not a geometry difference that has been previously
 * obtained using visu_geodiff_new(). The representation is done via
 * arrows positioned on nodes.
 *
 * Since: 3.5
 *
 * Returns: TRUE if the status of the geometry difference changed.
 */
gboolean visu_geodiff_setActive(gboolean value)
{
  if (!extGeo)
    geometryCreate_extGeo();

  if (value == extGeo->used)
    return FALSE;

  extGeo->used = value;

  return TRUE;
}

/*****************/
/* Init routine. */
/*****************/
/**
 * geometryInit: (skip)
 *
 * Initialise the geometry routines. Should not be called except at
 * initialisation time.
 *
 * Since: 3.5
 */
void geometryInit()
{
  VisuConfigFileEntry *conf;

  DBG_fprintf(stderr, "Geometry: set the conf entries for this module.\n");
  conf = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_ARROW,
				 DESC_RESOURCE_ARROW,
				 1, readArrow);
  visu_configFile_entry_setVersion(conf, 3.5f);
  conf = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_RATIO_MIN,
				 DESC_RESOURCE_RATIO_MIN,
				 1, readRatioMin);
  visu_configFile_entry_setVersion(conf, 3.5f);
  conf = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_RATIO_STR,
				 DESC_RESOURCE_RATIO_STR,
				 1, readRatioStr);
  visu_configFile_entry_setVersion(conf, 3.5f);
  conf = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_MULT,
				 DESC_RESOURCE_MULT,
				 1, readMult);
  visu_configFile_entry_setVersion(conf, 3.5f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResources);
}

/*************************/
/* Resources management. */
/*************************/
static gboolean readArrow(gchar **lines, int nbLines, int position,
			  VisuData *dataObj _U_, GError **error)
{
  float vals[4];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, vals, 4, error))
    return FALSE;
  arrow[0] = vals[0];
  arrow[1] = vals[1];
  arrow[2] = vals[2];
  arrow[3] = vals[3];

  return TRUE;
}
static gboolean readRatioMin(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &val, 1, error))
    return FALSE;
  ratioMin = val;

  return TRUE;
}
static gboolean readRatioStr(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &val, 1, error))
    return FALSE;
  ratioStr = val;

  return TRUE;
}
static gboolean readMult(gchar **lines, int nbLines, int position,
			 VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &val, 1, error))
    return FALSE;
  mult = val;

  return TRUE;
}
static void exportResources(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_ARROW);
  g_string_append_printf(data, "%s:\n    %f %f %f %f\n", FLAG_RESOURCE_ARROW,
			 arrow[0], arrow[1], arrow[2], arrow[3]);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_MULT);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_MULT, mult);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RATIO_MIN);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_RATIO_MIN, ratioMin);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RATIO_STR);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_RATIO_STR, ratioStr);
}

/*****************************/
/* XML files for iso-values. */
/*****************************/

/*
<pathes translat="0.000000;0.000000;0.000000">
  <path nodeId="214" translat="0.000000;0.000000;0.000000">
    <item time="0" type="dot" coordinates="24.46;29.39;29.65" totalEnergy="-23195.20" />
    <item time="0" type="delta" coordinates="0.23;0.05;0.07" totalEnergy="-23195.20" />
  </path>
</pathes>
*/

/* Known elements. */
#define GEOMETRY_ELEMENT_PATHES  "pathes"
#define GEOMETRY_ELEMENT_PATH    "path"
#define GEOMETRY_ELEMENT_ITEM    "item"
/* Known attributes. */
#define GEOMETRY_ATTRIBUTE_TIME  "time"
#define GEOMETRY_ATTRIBUTE_TRANS "translat"
#define GEOMETRY_ATTRIBUTE_NODE  "nodeId"
#define GEOMETRY_ATTRIBUTE_TYPE  "type"
#define GEOMETRY_ATTRIBUTE_COORD "coordinates"
#define GEOMETRY_ATTRIBUTE_TOT_E "totalEnergy"

static guint timeShift;
static gboolean startVisuPathes;
static Path *currentPath;

/* This method is called for every element that is parsed.
   The user_data must be a GList of _pick_xml. When a 'surface'
   element, a new struct instance is created and prepend in the list.
   When 'hidden-by-planes' or other qualificative elements are
   found, the first surface of the list is modified accordingly. */
static void geometryXML_element(GMarkupParseContext *context _U_,
				const gchar         *element_name,
				const gchar        **attribute_names,
				const gchar        **attribute_values,
				gpointer             user_data,
				GError             **error)
{
  VisuPathes *pathes;
  int i, n;
  guint val, time, type;
  float t[3], en;
  gboolean ok;
  GList *tmpLst;

  g_return_if_fail(user_data);
  pathes = (VisuPathes*)user_data;

  DBG_fprintf(stderr, "Geometry: found '%s' element.\n", element_name);
  if (!g_ascii_strcasecmp(element_name, GEOMETRY_ELEMENT_PATHES))
    {
      /* Initialise the pathList. */
      if (startVisuPathes)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
		      _("DTD error: element '%s' should appear only once."),
		      GEOMETRY_ELEMENT_PATHES);
	  return;
	}
      startVisuPathes = TRUE;
      /* Parse the attributes. */
      for (i = 0; attribute_names[i]; i++)
	{
	  /* Possible translation. */
	  if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_TRANS))
	    {
	      n = sscanf(attribute_values[i], "%f;%f;%f", t, t + 1, t + 2);
	      if (n != 3)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      GEOMETRY_ATTRIBUTE_TRANS, attribute_values[i]);
		  return;
		}
	      pathes->translation[0] = t[0];
	      pathes->translation[1] = t[1];
	      pathes->translation[2] = t[2];
	    }
	}
    }
  else if (!g_ascii_strcasecmp(element_name, GEOMETRY_ELEMENT_PATH))
    {
      if (!startVisuPathes)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
		      _("DTD error: parent element '%s' of element '%s' is missing."),
		      GEOMETRY_ELEMENT_PATHES, GEOMETRY_ELEMENT_PATH);
	  return;
	}
      /* We parse the attributes. */
      val = 123456789;
      t[0] = 0.f;
      t[1] = 0.f;
      t[2] = 0.f;
      for (i = 0; attribute_names[i]; i++)
	{
	  ok = TRUE;
	  if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_NODE))
	    ok = (sscanf(attribute_values[i], "%u", &val) == 1);
	  else if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_TRANS))
	    ok = (sscanf(attribute_values[i], "%f;%f;%f", t, t + 1, t + 2) == 3);
	  if (!ok)
	    {
	      g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			  _("DTD error: attribute '%s' has an unknown value '%s'."),
			  GEOMETRY_ATTRIBUTE_NODE, attribute_values[i]);
	      return;
	    }
	}
      if (val == 123456789)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
		      _("DTD error: element '%s' have missing mandatory attributes."),
		      GEOMETRY_ELEMENT_PATH);
	  return;
	}
      for (tmpLst = pathes->lst; tmpLst && ((Path*)tmpLst->data)->nodeId != val;
	   tmpLst = g_list_next(tmpLst));
      if (!tmpLst)
	{
	  currentPath = newPath();
	  currentPath->nodeId = val;
	  currentPath->translation[0] = t[0];
	  currentPath->translation[1] = t[1];
	  currentPath->translation[2] = t[2];
	  pathes->lst = g_list_prepend(pathes->lst, (gpointer)currentPath);
	}
      else
	currentPath = (Path*)tmpLst->data;
    }
  else if (!g_ascii_strcasecmp(element_name, GEOMETRY_ELEMENT_ITEM))
    {
      if (!currentPath)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
		      _("DTD error: parent element '%s' of element '%s' is missing."),
		      GEOMETRY_ELEMENT_PATH, GEOMETRY_ELEMENT_ITEM);
	  return;
	}

      /* We parse the attributes. */
      type = 999;
      time = 123456789;
      t[0] = G_MAXFLOAT;
      en = G_MAXFLOAT;
      for (i = 0; attribute_names[i]; i++)
	{
	  ok = TRUE;
	  if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_TIME))
	    ok = (sscanf(attribute_values[i], "%u", &time) == 1);
	  else if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_TYPE))
	    {
	      if (!g_ascii_strcasecmp(attribute_values[i], "dot"))
		type = PATH_ITEM_COORD;
	      else if (!g_ascii_strcasecmp(attribute_values[i], "delta"))
		type = PATH_ITEM_DELTA;
	      ok = (type != 999);
	    }
	  else if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_COORD))
	    ok = (sscanf(attribute_values[i], "%f;%f;%f", t, t + 1, t + 2) == 3);
	  else if (!g_ascii_strcasecmp(attribute_names[i], GEOMETRY_ATTRIBUTE_TOT_E))
	    ok = (sscanf(attribute_values[i], "%f", &en) == 1);
	  if (!ok)
	    {
	      g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			  _("DTD error: attribute '%s' has an unknown value '%s'."),
			  GEOMETRY_ATTRIBUTE_NODE, attribute_values[i]);
	      return;
	    }
	}
      if (time == 123456789 || type == 999 || t[0] == G_MAXFLOAT)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
		      _("DTD error: element '%s' have missing mandatory attributes."),
		      GEOMETRY_ELEMENT_PATH);
	  return;
	}
      addPathItem(currentPath, time + timeShift, t, type, en);
      pathes->time = MAX(time + timeShift + 1, pathes->time);
      if (en != G_MAXFLOAT)
	{
	  pathes->minE = MIN(en, pathes->minE);
	  pathes->maxE = MAX(en, pathes->maxE);
	}
    }
  else if (startVisuPathes)
    {
      /* We silently ignore the element if pathList is unset, but
	 raise an error if pathList has been set. */
      g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
		  _("Unexpected element '%s'."), element_name);
    }
}

/**
 * visu_pathes_parseFromXML:
 * @filename: a location on disk.
 * @pathes: a #VisuPathes object.
 * @error: a pointer on an error.
 *
 * Read an XML containing a description of @pathes. @pathes is newly
 * created on success and should be freed with visu_pathes_free().
 *
 * Since: 3.6
 * 
 * Returns: TRUE on success.
 */
gboolean visu_pathes_parseFromXML(const gchar* filename, VisuPathes *pathes, GError **error)
{
  GMarkupParseContext* xmlContext;
  GMarkupParser parser;
  gboolean status;
  gchar *buffer;
  gsize size;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(pathes, FALSE);

  buffer = (gchar*)0;
  if (!g_file_get_contents(filename, &buffer, &size, error))
    return FALSE;

  /* Create context. */
  currentPath          = (Path*)0;
  timeShift            = pathes->time;
  parser.start_element = geometryXML_element;
  parser.end_element   = NULL;
  parser.text          = NULL;
  parser.passthrough   = NULL;
  parser.error         = NULL;
  xmlContext = g_markup_parse_context_new(&parser, 0, pathes, NULL);

  /* Parse data. */
  startVisuPathes = FALSE;
  status = g_markup_parse_context_parse(xmlContext, buffer, size, error);

  /* Free buffers. */
  g_markup_parse_context_free(xmlContext);
  g_free(buffer);

  if (!startVisuPathes)
    {
      *error = g_error_new(G_MARKUP_ERROR, G_MARKUP_ERROR_EMPTY,
			  _("No pathes found."));
      status = FALSE;
    }

  if (!status)
    return FALSE;

  return TRUE;
}
/**
 * visu_pathes_exportToXML:
 * @pathes: a #VisuPathes object.
 * @filename: a location on disk.
 * @error: a pointer on an error.
 *
 * Write an XML file with the description of the given @pathes.
 *
 * Since: 3.6
 *
 * Returns: TRUE if no error.
 */
gboolean visu_pathes_exportToXML(const VisuPathes *pathes,
				const gchar *filename, GError **error)
{
  GString *output;
  GList *tmpLst;
  Path *path;
  guint i;
  gboolean valid;

  output = g_string_new("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
			"\n"
			"<pathes");
  g_string_append_printf(output, " translat=\"%f;%f;%f\">\n", pathes->translation[0],
			 pathes->translation[1], pathes->translation[2]);

  for (tmpLst = pathes->lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      path = (Path*)tmpLst->data;
      g_string_append_printf(output, "  <path nodeId=\"%d\" translat=\"%f;%f;%f\">\n",
			     path->nodeId, path->translation[0],
			     path->translation[1], path->translation[2]);
      for (i = 0; i < path->nItems; i++)
	if (ABS(path->items[i].energy) != G_MAXFLOAT)
	  g_string_append_printf
	    (output, "    <item time=\"%d\" type=\"%s\" coordinates=\"%f;%f;%f\""
	     " totalEnergy=\"%f\" />\n", path->items[i].time,
	     (path->items[i].type == PATH_ITEM_COORD)?"dot":"delta",
	     path->items[i].dxyz[0], path->items[i].dxyz[1],
	     path->items[i].dxyz[2], path->items[i].energy);
	else
	  g_string_append_printf
	    (output, "    <item time=\"%d\" type=\"%s\""
	     " coordinates=\"%f;%f;%f\" />\n", path->items[i].time,
	     (path->items[i].type == PATH_ITEM_COORD)?"dot":"delta",
	     path->items[i].dxyz[0], path->items[i].dxyz[1],
	     path->items[i].dxyz[2]);
      g_string_append(output, "  </path>\n");
    }

  g_string_append(output, "</pathes>");

  valid = g_file_set_contents(filename, output->str, -1, error);
  g_string_free(output, TRUE);
  return valid;
}
