/*  fft.c
 *
 *  FFT functions of xdemorse application
 */

/*
 *  xdemorse: An application to decode Morse code signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "fft.h"
#include "shared.h"

/* Scaled-up, integer, sin/cos tables */
static int
  *isin = NULL,
  *icos = NULL;

/*------------------------------------------------------------------------*/

/* Ifft_Init()
 *
 * Initializes Ifft()
 */
  void
Ifft_Init( int fft_input_size, int fft_bin_size )
{
  int i;
  size_t mreq;
  double w, dw;

  /* Allocate fft buffers */
  mreq = (size_t)fft_input_size * sizeof(int);
  mem_alloc( (void *)&fft_in_r, mreq );
  mem_alloc( (void *)&isin, mreq );
  mem_alloc( (void *)&icos, mreq );

  mreq = (size_t)fft_bin_size * sizeof(int);
  mem_alloc( (void *)&fft_out_r, mreq );
  mem_alloc( (void *)&fft_out_i, mreq );

  /* Make sin/cos tables */
  dw = 2.0 * M_PI / (double)fft_input_size;
  for( i = 0; i < fft_input_size; i++ )
  {
	w = dw * (double)i;
	isin[i] = (int)(127.0 * sin(w) + 0.5);
	icos[i] = (int)(127.0 * cos(w) + 0.5);
  }

} /* Ifft_Init() */

/*------------------------------------------------------------------------*/

/* Ifft()
 *
 * Simple, integer-only, FFT function
 */
  void
Ifft( int fft_input_size, int fft_bin_size )
{
  int i, j, w;

  /* In-phase and quadrature summation */
  int sum_i, sum_q;
  
  /* Calculate output bins */
  for( i = 0; i < fft_bin_size; i++ )
  {
	sum_i = sum_q = w = 0;

	/* Summate input values */
	for( j = 0; j < fft_input_size; j++ )
	{
	  sum_i += fft_in_r[j] * isin[w];
	  sum_q += fft_in_r[j] * icos[w];
	  w += i;
	  if( w >= fft_input_size ) w -= fft_input_size;
	}

	/* Normalized summations to bins */
	fft_out_r[i] = sum_i/fft_input_size;
	fft_out_i[i] = sum_q/fft_input_size;

  } /* for( i = 0; i < k; i++ ) */

} /* Ifft() */

/*------------------------------------------------------------------------*/

