/* Copyright (c) 2011 Nanakos Chrysostomos <nanakos@wired-net.gr>
   Simple and lightweight Yubikey OTP-OATH/HOTP Validation Server

   yubiserver is placed under the GNU General Public License, version 2 or later.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <gcrypt.h>
#include <sqlite3.h>
#include <time.h>
#include <sys/timeb.h>
#include <getopt.h>
#include <math.h>
#include <mhash.h>

#include <ev.h>

#include <libconfig.h>
#include "yubiserver.h"

void yubilog(int type, const char *s1, const char *s2, int num)
{
	int fd ;
	char *logbuffer=malloc(BUFSIZE*2);

	switch (type) 
	{
		case ERROR: 
			(void)sprintf(logbuffer,"ERROR: %s:%s Errno=%d exiting pid=%d",s1, s2, errno,getpid()); 
			break;
		case WARNING: 
			(void)sprintf(logbuffer, "<HTML><BODY><H2>Yubikey Validation Server Error: %s %s</H2></BODY></HTML>\r\n", s1, s2);
			(void)write(num,logbuffer,strlen(logbuffer));
			(void)sprintf(logbuffer,"WARNING: %s:%s",s1, s2); 
			break;
		case LOG: 
			(void)sprintf(logbuffer," INFO: %s:%s:%d",s1, s2,num); 
			break;
	}	
	/* no checks here, nothing can be done a failure anyway */
	if((fd = open(yubiserver_log, O_CREAT| O_WRONLY | O_APPEND,0644)) >= 0) 
	{
		(void)write(fd,logbuffer,strlen(logbuffer)); 
		(void)write(fd,"\n",1);      
		(void)close(fd);
	}
	free(logbuffer);
	if(type == ERROR || type == WARNING) 
		exit(3);
}

void null_terminate(char *buffer)
{
	int i;
	for(i=4;i<BUFSIZE;i++) { /* null terminate after the second space to ignore extra stuff */
		if(buffer[i] == ' ') { /* string is "GET URL " */
			buffer[i] = 0;
			break;
		}
	}
}

/* Decode Modhex to Hex */
void modhex2hex(char *otp_n,char *otp)
{
	char *hex="0123456789abcdef";
	char *modhex="cbdefghijklnrtuv";
	int i;
	char *end;
	int pos;
	for(i=0;i<strlen(otp);i++)
	{
		end=index(modhex,otp[i]);
		if(end==NULL) goto out;
		pos=end-modhex;
		otp_n[i]=hex[pos];
	}
	return;
out:
	memcpy(otp_n,otp,strlen(otp));
	return;
}

/* Convert ASCII to Hexadecimal  for OATH*/
void oath_atoh(char *token,char *out)
{
	int temp,index;
	char buf[2]={' ','\0'};
	for(index=0;index<20;index++)
	{
		buf[0]=token[2*index];
		temp = 16*strtol(buf,NULL,16);
		buf[0]=token[2*index+1];
		temp += strtol(buf,NULL,16);
		out[index]=temp;
	}
}

/* Convert ASCII to Hexadecimal */
void atoh(char *token,char *out)
{
	int temp,index;
	char buf[2]={' ','\0'};
	for(index=0;index<16;index++)
	{
		buf[0]=token[2*index];
		temp = 16*strtol(buf,NULL,16);
		buf[0]=token[2*index+1];
		temp += strtol(buf,NULL,16);
		out[index]=temp;
	}
}

/*Convert Hexadecimal to ASCII*/
void htoa(char *token,char *out)
{
	char *hex="0123456789abcdef";
	int index;
        for (index = 0; index<16; index++)
        {
      	 	sprintf((out+2*index),"%c",hex[((unsigned char)token[index]/16)]);
	        sprintf((out+2*index+1),"%c",hex[((unsigned char)token[index]%16)]);
        }
        out[33]='\0';
}
	
/* Get Public ID from Encoded or Decoded OTP */
void get_publicid(char *otp,char *pubid)
{
	int index;
	for(index=0;index<12;index++)
		pubid[index]=otp[index];
}

/* Retrieve AES Key from Database */
int get_aeskey(char *otp,char *aeskey,char *private_id)
{
	sqlite3 *handle;
	sqlite3_stmt *stmt;
	int retval,rows=0;
	char * query = (char *)malloc((size_t)QUERY_SIZE);
	char * public_id = (char *)malloc((size_t)PRIVATE_ID_SIZE+1);
	/* Get public_id*/
	get_publicid(otp,public_id);
	public_id[PRIVATE_ID_SIZE]='\0';
	/* Create query for aeskey and private id */
	sprintf(query,"SELECT aeskey,internalname FROM yubikeys WHERE publicname='%.12s' AND active='1'",public_id);
	retval = sqlite3_open(sqlite3_dbpath,&handle);

	if(retval)
		yubilog(LOG,"Database connection failed",0,0);

	retval = sqlite3_prepare_v2(handle,query,-1,&stmt,0);

	if(retval)
		yubilog(LOG,"get_aeskey: Selecting data from DB failed",0,0);
	while(1)
	{
		retval = sqlite3_step(stmt);
		if(retval==SQLITE_ROW)
		{	
			sprintf(aeskey,"%s",sqlite3_column_text(stmt,0));
			sprintf(private_id,"%s",sqlite3_column_text(stmt,1));
			rows++;
		}
		else if(retval==SQLITE_DONE)
		{
			break;
		}
		else
		{
			yubilog(LOG,"Database error encountered",0,0);
			break;
		}
	}
	sqlite3_finalize(stmt);
	sqlite3_close(handle);	
	free(query);
	free(public_id);

	if(rows)
		return 0; /* If user found return zero value */
	else
		return -1; /* If the user does not belong to the DB or is deactivated return negative value */
}

/* Decrypt AES encrypted OTP */
char *aes128ecb_decrypt(char *otp,char *premod_otp,char *private_id)
{
	#define GCRY_CIPHER GCRY_CIPHER_AES128   // cipher
	#define GCRY_C_MODE GCRY_CIPHER_MODE_ECB // cipher mode
	gcry_error_t     gcryError;
    	gcry_cipher_hd_t gcryCipherHd;
	size_t           index;
	size_t		 otpLength=16;
    	size_t keyLength = gcry_cipher_get_algo_keylen(GCRY_CIPHER);
	char * decoded_otp = (char *)malloc((size_t)16);
	char * otp_token = (char *)malloc((size_t)16);
	char * otp_buffer = (char *)malloc((size_t)32);
	char * final_otp = (char *)malloc(33);
	char * aesSymKey = (char *)malloc(32);
	gcry_control (GCRYCTL_ANY_INITIALIZATION_P);
	//gcry_check_version ( NULL );
        gcry_control ( GCRYCTL_DISABLE_SECMEM_WARN );
        gcry_control ( GCRYCTL_INIT_SECMEM, 16384, 0 );
	gcry_control (GCRYCTL_INITIALIZATION_FINISHED, NULL,0);
	int retval = get_aeskey(premod_otp,aesSymKey,private_id);
	if(retval<0)
	{
		free(decoded_otp);
		free(otp_token);
		free(otp_buffer);
		free(final_otp);
		free(aesSymKey);
		return NULL;
	}

	char * aesKey = (char *)malloc((size_t)16);

	gcryError = gcry_cipher_open(
			&gcryCipherHd, // gcry_cipher_hd_t *
			GCRY_CIPHER,   // int
			GCRY_C_MODE,   // int
			0);            // unsigned int

	if(gcryError)
	{
		//yubilog(ERROR,"gcry_cipher_open failed",gcry_strsource(gcryError),gcry_strerror(gcryError));
		yubilog(LOG,"gcry_cipher_open failed",gcry_strsource(gcryError),0);
		free(decoded_otp);
		free(otp_token);
		free(otp_buffer);
		free(final_otp);
		free(aesSymKey);
		free(aesKey);
		return NULL;
	}
	atoh(aesSymKey,aesKey);
	gcryError = gcry_cipher_setkey(gcryCipherHd, aesKey, keyLength);

	if(gcryError)
	{
		//yubilog(ERROR,"gcry_cipher_setkey failed",gcry_strsource(gcryError),gcry_strerror(gcryError));
		yubilog(LOG,"gcry_cipher_setkey failed",gcry_strsource(gcryError),0);
		free(decoded_otp);
		free(otp_token);
		free(otp_buffer);
		free(final_otp);
		free(aesSymKey);
		free(aesKey);
		gcry_cipher_close(gcryCipherHd);
		return NULL;
	}

	memcpy(otp_buffer,otp+12,32);
/*	for(index=0;index<32;index++)
	{
		otp_buffer[index]=otp[index+12];
	}
*/
	atoh(otp_buffer,otp_token);
	gcryError = gcry_cipher_decrypt(
			gcryCipherHd, // gcry_cipher_hd_t
			decoded_otp,  // void *
			otpLength,    // size_t
			otp_token,    // const void *
			keyLength);   // size_t
	if(gcryError)
	{
		//yubilog(ERROR,"gcry_cipher_decrypt failed",gcry_strsource(gcryError),gcry_strerror(gcryError));
		yubilog(LOG,"gcry_cipher_decrypt failed",gcry_strsource(gcryError),0);
		free(decoded_otp);
		free(otp_token);
		free(otp_buffer);
		free(final_otp);
		free(aesSymKey);
		free(aesKey);
		gcry_cipher_close(gcryCipherHd);
		return NULL;
	}
	gcry_cipher_reset(gcryCipherHd);
	gcry_cipher_close(gcryCipherHd);
	htoa(decoded_otp,final_otp);
	free(decoded_otp);
	free(otp_token);
	free(otp_buffer);
	free(aesSymKey);
	free(aesKey);
	return final_otp;
}

/* Compute CRC16-IBM {ANSI X3.28, Modbus,USB, Bisync} (reversed polynomial representation (MSB-first code) = 0xA001) */
unsigned short crc16_ansi(unsigned char *token,unsigned int len)
{
	unsigned short crc = 0xffff;
	unsigned int j;
	unsigned char b,i;
	for(j=0;j<len;j++)
	{
		b=token[j];
		for(i=0;i<8;i++)
		{
			crc = ((b ^ (unsigned char)crc) & 1) ? ((crc >> 1) ^ 0x8408) : (crc >> 1);
			b >>=1;
		}
	}
	return crc;
}

static char encoding_table[] = {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
        'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
        'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
        'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
        'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
        'w', 'x', 'y', 'z', '0', '1', '2', '3',
        '4', '5', '6', '7', '8', '9', '+', '/'};

static int mod_table[] = {0, 2, 1};

char *base64_encode(const unsigned char *data,size_t input_length,size_t output_length) 
{
        int i,j;
        output_length = (size_t) (4.0 * ceil((double) input_length / 3.0));
        char *encoded_data = malloc(output_length);
        if (encoded_data == NULL) return NULL;
        for (i = 0, j = 0; i < input_length;) {
                uint32_t octet_a = i < input_length ? data[i++] : 0;
                uint32_t octet_b = i < input_length ? data[i++] : 0;
                uint32_t octet_c = i < input_length ? data[i++] : 0;
                uint32_t triple = (octet_a << 0x10) + (octet_b << 0x08) + octet_c;

                encoded_data[j++] = encoding_table[(triple >> 3 * 6) & 0x3F];
                encoded_data[j++] = encoding_table[(triple >> 2 * 6) & 0x3F];
                encoded_data[j++] = encoding_table[(triple >> 1 * 6) & 0x3F];
                encoded_data[j++] = encoding_table[(triple >> 0 * 6) & 0x3F];
        }

        for (i = 0; i < mod_table[input_length % 3]; i++)
                encoded_data[output_length - 1 - i] = '=';
	encoded_data[output_length]='\0';
        return encoded_data;
}


int calc_counter(char *otp)
{
	static char chrotpcounter[7];
        sprintf(chrotpcounter,"%2s",(otp+14));
        sprintf((chrotpcounter+2),"%2s",(otp+12));
        sprintf((chrotpcounter+4),"%2s",(otp+22));
	chrotpcounter[6]='\0';
	return (int)strtol(chrotpcounter,NULL,16);
}

int calc_timestamp(char *otp)
{
	static char chrotptimestamp[7];
        sprintf(chrotptimestamp,"%2s",(otp+20));
        sprintf((chrotptimestamp+2),"%2s",(otp+18));
        sprintf((chrotptimestamp+4),"%2s",(otp+16));
	chrotptimestamp[6]='\0';
	return (int)strtol(chrotptimestamp,NULL,16);
}

int calc_sessioncounter(char *otp)
{
	static char chrotpcounter[5];
        sprintf(chrotpcounter,"%2s",(otp+14));
        sprintf((chrotpcounter+2),"%2s",(otp+12));
	chrotpcounter[4]='\0';
	return (int)strtol(chrotpcounter,NULL,16);
}

int calc_sessiontokencounter(char *otp)
{
	static char chrotpcounter[3];
        sprintf(chrotpcounter,"%2s",(otp+22));
	chrotpcounter[2]='\0';
	return (int)strtol(chrotpcounter,NULL,16);
}

int sqlite3_countertimestamp(char *otp,int *db_counter,int *db_timestamp,struct Yubikey *yubikey)
{
	sqlite3 *handle;
        sqlite3_stmt *stmt;
        int retval,rows=0;
	char * query = (char *)malloc((size_t)QUERY_SIZE);
	char * public_id = (char *)malloc((size_t)PRIVATE_ID_SIZE+1);
        /* Get public_id*/
        get_publicid(otp,public_id);
	public_id[PRIVATE_ID_SIZE]='\0';
        /* Create query for counter and time */
        sprintf(query,"SELECT counter,time,created FROM yubikeys WHERE publicname='%.12s' AND active='1'",public_id);
	retval = sqlite3_open(sqlite3_dbpath,&handle);

	if(retval)
                yubilog(LOG,"Database connection failed",0,0);

        retval = sqlite3_prepare_v2(handle,query,-1,&stmt,0);

	if(retval)
   		yubilog(LOG,"sqlite3_countertimestamp: Selecting data from DB failed",0,0);

	while(1)
        {
                retval = sqlite3_step(stmt);
                if(retval==SQLITE_ROW)
                {
                        *db_counter = atoi((const char *)sqlite3_column_text(stmt,0));
                        *db_timestamp = atoi((const char *)sqlite3_column_text(stmt,1));
			sprintf(yubikey->creation_date,"%s",sqlite3_column_text(stmt,2));
			yubikey->creation_date[24]='\0';
                        rows++;
                }
                else if(retval==SQLITE_DONE)
                {
                        break;
                }
                else
                {
                        yubilog(LOG,"sqlite3_countertimestamp: Database error encountered",0,0);
			break;
                }
        }
	sqlite3_finalize(stmt);
        sqlite3_close(handle);
        free(query);
        free(public_id);
        
	if(rows)
                return 0; /* If user found return zero value */
        else
                return -1; /* If the user does not belong to the DB return negative value */
}

/* Update SQLITE3 counter and timestamp */
void sqlite3_updatecounter(char *otp,int counter,int timestamp)
{
	sqlite3 *handle;
        int retval;
        char * query = (char *)malloc((size_t)QUERY_SIZE);
        char * public_id = (char *)malloc((size_t)PRIVATE_ID_SIZE+1);
        /* Get public_id*/
        get_publicid(otp,public_id);
	public_id[PRIVATE_ID_SIZE]='\0';
        /* Update counter and time */
        sprintf(query,"UPDATE yubikeys SET counter='%d',time='%d' WHERE publicname='%.12s' AND active='1'",counter,timestamp,public_id);
        retval = sqlite3_open(sqlite3_dbpath,&handle);
        
	if(retval)
                yubilog(LOG,"sqlite3_updatecounter: Database connection failed",0,0);

        retval = sqlite3_exec(handle,query,0,0,0);
        
	if(retval != SQLITE_OK)
	{
                yubilog(LOG,"sqlite3_updatecounter:Updating counter/timestamp data to DB failed",0,0);
	}
        
	sqlite3_close(handle);
        
	free(query);
        free(public_id);
}

char *get_apikey(char *id)
{
	sqlite3 *handle;
	sqlite3_stmt *stmt;
	int retval,rows=0;
	char * query = (char *)malloc((size_t)QUERY_SIZE);
	char * secret = (char *)malloc((size_t)40);
	/* Create query for aeskey and private id */
	sprintf(query,"SELECT secret FROM apikeys WHERE id='%s'",id);
	retval = sqlite3_open(sqlite3_dbpath,&handle);
	
	if(retval)
		yubilog(LOG,"get_apikey: Database connection failed",0,0);
		
	retval = sqlite3_prepare_v2(handle,query,-1,&stmt,0);
	
	if(retval)
		yubilog(LOG,"get_apikey: Selecting data from DB failed",0,0);
	
	while(1)
	{
		retval = sqlite3_step(stmt);
		if(retval==SQLITE_ROW)
		{
			sprintf(secret,"%s",sqlite3_column_text(stmt,0));
			rows++;
		}
		else if(retval==SQLITE_DONE)
		{
			break;
		}
		else
		{
			yubilog(LOG,"get_apikey: Database error encountered",0,0);
			break;
		}
	}

	sqlite3_finalize(stmt);
	sqlite3_close(handle);
	free(query);

	if(rows)
		return secret; /* If user found return secret */
	else
	{
		free(secret);
		return NULL; /* If the user does not belong to the DB or is deactivated return NULL value */
	}
}

/* Validate OTP */
int validate_otp(char *otp,char *premod_otp,char *private_id,struct Yubikey *yubikey)
{
	unsigned short crc=0;
	unsigned char * bcrc = (unsigned char *)malloc((size_t)OTP_MSG_SIZE);
	int otp_counter,otp_timestamp;
	int db_counter,db_timestamp;
	int retval;

	if(strncmp(otp,private_id,PRIVATE_ID_SIZE)!=0)
	{	
		free(bcrc);
		return BAD_OTP;
	}

	/* Get CRC16 */
	atoh(otp,(char *)bcrc);
	crc=crc16_ansi(bcrc,CRC_BLOCK_SIZE);
	free(bcrc);

	if(crc!=CRC_OK)
		return BAD_OTP;

	/* Compute internal counter and timestamp */
	otp_counter=calc_counter(otp);
	otp_timestamp=calc_timestamp(otp);
	/* Fetch internal counter and timestamp from the database */
	retval = sqlite3_countertimestamp(premod_otp,&db_counter,&db_timestamp,yubikey);

	if(retval < 0)
	{
		yubikey->result = BAD_OTP;
		return BAD_OTP;
	}
	/* Do timestamp and internal counter checks */
	if(db_counter >= otp_counter)
	{
		yubikey->result = REPLAYED_OTP;
		return REPLAYED_OTP;
	}

	if( (db_timestamp >= otp_timestamp) && ((db_counter >> 8) == (otp_counter >> 8)) )
	{
		yubikey->result = DELAYED_OTP;
		return DELAYED_OTP;
	}

	yubikey->counter=otp_counter;
	yubikey->timestamp=otp_timestamp;
	yubikey->session_counter = calc_sessioncounter(otp);
	yubikey->session_token_counter = calc_sessiontokencounter(otp);
	yubikey->result = OK;

	return OK;
}

/* Create HMAC-SHA1-BASE64 Encoded string */ 
char *create_hmac(char *otp,char *status,char *datetime,char *id,char *nonce)
{
	int keylen,datalen;
	unsigned char mac[20];
	char *data = NULL;
	MHASH td;
	
	char *password = get_apikey(id);

	if(password == NULL)
	{
		 return NULL;
	}

	keylen = strlen(password);
	data = (char *)calloc(1,180);
	if(nonce!=NULL)
	{
		datalen = sprintf(data,"nonce=%s&otp=%s&sl=100&status=%s&t=%s",nonce,otp,status,datetime);
	}
	if(nonce==NULL)
	{
		datalen = sprintf(data,"otp=%s&sl=100&status=%s&t=%s",otp,status,datetime);
	}
	td = mhash_hmac_init(MHASH_SHA1,password,keylen,mhash_get_hash_pblock(MHASH_SHA1));
	mhash(td,data,datalen);
	mhash_hmac_deinit(td,mac);
	char *output = base64_encode(mac,20,20);
	free(data);
	free(password);
	return output;
}

char *hotp(char *key,long counter,int digits)
{
	MHASH td;
	int bin_code;
	unsigned char hmac_result[20];
	int offset;
	char *final_hotp = NULL;
	char Counter[8]={
		((long)counter >> 56) & 0xff,((long)counter >> 48) & 0xff,((long)counter >> 40) & 0xff,
		((long)counter >> 32) & 0xff,((long)counter >> 24) & 0xff,((long)counter >> 16) & 0xff,
		((long)counter >> 8) & 0xff,((long)counter >> 0) & 0xff};
	char *HOTP = malloc(20*sizeof(char));
	td = mhash_hmac_init(MHASH_SHA1,key,strlen(key),mhash_get_hash_pblock(MHASH_SHA1));
	mhash(td,Counter,8); /* Hashing length is fixed to 8, always */
	mhash_hmac_deinit(td,hmac_result);
	offset   =  hmac_result[19] & 0xf ;
	bin_code = (hmac_result[offset]  & 0x7f) << 24
		| (hmac_result[offset+1] & 0xff) << 16
		| (hmac_result[offset+2] & 0xff) <<  8
		| (hmac_result[offset+3] & 0xff) ;
	sprintf(HOTP,"%d",bin_code);
	final_hotp = calloc(1,digits+1);
	memcpy(final_hotp,HOTP+strlen(HOTP)-digits,digits); /* Digits is usually 6 */
	free(HOTP);
	return final_hotp;
//	return (HOTP+strlen(HOTP)-digits); /* Digits is usually 6 */
}


struct Yubikey *oath_counter_secret(char *id)
{
	sqlite3 *handle;
	sqlite3_stmt *stmt;
	int retval,rows=0;
	char * query = (char *)malloc((size_t)QUERY_SIZE);
	struct Yubikey *oyubikey = (struct Yubikey *)calloc(1,sizeof(struct Yubikey));
	
	sprintf(query,"SELECT counter,secret FROM oathtokens WHERE publicname='%.12s' AND active='1'",id);
	retval = sqlite3_open(sqlite3_dbpath,&handle);
	
	if(retval)
		yubilog(LOG,"oath_counter_secret: Database connection failed",0,0);
		
	retval = sqlite3_prepare_v2(handle,query,-1,&stmt,0);
	
	if(retval)
		yubilog(LOG,"oath_counter_secret: Selecting data from DB failed",0,0);
	
	while(1)
	{
		retval = sqlite3_step(stmt);
		if(retval==SQLITE_ROW)
		{
			oyubikey->counter = atoi((const char *)sqlite3_column_text(stmt,0));
			sprintf(oyubikey->private_id,"%s",sqlite3_column_text(stmt,1));
			rows++;
		}
		else if(retval==SQLITE_DONE)
		{
			break;
		}
		else
		{
			yubilog(LOG,"oath_counter_secret: Database error encountered",0,0);
			break;
		}
	}

	sqlite3_finalize(stmt);
	sqlite3_close(handle);
	free(query);

	if(rows)
		return oyubikey; /* If user found return secret */
	else
	{
		free(oyubikey);
		return NULL; /* If the user does not belong to the DB or is deactivated return NULL value */
	}
}

void sqlite3_oath_updatecounter(char *otp,int counter)
{
	sqlite3 *handle;
        int retval;
        char * query = (char *)malloc((size_t)QUERY_SIZE);
        char * public_id = (char *)malloc((size_t)PRIVATE_ID_SIZE+1);
        /* Get public_id*/
        get_publicid(otp,public_id);
	public_id[PRIVATE_ID_SIZE]='\0';
        /* Update counter */
        sprintf(query,"UPDATE oathtokens SET counter='%d' WHERE publicname='%.12s' AND active='1'",counter,public_id);
        retval = sqlite3_open(sqlite3_dbpath,&handle);
        
	if(retval)
                yubilog(LOG,"sqlite3_oath_updatecounter: Database connection failed",0,0);

        retval = sqlite3_exec(handle,query,0,0,0);
        
	if(retval)
                yubilog(LOG,"sqlite3_oath_updatecounter:Updating counter/timestamp data to DB failed",0,0);
	
	free(query);
        free(public_id);
        
	sqlite3_close(handle);
        
}

/* Validate HOTP/OATH */
int validate_hotp(char *id,char *otp,struct OATH_Tokens *tokens)
{

	char *tokenid = (char *)malloc((size_t)PRIVATE_ID_SIZE+1);
	char *hotp_val = NULL;
	char *Key = NULL;
	char *temp = NULL;
	int counter;
	get_publicid(otp,tokenid);
	tokenid[PRIVATE_ID_SIZE]='\0';
	struct Yubikey *oyubikey = oath_counter_secret(tokenid);

	if(oyubikey == NULL) 
	{
		free(tokenid);
		return BAD_OTP;
	}
	
	hotp_val = (char *)malloc(strlen(otp)-12+1);
	sprintf(hotp_val,"%s",otp+12);
	if(strlen(hotp_val) % 2 != 0) 
	{	
		free(tokenid);
		free(hotp_val);
		free(oyubikey);
		return BAD_OTP;
	}
	Key = (char *)malloc((size_t)21);
	oath_atoh(oyubikey->private_id,Key);
	Key[20]='\0';
	for(counter=1+oyubikey->counter ; counter < 256+oyubikey->counter; counter++)
	{
		temp=hotp(Key,counter,strlen(hotp_val)); 
		//if(!strcmp(hotp_val,hotp(Key,counter,strlen(hotp_val))))
		if(!strcmp(hotp_val,temp))
		{
			tokens->counter = counter;
			if(Key) free(Key);
			if(temp) free(temp);
			if(oyubikey) free(oyubikey);
			if(hotp_val) free(hotp_val);
			free(tokenid);
			return OK;
		}
		free(temp);
	}
	if(Key) free(Key);
	if(oyubikey) free(oyubikey);
	if(hotp_val) free(hotp_val);
	free(tokenid);
	return NO_AUTH;
}
/* Validate OTP-HOTP/OATH */
int validate_all(char *otp,char *premod_otp,char *private_id,char *id,struct Yubikey *yubikey,int method,char **addr)
{
	int retval=BAD_OTP;
	if(method == METHOD_OTP)
		retval = validate_otp(otp,premod_otp,private_id,yubikey);
	if(method == METHOD_OATH)
	{
		retval = validate_hotp(id,otp,container_of(addr,struct OATH_Tokens,otp));
	}
	return retval;
}

char *find_token(char *token)
{
	char *token1,*token2;
	token1 = strtok(token,"=");
	token2 = strtok(NULL,"\0");
	return token2;
}

struct Tokens *tokenize(char *buffer)
{
	struct Tokens  *tokens  = (struct Tokens  *)calloc(1,sizeof(struct Tokens));
	char *token1,*token2,*token[7];
	int j, i=0;
	tokens->id = NULL;
	tokens->otp = NULL;
	tokens->h = NULL;
	tokens->timestamp = NULL;
	tokens->nonce = NULL;
	tokens->sl = NULL;
	tokens->timeout = 0;
	
	token[0] = strtok(buffer," ");
	token1 = strtok(NULL," ");
	token2 = strtok(token1,"?");
	token[0] = strtok(NULL,"&");
	while(token[i]!= NULL)
	{
		i++;
		token[i] = strtok(NULL,"&");
	}
	
	for(j=0; j <= i-1; j++)
	{
		if(strstr(token[j],"id=")!= NULL) tokens->id = find_token(token[j]);
		if(strstr(token[j],"otp=")!= NULL) tokens->otp = find_token(token[j]);
		if(strstr(token[j],"h=")!= NULL) tokens->h = find_token(token[j]);
		if(strstr(token[j],"timestamp=")!= NULL) tokens->timestamp = find_token(token[j]);
		if(strstr(token[j],"nonce=")!= NULL) tokens->nonce = find_token(token[j]);
		if(strstr(token[j],"sl=")!= NULL) tokens->sl = find_token(token[j]);
		if(strstr(token[j],"timeout=")!= NULL) tokens->timeout = atoi(find_token(token[j]));
	}
	return tokens;
}

struct OATH_Tokens *oath_tokenize(char *buffer)
{
	struct OATH_Tokens *tokens = (struct OATH_Tokens *)calloc(1,sizeof(struct OATH_Tokens));
	char *token1,*token2,*token[7];
	int j,i=0;
	tokens->id = NULL;
	tokens->otp = NULL;

	token[0] = strtok(buffer," ");
	token1 = strtok(NULL," ");
	token2 = strtok(token1,"?");
	token[0] = strtok(NULL,"&");
	while(token[i]!= NULL)
	{
		i++;
		token[i] = strtok(NULL,"&");
	}
	
	for(j=0; j <= i-1; j++)
	{
		if(strstr(token[j],"otp=")!= NULL) tokens->otp = find_token(token[j]);
		if(strstr(token[j],"id=")!= NULL) tokens->id = find_token(token[j]);
	}

	return tokens;
}

/* Child authentication server process, exit on errors */
static void write_callback(struct ev_loop *loop, struct ev_io *w, int revents)
{
	long i, ret;
	char *fstr = NULL;
	char *otp = NULL;
	char *hmac = NULL;
	char *private_id = NULL;
	struct Tokens  *tokens = NULL;
	struct OATH_Tokens  *oath_tokens = NULL;
	static char validation_date[24];
	static char datetmp[20];
	time_t t;
	struct tm *tmp = NULL;
	struct timeb tp;
	char *buffer = (char *)calloc(1,BUFSIZE+1); /* zero filled */
	int result = BAD_OTP;
	int sessioncounter=0,sessionuse=0,session_ts=0;
	int sl = -1;
	char *status[12]={"OK","BAD_OTP","REPLAYED_OTP","DELAYED_OTP","NO_SUCH_CLIENT","BAD_SIGNATURE","MISSING_PARAMETER", 
		          "OPERATION_NOT_ALLOWED","BACKEND_ERROR","NOT_ENOUGH_ANSWERS","REPLAYED_REQUEST","NO_AUTH"
			 }; 
	struct Yubikey *yubikey = NULL;
	struct ev_client *cli= ((struct ev_client*) (((char*)w) - offsetof(struct ev_client,ev_write)));
	memcpy(buffer,cli->buffer,cli->ret);
	ret = cli->ret;
	
	//struct Yubikey *yubikey = (struct Yubikey *)calloc(1,sizeof(struct Yubikey));
	//struct Yubikey *yubikey = (struct Yubikey *)malloc(sizeof(struct Yubikey));	
	//yubikey->creation_date[0]='\0';
	
	if(ret == 0 || ret == -1) 	/* read failure then stop now */
	{
		if (revents & EV_WRITE)
			ev_io_stop(EV_A_ w);
		yubilog(LOG,"failed to read browser request",buffer,ret);
		goto read_error;
	}
	
	if(ret > 0 && ret < BUFSIZE)	/* return code is valid chars */
		buffer[ret]=0;		/* terminate the buffer */
	else buffer[0]=0;		/* twice again?? */

	for(i=0;i<ret;i++)	/* remove CF and LF characters */
		if(buffer[i] == '\r' || buffer[i] == '\n')
			buffer[i]='*';
	
	//yubilog(LOG,"Client request",buffer,cli->fd);

	/* Check here for Yubikey OTP request */
	if(cli->mode==EV_VAL_OTP)
	{
		if (revents & EV_WRITE)
		{
			fstr = (char *)malloc((size_t)BUFSIZE);
			private_id = (char *)malloc((size_t)PRIVATE_ID_SIZE);
			yubikey = (struct Yubikey *)calloc(1,sizeof(struct Yubikey));
			null_terminate(buffer);
			tokens = tokenize(buffer); /* Tokenize the buffer and retrieve main attributes */
			char *otp_n = NULL;
			/* tokens->otp --------->  Before modhex decoding OTP string */
			if(tokens->otp != NULL)
			{
				otp=(char *)malloc((size_t)OTP_TOKEN);
				modhex2hex(otp,tokens->otp);
				otp_n = aes128ecb_decrypt(otp,tokens->otp,private_id); /* otp is now decrypted if it is valid and if user exists */
				if(otp != NULL) free(otp);
			}
			else if(tokens->otp == NULL)
			{
				yubilog(LOG,"Only simple verification operations supported for OTP, OTP is NULL",0,cli->fd);
			}
			if(otp_n == NULL || strlen(tokens->otp) < OTP_TOKEN || strlen(tokens->otp) > OTP_TOKEN) 
				result = BAD_OTP;
			else
				result = validate_all(otp_n,tokens->otp,private_id,tokens->id,yubikey,METHOD_OTP,NULL); /*validate*/
			
			t = time(NULL);
			ftime(&tp);
			tmp = localtime(&t);
			strftime(datetmp,20,"%FT%T",tmp);
			sprintf(validation_date,"%s.%.3dZ",datetmp,tp.millitm);
			if(tokens->nonce != NULL)
				if(strlen(tokens->nonce) < 16 || strlen(tokens->nonce) > 40)
					tokens->nonce = NULL;
			
			if(tokens->id!=NULL)
			{
				hmac = create_hmac(tokens->otp,status[result],validation_date,tokens->id,tokens->nonce); /* Add nonce when computing hash */
				if(hmac==NULL)
					result=NO_SUCH_CLIENT;
			}
			if(tokens->timestamp!=NULL)
			{	
				if(atoi(tokens->timestamp) == 1)
				{
					sessioncounter = yubikey->session_counter;
					sessionuse = yubikey->session_token_counter;
					session_ts = 1;
				}
			}
			if(tokens->sl!=NULL)
			{
				sl = atoi(tokens->sl);
			}
			(void)sprintf(fstr,"HTTP/1.0 200 OK\r\nContent-Type: %s\r\n\r\n", "text/plain");
			(void)write(cli->fd,fstr,strlen(fstr));
			if(session_ts && result==OK)
			{
				(void)sprintf(fstr,"h=%s\r\nt=%s\r\notp=%s\r\nnonce=%s\r\nsl=%d\r\ntimestamp=%d\r\nsessioncounter=%d\r\nsessionuse=%d\r\nstatus=%s\r\n",
						hmac!=NULL ? hmac : " ",validation_date,tokens->otp,tokens->nonce!=NULL? tokens->nonce : " ",sl!=-1? sl:100,yubikey->timestamp,sessioncounter,sessionuse, status[result]);
			}
			else{
				(void)sprintf(fstr,"h=%s\r\nt=%s\r\notp=%s\r\nnonce=%s\r\nsl=%d\r\nstatus=%s\r\n",
						hmac!=NULL ? hmac : " ",validation_date,tokens->otp,tokens->nonce!=NULL? tokens->nonce : " ",sl!=-1? sl:100 ,status[result]);
			}
			(void)write(cli->fd,fstr,strlen(fstr));
			/* Update database internal counter and timestamp */
			/* We are doing it here because we don't want the user to experience a small response delay */
			if(result==OK) sqlite3_updatecounter(tokens->otp,yubikey->counter,yubikey->timestamp); 
			free(fstr);
			free(private_id);
			free(yubikey);
			if(hmac) free(hmac);
			if(otp_n) free(otp_n);
			free(tokens);
			ev_io_stop(EV_A_ w); /* Preserve SQLite3 counter update */
		}
	}else if(cli->mode==EV_VAL_OATH) /* Check here for OATH Yubikey request */
	{
		if (revents & EV_WRITE)
		{
			fstr = (char *)malloc((size_t)BUFSIZE);
	     		null_terminate(buffer);
	                oath_tokens = oath_tokenize(buffer); /* Tokenize the buffer and retrieve main attributes */
			t = time(NULL);
			ftime(&tp);
			tmp = localtime(&t);
			strftime(datetmp,20,"%FT%T",tmp);
			sprintf(validation_date,"%s.%.3dZ",datetmp,tp.millitm);
			if(oath_tokens->otp != NULL && ( strlen(oath_tokens->otp) == 18 || strlen(oath_tokens->otp) == 20  ) )
				result = validate_all(oath_tokens->otp,NULL,NULL,NULL,NULL,METHOD_OATH,&oath_tokens->otp);
			else
				result = 1;
			(void)sprintf(fstr,"HTTP/1.0 200 OK\r\nContent-Type: %s\r\n\r\n", "text/plain");
			(void)write(cli->fd,fstr,strlen(fstr));
			(void)sprintf(fstr,"h=%s\r\nt=%s\r\notp=%s\r\nstatus=%s\r\n",
						" ",validation_date,oath_tokens->otp,status[result]);
		
			(void)write(cli->fd,fstr,strlen(fstr));
			if(result==OK) sqlite3_oath_updatecounter(oath_tokens->otp,oath_tokens->counter);
			free(fstr);
			free(oath_tokens);
			ev_io_stop(EV_A_ w);
		}
	}else if(cli->mode=EV_DEFAULT_PAGE) /* Default HTTP/1.0 request */
	{
		if (revents & EV_WRITE)
		{
			(void)sprintf(buffer,"HTTP/1.0 200 OK\r\nContent-Type: %s\r\n\r\n", "text/html");
			(void)write(cli->fd,buffer,strlen(buffer));
			(void)sprintf(buffer,"<html>Yubico Yubikey:<br><form action='/wsapi/2.0/verify' method='GET'> \
			     <input type='text' name='otp'><br><input type='submit'></form><br>OATH/HOTP tokens:<br> \
			     <form action='/wsapi/2.0/oauthverify' method='GET'>\
			     <input type='text' name='otp'><br><input type='submit'></form></html>\r\n");
			(void)write(cli->fd,buffer,strlen(buffer));
			ev_io_stop(EV_A_ w);
		}
	}else if(cli->mode==EV_DEFAULT_PAGE) /* Default HTTP/1.1 request */
	{
		if (revents & EV_WRITE)
		{
			(void)sprintf(buffer,"HTTP/1.0 200 OK\r\nContent-Type: %s\r\n\r\n", "text/html");
			(void)write(cli->fd,buffer,strlen(buffer));
			(void)sprintf(buffer,"<html>Yubico Yubikey:<br><form action='/wsapi/2.0/verify' method='GET'> \
			     <input type='text' name='otp'><br><input type='submit'></form><br>OATH/HOTP tokens:<br> \
			     <form action='/wsapi/2.0/oauthverify' method='GET'> \
			     <input type='text' name='otp'><br><input type='submit'></form></html>\r\n");
	                (void)write(cli->fd,buffer,strlen(buffer));
			ev_io_stop(EV_A_ w);
		}
	}else
	{ /*Maybe we should redirect this back to the default request???*/
		if (revents & EV_WRITE)
		{
			//yubilog(WARNING,"Only simple verification operations supported for OTP/OATH",buffer,cli->fd);
			ev_io_stop(EV_A_ w);
		}
	}
read_error:
	close(cli->fd); /* Don't let client to drain */
	free(buffer);
	free(cli);
}

void usage()
{
	fprintf(stderr,
			"Simple and lightweight Yubikey OTP and HOTP/OATH validation server.\n"
		        "Version " VERSION ". Written and copyrights by Nanakos Chrysostomos.\n"
			"THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY! USE AT YOUR OWN RISK!\n"	
			"\nUsage: yubiserver [options] \n\n"
			"Options supported:\n"
			"   --version  or -V		Print version information\n"
			"   --help     or -h		Print this help screen\n"
			"   --database or -d		Use this SQLite3 database file\n"
			"   --port     or -p		Port to bind the server. Default port is 8000\n"
			"   --logfile  or -l		Use this as logfile. Default is '" YUBISERVER_LOG_PATH "'\n"
			"This version of yubiserver has been configured with '" SQLITE3_DB_PATH "' as its default\n"
			"SQLite3 database file.\n");
}

int setnonblock(int fd)
{
    int flags;
    flags = fcntl(fd, F_GETFL);
    if (flags < 0)
            return flags;
    flags |= O_NONBLOCK;
    if (fcntl(fd, F_SETFL, flags) < 0) 
            return -1;
    return 0;
}

static void read_callback(struct ev_loop *loop, struct ev_io *w, int revents)
{ 
	
	struct ev_client *cli= ((struct ev_client*) (((char*)w) - offsetof(struct ev_client,ev_read)));
	long r=0;
	char rbuff[BUFSIZE+1];
	if (revents & EV_READ){
		r=read(cli->fd,&rbuff,BUFSIZE);
		rbuff[r]='\0';
		cli->buffer = rbuff;
		cli->ret = r;
		if(!strncmp(rbuff,"GET / HTTP/1.0",14))
		{
			cli->mode = EV_DEFAULT_PAGE;
		}else if(!strncmp(rbuff,"GET / HTTP/1.1",14))
		{
			cli->mode = EV_DEFAULT_PAGE;
		}else if(!strncmp(rbuff,"GET /wsapi/2.0/verify",21))
	        {
			cli->mode = EV_VAL_OTP;
		}else if(!strncmp(rbuff,"GET /wsapi/2.0/oauthverify",26))
		{
			cli->mode = EV_VAL_OATH;
		}

	}
	ev_io_stop(EV_A_ w);
	ev_io_init(&cli->ev_write,write_callback,cli->fd,EV_WRITE);
	ev_io_start(loop,&cli->ev_write);
	
}

static void accept_callback(struct ev_loop *loop, struct ev_io *w, int revents)
{
	int client_fd;
	struct ev_client *client;
	struct sockaddr_in client_addr;
	socklen_t client_len = sizeof(client_addr);
	client_fd = accept(w->fd, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd == -1) {
                return;
        }
   	 
	client = calloc(1,sizeof(*client));
	client->fd=client_fd;
	if (setnonblock(client->fd) < 0)
                yubilog(LOG, "failed to set client socket to non-blocking",0,0);
	ev_io_init(&client->ev_read,read_callback,client->fd,EV_READ);
	ev_io_start(loop,&client->ev_read);
}

//#define EVFLAG_FORKCHECK 1

int main(int argc, char **argv)
{
	int i, port, listenfd ;
	char portStr[6];
	int reuseaddr_on = 1;
	static struct sockaddr_in serv_addr; /* static = initialised to zeros */
	
	/* EV */
	ev_io *ev_accept = (ev_io *)malloc(sizeof(ev_io));
	struct ev_loop *loop = ev_default_loop (0);

	struct option long_options[] = 
	{
		/*Takes no parameters*/
		{"version",0,0,'V'},
		{"help",0,0,'h'},

		/*Takes parameters*/
		{"port",1,0,'p'},
		{"database",1,0,'d'},
		{"logfile",1,0,'l'}
	};
	int option_index =0,c;
	/* Define default port */
	port = 8000;
	/* Check here for arguments and please write a usage/help message!*/
        while( (c=getopt_long(argc,argv,"Vhp:d:",long_options,&option_index)) != -1)
	{
		switch(c)
		{
			case 'V':
				 printf("yubiserver version " VERSION ". Copyright (C) 2011 Nanakos Chrysostomos.\n"
			 		"This program is free software; you can redistribute it and/or modify\n"
					"it under the terms of the GNU General Public License as published by\n"
					"the Free Software Foundation; either version 2 of the License, or\n"
					"(at your option) any later version.\n" );
				 exit(0);
			case 'h':
				 usage();
				 exit(0);
			case 'p':
				 port=atoi(optarg);
				 break;
			case 'd':
				 sqlite3_dbpath = strdup(optarg);
				 break;
			case 'l':
				 yubiserver_log = strdup(optarg);
				 break;
		}
	}
	fprintf(stderr,"Database file used: %s\n",sqlite3_dbpath);
	fprintf(stderr,"Logfile used: %s\n",yubiserver_log);
	fprintf(stderr,"Server starting at port: %d\n",port);

	/* Become daemon + unstopable and no zombies children (= no wait()) */
	if(fork() != 0)
		return 0; /* parent returns OK to shell */

	(void)signal(SIGCLD, SIG_IGN); /* ignore child death */
	(void)signal(SIGHUP, SIG_IGN); /* ignore terminal hangups */

	for(i=0;i<32;i++)
		(void)close(i);		/* close open files */
	
	(void)setpgrp();		/* break away from process group */

	/* setup the network socket */
	if((listenfd = socket(AF_INET, SOCK_STREAM,0)) <0)
		yubilog(ERROR, "system call","socket",0);

	if (setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &reuseaddr_on,sizeof(reuseaddr_on)) == -1)
            yubilog(ERROR, "setsockopt failed",NULL,0);

	sprintf(portStr,"%d",port);

	if(port < 1024 || port > 65534)
		yubilog(ERROR,"Invalid port number (try 1025->65534)",portStr,0);

	yubilog(LOG,"yubikey validation server starting",portStr,getpid());
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port = htons(port);

	if(bind(listenfd, (struct sockaddr *)&serv_addr,sizeof(serv_addr)) <0)
		yubilog(ERROR,"system call","bind",0);

	if( listen(listenfd,64) <0)
		yubilog(ERROR,"system call","listen",0);
	
	if (setnonblock(listenfd) < 0)
            yubilog(ERROR, "cannot set server listening socket to non-blocking",0,0);

	ev_io_init(ev_accept,accept_callback,listenfd,EV_READ);
	ev_io_start(loop,ev_accept);
	ev_loop_fork(loop);
	ev_loop (loop, 0);

	return 0;
}
