/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTTrack.cpp,v 1.8 1999/04/15 22:36:27 lecroy Exp $
//
// QTTrack:
//   The central point of control for a track in a QTFile.


// -------------------------------------
// Includes
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "QTFile.h"

#include "QTAtom.h"
#include "QTAtom_dref.h"
#include "QTAtom_elst.h"
#include "QTAtom_mdhd.h"
#include "QTAtom_tkhd.h"
#include "QTAtom_stco.h"
#include "QTAtom_stsc.h"
#include "QTAtom_stsd.h"
#include "QTAtom_stss.h"
#include "QTAtom_stsz.h"
#include "QTAtom_stts.h"

#include "QTTrack.h"



// -------------------------------------
// Macros
//
#define DEBUG_PRINT(s) if(fDebug) printf s
#define DEEP_DEBUG_PRINT(s) if(fDeepDebug) printf s



// -------------------------------------
// Constructors and destructors
//
QTTrack::QTTrack(QTFile * File, QTFile::AtomTOCEntry * Atom, bool Debug, bool DeepDebug)
	: fDebug(Debug), fDeepDebug(DeepDebug),
	  fFile(File),
	  fIsInitialized(false),
	  fTrackHeaderAtom(NULL),
	  fTrackName(NULL),
	  fMediaHeaderAtom(NULL),
	  fEditListAtom(NULL), fDataReferenceAtom(NULL),
	  fTimeToSampleAtom(NULL), fSampleToChunkAtom(NULL), fSampleDescriptionAtom(NULL),
	  fChunkOffsetAtom(NULL), fSampleSizeAtom(NULL), fSyncSampleAtom(NULL),
	  fFirstEditMediaTime(0)
{
	// Temporary vars
	QTFile::AtomTOCEntry	*tempTOCEntry;


	//
	// Make a copy of the TOC entry.
	memcpy(&fTOCEntry, Atom, sizeof(QTFile::AtomTOCEntry));

	//
	// Load in the track header atom for this track.
	if( !fFile->FindTOCEntry(":tkhd", &tempTOCEntry, &fTOCEntry) )
		return;
	
	fTrackHeaderAtom = NEW('tkhd', QTAtom_tkhd, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fTrackHeaderAtom == NULL )
		return;
	if( !fTrackHeaderAtom->Initialize() ) {
		delete fTrackHeaderAtom;
		fTrackHeaderAtom = NULL;
	}
}

QTTrack::~QTTrack(void)
{
	//
	// Free our variables
	if( fTrackHeaderAtom != NULL )
		delete fTrackHeaderAtom;
	if( fTrackName != NULL )
		delete[] fTrackName;

	if( fMediaHeaderAtom != NULL )
		delete fMediaHeaderAtom;

	if( fEditListAtom != NULL )
		delete fEditListAtom;
	if( fDataReferenceAtom != NULL )
		delete fDataReferenceAtom;

	if( fTimeToSampleAtom != NULL )
		delete fTimeToSampleAtom;
	if( fSampleToChunkAtom != NULL )
		delete fSampleToChunkAtom;
	if( fSampleDescriptionAtom != NULL )
		delete fSampleDescriptionAtom;
	if( fChunkOffsetAtom != NULL )
		delete fChunkOffsetAtom;
	if( fSampleSizeAtom != NULL )
		delete fSampleSizeAtom;
	if( fSyncSampleAtom != NULL )
		delete fSyncSampleAtom;
}



// -------------------------------------
// Initialization functions
//
QTTrack::ErrorCode QTTrack::Initialize(void)
{
	// Temporary vars
	QTFile::AtomTOCEntry	*tempTOCEntry;


	//
	// Don't initialize more than once.
	if( IsInitialized() )
		return errNoError;

	//
	// Make sure that we were able to read in our track header atom.
	if( fTrackHeaderAtom == NULL )
		return errInvalidQuickTimeFile;

	
	//
	// See if this track has a name and load it in.
	if( fFile->FindTOCEntry(":udta:name", &tempTOCEntry, &fTOCEntry) ) {
		fTrackName = NEW_ARRAY('Ttn ', char, tempTOCEntry->AtomDataLength + 1);
		if( fTrackName != NULL )
			fFile->Read(tempTOCEntry->AtomDataPos, fTrackName, tempTOCEntry->AtomDataLength);
	}


	//
	// Load in the media header atom for this track.
	if( !fFile->FindTOCEntry(":mdia:mdhd", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;
	
	fMediaHeaderAtom = NEW('mdhd', QTAtom_mdhd, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fMediaHeaderAtom == NULL )
		return errInternalError;
	if( !fMediaHeaderAtom->Initialize() )
		return errInvalidQuickTimeFile;

	
	//
	// Load in the edit list atom for this track.
	DEEP_DEBUG_PRINT(("Searching track #%lu 'elst' atom.\n", GetTrackID()));
	if( fFile->FindTOCEntry(":edts:elst", &tempTOCEntry, &fTOCEntry) ) {
		fEditListAtom = NEW('elst', QTAtom_elst, (fFile, tempTOCEntry, fDebug, fDeepDebug));
		if( fEditListAtom == NULL )
			return errInternalError;
		if( !fEditListAtom->Initialize() )
			return errInvalidQuickTimeFile;

		//
		// Compute the first edit's media time.
		fFirstEditMediaTime = (UInt32)((GetFirstEditMovieTime() / fFile->GetTimeScale()) * GetTimeScale());
	} else {
		fEditListAtom = NULL;
	}


	//
	// Load in the data reference atom for this track.
	if( !fFile->FindTOCEntry(":mdia:minf:dinf:dref", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;
	
	fDataReferenceAtom = NEW('dref', QTAtom_dref, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fDataReferenceAtom == NULL )
		return errInternalError;
	if( !fDataReferenceAtom->Initialize() )
		return errInvalidQuickTimeFile;

	
	//
	// Load in the sample table atoms.
	if( !fFile->FindTOCEntry(":mdia:minf:stbl:stts", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;

	fTimeToSampleAtom = NEW('stts', QTAtom_stts, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fTimeToSampleAtom == NULL )
		return errInternalError;
	if( !fTimeToSampleAtom->Initialize() )
		return errInvalidQuickTimeFile;


	if( !fFile->FindTOCEntry(":mdia:minf:stbl:stsc", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;

	fSampleToChunkAtom = NEW('stsc', QTAtom_stsc, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fSampleToChunkAtom == NULL )
		return errInternalError;
	if( !fSampleToChunkAtom->Initialize() )
		return errInvalidQuickTimeFile;


	if( !fFile->FindTOCEntry(":mdia:minf:stbl:stsd", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;

	fSampleDescriptionAtom = NEW('stsd', QTAtom_stsd, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fSampleDescriptionAtom == NULL )
		return errInternalError;
	if( !fSampleDescriptionAtom->Initialize() )
		return errInvalidQuickTimeFile;


	if( !fFile->FindTOCEntry(":mdia:minf:stbl:stco", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;

	fChunkOffsetAtom = NEW('stco', QTAtom_stco, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fChunkOffsetAtom == NULL )
		return errInternalError;
	if( !fChunkOffsetAtom->Initialize() )
		return errInvalidQuickTimeFile;


	if( !fFile->FindTOCEntry(":mdia:minf:stbl:stsz", &tempTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;

	fSampleSizeAtom = NEW('stsz', QTAtom_stsz, (fFile, tempTOCEntry, fDebug, fDeepDebug));
	if( fSampleSizeAtom == NULL )
		return errInternalError;
	if( !fSampleSizeAtom->Initialize() )
		return errInvalidQuickTimeFile;


	if( fFile->FindTOCEntry(":mdia:minf:stbl:stss", &tempTOCEntry, &fTOCEntry) ) {
		fSyncSampleAtom = NEW('stss', QTAtom_stss, (fFile, tempTOCEntry, fDebug, fDeepDebug));
		if( fSyncSampleAtom == NULL )
			return errInternalError;
		if( !fSyncSampleAtom->Initialize() )
			return errInvalidQuickTimeFile;
	} else {
		fSyncSampleAtom = NULL;
	}
	
	
	//
	// This track has been successfully initialiazed.
	fIsInitialized = true;
	return errNoError;
}



// -------------------------------------
// Sample functions
//
bool QTTrack::GetSampleInfo(UInt32 SampleNumber, UInt32 * const Length, UInt64 * const Offset, UInt32 * const SampleDescriptionIndex, QTAtom_stsc_SampleTableControlBlock * STCB)
{
	// Temporary vars
	UInt32		SampleLength;

	// General vars
	UInt32		ChunkNumber, SampleOffsetInChunk,
				ChunkOffset;
	
	
	//
	// Locate this sample, compute its offset, and get its size.
	if( !SampleNumberToChunkNumber(SampleNumber, &ChunkNumber, SampleDescriptionIndex, &SampleOffsetInChunk, STCB) )
		return false;
	if( !fChunkOffsetAtom->ChunkOffset(ChunkNumber, &ChunkOffset) )
		return false;

	if( !fSampleSizeAtom->SampleSize(SampleNumber, &SampleLength) )
		return false;
	if( Length != NULL )
		*Length = SampleLength;

	if( Offset != NULL ) {
		//
		// Walk through all of the samples previous to this one, adding up
		// their lengths to figure out what the offset from the start of
		// the chunk to this sample is.
		*Offset = ChunkOffset;
		for( UInt32 CurSample = (SampleNumber - SampleOffsetInChunk);
			 CurSample < SampleNumber;
			 CurSample++
		) {
			//
			// Get the length of this sample and add it to our offset.
			if( !fSampleSizeAtom->SampleSize(CurSample, &SampleLength) )
				return false;
			*Offset += SampleLength;
		}
	}
	
	//
	// The sample was successfully located.
	return true;
}

bool QTTrack::GetSampleMediaTime(UInt32 SampleNumber, UInt32 * const MediaTime, QTAtom_stts_SampleTableControlBlock * STCB)
{
	//
	// Get this sample's media time.
	if( !fTimeToSampleAtom->SampleNumberToMediaTime(SampleNumber, MediaTime, STCB) )
		return false;
	
	//
	// The sample's media time was successfully located.
	return true;
}

bool QTTrack::GetSample(UInt32 SampleNumber, char * Buffer, UInt32 * Length, QTFile_FileControlBlock * FCB, QTAtom_stsc_SampleTableControlBlock * STCB)
{
	// General vars
	UInt32		SampleDescriptionIndex;
	UInt64		SampleOffset;
	
	
	//
	// Get the location and size of this sample.
	if( !GetSampleInfo(SampleNumber, Length, &SampleOffset, &SampleDescriptionIndex, STCB) )
		return false;
	
	//
	// Read in the sample
	if( !fDataReferenceAtom->Read(SampleDescriptionIndex, SampleOffset, Buffer, *Length, FCB) )
		return false;
	
	//
	// The sample was successfully read in.
	return true;
}


bool QTTrack::GetSampleNumberFromMediaTime(UInt32 MediaTime, UInt32 * const SampleNumber, QTAtom_stts_SampleTableControlBlock * STCB)
{
	//
	// Get the sample at this media time.
	if( !fTimeToSampleAtom->MediaTimeToSampleNumber(MediaTime, SampleNumber, STCB) )
		return false;
	
	//
	// This sample was successfully located.
	return true;
}




// -------------------------------------
// Debugging functions
//
void QTTrack::DumpTrack(void)
{
	//
	// Dump this track's information.
	DEBUG_PRINT(("QTTrack::DumpTrack - Dumping track.\n"));
	DEBUG_PRINT(("QTTrack::DumpTrack - ..Track name: \"%s\".\n", fTrackName ? fTrackName : "<untitled>"));

	//
	// Dump the sub-atoms of this track.
	if( fTrackHeaderAtom != NULL )
		fTrackHeaderAtom->DumpAtom();

	if( fMediaHeaderAtom != NULL )
		fMediaHeaderAtom->DumpAtom();
	
	if( fDataReferenceAtom != NULL )
		fDataReferenceAtom->DumpAtom();
	
	if( fTimeToSampleAtom != NULL )
		fTimeToSampleAtom->DumpAtom();
	if( fSampleToChunkAtom != NULL )
		fSampleToChunkAtom->DumpAtom();
	if( fSampleDescriptionAtom != NULL )
		fSampleDescriptionAtom->DumpAtom();
	if( fChunkOffsetAtom != NULL )
		fChunkOffsetAtom->DumpAtom();
	if( fSampleSizeAtom != NULL )
		fSampleSizeAtom->DumpAtom();
}
