/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPPlayListModule.cpp

	Contains:	Implements web stats module

	$Log: RTSPPlayListModule.cpp,v $
	Revision 1.9  1999/06/30 21:22:14  lecroy
	fixed RTSPPlayModule (not cllsing a file) and disabled the RTSPPlayModule, updated v-num
	
	Revision 1.8  1999/06/15 23:35:09  serenyi
	Changed OSCharDeleter to OSCharArrayDeleter
	
	Revision 1.7  1999/05/28 22:15:20  lecroy
	web stats changes: "help" paramter, rewrote the code as a loop to help out individual field support
	
	Revision 1.6  1999/05/21 22:24:39  lecroy
	more RTSPPlayModule stuff
	changed IDForTag to GenerateIDForSignature
	added RTSPRequestInterface::GetTruncatedAbsoluteURL method
	
	Revision 1.5  1999/05/20 06:33:22  lecroy
	nuttin
	
	Revision 1.4  1999/05/20 06:30:41  lecroy
	so, RTSPPlayModule seems to work - still needs a bunch of work in terms of modes, etc.
	
	Revision 1.3  1999/05/19 03:31:06  lecroy
	dfkjhsljhdslkghdslgh
	
	Revision 1.1  1999/02/26 05:14:57  cl
	Added HTTP stats (RTSPPlayListModule)
	

	

*/

#ifndef __MW_
#include <unistd.h>		/* for getopt() et al */
#include <stdio.h>		/* for printf */
#include <stdlib.h>		/* for getloadavg & other useful stuff */
#include <sys/stat.h>
#include <fcntl.h>
#endif

#include "RTSPPlayListModule.h"
#include "RTPServerInterface.h"
#include "OS.h"
#include "StrPtrLen.h"
#include "StringParser.h"


//**************************************************

UInt32 RTSPPlayListModule::sPlayListIndexDictID = 0;
UInt32 RTSPPlayListModule::sPlayListDictID = 0;

PlayList* RTSPPlayListModule::GetPlayListFromPath(char* inFilePath)
{
	PlayList* playList  = NULL;
	PlayList* prevPlayList  = NULL;
	// Get the specified file's inode.
	struct stat st = {};
	if( ::stat(inFilePath, (struct stat *)&st) != 0 )
		return NULL;
	
	for (playList = fFirstPlayList; playList != NULL; playList = playList->GetNextPlayList())
	{
		if ( playList->GetFileID() == st.st_ino )
		{
			//move this play list to top of cache
			if (prevPlayList != NULL)	//if it's not already at the top
			{
				prevPlayList->SetNextPlayList(playList->GetNextPlayList());
				playList->SetNextPlayList(fFirstPlayList);
				fFirstPlayList  = playList;
			}
			
			return playList;
		}
		prevPlayList = playList;
	}
	
	playList = PlayList::NewPlayList(inFilePath);
	
	if (playList != NULL)
	{
		//insert this play list at the top of the cache
		playList->SetNextPlayList(fFirstPlayList);
		fFirstPlayList = playList;

		
		//and delete the last element (if cache is greater than kPlayListCacheSize)
		PlayList* deadPlayList = NULL;
		int i=0;
		for (i=0,deadPlayList = fFirstPlayList; 
			i<=kPlayListCacheSize && deadPlayList != NULL; 
			i++,deadPlayList = deadPlayList->GetNextPlayList()) 
		{
			prevPlayList = deadPlayList;
		}
		if (deadPlayList != NULL)
		{
			prevPlayList->SetNextPlayList(NULL);
			delete deadPlayList;
		}
	}
		
	return playList;
}

void  RTSPPlayListModule::SessionClosing(RTSPSessionInterface* session)
{
	UInt32 valLen = 0;
	PlayList* playList = NULL;
	
	session->GetValue(sPlayListDictID, &playList, sizeof(playList), &valLen);

}

RTSPProtocol::RTSPStatusCode RTSPPlayListModule::ProcessRequest(RTSPRequestInterface* request) 
{ 
	if (request->GetMethod() == RTSPProtocol::kDescribeMethod)
	{
		OSCharArrayDeleter filePath(request->GetFullPath(qtssFilePathParam));
		
		return this->DoRedirect(request, filePath);
	}

	return RTSPProtocol::kSuccessOK; 
}

bool  RTSPPlayListModule::Initialize()
{
	sPlayListIndexDictID = RTSPSessionInterface::GenerateDictionaryID('pidx');
	sPlayListDictID = RTSPSessionInterface::GenerateDictionaryID('list');

	
	return RTSPModule::Initialize();
}

RTSPProtocol::RTSPStatusCode RTSPPlayListModule::DoRedirect(RTSPRequestInterface* request, char* inURL)
{
	//TODO:
	//resolve recursively until we hit non-playlist URL - small optimization

	PlayList* playList = NULL;
	if ( (playList = this->GetPlayListFromPath(inURL)) != NULL )
	{
		request->GetSession()->SetValue(sPlayListDictID, &playList, sizeof(playList));
	
		UInt32 valLen = 0;
		int playIndex = 0;
		
		request->GetSession()->GetValue(sPlayListIndexDictID, &playIndex, sizeof(playIndex), &valLen);
		if ( valLen != 0 ) playIndex++;

		//generate and send a redirect
		char* redirectURL = playList->GetRedirectURL(&playIndex);
		
		request->GetSession()->SetValue(sPlayListIndexDictID, &playIndex, sizeof(playIndex));

		if ( redirectURL != NULL )
		{
			OSCharArrayDeleter urlDeleter(NULL);
			
			 //prepend proto, address, port# if missing
			if ( ::strncmp(redirectURL, "rtsp://", ::strlen("rtsp://")) != 0 )
			{
				StrPtrLen* truncPath = request->GetQTSSParameter(qtssTruncAbsoluteURLParam);
				
				char* fullURL = new char[truncPath->Len + ::strlen(redirectURL) + 2];
				urlDeleter.SetObject(fullURL);
				
				::strncat(fullURL, truncPath->Ptr, truncPath->Len);
				::strcat(fullURL, "/");
				::strcat(fullURL, redirectURL);
				
				redirectURL = fullURL;
			}
			
			request->SetStatus(RTSPProtocol::kRedirectSeeOther);
			
			request->AppendHeader(RTSPProtocol::kLocationHeader, &StrPtrLen(redirectURL));

			request->SendHeader();
			
			return RTSPProtocol::kRedirectSeeOther;
			
		}
	}
	
	return RTSPProtocol::kSuccessOK; 
}





PlayList::PlayList(int inFileNodeID) :
	fFileInode(inFileNodeID),
	fFirstPlayListElement(NULL),
	fNextPlayList(NULL)
{
}

char* PlayList::GetRedirectURL(int* inPlayIndex)
{
	srand(time(NULL));

	PlayListElement* element = NULL;

	int theIndex = (inPlayIndex != NULL ? *inPlayIndex : 0);

	if (theIndex == 0)
		theIndex = rand() % this->GetElementCount();

	int i = 0;
	for ( i = theIndex, element = fFirstPlayListElement; 
			i > 0 && element != NULL; 
			i-- ) 
		{element = element->GetNextPlayListElement();}

	if ( inPlayIndex != NULL )
		*inPlayIndex = theIndex;

	if (element == NULL)
		return NULL;
	else
		return element->GetURL();
}

bool PlayList::ParseFile(int inFileDesc)
{
	UInt8 sTabStopsMask[] = {
		0, 0, 0, 0, 0, 0, 0, 0, 0, 1, //0-9      //stop on a ' ', '\t','\n','\r'
		1, 0, 0, 1, 0, 0, 0, 0, 0, 0, //10-19  
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //20-29
		0, 0, 1, 0, 0, 0, 0, 0, 0, 0, //30-39   
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //40-49
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //50-59
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //60-69
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //70-79
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //80-89
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //90-99
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //100-109
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //110-119
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //120-129
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //130-139
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //140-149
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //150-159
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //160-169
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //170-179
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //180-189
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //190-199
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //200-209
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //210-219
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //220-229
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //230-239
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //240-249
		0, 0, 0, 0, 0, 0 			 //250-255
	};

	FILE* file = ::fdopen(inFileDesc, "r");
	
	if (file == NULL)
		return false;
	
//#playlist settings
//#
//mode=randomplay
//repeatmin=60
//
//#
//#url, weight, playtime, artist, song, album
//#
//donttalk.mov, .4, 3:12, The Beach Boys, Don't Talk, Pet Sounds


	int kbufLen = 2048;
	char buff[kbufLen];
	
	//make reasonably sure it's a play list file
	if ( ::fgets(buff, kbufLen, file) != NULL )
	{
		StringParser magicValue(&StrPtrLen(buff));
		StrPtrLen theMagicWord;
		magicValue.ConsumeWord(&theMagicWord);
		if ( !theMagicWord.Equal(StrPtrLen("-StreamingPlayList-")) )
		{
			fclose(file);
			return false;
		}
	}
	else
	{
		fclose(file); 
		return false;
	}

	//most likely a play list file - build up a play list
	while ( ::fgets(buff, kbufLen, file) != NULL )
	{
		StrPtrLen	url;
		int	weight;
		StrPtrLen	playTime;
		StrPtrLen	artist;
		StrPtrLen	title;
		StrPtrLen	album;
		
		StringParser parser(&StrPtrLen(buff));
		parser.ConsumeUntil(&url, (UInt8*)sTabStopsMask);
		
		if (url.Len != 0)
		{
			weight = (int)parser.ConsumeInteger(NULL);
			parser.ConsumeUntil(&playTime, (UInt8*)sTabStopsMask);
			parser.ConsumeUntil(&artist, (UInt8*)sTabStopsMask);
			parser.ConsumeUntil(&title, (UInt8*)sTabStopsMask);
			parser.ConsumeUntil(&album, (UInt8*)sTabStopsMask);
	
			char* urlString = new char[url.Len+1];
			if ( urlString != NULL )
			{
				::memcpy(urlString, url.Ptr, url.Len);
				urlString[url.Len] = '\0';
			}
			
			PlayListElement* playListElement = new PlayListElement(urlString, weight);
			
			if (playListElement != NULL)
			{
				//insert it at the top of the list
				playListElement->SetNextPlayListElement(fFirstPlayListElement);
				fFirstPlayListElement = playListElement;
			}
		}
	}
	

	::fclose(file);
	
	return true;
}


int PlayList::GetElementCount()
{
	int count = 0;

	for ( PlayListElement* element = fFirstPlayListElement; element != NULL; element = element->GetNextPlayListElement() ) 
		count++;
		
	return count;
}


PlayListElement::PlayListElement(char* inURL, int inWeight) :
	fURL(inURL),
	fWeight(inWeight),
	fLastAccessTime(0),
	fNextPlayListElement(NULL)
{}


PlayListElement::~PlayListElement()
{
	delete fURL;
}



PlayList::~PlayList()
{
	for (PlayListElement* element = fFirstPlayListElement; element != NULL; )
	{
		PlayListElement* victim = element;
		element = victim->GetNextPlayListElement();
		delete victim;
	}

	fFirstPlayListElement = NULL;

}

PlayList* PlayList::NewPlayList(const char* inFilePath)
{
	int fileDesc = -1;
	
#if __MacOSX__
	fileDesc = ::open(inFilePath, O_RDONLY | O_NO_MFS, 0);
#else
	fileDesc = ::open(inFilePath, O_RDONLY, 0);
#endif
	if (fileDesc == -1)
		return NULL;
	
	struct stat fileInfo = {};
	if ( ::fstat(fileDesc, &fileInfo) == -1 )
	{
		::close(fileDesc);
		return NULL;
	}

	PlayList* playList = new PlayList(fileInfo.st_ino);
	
	//if it can't be parsed kill it
	if (!playList->ParseFile(fileDesc))
	{
		delete playList;
		playList = NULL;
	}
	
	::close(fileDesc);
	
	return playList;
}

