/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPRequestInterface.h

	Contains:	Provides a simple API for modules to access request information and
				manipulate (and possibly send) the client response.
	
	$Log: RTSPRequestInterface.h,v $
	Revision 1.2  1999/02/19 23:08:33  ds
	Created
	

*/


#ifndef __RTSPREQUESTINTERFACE_H__
#define __RTSPREQUESTINTERFACE_H__

//INCLUDES:
#include "QTSS.h"
#include "StrPtrLen.h"
#include "StringFormatter.h"
#include "RTSPSessionInterface.h"
#include "RTSPProtocol.h"
#include "RTSPMessages.h"


class RTSPResponseFormatter : public StringFormatter
{
	private:
	
		//This class is used in the implementation of RTSPRequestInterface, as a
		//resizeable output buffer. This way, no matter how much data is included in
		//an RTSP response, it can be handled by the server, and clients can write
		//into it transparently by calling Write.
		
		RTSPResponseFormatter() : 	StringFormatter(fOutputBuf, kOutputBufferSizeInBytes) {}
		
		//If we've been forced to increase the buffer size, fStartPut WILL be a dynamically allocated
		//buffer, and it WON'T be equal to fOutputBuf (obviously).
		virtual ~RTSPResponseFormatter() { if (fStartPut != &fOutputBuf[0]) delete [] fStartPut; }
		
		virtual void	BufferIsFull(char* inBuffer, UInt32 inBufferLen);
		
		//Resets the buffer to be the minimum size, if there was anything in it at the time,
		//it clears it out
		void			ShrinkBuffer() { if (fStartPut != &fOutputBuf[0]) delete [] fStartPut; this->SetBuffer(fOutputBuf, kOutputBufferSizeInBytes, 0); }
		void 			SetBuffer(char* inBuffer, UInt32 inBufferLen, UInt32 currentOffset);
		
		
	private:
	
		enum
		{
			kOutputBufferSizeInBytes = 512	//UInt32
		};
		
		//The default buffer size is allocated inline as part of the object. Because this size
		//is good enough for 99.9% of all requests, we avoid the dynamic memory allocation in most
		//cases. But if the response is too big for this buffer, the BufferIsFull function will
		//allocate a larger buffer.
		char					fOutputBuf[kOutputBufferSizeInBytes];
		
		friend class RTSPRequestInterface;
};


class RTSPRequestInterface
{
public:

	//Initialize
	//Call initialize before instantiating this class. For maximum performance, this class builds
	//any response header it can at startup time.
	static void 		Initialize();

	//CONSTRUCTOR:
	RTSPRequestInterface(RTSPRequestStream *stream, RTSPSessionInterface *session);
	virtual ~RTSPRequestInterface() {}
	
	//FUNCTIONS FOR SENDING OUTPUT:
	
	//Adds a new header to this object's list of headers to be sent out.
	//Note that this is only needed for "special purpose" headers. The Server,
	//CSeq, SessionID, and Connection headers are taken care of automatically
	void	AppendHeader(RTSPProtocol::RTSPHeader inHeader, StrPtrLen* inValue);

	//These functions format headers with complex formats
	void 	AppendTransportHeader(StrPtrLen* serverIPAddr,
									StrPtrLen* serverPortA,
									StrPtrLen* serverPortB);
	void 	AppendContentBaseHeader(StrPtrLen* theURL);
	void 	AppendRTPInfoHeader(RTSPProtocol::RTSPHeader inHeader,
								StrPtrLen* url, StrPtrLen* seqNumber,
								StrPtrLen* ssrc, StrPtrLen* rtpTime);
	void	AppendSessionHeader(StrPtrLen* inSessionID, StrPtrLen* inTimeout);
	void 	AppendContentLength(SInt32 inLength);

	//Modifying the response headers
	void SetStatus(RTSPProtocol::RTSPStatusCode newStatus) 	{ fStatus = newStatus; }
	void SetKeepAlive(bool newVal)							{ fResponseKeepAlive = newVal; }
	
	//This clears out the header data currently written, so you can start over.
	//Only works up until SendHeader actually gets called.
	void ResetHeaders();
	
	//SendHeader:
	//Sends the RTSP headers, in their current state, to the client.
	void SendHeader();
	
	//Send:
	//A slightly faster, simpler, & non-virtual Send function for compile-time
	//modules. THE FIRST ENTRY OF THE IOVEC MUST BE BLANK!!!
	QTSS_ErrorCode Send(iovec* inVec, UInt32 inNumVectors, UInt32 inTotalLength);
	
	//Write:
	//A "buffered send" that can be used for sending small chunks of data at a time.
	QTSS_ErrorCode Write(char* inBuffer, UInt32 inLength);
	
	//Close
	//Tells the server that the module is done processing this request.
	//This doesn't correspond to a TCP close. If the module doesn't call it, the
	//server will call it automatically when done with the request
	QTSS_ErrorCode 	Close();

	//ACCESS FUNCTIONS:
	
	//These should really be moved to QTSS API params.
	RTSPProtocol::RTSPMethod 	GetMethod() const	{ return fMethod; }
	RTSPProtocol::RTSPVersion 	GetVersion() const	{ return fVersion; }
	RTSPProtocol::RTSPStatusCode GetStatus() const 	{ return fStatus; }
	bool						GetRequestKeepAlive() const{ return fRequestKeepAlive; } 
	bool						GetResponseKeepAlive() const { return fResponseKeepAlive; }
	//will be -1 unless there was a Range header. May have one or two values
	Float64						GetStartTime()		{ return fStartTime; }
	Float64						GetStopTime()		{ return fStopTime; }
	UInt16						GetClientPortA()	{ return fClientPortA; }
	UInt16						GetClientPortB()	{ return fClientPortB; }
	bool						HasResponseBeenSent() { return fBytesSent > 0; }
	UInt32						GetBytesSent()		{ return fBytesSent; }
		
	RTSPSessionInterface*		GetSession()		 { return fSession; }
	
	//Principle functions for getting information about the request
	StrPtrLen*				GetQTSSParameter(QTSS_ParamKeywords whichParam);
	StrPtrLen*				GetHeaderValue(RTSPProtocol::RTSPHeader inHeader);
	
	//UTILITY FUNCTIONS:
	
	//This function constructs a C-string of the full path to the file being requested.
	//You may opt to append an optional suffix, or pass in NULL. You are responsible
	//for disposing this memory
	char*					GetFullPath(QTSS_ParamKeywords whichFileType, StrPtrLen* suffix = NULL);
	
	RTSPProtocol::RTSPStatusCode SendErrorResponse(RTSPProtocol::RTSPStatusCode inStatusCode,
													RTSPMessages::Message inMessageNum,
													StrPtrLen* inStringArg = NULL);
	
protected:

	//ALL THIS STUFF HERE IS SETUP BY RTSPREQUEST object (derived)
	
	//RESPONSE SENDING DATA
	bool							fResponseKeepAlive;
	RTSPProtocol::RTSPStatusCode 	fStatus;
	
	//REQUEST HEADER DATA
	enum
	{
		kMaxFilePathSizeInBytes = 256	//Uint32
	};
	
	RTSPProtocol::RTSPMethod	fMethod;
	RTSPProtocol::RTSPVersion 	fVersion;
	Float64						fStartTime;
	Float64						fStopTime;
	UInt16						fClientPortA;
	UInt16						fClientPortB;
	bool						fRequestKeepAlive;
	//Because of URL decoding issues, we need to make a copy of the file path.
	//Here is a buffer for it.
	char					fFilePath[kMaxFilePathSizeInBytes];
	//array of pointers to all the various header values
	//we can probably optimize this to reduce the 88 * 4 byte memory usage,
	//but let's not be overzealous in premature optimization
	StrPtrLen				fHeaders[RTSPProtocol::kIllegalHeader + 1];
	StrPtrLen				fQTSSConnParams[kQTSSNumRequestParameters];
	
	
private:

	RTSPSessionInterface*	fSession;
	RTSPRequestStream*		fStream;

	//Each request has an output buffer associated with it.
	//This is used for a number of things, such as writing headers
	RTSPResponseFormatter	fOutputFormatter;
	enum
	{
		kStaticHeaderSizeInBytes = 512	//UInt32
	};
	
	bool					fStandardHeadersWritten;
	UInt32					fBytesSent;
		
	void 					WriteStandardHeaders();
	static void 			PutStatusLine(	StringFormatter* putStream,
											RTSPProtocol::RTSPStatusCode status,
											RTSPProtocol::RTSPVersion version);

	//The server tends to supply parameters using lazy evaluation... ie, the value
	//of a given parameter is not computed until a module actually requests it. (only
	//in the case where computing a parameter requires some work). Here is a table
	//of function pointers. Each entry in the table is for one QTSS parameter  
	typedef StrPtrLen* (*ParamRetrievalFunction)(RTSPRequestInterface*);
	static ParamRetrievalFunction 	sParamRetrievalFunctions[];
	

	//Individual param retrieval functions
	static StrPtrLen* 	GetTruncatedPath(RTSPRequestInterface *req);
	static StrPtrLen* 	GetFileName(RTSPRequestInterface *req);
	static StrPtrLen* 	GetFileDigit(RTSPRequestInterface *req);
	static StrPtrLen* 	GetTruncatedAbsoluteURL(RTSPRequestInterface *req);


	//optimized preformatted response header strings
	static char				sPremadeHeader[kStaticHeaderSizeInBytes];
	static StrPtrLen		sPremadeHeaderPtr;
	static char				sServerHeader[kStaticHeaderSizeInBytes];
	static StrPtrLen		sServerHeaderPtr;
	static StrPtrLen		sColonSpace;
};
#endif // __RTSPREQUESTINTERFACE_H__

