import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.ConstructorDoc;
import com.sun.javadoc.Doc;
import com.sun.javadoc.FieldDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.ParamTag;
import com.sun.javadoc.Parameter;
import com.sun.javadoc.SeeTag;
import com.sun.javadoc.Tag;
import com.sun.javadoc.ThrowsTag;
import com.sun.javadoc.Type;

/**
 * Klasse, die die Javadoc-Informationen zusammenstellt und in die Zieldatei schreiben lsst. 
 * @author Jolle
 * @since 13.05.2008
 * @version 1.0
 *
 */
public class ClassWriter{

	private static final String JDclass = "jdclass";
	private static final String JDheader = "jdclassheader";
	private static final String JDmethod = "jdmethod";
	private static final String JDconstructor = "jdconstructor";
	private static final String JDfield = "jdfield";
	private static final String JDinterfaceOpt = "interface";
	private static final String JDclassOpt = "class";
	private static final String JDCpublic = "jdpublic ";	
	private static final String JDCprivate = "jdprivate ";
	private static final String JDCprotected = "jdprotected ";
	private static final String JDCfinal = "jdfinal ";
	private static final String JDCstatic = "jdstatic ";
	private static final String JDCtransient = "jdtransient ";
	private static final String JDCvolatile = "jdvolatile ";
	private static final String JDCabstract = "jdabstract ";
	private static final String JDCpackage = "jdpackage";
	private static final String JDCinherits = "jdinherits";
	private static final String JDCinhArrow = "jdinh ";
	private static final String JDCimplements = "jdimplements";
	private static final String JDCouterclass = "jdouterclass";
	private static final String JDCtype = "jdtype";
	private static final String JDcategory = "JDcategory";
	private static final String JDdeprecated = "JDdeprecated";
	private static final String JDsee = "JDsee";
	private static final String JDserial = "JDserial";
	private static final String JDserialData = "JDserialData";
	private static final String JDserialField = "JDserialField";
	private static final String JDsince = "JDsince";
	private static final String JDtext = "JDtext";
	private static final String JDversion = "JDversion";
	private static final String JDreturn = "JDreturn";
	private static final String JDauthor = "JDauthor";
	private static final String JDpara = "JDpara";
	private static final String JDthrows = "JDthrows";
	private static final String JDinhtable = "jdinheritancetable";

	private static final String JDClinksimple = "jdtypesimple";
	private static final String JDClinkarray = "jdtypearray";
	
	/**
	 * Stream, auf den die Texte geschrieben werden, und der diese in die Datei schreibt.
	 * @author Jolle
	 * @since version 1.0 vom 13.05.2008
	 */
	private TexPrintStream ps_dateiausgabe = null;
	
	/**
	 * Initialisiert den ClassWriter, indem aus dem bergebenen Pfad eine neue Textdatei erzeugt wird, 
	 * und ein Stream in die Datei gelegt wird
	 * @author Jolle
	 * @since version 1.0 vom 13.05.2008
	 * @param s_zielpath Pfad der neuen Textdatei
	 * @throws IOException Wenn die neue Datei nicht erstellt werden kann
	 * @throws FileNotFoundException Wenn nach dem Erstellen die Datei trotzdem nicht vorhanden ist
	 */
	public ClassWriter(String s_zielpath) throws IOException, FileNotFoundException{
		
		File f_ausgabedatei = new File(s_zielpath);
		f_ausgabedatei.createNewFile();
		
		if( !f_ausgabedatei.canWrite() ){
			throw new IOException("Auf die Datei "+ f_ausgabedatei.getAbsolutePath()+" kann nicht schreibend zugegriffen werden.");			
		}
				
		ps_dateiausgabe = new TexPrintStream( new File(s_zielpath) );
		
		// Todo: Zeile rauslschen fr Dateiausgabe 
		//ps_dateiausgabe = System.out;
	}
	
	/**
	 * Schreibt die Informationen zu einer Klasse
	 * @param cd Klassenobjekt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle 
	 */
	public void print( ClassDoc cd ){
		printBegin( JDclass );
		printOpt( cd.isClass()?JDclassOpt : JDinterfaceOpt );
		printArgument(cd.name());
			print();
			printBegin( JDheader );
				print();
				InhTable inhtable = printClassInfo(cd);				
				printTags(cd);								
			printEnd( JDheader );
			print();
			
			printInhTable(inhtable);
			print();
			for( FieldDoc fd : cd.fields()){
				print(fd);
			}
			for( ConstructorDoc constd : cd.constructors()){
				print(constd);
			}
			
			for( MethodDoc md : cd.methods()){
				print(md);
			}
		printEnd( JDclass );
		print();
		
		ps_dateiausgabe.flush();
		ps_dateiausgabe.close();
	}
	
	/**
	 * Schreibt die Informationen zu einem Feld
	 * @param fd Feld-objekt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle 
	 */
	private void print(FieldDoc fd ){
		printBegin( JDfield );
		printArgument( fd.name() );
		print();
		if( fd.isPrivate() ) 	printCommand( JDCprivate );
		if( fd.isPublic() ) 	printCommand( JDCpublic );
		if( fd.isProtected() )	printCommand( JDCprotected );
		if( fd.isFinal() )		printCommand( JDCfinal );
		if( fd.isStatic() ) 	printCommand( JDCstatic );
		if( fd.isTransient() )	printCommand( JDCtransient );
		if( fd.isVolatile() )	printCommand( JDCvolatile );
		print();
		
		printCommand( JDCtype );
		printLinks(fd.type());
		print();
		
		printTags(fd);
		
		printEnd( JDfield );
		print();
	}
	
	/**
	 * Schreibt die Informationen zu einem Konstruktor
	 * @param cd Konstruktorobjekt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle 
	 */
	private void print(ConstructorDoc cd){
		printBegin( JDconstructor );
		print();
		if( cd.isPrivate() ) 	printCommand( JDCprivate );
		if( cd.isPublic() ) 	printCommand( JDCpublic );
		if( cd.isProtected() )	printCommand( JDCprotected );
		if( cd.isFinal() )		printCommand( JDCfinal );
		if( cd.isStatic() ) 	printCommand( JDCstatic );
		print();
		
		for( Parameter p : cd.parameters() ){
			boolean tagfound = false;
			printCommand(JDpara);
			printLinks(p.type());
			printArgument( p.name());
			for( ParamTag par : cd.paramTags()  ){
				
				if( par.parameterName().equals( p.name() ) ){
					printArgument( par.parameterComment() );
					tagfound = true;
				}
			}
			if(! tagfound ) printArgument("");
			print();
		}
		
		for( Type excep : cd.thrownExceptionTypes() ){
			printCommand( JDthrows );
			printArgument( excep.typeName() );
			boolean tagfound = false;
			for( ThrowsTag tt : cd.throwsTags()  ){
				if( excep.equals(tt.exceptionType()) ){
					printArgument( tt.exceptionComment() );
					tagfound = true;
				}
			}
			if( !tagfound ) printArgument("");
			print();
		}
		
		printTags(cd);
		
		printEnd( JDconstructor );
		print();
	}
	
	/**
	 * Schreibt die Informationen zu einer Methode
	 * @param md Methodenobjekt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle 
	 */	
	private void print(MethodDoc md){
		printBegin( JDmethod );
		printArgument( md.name() );
		print();
		if( md.isPrivate() ) 	printCommand( JDCprivate );
		if( md.isPublic() ) 	printCommand( JDCpublic );
		if( md.isProtected() )	printCommand( JDCprotected );
		if( md.isFinal() )		printCommand( JDCfinal );
		if( md.isStatic() ) 	printCommand( JDCstatic );
		if( md.isAbstract() )	printCommand( JDCabstract );
		print();
		
		printCommand( JDCtype );
		printLinks( md.returnType() );
		print();
		
		for( Parameter p : md.parameters() ){
			printCommand(JDpara);
			printLinks(p.type());
			printArgument( p.name());
			boolean tagfound = false;
			for( ParamTag par : md.paramTags()  ){
				if( par.parameterName().equals( p.name() ) ){
					printArgument( par.parameterComment() );
					tagfound = true;
				}
				
			}
			if( !tagfound ) printArgument("");
			print();
		}
		
		for( Type excep : md.thrownExceptionTypes() ){
			printCommand( JDthrows );
			printArgument( excep.typeName() );
			boolean tagfound = false;
			for( ThrowsTag tt : md.throwsTags()  ){
				if( excep.equals(tt.exceptionType()) ){
					printArgument( tt.exceptionComment() );
					tagfound = true;
				}
			}
			if ( !tagfound ) printArgument("");
			print();
		}
				
		printTags(md);
		
		printEnd( JDmethod );	
		print();
	}
	
	/**
	 * Schreibt alle primitiven Javadoc-Tags
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 * @param d Doc-Typ mit allen Informationen ber das Element
	 */
	private void printTags(Doc d){
		
		if(d.commentText() != null && d.commentText().length() > 0 ){
			printCommand( JDtext );
			printArgument( d.commentText() );
			print();
		}
		
		for( Tag tag : d.tags()){
			if( tag.name().equals("@author") ){
				printCommand( JDauthor );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@category") ){
				printCommand( JDcategory );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@deprecated") ){
				printCommand( JDdeprecated );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@return") ){
				printCommand( JDreturn );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@serial") ){
				printCommand( JDserial );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@serialData") ){
				printCommand( JDserialData );
				printArgument( tag.text() );
				print();
			}
								
			if( tag.name().equals("@serialField") ){
				printCommand( JDserialField );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@since") ){
				printCommand( JDsince );
				printArgument( tag.text() );
				print();
			}
			
			if( tag.name().equals("@version") ){
				printCommand( JDversion );
				printArgument( tag.text() );
				print();
			}
		}
		
		for( SeeTag st : d.seeTags()){
			printCommand( JDsee );
			printArgument( st.text() );
			print();
		}
	}
	
	/**
	 * Schreibt die Informationen, die zu einer Klasse gehren
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 * @param cd Element mit den Informationen
	 * @return Beim Erzeugen der ClassInfo wird auch die InhTable erzeugt, die aber erst spter in den Ausgabestrom geschrieben werden darf. (Nmlich nach dem Header und vor Fields)
	 */
	private InhTable printClassInfo(ClassDoc cd){
		print();
		if( cd.isAbstract() )	printCommand( JDCabstract );
		if( cd.isPrivate() ) 	printCommand( JDCprivate );
		if( cd.isPublic() ) 	printCommand( JDCpublic );
		if( cd.isProtected() )	printCommand( JDCprotected );
		if( cd.isFinal() )		printCommand( JDCfinal );
		if( cd.isStatic() ) 	printCommand( JDCstatic );
		print();
		
		if( cd.containingClass() != null ){
			printCommand( JDCouterclass );
			printArgument( cd.containingClass().name() );
			print();
		}
		
		if( cd.containingPackage() != null && cd.containingPackage().name().length() > 0 ){
			printCommand( JDCpackage );
			printArgument( cd.containingPackage().name() );
			print();
		}
		
		for(ClassDoc interf : cd.interfaces() ){
			printCommand( JDCimplements );
			 printArgument( interf.name() );
			 print();
		}

		InhTable it = new InhTable();
		if( cd.superclass() != null ){
			ClassDoc cdrek = cd.superclass();
			it.addEntries(cdrek);
			
			String vererbung = getLinks(cdrek);
			while( cdrek.superclass() != null){
				cdrek = cdrek.superclass();
				it.addEntries(cdrek);
				vererbung = getLinks(cdrek) + "\\"+JDCinhArrow + vererbung;
			}
			printCommand( JDCinherits );
			ps_dateiausgabe.print("{");
			ps_dateiausgabe.print(vererbung);
			ps_dateiausgabe.print("}");
			print();
		}
		return it;
	}
	
	/*
	private void printComment(String comment){
		comment.replace("\n", "\n%");
		ps_dateiausgabe.println("%"+comment);
	}
	*/
	
	/**
	 * Gibt den Type (Array oder Simple) anhand des bergebenen Typs zurck 
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 * @param t Type-Objekt
	 * @return String mit dem entsprechenden texbefehl und den Argumenten
	 */
	public static String getLinks(Type t){
		StringBuffer sb = new StringBuffer();
		if( t.dimension().length() == 0 ){
			sb.append( "\\"+JDClinksimple +"{");
			sb.append( TexPrintStream.umwandlung(t.simpleTypeName()));
		}
		else{
			sb.append( "\\"+JDClinkarray +"{" );
			sb.append( TexPrintStream.umwandlung(t.simpleTypeName()));
			sb.append( "}{" );
			sb.append( TexPrintStream.umwandlung(t.dimension()));
		}
		sb.append("}");
		return sb.toString();
	}

	/**
	 * Schreibt den Typ eines Objektes
	 * @author Jolle
	 * @since version 1.0 vom 13.05.2008
	 * @param t Type-Objekt
	 */
	private void printLinks( Type t){
		ps_dateiausgabe.print("{");
		ps_dateiausgabe.print(getLinks(t));
		ps_dateiausgabe.print("}");
		
	}
	
	/**
	 * Schreibt die Umgebung mit der Tabelle der geerbten Elemente
	 * @param it das Objekt mit einer (noch unsortierten Tabelle)
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void printInhTable(InhTable it){
		printBegin(JDinhtable);
		it.sortTable();
		ps_dateiausgabe.print(it.getTexTableEntries());
		printEnd(JDinhtable );
	}
	
	/**
	 * Schreibt einen Zeilenumbruch
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void print(){
		ps_dateiausgabe.println();
	}
	
	/**
	 * Schreibt einen Tex-Befehl
	 * @param befehl Name des Befehls 
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void printCommand(String befehl){
		ps_dateiausgabe.print("\\"+befehl);
	}
	
	/**
	 * Schreibt eine Option zu einem Befehl
	 * @param option Optionsinhalt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void printOpt(String option){
		ps_dateiausgabe.print("[");
		ps_dateiausgabe.printTex(option);
		ps_dateiausgabe.print("]");	
	}
	
	/**
	 * Schreibt ein Argument zu einem Befehl
	 * @param arg Argumentinhalt
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void printArgument(String arg){
		ps_dateiausgabe.print("{");
		ps_dateiausgabe.printTex(arg);
		ps_dateiausgabe.print("}");		
	}
	
	/**
	 * Schreibt den Anfang einer Tex-Umgebung
	 * @param umgebung Name der Umgebung
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle
	 */
	private void printBegin(String umgebung){
		ps_dateiausgabe.print("\\begin{");
		ps_dateiausgabe.printTex(umgebung);
		ps_dateiausgabe.print("}");	
	}
	
	/**
	 * Schreibt das Ende einer Tex-Umgebung
	 * @param umgebung Name der Umgebung
	 * @since version 1.0 vom 13.05.2008
	 * @author Jolle	 */
	private void printEnd(String umgebung){
		ps_dateiausgabe.print("\\end{");
		ps_dateiausgabe.printTex(umgebung);
		ps_dateiausgabe.print("}");	

	}
}
