/* dvi.cpp -- read dvi files
   Time-stamp: "97/05/11 12:50:40 mik"

   Copyright (C) 1991, 92, 93, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.
   
   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Much of the code is borrowed from dvitype.web. */

#include <math.h>
#include <string.h>
#include <errno.h>

#include "dvi.h"
#include "common.h"
#include "pkfont.h"
#include "pkchar.h"

inline dvi_integer
xmax (dvi_integer a,
      dvi_integer b)

{
  return (a < b ? b : a);
}

const dvi_real pointsperinch = 72.27; // fixme

class _context

{
public:
  _context ();
  void save (const dvi &);
  void restore (dvi &) const;

private:
  int resolution;
  dvi_real conv;
  const char *mode;
 
  dvi_page **pages;
  dvi_integer totalpages;
  dvi_integer curpageidx;
  class _pkfont **fonts;
  dvi_integer totalfonts;
  dvi_integer maxhpixel;
  dvi_integer maxvpixel;
};

_context::_context ()

  : pages (0),
    fonts (0),
    totalfonts (0)

{
  ;
}

/* Save a DVI context. */
void
_context::save (const dvi &d)

{
  resolution = d.resolution;
  conv = d.conv;
  mode = d.mode;
  pages = d.pages;
  totalpages = d.totalpages;
  curpageidx = d.curpageidx;
  fonts = d.fonts;
  totalfonts = d.totalfonts;
  maxhpixel = d.maxhpixel;
  maxvpixel = d.maxvpixel;
}

/* Restore a DVI context. */
void
_context::restore (dvi &d) const

{
  d.resolution = resolution;
  d.conv = conv;
  d.mode = mode;
  d.pages = pages;
  d.totalpages = totalpages;
  d.curpageidx = curpageidx;
  d.fonts = fonts;
  d.totalfonts = totalfonts;
  d.maxhpixel = maxhpixel;
  d.maxvpixel = maxvpixel;
}

/* Switch to another DVI context. */
inline void
dvi::switchcontext (int c)

{
  contab[curcontext].save (*this);
  curcontext = c;
  contab[curcontext].restore (*this);
}

class _dvistate

{
public:
  dvi_integer h, v, w, x, y, z, hh, vv;
};

class _dvistatestack

{
public:
  _dvistatestack (dvi *, dvi_integer);
  ~_dvistatestack ();
  inline dvi_integer depth () const;
  void save ();
  void restore ();

private:
  dvi_integer stackdepth;
  dvi_integer maxdepth;
  dvi *d;
  _dvistate *stack;
};

inline dvi_integer
_dvistatestack::depth () const

{
  return (stackdepth);
}

_dvistatestack::~_dvistatestack ()

{
  if (stack)
    {
      delete [] stack;
      stack = 0;
    }
}

_dvistatestack::_dvistatestack (dvi *_d,
				dvi_integer _maxdepth)

  : d (_d),
    maxdepth (_maxdepth),
    stackdepth (0)

{
  stack = new _dvistate[ (size_t) maxdepth ];
}

void
_dvistatestack::save ()
     
{
  stack[stackdepth].h = d->h;
  stack[stackdepth].v = d->v;
  stack[stackdepth].w = d->w;
  stack[stackdepth].x = d->x;
  stack[stackdepth].y = d->y;
  stack[stackdepth].z = d->z;
  stack[stackdepth].hh = d->hh;
  stack[stackdepth].vv = d->vv;
  stackdepth++;
}

void
_dvistatestack::restore ()

{
  stackdepth--;
  d->h = stack[stackdepth].h;
  d->v = stack[stackdepth].v;
  d->w = stack[stackdepth].w;
  d->x = stack[stackdepth].x;
  d->y = stack[stackdepth].y;
  d->z = stack[stackdepth].z;
  d->hh = stack[stackdepth].hh;
  d->vv = stack[stackdepth].vv;
}

/* Round a DVI unit to the nearest pixel value. */
dvi_pixel
dvi::pixelround (dvi_integer du) const

{
#if 0
  if (du > 0)
    return ((dvi_integer) (conv * (real) du + 0.5));
  else
    return ((dvi_integer) (conv * (real) du + 0.5));
#else
  return ((dvi_integer) (floor (((du) * conv) + 0.5)));
#endif
}

/* Compute the number of pixels in the height or width of a rule. */
dvi_pixel
dvi::rulepixels (dvi_integer x) const

{
  dvi_pixel n;
  n = (dvi_pixel) (conv * x);
  if (n < (conv * x))
    return (n + 1);
  else
    return (n);
}

/* Open a DVI file and scan it. */
dvi::dvi (const char *_filename,
	  const char *mode1,
	  int resolution1,
	  const char *mode2,
	  int resolution2,
	  size_t _bytes_per_rword)

  : fonts (0),
    pages (0),
    dvifilename (strdup (_filename)),
    dvistream (0),
    curpageidx (-1),
    states (0),
    isok (dvi_false),
    bytesperrword (_bytes_per_rword)
     
{
  emptypage = new dvi_page ();
  dvistreamstat.st_mtime = 0;
  contab = new _context[2];
  maxhpixel = 0;
  maxvpixel = 0;
  mode = strdup (mode1);
  resolution = resolution1;
  contab[0].save (*this);
  contabsize = 1;
  curcontext = 0;
  if (mode2)
    {
      contabsize++;
      switchcontext (contabsize-1);
      maxhpixel = 0;
      maxvpixel = 0;
      mode = strdup (mode2);
      resolution = resolution2;
    }
  switchcontext (0);
  scan ();
}

void
dvi::freecontents ()

{
  dvi_integer oldcontext = curcontext;
  for (size_t i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      if (fonts)
	{
	  for (size_t i = 0; i < totalfonts; i++)
	    {
	      if (fonts[i])
		delete fonts[i];
	    }
	  delete [] fonts;
	  fonts = 0;
	  totalfonts = 0;
	}
      if (pages)
	{
	  for (size_t i = 0; i < totalpages; i++)
	    {
	      if (pages[i])
		delete pages[i];
	    }
	  delete [] pages;
	  pages = 0;
	  totalpages = 0;
	  curpageidx = -1;
	}
      maxhpixel = 0;
      maxvpixel = 0;
    }
  switchcontext (oldcontext);
  if (states)
    {
      delete states;
      states = 0;
    }
}

dvi::~dvi ()

{
  dprintf1 ("dvi::~dvi (%p)\n", this);
  freecontents ();
  if (dvifilename)
    {
      free ((void *) dvifilename);
      dvifilename = 0;
    }
  for (size_t i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      if (mode)
	{
	  free ((void *) mode);
	  mode = 0;
	}
    }
  if (dvistream)
    {
      delete dvistream;
      dvistream = 0;
    }
  if (emptypage)
    {
      delete emptypage;
      emptypage = 0;
    }
  if (contab)
    {
      delete [] contab;
      contab = 0;
    }
}

void
dvi::bad (dvi::dvierror e)

{
  ;				// FIXME
}

const int set_char = 0;
const int set1 = 128;
const int set2 = 129;
const int set3 = 130;
const int set4 = 131;
const int set_rule = 132;
const int put1 = 133;
const int put2 = 134;
const int put3 = 135;
const int put4 = 136;
const int put_rule = 137;
const int nop = 138;
const int bop = 139;
const int eop = 140;
const int p_ush = 141;
const int p_op = 142;
const int right1 = 143;
const int right2 = 144;
const int right3 = 145;
const int right4 = 146;
const int w0 = 147;
const int w1 = 148;
const int w2 = 149;
const int w3 = 150;
const int w4 = 151;
const int x0 = 152;
const int x1 = 153;
const int x2 = 154;
const int x3 = 155;
const int x4 = 156;
const int down1 = 157;
const int down2 = 158;
const int down3 = 159;
const int down4 = 160;
const int y_0 = 161;
const int y_1 = 162;
const int y2 = 163;
const int y3 = 164;
const int y4 = 165;
const int z0 = 166;
const int z1 = 167;
const int z2 = 168;
const int z3 = 169;
const int z4 = 170;
const int fnt_num = 171;
const int fnt1 = 235;
const int fnt2 = 236;
const int fnt3 = 237;
const int fnt4 = 238;
const int xxx1 = 239;
const int xxx2 = 240;
const int xxx3 = 241;
const int xxx4 = 242;
const int fnt_def1 = 243;
const int fnt_def2 = 244;
const int fnt_def3 = 245;
const int fnt_def4 = 246;
const int pre = 247;
const int post = 248;
const int post_post = 249;

#define undefined_commands 250: case 251: case 252: case 253: case 254: case 255
#define four_cases(x) x: case x+1: case x+2: case x+3

dvi_integer
dvi::firstparam (int opcode)

{
  if (opcode >= set_char && opcode <= set_char+127)
    return (opcode - set_char);
  switch (opcode)
    {
    case set1: case put1: case fnt1: case xxx1: case fnt_def1:
      return (dvistream->readbyte ());
    case set1+1: case put1+1: case fnt1+1: case xxx1+1: case fnt_def1+1:
      return (dvistream->readpair ());
    case set1+2: case put1+2: case fnt1+2: case xxx1+2: case fnt_def1+2:
      return (dvistream->readtrio ());
    case right1: case w1: case x1: case down1: case y_1: case z1:
      return (dvistream->readsignedbyte ());
    case right1+1: case w1+1: case x1+1: case down1+1: case y_1+1: case z1+1:
      return (dvistream->readsignedpair ());
    case right1+2: case w1+2: case x1+2: case down1+2: case y_1+2: case z1+2:
      return (dvistream->readsignedtrio ());
    case set1+3: case set_rule: case put1+3: case put_rule: case right1+3:
    case w1+3: case x1+3: case down1+3: case y_1+3: case z1+3: case fnt1+3:
    case xxx1+3: case fnt_def1+3:
      return (dvistream->readsignedquad ());
    case nop: case bop: case eop: case p_ush: case p_op: case pre: case post:
    case post_post: case undefined_commands:
      return (0);
    case w0:
      return (w);
    case x0:
      return (x);
    case y_0:
      return (y);
    case z0:
      return (z);
    }
  if (opcode >= fnt_num && opcode <= fnt_num+63)
    return (opcode - fnt_num);
  bad (fatalerror);
  return (-1);			// this can't happen
}

const int dvi_id = 2;
 
enum dvi_bool
dvi::scan ()

{
  int oldcontext = curcontext;

  dvi_integer firstbackpointer;	// pointer to last page

  // Prepare reading the dvi file
  if (_stat (dvifilename, &dvistreamstat) < 0)
    {
      bad (fileerror);
      return (dvi_false);
    }
  if (dvistream == 0)
    dvistream = new _ifwrdstream (dvifilename);
  if (! dvistream->is_open ())
    {
      bad (fileerror);
      return (dvi_false);
    }

  // Process the preamble
  if (dvistream->readbyte () != pre || dvistream->readbyte () != dvi_id)
    {
      bad (fileerror);
      return (dvi_false);
    }
  
  // Compute the conversion factor
  numerator = dvistream->readsignedquad ();
  denominator = dvistream->readsignedquad ();
  if (numerator <= 0 || denominator <= 0)
    {
      bad (dviformaterror);
      return (dvi_false);
    }
  dprintf2 ("numerator/denominator == %ld/%ld\n", numerator, denominator);
  mag = dvistream->readsignedquad ();
  dprintf1 ("mag == %ld\n", mag);

  for (size_t i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      conv = (((dvi_real) numerator * (dvi_real) mag * (dvi_real) resolution) /
	      ((dvi_real) denominator * 254000000.0));
      conv = (((dvi_real) numerator * (dvi_real) mag * (dvi_real) resolution) /
	      ((dvi_real) denominator * 254000000.0));
    }
  switchcontext (oldcontext);

  // Skip the introductory comment
  int len = dvistream->readbyte ();
  dvistream->skipg (len);
  
  // Find the postamble, working back from the end
  int k;
  dvistream->seekg (0, _ifwrdstream::end);
  dvi_integer filelength = dvistream->tellg ();
  dvi_integer m = filelength - 1;
  do
    {
      dvistream->seekg (m, _ifwrdstream::beg);
      k = dvistream->readbyte ();
      m--;
    }
  while (k == 223);
  if (k != dvi_id)
    {
      bad (dviformaterror);
      return (dvi_false);
    }
  dvistream->seekg (m - 3, _ifwrdstream::beg);
  dvi_integer q = dvistream->readsignedquad ();
  if (q < 0 || q > m - 33)
    {
      bad (dviformaterror);
      return (dvi_false);
    }
  dvistream->seekg (q, _ifwrdstream::beg);
  k = dvistream->readbyte ();
  if (k != post)
    {
      bad (dviformaterror);
      return (dvi_false);
    }
  firstbackpointer = dvistream->readsignedquad ();
    
  dvi_integer postamble_num = dvistream->readsignedquad ();
  dvi_integer postamble_den = dvistream->readsignedquad ();
  dvi_integer postamble_mag = dvistream->readsignedquad ();
  maxv = dvistream->readsignedquad ();
  maxh = dvistream->readsignedquad ();
  dprintf2 ("maxv ==%ld; maxh == %ld", maxv, maxh);
  dvi_integer maxs = dvistream->readpair ();
  dvi_integer numberofpages = dvistream->readpair ();

  dprintf2 ("; maxs == %ld; totalpages == %ld\n", maxs, numberofpages);
  states = new _dvistatestack (this, maxs);

  // Process the font definitions of the postamble
  do
    {
      k = dvistream->readbyte ();
      if (k >= fnt_def1 && k < fnt_def1 + 4)
	{
	  dvi_integer p = firstparam (k);
	  if (! installfont (p))
	    return (dvi_false);
	  k = nop;
	}
    }
  while (k == nop);
  if (k != post_post)
    {
      bad (dviformaterror);
      return (dvi_false);
    }
  
  // Build the page table
  dvi_integer backpointer = firstbackpointer;
  for (i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      totalpages = numberofpages;
      pages = new dvi_page * [totalpages];
      memset (pages, 0, sizeof (dvi_page *) * totalpages);
    }
  switchcontext (oldcontext);
  dvi_integer pageno = totalpages;
  while (backpointer >= 0 && pageno--)
    {
      dvistream->seekg (backpointer, _ifwrdstream::beg);
      if (dvistream->readbyte () != bop)
	{
	  bad (dviformaterror);
	  return (dvi_false);
	}
      dvi_integer count0 = dvistream->readsignedquad ();
      dvi_integer count1 = dvistream->readsignedquad ();
      dvi_integer count2 = dvistream->readsignedquad ();
      dvi_integer count3 = dvistream->readsignedquad ();
      dvi_integer count4 = dvistream->readsignedquad ();
      dvi_integer count5 = dvistream->readsignedquad ();
      dvi_integer count6 = dvistream->readsignedquad ();
      dvi_integer count7 = dvistream->readsignedquad ();
      dvi_integer count8 = dvistream->readsignedquad ();
      dvi_integer count9 = dvistream->readsignedquad ();
      backpointer = dvistream->readsignedquad ();
      for (i = 0; i < contabsize; i++)
	{
	  switchcontext (i);
	  dvi_page *p = new dvi_page (dvistream->tellg (),
				      count0, count1, count2, count3, count4,
				      count5, count6, count7, count8, count9);
	  pages[pageno] = p;
	}
      switchcontext (oldcontext);
    }

  delete dvistream;
  dvistream = 0;
  isok = dvi_true;
  return (isok);
}

enum dvi_bool
dvi::installfont (dvi_integer fontnum)

{
  dprintf1 ("dvi::installfont (%ld)\n", fontnum);
  dvi_integer checksum = dvistream->readsignedquad ();
  dvi_integer scaledsize = dvistream->readsignedquad ();
  dvi_integer designsize = dvistream->readsignedquad ();
  int areanamelen = dvistream->readbyte ();
  int fontnamelen = dvistream->readbyte ();
  dvistream->skipg (areanamelen); // ignore the area spec
  char fontname[260];
  dvistream->read (fontname, fontnamelen);
  fontname[fontnamelen] = 0;
  int oldcontext = curcontext;
  for (size_t i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      _pkfont *f = new _pkfont (fontname, mag, designsize, scaledsize, mode, resolution);
      if (fonts == 0)
	{
	  totalfonts = (fontnum < 128 ? 128 : fontnum + 10);
	  fonts = new _pkfont * [totalfonts];
	  memset (fonts, 0, sizeof (_pkfont *) * totalfonts);
	}
      if (fontnum >= totalfonts)
	{
	  dvi_integer newtotal = totalfonts * 2;
	  if (fontnum >= newtotal)
	    newtotal = fontnum + 10;
	  _pkfont **newfonts = new _pkfont * [newtotal];
	  memset (newfonts, 0, sizeof (_pkfont *) * newtotal);
	  memcpy (newfonts, fonts, sizeof (_pkfont *) totalfonts);
	  delete [] fonts;
	  fonts = newfonts;
	  totalfonts = newtotal;
	}
      fonts[fontnum] = f;
    }
  switchcontext (oldcontext);
  return (dvi_true);
}

const dvi_page &
dvi::operator[] (dvi_integer pageno)

{
  time_t oldmtime = dvistreamstat.st_mtime;
  if (_stat (dvifilename, &dvistreamstat) < 0)
    return (*emptypage);
  if (oldmtime != dvistreamstat.st_mtime)
    {
      freecontents ();
      if (! scan ())
	return (*emptypage);
    }
  dprintf1 ("dvi::operator[] (%ld)\n", pageno);
  curpageidx = pageno;
  dvi_page *page = pages[curpageidx];
  if (page)
    {
      if (! page->done ())
	dopage ();
      page->makebitmaps (bytesperrword);
      return (*page);
    }
  else
    return (*emptypage);
}

const int max_drift = 1;	// we insist that abs(hh-pixelRound(h))<=max_drift

enum dvi_bool
dvi::dopage ()

{
  dprintf ("dvi::dopage\n");
  curfontidx = 0;
  curfont = 0;
  curchar = 0;
  h = v = w = x = y = z = hh = vv = curfontidx = 0;
  dvi_page &page = *pages[curpageidx];
  if (dvistream == 0)
    dvistream = new _ifwrdstream (dvifilename);
  dvistream->seekg (page.fileoffset (), _ifwrdstream::beg);
  while (dvistream->good () && ! dvistream->eof ())
    {
      /* Translate the next command in the dvi file; goto L9999 with
	 result==true if it was eop; goto L9998 if premature
	 termination is needed */
      int opcode = dvistream->readbyte (); // operation code of the current command
      dvi_integer p, q;		// parameters of the current command
      p = firstparam (opcode);
      dprintf2 ("o=%d; p=%ld\n", opcode, (long) p);
      
      /* Start translation of command opcode and goto the appropriate
	 label to finish the job */
      if (opcode <= set4 || opcode >= put1 && opcode <= put4)
	{
	  /* Translate a char command */
	  if (curfont == 0)
	    {
	      bad (dviformaterror);
	      return (dvi_false);
	    }
	  curchar = (*curfont)[p];
	  if (curchar == 0)
	    {
	      bad (fatalerror);
	      goto L9998;
	    }
	  dvi_item item;
	  item.x = (hh - curchar->hoffset ());
	  item.y = (vv + (curchar->vpixels () - curchar->voffset () - 1));
	  item.x += resolution;
	  item.y += resolution;
	  item.code = curchar->code ();
	  item.bitmap = curchar->bitmap ();
	  item.width = curchar->hpixels ();
	  item.height = curchar->vpixels ();
	  maxvpixel = xmax (maxvpixel, item.y + item.height);
	  maxhpixel = xmax (maxhpixel, item.x + item.width);
	  page.additem (item);
	  goto fin_set;
	}
      else
	{
	  switch (opcode)
	    {
	    case set_rule:
	      goto fin_rule;
	    case put_rule:
	      goto fin_rule;
	    case nop:
	      goto done;
	    case bop:
	      bad (dviformaterror);
	      goto L9998;
	    case eop:
	      if (states->depth () != 0)
		{
		  bad (dviformaterror);
		  goto L9998;
		}
	      goto L9999;
	    case p_ush:
	      states->save ();
	      goto done;
	    case p_op:
	      states->restore ();
	      goto done;
	      
	    case w0:
	    case w1: case w1+1: case w1+2: case w1+3:
	      w = p;
	      goto out_space;
	    case x0:
	    case x1: case x1+1: case x1+2: case x1+3:
	      x = p;		// intended fall through!
	    case four_cases(right1):
	    out_space:
	    if (curfont && (p >= curfont->wordspace ()
			    || p <= - curfont->backspace ()))
	      hh = pixelround (h + p);
	    else
	      hh += pixelround (p);
	    q = p;
	    goto move_right;
	    
	    default:
	      if (specialcases(opcode, p))
		goto done;
	      else
		goto L9998;
	    }
	}
      
    fin_set:
      /* Finish a command that either sets or puts a character, then
	 goto move_right or done */
      dprintf1 ("char %ld\n", p);
      if (p < 0)
	p = 255 - ((-1 - p) % 256);
      else
	if (p >= 256)
	  p %= 256;		// width computation for oriental fonts
      q = curchar->width ();
      if (opcode >= put1)
	goto done;
      hh += curchar->escapement ();
      goto move_right;

    fin_rule:
      /* Finish a command that either sets or puts a rule, then goto
	 move_right or done */
      q = dvistream->readsignedquad ();
      if (p > 0 && q > 0)
	{
	  dvi_item item;
	  item.bitmap = 0;
	  item.x = (int) hh;
	  item.y = (int) vv;
	  item.x += resolution;
	  item.y += resolution;
	  item.width = (int) rulepixels (q);
	  item.height = (int) rulepixels (p);
	  maxvpixel = xmax (maxvpixel, item.y + item.height);
	  maxhpixel = xmax (maxhpixel, item.x + item.width);
	  page.additem (item);
	}
      if (opcode == put_rule)
	goto done;
      hh = hh + rulepixels (q);
      goto move_right;
      
    move_right:
      /* Finish a command that sets h=h+q, then goto done */
      {
	dvi_integer hhh = pixelround (h + q); // h, rounded to the nearest pxl
	if (abs (hhh - hh) > max_drift)
	  if (hhh > hh)
	    hh = hhh - max_drift;
	  else
	    hh = hhh + max_drift;
	h += q;
	dprintf1 ("hh=%ld\n", hh);
	goto done;
      }
      
    done:
      ;
    }

L9998:
  delete dvistream;
  dvistream = 0;
  return (dvi_false);

L9999:
  delete dvistream;
  dvistream = 0;
  return (dvi_true);
}

dvi_bool
dvi::specialcases (int opcode,
		   dvi_integer p)

{
  dvi_bool pure = dvi_true;	// is the command error-free?

  if (opcode >= fnt_num && opcode <= fnt_num+63)
    goto change_font;
  
  switch (opcode)
    {
    case y_0:
    case four_cases(y_1):
      y = p;
      goto out_vmove;
    case z0:
    case z1: case z1+1: case z1+2: case z1+3:
      z = p;
    case down1: case down2: case down3: case down4:
    out_vmove:
      if (curfont && abs(p) >= curfont->linespace ())
	vv = pixelround (v + p);
      else
	vv += pixelround (p);
      goto move_down;
    case fnt1: case fnt2: case fnt3: case fnt4:
      goto change_font;
    case fnt_def1: case fnt_def1+1: case fnt_def1+2: case fnt_def1+3:
      // Skip a font definition
      {
	dvistream->skipg (12);
	int areanamelen = dvistream->readbyte ();
	int fontnamelen = dvistream->readbyte ();
	dvistream->skipg (areanamelen + fontnamelen);
      }
      goto done;
    case xxx1: case xxx1+1: case xxx1+2: case xxx1+3:
      // Translate an xxx command and goto done
      {
	if (p < 0)
	  bad (dviformaterror);
	dvi_bool bad_char = dvi_false;
	for (dvi_integer k = 1; k <= p; k++)
	  {
	    dvi_integer q = dvistream->readbyte ();
	    if (q <' ' || q > '~')
	      bad_char = dvi_true;
	  }
	if (bad_char)
	  bad (dviformaterror);
	goto done;
      }
    case pre:
      bad (dviformaterror);
      goto L9998;
    case post:
    case post_post:
      bad (dviformaterror);
      goto L9998;
    default:
      bad (dviformaterror);
      goto done;
    }
  
move_down:
  // Finish a command that sets v=v+p, then goto done
  {
    dvi_integer vvv = pixelround (v + p); // |v|, rounded to the nearest pixel
    if (abs (vvv - vv) > max_drift)
      if (vvv > vv)
	vv = vvv - max_drift;
      else
	vv = vvv + max_drift;
    v += p;
    dprintf1 ("vv=%ld\n", vv);
    goto done;
  }
  
change_font:
  // Finish a command that changes the current font, then goto done
  curfontidx = p;
  dprintf1 ("selecting font %ld\n", (long) p);
  curfont = fonts[p];
  if (curfont == 0)
    {
      bad (dviformaterror);
      goto L9998;
    }
  goto done;

L9998:
  pure = dvi_false;

done:
  return (pure);
}

dvi *
dvi::create (const char *filename,
	     const char *mode1,
	     int resolution1,
	     const char *mode2,
	     int resolution2,
	     size_t bytesperrword)

{
  return (new dvi (filename, mode1, resolution1, mode2, resolution2,
		   bytesperrword));
}

void
dvi::destroy (dvi *dviobj)

{
  delete dviobj;
}

void
dvi::newmode (const char *mode1,
	      unsigned resolution1,
	      const char *mode2,
	      unsigned resolution2)

{
  freecontents ();
  for (size_t i = 0; i < contabsize; i++)
    {
      switchcontext (i);
      if (mode)
	{
	  free ((void *) mode);
	  mode = 0;
	}
    }
  if (contab)
    {
      delete [] contab;
      contab = 0;
    }
  contab = new _context[2];
  curcontext = 0;
  maxhpixel = 0;
  maxvpixel = 0;
  mode = strdup (mode1);
  resolution = resolution1;
  contab[0].save (*this);
  contabsize = 1;
  curcontext = 0;
  if (mode2)
    {
      contabsize++;
      switchcontext (contabsize-1);
      maxhpixel = 0;
      maxvpixel = 0;
      mode = strdup (mode2);
      resolution = resolution2;
    }

  switchcontext (0);

  scan ();
}

#ifdef USE_MFC
#include "afxdllx.h"
static AFX_EXTENSION_MODULE extensionDLL;

extern "C" int APIENTRY 
DllMain (HINSTANCE hInstance,
	 DWORD dwReason,
	 LPVOID)

{
  if (dwReason == DLL_PROCESS_ATTACH)
    {
      if (! AfxInitExtensionModule (extensionDLL, hInstance))
	return 0;
    }
  return (1);
}
#endif

/// dvi.cpp ends here
